/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/


/*
 * iocontrols.h
 *
 *        The driver io controls.
 */

#ifndef _IOCONTROLS_H_
#define _IOCONTROLS_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_MODULE
#include "includeCheck.h"

#include "pshare_ext.h"
#include "basic_initblock.h"
#include "vmci_defs.h"

#ifndef VMX86_SERVER
#include "numa_defs.h"
#endif

/*
 * Driver version.
 *
 * Increment major version when you make an incompatible change.
 * Compatibility goes both ways (old driver with new executable
 * as well as new driver with old executable).
 *
 * Note: Vmcore compatibility is different from driver versioning.
 * For vmcore puposes, the bora tree is conceptually split in two:
 * vmcore, and rest-of-bora. The vmmon driver is largely outside
 * vmcore and vmcore imports functionality from vmmon. Addition,
 * deletion or modification of an iocontrol used only by rest-of-bora
 * does not break vmcore compatibility. 
 *
 * See bora/doc/vmcore details.
 *
 */

#define VMMON_VERSION           (161 << 16 | 0)
#define VMMON_VERSION_MAJOR(v)  ((uint32) (v) >> 16)
#define VMMON_VERSION_MINOR(v)  ((uint16) (v))


/*
 * ENOMEM returned after MAX_VMS virtual machines created
 */

#ifdef VMX86_SERVER
#ifdef VMX86_LCH /* SMALL_FOOTPRINT */
#define MAX_VMS 4
#else
#define MAX_VMS	128
#endif
#else
#define MAX_VMS 128
#endif


#if !__linux__
/*
 * On platforms other than Linux, IOCTLCMD_foo values are just numbers, and
 * we build the IOCTL_VMX86_foo values around these using platform-specific
 * format for encoding arguments and sizes.
 */
#  define IOCTLCMD(_cmd) IOCTLCMD_ ## _cmd
#else // if __linux__
/*
 * Linux defines _IO* macros, but the core kernel code ignore the encoded
 * ioctl value. It is up to individual drivers to decode the value (for
 * example to look at the size of a structure to determine which version
 * of a specific command should be used) or not (which is what we
 * currently do, so right now the ioctl value for a given command is the
 * command itself).
 *
 * Hence, we just define the IOCTL_VMX86_foo values directly, with no
 * intermediate IOCTLCMD_ representation.
 */
#  define IOCTLCMD(_cmd) IOCTL_VMX86_ ## _cmd
#endif


enum IOCTLCmd {
   /*
    * We need to bracket the range of values used for ioctls, because x86_64
    * Linux forces us to explicitly register ioctl handlers by value for
    * handling 32 bit ioctl syscalls.  Hence FIRST and LAST.  FIRST must be
    * 2001 so that VERSION is 2001 for backwards compatibility.
    */
#if defined(__linux__) || defined(_WIN32)
   IOCTLCMD(NOT_IMPLEMENTED) = 1,
   /* Start at 2001 because legacy code did. */
   IOCTLCMD(FIRST) = 2001,
#else
   /* Start at 0. */
   IOCTLCMD(FIRST),
#endif
   IOCTLCMD(VERSION) = IOCTLCMD(FIRST),
   IOCTLCMD(CREATE_VM),
   IOCTLCMD(RELEASE_VM),
   IOCTLCMD(GET_NUM_VMS),
   IOCTLCMD(INIT_VM),
   IOCTLCMD(LATE_INIT_VM),
   IOCTLCMD(RUN_VM),
   IOCTLCMD(LOOK_UP_MPN),
   IOCTLCMD(LOCK_PAGE),
   IOCTLCMD(UNLOCK_PAGE),
   IOCTLCMD(APIC_BASE),
   IOCTLCMD(SET_HARD_LIMIT),
   IOCTLCMD(GET_MEM_INFO),
   IOCTLCMD(ADMIT),
   IOCTLCMD(SET_MEM_USAGE),
   IOCTLCMD(READMIT),
   IOCTLCMD(PAE_ENABLED),
#ifndef __APPLE__
   IOCTLCMD(HOST_X86_64),
#else
   IOCTLCMD(HOST_X86_CM),
#endif
   IOCTLCMD(GET_TOTAL_MEM_USAGE),
   IOCTLCMD(COMPLETE_USER_CALL),
   IOCTLCMD(GET_KHZ_ESTIMATE),
   IOCTLCMD(SET_HOST_CLOCK_RATE),
   IOCTLCMD(READ_PAGE),
   IOCTLCMD(WRITE_PAGE),
   IOCTLCMD(LOCK_PAGE_NEW),
   IOCTLCMD(UNLOCK_PAGE_BY_MPN),
   IOCTLCMD(MARK_LOCKEDVARANGE_CLEAN),
   IOCTLCMD(COW_SHARE),
   IOCTLCMD(COW_CHECK),
   IOCTLCMD(COW_UPDATE_HINT),
   IOCTLCMD(COW_COPY_PAGE),
   IOCTLCMD(COW_REMOVE_HINT),
   IOCTLCMD(COW_GET_ZERO_MPN),
    /* AWE calls */
   IOCTLCMD(ALLOC_LOCKED_PAGES),
   IOCTLCMD(FREE_LOCKED_PAGES),
   IOCTLCMD(GET_LOCKED_PAGES_LIST),

   IOCTLCMD(APIC_ID),
   IOCTLCMD(SVM_ENABLED_CPU),
   IOCTLCMD(VT_ENABLED_CPU),
   IOCTLCMD(VT_SUPPORTED_CPU),
   IOCTLCMD(BROKEN_CPU_HELPER),
   IOCTLCMD_VMCI_INIT_CONTEXT,
   IOCTLCMD_VMCI_CREATE_PROCESS,
   IOCTLCMD_VMCI_CREATE_DATAGRAM_PROCESS,
   IOCTLCMD_VMCI_HYPERCALL,
   IOCTLCMD_VMCI_READ_CALL,
   IOCTLCMD_VMCI_SHAREDMEM_CREATE,
   IOCTLCMD_VMCI_SHAREDMEM_ATTACH,
   IOCTLCMD_VMCI_SHAREDMEM_QUERY,
   IOCTLCMD_VMCI_SHAREDMEM_DETACH,

   IOCTLCMD(COUNT_PRESENT_PAGES),
  
   IOCTLCMD(INIT_NUMA_INFO),
   IOCTLCMD(GET_NUMA_MEM_STATS),
    
   /*
    * Keep host-specific calls at the end so they can be undefined
    * without renumbering the common calls.
    */

#if __linux__ || defined __APPLE__
   IOCTLCMD(SET_UID),		// VMX86_DEVEL only
   IOCTLCMD(ALLOW_CORE_DUMP),

   IOCTLCMD(BROADCAST_IPI),	// SMP 2.2.8+ only

   // Not implemented in the main branch.
   IOCTLCMD(REGISTER_PASSTHROUGH_IO),
   IOCTLCMD(REGISTER_PASSTHROUGH_IRQ),
   IOCTLCMD(FREE_PASSTHROUGH_IO),
   IOCTLCMD(FREE_PASSTHROUGH_IRQ),
   IOCTLCMD(START_PASSTHROUGH),
   IOCTLCMD(STOP_PASSTHROUGH),
   IOCTLCMD(QUERY_PASSTHROUGH),

   IOCTLCMD(REGISTER_PERFCTR),
   IOCTLCMD(START_PERFCTR),
   IOCTLCMD(STOP_PERFCTR),
   IOCTLCMD(RELEASE_PERFCTR),

   IOCTLCMD(GET_ALL_CPUID),

   IOCTLCMD(SET_THREAD_AFFINITY),
   IOCTLCMD(GET_THREAD_AFFINITY),

   IOCTLCMD(GET_KERNEL_CLOCK_RATE),
#endif

#if defined _WIN32 || defined __APPLE__
   IOCTLCMD(ALLOC_CONTIG_PAGES),
   IOCTLCMD(SEND_IPI),
#endif

#if defined _WIN32 || __linux__
   IOCTLCMD(ACK_USER_CALL),
#endif

#if defined _WIN32
   IOCTLCMD(FREE_CONTIG_PAGES),
   IOCTLCMD(BEEP),
   IOCTLCMD(HARD_LIMIT_MONITOR_STATUS),	// Windows 2000 only
   IOCTLCMD(BLUE_SCREEN),	// USE_BLUE_SCREEN only
   IOCTLCMD(CHANGE_HARD_LIMIT),
   IOCTLCMD(SYNC_GET_TSCS),
   IOCTLCMD(SYNC_SET_TSCS),
   IOCTLCMD(GET_KERNEL_PROC_ADDRESS),
   IOCTLCMD(READ_VA64),
   IOCTLCMD(SET_MEMORY_PARAMS),
#endif

#if defined __APPLE__
   IOCTLCMD(ALLOC_LOW_PAGES),
   IOCTLCMD(SET_PRIV_SYMS),
#endif

   IOCTLCMD(SET_POLL_TIMEOUT_PTR),
   IOCTLCMD(IS_MP_SAFE),

   IOCTLCMD(FAST_MON_SWITCH_START),
   IOCTLCMD(FAST_MON_SWITCH_IS_DONE),
   IOCTLCMD(FAST_MON_SWITCH_TRANSFER),
   IOCTLCMD(FAST_MON_SWITCH_END),

   // Must be last.
   IOCTLCMD(LAST)
};


#if defined _WIN32
/*
 * Windows ioctl definitions.
 *
 * We use the IRP Information field for the return value
 * of IOCTLCMD_RUN_VM, to be faster since it is used a lot.
 */

#define FILE_DEVICE_VMX86        0x8101
#define VMX86_IOCTL_BASE_INDEX   0x801
#define VMIOCTL_BUFFERED(name) \
     CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX + IOCTLCMD_ ## name, \
	       METHOD_BUFFERED, \
	       FILE_ANY_ACCESS)
#define VMIOCTL_NEITHER(name) \
      CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX + IOCTLCMD_ ## name, \
	       METHOD_NEITHER, \
	       FILE_ANY_ACCESS)

#define IOCTL_VMX86_VERSION		VMIOCTL_BUFFERED(VERSION)
#define IOCTL_VMX86_CREATE_VM		VMIOCTL_BUFFERED(CREATE_VM)
#define IOCTL_VMX86_RELEASE_VM		VMIOCTL_BUFFERED(RELEASE_VM)
#define IOCTL_VMX86_GET_NUM_VMS		VMIOCTL_BUFFERED(GET_NUM_VMS)
#define IOCTL_VMX86_INIT_VM		VMIOCTL_BUFFERED(INIT_VM)
#define IOCTL_VMX86_INIT_NUMA_INFO	VMIOCTL_BUFFERED(INIT_NUMA_INFO)
#define IOCTL_VMX86_GET_NUMA_MEM_STATS  VMIOCTL_BUFFERED(GET_NUMA_MEM_STATS)
#define IOCTL_VMX86_LATE_INIT_VM	VMIOCTL_BUFFERED(LATE_INIT_VM)
#define IOCTL_VMX86_RUN_VM		VMIOCTL_NEITHER(RUN_VM)
#define IOCTL_VMX86_SEND_IPI		VMIOCTL_NEITHER(SEND_IPI)
#define IOCTL_VMX86_LOOK_UP_MPN		VMIOCTL_BUFFERED(LOOK_UP_MPN)
#define IOCTL_VMX86_LOCK_PAGE		VMIOCTL_BUFFERED(LOCK_PAGE)
#define IOCTL_VMX86_UNLOCK_PAGE		VMIOCTL_BUFFERED(UNLOCK_PAGE)
#define IOCTL_VMX86_APIC_BASE		VMIOCTL_BUFFERED(APIC_BASE)
#define IOCTL_VMX86_SET_HARD_LIMIT	VMIOCTL_BUFFERED(SET_HARD_LIMIT)
#define IOCTL_VMX86_GET_MEM_INFO	VMIOCTL_BUFFERED(GET_MEM_INFO)
#define IOCTL_VMX86_ADMIT  	        VMIOCTL_BUFFERED(ADMIT)
#define IOCTL_VMX86_READMIT             VMIOCTL_BUFFERED(READMIT)
#define IOCTL_VMX86_SET_MEM_USAGE       VMIOCTL_BUFFERED(SET_MEM_USAGE)
#define IOCTL_VMX86_PAE_ENABLED		VMIOCTL_BUFFERED(PAE_ENABLED)
#define IOCTL_VMX86_HOST_X86_64         VMIOCTL_BUFFERED(HOST_X86_64)
#define IOCTL_VMX86_COW_SHARE           VMIOCTL_BUFFERED(COW_SHARE)
#define IOCTL_VMX86_COW_CHECK           VMIOCTL_BUFFERED(COW_CHECK)
#define IOCTL_VMX86_COW_UPDATE_HINT     VMIOCTL_BUFFERED(COW_UPDATE_HINT)
#define IOCTL_VMX86_COW_COPY_PAGE       VMIOCTL_BUFFERED(COW_COPY_PAGE)
#define IOCTL_VMX86_COW_REMOVE_HINT     VMIOCTL_BUFFERED(COW_REMOVE_HINT)
#define IOCTL_VMX86_COW_GET_ZERO_MPN    VMIOCTL_BUFFERED(COW_GET_ZERO_MPN)
#define IOCTL_VMX86_BEEP		VMIOCTL_BUFFERED(BEEP)
#define IOCTL_VMX86_HARD_LIMIT_MONITOR_STATUS	VMIOCTL_BUFFERED(HARD_LIMIT_MONITOR_STATUS)
#define IOCTL_VMX86_CHANGE_HARD_LIMIT   VMIOCTL_BUFFERED(CHANGE_HARD_LIMIT)
#define IOCTL_VMX86_ALLOC_CONTIG_PAGES  VMIOCTL_BUFFERED(ALLOC_CONTIG_PAGES)
#define IOCTL_VMX86_FREE_CONTIG_PAGES   VMIOCTL_BUFFERED(FREE_CONTIG_PAGES)
#ifdef USE_BLUE_SCREEN
#define IOCTL_VMX86_BLUE_SCREEN		VMIOCTL_BUFFERED(BLUE_SCREEN)
#endif

#define IOCTL_VMX86_GET_TOTAL_MEM_USAGE	VMIOCTL_BUFFERED(GET_TOTAL_MEM_USAGE)
#define IOCTL_VMX86_ACK_USER_CALL       VMIOCTL_BUFFERED(ACK_USER_CALL)
#define IOCTL_VMX86_COMPLETE_USER_CALL  VMIOCTL_BUFFERED(COMPLETE_USER_CALL)
#define IOCTL_VMX86_GET_KHZ_ESTIMATE    VMIOCTL_BUFFERED(GET_KHZ_ESTIMATE)
#define IOCTL_VMX86_SET_HOST_CLOCK_RATE VMIOCTL_BUFFERED(SET_HOST_CLOCK_RATE)
#define IOCTL_VMX86_SYNC_GET_TSCS       VMIOCTL_BUFFERED(SYNC_GET_TSCS)
#define IOCTL_VMX86_SYNC_SET_TSCS       VMIOCTL_BUFFERED(SYNC_SET_TSCS)
#define IOCTL_VMX86_READ_PAGE           VMIOCTL_BUFFERED(READ_PAGE)
#define IOCTL_VMX86_WRITE_PAGE          VMIOCTL_BUFFERED(WRITE_PAGE)
#define IOCTL_VMX86_LOCK_PAGE_NEW	VMIOCTL_BUFFERED(LOCK_PAGE_NEW)
#define IOCTL_VMX86_UNLOCK_PAGE_BY_MPN  VMIOCTL_BUFFERED(UNLOCK_PAGE_BY_MPN)
#define IOCTL_VMX86_ALLOC_LOCKED_PAGES	VMIOCTL_BUFFERED(ALLOC_LOCKED_PAGES)
#define IOCTL_VMX86_FREE_LOCKED_PAGES   VMIOCTL_BUFFERED(FREE_LOCKED_PAGES)
#define IOCTL_VMX86_GET_LOCKED_PAGES_LIST VMIOCTL_BUFFERED(GET_LOCKED_PAGES_LIST)

#define IOCTL_VMX86_GET_KERNEL_PROC_ADDRESS  VMIOCTL_BUFFERED(GET_KERNEL_PROC_ADDRESS)
#define IOCTL_VMX86_READ_VA64		VMIOCTL_BUFFERED(READ_VA64)
#define IOCTL_VMX86_SET_MEMORY_PARAMS   VMIOCTL_BUFFERED(SET_MEMORY_PARAMS)

#define IOCTL_VMX86_APIC_ID		VMIOCTL_BUFFERED(APIC_ID)
#define IOCTL_VMX86_SVM_ENABLED_CPU	VMIOCTL_BUFFERED(SVM_ENABLED_CPU)
#define IOCTL_VMX86_VT_ENABLED_CPU	VMIOCTL_BUFFERED(VT_ENABLED_CPU)
#define IOCTL_VMX86_VT_SUPPORTED_CPU	VMIOCTL_BUFFERED(VT_SUPPORTED_CPU)
#define IOCTL_VMX86_BROKEN_CPU_HELPER	VMIOCTL_BUFFERED(BROKEN_CPU_HELPER)
#define IOCTL_VMX86_COUNT_PRESENT_PAGES	VMIOCTL_BUFFERED(COUNT_PRESENT_PAGES)
#define IOCTL_VMX86_VMCI_INIT_CONTEXT VMIOCTL_BUFFERED(VMCI_INIT_CONTEXT)
#define IOCTL_VMX86_VMCI_CREATE_PROCESS VMIOCTL_BUFFERED(VMCI_CREATE_PROCESS)
#define IOCTL_VMX86_VMCI_CREATE_DATAGRAM_PROCESS \
               VMIOCTL_BUFFERED(VMCI_CREATE_DATAGRAM_PROCESS)
#define IOCTL_VMX86_VMCI_HYPERCALL VMIOCTL_BUFFERED(VMCI_HYPERCALL)
#define IOCTL_VMX86_VMCI_READ_CALL VMIOCTL_NEITHER(VMCI_READ_CALL)
#define IOCTL_VMX86_VMCI_SHAREDMEM_CREATE  \
               VMIOCTL_BUFFERED(VMCI_SHAREDMEM_CREATE)
#define IOCTL_VMX86_VMCI_SHAREDMEM_ATTACH  \
               VMIOCTL_BUFFERED(VMCI_SHAREDMEM_ATTACH)
#define IOCTL_VMX86_VMCI_SHAREDMEM_QUERY   \
               VMIOCTL_BUFFERED(VMCI_SHAREDMEM_QUERY)
#define IOCTL_VMX86_VMCI_SHAREDMEM_DETACH  \
               VMIOCTL_BUFFERED(VMCI_SHAREDMEM_DETACH)

#define IOCTL_VMX86_FAST_MON_SWITCH_START    VMIOCTL_BUFFERED(FAST_MON_SWITCH_START)
#define IOCTL_VMX86_FAST_MON_SWITCH_IS_DONE  VMIOCTL_BUFFERED(FAST_MON_SWITCH_IS_DONE)
#define IOCTL_VMX86_FAST_MON_SWITCH_TRANSFER VMIOCTL_BUFFERED(FAST_MON_SWITCH_TRANSFER)
#define IOCTL_VMX86_FAST_MON_SWITCH_END      VMIOCTL_BUFFERED(FAST_MON_SWITCH_END)

#endif


/*
 * Return codes from page locking, unlocking, and MPN lookup.
 * They share an error code space because they call one another
 * internally.
 *
 *    PAGE_LOCK_FAILED              The host refused to lock a page.
 *    PAGE_LOCK_LIMIT_EXCEEDED      We have reached the limit of locked
 *                                  pages for all VMs
 *    PAGE_LOCK_TOUCH_FAILED        Failed to touch page after lock.
 *    PAGE_LOCK_IN_TRANSITION       The page is locked but marked by Windows
 *                                  as nonpresent in CPU PTE and in transition 
 *                                  in Windows PFN.  
 *
 *    PAGE_LOCK_SYS_ERROR           System call error.
 *    PAGE_LOCK_ALREADY_LOCKED      Page already locked.
 *    PAGE_LOCK_MEMTRACKER_ERROR    MemTracker fails.
 *    PAGE_LOCK_PHYSTRACKER_ERROR   PhysTracker fails.
 *    PAGE_LOCK_MDL_ERROR           Mdl error on Windows.
 *
 *    PAGE_UNLOCK_NO_ERROR          Unlock successful (must be 0).
 *    PAGE_UNLOCK_NOT_TRACKED       Not in memtracker.
 *    PAGE_UNLOCK_NO_MPN            Tracked but no MPN.
 *    PAGE_UNLOCK_NOT_LOCKED        Not locked.
 *    PAGE_UNLOCK_TOUCH_FAILED      Failed to touch page.
 *    PAGE_UNLOCK_MISMATCHED_TYPE   Tracked but was locked by different API 
 *
 *    PAGE_LOOKUP_INVALID_ADDR      Consistency checking on Windows.
 *    PAGE_LOOKUP_BAD_HIGH_ADDR     Consistency checking on Windows.
 *    PAGE_LOOKUP_ZERO_ADDR         Consistency checking on Windows.
 *    PAGE_LOOKUP_SMALL_ADDR        Consistency checking on Windows.
 *
 * PAGE_LOCK_ERROR_BIT is used to encode the error value as a positive
 * number to get through ioctl() on Linux.
 *
 * -- edward
 */

#define PAGE_LOCK_FAILED                  -1
#define PAGE_LOCK_LIMIT_EXCEEDED          -2
#define PAGE_LOCK_TOUCH_FAILED            -3
#define PAGE_LOCK_IN_TRANSITION           -4

#define PAGE_LOCK_SYS_ERROR              -10
#define PAGE_LOCK_ALREADY_LOCKED         -11
#define PAGE_LOCK_MEMTRACKER_ERROR       -12
#define PAGE_LOCK_PHYSTRACKER_ERROR      -13
#define PAGE_LOCK_MDL_ERROR              -14

#define PAGE_UNLOCK_NO_ERROR		   0
#define PAGE_UNLOCK_NOT_TRACKED		-100
#define PAGE_UNLOCK_NO_MPN		-101
#define PAGE_UNLOCK_NOT_LOCKED		-102
#define PAGE_UNLOCK_TOUCH_FAILED	-103
#define PAGE_UNLOCK_MISMATCHED_TYPE	-104

#define PAGE_LOOKUP_INVALID_ADDR        -200
#define PAGE_LOOKUP_BAD_HIGH_ADDR       -201
#define PAGE_LOOKUP_ZERO_ADDR           -202
#define PAGE_LOOKUP_SMALL_ADDR          -203
#define PAGE_LOOKUP_NOT_TRACKED          -10	// added to another code
#define PAGE_LOOKUP_NO_MPN               -20	// added to another code
#define PAGE_LOOKUP_NOT_LOCKED           -30	// added to another code
#define PAGE_LOOKUP_NO_VM                -40	// added to another code

#define PAGE_LOCK_SUCCESS(mpn) ((int) (mpn) >= 0)
#define PAGE_LOCK_SOFT_FAILURE(mpn) ((int) (mpn) < 0 && (int) (mpn) > -10)

#define PAGE_LOCK_ERROR_BIT 0x4000000


/*
 * Flags sent into APICBASE ioctl
 */

#define APIC_FLAG_DISABLE_NMI       0x00000001
#define APIC_FLAG_PROBE             0x00000002
#define APIC_FLAG_FORCE_ENABLE      0x00000004


/*
 * Structure sent into REGISTER_PERFCOUNTERS ioctl
 */

#define PERFCTR_MAX_COUNTERS        2
#define PERFCTR_INVALID_EVENT       0
#define PERFCTR_INVALID_IRQ         -1

typedef struct PerfCtrInfo {
   uint32 eventNum;
   uint32 samplingRate;
} PerfCtrInfo;

typedef struct PerfCtrRegisterArgs {
   PerfCtrInfo counterInfo[PERFCTR_MAX_COUNTERS];
   int irq;
   Bool totalOnly;
} PerfCtrRegisterArgs;

typedef struct VMAPICInfo {
   uint32 flags;
   MA32   apicBase;
} VMAPICInfo; 

#ifndef MAX_PROCESSORS
#define MAX_PROCESSORS 32
#endif

typedef struct VMSyncWriteTSCs {
   uint32 cpuSet;
   uint32 TSCs[MAX_PROCESSORS];
} VMSyncWriteTSCs;


#define VMX86_DRIVER_VCPUID_OFFSET	1000


/*
 * We keep track of 3 different limits on the number of pages we can lock.
 * The host limit is determined at driver load time (in windows only) to
 * make sure we do not starve the host by locking too many pages.
 * The static limit is user defined in the UI and the dynamic limit is 
 * set by authd's hardLimitMonitor code (windows only), which queries
 * host load and adjusts the limit accordingly.  We lock the minimum of 
 * all these values at any given time.
 */
typedef struct LockedPageLimit {
   uint32 host;        // driver calculated maximum for this host
   uint32 configured;  // user defined maximum pages to lock
   uint32 dynamic;     // authd hardLimitMonitor pages to lock
} LockedPageLimit;

/*
 * Data structures for the GET_MEM_INFO and ADMIT ioctls.
 */

typedef struct VMMemMgmtInfo {
   uint32          minAllocation;   // minimum pages for vm
   uint32          maxAllocation;   // maximum pages the vm could lock
   uint32          shares;          // proportional sharing weight
   uint32          nonpaged;        // overhead memory
   uint32          paged;           // guest memory(mainmem + other regions)
   uint32          mainMemSize;     // guest main memory size
   uint32          locked;          // number of pages locked by this vm
   uint32          shared;          // number of guest pages shared
   uint32          usedPct;         // % of guest memory being touched
   uint32          admitted;        // admission control
} VMMemMgmtInfo;

#define VMMEM_COW_HOT_PAGES 5

typedef struct VMMemCOWHotPage {
   uint32          mpn;
   uint32          ref;
   uint64          key;
} VMMemCOWHotPage;

typedef struct VMMemCOWInfo {
   uint32          numHints;
   uint32          uniqueMPNs;       // current MPNs used for COW
   uint32          totalUniqueMPNs;  // total MPNs ever used as COW
   uint32          numBreaks;
   uint32          numRef;
   uint32          _pad[1];
   VMMemCOWHotPage hot[VMMEM_COW_HOT_PAGES];
} VMMemCOWInfo;

typedef struct VMMemInfoArgs {
   VMMemCOWInfo    cowInfo;            // global page sharing state
   uint32          numVMs;             
   uint32          minVmMemPct;        // % of vm that must fit in maxLockedPages
   uint32          globalMinAllocation;// pages that must fit in maxLockedPages
   uint32          numLockedPages;     // total locked pages by all vms
   LockedPageLimit lockedPageLimit;    // set of locked page limits
   uint32          maxLockedPages;     // effective limit on locked pages
   uint32          callerIndex;        // this vm's index memInfo array
   VMMemMgmtInfo   memInfo[1];
   uint8           _pad[4];
} VMMemInfoArgs;

#define VM_GET_MEM_INFO_SIZE(numVMs) \
   (sizeof(VMMemInfoArgs) - sizeof(VMMemMgmtInfo) + (numVMs) * sizeof(VMMemMgmtInfo))

typedef struct VMMPNList {
   uint32 mpnCount; // IN (and OUT on Mac OS)
   VA32 mpn32List;  // IN: 32-bit user VA to an array of 32-bit MPNs.
} VMMPNList;

typedef struct VARange {
   VA32     addr;
   unsigned len;
   VA32     bv;
} VARange;

typedef struct VMMPNListVA {
   uint32       mpnCount;
   VA32         mpn32List;  /* list of 32 bit mpns */
   VA32		va;	    /* userspace VA. */
} VMMPNListVA;

typedef struct VMMUnlockPageByMPN {
   MPN32	  mpn;
   VA32 	  va;	    /* optional userspace VA */
} VMMUnlockPageByMPN;

typedef struct VMMReadWritePage {
   MPN32        mpn; // IN
   VA32         buf; // IN
} VMMReadWritePage;

#define VMM_CPP_MAXPAGES 400
typedef struct VMMCountPresentPages {
   uint32       numPages;                    /* number of pages in pageList */
   VA32         pageList[VMM_CPP_MAXPAGES];  /* random sample of pages to check */
} VMMCountPresentPages;

struct passthrough_iorange {
   unsigned short ioBase;   /* Base of range to pass through. */
   unsigned short numPorts; /* Length of range. */
};

typedef struct COWShareInfo {
   uint32       numPages;
   MPN          pshareMPN;
   Bool         shareFailure;
   Bool         updateHints;
} COWShareInfo;

/* 
 * Classes of page contents. Basicly, we divide all the page contents into
 * two classes - regular and non-regular. Regular class constitute contents
 * that can be described with some regular pattern. Otherwise, contents is
 * referred to as a non-regular. We also differentiate between trivial (all 
 * zeroes) and non-trivial contents. PageClass sets partial ordering on 
 * contents class identifiers to simplify contents-sensitive lookups. We 
 * keep the following ordering: all non-trivial classes precede the trivial
 * one, then all regular classes precede non-regular ones. In terms of a
 * layout it means: all identifiers for non-trivial classes of regular page
 * contents must precede PC_UNKNOWN; PC_ZERO must be the last one). Compile 
 * time assertions in pshare.c provide some guarding on that.
 */
typedef enum {
   PC_WORD5,      // zero page except for the 5th 32-bit word
   PC_UNKNOWN,    // unidentified contents class
   PC_ZERO        // zero page
} PageClass;

#define MAX_NONTRIVIAL_REGULAR_CLASS PC_WORD5
#define MAX_NONTRIVIAL_CLASS         PC_UNKNOWN
#define MAX_CLASS                    PC_ZERO

typedef struct COWHintInfo {
   uint32            nUpdates;
   PShare_HintUpdate updates[PSHARE_HINT_BATCH_PAGES_MAX];
} COWHintInfo;

typedef struct COWCheckInfo {
   uint32              numPages;
   PShare_COWCheckInfo check[PSHARE_MAX_COW_CHECK_PAGES];
} COWCheckInfo;

#ifndef VMX86_SERVER
typedef struct NUMAInfoArgs {
   uint32           numNodes;              // Keep this first in the structure
   uint32           numMemRanges;
   NUMA_NodeInfo    nodes[NUMA_MAX_NODES]; // Info about all NUMA nodes
} NUMAInfoArgs;
 
#define NUMA_INFO_ARGS_SIZE(args, numNodes) \
  (sizeof(args) - (sizeof(NUMA_NodeInfo) * NUMA_MAX_NODES) \
                                     + (sizeof(NUMA_NodeInfo) * numNodes))

typedef struct VMNUMAMemStatsArgs {
   uint32     	curCPU;
   uint32	curNUMANode;
   uint32     	numPagesPerNode[NUMA_MAX_NODES]; //For each NUMA node
} VMNUMAMemStatsArgs;
#endif
 
/*
 * VMCI driver initialization. This block can also be used to
 * pass initial group membership etc.
 */
typedef struct VMCIInitBlock {
   VMCIId cid;
#ifdef _WIN32
   int         event; // Hnd for signalling vmci calls on windows
#endif // _WIN32
} VMCIInitBlock;

typedef struct VMCIHypercallInfo {
   union {
      uint32 val32;
      uint64 val64;
   } result;
   VA64 addr;
   uint32 len;
} VMCIHypercallInfo;

/* 
 * XXX Merge hypercalls and driver calls, need to decide whether handles should 
 * be returned to userlevel or only resource ID. Makes it weird to send datagrams.
 */
typedef struct VMCIDrivercallInfo {
   uint32 len;
   VA32 addr;
   uint64 retval;
} VMCIDrivercallInfo;

typedef struct VMCISharedMemInfo {
   VMCIHandle handle;
   uint32 size;
   uint32 result;     
   VA64 va;           /* Currently only used in the guest. */ 
   char pageFileName[VMCI_PATH_MAX];
} VMCISharedMemInfo;


#if !defined _WIN32

typedef struct {
   uint64              ioarg;	// 64bit address
   uint64              iocmd;	// 64bit command
} VMIoctl64;

/*
 * We can move it at the beginning of our ioctl list (just below
 * IOCTLCMD_VERSION). It just should not change very often...
 */
// XXXMACOS Do we need this?
#define IOCTL_VMX86_IOCTL64  _IOW(0x99, 0x80, VMIoctl64)

/*
 * Structures for passing process affinity between vmmon and userspace.
 */
struct VMMonAffinity {
   uint32   pid;      // IN
   uint32   affinity; // IN for set, OUT for get, bit 0 == processor 0...
};
#endif

#if __linux__
/*
 * Linux uses mmap(2) to allocate contiguous locked pages, and uses these
 * macros to marshall real arguments to mmap's made-up 'offset' argument.
 */

#define VMMON_MAP_MT_LOW4GB	0
#define VMMON_MAP_MT_LOW16MB	1
#define VMMON_MAP_MT_ANY	2

#define VMMON_MAP_OFFSET_SHIFT	0
#define VMMON_MAP_OFFSET_MASK	0x00000FFF
#define VMMON_MAP_ORDER_SHIFT	12
#define VMMON_MAP_ORDER_MASK	0xF
#define VMMON_MAP_MT_SHIFT	16
#define VMMON_MAP_MT_MASK	0x7
#define VMMON_MAP_RSVD_SHIFT	19

#define VMMON_MAP_RSVD(base)	\
		((base) >> VMMON_MAP_RSVD_SHIFT)
#define VMMON_MAP_MT(base)	\
		(((base) >> VMMON_MAP_MT_SHIFT) & VMMON_MAP_MT_MASK)
#define VMMON_MAP_ORDER(base)	\
		(((base) >> VMMON_MAP_ORDER_SHIFT) & VMMON_MAP_ORDER_MASK)
#define VMMON_MAP_OFFSET(base)	\
		(((base) >> VMMON_MAP_OFFSET_SHIFT) & VMMON_MAP_OFFSET_MASK)

#define VMMON_MAP_BASE(mt, order)	(((mt) << VMMON_MAP_MT_SHIFT) | \
					 ((order) << VMMON_MAP_ORDER_SHIFT))

#elif defined _WIN32
/*
 * Windows uses an ioctl to allocate contiguous locked pages.
 */

typedef struct VMAllocContiguousMem {
   uint32 mpnCount; // IN
   VA32 mpn32List;  // IN: 32-bit user VA to an array of 32-bit MPNs.
   uint32 order;    // IN
   MPN32 maxMPN;    // IN
} VMAllocContiguousMem;
#elif defined __APPLE__
/*
 * Mac OS uses an ioctl to allocate contiguous locked pages.
 */

typedef struct VMAllocSpecialMem {
   uint32 numPages; // IN
   VA32 uAddr;      // OUT
} VMAllocSpecialMem;
#endif


#if defined __APPLE__
/*
 * Mac OS ioctl definitions.
 *
 * Mac OS defines _IO* macros, and the core kernel code uses the size encoded
 * in the ioctl value to copy the memory back and forth (depending on the
 * direction encoded in the ioctl value) between the user and kernel address
 * spaces.
 */

#include "cpuid_info.h"
// XXXMACOS lib/user/utilPosix.c depends on this file, but does not have access to vmx/public/vcpuset.h. So move the header to a more public area.
//#include "vcpuset.h"

typedef union {
   Vcpuid vcpuid; // IN
   int userCall;  // OUT
} VMRunVM;

typedef union {
   // XXX Should be of type 'VA', but now is not the time to fix this. --hpreg
   void *uvAddr; // IN: User virtual address
   MPN pageCode; // OUT: See PAGE_* above.
} VMLockPage;

typedef union {
   VMMUnlockPageByMPN in; // IN
   MPN pageCode;          // OUT: See PAGE_* above.
} VMUnlockPage;

typedef union {
   Bool use64Bit;       // IN
   Bool support;        // OUT
} VMVTSupport;

// Private kernel symbols.
typedef
#include "vmware_pack_begin.h"
struct {
   Bool capHighMem;

   unsigned int *vm_page_wire_count;
   unsigned int *vm_page_active_count;
   unsigned int *vm_page_inactive_count;
   void *kernel_memory_allocate;
   void *vm_map_get_phys_page;
}
#include "vmware_pack_end.h"
PrivSyms;

#include <sys/ioccom.h>

#define VMIOCTL(_cmd) \
   IOCTL_VMX86_ ## _cmd = _IO('V', IOCTLCMD_ ## _cmd)
#define VMIOCTL_I(_cmd, _type) \
   IOCTL_VMX86_ ## _cmd = _IOW('V', IOCTLCMD_ ## _cmd, _type)
#define VMIOCTL_O(_cmd, _type) \
   IOCTL_VMX86_ ## _cmd = _IOR('V', IOCTLCMD_ ## _cmd, _type)
#define VMIOCTL_IO(_cmd, _type) \
   IOCTL_VMX86_ ## _cmd = _IOWR('V', IOCTLCMD_ ## _cmd, _type)

enum IOCTLVmx86 {
   VMIOCTL_O(VERSION, int),
   VMIOCTL_O(CREATE_VM, int),
   VMIOCTL(RELEASE_VM),
   VMIOCTL_O(GET_NUM_VMS, int),
   VMIOCTL_I(INIT_VM, InitBlock),
   VMIOCTL(LATE_INIT_VM),
   VMIOCTL_IO(RUN_VM, VMRunVM),
   VMIOCTL_IO(LOOK_UP_MPN, VMLockPage),
   VMIOCTL_IO(LOCK_PAGE, VMLockPage),
   VMIOCTL_IO(UNLOCK_PAGE, VMLockPage),
   VMIOCTL(APIC_BASE), // XXXMACOS
   VMIOCTL_I(SET_HARD_LIMIT, int32),
   VMIOCTL_I(GET_MEM_INFO, VMMemInfoArgs *),
   VMIOCTL_IO(ADMIT, VMMemInfoArgs),
   VMIOCTL_I(SET_MEM_USAGE, int32),
   VMIOCTL_I(READMIT, int32),
   VMIOCTL_O(PAE_ENABLED, Bool),
   VMIOCTL_O(HOST_X86_CM, Bool),
   VMIOCTL_O(GET_TOTAL_MEM_USAGE, int32),
   VMIOCTL_I(COMPLETE_USER_CALL, Vcpuid),
   VMIOCTL_O(GET_KHZ_ESTIMATE, uint32),
   VMIOCTL_I(SET_HOST_CLOCK_RATE, int),
   VMIOCTL_I(READ_PAGE, VMMReadWritePage),
   VMIOCTL_I(WRITE_PAGE, VMMReadWritePage),
   VMIOCTL_IO(LOCK_PAGE_NEW, VMLockPage),
   VMIOCTL_IO(UNLOCK_PAGE_BY_MPN, VMUnlockPage),
   VMIOCTL(MARK_LOCKEDVARANGE_CLEAN), // XXXMACOS
   VMIOCTL_IO(COW_SHARE, COWShareInfo),
   VMIOCTL_IO(COW_CHECK, COWCheckInfo),
   VMIOCTL_IO(COW_UPDATE_HINT, COWHintInfo),
   VMIOCTL_I(COW_COPY_PAGE, MPN),
   VMIOCTL_I(COW_REMOVE_HINT, MPN),
   VMIOCTL_O(COW_GET_ZERO_MPN, MPN),

   VMIOCTL_IO(ALLOC_CONTIG_PAGES, VMAllocSpecialMem),
   VMIOCTL_IO(ALLOC_LOW_PAGES, VMAllocSpecialMem),
   VMIOCTL_IO(ALLOC_LOCKED_PAGES, VMMPNList),
   VMIOCTL_I(FREE_LOCKED_PAGES, VMMPNList),
   VMIOCTL_IO(GET_LOCKED_PAGES_LIST, VMMPNList),

   VMIOCTL(APIC_ID), // XXXMACOS
   VMIOCTL_O(SVM_ENABLED_CPU, Bool),
   VMIOCTL_O(VT_ENABLED_CPU, Bool),
   VMIOCTL_IO(VT_SUPPORTED_CPU, VMVTSupport),
   VMIOCTL_O(BROKEN_CPU_HELPER, uint32),

   VMIOCTL(COUNT_PRESENT_PAGES), // XXXMACOS

   VMIOCTL(INIT_NUMA_INFO), // XXXMACOS
   VMIOCTL(GET_NUMA_MEM_STATS), // XXXMACOS

#ifdef VMX86_DEVEL
   VMIOCTL(SET_UID),
#endif
   VMIOCTL(ALLOW_CORE_DUMP), // XXXMACOS

// XXXMACOS lib/user/utilPosix.c depends on this file, but does not have access to vmx/public/vcpuset.h. So move the header to a more public area.
   VMIOCTL_I(SEND_IPI, uint32),

   VMIOCTL_I(SET_PRIV_SYMS, PrivSyms),

   VMIOCTL(REGISTER_PERFCTR), // XXXMACOS
   VMIOCTL(START_PERFCTR), // XXXMACOS
   VMIOCTL(STOP_PERFCTR), // XXXMACOS
   VMIOCTL(RELEASE_PERFCTR), // XXXMACOS

   VMIOCTL_I(GET_ALL_CPUID, CPUIDResult *),

   VMIOCTL(SET_THREAD_AFFINITY), // XXXMACOS
   VMIOCTL(GET_THREAD_AFFINITY), // XXXMACOS
};


enum VMMonSockOpt {
   VMMON_SO_LINK_VM,
   VMMON_SO_GET_KERNEL_HANDLE,
   VMMON_SO_LINK_SOCKET,
   VMMON_SO_LINK_TIMER,
   VMMON_SO_SET_TIMER,
};

enum VMCISockOpt {
   VMCI_SO_VERSION = 0,
   VMCI_SO_USERFD,
   VMCI_SO_CONTEXT = IOCTLCMD_VMCI_INIT_CONTEXT,
   VMCI_SO_PROCESS = IOCTLCMD_VMCI_CREATE_PROCESS,
   VMCI_SO_DATAGRAM_PROCESS = IOCTLCMD_VMCI_CREATE_DATAGRAM_PROCESS,
   VMCI_SO_HYPERCALL = IOCTLCMD_VMCI_HYPERCALL,
   VMCI_SO_READ_CALL = IOCTLCMD_VMCI_READ_CALL,
   VMCI_SO_SHAREDMEM_CREATE = IOCTLCMD_VMCI_SHAREDMEM_CREATE,
   VMCI_SO_SHAREDMEM_ATTACH = IOCTLCMD_VMCI_SHAREDMEM_ATTACH,
   VMCI_SO_SHAREDMEM_QUERY = IOCTLCMD_VMCI_SHAREDMEM_QUERY,
   VMCI_SO_SHAREDMEM_DETACH = IOCTLCMD_VMCI_SHAREDMEM_DETACH,
};

#endif


#endif // ifndef _IOCONTROLS_H_
