#ifndef _MODULECALL_COMPAT_H_
#define _MODULECALL_COMPAT_H_

/* ----------------------------- VMware Workstation 6.0.0 --------------------------- */

typedef enum ModuleCallTypeV6 {
   MODULECALL_V6_NONE = 100,

   MODULECALL_V6_INTR,
   MODULECALL_V6_SEMAWAIT,
   MODULECALL_V6_SEMASIGNAL,
   MODULECALL_V6_SEMAFORCEWAKEUP,
   MODULECALL_V6_IPI,                   // hit thread with IPI
   MODULECALL_V6_SWITCH_TO_PEER,

   /*
    * Return codes for user calls
    */

   MODULECALL_V6_USERRETURN,
   MODULECALL_V6_USERTIMEOUT,

   MODULECALL_V6_GET_RECYCLED_PAGE,
   MODULECALL_V6_RELEASE_ANON_PAGE,
   MODULECALL_V6_IS_ANON_PAGE,

   MODULECALL_V6_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV6;


typedef struct VMCrossPageV6 {
   /*
    * Version checking. Should remain at offset 0
    */
   uint32 version;
   uint32 crosspage_size;
   uint32 _pad[2]; /* Align context switching code on 16 byte boundary. */

   /* Pad to offset 1024 to appease the P4 trace cache.  Smaller than
    * 1024 causes the trace cache to be flushed each time data is
    * written to the context switch.
    *
    * Note for Hosted:
    *    The crosspage code is actually stored in this 
    *    array for now.  Will move it to the end later.
    *    The offsetChecker can't handle unions so we can't 
    *    put WSModule directly here, though it ought to be.
    */
   uint8 wsModule[1024-16];

   /*
    * The interrupt redirection bitmap, must immediately follow
    * monContext.task (ASSERT fail otherwise).
    */
   ContextInfo hostContext;                 /* host32 */
   ContextInfo monContext;                  /* vmm32 */
   Task32      monTask32;
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE]; /* vmm32 */

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;           /* vmm32 */
   uint32 switchL4Off;                      /* vmm64 */
   MPN32  crossMonPageTableMPN;             /* vmm32 */

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;                       /* vmm32 */
   VM_PAE_PDE paeSwitchPDE;                 /* vmm32 */
   VM_L4E lmSwitchPE;                       /* vmm64 */

   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV6 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;       // VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;

   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool runVmm64;

   // host irq relocation values
   int irqRelocateOffset[2];

#if !defined(VMX86_SERVER)
   uint64 ucTimeStamps[UCCOST_MAX];
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   SharedUReg64 _shadowDR[8];
   Assert_MonSrcLoc switchError;

   SystemCallState systemCall;

   /*
    * Adjustment for machines where the hardware TSC does not run
    * constantly (laptops) or is out of sync between different PCPUs.
    * Updated as needed by vmmon.  See VMK_SharedData for the ESX
    * analog, which is updated by the vmkernel.
    */
   RateConv_Params pseudoTSCConv;

   /*
    * PTSC value and target VCPUSet for the next MonitorPoll callback,
    * as last known by the VCPU that this crosspage belongs to.  To
    * find up-to-date information, vmmon must loop through the
    * crosspages of all the VCPUs in this VM and take the one with the
    * latest time (ugh).  When the time arrives, if a target VCPU
    * thread is in the monitor, it wants to receive a hardware
    * interrupt (e.g., an IPI) as soon as possible; if it has called
    * up to userlevel to halt, it wants to wake up as soon as
    * possible.
    */
   VmAbsoluteTS monitorPollFirstTS;
   VCPUSet monitorPollFirstVCS;

   ContextInfo64 hostContext64;             /* host64 */

   /*
    * Location of crosspage in different address spaces.
    */
   LA32   vmm32CrossPageLA;  // where 32-bit mon has crosspage double-mapped
   LA64   vmm64CrossPageLA;  // where 64-bit mon has crosspage double-mapped
   LA64   hostCrossPageLA;   // where host has crosspage mapped
   MA64   crosspageMA;       // crosspage machine address

   uint64 hostCR4;
   uint64 hostDR[8];
   uint16 hostDRSaved;             // Host DR spilled to hostDR[x].
   uint16 hostDRInHW;              // 0: shadowDR in h/w, 1: hostDR in h/w.
   Bool debugRegistersReady;       // DR invariants are active, may touch DRs
   Bool restoreGeneralDetect;      // vmmon must restore DR7_GD

   char _pad4[2];
   ContextInfo64 monContext64;              /* vmm64 */
   Task64        monTask64;                 /* vmm64's task */
   LongModeSwitch lmSwitch;                 /* vmm32+host64 or vmm64+host32 */
} __attribute__((packed)) VMCrossPageV6;

#define CROSSPAGE_VERSION_V6    0x178E


/* ----------------------------- VMware Server 1.0 beta1 ---------------------------- */

typedef struct {
   uint64 cr3;
   uint64 rax;
   uint64 rcx;
   uint64 rdx;
   uint64 rbx;
   uint64 rsi;
   uint64 rdi;
   uint64 rbp;
   uint64 rsp;
   uint64 r8;
   uint64 r9;
   uint64 r10;
   uint64 r11;
   uint64 r12;
   uint64 r13;
   uint64 r14;
   uint64 r15;
   uint32 cs;
   uint32 ds;
   uint32 ss;
   uint32 es;
   uint32 fs;
   uint32 gs;
   uint64 rip;
   uint64 eflags;
} __attribute__ ((packed)) Context64S1B1;

typedef struct {
   DTRWords64 gdtr;
   DTRWords64 idtr;
   Context64S1B1  context;
   uint16     tr;
   uint16     _pad0;
   Task64     task;
} __attribute__ ((packed)) ContextInfo64S1B1;

typedef struct {
   SystemCallRegisters scs32;   // Cannot use 'vmm32' and 'vmm64' as they
   SystemCallRegisters scs64;   // are defined as constants '0' or '1'.
   Bool                sce;
   uint8               _pad0[3];
   uint32              _pad1;
} SystemCallStateS1B1;

typedef enum ModuleCallTypeV55 {
   MODULECALL_V55_NONE = 100,

   MODULECALL_V55_INTR,                  
   MODULECALL_V55_SEMAWAIT,
   MODULECALL_V55_SEMASIGNAL,
   MODULECALL_V55_SEMAFORCEWAKEUP,
   MODULECALL_V55_IPI,                   // hit thread with IPI
   MODULECALL_V55_SWITCH_TO_PEER,

   /*
    * Return codes for user calls
    */

   MODULECALL_V55_USERRETURN,
   MODULECALL_V55_USERTIMEOUT,

   MODULECALL_V55_GET_RECYCLED_PAGE,
   MODULECALL_V55_RELEASE_ANON_PAGE,
   MODULECALL_V55_IS_ANON_PAGE,

   MODULECALL_V55_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV55;

typedef struct VMCrossPageS1B1 {
   /*
    * Version checking. Should remain at offset 0
    */
   uint32 version;
   uint32 _pad[3]; /* Align context switching code on 16 byte boundary. */

   ContextSwitch contextSwitch;

   /* Pad to offset 1024 to appease the P4 trace cache.  Smaller than
    * 1024 causes the trace cache to be flushed each time data is
    * written to the context switch.
    */
   uint8 _trace_pad[1024 - (16 /* alignment */ + sizeof(ContextSwitch))];

   /*
    * The interrupt redirection bitmap, must immediately follow
    * monContext.task (ASSERT fail otherwise).
    */
   ContextInfo hostContext;                 /* host32 */
   ContextInfo monContext;                  /* vmm32 */
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE]; /* vmm32 */

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;           /* vmm32 */
   uint32 switchL4Off;                      /* vmm64 */
   MPN32  crossMonPageTableMPN;             /* vmm32 */

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;                       /* vmm32 */
   VM_PAE_PDE paeSwitchPDE;                 /* vmm32 */
   VM_L4E lmSwitchPE;                       /* vmm64 */

   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV55 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;       // VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;

   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool runVmm64;

   // host irq relocation values
   int irqRelocateOffset[2];

#if !defined(VMX86_SERVER)
   uint64 ucTimeStamps[UCCOST_MAX];
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   SharedUReg64 _shadowDR[8];
   Assert_MonSrcLoc switchError;

   SystemCallStateS1B1 systemCall;

   /*
    * Adjustment for machines where the hardware TSC does not run
    * constantly (laptops).  See VMK_SharedData for the ESX analog.
    * Within the monitor, PTSC_Get() = RDTSC() + tscAdjustment.  To
    * handle the case where the TSC slows or stops when the host OS
    * halts the processor using APM or ACPI, tscAdjustment is updated
    * whenever we go to userlevel.  To handle out of sync MP TSCs,
    * tscAdjustment will need to be per-PCPU and thus will need to be
    * updated whenever this VCPU migrates to a different PCPU.
    */
   VmRelativeTS tscAdjustment;

   /*
    * PTSC value and target VCPUSet for the next MonitorPoll callback,
    * as last known by the VCPU that this crosspage belongs to.  To
    * find up-to-date information, vmmon must loop through the
    * crosspages of all the VCPUs in this VM and take the one with the
    * latest time (ugh).  When the time arrives, if a target VCPU
    * thread is in the monitor, it wants to receive a hardware
    * interrupt (e.g., an IPI) as soon as possible; if it has called
    * up to userlevel to halt, it wants to wake up as soon as
    * possible.
    */
   VmAbsoluteTS monitorPollFirstTS;
   VCPUSet monitorPollFirstVCS;

   ContextInfo64S1B1 hostContext64;             /* host64 */

   /*
    * Location of crosspage in different address spaces.
    */
   LA32   vmm32CrossPageLA;                 /* vmm32 */
   LA64   vmm64CrossPageLA;                 /* vmm64 */
   LA64   hostCrossPageLA;
   MA64   crosspageMA;

   uint64 hostCR4;
   uint64 hostDR[8];
   uint16 hostDRSaved;             // Host DR spilled to hostDR[x].
   uint16 hostDRInHW;              // 0: shadowDR in h/w, 1: hostDR in h/w.
   Bool debugRegistersReady;       // DR invariants are active, may touch DRs
   Bool restoreGeneralDetect;      // vmmon must restore DR7_GD

   char _pad4[2];
   ContextInfo64S1B1 monContext64;              /* vmm64 */
   LongModeSwitch lmSwitch;                 /* vmm32+host64 or vmm64+host32 */
#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V55_INTR; }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageS1B1;


/* ----------------------------- VMware 5.5 ---------------------------- */

typedef struct VMCrossPageV55 {
   /*
    * Version checking. Should remain at offset 0
    */
   uint32 version;
   uint32 _pad[3]; /* Align context switching code on 16 byte boundary. */

   ContextSwitch contextSwitch;

   /* Pad to offset 1024 to appease the P4 trace cache.  Smaller than
    * 1024 causes the trace cache to be flushed each time data is
    * written to the context switch.
    */
   uint8 _trace_pad[1024 - (16 /* alignment */ + sizeof(ContextSwitch))];

   /*
    * The interrupt redirection bitmap, must immediately follow
    * monContext.task (ASSERT fail otherwise).
    */
   ContextInfo hostContext;                 /* host32 */
   ContextInfo monContext;                  /* vmm32 */
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE]; /* vmm32 */

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;           /* vmm32 */
   uint32 switchL4Off;                      /* vmm64 */
   MPN32  crossMonPageTableMPN;             /* vmm32 */

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;                       /* vmm32 */
   VM_PAE_PDE paeSwitchPDE;                 /* vmm32 */
   VM_L4E lmSwitchPE;                       /* vmm64 */

   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV55 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;       // VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;

   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool runVmm64;

   // host irq relocation values
   int irqRelocateOffset[2];

#if !defined(VMX86_SERVER)
   uint64 ucTimeStamps[UCCOST_MAX];
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   SharedUReg64 _shadowDR[8];
   Assert_MonSrcLoc switchError;

   SystemCallStateS1B1 systemCall;

   /*
    * Adjustment for machines where the hardware TSC does not run
    * constantly (laptops).  See VMK_SharedData for the ESX analog.
    * Within the monitor, PTSC_Get() = RDTSC() + tscAdjustment.  To
    * handle the case where the TSC slows or stops when the host OS
    * halts the processor using APM or ACPI, tscAdjustment is updated
    * whenever we go to userlevel.  To handle out of sync MP TSCs,
    * tscAdjustment will need to be per-PCPU and thus will need to be
    * updated whenever this VCPU migrates to a different PCPU.
    */
   VmRelativeTS tscAdjustment;

   /*
    * PTSC value and target VCPUSet for the next MonitorPoll callback,
    * as last known by the VCPU that this crosspage belongs to.  To
    * find up-to-date information, vmmon must loop through the
    * crosspages of all the VCPUs in this VM and take the one with the
    * latest time (ugh).  When the time arrives, if a target VCPU
    * thread is in the monitor, it wants to receive a hardware
    * interrupt (e.g., an IPI) as soon as possible; if it has called
    * up to userlevel to halt, it wants to wake up as soon as
    * possible.
    */
   VmAbsoluteTS monitorPollFirstTS;
   VCPUSet monitorPollFirstVCS;

   ContextInfo64S1B1 hostContext64;             /* host64 */

   /*
    * Location of crosspage in different address spaces.
    */
   LA32   vmm32CrossPageLA;                 /* vmm32 */
   LA64   vmm64CrossPageLA;                 /* vmm64 */
   LA64   hostCrossPageLA;
   MA64   crosspageMA;

   uint64 hostCR4;
   uint64 hostDR[8];
   uint32 hostDRSaved;
   Bool debugRegistersReady;       // DR invariants are active, may touch DRs
   Bool restoreGeneralDetect;      // vmmon must restore DR7_GD

   char _pad4[2];
   ContextInfo64S1B1 monContext64;              /* vmm64 */
   LongModeSwitch lmSwitch;                 /* vmm32+host64 or vmm64+host32 */
#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V55_INTR; }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageV55;




#define MAX_PASSTHROUGH_PORT 1024
#define NUM_PASSTHROUGH_WORDS (MAX_PASSTHROUGH_PORT / 32)

/* ----------------------------- VMware 5 ---------------------------- */

typedef enum ModuleCallTypeV5 {
   MODULECALL_V5_NONE = 100,
   
   MODULECALL_V5_INTR,                  
   MODULECALL_V5_SEMAWAIT,
   MODULECALL_V5_SEMASIGNAL,
   MODULECALL_V5_SEMAFORCEWAKEUP,
   MODULECALL_V5_IPI,                   // hit thread with IPI
   MODULECALL_V5_SWITCH_TO_PEER,

   /*
    * Return codes for user calls
    */

   MODULECALL_V5_USERRETURN,
   MODULECALL_V5_USERTIMEOUT,

   MODULECALL_V5_GET_RECYCLED_PAGE,
   MODULECALL_V5_RELEASE_ANON_PAGE,

   MODULECALL_V5_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV5;

typedef struct {
   DTR64           switchGdtr64;
   uint16          switchDS;
   FarPtr32        jump32Code;       /* &worldswitch_64h_32v_mode_32compat */
   uint16          _pad;
   FarPtr32        jump64Code;       /* &worldswitch_64h_32v_mode_64 */
   uint16          _pad2;
   uint32          _pad3;
   uint64          farPtr;           /* &worldswitch_64h_32v_mon_switch */
   DescriptorUnion switchGdt[4];
   uint64          hostSwitchCR3;
} __attribute__ ((packed)) LongModeSwitchV5;

typedef struct ContextInfo64V5 {
   DTRWords64 gdtr;
   DTRWords64 ldtr64;
   DTRWords64 idtr;
   Context64S1B1  context;
   uint16     tr;
   uint16     _pad0;
   Task64     task;
} __attribute__ ((packed)) ContextInfo64V5;

typedef struct SysenterRegsV5 {
   Selector     sysenterCS;
   uint16       _pad[3];
   uint64       sysenterRIP;
   uint64       sysenterRSP;
} SysenterRegsV5;


typedef struct SysenterStateV5 {
   SysenterRegsV5 hw;             /* currently 'on the hardware' in the vmm */
   Selector       requestedCS;    /* cs requested by monitor */
   Bool           sysenterCacheMSR;
   uint32         _pad;
} SysenterStateV5;


typedef struct {
   uint8  hostXToVmm32[0x0C0];
   uint8  vmmXToHostY [0x100];
   uint8  hostXToVmm64[0x100];
} ContextSwitchV5;

typedef struct VMCrossPageV5 {
   /*
    * Version checking. Should remain at offset 0
    */
   uint32 version;
   uint32 _pad[3]; /* Align context switching code on 16 byte boundary. */

   ContextSwitchV5 contextSwitch;

   /* Pad to offset 1024 to appease the P4 trace cache.  Smaller than
    * 1024 causes the trace cache to be flushed each time data is
    * written to the context switch.
    */
   uint8 _trace_pad[1024 - (16 /* alignment */ + sizeof(ContextSwitchV5))];

   /*
    * The interrupt redirection bitmap, must immediately follow
    * monContext.task (ASSERT fail otherwise).
    */
   ContextInfo hostContext;                 /* host32 */
   ContextInfo monContext;                  /* vmm32 */
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE]; /* vmm32 */

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;           /* vmm32 */
   uint32 switchL4Off;                      /* vmm64 */
   MPN32  crossMonPageTableMPN;             /* vmm32 */

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;                       /* vmm32 */
   VM_PAE_PDE paeSwitchPDE;                 /* vmm32 */
   VM_L4E lmSwitchPE;                       /* vmm64 */

   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV5 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;        // VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;

   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool runVmm64;

   // host irq relocation values
   int irqRelocateOffset[2];

#if defined(VMX86_UCCOST) && !defined(VMX86_SERVER)
   uint64 ucTimeStamps[UCCOST_MAX];
#define UCCOST_CROSSPAGE_SIZE (8 * UCCOST_MAX)
#else
#define UCCOST_CROSSPAGE_SIZE 0
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   uint32 shadowDR[8];
   int32 switchError;

   SysenterStateV5 vmm32Sysenter;
   SysenterStateV5 vmm64Sysenter;

   /*
    * Adjustment for machines where the hardware TSC does not run
    * constantly (laptops).  See VMK_SharedData for the ESX analog.
    * Within the monitor, PTSC_Get() = RDTSC() + tscAdjustment.  To
    * handle the case where the TSC slows or stops when the host OS
    * halts the processor using APM or ACPI, tscAdjustment is updated
    * whenever we go to userlevel.  To handle out of sync MP TSCs,
    * tscAdjustment will need to be per-PCPU and thus will need to be
    * updated whenever this VCPU migrates to a different PCPU.
    */
   VmRelativeTS tscAdjustment;

   /*
    * PTSC value and target VCPUSet for the next MonitorPoll callback,
    * as last known by the VCPU that this crosspage belongs to.  To
    * find up-to-date information, vmmon must loop through the
    * crosspages of all the VCPUs in this VM and take the one with the
    * latest time (ugh).  When the time arrives, if a target VCPU
    * thread is in the monitor, it wants to receive a hardware
    * interrupt (e.g., an IPI) as soon as possible; if it has called
    * up to userlevel to halt, it wants to wake up as soon as
    * possible.
    */
   VmAbsoluteTS monitorPollFirstTS;
   VCPUSet monitorPollFirstVCS;
   

   ContextInfo64V5 hostContext64;             /* host64 */

   /*
    * Location of crosspage in different address spaces.
    */
   LA32   vmm32CrossPageLA;                 /* vmm32 */
   LA64   vmm64CrossPageLA;                 /* vmm64 */
   LA64   hostCrossPageLA;
   MA64   crosspageMA;

   uint64 hostCR4;
   uint64 hostDR[8];
   uint32 hostDRSaved;
   Bool debugRegistersReady;       // DR invariants are active, may touch DRs
   Bool restoreGeneralDetect;      // vmmon must restore DR7_GD

   char _pad4[2];
   ContextInfo64V5 monContext64;              /* vmm64 */
   LongModeSwitchV5 lmSwitch;                 /* vmm32+host64 or vmm64+host32 */
#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V5_INTR; }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageV5;

#define MODULECALL_CROSSPAGE_SIZE_V5 (0x940 + UCCOST_CROSSPAGE_SIZE)

/* ---------------------------- VMware 4.5 --------------------------- */

typedef enum ModuleCallTypeV45 {
   MODULECALL_V45_NONE = 100,
   
   MODULECALL_V45_INTR,                  
   MODULECALL_V45_SEMAWAIT,
   MODULECALL_V45_SEMASIGNAL,
   MODULECALL_V45_SEMAFORCEWAKEUP,
   MODULECALL_V45_IPI,                   // hit thread with IPI

   /*
    * Return codes for user calls
    */

   MODULECALL_V45_USERRETURN,
   MODULECALL_V45_USERTIMEOUT,

   MODULECALL_V45_GET_RECYCLED_PAGE,
   MODULECALL_V45_RELEASE_ANON_PAGE,
   MODULECALL_V45_SET_FASTCLOCKRATE,

   MODULECALL_V45_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV45;

#define CONTEXT_SWITCH_CODE_SIZE_V45 1008

typedef struct {
   DTRWords32  gdtr;
   DTRWords32  ldtr;
   DTRWords32  idtr;
   uint16 tr;
   uint8 jump[6];
   Task task;   
} __attribute__ ((packed)) ContextInfoV45;

typedef struct {
   uint64 cr3;

   // general registers
   uint64 rax;
   uint64 rcx;
   uint64 rdx;
   uint64 rbx;
   uint64 rsi;
   uint64 rdi;
   uint64 rbp;
   uint64 rsp;

   // segment registers

   uint16 cs;
   uint16 ds;
   uint16 ss;
   uint16 es;
   uint16 fs;
   uint16 gs;

   // rip
   uint64 rip;

} __attribute__ ((packed)) Context64V45;

typedef struct {
   DTRWords64  gdtr64;
   DTRWords64  ldtr64;
   DTRWords64  idtr64;
   uint16 tr;
   DTR64 switchGdtr64;
   FarPtr32 jump32Code;
   uint16   pad1;
   FarPtr32 jump64Code;
   uint16   pad2;
   uint16   switchDS;
   uint64   farPtr;
   Context64V45 context;
} __attribute__ ((packed)) ContextInfo64V45;

typedef struct VMCrossPageV45 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   uint32 _pad[3]; /* Align the code on a 16 byte boundary */
   struct {
   	uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V45];
   } contextSwitch;

   /*
    * The interrupt redirection bitmap, must immediately follow 
    * monContext.task (ASSERT fail otherwise).
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE];

   VM_PTE hostSharedPTE;

   uint32 hostContextVA;

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;

   MPN32 crossMonPageTableMPN;

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;
   VM_PAE_PDE paeSwitchPDE; 

   /*
    * In x86,
    * Linear address of the crosspage. It must be the same in both the kernel
    * and the monitor address spaces, because the context switch code must
    * continue to run after it flips CR3 --hpreg
    *
    * In x86 VM on x86-64 host,
    * In oder to make the transition to legacy mode from long mode,
    * we must first turn paging off on the processor. For this we need to
    * be executing code on an identity mapped page (the crosspage). Here
    * is where we keep the phisical address of the crosspage. We will use
    * this address to create an identity mapping (a translation for which
    * PA == LA)
    *
    */
   LA32 crosspageLinearAddr;
   
   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV45 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;	// VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;
   
   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool _pad1[1];

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2];

#if defined(VMX86_UCCOST) && !defined(VMX86_SERVER)
   uint64 ucTimeStamps[UCCOST_MAX];
#define UCCOST_CROSSPAGE_SIZE (8 * UCCOST_MAX)
#else
#define UCCOST_CROSSPAGE_SIZE 0
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   uint32 shadowDR[8];
   int32 switchError;

   /*
    * monUsesSysenter is TURE iff the virtual MSR_SYSENTER_CS is
    * nonzero and the monitor uses the sysenter/sysexit MSRs
    * for fast system calls.
    */
   Bool monUsesSysenter;

   Bool _pad2[3];

   /*
    * Adjustment for machines where the hardware TSC does not run
    * constantly (laptops).  See VMK_SharedData for the ESX analog.
    * Within the monitor, PTSC_Get() = RDTSC() + tscAdjustment.  To
    * handle the case where the TSC slows or stops when the host OS
    * halts the processor using APM or ACPI, tscAdjustment is updated
    * whenever we go to userlevel.  To handle out of sync MP TSCs,
    * tscAdjustment will need to be per-PCPU and thus will need to be
    * updated whenever this VCPU migrates to a different PCPU.
    */
   VmRelativeTS tscAdjustment;

   /*
    * PTSC value and target VCPUSet for the next MonitorPoll callback,
    * as last known by the VCPU that this crosspage belongs to.  To
    * find up-to-date information, vmmon must loop through the
    * crosspages of all the VCPUs in this VM and take the one with the
    * latest time (ugh).  When the time arrives, if a target VCPU
    * thread is in the monitor, it wants to receive a hardware
    * interrupt (e.g., an IPI) as soon as possible; if it has called
    * up to userlevel to halt, it wants to wake up as soon as
    * possible.
    */
   VmAbsoluteTS monitorPollFirstTS;
   VCPUSet monitorPollFirstVCS;
   
   /*
    * Fields used by x86_64.
    *
    * Since we need to share the same VMX and VMM binary for
    * x86 and x86_64, we don't use ifdef to define x86_64
    * specific fields. If we run out of crosspage space,
    * we need to define these structures using unions.
    */
   DescriptorUnion switchGdt[4];
   ContextInfo64V45 hostContext64;         // replace hostContext
   Bool _pad3[2];
   uint64 hostContext64VA;              // replace hostContextVA
   uint64 hostVA64;                     
   uint64 hostSwitchCR3;

   uint64 switchHostCR4;
   uint64 hostDR[8];
   uint32 hostDRSaved;

#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V45_INTR; }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageV45;

#define MODULECALL_CROSSPAGE_SIZE_V45 (0x6f4 + UCCOST_CROSSPAGE_SIZE)
#define MODULECALL_CROSSPAGE_LAOFFSET_V45 0x54c

/* ---------------------------- VMware 4 --------------------------- */

typedef enum ModuleCallTypeV4 {
   MODULECALL_V4_NONE = 100,
   
   MODULECALL_V4_INTR,                  
   MODULECALL_V4_LOCKPAGE,                  
   MODULECALL_V4_UNLOCKPAGE,                  
   MODULECALL_V4_ISMPNLOCKED, 
   MODULECALL_V4_SEMAWAIT,
   MODULECALL_V4_SEMASIGNAL,
   MODULECALL_V4_SEMAFORCEWAKEUP,
   MODULECALL_V4_IPI,                   // hit thread with IPI

   /*
    * Return codes for user calls
    */

   MODULECALL_V4_USERRETURN,
   MODULECALL_V4_USERTIMEOUT,

   MODULECALL_V4_GET_RECYCLED_PAGE,
   MODULECALL_V4_RELEASE_ANON_PAGE,

   MODULECALL_V4_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV4;

#define CONTEXT_SWITCH_CODE_SIZE_V4 1008

typedef struct VMCrossPageV4 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   uint32 _pad[3]; /* Align the code on a 16 byte boundary */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V4];
   } contextSwitch;

   /*
    * iobitMap must immediately follow the interrupt redirection
    * bitmap, which must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE];
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   /* Kernel virtual address of the crosspage --hpreg */
   VA32 hostVA;

   VM_PTE hostSharedPTE;

   uint32 hostContextVA;

   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;

   MPN32 crossMonPageTableMPN;

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;
   VM_PAE_PDE paeSwitchPDE; 

   /*
    * Linear address of the crosspage. It must be the same in both the kernel
    * and the monitor address spaces, because the context switch code must
    * continue to run after it flips CR3 --hpreg
    */
   LA32 crosspageLinearAddr;

   /*
    * The monitor may requests up to two actions when
    * returning to the host.  The moduleCallType field and
    * args encode a request for some action in the driver.
    * The userCallType field (together with the RPC block)
    * encodes a user call request.  The two requests are
    * independent.  The user call is executed first, with
    * the exception of MODULECALL_INTR which has a special
    * effect.
    */
   ModuleCallTypeV4 moduleCallType;
   uint32 args[4];
   uint32 retval;

   int userCallType;
   volatile int userCallRequest;	// VCPU/VMX synchronization
   Bool userCallCross;
   Bool userCallRestart;
   
   /*
    * TRUE if moduleCall was interrupted by signal. Only
    * vmmon uses this field to remember that it should
    * restart RunVM call, nobody else should look at it.
    */
   Bool moduleCallInterrupted;
   Bool _pad1[1];

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2];

#ifdef VMX86_UCCOST
    uint64 ucTimeStamps[UCCOST_MAX];
#define UCCOST_CROSSPAGE_SIZE (8 * UCCOST_MAX)
#else
#define UCCOST_CROSSPAGE_SIZE 0
#endif

   /*
    * The values in the shadow debug registers must match
    * those in the hardware debug register immediately after
    * a task switch in either direction.  They are used to
    * minimize moves to and from the debug registers.
    */
   uint32 shadowDR[8];
   uint32 switchPanic;
   Bool monUsesSysenter;

#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V4_INTR; }
   Bool isDRModified(int reg, uint32 savedDR) { return (savedDR != shadowDR[reg]); }
   void setDR(int reg, uint32 dr) { shadowDR[reg] = dr; }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageV4;

#define MODULECALL_CROSSPAGE_SIZE_V4 (0x629 + UCCOST_CROSSPAGE_SIZE)
#define MODULECALL_CROSSPAGE_LAOFFSET_V4 0x5d0

/* ---------------------------- GSX 2.5 ---------------------------- */

typedef enum ModuleCallTypeV3 {
   MODULECALL_V3_NONE = 100,            //
   
   /*
    * All I/O interrupts go under MODULECALL_INTR, special-cased
    * in task switching code
    */
   MODULECALL_V3_INTR,                  
   MODULECALL_V3_LOCKPAGE,                  
   MODULECALL_V3_UNLOCKPAGE,                  

   MODULECALL_V3_ISMPNLOCKED, 
   /*
    * calls to user mode
    */
   MODULECALL_V3_USERRETURN,            // return code for USER RPC
   MODULECALL_V3_LAST                   // Number of entries. Must be the last one
} ModuleCallTypeV3;

#define CONTEXT_SWITCH_CODE_SIZE_GSX25  1008

typedef struct VMCrossPageGSX25 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   uint32 _pad[3]; /* Align the code on a 16 byte boundary */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_GSX25];
   } contextSwitch;

   /*
    * iobitMap must immediately follow the interrupt redirection
    * bitmap, which must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE];
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   /* Kernel virtual address of the crosspage --hpreg */
   VA32 hostVA;

   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;
  
   MPN32 crossMonPageTableMPN; 

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;
   VM_PAE_PDE paeSwitchPDE; 

   LA32 monitorLinearAddr;      // base location of monitor in linear space
  
   /*
    * Linear address of the crosspage. It must be the same in both the kernel
    * and the monitor address spaces, because the context switch code must
    * continue to run after it flips CR3 --hpreg
    */
   LA32 crosspageLinearAddr;

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

   Bool monUsesSysenter;

#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageGSX25;

#define MODULECALL_CROSSPAGE_SIZE_GSX25     0x5fd
#define MODULECALL_CROSSPAGE_LAOFFSET_GSX25 0x5d4

/* ---------------------------- GSX 2.0 ---------------------------- */

#define CONTEXT_SWITCH_CODE_SIZE_GSX2  1008

typedef struct VMCrossPageGSX2 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   uint32 _pad[3]; /* Align the code on a 16 byte boundary */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_GSX2];
   } contextSwitch;

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   /* Kernel virtual address of the crosspage --hpreg */
   VA32 hostVA;

   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   /*
    * Most significant bits (10 if the host is not in PAE mode, 11 if the host
    * is in PAE mode) of the linear address of the crosspage, which allow to
    * locate the page directory entry associated to the crosspage --hpreg
    */
   uint32 switchLinearRangeIndex;
  
   MPN32 crossMonPageTableMPN; 

   /* PDE that points to switch page table */
   VM_PDE  switchPDE;
   VM_PAE_PDE paeSwitchPDE; 

   LA32 monitorLinearAddr;      // base location of monitor in linear space
  
   /*
    * Linear address of the crosspage. It must be the same in both the kernel
    * and the monitor address spaces, because the context switch code must
    * continue to run after it flips CR3 --hpreg
    */
   LA32 crosspageLinearAddr;

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

   Bool monUsesSysenter;

#ifdef __cplusplus
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 6; }
#endif
} __attribute__ ((packed)) VMCrossPageGSX2;

#define MODULECALL_CROSSPAGE_SIZE_GSX2     0x5dd
#define MODULECALL_CROSSPAGE_LAOFFSET_GSX2 0x5b4

/* ---------------------------- WS 3.2.1 ---------------------------- */

#define CONTEXT_SWITCH_CODE_SIZE_V321	1008

typedef struct VMCrossPageV321 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
   uint32 _pad[3];

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V321];
   } contextSwitch;

   /*
    * iobitMap must immediately follow the interrupt redirection
    * bitmap, which must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint8       interruptRedirectionBitMap[INTERRUPT_REDIRECTION_BITMAP_SIZE];
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   VA32 hostVA;
   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     // index into the page directory of switchPDE
   VM_PDE switchPDE;          // PDE that points to switch page table


   LA32 monitorLinearAddr;      // base location of monitor in linear space
   LA32 crosspageLinearAddr;    // location of crosspage page in both linear spaces

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

#ifdef __cplusplus
   static const int monUsesSysenter = 0;
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 4; }
#endif
} __attribute__ ((packed)) VMCrossPageV321;

#define MODULECALL_CROSSPAGE_SIZE_V321     0x5f0
#define MODULECALL_CROSSPAGE_LAOFFSET_V321 0x5c8

/* ---------------------------- WS 3.2 ---------------------------- */

#define CONTEXT_SWITCH_CODE_SIZE_V32	1008

typedef struct VMCrossPageV32 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
   uint32 _pad[3];

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V32];
   } contextSwitch;

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   VA32 hostVA;
   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     // index into the page directory of switchPDE
   VM_PDE switchPDE;          // PDE that points to switch page table


   LA32 monitorLinearAddr;      // base location of monitor in linear space
   LA32 crosspageLinearAddr;    // location of crosspage page in both linear spaces

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

#ifdef __cplusplus
   static const int monUsesSysenter = 0;
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 4; }
#endif
} __attribute__ ((packed)) VMCrossPageV32;

#define MODULECALL_CROSSPAGE_SIZE_V32     0x5d0
#define MODULECALL_CROSSPAGE_LAOFFSET_V32 0x5a8

/* ---------------------------- WS 3.0 ---------------------------- */

#define CONTEXT_SWITCH_CODE_SIZE_V3	192

typedef struct VMCrossPageV3 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V3];
   } contextSwitch;

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   VA32 hostVA;
   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     // index into the page directory of switchPDE
   VM_PDE switchPDE;          // PDE that points to switch page table


   LA32 monitorLinearAddr;      // base location of monitor in linear space
   LA32 crosspageLinearAddr;    // location of crosspage page in both linear spaces

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

#ifdef __cplusplus
   static const int monUsesSysenter = 0;
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 4; }
#endif
} __attribute__ ((packed)) VMCrossPageV3;

typedef struct VMCrossPageV2 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V3];
   } contextSwitch;

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   VA32 hostVA;
   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     // index into the page directory of switchPDE
   VM_PDE switchPDE;          // PDE that points to switch page table


   LA32 monitorLinearAddr;      // base location of monitor in linear space
   LA32 crosspageLinearAddr;    // location of crosspage page in both linear spaces

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

#ifdef __cplusplus
   static const int monUsesSysenter = 0;
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 2; }
#endif
} __attribute__ ((packed)) VMCrossPageV2;

typedef struct VMCrossPageGSX1 {
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   struct {
      uint8 hostXToVmm32[CONTEXT_SWITCH_CODE_SIZE_V3];
   } contextSwitch;

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfoV45 hostContext;
   ContextInfoV45 monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   VA32 hostVA;
   VM_PTE hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     // index into the page directory of switchPDE
   VM_PDE switchPDE;          // PDE that points to switch page table


   LA32 monitorLinearAddr;      // base location of monitor in linear space
   LA32 crosspageLinearAddr;    // location of crosspage page in both linear spaces

   //
   // ModuleCall parameters
   //
   ModuleCallTypeV3 moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA32 userSharedAreaAddr;

   // host irq relocation values
   int irqRelocateOffset[2]; 

#ifdef __cplusplus
   static const int monUsesSysenter = 0;
   Bool isMCIntr(void) { return moduleCallType == MODULECALL_V3_INTR; }
   static Bool isDRModified(int reg, uint32 savedDR) { return TRUE; }
   static void setDR(int reg, uint32 dr) { }
   static unsigned int crosspageStart(void) { return 2; }
#endif
} __attribute__ ((packed)) VMCrossPageGSX1;

#define MODULECALL_CROSSPAGE_SIZE_V3     0x294
#define MODULECALL_CROSSPAGE_LAOFFSET_V3 0x26c

#endif /* _MODULECALL_COMPAT_H_ */
