/* **********************************************************
 * Copyright 2004-2005 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 * **********************************************************/

#ifndef _X86VT_H_
#define _X86VT_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMX
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_VMMON
#include "includeCheck.h"

#include "x86msr.h"
#include "vm_asm.h"

/* VMX related MSRs */
#define MSR_VMX_BASIC          0x00000480
#define MSR_VMX_PINBASED_CTLS  0x00000481
#define MSR_VMX_PROCBASED_CTLS 0x00000482
#define MSR_VMX_EXIT_CTLS      0x00000483
#define MSR_VMX_ENTRY_CTLS     0x00000484
#define MSR_VMX_MISC           0x00000485
#define MSR_VMX_CR0_FIXED0     0x00000486
#define MSR_VMX_CR0_FIXED1     0x00000487
#define MSR_VMX_CR4_FIXED0     0x00000488
#define MSR_VMX_CR4_FIXED1     0x00000489
#define MSR_VMX_VMCS_ENUM      0x0000048a

#define MSR_VMX_BASIC_VMCS_SIZE_SHIFT  32
#define MSR_VMX_BASIC_VMCS_SIZE_MASK  0x1fff
#define MSR_VMX_BASIC_32BITPA_SHIFT    48
#define MSR_VMX_BASIC_DUALVMM_SHIFT    49
#define MSR_VMX_BASIC_MEMTYPE_SHIFT    50
#define MSR_VMX_BASIC_MEMTYPE_MASK    0xf

/*
 * Structure of VMCS Component Encoding (Table 20-15)
 */
#define VT_ENCODING_ACCESS_HIGH        0x00000001
#define VT_ENCODING_INDEX_MASK         0x000003fe
#define VT_ENCODING_INDEX_SHIFT                 1
#define VT_ENCODING_TYPE_MASK          0x00000c00
#define VT_ENCODING_TYPE_CTL           0x00000000
#define VT_ENCODING_TYPE_RODATA        0x00000400
#define VT_ENCODING_TYPE_GUEST         0x00000800
#define VT_ENCODING_TYPE_HOST          0x00000c00
#define VT_ENCODING_SIZE_MASK          0x00006000
#define VT_ENCODING_SIZE_16BIT         0x00000000
#define VT_ENCODING_SIZE_64BIT         0x00002000
#define VT_ENCODING_SIZE_32BIT         0x00004000
#define VT_ENCODING_SIZE_NATURAL       0x00006000
#define VT_ENCODING_RSVD               0xffff9000

/* 
 * VMCS encodings; volume 3B Appendix H. These are the values passed to
 * VMWrite and VMRead.
 */

/* 16-bit guest state: table H-1 */
#define VT_VMCS_ES                     0x00000800
#define VT_VMCS_CS                     0x00000802
#define VT_VMCS_SS                     0x00000804
#define VT_VMCS_DS                     0x00000806
#define VT_VMCS_FS                     0x00000808
#define VT_VMCS_GS                     0x0000080A
#define VT_VMCS_LDTR                   0x0000080C
#define VT_VMCS_TR                     0x0000080E

/* 16-bit host state: table H-2 */
#define VT_VMCS_HOST_ES                0x00000C00
#define VT_VMCS_HOST_CS                0x00000C02
#define VT_VMCS_HOST_SS                0x00000C04
#define VT_VMCS_HOST_DS                0x00000C06
#define VT_VMCS_HOST_FS                0x00000C08
#define VT_VMCS_HOST_GS                0x00000C0A
#define VT_VMCS_HOST_TR                0x00000C0C

/* 64-bit control fields: table H-3 */
#define VT_VMCS_IOBITMAPA              0x00002000
#define VT_VMCS_IOBITMAPB              0x00002002
#define VT_VMCS_MSRBITMAP              0x00002004
#define VT_VMCS_VMEXIT_MSR_STORE_ADDR  0x00002006
#define VT_VMCS_VMEXIT_MSR_LOAD_ADDR   0x00002008
#define VT_VMCS_VMENTRY_MSR_LOAD_ADDR  0x0000200A
#define VT_VMCS_EXECUTIVE_VMCS_PTR     0x0000200C
#define VT_VMCS_TSC_OFF                0x00002010
#define VT_VMCS_VIRT_APIC_ADDR         0x00002012

/* 64-bit guest state: table H-4 */
#define VT_VMCS_LINK_PTR               0x00002800
#define VT_VMCS_DEBUGCTL               0x00002802

/* 32-bit control fields: table H-5 */
#define VT_VMCS_PIN_VMEXEC_CTL         0x00004000
#define VT_VMCS_CPU_VMEXEC_CTL         0x00004002
#define VT_VMCS_XCP_BITMAP             0x00004004
#define VT_VMCS_PF_ERR_MASK            0x00004006
#define VT_VMCS_PF_ERR_MATCH           0x00004008
#define VT_VMCS_CR3_TARG_COUNT         0x0000400A
#define VT_VMCS_VMEXIT_CTL             0x0000400C
#define VT_VMCS_VMEXIT_MSR_STORE_COUNT 0x0000400E
#define VT_VMCS_VMEXIT_MSR_LOAD_COUNT  0x00004010
#define VT_VMCS_VMENTRY_CTL            0x00004012
#define VT_VMCS_VMENTRY_MSR_LOAD_COUNT 0x00004014
#define VT_VMCS_VMENTRY_INTR_INFO      0x00004016
#define VT_VMCS_VMENTRY_XCP_ERR        0x00004018
#define VT_VMCS_VMENTRY_INSTR_LEN      0x0000401A
#define VT_VMCS_TPR_THRESHOLD          0x0000401C

/* 32-bit read-only data fields: table H-6 */
#define VT_VMCS_VMINSTR_ERR            0x00004400
#define VT_VMCS_EXIT_REASON            0x00004402
#define VT_VMCS_EXIT_INTR_INFO         0x00004404
#define VT_VMCS_EXIT_INTR_ERR          0x00004406
#define VT_VMCS_IDTVEC_INFO            0x00004408
#define VT_VMCS_IDTVEC_ERR             0x0000440A
#define VT_VMCS_INSTRLEN               0x0000440C
#define VT_VMCS_VMX_INSTR_INFO         0x0000440E

/* 32-bit guest state: table H-7 */
#define VT_VMCS_ES_LIMIT               0x00004800
#define VT_VMCS_CS_LIMIT               0x00004802
#define VT_VMCS_SS_LIMIT               0x00004804
#define VT_VMCS_DS_LIMIT               0x00004806
#define VT_VMCS_FS_LIMIT               0x00004808
#define VT_VMCS_GS_LIMIT               0x0000480A
#define VT_VMCS_LDTR_LIMIT             0x0000480C
#define VT_VMCS_TR_LIMIT               0x0000480E
#define VT_VMCS_GDTR_LIMIT             0x00004810
#define VT_VMCS_IDTR_LIMIT             0x00004812
#define VT_VMCS_ES_AR                  0x00004814
#define VT_VMCS_CS_AR                  0x00004816
#define VT_VMCS_SS_AR                  0x00004818
#define VT_VMCS_DS_AR                  0x0000481A
#define VT_VMCS_FS_AR                  0x0000481C
#define VT_VMCS_GS_AR                  0x0000481E
#define VT_VMCS_LDTR_AR                0x00004820
#define VT_VMCS_TR_AR                  0x00004822
#define VT_VMCS_INTRABILITY            0x00004824
#define VT_VMCS_ACTSTATE               0x00004826
#define VT_VMCS_SMBASE                 0x00004828
#define VT_VMCS_SYSENTER_CS            0x0000482A

/* 32-bit host state: table H-8 */
#define VT_VMCS_HOST_SYSENTER_CS       0x00004C00

/* natural-width control fields: table H-9 */
#define VT_VMCS_CR0_GHMASK             0x00006000
#define VT_VMCS_CR4_GHMASK             0x00006002
#define VT_VMCS_CR0_SHADOW             0x00006004
#define VT_VMCS_CR4_SHADOW             0x00006006
#define VT_VMCS_CR3_TARGVAL0           0x00006008
#define VT_VMCS_CR3_TARGVAL1           0x0000600A
#define VT_VMCS_CR3_TARGVAL2           0x0000600C
#define VT_VMCS_CR3_TARGVAL3           0x0000600E

/* natural-width RO data fields: table H-10 */
#define VT_VMCS_EXIT_QUAL              0x00006400
#define VT_VMCS_IO_ECX                 0x00006402
#define VT_VMCS_IO_ESI                 0x00006404
#define VT_VMCS_IO_EDI                 0x00006406
#define VT_VMCS_IO_EIP                 0x00006408
#define VT_VMCS_LINEAR_ADDR            0x0000640A

/* natural-width guest state: table H-11 */
#define VT_VMCS_CR0                    0x00006800
#define VT_VMCS_CR3                    0x00006802
#define VT_VMCS_CR4                    0x00006804
#define VT_VMCS_ES_BASE                0x00006806
#define VT_VMCS_CS_BASE                0x00006808
#define VT_VMCS_SS_BASE                0x0000680A
#define VT_VMCS_DS_BASE                0x0000680C
#define VT_VMCS_FS_BASE                0x0000680E
#define VT_VMCS_GS_BASE                0x00006810
#define VT_VMCS_LDTR_BASE              0x00006812
#define VT_VMCS_TR_BASE                0x00006814
#define VT_VMCS_GDTR_BASE              0x00006816
#define VT_VMCS_IDTR_BASE              0x00006818
#define VT_VMCS_DR7                    0x0000681A
#define VT_VMCS_ESP                    0x0000681C
#define VT_VMCS_EIP                    0x0000681E
#define VT_VMCS_EFLAGS                 0x00006820
#define VT_VMCS_PENDDBG                0x00006822
#define VT_VMCS_SYSENTER_ESP           0x00006824
#define VT_VMCS_SYSENTER_EIP           0x00006826

/* natural-width host state: table H-12 */
#define VT_VMCS_HOST_CR0               0x00006C00
#define VT_VMCS_HOST_CR3               0x00006C02
#define VT_VMCS_HOST_CR4               0x00006C04
#define VT_VMCS_HOST_FSBASE            0x00006C06
#define VT_VMCS_HOST_GSBASE            0x00006C08
#define VT_VMCS_HOST_TRBASE            0x00006C0A
#define VT_VMCS_HOST_GDTRBASE          0x00006C0C
#define VT_VMCS_HOST_IDTRBASE          0x00006C0E
#define VT_VMCS_HOST_SYSENTER_ESP      0x00006C10
#define VT_VMCS_HOST_SYSENTER_EIP      0x00006C12
#define VT_VMCS_HOST_ESP               0x00006C14
#define VT_VMCS_HOST_EIP               0x00006C16

/*
 * Sizes of referenced fields
 */
#define VT_VMCS_IO_BITMAP_SIZE    (2 * PAGE_SIZE)
#define VT_VMCS_MSR_BITMAP_SIZE   (1 * PAGE_SIZE)

/*
 * Bits for exec control
 */
#define VT_VMCS_PIN_VMEXEC_CTL_HOSTINTRMASK 0x1
#define VT_VMCS_PIN_VMEXEC_CTL_NMI          0x8

#define VT_VMCS_CPU_VMEXEC_CTL_VINTRPEND   0x00000004
#define VT_VMCS_CPU_VMEXEC_CTL_TSCOFF      0x00000008
#define VT_VMCS_CPU_VMEXEC_CTL_HLT         0x00000080
#define VT_VMCS_CPU_VMEXEC_CTL_INVLPG      0x00000200
#define VT_VMCS_CPU_VMEXEC_CTL_MWAIT       0x00000400
#define VT_VMCS_CPU_VMEXEC_CTL_RDPMC       0x00000800
#define VT_VMCS_CPU_VMEXEC_CTL_RDTSC       0x00001000
#define VT_VMCS_CPU_VMEXEC_CTL_LDCR8       0x00080000
#define VT_VMCS_CPU_VMEXEC_CTL_STCR8       0x00100000
#define VT_VMCS_CPU_VMEXEC_CTL_USECR8SHAD  0x00200000
#define VT_VMCS_CPU_VMEXEC_CTL_MOVDR       0x00800000
#define VT_VMCS_CPU_VMEXEC_CTL_IO          0x01000000
#define VT_VMCS_CPU_VMEXEC_CTL_IOBITMAP    0x02000000
#define VT_VMCS_CPU_VMEXEC_CTL_MSRBITMAP   0x10000000
#define VT_VMCS_CPU_VMEXEC_CTL_MONITOR     0x20000000
#define VT_VMCS_CPU_VMEXEC_CTL_PAUSE       0x40000000

/*
 * Bits for entry control.
 */
#define VT_VMCS_VMENTRY_CTL_LONGMODE      0x0200
#define VT_VMCS_VMENTRY_CTL_SMM           0x0400
#define VT_VMCS_VMENTRY_CTL_SMMTEARDOWN   0x0800

/*
 * Bits for exit control.
 */
#define VT_VMCS_VMEXIT_CTL_LONGMODE       0x0200
#define VT_VMCS_VMEXIT_CTL_INTRACK        0x8000

/*
 * The AR format is mostly the same as the SMM segment format; i.e.,
 * a descriptor shifted by a byte. However, there is an extra bit in
 * the high-order word which indicates an "unusable" selector.  A NULL
 * selector is generally unusable, as are a few other corner cases.  
 * For details, see the public VT specification, section 2.4.1).
 */
#define VT_VMCS_AR_ACCESSED   DT_ACCESS_RIGHTS_ACCESSED
#define VT_VMCS_AR_WRITE      DT_ACCESS_RIGHTS_WRITE
#define VT_VMCS_AR_READ       DT_ACCESS_RIGHTS_READ
#define VT_VMCS_AR_CONFORM    DT_ACCESS_RIGHTS_CONFORM
#define VT_VMCS_AR_CODE       DT_ACCESS_RIGHTS_CODE
#define VT_VMCS_AR_TYPE       DT_ACCESS_RIGHTS_TYPE
#define VT_VMCS_AR_S          DT_ACCESS_RIGHTS_S
#define VT_VMCS_AR_DPL        DT_ACCESS_RIGHTS_DPL
#define VT_VMCS_AR_PRES       DT_ACCESS_RIGHTS_PRES
#define VT_VMCS_AR_LONGMODE   DT_ACCESS_RIGHTS_LONGMODE
#define VT_VMCS_AR_DB         DT_ACCESS_RIGHTS_DB
#define VT_VMCS_AR_GRAN       DT_ACCESS_RIGHTS_GRAN
#define VT_VMCS_AR_UNUSABLE   0x00010000
#define VT_VMCS_AR_RESERVED   0xfffe0f00

#define VT_VMCS_AR_TYPE_SHIFT      DT_ACCESS_RIGHTS_TYPE_SHIFT
#define VT_VMCS_AR_S_SHIFT         DT_ACCESS_RIGHTS_S_SHIFT
#define VT_VMCS_AR_DPL_SHIFT       DT_ACCESS_RIGHTS_DPL_SHIFT
#define VT_VMCS_AR_PRES_SHIFT      DT_ACCESS_RIGHTS_PRES_SHIFT
#define VT_VMCS_AR_LONGMODE_SHIFT  DT_ACCESS_RIGHTS_LONGMODE_SHIFT
#define VT_VMCS_AR_DB_SHIFT        DT_ACCESS_RIGHTS_DB_SHIFT
#define VT_VMCS_AR_GRAN_SHIFT      DT_ACCESS_RIGHTS_GRAN_SHIFT

/* Exception error must-be-zero bits for VMEntry */
#define VT_XCP_ERR_MBZ        0xffff8000

/* Exit reasons: table I-1 */
#define VT_EXITREASON_SOFTINT            0
#define VT_EXITREASON_EXTINT             1
#define VT_EXITREASON_TRIPLEFAULT        2
#define VT_EXITREASON_INIT               3
#define VT_EXITREASON_SIPI               4
#define VT_EXITREASON_IOSMI              5
#define VT_EXITREASON_OTHERSMI           6
#define VT_EXITREASON_PENDINTR           7
#define VT_EXITREASON_TS                 9
#define VT_EXITREASON_CPUID             10
#define VT_EXITREASON_HLT               12
#define VT_EXITREASON_INVD              13
#define VT_EXITREASON_INVLPG            14
#define VT_EXITREASON_RDPMC             15
#define VT_EXITREASON_RDTSC             16
#define VT_EXITREASON_RSM               17
#define VT_EXITREASON_VMCALL            18
#define VT_EXITREASON_VMCLEAR           19
#define VT_EXITREASON_VMLAUNCH          20
#define VT_EXITREASON_VMPTRLD           21
#define VT_EXITREASON_VMPTRST           22
#define VT_EXITREASON_VMREAD            23
#define VT_EXITREASON_VMRESUME          24
#define VT_EXITREASON_VMWRITE           25
#define VT_EXITREASON_VMXOFF            26
#define VT_EXITREASON_VMXON             27
#define VT_EXITREASON_CR                28
#define VT_EXITREASON_DR                29
#define VT_EXITREASON_IO                30
#define VT_EXITREASON_MSRREAD           31
#define VT_EXITREASON_MSRWRITE          32
#define VT_EXITREASON_VMENTRYFAIL_GUEST 33
#define VT_EXITREASON_VMENTRYFAIL_MSR   34
#define VT_EXITREASON_MWAIT             36
#define VT_EXITREASON_MONITOR           39
#define VT_EXITREASON_PAUSE             40
#define VT_EXITREASON_VMENTRYFAIL_MC    41
#define VT_EXITREASON_TPR               43

#define VT_NUM_EXIT_REASONS (VT_EXITREASON_TPR + 1)

/* Instruction error codes: table J-1 */
#define VT_ERROR_VMCALL_VMX_ROOT            1
#define VT_ERROR_VMCLEAR_INVALID_PA         2
#define VT_ERROR_VMCLEAR_ROOT_PTR           3
#define VT_ERROR_VMLAUNCH_NOT_CLEAR         4
#define VT_ERROR_VMRESUME_NOT_LAUNCHED      5
#define VT_ERROR_VMRESUME_CORRUPT_VMCS      6
#define VT_ERROR_VMENTRY_INVALID_CTL        7
#define VT_ERROR_VMENTRY_INVALID_HOST       8
#define VT_ERROR_VMPTRLD_INVALID_PA         9
#define VT_ERROR_VMPTRLD_ROOT_PTR          10
#define VT_ERROR_VMPTRLD_BAD_REVISION      11
#define VT_ERROR_VMACCESS_UNSUPPORTED      12
#define VT_ERROR_VMWRITE_READ_ONLY         13
#define VT_ERROR_VMXON_VMX_ROOT            15
#define VT_ERROR_VMENTRY_INVALID_EXEC      16
#define VT_ERROR_VMENTRY_EXEC_NOT_LAUNCHED 17
#define VT_ERROR_VMENTRY_EXEC_NOT_ROOT     18
#define VT_ERROR_VMCALL_NOT_CLEAR          19
#define VT_ERROR_VMCALL_INVALID_CTL        20
#define VT_ERROR_VMCALL_WRONG_MSEG         22
#define VT_ERROR_VMXOFF_DUALVMM            23
#define VT_ERROR_VMCALL_INVALID_SMM        24
#define VT_ERROR_VMENTRY_INVALID_EXEC_CTL  25
#define VT_ERROR_VMENTRY_MOVSS_SHADOW      26

/* interrupt information fields. Low order 8 bits are vector. */
#define VT_INTRINFO_TYPE_SHIFT      8
#define VT_INTRINFO_TYPE_MASK       (7 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_EXTINT     (0 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_RSVD0      (1 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_NMI        (2 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_EXC        (3 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_INTN       (4 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_PRIVTRAP   (5 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_UNPRIVTRAP (6 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_TYPE_RSVD1      (7 << VT_INTRINFO_TYPE_SHIFT)
#define VT_INTRINFO_ERRORCODE       (1 << 11)
#define VT_INTRINFO_NMIUNMASK       (1 << 12)
#define VT_INTRINFO_VALID           (1 << 31)
#define VT_INTRINFO_VECTOR_MASK     ((1 << VT_INTRINFO_TYPE_SHIFT) - 1)
#define VT_INTRINFO_RESERVED        0x7fffe000

/* Activity State */
#define VT_ACTSTATE_ACTIVE     0
#define VT_ACTSTATE_HLT        1
#define VT_ACTSTATE_SHUT_DOWN  2
#define VT_ACTSTATE_WFSIPI     3

/* Interruptability */
#define VT_INTRABILITY_STI  0x1
#define VT_INTRABILITY_SS   0x2
#define VT_INTRABILITY_NMI  0x8
#define VT_INTRABILITY_INST (VT_INTRABILITY_STI | VT_INTRABILITY_SS) /* instruction only */
#define VT_INTRABILITY_RSV  0xFFFFFFF4

/* Required feature bits */

#define VT32_REQUIRED_PINBASED_CTLS                    \
   (VT_VMCS_PIN_VMEXEC_CTL_HOSTINTRMASK              | \
    VT_VMCS_PIN_VMEXEC_CTL_NMI)

#define VT32_REQUIRED_PROCBASED_CTLS                   \
   (VT_VMCS_CPU_VMEXEC_CTL_TSCOFF                    | \
    VT_VMCS_CPU_VMEXEC_CTL_HLT                       | \
    VT_VMCS_CPU_VMEXEC_CTL_INVLPG                    | \
    VT_VMCS_CPU_VMEXEC_CTL_RDPMC                     | \
    VT_VMCS_CPU_VMEXEC_CTL_RDTSC                     | \
    VT_VMCS_CPU_VMEXEC_CTL_IOBITMAP                  | \
    VT_VMCS_CPU_VMEXEC_CTL_MOVDR)

#define VT32_REQUIRED_ENTRY_CTLS 0

#define VT32_REQUIRED_EXIT_CTLS  0                     \

#define VT64_REQUIRED_PINBASED_CTLS                    \
   (VT32_REQUIRED_PINBASED_CTLS)

#define VT64_REQUIRED_PROCBASED_CTLS                   \
   (VT32_REQUIRED_PROCBASED_CTLS                     | \
    VT_VMCS_CPU_VMEXEC_CTL_LDCR8                     | \
    VT_VMCS_CPU_VMEXEC_CTL_STCR8                     | \
    VT_VMCS_CPU_VMEXEC_CTL_USECR8SHAD)

#define VT64_REQUIRED_ENTRY_CTLS                       \
   (VT32_REQUIRED_ENTRY_CTLS                         | \
    VT_VMCS_VMENTRY_CTL_LONGMODE)

#define VT64_REQUIRED_EXIT_CTLS                        \
   (VT32_REQUIRED_EXIT_CTLS                          | \
    VT_VMCS_VMEXIT_CTL_LONGMODE)

/*
 *----------------------------------------------------------------------
 * VT_CapableCPU --
 * 
 *   Verify that CPU is VT Capable.
 *----------------------------------------------------------------------
 */
static INLINE Bool
VT_CapableCPU(void)
{
   return (__GET_ECX_FROM_CPUID(1) & CPUID_FEATURE_INTEL_ID1ECX_VMX) != 0;
}


/*
 *----------------------------------------------------------------------
 * VT_EnabledCPU --
 * 
 *   Returns TRUE if VT is enabled in the BIOS.  This function assumes
 *   that the processor is VT_Capable().
 *----------------------------------------------------------------------
 */
static INLINE Bool
VT_EnabledCPU(void)
{
   uint64 msr;

   msr = __GET_MSR(MSR_FEATCTL);
   return ((msr & (MSR_FEATCTL_VMXE | MSR_FEATCTL_LOCK)) ==
           (MSR_FEATCTL_VMXE | MSR_FEATCTL_LOCK));
}


/*
 *----------------------------------------------------------------------
 * VT_SupportedCPU --
 * 
 *   Returns TRUE if the CPU has all of the features that we need to 
 *   run our VT monitor.  This function assumes that the processor is
 *   VT_Capable().
 *   
 *   Note that all currently shipping VT-capable processors meet these
 *   criteria, and that we do not expect any surprises in the field.
 *----------------------------------------------------------------------
 */

static INLINE uint32
VTComputeMandatoryBits(uint32 msrNum, uint32 bits)
{
   uint64 msr =__GET_MSR(msrNum);
   uint32 ones = LODWORD(msr);
   uint32 zeros = HIDWORD(msr); 
   return (bits | ones) & zeros;
}

static INLINE Bool
VT_SupportedCPU(Bool use64Bit)
{  
   uint64 msr;
   unsigned memType;

   if (use64Bit) {
      if ((VT64_REQUIRED_PINBASED_CTLS
           & ~VTComputeMandatoryBits(MSR_VMX_PINBASED_CTLS,
                                     VT64_REQUIRED_PINBASED_CTLS))
          || (VT64_REQUIRED_PROCBASED_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_PROCBASED_CTLS,
                                        VT64_REQUIRED_PROCBASED_CTLS))
          || (VT64_REQUIRED_ENTRY_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_ENTRY_CTLS, 
                                        VT64_REQUIRED_ENTRY_CTLS))
          || (VT64_REQUIRED_EXIT_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_EXIT_CTLS,
                                        VT64_REQUIRED_EXIT_CTLS))) {
         return FALSE;
      }
   } else {
      if ((VT32_REQUIRED_PINBASED_CTLS
           & ~VTComputeMandatoryBits(MSR_VMX_PINBASED_CTLS,
                                     VT32_REQUIRED_PINBASED_CTLS))
          || (VT32_REQUIRED_PROCBASED_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_PROCBASED_CTLS,
                                        VT32_REQUIRED_PROCBASED_CTLS))
          || (VT32_REQUIRED_ENTRY_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_ENTRY_CTLS, 
                                        VT32_REQUIRED_ENTRY_CTLS))
          || (VT32_REQUIRED_EXIT_CTLS
              & ~VTComputeMandatoryBits(MSR_VMX_EXIT_CTLS,
                                        VT32_REQUIRED_EXIT_CTLS))) {
         return FALSE;
      }
   }

   msr = __GET_MSR(MSR_VMX_BASIC);

   memType = (unsigned) ((msr >> MSR_VMX_BASIC_MEMTYPE_SHIFT) &
                         MSR_VMX_BASIC_MEMTYPE_MASK);
   if (memType != MTRR_TYPE_WB) {
      return FALSE;
   }

   return TRUE;
}

#endif /* _X86VT_H_ */
