/*********************************************************
 * Copyright (C) 2007 VMware, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation version 2 and no later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 *********************************************************/

/*
 * vmciQueuePair.c --
 *
 *    VMCI QueuePair API implementation in the host driver.
 */

/* Must come before any kernel header file. */
#if defined(__linux__) && !defined(VMKERNEL)
#  include "driver-config.h"
#endif

#include "vmci_kernel_if.h"
#ifdef VMKERNEL
#  define LOGLEVEL_MODULE_LEN 0
#  define LOGLEVEL_MODULE VMCIVMK
#  include "log.h"
#endif // VMKERNEL
#include "vm_assert.h"
#include "vmci_handle_array.h"
#include "vmciQueuePair.h"
#include "vmciDriver.h"
#include "vmciContext.h"
#include "vmciDatagram.h"
#include "vmciCommonInt.h"
#include "circList.h"

#define LGPFX "VMCIQueuePair: "

typedef struct QueueInfo {
   uint64 size;
#ifndef VMKERNEL
   char   pageFile[VMCI_PATH_MAX];
#endif // !VMKERNEL
} QueueInfo;


/*
 * The context that creates the QueuePair becomes producer of produce queue,
 * and consumer of consume queue. The context on other end for the QueuePair
 * has roles reversed for these two queues.
 */

typedef struct QueuePairEntry {
   VMCIHandle         handle;
   VMCIId             peer;
   uint32             flags;
   QueueInfo          produceQ;
   QueueInfo          consumeQ;
   VMCIId             createId;
   VMCIId             attachId;
   uint32             refCount;
   Bool               pageStoreSet;
   ListItem           listItem;
#ifdef VMKERNEL
   QueuePairPageStore store;
#endif   
} QueuePairEntry;

typedef struct QueuePairList {
   ListItem  *head;
   VMCIMutex mutex;
} QueuePairList;

static QueuePairList queuePairList;

static QueuePairEntry *QueuePairList_FindEntry(VMCIHandle handle);
static void QueuePairList_AddEntry(QueuePairEntry *entry);
static void QueuePairList_RemoveEntry(QueuePairEntry *entry);
static QueuePairEntry *QueuePairList_GetHead(void);
static int QueuePairNotifyPeer(Bool attach, VMCIHandle handle, VMCIId myId,
                               VMCIId peerId);



/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_Init --
 *
 *      Initializes the list of QueuePairs.
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE int
QueuePairList_Init(void)
{
   memset(&queuePairList, 0, sizeof queuePairList);
   return VMCIMutex_Init(&queuePairList.mutex);
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_Exit --
 *
 *      Destroy the list's lock.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
QueuePairList_Exit(void)
{
   VMCIMutex_Destroy(&queuePairList.mutex);
   memset(&queuePairList, 0, sizeof queuePairList);
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_Lock --
 *
 *      Acquires the lock protecting the QueuePair list.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

void
QueuePairList_Lock(void)
{
   VMCIMutex_Acquire(&queuePairList.mutex);
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_Unlock --
 *
 *      Releases the lock protecting the QueuePair list.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

void
QueuePairList_Unlock(void)
{
   VMCIMutex_Release(&queuePairList.mutex);
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_FindEntry --
 *
 *      Finds the entry in the list corresponding to a given handle. Assumes
 *      that the list is locked.
 *
 * Results:
 *      Pointer to entry.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static QueuePairEntry *
QueuePairList_FindEntry(VMCIHandle handle) // IN:
{
   ListItem *next;

   ASSERT(!VMCI_HANDLE_INVALID(handle));
   LIST_SCAN(next, queuePairList.head) {
      QueuePairEntry *entry = LIST_CONTAINER(next, QueuePairEntry, listItem);

      if (VMCI_HANDLE_EQUAL(entry->handle, handle)) {
         return entry;
      }
   }

   return NULL;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_AddEntry --
 *
 *      Adds the given entry to the list. Assumes that the list is locked.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static void
QueuePairList_AddEntry(QueuePairEntry *entry) // IN:
{
   if (entry) {
      LIST_QUEUE(&entry->listItem, &queuePairList.head);
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_RemoveEntry --
 *
 *      Removes the given entry from the list. Assumes that the list is locked.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static void
QueuePairList_RemoveEntry(QueuePairEntry *entry) // IN:
{
   if (entry) {
      LIST_DEL(&entry->listItem, &queuePairList.head);
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairList_GetHead --
 *
 *      Returns the entry from the head of the list. Assumes that the list is
 *      locked.
 *
 * Results:
 *      Pointer to entry.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static QueuePairEntry *
QueuePairList_GetHead(void)
{
   ListItem *first = LIST_FIRST(queuePairList.head);

   if (first) {
      QueuePairEntry *entry = LIST_CONTAINER(first, QueuePairEntry, listItem);
      return entry;
   }

   return NULL;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairDenyConnection --
 *
 *      On ESX we check if the domain names of the two contexts match.
 *      Otherwise we deny the connection.  We always allow the connection on
 *      hosted.
 *
 * Results:
 *      Boolean result.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE Bool
QueuePairDenyConnection(VMCIId contextId, // IN:  Unused on hosted
                        VMCIId peerId)    // IN:  Unused on hosted
{
#ifndef VMKERNEL
   return FALSE; /* Allow on hosted. */
#else
   char contextDomain[VMCI_DOMAIN_NAME_MAXLEN];
   char peerDomain[VMCI_DOMAIN_NAME_MAXLEN];

   ASSERT(contextId != VMCI_INVALID_ID);
   if (peerId == VMCI_INVALID_ID) {
      return FALSE; /* Allow. */
   }
   if (VMCIContext_GetDomainName(contextId, contextDomain,
                                 sizeof contextDomain) != VMCI_SUCCESS) {
      return TRUE; /* Deny. */
   }
   if (VMCIContext_GetDomainName(peerId, peerDomain, sizeof peerDomain) !=
       VMCI_SUCCESS) {
      return TRUE; /* Deny. */
   }
   return strcmp(contextDomain, peerDomain) ? TRUE : /* Deny. */
                                              FALSE; /* Allow. */
#endif
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_Init --
 *
 *      Initalizes QueuePair state in the host driver.
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePair_Init(void)
{
   return QueuePairList_Init();
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_Exit --
 *
 *      Destroys QueuePair state in the host driver.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

void
QueuePair_Exit(void)
{
   QueuePairEntry *entry;

   QueuePairList_Lock();

   while ((entry = QueuePairList_GetHead())) {
      QueuePairList_RemoveEntry(entry);
      VMCI_FreeKernelMem(entry, sizeof *entry);
   }
   
   QueuePairList_Unlock();
   QueuePairList_Exit();
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_Alloc --
 *
 *      Does all the work for the QueuePairAlloc host driver call. Allocates a
 *      QueuePair entry if one does not exist. Attaches to one if it exists,
 *      and retrieves the page files backing that QueuePair.  Assumes that the
 *      QP list lock is held.
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      Memory may be allocated.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePair_Alloc(VMCIHandle handle,             // IN:
                VMCIId peer,                   // IN:
                uint32 flags,                  // IN:
                uint64 produceSize,            // IN:
                uint64 consumeSize,            // IN:
                QueuePairPageStore *pageStore, // IN/OUT
                VMCIContext *context)          // IN: Caller
{
   QueuePairEntry *entry;
   int result;
   const VMCIId contextId = VMCIContext_GetId(context);

   if (VMCI_HANDLE_INVALID(handle) ||
       (flags & ~VMCI_QP_ALL_FLAGS) || (flags & VMCI_QPFLAG_LOCAL) ||
       !(produceSize || consumeSize) || !pageStore ||
#ifdef VMKERNEL
       !pageStore->shared ||
#else
       !pageStore->producePageFile || !pageStore->consumePageFile ||
       !pageStore->producePageFileSize || !pageStore->consumePageFileSize ||
#endif // VMKERNEL
       !context || contextId == VMCI_INVALID_ID ||
       handle.context == VMCI_INVALID_ID) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (VMCIHandleArray_HasEntry(context->queuePairArray, handle)) {
      Log(LGPFX "Context %u already attached to queue pair 0x%x:0x%x.\n",
          contextId, handle.context, handle.resource);
      result = VMCI_ERROR_ALREADY_EXISTS;
      goto out;
   }

   /*
    * Currently, we do not support trusted well known services that
    * use queue pairs, so any request to create/attach from a
    * restricted context will fail. This will change once we may have
    * trusted queue pair endpoints.
    */
   if (context->privFlags & VMCI_PRIVILEGE_FLAG_RESTRICTED) {
      result = VMCI_ERROR_NO_ACCESS;
      goto out;
   }

   entry = QueuePairList_FindEntry(handle);
   if (!entry) { /* Create case. */
      /*
       * Do not create if the caller asked not to.
       */
      if (flags & VMCI_QPFLAG_ATTACH_ONLY) {
         result = VMCI_ERROR_NOT_FOUND;
         goto out;
      }
      
      /*
       * Creator's context ID should match handle's context ID or the creator
       * must allow the context in handle's context ID as the "peer".
       */
      if (handle.context != contextId && handle.context != peer) {
         result = VMCI_ERROR_NO_ACCESS;
         goto out;
      }

      /*
       * Check if we should allow this QueuePair connection.
       */
      if (QueuePairDenyConnection(contextId, peer)) {
         result = VMCI_ERROR_NO_ACCESS;
         goto out;
      }

      entry = VMCI_AllocKernelMem(sizeof *entry, VMCI_MEMORY_NORMAL);
      if (!entry) {
         result = VMCI_ERROR_NO_MEM;
         goto out;
      }

      memset(entry, 0, sizeof *entry);
      entry->handle = handle;
      entry->peer = peer;
      entry->flags = flags;
      entry->createId = contextId;
      entry->attachId = VMCI_INVALID_ID;
      entry->produceQ.size = produceSize;
      entry->consumeQ.size = consumeSize;
      entry->refCount = 1;
#ifdef VMKERNEL
      ASSERT_NOT_IMPLEMENTED(pageStore->shared);
      entry->pageStoreSet = FALSE;
#endif
      INIT_LIST_ITEM(&entry->listItem);

      QueuePairList_AddEntry(entry);
      result = VMCI_SUCCESS_QUEUEPAIR_CREATE;
   } else { /* Attach case. */
      /* Check for failure conditions. */
      if (contextId == entry->createId || contextId == entry->attachId) {
         result = VMCI_ERROR_ALREADY_EXISTS;
         goto out;
      }
      if (entry->refCount >= 2 || entry->attachId != VMCI_INVALID_ID) {
         result = VMCI_ERROR_UNAVAILABLE;
         goto out;
      }
      /*
       * If the creator specifies VMCI_INVALID_ID in "peer" field, access
       * control check is not performed.
       */
      if (entry->peer != VMCI_INVALID_ID && entry->peer != contextId) {
         result = VMCI_ERROR_NO_ACCESS;
         goto out;
      }
      if (entry->produceQ.size != consumeSize ||
          entry->consumeQ.size != produceSize ||
          entry->flags != (flags & ~VMCI_QPFLAG_ATTACH_ONLY)) {
         result = VMCI_ERROR_QUEUEPAIR_MISMATCH;
         goto out;
      }
      if (!entry->pageStoreSet) {
         result = VMCI_ERROR_QUEUEPAIR_NOTSET;
         goto out;
      }

      /*
       * Check if we should allow this QueuePair connection.
       */
      if (QueuePairDenyConnection(contextId, entry->createId)) {
         result = VMCI_ERROR_NO_ACCESS;
         goto out;
      }

#ifdef VMKERNEL
      ASSERT_NOT_IMPLEMENTED(entry->store.shared);
      pageStore->store.shmID = entry->store.store.shmID;
#else
      ASSERT(entry->produceQ.pageFile[0] && entry->consumeQ.pageFile[0]);
      if (pageStore->producePageFileSize < sizeof entry->consumeQ.pageFile) {
         result = VMCI_ERROR_NO_MEM;
         goto out;
      }
      if (VMCI_CopyToUser((void *)(VA)pageStore->producePageFile,
                          entry->consumeQ.pageFile,
                          sizeof entry->consumeQ.pageFile)) {
         result = VMCI_ERROR_GENERIC;
         goto out;
      }
      if (pageStore->consumePageFileSize < sizeof entry->produceQ.pageFile) {
         result = VMCI_ERROR_NO_MEM;
         goto out;
      }
      if (VMCI_CopyToUser((void *)(VA)pageStore->consumePageFile,
                          entry->produceQ.pageFile,
                          sizeof entry->produceQ.pageFile)) {
         result = VMCI_ERROR_GENERIC;
         goto out;
      }
#endif // VMKERNEL

      result = QueuePairNotifyPeer(TRUE, handle, contextId, entry->createId);
      if (result < VMCI_SUCCESS) {
         goto out;
      }

      entry->attachId = contextId;
      entry->refCount++;
      result = VMCI_SUCCESS_QUEUEPAIR_ATTACH;
   }

out:
   if (result >= VMCI_SUCCESS) {
      VMCIHandleArray_AppendEntry(&context->queuePairArray, handle);
   }
   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_SetPageStore --
 *
 *      The creator of a QueuePair uses this to set the page file names for a
 *      given QueuePair.  Assumes that the QP list lock is held.
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePair_SetPageStore(VMCIHandle handle,             // IN:
		       QueuePairPageStore *pageStore, // IN:
		       VMCIContext *context)          // IN: Caller
{
   QueuePairEntry *entry;
   int result;
   const VMCIId contextId = VMCIContext_GetId(context);

   if (VMCI_HANDLE_INVALID(handle) || !pageStore ||
#ifdef VMKERNEL       
       (pageStore->shared && pageStore->store.shmID == SHM_INVALID_ID) ||
#else
       !pageStore->producePageFile || !pageStore->consumePageFile ||
       !pageStore->producePageFileSize || !pageStore->consumePageFile ||
#endif // VMKERNEL
       !context || contextId == VMCI_INVALID_ID) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (!VMCIHandleArray_HasEntry(context->queuePairArray, handle)) {
      Log(LGPFX "Context %u not attached to queue pair 0x%x:0x%x.\n",
          contextId, handle.context, handle.resource);
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }

   entry = QueuePairList_FindEntry(handle);
   if (!entry) {
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }
   if (entry->createId != contextId) {
      result = VMCI_ERROR_QUEUEPAIR_NOTOWNER;
      goto out;
   }
   if (entry->pageStoreSet) {
      result = VMCI_ERROR_UNAVAILABLE;
      goto out;
   }
#ifdef VMKERNEL
   entry->store = *pageStore;
#else
   if (pageStore->producePageFileSize > sizeof entry->produceQ.pageFile) {
      result = VMCI_ERROR_NO_MEM;
      goto out;
   }
   if (VMCI_CopyFromUser(entry->produceQ.pageFile,
			 (void *)(VA)pageStore->producePageFile,
                         (size_t)pageStore->producePageFileSize)) {
      result = VMCI_ERROR_GENERIC;
      goto out;
   }
   if (pageStore->consumePageFileSize > sizeof entry->consumeQ.pageFile) {
      result = VMCI_ERROR_NO_MEM;
      goto out;
   }
   if (VMCI_CopyFromUser(entry->consumeQ.pageFile,
			 (void *)(VA)pageStore->consumePageFile,
                         (size_t)pageStore->consumePageFileSize)) {
      result = VMCI_ERROR_GENERIC;
      goto out;
   }
   Log(LGPFX "%s: produce file = %s\n", __FUNCTION__, entry->produceQ.pageFile);
   Log(LGPFX "%s: consume file = %s\n", __FUNCTION__, entry->consumeQ.pageFile);
#endif // VMKERNEL
   entry->pageStoreSet = TRUE;
   result = VMCI_SUCCESS;

out:
   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_Detach --
 *
 *      Detach a context from a given QueuePair handle.  Assumes that the QP
 *      list lock is held.  If the "detach" input parameter is FALSE, the QP
 *      entry is not removed from the list of QPs, and the context is not
 *      detached from the given handle.  If "detach" is TRUE, the detach
 *      operation really happens.  With "detach" set to FALSE, the caller can
 *      query if the "actual" detach operation would succeed or not.  The
 *      return value from this function remains the same irrespective of the
 *      value of the boolean "detach".
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePair_Detach(VMCIHandle  handle,   // IN:
                 VMCIContext *context, // IN:
                 Bool detach)          // IN: Really detach?
{
   QueuePairEntry *entry;
   int result;
   const VMCIId contextId = VMCIContext_GetId(context);

   if (VMCI_HANDLE_INVALID(handle) ||
       !context || contextId == VMCI_INVALID_ID) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (!VMCIHandleArray_HasEntry(context->queuePairArray, handle)) {
      Log(LGPFX "Context %u not attached to queue pair 0x%x:0x%x.\n",
          contextId, handle.context, handle.resource);
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }

   entry = QueuePairList_FindEntry(handle);
   if (!entry) {
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }

   ASSERT(!(entry->flags & VMCI_QPFLAG_LOCAL));

   if (contextId != entry->createId && contextId != entry->attachId) {
      result = VMCI_ERROR_QUEUEPAIR_NOTATTACHED;
      goto out;
   }

   if (!detach) {
      /* Do not update the QP entry. */
      ASSERT(entry->refCount == 1 || entry->refCount == 2);
      result = entry->refCount == 1 ? VMCI_SUCCESS_LAST_DETACH : VMCI_SUCCESS;
      goto out;
   }

   if (contextId == entry->createId) {
      entry->createId = VMCI_INVALID_ID;
   } else {
      entry->attachId = VMCI_INVALID_ID;
   }
   entry->refCount--;

   if (!entry->refCount) {
      QueuePairList_RemoveEntry(entry);
      VMCI_FreeKernelMem(entry, sizeof *entry);
      result = VMCI_SUCCESS_LAST_DETACH;
   } else {
      VMCIId peerId = entry->createId != VMCI_INVALID_ID ? entry->createId :
                                                           entry->attachId;

      /*
       * XXX: If we ever allow the creator to detach and attach again
       * to the same queue pair, we need to handle the mapping of the
       * shared memory region in vmkernel differently. Currently, we
       * assume that an attaching VM always needs to swap the two
       * queues.
       */

      ASSERT(peerId != VMCI_INVALID_ID);
      QueuePairNotifyPeer(FALSE, handle, contextId, peerId);
      result = VMCI_SUCCESS;
   }

out:
   if (result >= VMCI_SUCCESS && detach) {
      VMCIHandleArray_RemoveEntry(context->queuePairArray, handle);
   }
   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePair_ResetPageStore --
 *
 *      The creator of a QueuePair uses this to clear the page store for a
 *      given QueuePair.  Assumes that the QP list lock is held.
 *
 * Results:
 *      Success or failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePair_ResetPageStore(VMCIHandle handle,    // IN:
			 VMCIContext *context) // IN: Caller
{
   QueuePairEntry *entry;
   int result;
   const VMCIId contextId = VMCIContext_GetId(context);

   if (VMCI_HANDLE_INVALID(handle) ||
       !context || contextId == VMCI_INVALID_ID) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (!VMCIHandleArray_HasEntry(context->queuePairArray, handle)) {
      Log(LGPFX "Context %u not attached to queue pair 0x%x:0x%x.\n",
          contextId, handle.context, handle.resource);
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }

   entry = QueuePairList_FindEntry(handle);
   if (!entry) {
      result = VMCI_ERROR_NOT_FOUND;
      goto out;
   }
   if (entry->createId != contextId) {
      result = VMCI_ERROR_QUEUEPAIR_NOTOWNER;
      goto out;
   }
   if (!entry->pageStoreSet) {
      result = VMCI_ERROR_UNAVAILABLE;
      goto out;
   }
#ifdef VMKERNEL
   entry->store.shared = TRUE;
   entry->store.store.shmID = SHM_INVALID_ID;
#else
   memset(entry->produceQ.pageFile, 0, sizeof entry->produceQ.pageFile);
   memset(entry->consumeQ.pageFile, 0, sizeof entry->consumeQ.pageFile);
#endif
   entry->pageStoreSet = FALSE;
   result = VMCI_SUCCESS;

out:
   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * QueuePairNotifyPeer --
 *
 *      Enqueues an event datagram to notify the peer VM attached to the given
 *      QP handle about attach/detach event by the given VM.
 *
 * Results:
 *      Payload size of datagram enqueued on success, error code otherwise.
 *
 * Side effects:
 *      Memory is allocated.
 *
 *-----------------------------------------------------------------------------
 */

int
QueuePairNotifyPeer(Bool attach,       // IN: attach or detach?
                    VMCIHandle handle, // IN:
                    VMCIId myId,       // IN:
                    VMCIId peerId)     // IN: CID of VM to notify
{
   int rv;
   VMCIEventMsg *eMsg;
   VMCIEventPayload_QP *evPayload;
   char buf[sizeof *eMsg + sizeof *evPayload];

   if (VMCI_HANDLE_INVALID(handle) || myId == VMCI_INVALID_ID ||
       peerId == VMCI_INVALID_ID) {
      return VMCI_ERROR_INVALID_ARGS;
   }
   
   /*
    * Notification message contains:  QP handle and attaching/detaching VM's
    * context id.
    */
   eMsg = (VMCIEventMsg *)buf;

   /*
    * In VMCIContext_EnqueueDatagram() we enforce the upper limit on number of
    * pending events from the hypervisor to a given VM otherwise a rogue VM
    * could do arbitrary number of attached and detaches causing memory
    * pressure in the host kernel.
   */

   /* Clear out any garbage. */
   memset(eMsg, 0, sizeof *eMsg + sizeof *evPayload);

   eMsg->hdr.dst = VMCI_MAKE_HANDLE(peerId, VMCI_EVENT_HANDLER);
   eMsg->hdr.src = VMCI_MAKE_HANDLE(VMCI_HYPERVISOR_CONTEXT_ID,
                                    VMCI_CONTEXT_RESOURCE_ID);
   eMsg->hdr.payloadSize = sizeof *eMsg + sizeof *evPayload - sizeof eMsg->hdr;
   eMsg->eventData.event = attach ? VMCI_EVENT_QP_PEER_ATTACH :
                                    VMCI_EVENT_QP_PEER_DETACH;
   evPayload = VMCIEventMsgPayload(eMsg);
   evPayload->handle = handle;
   evPayload->peerId = myId;

   rv = VMCIDatagram_Dispatch(VMCI_HYPERVISOR_CONTEXT_ID, (VMCIDatagram *)eMsg);
   if (rv < VMCI_SUCCESS) {
      Log(LGPFX "Failed to enqueue QueuePair %s event datagram for "
          "context %u.\n", attach ? "ATTACH" : "DETACH", peerId);
   }

   return rv;
}
