/*
 * linux/drivers/char/ppuser.c
 *
 * Copyright (C) 1998 Tim Waugh <tim@cyberelk.demon.co.uk>
 *
 * May be freely distributed as part of Linux
 */

#include "driver-config.h"

#include <linux/module.h>
#include <linux/sched.h>
#include <linux/ioctl.h>
#include <linux/parport.h>
#include <linux/ctype.h>
#include <linux/poll.h>
#include <asm/uaccess.h>
#ifdef VMWARE
#include <linux/parport_pc.h>
#endif
#include "ppuser.h"

#define PP_VERSION "ppuser: User-space parallel port driver"

#ifndef min
#define min(a,b) ((a) < (b) ? (a) : (b))
#endif

/* The device minor encodes the parport number and (arbitrary) 
 * pardevice number as (port << 4) | dev. */
#define PP_PORT(minor) ((minor >> 4) & 0xf)
#define PP_DEV(minor) ((minor) & 0xf)

struct pp_struct {
	struct pardevice * pdev;
	struct wait_queue * irq_wait;
	int mode;
	unsigned int flags;
	unsigned char interrupts;
};

/* pp_struct.flags bitfields */
#define PP_CLAIMED	(1<<0)
#ifdef VMWARE
#define PP_PASSTHROUGH	(1<<1)
#endif

/* Other constants */
#define PP_INTERRUPT_TIMEOUT (10 * HZ) /* 10s */
#define PP_BUFFER_SIZE 256
#define PARDEVICE_MAX 8

static struct pp_struct pp_table[PARPORT_MAX][PARDEVICE_MAX] = {
	[0 ... PARPORT_MAX-1 ] = {
		[0 ... PARDEVICE_MAX-1 ] = { NULL, NULL, 0, 0 }
	}
};

static loff_t pp_lseek (struct file * file, long long offset, int origin)
{
	return -ESPIPE;
}

struct callback_info {
	struct wait_queue * waitq;
	size_t n;
};

static void pp_ecp_callback (struct parport * port, void * handle, size_t n)
{
	struct callback_info * cbinfo = (struct callback_info *) handle;

	cbinfo->n = n;
	wake_up (&cbinfo->waitq);
}

static ssize_t pp_read (struct file * file, char * buf, size_t count,
			loff_t * ppos)
{
	unsigned int minor = MINOR (file->f_dentry->d_inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);
	struct parport * port = pp_table[portnum][dev].pdev->port;
	char * kbuffer;
	ssize_t bytes_read = 0;
	ssize_t error = 0;
	int mode = pp_table[portnum][dev].mode;

	if (!(pp_table[portnum][dev].flags & PP_CLAIMED)) {
		/* Don't have the port claimed */
		printk (KERN_DEBUG "ppuser%d: claim the port first\n", minor);
		return -EPERM;
	}

	kbuffer = kmalloc (min (count, PP_BUFFER_SIZE), GFP_KERNEL);
	if (!kbuffer)
		return -ENOMEM;

	if (mode == PARPORT_MODE_PCECP) {
		struct callback_info cbinfo;
		init_waitqueue (&cbinfo.waitq);
		while (bytes_read < count) {
			ssize_t need = min(count - bytes_read, PP_BUFFER_SIZE);

			cbinfo.n = 0;
			error = port->ops->ecp_read_block (port, kbuffer, need,
							   pp_ecp_callback,
							   &cbinfo);
			if (error < 0)
				break;

			interruptible_sleep_on_timeout (&cbinfo.waitq,
							PP_INTERRUPT_TIMEOUT);

			error = -EINTR;
			if (signal_pending (current))
				break;

			error = -EFAULT;
			if (copy_to_user (kbuffer, buf + bytes_read, cbinfo.n))
				break;

			bytes_read += cbinfo.n;
		}
	}
	else if (mode == PARPORT_MODE_PCEPP) {
		ssize_t got;
		while (bytes_read < count) {
			ssize_t need = min(count - bytes_read, PP_BUFFER_SIZE);

			got = port->ops->epp_read_block (port, kbuffer, need);

			error = got;
			if (error < 0)
				break;

			error = -EFAULT;
			if (copy_to_user (kbuffer, buf + bytes_read, got))
				break;

			bytes_read += got;
		}
	} else {
		if (mode)
			printk (KERN_DEBUG
				"ppuser%02x: don't how to to read\n",
				minor);
		else printk (KERN_DEBUG "ppuser%02x: set mode first\n", minor);
		error = -EINVAL;
	}

	kfree (kbuffer);
	return error ? error : bytes_read;
}

static ssize_t pp_write (struct file * file, const char * buf, size_t count,
			 loff_t * ppos)
{
	unsigned int minor = MINOR (file->f_dentry->d_inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);
	struct parport * port = pp_table[portnum][dev].pdev->port;
	char * kbuffer;
	ssize_t bytes_written = 0;
	ssize_t error = 0;
	int mode = pp_table[portnum][dev].mode;

	if (!(pp_table[portnum][dev].flags & PP_CLAIMED)) {
		/* Don't have the port claimed */
		printk (KERN_DEBUG "ppuser%02x: claim the port first\n",
			minor);
		return -EPERM;
	}

	kbuffer = kmalloc (min (count, PP_BUFFER_SIZE), GFP_KERNEL);
	if (!kbuffer)
		return -ENOMEM;

	if (mode == PARPORT_MODE_PCECP) {
		struct callback_info cbinfo;
		init_waitqueue (&cbinfo.waitq);
		while (bytes_written < count) {
			ssize_t n = min(count - bytes_written, PP_BUFFER_SIZE);

			error = -EFAULT;
			if (copy_from_user (kbuffer, buf + bytes_written, n))
				break;

			cbinfo.n = 0;
			error = port->ops->ecp_write_block (port, kbuffer, n,
							    pp_ecp_callback,
							    &cbinfo);
			if (error < 0)
				break;

			interruptible_sleep_on_timeout (&cbinfo.waitq,
							PP_INTERRUPT_TIMEOUT);

			error = -EINTR;
			if (signal_pending (current))
				break;

			bytes_written += cbinfo.n;
		}
	}
	else if (mode == PARPORT_MODE_PCEPP) {
		ssize_t wrote;
		while (bytes_written < count) {
			ssize_t n = min(count - bytes_written, PP_BUFFER_SIZE);

			error = -EFAULT;
			if (copy_from_user (kbuffer, buf + bytes_written, n))
				break;

			wrote = port->ops->epp_write_block (port, kbuffer, n);

			error = wrote;
			if (error < 0)
				break;

			bytes_written += wrote;
		}
	} else {
		if (mode)
			printk (KERN_DEBUG
				"ppuser%02x: don't how to to write\n",
				minor);
		else printk (KERN_DEBUG "ppuser%02x: set mode first\n", minor);
		return -EINVAL;
	}

	kfree (kbuffer);
	return error ? error : bytes_written;
}

static int pp_ioctl(struct inode *inode, struct file *file,
		    unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);
	struct pp_struct * pp = &pp_table[portnum][dev];
	struct parport * port = pp->pdev->port;
	unsigned char reg;
	unsigned int value;
	int dir;

	if (cmd == PPCLAIM) {
		if (pp_table[portnum][dev].flags & PP_CLAIMED) {
			printk (KERN_DEBUG
				"ppuser%02x: you've already got it!\n", minor);
			return -EINVAL;
		}
		parport_claim_or_block (pp_table[portnum][dev].pdev);
		pp_table[portnum][dev].flags |= PP_CLAIMED;
		return 0;
	}

	if (cmd == PPRELEASE) {
		if ((pp_table[portnum][dev].flags & PP_CLAIMED) == 0) {
			printk (KERN_DEBUG
				"ppuser%02x: you don't have it!\n", minor);
			return -EINVAL;
		}
		parport_release (pp_table[portnum][dev].pdev);
		pp_table[portnum][dev].flags &= ~PP_CLAIMED;
		return 0;
	}

	if ((pp_table[portnum][dev].flags & PP_CLAIMED) == 0) {
		printk (KERN_DEBUG "ppuser%02x: claim the port first\n",
			minor);
		return -EPERM;
	}

	dir = VERIFY_WRITE;
	switch (cmd) {
	case PPRSTATUS:
		reg = parport_read_status (port);
		break;

	case PPRCONTROL:
		reg = parport_read_control (port);
		break;

	case PPRDATA:
		reg = parport_read_data (port);
		break;

	case PPRECONTROL:
		/* Should check to see if it exists */
		reg = parport_read_econtrol (port);
		break;

	case PPRFIFO:
		reg = parport_read_fifo (port);
		break;

	case PPYIELD:
		parport_yield_blocking (pp_table[portnum][dev].pdev);
		return 0;

	case PPGETMODES:
	        value = port->modes;
		break;

	case PPGETBASE:
		value = port->base;
		break;

	default:
		dir = VERIFY_READ;
	}
			
	if (dir == VERIFY_WRITE) {
	   if (cmd == PPGETMODES || cmd == PPGETBASE) {
		return copy_to_user ((unsigned int *) arg, &value,
				     sizeof (value)) ? -EFAULT : 0;
	   } else {
		return copy_to_user ((unsigned char *) arg, &reg,
				     sizeof (reg)) ? -EFAULT : 0;
	   }
	}

	if (copy_from_user (&reg, (unsigned char *) arg, sizeof (reg)))
		return -EFAULT;

	switch (cmd) {
	case PPSETMODE:
		/* FIXME! Should be unsigned int here */
		parport_change_mode (port, reg);
		pp_table[portnum][dev].mode = reg;
		break;

	case PPWSTATUS:
		parport_write_status (port, reg);
		break;

	case PPWCONTROL:
		parport_write_control (port, reg);
		break;

	case PPWDATA:
		parport_write_data (port, reg);
		break;

	case PPWECONTROL:
		/* Should check to see if it exists */
		parport_write_econtrol (port, reg);
		break;

	case PPWFIFO:
		parport_write_fifo (port, reg);
		break;

	default:
		printk (KERN_DEBUG "ppuser%02x: What?\n", minor);
		return -EINVAL;
	}

	return 0;
}

static void pp_irq (int irq, void * private, struct pt_regs * unused)
{
	struct pp_struct * pp = (struct pp_struct *) private;
	if (pp->interrupts < 255) {
		pp->interrupts++;
	}
	wake_up_interruptible (&pp->irq_wait);
}

static int pp_open (struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR (inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);
	struct parport * port;
	struct pardevice * pdev = NULL;
	char *name;

	if (portnum >= PARPORT_MAX)
		return -ENXIO;

	if (pp_table[portnum][dev].pdev)
		return -EBUSY;

	name = kmalloc (16, GFP_KERNEL);
	if (name == NULL)
		return -ENOMEM;

	pp_table[portnum][dev].mode = 0;
	pp_table[portnum][dev].flags = 0;
	pp_table[portnum][dev].interrupts = 0;

	sprintf (name, "ppuser%02x", minor);
	port = parport_enumerate ();

	while (port && port->number != portnum)
		port = port->next;

	if (!port) {
		printk (KERN_WARNING "%s: no associated port!\n", name);
		kfree (name);
		return -ENXIO;
	}

	pdev = parport_register_device (port, name, NULL, NULL, pp_irq, 0,
					&pp_table[portnum][dev]);

	if (!pdev) {
		printk (KERN_WARNING "%s: failed to register device!\n", name);
		kfree (name);
		return -ENXIO;
	}

	MOD_INC_USE_COUNT;

	pp_table[portnum][dev].pdev = pdev;
	init_waitqueue (&pp_table[portnum][dev].irq_wait);
	printk (KERN_DEBUG "%s: registered pardevice\n", name);
	return 0;
}

static int pp_release (struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR (inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);

	if (pp_table[portnum][dev].flags & PP_CLAIMED) {
		parport_release (pp_table[portnum][dev].pdev);
		printk (KERN_DEBUG "ppuser%02x: released pardevice because "
			"user-space forgot\n", minor);
	}

	kfree (pp_table[portnum][dev].pdev->name);
	parport_unregister_device (pp_table[portnum][dev].pdev);
	pp_table[portnum][dev].pdev = NULL;
	printk (KERN_DEBUG "ppuser%02x: unregistered pardevice\n", minor);
	MOD_DEC_USE_COUNT;
	return 0;
}

#if 0
static unsigned int pp_poll (struct file * file, poll_table * wait)
{
	unsigned int minor = MINOR (file->f_dentry->d_inode->i_rdev);
	unsigned int portnum = PP_PORT (minor);
	unsigned int dev = PP_DEV (minor);

	poll_wait (file, &pp_table[portnum][dev].irq_wait, wait);
	return POLLIN | POLLRDNORM;
}
#endif

static struct file_operations pp_fops = {
	pp_lseek,
	pp_read,
	pp_write,
	NULL,	/* pp_readdir */
	NULL,	/* pp_poll */
	pp_ioctl,
	NULL,	/* pp_mmap */
	pp_open,
	NULL,   /* pp_flush */
	pp_release
};

#ifdef MODULE
#define pp_init init_module
#endif

int pp_init (void)
{
	if (register_chrdev (PP_MAJOR, "ppuser", &pp_fops)) {
		printk (KERN_WARNING "parport_user: unable to get major %d\n",
			PP_MAJOR);
		return -EIO;
	}

	printk (KERN_INFO PP_VERSION "\n");
	return 0;
}

#ifdef MODULE
void cleanup_module (void)
{
	/* Clean up all parport stuff */
	unregister_chrdev (PP_MAJOR, "ppuser");
}
#endif /* MODULE */
