/*********************************************************
 * Copyright (C) 2007 VMware, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation version 2 and no later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 *********************************************************/

/* Must come before any kernel header file */
#include "driver-config.h"

#define EXPORT_SYMTAB

#include "compat_version.h"
#include "compat_kernel.h"
#include <linux/miscdevice.h>
#include "compat_module.h"
#include "compat_sched.h"
#include "compat_file.h"
#include "compat_slab.h"
#include "compat_interrupt.h"

#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
#include "compat_semaphore.h"
#endif
#include <linux/smp.h>
#include <linux/smp_lock.h>

#include <linux/poll.h>

#include <asm/io.h>
#if defined(__x86_64__) && LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
#   if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
#      include <asm/ioctl32.h>
#   else
#      include <linux/ioctl32.h>
#   endif
/* Use weak: not all kernels export sys_ioctl for use by modules */
#   if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 66)
asmlinkage __attribute__((weak)) long
sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);
#   else
asmlinkage __attribute__((weak)) int
sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);
#   endif
#endif

#include "vmware.h"
#include "driverLog.h"
#include "circList.h"
#include "vmci_defs.h"
#include "vmci_iocontrols.h"
#include "vmci_kernel_if.h"
#include "vmci_infrastructure.h"
#include "vmciCommonInt.h"
#include "vmciDriver.h"
#include "vmciDatagram.h"
#include "vmciProcess.h"
#include "vmciDsInt.h"
#include "vmciGroup.h"
#include "vmciQueuePair.h"

/* Compatibility reference counting */
#ifdef KERNEL_2_4_0
#   define compat_fop_set_owner(_pFop) do { \
   (_pFop)->owner = THIS_MODULE;            \
} while (0)
#   define compat_mod_inc_refcount
#   define compat_mod_dec_refcount
#else
#   define compat_fop_set_owner(_pFop)
#   define compat_mod_inc_refcount do { \
   MOD_INC_USE_COUNT;                   \
} while (0)
#   define compat_mod_dec_refcount do { \
   MOD_DEC_USE_COUNT;                   \
} while (0)
#endif


/*
 * Per-instance driver state
 */

typedef struct VMCILinux {
   union {
      VMCIContext *context;
      VMCIProcess *process;
      VMCIDatagramProcess *dgmProc;
   } ct;
   VMCIObjType ctType;
#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
   struct semaphore lock;
#endif
} VMCILinux;

#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
#define LinuxDriverLockIoctlPerFD(mutex) down(mutex)
#define LinuxDriverUnlockIoctlPerFD(mutex) up(mutex) 
#else
#define LinuxDriverLockIoctlPerFD(mutex) do {} while (0)
#define LinuxDriverUnlockIoctlPerFD(mutex) do {} while (0)
#endif

/*
 * Static driver state.
 */

#define VM_DEVICE_NAME_SIZE 32
#define LINUXLOG_BUFFER_SIZE  1024

typedef struct VMCILinuxState {
   int major;
   int minor;
   struct miscdevice misc;
   char deviceName[VM_DEVICE_NAME_SIZE];
   char buf[LINUXLOG_BUFFER_SIZE];
} VMCILinuxState;

static struct VMCILinuxState linuxState;


/*
 *----------------------------------------------------------------------
 *
 * Device Driver Interface --
 *
 *      Implements VMCI by implementing open/close/ioctl functions
 *
 *
 *----------------------------------------------------------------------
 */

static int LinuxDriver_Open(struct inode *inode, struct file *filp);

static int LinuxDriver_Ioctl(struct inode *inode, struct file *filp,
                             u_int iocmd, unsigned long ioarg);

#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
static long LinuxDriver_UnlockedIoctl(struct file *filp,
                                      u_int iocmd, unsigned long ioarg);
#endif

static int LinuxDriver_Close(struct inode *inode, struct file *filp);
static unsigned int LinuxDriverPoll(struct file *file, poll_table *wait);

static struct file_operations vmuser_fops;


#ifdef VM_X86_64
#ifndef HAVE_COMPAT_IOCTL
static int 
LinuxDriver_Ioctl32_Handler(unsigned int fd, unsigned int iocmd, 
                            unsigned long ioarg, struct file * filp)
{
   int ret;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 26) || \
   (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 0) && LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 3))
   lock_kernel();
#endif
   ret = -ENOTTY;
   if (filp && filp->f_op && filp->f_op->ioctl == LinuxDriver_Ioctl) {
      ret = LinuxDriver_Ioctl(filp->f_dentry->d_inode, filp, iocmd, ioarg);
   }
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 26) || \
   (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 0) && LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 3))
   unlock_kernel();
#endif
   return ret;
}
#endif /* !HAVE_COMPAT_IOCTL */

static int
register_ioctl32_handlers(void)
{
#ifndef HAVE_COMPAT_IOCTL
   {
      int i;
      for (i = IOCTL_VMCI_FIRST; i < IOCTL_VMCI_LAST; i++) {
         int retval = register_ioctl32_conversion(i, LinuxDriver_Ioctl32_Handler);
         if (retval) {
            Warning("Fail to register ioctl32 conversion for cmd %d\n", i);
            return retval;
         }
      }
   }
#endif /* !HAVE_COMPAT_IOCTL */
   return 0;
}

static void
unregister_ioctl32_handlers(void)
{
#ifndef HAVE_COMPAT_IOCTL
   {
      int i;
      for (i = IOCTL_VMCI_FIRST; i < IOCTL_VMCI_LAST; i++) {
         int retval = unregister_ioctl32_conversion(i);
         if (retval) {
            Warning("Fail to unregister ioctl32 conversion for cmd %d\n", i);
         }
      }
   }
#endif /* !HAVE_COMPAT_IOCTL */
}
#else /* VM_X86_64 */
#define register_ioctl32_handlers() (0)
#define unregister_ioctl32_handlers() do { } while (0)
#endif /* VM_X86_64 */


/*
 *----------------------------------------------------------------------
 *
 * init_module --
 *
 *      linux module entry point. Called by /sbin/insmod command
 *
 * Results:
 *      registers a device driver for a major # that depends
 *      on the uid. Add yourself to that list.  List is now in
 *      private/driver-private.c.
 *
 *----------------------------------------------------------------------
 */

int
init_module(void)
{
   int retval;

   DriverLog_Init("/dev/vmci");

   /* Initialize VMCI core and APIs. */
   if (VMCI_Init() < VMCI_SUCCESS) {
      return -ENOMEM;
   }

   /*
    * Initialize the file_operations structure. Because this code is always
    * compiled as a module, this is fine to do it here and not in a static
    * initializer.
    */

   memset(&vmuser_fops, 0, sizeof vmuser_fops);
   compat_fop_set_owner(&vmuser_fops);
   vmuser_fops.poll = LinuxDriverPoll;
#ifdef HAVE_UNLOCKED_IOCTL
   vmuser_fops.unlocked_ioctl = LinuxDriver_UnlockedIoctl;
#else
   vmuser_fops.ioctl = LinuxDriver_Ioctl;
#endif
#ifdef HAVE_COMPAT_IOCTL
   vmuser_fops.compat_ioctl = LinuxDriver_UnlockedIoctl;
#endif
   vmuser_fops.open = LinuxDriver_Open;
   vmuser_fops.release = LinuxDriver_Close;

   sprintf(linuxState.deviceName, "vmci");
   linuxState.major = 10;
   linuxState.misc.minor = MISC_DYNAMIC_MINOR;
   linuxState.misc.name = linuxState.deviceName;
   linuxState.misc.fops = &vmuser_fops;

   retval = misc_register(&linuxState.misc);

   if (retval) {
      Warning("Module %s: error %u registering with major=%d minor=%d\n",
	      linuxState.deviceName, -retval, linuxState.major, linuxState.minor);
      return -ENOENT;
   }
   linuxState.minor = linuxState.misc.minor;
   Log("Module %s: registered with major=%d minor=%d\n",
       linuxState.deviceName, linuxState.major, linuxState.minor);


   retval = register_ioctl32_handlers();
   if (retval) {
      misc_deregister(&linuxState.misc);
      return retval;
   }

   Log("Module %s: initialized\n", linuxState.deviceName);

   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * cleanup_module --
 *
 *      Called by /sbin/rmmod
 *
 *
 *----------------------------------------------------------------------
 */

void
cleanup_module(void)
{
   int retval;

   unregister_ioctl32_handlers();

   VMCI_Cleanup();

   /*
    * XXX smp race?
    */
   retval = misc_deregister(&linuxState.misc);

   if (retval) {
      Warning("Module %s: error unregistering\n", linuxState.deviceName);
   } else {
      Log("Module %s: unloaded\n", linuxState.deviceName);
   }
}



/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Open  --
 *
 *      called on open of /dev/vmci. Use count used
 *      to determine eventual deallocation of the module
 *
 * Side effects:
 *     Increment use count used to determine eventual deallocation of
 *     the module
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_Open(struct inode *inode, // IN
                 struct file *filp)   // IN
{
   VMCILinux *vmciLinux;

   compat_mod_inc_refcount;

   vmciLinux = kmalloc(sizeof *vmciLinux, GFP_KERNEL);
   if (vmciLinux == NULL) {
      compat_mod_dec_refcount;
      return -ENOMEM;
   }
   memset(vmciLinux, 0, sizeof *vmciLinux);
   vmciLinux->ctType = VMCIOBJ_NOT_SET;
#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
   init_MUTEX(&vmciLinux->lock);
#endif

   filp->private_data = vmciLinux;

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Close  --
 *
 *      called on close of /dev/vmci, most often when the
 *      process exits. Decrement use count, allowing for possible uninstalling
 *      of the module.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_Close(struct inode *inode, // IN
                  struct file *filp)   // IN
{
   VMCILinux *vmciLinux;

   vmciLinux = (VMCILinux *)filp->private_data;
   ASSERT(vmciLinux);

   if (vmciLinux->ctType == VMCIOBJ_CONTEXT) {
      VMCIId cid;
      
      ASSERT(vmciLinux->ct.context);
      cid = VMCIContext_GetId(vmciLinux->ct.context);

      /*
       * Remove the context from the datagram and DS API groups. Meaning it
       * can no longer access the API functions.
       */
      VMCIDs_RemoveContext(cid);
      
      /* Remove context from the public group handle. */
      VMCIPublicGroup_RemoveContext(cid);

      VMCIContext_ReleaseContext(vmciLinux->ct.context);
      vmciLinux->ct.context = NULL;
   } else if (vmciLinux->ctType == VMCIOBJ_PROCESS) {
      VMCIProcess_Destroy(vmciLinux->ct.process);
      vmciLinux->ct.process = NULL;
   } else if (vmciLinux->ctType == VMCIOBJ_DATAGRAM_PROCESS) {
      VMCIDatagramProcess_Destroy(vmciLinux->ct.dgmProc);
      vmciLinux->ct.dgmProc = NULL;
   }
   vmciLinux->ctType = VMCIOBJ_NOT_SET;

   kfree(vmciLinux);
   filp->private_data = NULL;
   compat_mod_dec_refcount;
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverPoll  --
 *
 *      This is used to wake up the VMX when a VMCI call arrives, or
 *      to wake up select() or poll() at the next clock tick.
 *
 *----------------------------------------------------------------------
 */

static unsigned int
LinuxDriverPoll(struct file *filp,
		poll_table *wait)
{
   VMCILockFlags flags;
   VMCILinux *vmciLinux = (VMCILinux *) filp->private_data;
   unsigned int mask = 0;

   if (vmciLinux->ctType == VMCIOBJ_CONTEXT) {
      ASSERT(vmciLinux->ct.context != NULL);
      /* 
       * Check for VMCI calls to this VM context. 
       */

      if (wait != NULL) {
         poll_wait(filp, &vmciLinux->ct.context->hostContext.waitQueue, wait);
      }

      VMCI_GrabLock(&vmciLinux->ct.context->lock, &flags);
      if (vmciLinux->ct.context->pendingDatagrams > 0) {
         mask = POLLIN;
      }
      VMCI_ReleaseLock(&vmciLinux->ct.context->lock, flags);

   } else if (vmciLinux->ctType == VMCIOBJ_PROCESS) {
      /* nop */
   } else if (vmciLinux->ctType == VMCIOBJ_DATAGRAM_PROCESS) {
      ASSERT(vmciLinux->ct.dgmProc);

      /* 
       * Check for messages to this datagram fd. 
       */

      if (wait) {
         poll_wait(filp, &vmciLinux->ct.dgmProc->host.waitQueue, wait);
      }

      VMCI_GrabLock(&vmciLinux->ct.dgmProc->lock, &flags);
      if (vmciLinux->ct.dgmProc->pendingDatagrams > 0) {
         mask = POLLIN;
      }
      VMCI_ReleaseLock(&vmciLinux->ct.dgmProc->lock, flags);
   }
   return mask;
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriver_Ioctl --
 *
 *      Main path for UserRPC
 *
 * Results:
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static int
LinuxDriver_Ioctl(struct inode *inode,
                  struct file *filp,
                  u_int iocmd,
                  unsigned long ioarg)
{
   VMCILinux *vmciLinux = (VMCILinux *) filp->private_data;
   int retval = 0;

   switch (iocmd) {
   case IOCTL_VMCI_VERSION2: {
      int verFromUser;

      if (copy_from_user(&verFromUser, (void *)ioarg, sizeof verFromUser)) {
         retval = -EFAULT;
         break;
      }
   }
      /* Fall through. */
   case IOCTL_VMCI_VERSION:
      retval = VMCI_VERSION;
      break;

   case IOCTL_VMCI_INIT_CONTEXT: {
      VMCIInitBlock initBlock;

      retval = copy_from_user(&initBlock, (void *)ioarg, sizeof initBlock);
      if (retval != 0) {
         Log("VMCI: Error reading init block.\n");
         retval = -EFAULT;
         break;
      }

      LinuxDriverLockIoctlPerFD(&vmciLinux->lock);
      if (vmciLinux->ctType != VMCIOBJ_NOT_SET) {
         Log("VMCI: Received VMCI init on initialized handle\n");
         retval = -EINVAL;
         goto init_release;
      }

      if (initBlock.flags & ~VMCI_PRIVILEGE_FLAG_RESTRICTED) {
         Log("VMCI: Unsupported VMCI restriction flag.\n");
         retval = -EINVAL;
         goto init_release;
      }

      if ((retval = VMCIContext_InitContext(initBlock.cid, initBlock.flags,
					    -1, &vmciLinux->ct.context)) <
          VMCI_SUCCESS) {
         Log("VMCI: Error initializing context.\n");
         retval = retval == VMCI_ERROR_DUPLICATE_ENTRY ? -EEXIST : -EINVAL;
         goto init_release;
      }


      /* 
       * Copy cid to userlevel, we do this to allow the VMX to enforce its 
       * policy on cid generation.
       */
      initBlock.cid = VMCIContext_GetId(vmciLinux->ct.context); 
      retval = copy_to_user((void *)ioarg, &initBlock, sizeof initBlock); 
      if (retval != 0) {
	 VMCIContext_ReleaseContext(vmciLinux->ct.context);
	 vmciLinux->ct.context = NULL;
	 Log("VMCI: Error writing init block.\n");
	 retval = -EFAULT;
	 goto init_release;
      }
      ASSERT(initBlock.cid != VMCI_INVALID_ID);

      /* Give VM context access to the datagram and DS API. */
      VMCIDs_AddContext(initBlock.cid);

      /* Add VM to the public group handle. */
      VMCIPublicGroup_AddContext(initBlock.cid);

      vmciLinux->ctType = VMCIOBJ_CONTEXT;
      
     init_release:
      LinuxDriverUnlockIoctlPerFD(&vmciLinux->lock);
      break;
   }

   case IOCTL_VMCI_CREATE_PROCESS: {
      LinuxDriverLockIoctlPerFD(&vmciLinux->lock);
      if (vmciLinux->ctType != VMCIOBJ_NOT_SET) {
         Log("VMCI: Received VMCI init on initialized handle\n");
         retval = -EINVAL;
         goto create_release;
      }

      if (VMCIProcess_Create(&vmciLinux->ct.process, -1) < VMCI_SUCCESS) {
         Log("VMCI: Error initializing process.\n");
         retval = -EINVAL;
         goto create_release;
      }
      vmciLinux->ctType = VMCIOBJ_PROCESS;

     create_release:
      LinuxDriverUnlockIoctlPerFD(&vmciLinux->lock);
      break;
   }

   case IOCTL_VMCI_CREATE_DATAGRAM_PROCESS: {
      VMCIDatagramCreateInfo dgCreateInfo;
      VMCIDatagramProcess *dgmProc;

      /* Get datagram create info. */
      retval = copy_from_user(&dgCreateInfo, (char *)ioarg, sizeof dgCreateInfo);
      if (retval != 0) {
         Log("VMCI: Error getting datagram create info, %d\n", retval);
         retval = -EFAULT;
	 break;
      }

      LinuxDriverLockIoctlPerFD(&vmciLinux->lock);
      if (vmciLinux->ctType != VMCIOBJ_NOT_SET) {
         Log("VMCI: Received IOCTLCMD_VMCI_CREATE_DATAGRAM_PROCESS on initialized handle\n");
         retval = -EINVAL;
         goto create_dg_release;
      }

      /* Create process and datagram. */
      if (VMCIDatagramProcess_Create(&dgmProc, &dgCreateInfo) < VMCI_SUCCESS) {
	 retval = -EINVAL;
	 goto create_dg_release;
      }
      retval = copy_to_user((void *)ioarg, &dgCreateInfo, sizeof dgCreateInfo);
      if (retval != 0) {
	 VMCIDatagramProcess_Destroy(dgmProc);
         Log("VMCI: Error copying create info out, %d.\n", retval);
         retval = -EFAULT;
	 break;
      }
      vmciLinux->ct.dgmProc = dgmProc;
      vmciLinux->ctType = VMCIOBJ_DATAGRAM_PROCESS;

     create_dg_release:
      LinuxDriverUnlockIoctlPerFD(&vmciLinux->lock);
      break;
   }

   case IOCTL_VMCI_DATAGRAM_SEND: {
      VMCIDatagramSendRecvInfo sendInfo;
      VMCIDatagram *dg = NULL;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_DATAGRAM_PROCESS &&
	  vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Warning("VMCI: Ioctl %d only valid for context and process datagram "
		 "handle.\n", iocmd);
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&sendInfo, (void *) ioarg, sizeof sendInfo);
      if (retval) {
         Warning("VMCI: copy_from_user failed.\n");
         retval = -EFAULT;
         break;
      }

      if (sendInfo.len > VMCI_MAX_DG_SIZE) {
         Warning("VMCI: datagram size too big.\n");
	 retval = -EINVAL;
	 break;
      }

      if (sendInfo.len < sizeof *dg) {
         Warning("VMCI: datagram size too small.\n");
	 retval = -EINVAL;
	 break;
      }

      dg = VMCI_AllocKernelMem(sendInfo.len, VMCI_MEMORY_NORMAL);
      if (dg == NULL) {
         Log("VMCI: Cannot allocate memory to dispatch datagram.\n");
         retval = -ENOMEM;
         break;
      }

      retval = copy_from_user(dg, (char *)(VA)sendInfo.addr, sendInfo.len);
      if (retval != 0) {
         Log("VMCI: Error getting datagram: %d\n", retval);
         VMCI_FreeKernelMem(dg, sendInfo.len);
         retval = -EFAULT;
         break;
      }

      VMCI_LOG(("VMCI: Datagram dst handle 0x%"FMT64"x, src handle 0x%"FMT64"x, "
	        "payload size %"FMT64"u.\n", 
                dg->dstHandle, dg->srcHandle, dg->payloadSize));

      /* Get source context id. */
      if (vmciLinux->ctType == VMCIOBJ_CONTEXT) {
	 ASSERT(vmciLinux->ct.context);
	 cid = VMCIContext_GetId(vmciLinux->ct.context);               
      } else {
	 /* XXX Will change to dynamic id when we make host context id random. */
	 cid = VMCI_HOST_CONTEXT_ID;
      }
      ASSERT(cid != VMCI_INVALID_ID);
      sendInfo.result = VMCIDatagram_Dispatch(cid, dg);
      VMCI_FreeKernelMem(dg, sendInfo.len);
      retval = copy_to_user((void *)ioarg, &sendInfo, sizeof sendInfo);
      break;
   }

   case IOCTL_VMCI_DATAGRAM_RECEIVE: {
      VMCIDatagramSendRecvInfo recvInfo;
      VMCIDatagram *dg = NULL;

      if (vmciLinux->ctType != VMCIOBJ_DATAGRAM_PROCESS &&
	  vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Warning("VMCI: Ioctl %d only valid for context and process datagram "
		 "handle.\n", iocmd);
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&recvInfo, (void *) ioarg, sizeof recvInfo);
      if (retval) {
         Warning("VMCI: copy_from_user failed.\n");
         retval = -EFAULT;
         break;
      }

      if (vmciLinux->ctType == VMCIOBJ_CONTEXT) {
	 size_t size = recvInfo.len;
	 ASSERT(vmciLinux->ct.context);
	 recvInfo.result = VMCIContext_DequeueDatagram(vmciLinux->ct.context,
						       &size, &dg);
      } else {
	 ASSERT(vmciLinux->ctType == VMCIOBJ_DATAGRAM_PROCESS);
	 ASSERT(vmciLinux->ct.dgmProc);
	 recvInfo.result = VMCIDatagramProcess_ReadCall(vmciLinux->ct.dgmProc, 
							recvInfo.len, &dg);
      }	 
      if (recvInfo.result >= VMCI_SUCCESS) {
	 ASSERT(dg);
	 retval = copy_to_user((void *) ((uintptr_t) recvInfo.addr), dg,
			       VMCI_DG_SIZE(dg));
	 VMCI_FreeKernelMem(dg, VMCI_DG_SIZE(dg));
	 if (retval != 0) {
	    break;
	 }
      }
      retval = copy_to_user((void *)ioarg, &recvInfo, sizeof recvInfo);
      break;
   }

   case IOCTL_VMCI_QUEUEPAIR_ALLOC: {
      VMCIQueuePairAllocInfo queuePairAllocInfo;
      VMCIQueuePairAllocInfo *info = (VMCIQueuePairAllocInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_QUEUEPAIR_ALLOC only valid for contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&queuePairAllocInfo, (void *)ioarg,
                              sizeof queuePairAllocInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      QueuePairList_Lock();

      {
	 QueuePairPageStore pageStore = { queuePairAllocInfo.producePageFile,
					  queuePairAllocInfo.consumePageFile,
					  queuePairAllocInfo.producePageFileSize,
					  queuePairAllocInfo.consumePageFileSize };

	 result = QueuePair_Alloc(queuePairAllocInfo.handle,
	                          queuePairAllocInfo.peer,
				  queuePairAllocInfo.flags,
				  queuePairAllocInfo.produceSize,
				  queuePairAllocInfo.consumeSize,
				  &pageStore,
				  vmciLinux->ct.context);
      }
      Log("VMCI: IOCTL_VMCI_QUEUEPAIR_ALLOC cid = %u result = %d.\n", cid,
          result);
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         if (result >= VMCI_SUCCESS) {
            result = QueuePair_Detach(queuePairAllocInfo.handle,
                                      vmciLinux->ct.context, TRUE);
            ASSERT(result >= VMCI_SUCCESS);
         }
      }

      QueuePairList_Unlock();
      break;
   }

   case IOCTL_VMCI_QUEUEPAIR_SETPAGEFILE: {
      VMCIQueuePairPageFileInfo pageFileInfo;
      VMCIQueuePairPageFileInfo *info = (VMCIQueuePairPageFileInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_QUEUEPAIR_SETPAGEFILE only valid for "
             "contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&pageFileInfo, (void *)ioarg,
                              sizeof pageFileInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      QueuePairList_Lock();

      {
	 QueuePairPageStore pageStore = { pageFileInfo.producePageFile,
					  pageFileInfo.consumePageFile,
					  pageFileInfo.producePageFileSize,
					  pageFileInfo.consumePageFileSize };
	 
	 result = QueuePair_SetPageStore(pageFileInfo.handle,
					 &pageStore,
					 vmciLinux->ct.context);
      }
      Log("VMCI: IOCTL_VMCI_QUEUEPAIR_SETPAGEFILE cid = %u result = %d.\n",
          cid, result);
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         if (result >= VMCI_SUCCESS) {
            result = QueuePair_ResetPageStore(pageFileInfo.handle,
					      vmciLinux->ct.context);
            ASSERT(result >= VMCI_SUCCESS);
         }
      }

      QueuePairList_Unlock();
      break;
   }

   case IOCTL_VMCI_QUEUEPAIR_DETACH: {
      VMCIQueuePairDetachInfo detachInfo;
      VMCIQueuePairDetachInfo *info = (VMCIQueuePairDetachInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_QUEUEPAIR_DETACH only valid for contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&detachInfo, (void *)ioarg, sizeof detachInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      QueuePairList_Lock();
      result = QueuePair_Detach(detachInfo.handle, vmciLinux->ct.context,
                                FALSE); /* Probe detach operation. */
      Log("VMCI: IOCTL_VMCI_QUEUEPAIR_DETACH cid = %u result = %d.\n",
          cid, result);
      
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         /* Could not copy to userland, don't perform the actual detach. */
         retval = -EFAULT;
      } else {
         if (result >= VMCI_SUCCESS) {
            /* Now perform the actual detach. */
            int32 result2 = QueuePair_Detach(detachInfo.handle,
                                             vmciLinux->ct.context, TRUE);
            if (UNLIKELY(result != result2)) {
               /*
                * This should never happen.  But it's better to log a warning
                * than to crash the host.
                */
               Warning("QueuePair_Detach returned different results:  "
                       "previous = %d, current = %d.\n", result, result2);
            }
         }
      }

      QueuePairList_Unlock();
      break;
   }

   case IOCTL_VMCI_DATAGRAM_REQUEST_MAP: {
      VMCIDatagramMapInfo mapInfo;
      VMCIDatagramMapInfo *info = (VMCIDatagramMapInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_REQUEST_MAP only valid for contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&mapInfo, (void *)ioarg, sizeof mapInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      result = VMCIDatagramRequestWellKnownMap(mapInfo.wellKnownID, cid,
					       VMCIContext_GetPrivFlags(cid));
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_DATAGRAM_REMOVE_MAP: {
      VMCIDatagramMapInfo mapInfo;
      VMCIDatagramMapInfo *info = (VMCIDatagramMapInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_REMOVE_MAP only valid for contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&mapInfo, (void *)ioarg, sizeof mapInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      result = VMCIDatagramRemoveWellKnownMap(mapInfo.wellKnownID, cid);
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_CTX_ADD_NOTIFICATION: {
      VMCINotifyAddRemoveInfo arInfo;
      VMCINotifyAddRemoveInfo *info = (VMCINotifyAddRemoveInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_CTX_ADD_NOTIFICATION only valid for contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&arInfo, (void *)ioarg, sizeof arInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      result = VMCIContext_AddNotification(cid, arInfo.remoteCID);
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_CTX_REMOVE_NOTIFICATION: {
      VMCINotifyAddRemoveInfo arInfo;
      VMCINotifyAddRemoveInfo *info = (VMCINotifyAddRemoveInfo *)ioarg;
      int32 result;
      VMCIId cid;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_CTX_REMOVE_NOTIFICATION only valid for "
	     "contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&arInfo, (void *)ioarg, sizeof arInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      result = VMCIContext_RemoveNotification(cid, arInfo.remoteCID);
      retval = copy_to_user(&info->result, &result, sizeof result);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_CTX_GET_CPT_STATE: {
      VMCICptBufInfo getInfo;
      VMCIId cid;
      char *cptBuf;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_CTX_GET_CPT_STATE only valid for "
	     "contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&getInfo, (void *)ioarg, sizeof getInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      getInfo.result = VMCIContext_GetCheckpointState(cid, getInfo.cptType,
						      &getInfo.bufSize,
						      &cptBuf);
      if (getInfo.result == VMCI_SUCCESS && getInfo.bufSize) {
	 retval = copy_to_user((void *)(VA)getInfo.cptBuf, cptBuf,
			       getInfo.bufSize);
	 VMCI_FreeKernelMem(cptBuf, getInfo.bufSize);
	 if (retval) {
	    retval = -EFAULT;
	    break;
	 }
      }
      retval = copy_to_user((void *)ioarg, &getInfo, sizeof getInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_CTX_SET_CPT_STATE: {
      VMCICptBufInfo setInfo;
      VMCIId cid;
      char *cptBuf;

      if (vmciLinux->ctType != VMCIOBJ_CONTEXT) {
         Log("VMCI: IOCTL_VMCI_CTX_SET_CPT_STATE only valid for "
	     "contexts.\n");
         retval = -EINVAL;
         break;
      }

      retval = copy_from_user(&setInfo, (void *)ioarg, sizeof setInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }

      cptBuf = VMCI_AllocKernelMem(setInfo.bufSize, VMCI_MEMORY_NORMAL);
      if (cptBuf == NULL) {
         Log("VMCI: Cannot allocate memory to set cpt state of type %d.\n",
	     setInfo.cptType);
         retval = -ENOMEM;
         break;
      }
      retval = copy_from_user(cptBuf, (void *)(VA)setInfo.cptBuf,
			      setInfo.bufSize);
      if (retval) {
	 VMCI_FreeKernelMem(cptBuf, setInfo.bufSize);
         retval = -EFAULT;
         break;
      }

      cid = VMCIContext_GetId(vmciLinux->ct.context);
      setInfo.result = VMCIContext_SetCheckpointState(cid, setInfo.cptType,
						      setInfo.bufSize, cptBuf);
      VMCI_FreeKernelMem(cptBuf, setInfo.bufSize);
      retval = copy_to_user((void *)ioarg, &setInfo, sizeof setInfo);
      if (retval) {
         retval = -EFAULT;
         break;
      }
      break;
   }

   case IOCTL_VMCI_GET_CONTEXT_ID: {
      VMCIId cid = VMCI_HOST_CONTEXT_ID;
      retval = copy_to_user((void *)ioarg, &cid, sizeof cid);
      break;
   }

   default:
      Warning("Unknown ioctl %d\n", iocmd);
      retval = -EINVAL;
   }

   return retval;
}


#if defined(HAVE_COMPAT_IOCTL) || defined(HAVE_UNLOCKED_IOCTL)
/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriver_UnlockedIoctl --
 *
 *      Wrapper for LinuxDriver_Ioctl supporting the compat_ioctl and
 *      unlocked_ioctl methods that have signatures different from the
 *      old ioctl. Used as compat_ioctl method for 32bit apps running
 *      on 64bit kernel and for unlocked_ioctl on systems supporting
 *      those.  LinuxDriver_Ioctl may safely be called without holding
 *      the BKL.
 *
 * Results:
 *      Same as LinuxDriver_Ioctl.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static long
LinuxDriver_UnlockedIoctl(struct file *filp,
                          u_int iocmd,
                          unsigned long ioarg)
{
   return LinuxDriver_Ioctl(NULL, filp, iocmd, ioarg);
}
#endif


MODULE_AUTHOR("VMware, Inc.");
MODULE_DESCRIPTION("VMware Virtual Machine Communication Interface (VMCI).");
MODULE_LICENSE("GPL v2");
