/*
 * crypt32 certificate chain functions tests
 *
 * Copyright 2006 Juan Lang
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */
#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <windef.h>
#include <winbase.h>
#include <winerror.h>
#include <wincrypt.h>

#include "wine/test.h"

static const BYTE selfSignedCert[] = {
 0x30, 0x82, 0x01, 0x1f, 0x30, 0x81, 0xce, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
 0x10, 0xeb, 0x0d, 0x57, 0x2a, 0x9c, 0x09, 0xba, 0xa4, 0x4a, 0xb7, 0x25, 0x49,
 0xd9, 0x3e, 0xb5, 0x73, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d,
 0x05, 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03,
 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30,
 0x1e, 0x17, 0x0d, 0x30, 0x36, 0x30, 0x36, 0x32, 0x39, 0x30, 0x35, 0x30, 0x30,
 0x34, 0x36, 0x5a, 0x17, 0x0d, 0x30, 0x37, 0x30, 0x36, 0x32, 0x39, 0x31, 0x31,
 0x30, 0x30, 0x34, 0x36, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e,
 0x67, 0x00, 0x30, 0x5c, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x4b, 0x00, 0x30, 0x48, 0x02, 0x41,
 0x00, 0xe2, 0x54, 0x3a, 0xa7, 0x83, 0xb1, 0x27, 0x14, 0x3e, 0x59, 0xbb, 0xb4,
 0x53, 0xe6, 0x1f, 0xe7, 0x5d, 0xf1, 0x21, 0x68, 0xad, 0x85, 0x53, 0xdb, 0x6b,
 0x1e, 0xeb, 0x65, 0x97, 0x03, 0x86, 0x60, 0xde, 0xf3, 0x6c, 0x38, 0x75, 0xe0,
 0x4c, 0x61, 0xbb, 0xbc, 0x62, 0x17, 0xa9, 0xcd, 0x79, 0x3f, 0x21, 0x4e, 0x96,
 0xcb, 0x0e, 0xdc, 0x61, 0x94, 0x30, 0x18, 0x10, 0x6b, 0xd0, 0x1c, 0x10, 0x79,
 0x02, 0x03, 0x01, 0x00, 0x01, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02,
 0x1d, 0x05, 0x00, 0x03, 0x41, 0x00, 0x25, 0x90, 0x53, 0x34, 0xd9, 0x56, 0x41,
 0x5e, 0xdb, 0x7e, 0x01, 0x36, 0xec, 0x27, 0x61, 0x5e, 0xb7, 0x4d, 0x90, 0x66,
 0xa2, 0xe1, 0x9d, 0x58, 0x76, 0xd4, 0x9c, 0xba, 0x2c, 0x84, 0xc6, 0x83, 0x7a,
 0x22, 0x0d, 0x03, 0x69, 0x32, 0x1a, 0x6d, 0xcb, 0x0c, 0x15, 0xb3, 0x6b, 0xc7,
 0x0a, 0x8c, 0xb4, 0x5c, 0x34, 0x78, 0xe0, 0x3c, 0x9c, 0xe9, 0xf3, 0x30, 0x9f,
 0xa8, 0x76, 0x57, 0x92, 0x36 };

static void testCreateCertChainEngine(void)
{
    BOOL ret;
    CERT_CHAIN_ENGINE_CONFIG config = { 0 };
    HCERTCHAINENGINE engine;
    HCERTSTORE store;

    /* Crash
    ret = CertCreateCertificateChainEngine(NULL, NULL);
    ret = CertCreateCertificateChainEngine(NULL, &engine);
     */
    ret = CertCreateCertificateChainEngine(&config, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertCreateCertificateChainEngine(&config, &engine);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    /* Crashes
    config.cbSize = sizeof(config);
    ret = CertCreateCertificateChainEngine(&config, NULL);
     */
    config.cbSize = sizeof(config);
    ret = CertCreateCertificateChainEngine(&config, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08x\n", GetLastError());
    CertFreeCertificateChainEngine(engine);
    config.dwFlags = 0xff000000;
    ret = CertCreateCertificateChainEngine(&config, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08x\n", GetLastError());
    CertFreeCertificateChainEngine(engine);

    /* Creating a cert with no root certs at all is allowed.. */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    config.hRestrictedRoot = store;
    ret = CertCreateCertificateChainEngine(&config, &engine);
    ok(ret, "CertCreateCertificateChainEngine failed: %08x\n", GetLastError());
    CertFreeCertificateChainEngine(engine);

    /* but creating one with a restricted root with a cert that isn't a member
     * of the Root store isn't allowed.
     */
    CertAddEncodedCertificateToStore(store, X509_ASN_ENCODING, selfSignedCert,
     sizeof(selfSignedCert), CERT_STORE_ADD_ALWAYS, NULL);
    ret = CertCreateCertificateChainEngine(&config, &engine);
    ok(!ret && GetLastError() == CRYPT_E_NOT_FOUND,
     "Expected CRYPT_E_NOT_FOUND, got %08x\n", GetLastError());

    CertCloseStore(store, 0);
}

static const BYTE bigCert[] = { 0x30, 0x7a, 0x02, 0x01, 0x01, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x22,
 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30,
 0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30,
 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20,
 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x07, 0x30, 0x02, 0x06, 0x00, 0x03, 0x01,
 0x00, 0xa3, 0x16, 0x30, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01 };
static const BYTE iTunesCert0[] = {
0x30,0x82,0x03,0xc4,0x30,0x82,0x03,0x2d,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x47,0xbf,0x19,0x95,0xdf,0x8d,0x52,0x46,0x43,0xf7,0xdb,0x6d,0x48,0x0d,0x31,
0xa4,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x81,0x8b,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x5a,0x41,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x08,0x13,0x0c,0x57,0x65,
0x73,0x74,0x65,0x72,0x6e,0x20,0x43,0x61,0x70,0x65,0x31,0x14,0x30,0x12,0x06,
0x03,0x55,0x04,0x07,0x13,0x0b,0x44,0x75,0x72,0x62,0x61,0x6e,0x76,0x69,0x6c,
0x6c,0x65,0x31,0x0f,0x30,0x0d,0x06,0x03,0x55,0x04,0x0a,0x13,0x06,0x54,0x68,
0x61,0x77,0x74,0x65,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,0x04,0x0b,0x13,0x14,
0x54,0x68,0x61,0x77,0x74,0x65,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,
0x61,0x74,0x69,0x6f,0x6e,0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x03,0x13,
0x16,0x54,0x68,0x61,0x77,0x74,0x65,0x20,0x54,0x69,0x6d,0x65,0x73,0x74,0x61,
0x6d,0x70,0x69,0x6e,0x67,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x33,0x31,
0x32,0x30,0x34,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x33,0x31,
0x32,0x30,0x33,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x53,0x31,0x0b,0x30,
0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,0x06,
0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x2c,
0x20,0x49,0x6e,0x63,0x2e,0x31,0x2b,0x30,0x29,0x06,0x03,0x55,0x04,0x03,0x13,
0x22,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,0x54,0x69,0x6d,0x65,0x20,
0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,0x53,0x65,0x72,0x76,0x69,0x63,
0x65,0x73,0x20,0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,
0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xa9,0xca,0xb2,0xa4,0xcc,0xcd,0x20,
0xaf,0x0a,0x7d,0x89,0xac,0x87,0x75,0xf0,0xb4,0x4e,0xf1,0xdf,0xc1,0x0f,0xbf,
0x67,0x61,0xbd,0xa3,0x64,0x1c,0xda,0xbb,0xf9,0xca,0x33,0xab,0x84,0x30,0x89,
0x58,0x7e,0x8c,0xdb,0x6b,0xdd,0x36,0x9e,0x0f,0xbf,0xd1,0xec,0x78,0xf2,0x77,
0xa6,0x7e,0x6f,0x3c,0xbf,0x93,0xaf,0x0d,0xba,0x68,0xf4,0x6c,0x94,0xca,0xbd,
0x52,0x2d,0xab,0x48,0x3d,0xf5,0xb6,0xd5,0x5d,0x5f,0x1b,0x02,0x9f,0xfa,0x2f,
0x6b,0x1e,0xa4,0xf7,0xa3,0x9a,0xa6,0x1a,0xc8,0x02,0xe1,0x7f,0x4c,0x52,0xe3,
0x0e,0x60,0xec,0x40,0x1c,0x7e,0xb9,0x0d,0xde,0x3f,0xc7,0xb4,0xdf,0x87,0xbd,
0x5f,0x7a,0x6a,0x31,0x2e,0x03,0x99,0x81,0x13,0xa8,0x47,0x20,0xce,0x31,0x73,
0x0d,0x57,0x2d,0xcd,0x78,0x34,0x33,0x95,0x12,0x99,0x12,0xb9,0xde,0x68,0x2f,
0xaa,0xe6,0xe3,0xc2,0x8a,0x8c,0x2a,0xc3,0x8b,0x21,0x87,0x66,0xbd,0x83,0x58,
0x57,0x6f,0x75,0xbf,0x3c,0xaa,0x26,0x87,0x5d,0xca,0x10,0x15,0x3c,0x9f,0x84,
0xea,0x54,0xc1,0x0a,0x6e,0xc4,0xfe,0xc5,0x4a,0xdd,0xb9,0x07,0x11,0x97,0x22,
0x7c,0xdb,0x3e,0x27,0xd1,0x1e,0x78,0xec,0x9f,0x31,0xc9,0xf1,0xe6,0x22,0x19,
0xdb,0xc4,0xb3,0x47,0x43,0x9a,0x1a,0x5f,0xa0,0x1e,0x90,0xe4,0x5e,0xf5,0xee,
0x7c,0xf1,0x7d,0xab,0x62,0x01,0x8f,0xf5,0x4d,0x0b,0xde,0xd0,0x22,0x56,0xa8,
0x95,0xcd,0xae,0x88,0x76,0xae,0xee,0xba,0x0d,0xf3,0xe4,0x4d,0xd9,0xa0,0xfb,
0x68,0xa0,0xae,0x14,0x3b,0xb3,0x87,0xc1,0xbb,0x02,0x03,0x01,0x00,0x01,0xa3,
0x81,0xdb,0x30,0x81,0xd8,0x30,0x34,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x01,0x01,0x04,0x28,0x30,0x26,0x30,0x24,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,
0x07,0x30,0x01,0x86,0x18,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,
0x70,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x30,
0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,
0xff,0x02,0x01,0x00,0x30,0x41,0x06,0x03,0x55,0x1d,0x1f,0x04,0x3a,0x30,0x38,
0x30,0x36,0xa0,0x34,0xa0,0x32,0x86,0x30,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,
0x63,0x72,0x6c,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,
0x6d,0x2f,0x54,0x68,0x61,0x77,0x74,0x65,0x54,0x69,0x6d,0x65,0x73,0x74,0x61,
0x6d,0x70,0x69,0x6e,0x67,0x43,0x41,0x2e,0x63,0x72,0x6c,0x30,0x13,0x06,0x03,
0x55,0x1d,0x25,0x04,0x0c,0x30,0x0a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,
0x03,0x08,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
0x02,0x01,0x06,0x30,0x24,0x06,0x03,0x55,0x1d,0x11,0x04,0x1d,0x30,0x1b,0xa4,
0x19,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0c,0x54,
0x53,0x41,0x32,0x30,0x34,0x38,0x2d,0x31,0x2d,0x35,0x33,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,
0x4a,0x6b,0xf9,0xea,0x58,0xc2,0x44,0x1c,0x31,0x89,0x79,0x99,0x2b,0x96,0xbf,
0x82,0xac,0x01,0xd6,0x1c,0x4c,0xcd,0xb0,0x8a,0x58,0x6e,0xdf,0x08,0x29,0xa3,
0x5e,0xc8,0xca,0x93,0x13,0xe7,0x04,0x52,0x0d,0xef,0x47,0x27,0x2f,0x00,0x38,
0xb0,0xe4,0xc9,0x93,0x4e,0x9a,0xd4,0x22,0x62,0x15,0xf7,0x3f,0x37,0x21,0x4f,
0x70,0x31,0x80,0xf1,0x8b,0x38,0x87,0xb3,0xe8,0xe8,0x97,0x00,0xfe,0xcf,0x55,
0x96,0x4e,0x24,0xd2,0xa9,0x27,0x4e,0x7a,0xae,0xb7,0x61,0x41,0xf3,0x2a,0xce,
0xe7,0xc9,0xd9,0x5e,0xdd,0xbb,0x2b,0x85,0x3e,0xb5,0x9d,0xb5,0xd9,0xe1,0x57,
0xff,0xbe,0xb4,0xc5,0x7e,0xf5,0xcf,0x0c,0x9e,0xf0,0x97,0xfe,0x2b,0xd3,0x3b,
0x52,0x1b,0x1b,0x38,0x27,0xf7,0x3f,0x4a };
static const BYTE iTunesCert1[] = {
0x30,0x82,0x03,0xff,0x30,0x82,0x02,0xe7,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x0d,0xe9,0x2b,0xf0,0xd4,0xd8,0x29,0x88,0x18,0x32,0x05,0x09,0x5e,0x9a,0x76,
0x88,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x53,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,
0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,
0x69,0x53,0x69,0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x2b,0x30,0x29,
0x06,0x03,0x55,0x04,0x03,0x13,0x22,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,
0x20,0x54,0x69,0x6d,0x65,0x20,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,
0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,
0x30,0x33,0x31,0x32,0x30,0x34,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,
0x30,0x38,0x31,0x32,0x30,0x33,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x57,
0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,
0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,
0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x2f,0x30,0x2d,0x06,0x03,0x55,
0x04,0x03,0x13,0x26,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,0x54,0x69,
0x6d,0x65,0x20,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,0x53,0x65,0x72,
0x76,0x69,0x63,0x65,0x73,0x20,0x53,0x69,0x67,0x6e,0x65,0x72,0x30,0x82,0x01,
0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,
0xb2,0x50,0x28,0x48,0xdd,0xd3,0x68,0x7a,0x84,0x18,0x44,0x66,0x75,0x5d,0x7e,
0xc4,0xb8,0x9f,0x63,0x26,0xff,0x3d,0x43,0x9c,0x7c,0x11,0x38,0x10,0x25,0x55,
0x73,0xd9,0x75,0x27,0x69,0xfd,0x4e,0xb9,0x20,0x5c,0xd3,0x0a,0xf9,0xa0,0x1b,
0x2a,0xed,0x55,0x56,0x21,0x61,0xd8,0x1e,0xdb,0xe4,0xbc,0x33,0x6b,0xc7,0xef,
0xdd,0xa3,0x37,0x65,0x8e,0x1b,0x93,0x0c,0xb6,0x53,0x1e,0x5c,0x7c,0x66,0x35,
0x5f,0x05,0x8a,0x45,0xfe,0x76,0x4e,0xdf,0x53,0x80,0xa2,0x81,0x20,0x9d,0xae,
0x88,0x5c,0xa2,0x08,0xf7,0xe5,0x30,0xf9,0xee,0x22,0x37,0x4c,0x42,0x0a,0xce,
0xdf,0xc6,0x1f,0xc4,0xd6,0x55,0xe9,0x81,0x3f,0xb5,0x52,0xa3,0x2c,0xaa,0x01,
0x7a,0xf2,0xa2,0xaa,0x8d,0x35,0xfe,0x9f,0xe6,0x5d,0x6a,0x05,0x9f,0x3d,0x6b,
0xe3,0xbf,0x96,0xc0,0xfe,0xcc,0x60,0xf9,0x40,0xe7,0x07,0xa0,0x44,0xeb,0x81,
0x51,0x6e,0xa5,0x2a,0xf2,0xb6,0x8a,0x10,0x28,0xed,0x8f,0xdc,0x06,0xa0,0x86,
0x50,0x9a,0x7b,0x4a,0x08,0x0d,0x30,0x1d,0xca,0x10,0x9e,0x6b,0xf7,0xe9,0x58,
0xae,0x04,0xa9,0x40,0x99,0xb2,0x28,0xe8,0x8f,0x16,0xac,0x3c,0xe3,0x53,0x6f,
0x4b,0xd3,0x35,0x9d,0xb5,0x6f,0x64,0x1d,0xb3,0x96,0x2c,0xbb,0x3d,0xe7,0x79,
0xeb,0x6d,0x7a,0xf9,0x16,0xe6,0x26,0xad,0xaf,0xef,0x99,0x53,0xb7,0x40,0x2c,
0x95,0xb8,0x79,0xaa,0xfe,0xd4,0x52,0xab,0x29,0x74,0x7e,0x42,0xec,0x39,0x1e,
0xa2,0x6a,0x16,0xe6,0x59,0xbb,0x24,0x68,0xd8,0x00,0x80,0x43,0x10,0x87,0x80,
0x6b,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xca,0x30,0x81,0xc7,0x30,0x34,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x28,0x30,0x26,0x30,0x24,
0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x18,0x68,0x74,0x74,
0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,
0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x02,0x30,0x00,0x30,0x33,0x06,0x03,0x55,0x1d,0x1f,0x04,0x2c,0x30,
0x2a,0x30,0x28,0xa0,0x26,0xa0,0x24,0x86,0x22,0x68,0x74,0x74,0x70,0x3a,0x2f,
0x2f,0x63,0x72,0x6c,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,
0x6f,0x6d,0x2f,0x74,0x73,0x73,0x2d,0x63,0x61,0x2e,0x63,0x72,0x6c,0x30,0x16,
0x06,0x03,0x55,0x1d,0x25,0x01,0x01,0xff,0x04,0x0c,0x30,0x0a,0x06,0x08,0x2b,
0x06,0x01,0x05,0x05,0x07,0x03,0x08,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
0x01,0xff,0x04,0x04,0x03,0x02,0x06,0xc0,0x30,0x24,0x06,0x03,0x55,0x1d,0x11,
0x04,0x1d,0x30,0x1b,0xa4,0x19,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,
0x04,0x03,0x13,0x0c,0x54,0x53,0x41,0x32,0x30,0x34,0x38,0x2d,0x31,0x2d,0x35,
0x34,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x03,0x82,0x01,0x01,0x00,0x87,0x78,0x70,0xda,0x4e,0x52,0x01,0x20,0x5b,
0xe0,0x79,0xc9,0x82,0x30,0xc4,0xfd,0xb9,0x19,0x96,0xbd,0x91,0x00,0xc3,0xbd,
0xcd,0xcd,0xc6,0xf4,0x0e,0xd8,0xff,0xf9,0x4d,0xc0,0x33,0x62,0x30,0x11,0xc5,
0xf5,0x74,0x1b,0xd4,0x92,0xde,0x5f,0x9c,0x20,0x13,0xb1,0x7c,0x45,0xbe,0x50,
0xcd,0x83,0xe7,0x80,0x17,0x83,0xa7,0x27,0x93,0x67,0x13,0x46,0xfb,0xca,0xb8,
0x98,0x41,0x03,0xcc,0x9b,0x51,0x5b,0x05,0x8b,0x7f,0xa8,0x6f,0xf3,0x1b,0x50,
0x1b,0x24,0x2e,0xf2,0x69,0x8d,0x6c,0x22,0xf7,0xbb,0xca,0x16,0x95,0xed,0x0c,
0x74,0xc0,0x68,0x77,0xd9,0xeb,0x99,0x62,0x87,0xc1,0x73,0x90,0xf8,0x89,0x74,
0x7a,0x23,0xab,0xa3,0x98,0x7b,0x97,0xb1,0xf7,0x8f,0x29,0x71,0x4d,0x2e,0x75,
0x1b,0x48,0x41,0xda,0xf0,0xb5,0x0d,0x20,0x54,0xd6,0x77,0xa0,0x97,0x82,0x63,
0x69,0xfd,0x09,0xcf,0x8a,0xf0,0x75,0xbb,0x09,0x9b,0xd9,0xf9,0x11,0x55,0x26,
0x9a,0x61,0x32,0xbe,0x7a,0x02,0xb0,0x7b,0x86,0xbe,0xa2,0xc3,0x8b,0x22,0x2c,
0x78,0xd1,0x35,0x76,0xbc,0x92,0x73,0x5c,0xf9,0xb9,0xe6,0x4c,0x15,0x0a,0x23,
0xcc,0xe4,0xd2,0xd4,0x34,0x2e,0x49,0x40,0x15,0x3c,0x0f,0x60,0x7a,0x24,0xc6,
0xa5,0x66,0xef,0x96,0xcf,0x70,0xeb,0x3e,0xe7,0xf4,0x0d,0x7e,0xdc,0xd1,0x7c,
0xa3,0x76,0x71,0x69,0xc1,0x9c,0x4f,0x47,0x30,0x35,0x21,0xb1,0xa2,0xaf,0x1a,
0x62,0x3c,0x2b,0xd9,0x8e,0xaa,0x2a,0x07,0x7b,0xd8,0x18,0xb3,0x5c,0x7b,0xe2,
0x9d,0xa5,0x6f,0xfe,0x3c,0x89,0xad };
static const BYTE iTunesCert2[] = {
0x30,0x82,0x04,0xbf,0x30,0x82,0x04,0x28,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x41,0x91,0xa1,0x5a,0x39,0x78,0xdf,0xcf,0x49,0x65,0x66,0x38,0x1d,0x4c,0x75,
0xc2,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x5f,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,
0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,
0x69,0x53,0x69,0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x37,0x30,0x35,
0x06,0x03,0x55,0x04,0x0b,0x13,0x2e,0x43,0x6c,0x61,0x73,0x73,0x20,0x33,0x20,
0x50,0x75,0x62,0x6c,0x69,0x63,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,0x20,
0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,0x41,
0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,0x1e,0x17,0x0d,0x30,0x34,0x30,
0x37,0x31,0x36,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x34,0x30,
0x37,0x31,0x35,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x81,0xb4,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,
0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,
0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x1f,0x30,0x1d,0x06,0x03,0x55,0x04,0x0b,
0x13,0x16,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,0x54,0x72,0x75,0x73,
0x74,0x20,0x4e,0x65,0x74,0x77,0x6f,0x72,0x6b,0x31,0x3b,0x30,0x39,0x06,0x03,
0x55,0x04,0x0b,0x13,0x32,0x54,0x65,0x72,0x6d,0x73,0x20,0x6f,0x66,0x20,0x75,
0x73,0x65,0x20,0x61,0x74,0x20,0x68,0x74,0x74,0x70,0x73,0x3a,0x2f,0x2f,0x77,
0x77,0x77,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,
0x2f,0x72,0x70,0x61,0x20,0x28,0x63,0x29,0x30,0x34,0x31,0x2e,0x30,0x2c,0x06,
0x03,0x55,0x04,0x03,0x13,0x25,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,
0x43,0x6c,0x61,0x73,0x73,0x20,0x33,0x20,0x43,0x6f,0x64,0x65,0x20,0x53,0x69,
0x67,0x6e,0x69,0x6e,0x67,0x20,0x32,0x30,0x30,0x34,0x20,0x43,0x41,0x30,0x82,
0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,
0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,
0x00,0xbe,0xbc,0xee,0xbc,0x7e,0xef,0x83,0xeb,0xe0,0x37,0x4f,0xfb,0x03,0x10,
0x38,0xbe,0x08,0xd2,0x8c,0x7d,0x9d,0xfa,0x92,0x7f,0x19,0x0c,0xc2,0x6b,0xee,
0x42,0x52,0x8c,0xde,0xd3,0x1c,0x48,0x13,0x25,0xea,0xc1,0x63,0x7a,0xf9,0x51,
0x65,0xee,0xd3,0xaa,0x3b,0xf5,0xf0,0x94,0x9c,0x2b,0xfb,0xf2,0x66,0xd4,0x24,
0xda,0xf7,0xf5,0x9f,0x6e,0x19,0x39,0x36,0xbc,0xd0,0xa3,0x76,0x08,0x1e,0x22,
0x27,0x24,0x6c,0x38,0x91,0x27,0xe2,0x84,0x49,0xae,0x1b,0x8a,0xa1,0xfd,0x25,
0x82,0x2c,0x10,0x30,0xe8,0x71,0xab,0x28,0xe8,0x77,0x4a,0x51,0xf1,0xec,0xcd,
0xf8,0xf0,0x54,0xd4,0x6f,0xc0,0xe3,0x6d,0x0a,0x8f,0xd9,0xd8,0x64,0x8d,0x63,
0xb2,0x2d,0x4e,0x27,0xf6,0x85,0x0e,0xfe,0x6d,0xe3,0x29,0x99,0xe2,0x85,0x47,
0x7c,0x2d,0x86,0x7f,0xe8,0x57,0x8f,0xad,0x67,0xc2,0x33,0x32,0x91,0x13,0x20,
0xfc,0xa9,0x23,0x14,0x9a,0x6d,0xc2,0x84,0x4b,0x76,0x68,0x04,0xd5,0x71,0x2c,
0x5d,0x21,0xfa,0x88,0x0d,0x26,0xfd,0x1f,0x2d,0x91,0x2b,0xe7,0x01,0x55,0x4d,
0xf2,0x6d,0x35,0x28,0x82,0xdf,0xd9,0x6b,0x5c,0xb6,0xd6,0xd9,0xaa,0x81,0xfd,
0x5f,0xcd,0x83,0xba,0x63,0x9d,0xd0,0x22,0xfc,0xa9,0x3b,0x42,0x69,0xb2,0x8e,
0x3a,0xb5,0xbc,0xb4,0x9e,0x0f,0x5e,0xc4,0xea,0x2c,0x82,0x8b,0x28,0xfd,0x53,
0x08,0x96,0xdd,0xb5,0x01,0x20,0xd1,0xf9,0xa5,0x18,0xe7,0xc0,0xee,0x51,0x70,
0x37,0xe1,0xb6,0x05,0x48,0x52,0x48,0x6f,0x38,0xea,0xc3,0xe8,0x6c,0x7b,0x44,
0x84,0xbb,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0xa0,0x30,0x82,0x01,0x9c,
0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,
0x01,0xff,0x02,0x01,0x00,0x30,0x44,0x06,0x03,0x55,0x1d,0x20,0x04,0x3d,0x30,
0x3b,0x30,0x39,0x06,0x0b,0x60,0x86,0x48,0x01,0x86,0xf8,0x45,0x01,0x07,0x17,
0x03,0x30,0x2a,0x30,0x28,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,
0x16,0x1c,0x68,0x74,0x74,0x70,0x73,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x76,
0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x2f,0x72,0x70,0x61,
0x30,0x31,0x06,0x03,0x55,0x1d,0x1f,0x04,0x2a,0x30,0x28,0x30,0x26,0xa0,0x24,
0xa0,0x22,0x86,0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,
0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x2f,0x70,0x63,
0x61,0x33,0x2e,0x63,0x72,0x6c,0x30,0x1d,0x06,0x03,0x55,0x1d,0x25,0x04,0x16,
0x30,0x14,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x02,0x06,0x08,0x2b,
0x06,0x01,0x05,0x05,0x07,0x03,0x03,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x11,0x06,0x09,0x60,0x86,0x48,
0x01,0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x00,0x01,0x30,0x29,0x06,
0x03,0x55,0x1d,0x11,0x04,0x22,0x30,0x20,0xa4,0x1e,0x30,0x1c,0x31,0x1a,0x30,
0x18,0x06,0x03,0x55,0x04,0x03,0x13,0x11,0x43,0x6c,0x61,0x73,0x73,0x33,0x43,
0x41,0x32,0x30,0x34,0x38,0x2d,0x31,0x2d,0x34,0x33,0x30,0x1d,0x06,0x03,0x55,
0x1d,0x0e,0x04,0x16,0x04,0x14,0x08,0xf5,0x51,0xe8,0xfb,0xfe,0x3d,0x3d,0x64,
0x36,0x7c,0x68,0xcf,0x5b,0x78,0xa8,0xdf,0xb9,0xc5,0x37,0x30,0x81,0x80,0x06,
0x03,0x55,0x1d,0x23,0x04,0x79,0x30,0x77,0xa1,0x63,0xa4,0x61,0x30,0x5f,0x31,
0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,
0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,
0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x37,0x30,0x35,0x06,0x03,0x55,0x04,
0x0b,0x13,0x2e,0x43,0x6c,0x61,0x73,0x73,0x20,0x33,0x20,0x50,0x75,0x62,0x6c,
0x69,0x63,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,0x20,0x43,0x65,0x72,0x74,
0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,0x41,0x75,0x74,0x68,0x6f,
0x72,0x69,0x74,0x79,0x82,0x10,0x70,0xba,0xe4,0x1d,0x10,0xd9,0x29,0x34,0xb6,
0x38,0xca,0x7b,0x03,0xcc,0xba,0xbf,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0xae,0x3a,0x17,0xb8,
0x4a,0x7b,0x55,0xfa,0x64,0x55,0xec,0x40,0xa4,0xed,0x49,0x41,0x90,0x99,0x9c,
0x89,0xbc,0xaf,0x2e,0x1d,0xca,0x78,0x23,0xf9,0x1c,0x19,0x0f,0x7f,0xeb,0x68,
0xbc,0x32,0xd9,0x88,0x38,0xde,0xdc,0x3f,0xd3,0x89,0xb4,0x3f,0xb1,0x82,0x96,
0xf1,0xa4,0x5a,0xba,0xed,0x2e,0x26,0xd3,0xde,0x7c,0x01,0x6e,0x00,0x0a,0x00,
0xa4,0x06,0x92,0x11,0x48,0x09,0x40,0xf9,0x1c,0x18,0x79,0x67,0x23,0x24,0xe0,
0xbb,0xd5,0xe1,0x50,0xae,0x1b,0xf5,0x0e,0xdd,0xe0,0x2e,0x81,0xcd,0x80,0xa3,
0x6c,0x52,0x4f,0x91,0x75,0x55,0x8a,0xba,0x22,0xf2,0xd2,0xea,0x41,0x75,0x88,
0x2f,0x63,0x55,0x7d,0x1e,0x54,0x5a,0x95,0x59,0xca,0xd9,0x34,0x81,0xc0,0x5f,
0x5e,0xf6,0x7a,0xb5 };
static const BYTE iTunesCert3[] = {
0x30,0x82,0x04,0xf1,0x30,0x82,0x03,0xd9,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x0f,0x1a,0xa0,0xe0,0x9b,0x9b,0x61,0xa6,0xb6,0xfe,0x40,0xd2,0xdf,0x6a,0xf6,
0x8d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x81,0xb4,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
0x55,0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,
0x72,0x69,0x53,0x69,0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x1f,0x30,
0x1d,0x06,0x03,0x55,0x04,0x0b,0x13,0x16,0x56,0x65,0x72,0x69,0x53,0x69,0x67,
0x6e,0x20,0x54,0x72,0x75,0x73,0x74,0x20,0x4e,0x65,0x74,0x77,0x6f,0x72,0x6b,
0x31,0x3b,0x30,0x39,0x06,0x03,0x55,0x04,0x0b,0x13,0x32,0x54,0x65,0x72,0x6d,
0x73,0x20,0x6f,0x66,0x20,0x75,0x73,0x65,0x20,0x61,0x74,0x20,0x68,0x74,0x74,
0x70,0x73,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,
0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x2f,0x72,0x70,0x61,0x20,0x28,0x63,0x29,0x30,
0x34,0x31,0x2e,0x30,0x2c,0x06,0x03,0x55,0x04,0x03,0x13,0x25,0x56,0x65,0x72,
0x69,0x53,0x69,0x67,0x6e,0x20,0x43,0x6c,0x61,0x73,0x73,0x20,0x33,0x20,0x43,
0x6f,0x64,0x65,0x20,0x53,0x69,0x67,0x6e,0x69,0x6e,0x67,0x20,0x32,0x30,0x30,
0x34,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x36,0x30,0x31,0x31,0x37,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x38,0x30,0x31,0x32,0x32,0x32,
0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x81,0xb4,0x31,0x0b,0x30,0x09,0x06,0x03,
0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,
0x08,0x13,0x0a,0x43,0x61,0x6c,0x69,0x66,0x6f,0x72,0x6e,0x69,0x61,0x31,0x12,
0x30,0x10,0x06,0x03,0x55,0x04,0x07,0x13,0x09,0x43,0x75,0x70,0x65,0x72,0x74,
0x69,0x6e,0x6f,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,0x04,0x0a,0x14,0x14,0x41,
0x70,0x70,0x6c,0x65,0x20,0x43,0x6f,0x6d,0x70,0x75,0x74,0x65,0x72,0x2c,0x20,
0x49,0x6e,0x63,0x2e,0x31,0x3e,0x30,0x3c,0x06,0x03,0x55,0x04,0x0b,0x13,0x35,
0x44,0x69,0x67,0x69,0x74,0x61,0x6c,0x20,0x49,0x44,0x20,0x43,0x6c,0x61,0x73,
0x73,0x20,0x33,0x20,0x2d,0x20,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x20,0x53,0x6f,0x66,0x74,0x77,0x61,0x72,0x65,0x20,0x56,0x61,0x6c,0x69,0x64,
0x61,0x74,0x69,0x6f,0x6e,0x20,0x76,0x32,0x31,0x1d,0x30,0x1b,0x06,0x03,0x55,
0x04,0x03,0x14,0x14,0x41,0x70,0x70,0x6c,0x65,0x20,0x43,0x6f,0x6d,0x70,0x75,
0x74,0x65,0x72,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xd3,0xab,0x3b,0x7f,0xec,0x48,0x84,
0xce,0xa8,0x1a,0x12,0xf3,0x3c,0x87,0xcb,0x24,0x58,0x96,0x02,0x87,0x66,0x49,
0xeb,0x89,0xee,0x79,0x44,0x70,0x8d,0xe7,0xd4,0x1f,0x30,0x92,0xc0,0x9c,0x35,
0x78,0xc0,0xaf,0x1c,0xb6,0x28,0xd3,0xe0,0xe0,0x9d,0xd3,0x49,0x76,0x73,0x57,
0x19,0x4d,0x8d,0x70,0x85,0x64,0x4d,0x1d,0xc6,0x02,0x3e,0xe5,0x2c,0x66,0x07,
0xd2,0x27,0x4b,0xd6,0xc8,0x3c,0x93,0xb6,0x15,0x0c,0xde,0x5b,0xd7,0x93,0xdd,
0xbe,0x85,0x62,0x34,0x17,0x8a,0x05,0x60,0xf0,0x8a,0x1c,0x5a,0x40,0x21,0x8d,
0x51,0x6c,0xb0,0x62,0xd8,0xb5,0xd4,0xf9,0xb1,0xd0,0x58,0x7a,0x7a,0x82,0x55,
0xb3,0xf9,0x53,0x71,0xde,0xd2,0xc9,0x37,0x8c,0xf6,0x5a,0x1f,0x2d,0xcd,0x7c,
0x67,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x7f,0x30,0x82,0x01,0x7b,0x30,
0x09,0x06,0x03,0x55,0x1d,0x13,0x04,0x02,0x30,0x00,0x30,0x0e,0x06,0x03,0x55,
0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x40,0x06,0x03,
0x55,0x1d,0x1f,0x04,0x39,0x30,0x37,0x30,0x35,0xa0,0x33,0xa0,0x31,0x86,0x2f,
0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x43,0x53,0x43,0x33,0x2d,0x32,0x30,0x30,
0x34,0x2d,0x63,0x72,0x6c,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,
0x63,0x6f,0x6d,0x2f,0x43,0x53,0x43,0x33,0x2d,0x32,0x30,0x30,0x34,0x2e,0x63,
0x72,0x6c,0x30,0x44,0x06,0x03,0x55,0x1d,0x20,0x04,0x3d,0x30,0x3b,0x30,0x39,
0x06,0x0b,0x60,0x86,0x48,0x01,0x86,0xf8,0x45,0x01,0x07,0x17,0x03,0x30,0x2a,
0x30,0x28,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x1c,0x68,
0x74,0x74,0x70,0x73,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x76,0x65,0x72,0x69,
0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x2f,0x72,0x70,0x61,0x30,0x13,0x06,
0x03,0x55,0x1d,0x25,0x04,0x0c,0x30,0x0a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,
0x07,0x03,0x03,0x30,0x75,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,
0x04,0x69,0x30,0x67,0x30,0x24,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,
0x01,0x86,0x18,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,
0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x30,0x3f,0x06,
0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x33,0x68,0x74,0x74,0x70,
0x3a,0x2f,0x2f,0x43,0x53,0x43,0x33,0x2d,0x32,0x30,0x30,0x34,0x2d,0x61,0x69,
0x61,0x2e,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6e,0x2e,0x63,0x6f,0x6d,0x2f,
0x43,0x53,0x43,0x33,0x2d,0x32,0x30,0x30,0x34,0x2d,0x61,0x69,0x61,0x2e,0x63,
0x65,0x72,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
0x08,0xf5,0x51,0xe8,0xfb,0xfe,0x3d,0x3d,0x64,0x36,0x7c,0x68,0xcf,0x5b,0x78,
0xa8,0xdf,0xb9,0xc5,0x37,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,
0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x04,0x10,0x30,0x16,0x06,0x0a,0x2b,0x06,
0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x1b,0x04,0x08,0x30,0x06,0x01,0x01,0x00,
0x01,0x01,0xff,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x6a,0xa6,0x06,0xd0,0x33,0x18,0x64,
0xe2,0x69,0x82,0xee,0x6e,0x36,0x9e,0x9d,0x9a,0x0e,0x18,0xa8,0xac,0x9d,0x10,
0xed,0x01,0x3c,0xb9,0x61,0x04,0x62,0xf3,0x85,0x8f,0xcc,0x4f,0x2c,0x66,0x35,
0x54,0x25,0x45,0x8d,0x95,0x1c,0xd2,0x33,0xbe,0x2e,0xdd,0x7f,0x74,0xaf,0x03,
0x7b,0x86,0x63,0xb0,0xc9,0xe6,0xbd,0xc7,0x8e,0xde,0x03,0x18,0x98,0x82,0xc3,
0xbb,0xf8,0x15,0x99,0x1a,0xa9,0xdd,0xb9,0x5d,0xb9,0xbd,0x53,0x95,0x25,0x76,
0xfb,0x5c,0x53,0x90,0xea,0x01,0x0a,0xa0,0xb1,0xbf,0x09,0x1b,0x97,0x8f,0x40,
0xfa,0x85,0x12,0x74,0x01,0xdb,0xf6,0xdb,0x09,0xd6,0x5f,0x4f,0xd7,0x17,0xb4,
0xbf,0x9e,0x2f,0x86,0x52,0x5d,0x70,0x24,0x52,0x32,0x1e,0xa5,0x1d,0x39,0x8b,
0x66,0xf6,0xba,0x9b,0x69,0x8e,0x12,0x60,0xdb,0xb6,0xcf,0xe6,0x0d,0xd6,0x1c,
0x8f,0xd4,0x5b,0x4b,0x00,0xde,0x21,0x93,0xfb,0x6e,0xc7,0x3d,0xb4,0x66,0x0d,
0x29,0x0c,0x4e,0xe9,0x3f,0x94,0xd6,0xd6,0xdc,0xec,0xf8,0x53,0x3b,0x62,0xd5,
0x97,0x50,0x53,0x84,0x17,0xfe,0xe2,0xed,0x4c,0x23,0x0a,0x49,0xce,0x5b,0xe9,
0x70,0x31,0xc1,0x04,0x02,0x02,0x6c,0xb8,0x52,0xcd,0xc7,0x4e,0x70,0xb4,0x13,
0xd7,0xe0,0x92,0xba,0x44,0x1a,0x10,0x4c,0x6e,0x45,0xc6,0x86,0x04,0xc6,0x64,
0xd3,0x9c,0x6e,0xc1,0x9c,0xac,0x74,0x3d,0x77,0x06,0x5e,0x28,0x28,0x5c,0xf5,
0xe0,0x9c,0x19,0xd8,0xba,0x74,0x81,0x2d,0x67,0x77,0x93,0x8d,0xbf,0xd2,0x52,
0x00,0xe6,0xa5,0x38,0x4e,0x2e,0x73,0x66,0x7a };
static const BYTE verisignCA[] = {
0x30,0x82,0x02,0x3c,0x30,0x82,0x01,0xa5,0x02,0x10,0x70,0xba,0xe4,0x1d,0x10,
0xd9,0x29,0x34,0xb6,0x38,0xca,0x7b,0x03,0xcc,0xba,0xbf,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x02,0x05,0x00,0x30,0x5f,0x31,0x0b,
0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,
0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,
0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x37,0x30,0x35,0x06,0x03,0x55,0x04,0x0b,
0x13,0x2e,0x43,0x6c,0x61,0x73,0x73,0x20,0x33,0x20,0x50,0x75,0x62,0x6c,0x69,
0x63,0x20,0x50,0x72,0x69,0x6d,0x61,0x72,0x79,0x20,0x43,0x65,0x72,0x74,0x69,
0x66,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,
0x69,0x74,0x79,0x30,0x1e,0x17,0x0d,0x39,0x36,0x30,0x31,0x32,0x39,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x32,0x38,0x30,0x38,0x30,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x5f,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,
0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,
0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,
0x31,0x37,0x30,0x35,0x06,0x03,0x55,0x04,0x0b,0x13,0x2e,0x43,0x6c,0x61,0x73,
0x73,0x20,0x33,0x20,0x50,0x75,0x62,0x6c,0x69,0x63,0x20,0x50,0x72,0x69,0x6d,
0x61,0x72,0x79,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,
0x6f,0x6e,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,0x81,0x9f,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc9,0x5c,0x59,0x9e,
0xf2,0x1b,0x8a,0x01,0x14,0xb4,0x10,0xdf,0x04,0x40,0xdb,0xe3,0x57,0xaf,0x6a,
0x45,0x40,0x8f,0x84,0x0c,0x0b,0xd1,0x33,0xd9,0xd9,0x11,0xcf,0xee,0x02,0x58,
0x1f,0x25,0xf7,0x2a,0xa8,0x44,0x05,0xaa,0xec,0x03,0x1f,0x78,0x7f,0x9e,0x93,
0xb9,0x9a,0x00,0xaa,0x23,0x7d,0xd6,0xac,0x85,0xa2,0x63,0x45,0xc7,0x72,0x27,
0xcc,0xf4,0x4c,0xc6,0x75,0x71,0xd2,0x39,0xef,0x4f,0x42,0xf0,0x75,0xdf,0x0a,
0x90,0xc6,0x8e,0x20,0x6f,0x98,0x0f,0xf8,0xac,0x23,0x5f,0x70,0x29,0x36,0xa4,
0xc9,0x86,0xe7,0xb1,0x9a,0x20,0xcb,0x53,0xa5,0x85,0xe7,0x3d,0xbe,0x7d,0x9a,
0xfe,0x24,0x45,0x33,0xdc,0x76,0x15,0xed,0x0f,0xa2,0x71,0x64,0x4c,0x65,0x2e,
0x81,0x68,0x45,0xa7,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x02,0x05,0x00,0x03,0x81,0x81,0x00,0xbb,0x4c,
0x12,0x2b,0xcf,0x2c,0x26,0x00,0x4f,0x14,0x13,0xdd,0xa6,0xfb,0xfc,0x0a,0x11,
0x84,0x8c,0xf3,0x28,0x1c,0x67,0x92,0x2f,0x7c,0xb6,0xc5,0xfa,0xdf,0xf0,0xe8,
0x95,0xbc,0x1d,0x8f,0x6c,0x2c,0xa8,0x51,0xcc,0x73,0xd8,0xa4,0xc0,0x53,0xf0,
0x4e,0xd6,0x26,0xc0,0x76,0x01,0x57,0x81,0x92,0x5e,0x21,0xf1,0xd1,0xb1,0xff,
0xe7,0xd0,0x21,0x58,0xcd,0x69,0x17,0xe3,0x44,0x1c,0x9c,0x19,0x44,0x39,0x89,
0x5c,0xdc,0x9c,0x00,0x0f,0x56,0x8d,0x02,0x99,0xed,0xa2,0x90,0x45,0x4c,0xe4,
0xbb,0x10,0xa4,0x3d,0xf0,0x32,0x03,0x0e,0xf1,0xce,0xf8,0xe8,0xc9,0x51,0x8c,
0xe6,0x62,0x9f,0xe6,0x9f,0xc0,0x7d,0xb7,0x72,0x9c,0xc9,0x36,0x3a,0x6b,0x9f,
0x4e,0xa8,0xff,0x64,0x0d,0x64 };

/* chain0_0 -> chain0_1:
 * A simple, valid chain
 */
static const BYTE chain0_0[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xa0,0x9a,0x52,0x5b,0xea,0x24,0xa2,0xb8,0x42,0x63,0x07,0xc3,0xda,0x16,0x08,
0x85,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x98,0x32,0xf8,0xd2,0x7a,0x34,0x64,0x9d,0x40,0xd3,0x85,0x62,0x58,
0x13,0xba,0x2f,0xac,0x2f,0xbd,0x4c,0xe3,0xba,0x05,0xa5,0xbf,0x33,0x4b,0x4a,
0x38,0xd6,0xc9,0x9e,0xa7,0x69,0x95,0xc2,0x67,0x22,0xb6,0xec,0x1f,0x92,0xbc,
0xd3,0xd1,0x59,0x12,0xff,0x83,0xba,0xa1,0x17,0xec,0x52,0x8c,0x21,0x7d,0xf8,
0x68,0x63,0x46,0xbe,0x48,0xd7,0xeb,0x8d,0xbf,0x5e,0x87,0x5f,0xc7,0x56,0x2d,
0xf7,0xf8,0x41,0x8a,0xd1,0xb4,0xe8,0x18,0x71,0xf9,0xf1,0x89,0x7c,0xfc,0x67,
0x9b,0x83,0x5d,0x54,0xb5,0xfd,0x5d,0x94,0x41,0xcd,0x31,0x7d,0x5d,0x02,0x38,
0xff,0x75,0x7d,0xb9,0x86,0x70,0xe4,0x6d,0x96,0x97,0x30,0xeb,0x88,0x45,0xb5,
0x2b,0x7c,0x29,0x84,0x6c,0xe9,0xa3,0xb6,0x11,0x78 };
static const BYTE chain0_1[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,
0xc5,0x4b,0x3f,0xe5,0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,
0xba,0xff,0xb4,0xc6,0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,
0xce,0x30,0xc5,0xa7,0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,
0x16,0xde,0xc6,0x8b,0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,
0x3b,0xb9,0xcf,0x6b,0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,
0xb7,0x82,0x40,0xfc,0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,
0xcb,0x53,0xac,0x85,0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,
0x8b,0xa2,0x23,0xb2,0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,
0xcf,0x49,0x98,0x72,0xa3,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x3c,
0x90,0x45,0x5b,0x81,0x25,0x9d,0x7a,0x10,0x9f,0x15,0x25,0x03,0x52,0x78,0x09,
0x6f,0x62,0x66,0x78,0x01,0x69,0x6a,0x7f,0x19,0x02,0x18,0xd8,0xc0,0x89,0x2f,
0x0f,0x47,0xa7,0x17,0xc2,0xd7,0xe2,0x81,0x7e,0xea,0x54,0xd4,0xe6,0xcc,0x8f,
0x88,0x1a,0xee,0x28,0x42,0x6c,0x77,0xf5,0xec,0x8c,0x05,0xa5,0x7f,0xee,0x2e,
0x0b,0xbc,0xde,0xb0,0x71,0x2c,0x83,0x32,0xcb,0x49,0x00,0x46,0x03,0x95,0x58,
0x6c,0x57,0x0c,0x5a,0x69,0x3c,0xfa,0xbf,0x46,0x7a,0x4d,0xd0,0x23,0x77,0x23,
0xac,0x1d,0x33,0xb7,0x87,0xb4,0xc6,0x6b,0x52,0xa1,0xed,0x9a,0xb8,0x8e,0x1a,
0xcb,0xc3,0xd3,0x50,0xe0,0x0d,0x12,0x5d,0x0e,0x84,0x52,0xa5,0x12,0x76,0x6b,
0xbf,0xb8,0xbe,0x51,0xc0,0xa6,0xae };
/* chain0_0 -> chain1_1:
 * A chain whose signature is bad
 */
static const BYTE chain1_1[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xad,0x7e,0xca,
0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,
0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,
0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,
0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,
0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,
0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,
0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,
0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,
0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0xad,
0x16,0xaa,0x77,0xab,0xd5,0x1a,0x6c,0x90,0x5d,0x4e,0x61,0x49,0x89,0x6e,0x9f,
0x03,0x7b,0x4a,0x49,0xb1,0x46,0x75,0xf3,0x69,0xeb,0x96,0x13,0x3c,0x35,0xb5,
0x37,0x3b,0xc2,0x84,0x5c,0xe5,0x7c,0x46,0xf6,0x6b,0xc7,0x4e,0x72,0x91,0xf4,
0xde,0xc5,0x66,0x0f,0x2d,0x39,0xc6,0x9e,0x8e,0x25,0x4d,0x3f,0x7b,0x45,0xe1,
0xc6,0x1a,0x7e,0x28,0x83,0xf8,0x87,0x30,0x4e,0xa5,0xfd,0x32,0x7a,0xed,0x53,
0x10,0x3b,0x14,0xe5,0xf1,0x32,0x77,0xf1,0x29,0x72,0x98,0x2b,0xa3,0x17,0xd6,
0x8c,0x65,0xaa,0x3b,0x6d,0xab,0xb3,0xbe,0xfa,0x72,0x38,0xf7,0xd0,0xb6,0x0f,
0x5c,0x0d,0xec,0x17,0x43,0x2a,0xfd,0xc1,0x01,0x43,0x98,0xd3,0x1b,0x84,0x3e,
0x32,0xcf,0xda,0x1d,0xc2,0xc9,0x61 };
/* chain2_0 -> chain0_1:
 * A chain whose time nesting is invalid.
 */
static const BYTE chain2_0[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x23,0x82,0x3b,0x9e,0x51,0x6e,0xc9,0x8c,0x4f,0x22,0x48,0x21,0x20,0xb4,0xa3,
0xac,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x36,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x36,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x18,0x5c,0xb9,0x2f,0x27,0xa5,0x4b,0x2c,0xda,0x04,0x78,0x46,0x5e,
0x36,0xe7,0xc9,0xbb,0x42,0xf6,0x85,0x56,0x45,0xca,0x51,0xb2,0xcf,0x2b,0xbd,
0xed,0x77,0xcf,0x12,0xf5,0x29,0xc2,0x42,0x32,0xf4,0xad,0x25,0xc1,0x17,0xef,
0x93,0xbf,0xa5,0x61,0xb8,0xc7,0x4b,0xae,0xa8,0x1c,0x08,0xda,0x98,0xe0,0x5c,
0x55,0x08,0x74,0xf7,0xf4,0xdc,0x38,0x2b,0xa9,0x2b,0x1e,0x12,0x7b,0xc1,0x61,
0xe5,0x8d,0xf3,0xdc,0x21,0xe2,0xd2,0x56,0x4f,0xb4,0x8d,0x8c,0x36,0x69,0xf8,
0x82,0x83,0x28,0x6d,0x0d,0x81,0x73,0xec,0x41,0x5c,0x90,0x7a,0xc5,0x27,0x36,
0x9d,0xaa,0xf5,0xe0,0x8b,0xf0,0xfa,0x7d,0x41,0x8f,0x6a,0x36,0xa8,0x61,0x75,
0xc0,0xf2,0x06,0xe8,0x83,0xdb,0xbe,0xc4,0x5b,0x72 };
/* chain3_0 -> chain0_1:
 * A chain whose root cannot be a CA.
 */
static const BYTE chain3_0[] = {
0x30,0x82,0x01,0xad,0x30,0x82,0x01,0x1a,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x63,0x73,0xee,0x45,0x59,0x76,0x1a,0x9e,0x47,0xf1,0xfb,0xf0,0x42,0x44,0x26,
0xeb,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x10,0x30,0x0e,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x02,0x30,0x00,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,
0x00,0x03,0x81,0x81,0x00,0x53,0xb7,0xa9,0x90,0xe0,0x88,0x36,0xc0,0x2d,0xb1,
0xa3,0x97,0xff,0x16,0x62,0x29,0xa2,0xfd,0x92,0x90,0xb7,0x8d,0xd9,0xdf,0x57,
0x94,0x5a,0xce,0x36,0x5c,0x95,0x24,0xab,0x4f,0x63,0x55,0xa7,0xea,0x8b,0xaf,
0xd9,0x41,0x6a,0x2f,0xd8,0x9e,0x1c,0x2d,0xc0,0x11,0x52,0x09,0x4b,0x31,0x1f,
0x5e,0x8f,0x4b,0xbf,0x20,0x73,0x2d,0x04,0x11,0x56,0x6d,0xb1,0xbe,0xb4,0x5d,
0x70,0x6b,0xb2,0xd0,0xd3,0xf5,0x5d,0x60,0x08,0x65,0x2c,0xe3,0x78,0x5a,0x05,
0x36,0xe1,0xf1,0x67,0x3a,0x25,0xf3,0x47,0x83,0xce,0x4f,0xb0,0x74,0x7a,0x35,
0xa0,0x7f,0x70,0x64,0x8a,0x14,0x66,0x03,0xf6,0xae,0xdf,0x1b,0xf5,0x80,0x87,
0x06,0x39,0x76,0x70,0x7b,0xd2,0x83,0xe9,0xce,0x1f,0xf8,0x66,0xc1 };
/* chain4_0 -> chain0_1 -> chain4_2:
 * A chain whose path length constraint is violated.
 */
static const BYTE chain4_0[] = {
0x30,0x82,0x01,0xb3,0x30,0x82,0x01,0x20,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xcb,0xdc,0x8d,0xca,0xc1,0x0e,0x9c,0x9d,0x4f,0x0a,0xc1,0x09,0x31,0x32,0x94,
0xf9,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x09,0x06,0x05,
0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,0x81,0x00,0x93,0xe8,0x98,0xbe,
0x7b,0x5c,0x4c,0x58,0xbd,0xee,0x31,0x55,0xa8,0xb5,0x86,0xfc,0x23,0x55,0x48,
0xda,0x95,0x60,0x0f,0x53,0x64,0x66,0x50,0x78,0xdc,0x1c,0xf3,0xec,0x38,0x7d,
0x07,0x58,0x19,0xcf,0x83,0xfc,0x92,0xc4,0x3e,0x48,0x5b,0x45,0x54,0xf7,0x15,
0x6f,0xb0,0x5e,0x8c,0xbf,0x03,0x62,0x17,0x27,0x28,0x59,0x49,0xde,0x52,0x2d,
0x4b,0xb3,0x1e,0x4d,0x4f,0x2c,0xa8,0x1e,0x70,0xd1,0xdb,0xe7,0x98,0x6c,0x6f,
0xaa,0x94,0xea,0x5d,0x7f,0xc5,0x1d,0x2a,0x4d,0xb7,0x4c,0xac,0xa6,0xbf,0x79,
0x9a,0xf6,0x1e,0x98,0x64,0x14,0x2e,0xbd,0x8a,0xaf,0xac,0x52,0xd2,0x2c,0xdd,
0xcd,0x1b,0x3d,0x28,0x43,0xea,0x25,0x91,0x15,0xb8,0x54,0x72,0x33,0xca,0xd8,
0x5f,0x06,0x4c,0x97 };
static const BYTE chain4_2[] = {
0x30,0x82,0x01,0x98,0x30,0x82,0x01,0x01,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x32,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x31,0x31,0x32,
0x32,0x31,0x38,0x35,0x32,0x30,0x37,0x5a,0x18,0x0f,0x31,0x36,0x32,0x38,0x30,
0x38,0x32,0x36,0x30,0x35,0x32,0x30,0x35,0x37,0x5a,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,
0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc3,0x31,0x35,
0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,
0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,
0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,
0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,
0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,
0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,
0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,
0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,
0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0xaa,
0x0e,0x17,0x8d,0x5b,0xca,0x11,0xda,0x5a,0xb5,0xd4,0xe8,0x63,0xce,0xa5,0x67,
0x11,0x2f,0xf8,0xb8,0x9f,0x89,0x46,0x7d,0x20,0x94,0x5f,0x0d,0x02,0x7e,0x5e,
0x8c,0x33,0xfc,0x24,0x56,0x08,0xe6,0xf1,0x4d,0x29,0xd5,0xef,0x03,0xe3,0x9a,
0xc6,0xcd,0x6e,0xe3,0xe6,0x45,0xaa,0xce,0x5a,0x2f,0x37,0x6c,0x07,0xd2,0x28,
0x63,0x8e,0x55,0x8a,0xad,0xe3,0x8a,0x58,0x3d,0x24,0x09,0x1e,0x9a,0x7d,0x5f,
0x05,0xc4,0x51,0x18,0x17,0x46,0x33,0x4d,0x72,0x05,0xc2,0x02,0xb6,0x05,0x22,
0x23,0xcc,0xf9,0x1f,0x87,0x6c,0x4a,0x3d,0x6f,0xdb,0x59,0xf6,0x79,0x81,0x02,
0x45,0x9f,0xc8,0xc3,0xf0,0x90,0x82,0xaf,0x58,0x80,0x18,0xea,0x73,0x50,0x29,
0x71,0x8d,0xc1,0x9e,0x53,0x19,0x5a };
/* chain5_0 -> chain5_1:
 * A chain whose name constraints aren't met.
 */
static const BYTE chain5_0[] = {
0x30,0x82,0x01,0xbd,0x30,0x82,0x01,0x2a,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x80,0xc5,0x90,0x33,0xb6,0xb5,0xf7,0x99,0x4e,0x6d,0xfc,0xf9,0x4a,0xa7,0xcb,
0x20,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x20,0x30,0x1e,0x30,0x1c,0x06,0x03,0x55,0x1d,0x1e,0x01,0x01,
0xff,0x04,0x12,0x30,0x10,0xa0,0x0e,0x30,0x0c,0x82,0x0a,0x77,0x69,0x6e,0x65,
0x68,0x71,0x2e,0x6f,0x72,0x67,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,
0x05,0x00,0x03,0x81,0x81,0x00,0x17,0x4a,0x31,0x42,0x6a,0x43,0x85,0xf8,0x91,
0x17,0xc2,0x09,0xbe,0xba,0x1f,0x49,0x23,0x2e,0xa1,0x24,0x58,0x46,0xe8,0xfe,
0x6e,0xbc,0x61,0xb9,0x71,0x92,0x99,0x43,0xd4,0x60,0x54,0xe8,0x29,0x78,0x97,
0xb5,0xb4,0x79,0xad,0xe0,0x1e,0x2b,0x8c,0xb8,0x96,0x9f,0xf1,0x23,0x72,0x45,
0xb7,0x1d,0xe1,0xed,0x0a,0x91,0xf3,0x79,0x8b,0x23,0xd8,0xd4,0x10,0x1b,0x9d,
0xf2,0x7b,0xd9,0x6d,0x85,0x23,0x52,0xd5,0xbf,0x56,0x4f,0x81,0x3c,0x8d,0x62,
0x17,0x0b,0xd7,0x2e,0x5b,0xa1,0xee,0x09,0x99,0x39,0x41,0x57,0x8c,0x3b,0xc4,
0x30,0x1d,0xf5,0x87,0x4f,0x81,0x56,0xfa,0xc1,0x83,0x69,0xf2,0x4a,0x97,0xc9,
0xd9,0x20,0xfa,0x09,0x8c,0x9f,0x13,0xfe,0x0f,0x4d,0x50,0xd3,0x24,0x1f };
static const BYTE chain5_1[] = {
0x30,0x82,0x01,0xb2,0x30,0x82,0x01,0x1b,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,
0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x1c,0x30,0x1a,0x30,0x18,0x06,0x03,0x55,
0x1d,0x07,0x01,0x01,0xff,0x04,0x0e,0x30,0x0c,0x82,0x0a,0x77,0x69,0x6e,0x65,
0x68,0x71,0x2e,0x63,0x6f,0x6d,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x08,0x08,0xf5,0x1c,0xca,
0x3b,0x5a,0xd7,0x45,0xac,0x29,0x5c,0x10,0x9d,0xb0,0x4a,0xdb,0x36,0x0b,0x39,
0x14,0x87,0x3d,0x06,0xe8,0x0d,0x54,0xa7,0xb9,0x50,0x14,0x29,0x05,0xd2,0x75,
0xe4,0x88,0xd8,0x8e,0x7e,0xab,0x61,0xba,0x3c,0xb2,0x4f,0x8f,0xfd,0x00,0x6b,
0x94,0x69,0xf0,0x13,0x40,0x7d,0x4e,0x7c,0xac,0xf0,0x1b,0x07,0x2c,0x96,0x46,
0xf6,0x38,0x38,0xc6,0x65,0x2c,0x02,0x28,0x62,0x6f,0x7d,0xf9,0x4d,0xff,0xb1,
0x38,0x0e,0xce,0x89,0x0e,0xdd,0x22,0x51,0xd2,0x6b,0x0e,0x80,0x3c,0xa6,0x0c,
0xe7,0x8b,0x3b,0x2a,0x2f,0x46,0xed,0xda,0x96,0x1a,0xc1,0x5c,0xfe,0xe0,0xef,
0xdb,0xe0,0x11,0xc4,0x56,0x12,0xe2,0x4b,0x84,0xde,0xaf,0x7f,0x3c,0x9f,0xcb,
0xb6,0xac,0x92 };
/* chain0_0 -> chain6_1:
 * A chain whose end certificate is a CA.
 */
static const BYTE chain6_1[] = {
0x30,0x82,0x01,0xa9,0x30,0x82,0x01,0x12,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,
0x00,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,
0x65,0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,
0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,
0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,
0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,
0x33,0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,
0xdc,0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,
0x48,0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,
0x47,0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,
0x05,0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,
0x6a,0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,
0x85,0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,
0xd3,0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,
0xa3,0x02,0x03,0x01,0x00,0x01,0xa3,0x13,0x30,0x11,0x30,0x0f,0x06,0x03,0x55,
0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,
0x00,0x08,0x71,0x6d,0x6c,0xbf,0xf1,0x0f,0xe9,0x5b,0x9f,0x9c,0x9d,0xa5,0x07,
0x83,0x28,0xa1,0x05,0x0b,0x92,0x73,0xb9,0x8a,0x99,0xbe,0x1e,0x21,0x53,0x9f,
0xae,0x44,0x7b,0x89,0x09,0x36,0x4a,0xef,0xe7,0x73,0x9f,0x84,0x8d,0x9a,0xec,
0x52,0xcd,0x58,0x14,0x2d,0x22,0x5c,0x03,0x6a,0xb3,0x35,0x40,0x35,0x66,0xba,
0xc5,0xe2,0x32,0x72,0x60,0x52,0x86,0xbc,0x6a,0xc3,0x24,0x05,0x6b,0x02,0xe3,
0x70,0xa3,0xad,0x09,0x88,0x1e,0xd1,0x34,0xca,0x52,0x6a,0x55,0x38,0x88,0x60,
0xfa,0x9a,0x1f,0xe9,0x29,0x47,0x54,0x96,0x53,0x65,0x27,0xb1,0x66,0xd0,0x11,
0x1c,0xf6,0x20,0x07,0xb4,0x66,0x88,0x18,0xdc,0x0c,0x57,0x7e,0x1f,0x8a,0x49,
0x7a,0xe2,0x34,0x52,0x0b,0x46,0x6b,0xfd,0x27 };
/* chain0_0 -> chain 7_1:
 * A chain with a bad critical extension.
 */
static const BYTE chain7_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x9f,0x69,0xfd,0x26,0xd5,0x4b,
0xe0,0xab,0x12,0x21,0xb9,0xfc,0xf7,0xe0,0x0c,0x09,0x94,0xad,0x27,0xd7,0x9d,
0xa3,0xcc,0x46,0x2a,0x25,0x9a,0x24,0xa7,0x31,0x58,0x78,0xf5,0xfc,0x30,0xe1,
0x6d,0xfd,0x59,0xab,0xbe,0x69,0xa0,0xea,0xe3,0x7d,0x7a,0x7b,0xe5,0x85,0xeb,
0x86,0x6a,0x84,0x3c,0x96,0x01,0x1a,0x70,0xa7,0xb8,0xcb,0xf2,0x11,0xe7,0x52,
0x9c,0x58,0x2d,0xac,0x63,0xce,0x72,0x4b,0xad,0x62,0xa8,0x1d,0x75,0x96,0xe2,
0x27,0xf5,0x6f,0xba,0x91,0xf8,0xf1,0xb0,0xbf,0x90,0x24,0x6d,0xba,0x5d,0xd7,
0x39,0x63,0x3b,0x7c,0x04,0x5d,0x89,0x9d,0x1c,0xf2,0xf7,0xcc,0xdf,0x6e,0x8a,
0x43,0xa9,0xdd,0x86,0x05,0xa2,0xf3,0x22,0x2d,0x1e,0x70,0xa1,0x59,0xd7,0xa5,
0x94,0x7d };
/* chain8_0 -> chain8_1 -> chain8_2:
 * A chain whose root cert validity time doesn't overlap the end cert's
 * validity time.
 */
static const BYTE chain8_0[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xfb,0x25,0xa6,0x55,0x98,0xbc,0xdf,0xb7,0x40,0x86,0x44,0xb9,0x36,0x03,0x82,
0x9b,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x30,0x36,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x17,0xdf,0x1d,0x1d,0xa3,0x3b,0x0c,0x26,0x3e,0xc7,0x6c,0x8c,0xbd,
0xc7,0x62,0x01,0xba,0xbd,0x2e,0x6a,0x8d,0xe1,0x4e,0x0a,0xc1,0xc1,0x9a,0xb1,
0x43,0x82,0x05,0xb9,0xf8,0x27,0x4d,0x3e,0x8c,0xaa,0xff,0x5c,0x88,0x82,0x3b,
0x09,0xa4,0x23,0x06,0x54,0x16,0xbf,0xc6,0x57,0x6b,0xb1,0x80,0x2c,0xd2,0x7f,
0x28,0xfc,0xb5,0xac,0x5b,0x1f,0x14,0xb9,0xd8,0x49,0x6a,0x0d,0xf1,0x55,0x51,
0xcd,0x0c,0x2b,0x5b,0xfa,0x01,0x08,0x82,0x98,0x6d,0x39,0xde,0x9a,0xb1,0xbf,
0xc5,0xbd,0x05,0xd1,0x64,0x21,0xc4,0x99,0x40,0xdb,0x3e,0xa0,0x8e,0x46,0x80,
0xd0,0x74,0x93,0xbd,0xad,0x81,0xba,0x40,0x8e,0x0a,0x67,0x47,0xc4,0xae,0xd4,
0xca,0xd8,0x3b,0xa4,0xa6,0xbd,0xc8,0xa9,0x67,0xf9 };
static const BYTE chain8_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x22,0xf1,0x66,0x00,0x79,0xd2,
0xe6,0xb2,0xb2,0xf7,0x2f,0x98,0x92,0x7d,0x73,0xc3,0x6c,0x5c,0x77,0x20,0xe3,
0xbf,0x3e,0xe0,0xb3,0x5c,0x68,0xb4,0x9b,0x3a,0x41,0xae,0x94,0xa0,0x80,0x3a,
0xfe,0x5d,0x7a,0x56,0x87,0x85,0x44,0x45,0xcf,0xa6,0xd3,0x10,0xe7,0x73,0x41,
0xf2,0x7f,0x88,0x85,0x91,0x8e,0xe6,0xec,0xe2,0xce,0x08,0xbc,0xa5,0x76,0xe5,
0x4d,0x1d,0xb7,0x70,0x31,0xdd,0xc9,0x9a,0x15,0x32,0x11,0x5a,0x4e,0x62,0xc8,
0xd1,0xf8,0xec,0x46,0x39,0x5b,0xe7,0x67,0x1f,0x58,0xe8,0xa1,0xa0,0x5b,0xf7,
0x8a,0x6d,0x5f,0x91,0x18,0xd4,0x90,0x85,0xff,0x30,0xc7,0xca,0x9c,0xc6,0x92,
0xb0,0xca,0x16,0xc4,0xa4,0xc0,0xd6,0xe8,0xff,0x15,0x19,0xd1,0x30,0x61,0xf3,
0xef,0x9f };
static const BYTE chain8_2[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x39,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x33,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,
0xc6,0x32,0xe9,0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,
0x9c,0x7f,0x11,0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,
0x8f,0x0c,0xd0,0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,
0x9b,0xde,0x42,0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,
0x39,0x23,0x87,0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,
0xb7,0x7c,0x55,0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,
0xd5,0xad,0x0f,0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,
0xf9,0x2a,0xb1,0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x33,0x95,0x73,0x7c,0x7c,0xb0,
0xbb,0x2b,0xe0,0xc4,0x12,0xf0,0x26,0x85,0x97,0x0c,0x81,0xde,0xc5,0x52,0x04,
0xcb,0xee,0x11,0x9a,0x2e,0xa2,0xc4,0x76,0xd5,0x57,0xd1,0x2f,0x2d,0x6f,0x42,
0xbf,0x23,0x85,0x59,0x5c,0x96,0x8d,0xef,0xbc,0xb6,0xf5,0xd3,0x66,0x99,0x54,
0x95,0x09,0x28,0x0b,0x4f,0x05,0x5f,0x61,0x20,0xa8,0x0d,0x46,0x9a,0xab,0x29,
0x97,0x6b,0xa3,0xcd,0xd7,0x5a,0xda,0xaa,0x8e,0xad,0x5a,0x66,0x58,0xf6,0x63,
0x76,0x10,0xdc,0xb0,0xee,0x3f,0x82,0x6d,0x29,0xb9,0xcb,0x99,0x15,0xbb,0xb9,
0x74,0x73,0x00,0x14,0x74,0x79,0xad,0x89,0x73,0xaa,0xa2,0x0e,0x52,0xe8,0x28,
0x69,0xe6,0x8c,0x1e,0xce,0x3a,0x6d,0x85,0x33,0x1e,0x21,0xa3,0x42,0x61,0x3a,
0xeb,0x25 };
/* chain9_0 -> chain7_1:
 * A cyclic chain.
 */
static const BYTE chain9_0[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x32,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,
0xca,0x50,0x82,0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,
0xc5,0xc3,0x34,0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,
0x03,0x7a,0xe7,0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,
0xc1,0x33,0x91,0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,
0x1b,0x99,0xc5,0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,
0x49,0xba,0xcd,0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,
0x72,0xb6,0x1c,0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,
0x1f,0x2f,0x35,0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x2c,0xf5,0x25,0xaa,0xb4,0x63,
0x5a,0x5d,0xae,0x7b,0x42,0x79,0xbc,0x41,0x2f,0x8a,0x2a,0x70,0x9f,0x8e,0xb2,
0x14,0x1b,0x6f,0xb2,0xf3,0x3e,0xab,0x06,0x9c,0xa8,0x92,0x43,0xc4,0x57,0x63,
0xa4,0xee,0x83,0x04,0xab,0xa5,0x2c,0xd9,0x90,0xed,0xc9,0x4d,0xa8,0x7b,0xca,
0x3b,0x5f,0xdf,0x46,0xe1,0x3b,0x7f,0x72,0xb7,0x92,0x82,0xe6,0x05,0xc3,0x64,
0x38,0xfb,0x86,0x13,0xeb,0x19,0xb6,0x16,0x82,0x63,0x84,0x13,0x04,0x38,0x51,
0x4c,0x89,0x12,0xf1,0x13,0x46,0x75,0x4e,0x99,0x46,0xcb,0xc6,0x76,0x9f,0x97,
0x9c,0xda,0x63,0x01,0x6a,0x71,0x68,0xc4,0x21,0x67,0xa6,0x51,0x19,0x5e,0xc7,
0x8c,0x8d,0x63,0x5c,0xbe,0x04,0x8f,0xad,0xea,0xa5,0xcd,0xc3,0xd6,0x6b,0xc7,
0xb5,0x56 };
/* chain0_0 -+
 *            +-> chain7_1
 * chain10_1 -+
 * A chain with two issuers, only one of whose dates is valid.
 */
static const BYTE chain10_1[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0xbf,0x99,0x4f,0x14,0x03,0x77,0x44,0xb8,0x49,0x02,0x70,0xa1,0xb8,0x9c,0xa7,
0x24,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0xa8,0xec,0x8c,0x34,0xe7,0x2c,0xdf,0x75,0x87,0xc4,0xf7,0xda,0x71,
0x72,0x29,0xb2,0x48,0xa8,0x2a,0xec,0x7b,0x7d,0x19,0xb9,0x5f,0x1d,0xd9,0x91,
0x2b,0xc4,0x28,0x7e,0xd6,0xb5,0x91,0x69,0xa5,0x8a,0x1a,0x1f,0x97,0x98,0x46,
0x9d,0xdf,0x12,0xf6,0x45,0x62,0xad,0x60,0xb6,0xba,0xb0,0xfd,0xf5,0x9f,0xc6,
0x98,0x05,0x4f,0x4d,0x48,0xdc,0xee,0x69,0xbe,0xb8,0xc4,0xc4,0xd7,0x1b,0xb1,
0x1f,0x64,0xd6,0x45,0xa7,0xdb,0xb3,0x87,0x63,0x0f,0x54,0xe1,0x3a,0x6b,0x57,
0x36,0xd7,0x68,0x65,0xcf,0xda,0x57,0x8d,0xcd,0x84,0x75,0x47,0x26,0x2c,0xef,
0x1e,0x8f,0xc7,0x3b,0xee,0x5d,0x03,0xa6,0xdf,0x3a,0x20,0xb2,0xcc,0xc9,0x09,
0x2c,0xfe,0x2b,0x79,0xb0,0xca,0x2c,0x9a,0x81,0x6b };
/* chain0_0 -+
 *            +-> chain7_1
 * chain11_1 -+
 * A chain with two issuers, only one of whose public keys matches the
 * signature.
 */
static const BYTE chain11_1[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x72,0x41,0xa2,0x53,0xd2,0x33,0x0b,0xac,0x4a,0x30,0xfe,0x0a,0xe0,0x2e,0xd4,
0x03,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xc3,0x31,0x35,0xc3,0x9d,0x8a,0x87,0x20,0xc6,0x32,0xe9,
0xb1,0xf6,0x8f,0xf9,0x05,0x73,0x1d,0xa7,0xde,0xab,0x15,0x8a,0x9c,0x7f,0x11,
0x7e,0x77,0xa0,0x42,0x80,0xf4,0x79,0xda,0x98,0x7b,0x00,0xfa,0x8f,0x0c,0xd0,
0xeb,0x8a,0x80,0xed,0x07,0xfc,0x64,0x71,0x03,0xc1,0xd6,0x2f,0x9b,0xde,0x42,
0x63,0xd9,0x79,0xea,0xdd,0x10,0xe8,0x68,0xc8,0x69,0x4f,0x4a,0x39,0x23,0x87,
0xca,0xd1,0xc9,0x77,0x14,0x30,0x85,0x9e,0xf7,0x79,0xf9,0x07,0xb7,0x7c,0x55,
0xcb,0xa7,0xd5,0xb8,0x44,0xb5,0x20,0xb5,0x01,0x5c,0xa2,0xd1,0xd5,0xad,0x0f,
0x87,0xaf,0x37,0xd1,0x39,0x0c,0x0d,0xd5,0xde,0x26,0x7a,0xed,0xf9,0x2a,0xb1,
0x60,0x65,0x2d,0x08,0x24,0x51,0x1d,0xb0,0x0a,0xb5,0x13,0xc7,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x51,0x17,0xfb,0x2a,0x69,0x31,0x36,0xa7,0x37,0x2c,0x1f,0x38,0xee,
0xe3,0xf1,0xae,0x63,0x61,0xac,0x63,0x13,0xc2,0x3b,0x29,0xca,0xb1,0x8e,0x27,
0x41,0xfa,0xb3,0x7d,0x42,0xbe,0xf0,0x9c,0xe0,0xc7,0x69,0xba,0x80,0xf4,0x87,
0x7c,0x6f,0x77,0x70,0x6b,0x77,0x7d,0x5c,0x58,0x95,0x07,0xbd,0xac,0xe2,0x5d,
0x65,0x12,0xb0,0xae,0xf4,0x01,0x14,0x01,0xba,0xf9,0xa0,0x32,0x10,0xea,0x09,
0x12,0xcf,0xf1,0xbc,0x37,0xc9,0x42,0xc3,0x8c,0x9e,0xa2,0xc3,0x5e,0x2e,0x56,
0xe9,0xa7,0xd8,0x1c,0x1c,0x8c,0x1c,0x2f,0xd8,0xfb,0x67,0x44,0xd4,0xa3,0x79,
0x6d,0x53,0x17,0x56,0x3c,0x3e,0xd0,0x67,0x36,0x0c,0x8b,0x77,0xab,0x40,0xc7,
0x27,0xd9,0x44,0x69,0x13,0xeb,0x09,0xec,0x06,0xe2 };
/* chain12_0 -> chain7_1:
 * A chain whose root has an invalid signature.
 */
static const BYTE chain12_0[] = {
0x30,0x82,0x01,0x9b,0x30,0x82,0x01,0x08,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x2f,0x67,0x04,0x50,0xe4,0xac,0x81,0x81,0x44,0x25,0x25,0x81,0xbe,0x8a,0x6c,
0x3c,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,
0x81,0x00,0x82,0xa8,0x28,0x6e,0x85,0xbb,0xa4,0xac,0xf4,0xe3,0xbc,0xe5,0x08,
0x76,0x8a,0x1b,0x51,0xea,0xca,0x49,0x46,0x70,0x75,0x38,0x70,0x71,0xa1,0x1c,
0x67,0x77,0x70,0x7d,0xdf,0xd0,0xc6,0xa4,0xfc,0x30,0x22,0x2f,0x52,0x56,0x16,
0xe7,0xe2,0x7e,0x6a,0xcf,0x24,0x0b,0xe2,0xad,0x94,0x52,0x2c,0xcc,0x56,0xd3,
0x2e,0xe2,0xd3,0x3d,0x2f,0xdb,0x22,0xb2,0xff,0xf3,0x20,0xc1,0x62,0xc2,0x3c,
0x24,0x48,0x19,0x13,0x2c,0x8a,0x80,0xcd,0xb5,0xe1,0x84,0xc2,0xee,0x14,0xfe,
0x59,0xe6,0xed,0x7b,0x8b,0xec,0xcc,0x27,0x56,0xa1,0x0e,0x73,0xe7,0x37,0x20,
0x97,0x27,0x12,0xed,0x9f,0xe9,0xec,0x44,0xde,0x3f,0x07,0xa4,0x07,0x42,0x5b,
0x7e,0xe5,0xe1,0x42,0x10,0x4c,0x38,0xd7,0x2b,0xff };
/* chain0_0 -> chain13_1:
 * A chain whose time validity nesting is invalid.
 */
static const BYTE chain13_1[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x36,0x30,0x31,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,
0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x85,0xc9,0xc5,0x08,0x0b,0xb1,
0xff,0x58,0x8f,0xee,0x66,0x5f,0xa3,0x77,0x63,0x9f,0xd1,0x53,0x03,0xdc,0xf6,
0x59,0x08,0x02,0x1f,0xdc,0x52,0x5a,0x52,0x38,0xa4,0x0b,0x83,0x12,0x19,0xd6,
0xb5,0xb8,0x81,0x17,0xcd,0x10,0x5d,0x96,0x16,0x9d,0x51,0x5b,0xf9,0x31,0x3e,
0x36,0xac,0x30,0xd5,0xd6,0xa0,0x30,0x5c,0x9c,0xcb,0xe1,0x02,0xfd,0x73,0xc3,
0xa1,0x58,0xfe,0x12,0xb0,0xa6,0xbd,0x11,0x86,0xdb,0x20,0x17,0x29,0x1e,0x09,
0x63,0x4d,0x28,0xe4,0x55,0x2d,0xa1,0x78,0xde,0x28,0x0b,0x1b,0x35,0x40,0x4c,
0xf1,0xb0,0x6e,0x82,0x34,0x2b,0x7f,0x3b,0x2e,0x27,0x6b,0xf7,0xde,0x73,0x9a,
0x98,0xac,0x79,0x33,0x1a,0xdd,0x03,0x95,0x60,0x8a,0x08,0xba,0xcb,0x8a,0xbf,
0xea,0xe3 };

typedef struct _CONST_DATA_BLOB
{
    DWORD       cbData;
    const BYTE *pbData;
} CONST_DATA_BLOB;

typedef struct _CONST_BLOB_ARRAY
{
    DWORD cBlob;
    CONST_DATA_BLOB *rgBlob;
} CONST_BLOB_ARRAY;

#define TODO_CHAIN    1
#define TODO_ERROR    2
#define TODO_INFO     4
#define TODO_ELEMENTS 8
#define TODO_CHAINS   16
#define TODO_POLICY   32

/* Gets a certificate chain built from a store containing all the certs in
 * certArray, where the last certificate in the chain is expected to be the
 * end certificate (the one from which the chain is built.)
 */
static PCCERT_CHAIN_CONTEXT getChain(const CONST_BLOB_ARRAY *certArray,
 DWORD flags, BOOL includeStore, LPSYSTEMTIME checkTime, DWORD todo,
 DWORD testIndex)
{
    HCERTSTORE store;
    PCCERT_CHAIN_CONTEXT chain = NULL;

    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    if (store)
    {
        BOOL ret;
        PCCERT_CONTEXT endCert;

        if (certArray->cBlob > 1)
        {
            DWORD i;

            for (i = 0, ret = TRUE; ret && i < certArray->cBlob - 1; i++)
            {
                ret = CertAddEncodedCertificateToStore(store,
                 X509_ASN_ENCODING, certArray->rgBlob[i].pbData,
                 certArray->rgBlob[i].cbData, CERT_STORE_ADD_ALWAYS, NULL);
                ok(ret, "Chain %d: adding cert %d failed: %08x\n",
                 testIndex, i, GetLastError());
            }
        }
        ret = CertAddEncodedCertificateToStore(store,
         X509_ASN_ENCODING, certArray->rgBlob[certArray->cBlob - 1].pbData,
         certArray->rgBlob[certArray->cBlob - 1].cbData, CERT_STORE_ADD_ALWAYS,
         &endCert);
        ok(ret, "Chain %d: adding end cert failed: %08x\n",
         testIndex, GetLastError());
        if (ret)
        {
            /* FIXME: allow caller to specify usage matches? */
            CERT_CHAIN_PARA chainPara = { sizeof(chainPara), { 0 } };
            FILETIME fileTime;

            SystemTimeToFileTime(checkTime, &fileTime);
            ret = CertGetCertificateChain(NULL, endCert, &fileTime,
             includeStore ? store : NULL, &chainPara, flags, NULL, &chain);
            if (todo & TODO_CHAIN)
                todo_wine ok(ret, "Chain %d: CertGetCertificateChain failed: %08x\n",
                 testIndex, GetLastError());
            else
                ok(ret, "Chain %d: CertGetCertificateChain failed: %08x\n",
                 testIndex, GetLastError());
            CertFreeCertificateContext(endCert);
        }
        CertCloseStore(store, 0);
    }
    return chain;
}

typedef struct _SimpleChainStatusCheck
{
    DWORD                    cElement;
    const CERT_TRUST_STATUS *rgElementStatus;
} SimpleChainStatusCheck;

static void checkSimpleChainStatus(const CERT_SIMPLE_CHAIN *simpleChain,
 const SimpleChainStatusCheck *simpleChainStatus,
 const CERT_TRUST_STATUS *ignore, DWORD todo, DWORD testIndex, DWORD chainIndex)
{
    if (todo & TODO_ELEMENTS)
        todo_wine ok(simpleChain->cElement == simpleChainStatus->cElement,
         "Chain %d: expected %d elements, got %d\n", testIndex,
         simpleChainStatus->cElement, simpleChain->cElement);
    else
        ok(simpleChain->cElement == simpleChainStatus->cElement,
         "Chain %d: expected %d elements, got %d\n", testIndex,
         simpleChainStatus->cElement, simpleChain->cElement);
    if (simpleChain->cElement == simpleChainStatus->cElement)
    {
        DWORD i;

        for (i = 0; i < simpleChain->cElement; i++)
        {
            DWORD error =
             simpleChain->rgpElement[i]->TrustStatus.dwErrorStatus &
             ~ignore->dwErrorStatus;
            DWORD info =
             simpleChain->rgpElement[i]->TrustStatus.dwInfoStatus &
             ~ignore->dwInfoStatus;

            if (todo & TODO_ERROR && error !=
             simpleChainStatus->rgElementStatus[i].dwErrorStatus)
                todo_wine ok(error ==
                 simpleChainStatus->rgElementStatus[i].dwErrorStatus,
                 "Chain %d, element [%d,%d]: expected error %08x, got %08x\n",
                 testIndex, chainIndex, i,
                 simpleChainStatus->rgElementStatus[i].dwErrorStatus, error);
            else
                ok(error ==
                 simpleChainStatus->rgElementStatus[i].dwErrorStatus,
                 "Chain %d, element [%d,%d]: expected error %08x, got %08x\n",
                 testIndex, chainIndex, i,
                 simpleChainStatus->rgElementStatus[i].dwErrorStatus, error);
            if (todo & TODO_INFO && info !=
             simpleChainStatus->rgElementStatus[i].dwInfoStatus)
                todo_wine ok(info ==
                 simpleChainStatus->rgElementStatus[i].dwInfoStatus,
                 "Chain %d, element [%d,%d]: expected info %08x, got %08x\n",
                 testIndex, chainIndex, i,
                 simpleChainStatus->rgElementStatus[i].dwInfoStatus, info);
            else
                ok(info ==
                 simpleChainStatus->rgElementStatus[i].dwInfoStatus,
                 "Chain %d, element [%d,%d]: expected info %08x, got %08x\n",
                 testIndex, chainIndex, i,
                 simpleChainStatus->rgElementStatus[i].dwInfoStatus, info);
        }
    }
}

typedef struct _ChainStatusCheck
{
    CERT_TRUST_STATUS             statusToIgnore;
    CERT_TRUST_STATUS             status;
    DWORD                         cChain;
    const SimpleChainStatusCheck *rgChainStatus;
} ChainStatusCheck;

static void checkChainStatus(PCCERT_CHAIN_CONTEXT chain,
 const ChainStatusCheck *chainStatus, DWORD todo, DWORD testIndex)
{
    DWORD error = chainStatus->status.dwErrorStatus &
     ~chainStatus->statusToIgnore.dwErrorStatus;
    DWORD info = chainStatus->status.dwInfoStatus &
     ~chainStatus->statusToIgnore.dwInfoStatus;

    ok(chain->cChain == chainStatus->cChain,
     "Chain %d: expected %d simple chains, got %d\n", testIndex,
     chainStatus->cChain, chain->cChain);
    if (todo & TODO_ERROR && error != chainStatus->status.dwErrorStatus)
        todo_wine ok(error == chainStatus->status.dwErrorStatus,
         "Chain %d: expected error %08x, got %08x\n",
         testIndex, chainStatus->status.dwErrorStatus, error);
    else
        ok(error == chainStatus->status.dwErrorStatus,
         "Chain %d: expected error %08x, got %08x\n",
         testIndex, chainStatus->status.dwErrorStatus, error);
    if (todo & TODO_INFO && info != chainStatus->status.dwInfoStatus)
        todo_wine ok(info == chainStatus->status.dwInfoStatus,
         "Chain %d: expected info %08x, got %08x\n",
         testIndex, chainStatus->status.dwInfoStatus, info);
    else
        ok(info == chainStatus->status.dwInfoStatus,
         "Chain %d: expected info %08x, got %08x\n",
         testIndex, chainStatus->status.dwInfoStatus, info);
    if (chain->cChain == chainStatus->cChain)
    {
        DWORD i;

        for (i = 0; i < chain->cChain; i++)
            checkSimpleChainStatus(chain->rgpChain[i],
             &chainStatus->rgChainStatus[i], &chainStatus->statusToIgnore,
             todo, testIndex, i);
    }
}

typedef struct _ChainCheck
{
    CONST_BLOB_ARRAY certs;
    ChainStatusCheck status;
    DWORD            todo;
} ChainCheck;

static CONST_DATA_BLOB chain0[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus0[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus0[] = {
 { sizeof(elementStatus0) / sizeof(elementStatus0[0]), elementStatus0 },
};
static CONST_DATA_BLOB chain1[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain1_1), chain1_1 },
};
static const CERT_TRUST_STATUS elementStatus1[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_NOT_SIGNATURE_VALID,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus1[] = {
 { sizeof(elementStatus1) / sizeof(elementStatus1[0]), elementStatus1 },
};
static CONST_DATA_BLOB chain2[] = {
 { sizeof(chain2_0), chain2_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus2[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus2[] = {
 { sizeof(elementStatus2) / sizeof(elementStatus2[0]), elementStatus2 },
};
static CONST_DATA_BLOB chain3[] = {
 { sizeof(chain3_0), chain3_0 },
 { sizeof(chain0_1), chain0_1 },
};
static const CERT_TRUST_STATUS elementStatus3[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus3[] = {
 { sizeof(elementStatus3) / sizeof(elementStatus3[0]), elementStatus3 },
};
static CONST_DATA_BLOB chain4[] = {
 { sizeof(chain4_0), chain4_0 },
 { sizeof(chain0_1), chain0_1 },
 { sizeof(chain4_2), chain4_2 },
};
static const CERT_TRUST_STATUS elementStatus4[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_NOT_TIME_VALID,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus4[] = {
 { sizeof(elementStatus4) / sizeof(elementStatus4[0]), elementStatus4 },
};
static CONST_DATA_BLOB chain5[] = {
 { sizeof(chain5_0), chain5_0 },
 { sizeof(chain5_1), chain5_1 },
};
static const CERT_TRUST_STATUS elementStatus5[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
   CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT |
   CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus5[] = {
 { sizeof(elementStatus5) / sizeof(elementStatus5[0]), elementStatus5 },
};
static CONST_DATA_BLOB chain6[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain6_1), chain6_1 },
};
static const CERT_TRUST_STATUS elementStatus6[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus6[] = {
 { sizeof(elementStatus6) / sizeof(elementStatus6[0]), elementStatus6 },
};
static CONST_DATA_BLOB chain7[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus7[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus7[] = {
 { sizeof(elementStatus7) / sizeof(elementStatus7[0]), elementStatus7 },
};
static CONST_DATA_BLOB chain8[] = {
 { sizeof(chain8_0), chain8_0 },
 { sizeof(chain8_1), chain8_1 },
 { sizeof(chain8_2), chain8_2 },
};
static const CERT_TRUST_STATUS elementStatus8[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck simpleStatus8[] = {
 { sizeof(elementStatus8) / sizeof(elementStatus8[0]), elementStatus8 },
};
static CONST_DATA_BLOB chain9[] = {
 { sizeof(chain9_0), chain9_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus9[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_CYCLIC,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus9[] = {
 { sizeof(elementStatus9) / sizeof(elementStatus9[0]), elementStatus9 },
};
static CONST_DATA_BLOB chain10[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain10_1), chain10_1 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus10[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus10[] = {
 { sizeof(elementStatus10) / sizeof(elementStatus10[0]), elementStatus10 },
};
static CONST_DATA_BLOB chain11[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain11_1), chain11_1 },
 { sizeof(chain7_1), chain7_1 },
};
static CONST_DATA_BLOB chain12[] = {
 { sizeof(chain12_0), chain12_0 },
 { sizeof(chain7_1), chain7_1 },
};
static const CERT_TRUST_STATUS elementStatus12[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_NOT_SIGNATURE_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus12[] = {
 { sizeof(elementStatus12) / sizeof(elementStatus12[0]), elementStatus12 },
};
static CONST_DATA_BLOB chain13[] = {
 { sizeof(chain0_0), chain0_0 },
 { sizeof(chain13_1), chain13_1 },
};
static const CERT_TRUST_STATUS elementStatus13[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
 { CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_IS_SELF_SIGNED | CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus13[] = {
 { sizeof(elementStatus13) / sizeof(elementStatus13[0]), elementStatus13 },
};
static CONST_DATA_BLOB selfSignedChain[] = {
 { sizeof(selfSignedCert), selfSignedCert }
};
static const CERT_TRUST_STATUS selfSignedElementStatus[] = {
 { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED },
};
static const SimpleChainStatusCheck selfSignedSimpleStatus[] = {
 { sizeof(selfSignedElementStatus) / sizeof(selfSignedElementStatus[0]),
   selfSignedElementStatus },
};
static CONST_DATA_BLOB iTunesChain[] = {
 { sizeof(verisignCA), verisignCA },
 { sizeof(iTunesCert0), iTunesCert0 },
 { sizeof(iTunesCert1), iTunesCert1 },
 { sizeof(iTunesCert2), iTunesCert2 },
 { sizeof(iTunesCert3), iTunesCert3 },
};
static const CERT_TRUST_STATUS iTunesElementStatus[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_KEY_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_EXACT_MATCH_ISSUER },
 { CERT_TRUST_NO_ERROR,
   CERT_TRUST_HAS_NAME_MATCH_ISSUER | CERT_TRUST_IS_SELF_SIGNED},
};
static const SimpleChainStatusCheck iTunesSimpleStatus[] = {
 { sizeof(iTunesElementStatus) / sizeof(iTunesElementStatus[0]),
   iTunesElementStatus },
};
static ChainCheck chainCheck[] = {
 /* Windows XP erroneously sets CERT_TRUST_HAS_PREFERRED_ISSUER on all
  * elements, so ignore it
  */
 { { sizeof(chain0) / sizeof(chain0[0]), chain0 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus0 }, 0 },
 { { sizeof(chain1) / sizeof(chain1[0]), chain1 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_SIGNATURE_VALID |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus1 }, 0 },
 { { sizeof(chain2) / sizeof(chain2[0]), chain2 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT | CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus2 }, 0 },
 { { sizeof(chain3) / sizeof(chain3[0]), chain3 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus3 }, 0 },
 { { sizeof(chain4) / sizeof(chain4[0]), chain4 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus4 }, 0 },
 { { sizeof(chain5) / sizeof(chain5[0]), chain5 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT |
       CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT |
       CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus5 }, 0 },
 { { sizeof(chain6) / sizeof(chain6[0]), chain6 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus6 }, 0 },
 { { sizeof(chain7) / sizeof(chain7[0]), chain7 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus7 }, 0 },
 { { sizeof(chain8) / sizeof(chain8[0]), chain8 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus8 },
   TODO_ERROR },
 { { sizeof(chain9) / sizeof(chain9[0]), chain9 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_HAS_NOT_SUPPORTED_NAME_CONSTRAINT |
       CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_CYCLIC, 0 },
     1, simpleStatus9 },
   TODO_ERROR | TODO_INFO },
 { { sizeof(chain10) / sizeof(chain10[0]), chain10 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus10 }, 0 },
 { { sizeof(chain11) / sizeof(chain11[0]), chain11 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus10 }, 0 },
 { { sizeof(chain12) / sizeof(chain12[0]), chain12 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus12 }, 0 },
 { { sizeof(chain13) / sizeof(chain13[0]), chain13 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_UNTRUSTED_ROOT, 0 }, 1, simpleStatus13 },
   0 },
 { { sizeof(selfSignedChain) / sizeof(selfSignedChain[0]), selfSignedChain },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT, 0 },
     1, selfSignedSimpleStatus }, 0 },
 /* The iTunes chain may or may not have its root trusted, so ignore the
  * error
  */
 { { sizeof(iTunesChain) / sizeof(iTunesChain[0]), iTunesChain },
   { { CERT_TRUST_IS_UNTRUSTED_ROOT, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { 0, 0 },
       1, iTunesSimpleStatus }, 0 },
};

static const CERT_TRUST_STATUS elementStatus8NoStore[] = {
 { CERT_TRUST_NO_ERROR, CERT_TRUST_HAS_NAME_MATCH_ISSUER },
};
static const SimpleChainStatusCheck simpleStatus8NoStore[] = {
 { sizeof(elementStatus8NoStore) / sizeof(elementStatus8NoStore[0]),
   elementStatus8NoStore },
};
static ChainCheck chainCheckNoStore[] = {
 { { sizeof(selfSignedChain) / sizeof(selfSignedChain[0]), selfSignedChain },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_IS_NOT_TIME_VALID | CERT_TRUST_IS_UNTRUSTED_ROOT, 0 },
     1, selfSignedSimpleStatus },
   TODO_ERROR },
 { { sizeof(chain8) / sizeof(chain8[0]), chain8 },
   { { 0, CERT_TRUST_HAS_PREFERRED_ISSUER },
     { CERT_TRUST_INVALID_BASIC_CONSTRAINTS | CERT_TRUST_IS_UNTRUSTED_ROOT |
       CERT_TRUST_IS_NOT_TIME_VALID, 0 },
     1, simpleStatus8NoStore },
   TODO_INFO },
};

/* Wednesday, Oct 1, 2007 */
static SYSTEMTIME oct2007 = { 2007, 10, 1, 1, 0, 0, 0, 0 };

static void testGetCertChain(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    CERT_CHAIN_PARA para = { 0 };
    PCCERT_CHAIN_CONTEXT chain;
    DWORD i;

    /* Basic parameter checks */
    ret = CertGetCertificateChain(NULL, NULL, NULL, NULL, NULL, 0, NULL, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertGetCertificateChain(NULL, NULL, NULL, NULL, NULL, 0, NULL,
     &chain);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    /* Crash
    ret = CertGetCertificateChain(NULL, NULL, NULL, NULL, &para, 0, NULL, NULL);
    ret = CertGetCertificateChain(NULL, NULL, NULL, NULL, &para, 0, NULL,
     &chain);
     */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    ret = CertGetCertificateChain(NULL, cert, NULL, NULL, NULL, 0, NULL, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    /* Crash
    ret = CertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL, NULL);
     */

    /* Tests with an invalid cert (one whose signature is bad) */
    SetLastError(0xdeadbeef);
    ret = CertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL,
     &chain);
    ok(!ret && GetLastError() == ERROR_INVALID_DATA,
     "Expected ERROR_INVALID_DATA, got %d\n", GetLastError());
    para.cbSize = sizeof(para);
    SetLastError(0xdeadbeef);
    ret = CertGetCertificateChain(NULL, cert, NULL, NULL, &para, 0, NULL,
     &chain);
    ok(!ret && GetLastError() == ERROR_INVALID_DATA,
     "Expected ERROR_INVALID_DATA, got %d\n", GetLastError());
    CertFreeCertificateContext(cert);

    for (i = 0; i < sizeof(chainCheck) / sizeof(chainCheck[0]); i++)
    {
        chain = getChain(&chainCheck[i].certs, 0, TRUE, &oct2007,
         chainCheck[i].todo, i);
        if (chain)
        {
            checkChainStatus(chain, &chainCheck[i].status, chainCheck[i].todo,
             i);
            CertFreeCertificateChain(chain);
        }
    }
    for (i = 0; i < sizeof(chainCheckNoStore) / sizeof(chainCheckNoStore[0]);
     i++)
    {
        chain = getChain(&chainCheckNoStore[i].certs, 0, FALSE, &oct2007,
         chainCheckNoStore[i].todo, i);
        if (chain)
        {
            checkChainStatus(chain, &chainCheckNoStore[i].status,
             chainCheckNoStore[i].todo, i);
            CertFreeCertificateChain(chain);
        }
    }
}

typedef struct _ChainPolicyCheck
{
    CONST_BLOB_ARRAY         certs;
    CERT_CHAIN_POLICY_STATUS status;
    DWORD                    todo;
} ChainPolicyCheck;

static ChainPolicyCheck basePolicyCheck[] = {
 { { sizeof(chain0) / sizeof(chain0[0]), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain1) / sizeof(chain1[0]), chain1 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 0, NULL }, 0 },
 { { sizeof(chain2) / sizeof(chain2[0]), chain2 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain3) / sizeof(chain3[0]), chain3 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain4) / sizeof(chain4[0]), chain4 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, 0 },
 { { sizeof(chain5) / sizeof(chain5[0]), chain5 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain6) / sizeof(chain6[0]), chain6 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain7) / sizeof(chain7[0]), chain7 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain8) / sizeof(chain8[0]), chain8 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, 0 },
 { { sizeof(chain9) / sizeof(chain9[0]), chain9 },
   { 0, CERT_E_CHAINING, 0, -1, NULL }, 0 },
 { { sizeof(chain10) / sizeof(chain10[0]), chain10 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain11) / sizeof(chain11[0]), chain11 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain12) / sizeof(chain12[0]), chain12 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 1, NULL }, 0 },
 { { sizeof(selfSignedChain) / sizeof(selfSignedChain[0]), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, 0 },
};

static ChainPolicyCheck authenticodePolicyCheck[] = {
 { { sizeof(chain0) / sizeof(chain0[0]), chain0 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain1) / sizeof(chain1[0]), chain1 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 0, NULL }, 0 },
 { { sizeof(chain2) / sizeof(chain2[0]), chain2 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain3) / sizeof(chain3[0]), chain3 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain4) / sizeof(chain4[0]), chain4 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, 0 },
 { { sizeof(chain5) / sizeof(chain5[0]), chain5 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain6) / sizeof(chain6[0]), chain6 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain7) / sizeof(chain7[0]), chain7 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain8) / sizeof(chain8[0]), chain8 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 2, NULL }, 0 },
 { { sizeof(chain9) / sizeof(chain9[0]), chain9 },
   { 0, CERT_E_CHAINING, 0, -1, NULL }, 0 },
 { { sizeof(chain10) / sizeof(chain10[0]), chain10 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain11) / sizeof(chain11[0]), chain11 },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 1, NULL }, 0 },
 { { sizeof(chain12) / sizeof(chain12[0]), chain12 },
   { 0, TRUST_E_CERT_SIGNATURE, 0, 1, NULL }, 0 },
 { { sizeof(selfSignedChain) / sizeof(selfSignedChain[0]), selfSignedChain },
   { 0, CERT_E_UNTRUSTEDROOT, 0, 0, NULL }, 0 },
};

static ChainPolicyCheck basicConstraintsPolicyCheck[] = {
 { { sizeof(chain0) / sizeof(chain0[0]), chain0 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain1) / sizeof(chain1[0]), chain1 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain2) / sizeof(chain2[0]), chain2 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain3) / sizeof(chain3[0]), chain3 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, 0 },
 { { sizeof(chain4) / sizeof(chain4[0]), chain4 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL }, 0 },
 { { sizeof(chain5) / sizeof(chain5[0]), chain5 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain6) / sizeof(chain6[0]), chain6 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain7) / sizeof(chain7[0]), chain7 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain8) / sizeof(chain8[0]), chain8 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL },
   TODO_ERROR | TODO_CHAINS | TODO_ELEMENTS },
 { { sizeof(chain9) / sizeof(chain9[0]), chain9 },
   { 0, TRUST_E_BASIC_CONSTRAINTS, 0, 1, NULL },
   TODO_ERROR | TODO_CHAINS | TODO_ELEMENTS },
 { { sizeof(chain10) / sizeof(chain10[0]), chain10 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain11) / sizeof(chain11[0]), chain11 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(chain12) / sizeof(chain12[0]), chain12 },
   { 0, 0, -1, -1, NULL }, 0 },
 { { sizeof(selfSignedChain) / sizeof(selfSignedChain[0]), selfSignedChain },
   { 0, 0, -1, -1, NULL }, 0 },
};

static void checkChainPolicyStatus(LPCSTR policy, ChainPolicyCheck *check,
 DWORD testIndex)
{
    PCCERT_CHAIN_CONTEXT chain = getChain(&check->certs, 0, TRUE, &oct2007,
     check->todo, testIndex);

    if (chain)
    {
        CERT_CHAIN_POLICY_STATUS policyStatus = { 0 };
        BOOL ret = CertVerifyCertificateChainPolicy(policy, chain, NULL,
         &policyStatus);

        if (check->todo & TODO_POLICY)
            todo_wine ok(ret, "%d: CertVerifyCertificateChainPolicy failed: %08x\n",
             testIndex, GetLastError());
        else
            ok(ret, "%d: CertVerifyCertificateChainPolicy failed: %08x\n",
             testIndex, GetLastError());
        if (ret)
        {
            if (check->todo & TODO_ERROR)
                todo_wine ok(policyStatus.dwError == check->status.dwError,
                 "%d: expected %08x, got %08x\n", testIndex,
                 check->status.dwError, policyStatus.dwError);
            else
                ok(policyStatus.dwError == check->status.dwError,
                 "%d: expected %08x, got %08x\n", testIndex,
                 check->status.dwError, policyStatus.dwError);
            if (check->todo & TODO_CHAINS)
                todo_wine ok(policyStatus.lChainIndex ==
                 check->status.lChainIndex, "%d: expected %d, got %d\n",
                 testIndex, check->status.lChainIndex,
                 policyStatus.lChainIndex);
            else
                ok(policyStatus.lChainIndex == check->status.lChainIndex,
                 "%d: expected %d, got %d\n", testIndex,
                 check->status.lChainIndex, policyStatus.lChainIndex);
            if (check->todo & TODO_ELEMENTS)
                todo_wine ok(policyStatus.lElementIndex ==
                 check->status.lElementIndex,
                 "%d: expected %d, got %d\n", testIndex,
                 check->status.lElementIndex, policyStatus.lElementIndex);
            else
                ok(policyStatus.lElementIndex == check->status.lElementIndex,
                 "%d: expected %d, got %d\n", testIndex,
                 check->status.lElementIndex, policyStatus.lElementIndex);
        }
        CertFreeCertificateChain(chain);
    }
}

static void testVerifyCertChainPolicy(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    CERT_CHAIN_PARA chainPara = { sizeof(CERT_CHAIN_PARA), { 0 } };
    PCCERT_CHAIN_CONTEXT chain;
    CERT_CHAIN_POLICY_STATUS policyStatus = { 0 };
    CERT_CHAIN_POLICY_PARA policyPara = { 0 };
    DWORD i;

    /* Crash
    ret = CertVerifyCertificateChainPolicy(NULL, NULL, NULL, NULL);
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL, NULL,
     NULL);
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL,
     &chainPara, NULL);
     */
    SetLastError(0xdeadbeef);
    ret = CertVerifyCertificateChainPolicy(NULL, NULL, NULL, &policyStatus);
    ok(!ret && GetLastError() == ERROR_FILE_NOT_FOUND,
     "Expected ERROR_FILE_NOT_FOUND, got %08x\n", GetLastError());
    /* Crashes
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, NULL, NULL,
     &policyStatus);
     */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, selfSignedCert,
     sizeof(selfSignedCert));
    CertGetCertificateChain(NULL, cert, NULL, NULL, &chainPara, 0, NULL,
     &chain);
    /* Crash
    ret = CertVerifyCertificateChainPolicy(NULL, chain, NULL, NULL);
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain, NULL,
     NULL);
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain,
     &chainPara, NULL);
     */
    /* Size of policy status is apparently ignored, as is pChainPolicyPara */
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain, NULL,
     &policyStatus);
    ok(ret, "CertVerifyCertificateChainPolicy failed: %08x\n", GetLastError());
    ok(policyStatus.dwError == CERT_E_UNTRUSTEDROOT,
     "Expected CERT_E_UNTRUSTEDROOT, got %08x\n", policyStatus.dwError);
    ok(policyStatus.lChainIndex == 0 && policyStatus.lElementIndex == 0,
     "Expected both indexes 0, got %d, %d\n", policyStatus.lChainIndex,
     policyStatus.lElementIndex);
    ret = CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE, chain,
     &policyPara, &policyStatus);
    ok(ret, "CertVerifyCertificateChainPolicy failed: %08x\n", GetLastError());
    ok(policyStatus.dwError == CERT_E_UNTRUSTEDROOT,
     "Expected CERT_E_UNTRUSTEDROOT, got %08x\n", policyStatus.dwError);
    ok(policyStatus.lChainIndex == 0 && policyStatus.lElementIndex == 0,
     "Expected both indexes 0, got %d, %d\n", policyStatus.lChainIndex,
     policyStatus.lElementIndex);
    CertFreeCertificateChain(chain);

    for (i = 0;
     i < sizeof(basePolicyCheck) / sizeof(basePolicyCheck[0]); i++)
        checkChainPolicyStatus(CERT_CHAIN_POLICY_BASE, &basePolicyCheck[i], i);
    /* The authenticode policy doesn't seem to check anything beyond the base
     * policy.  It might check for chains signed by the MS test cert, but none
     * of these chains is.
     */
    for (i = 0; i <
     sizeof(authenticodePolicyCheck) / sizeof(authenticodePolicyCheck[0]); i++)
        checkChainPolicyStatus(CERT_CHAIN_POLICY_AUTHENTICODE,
         &authenticodePolicyCheck[i], i);
    for (i = 0; i <
     sizeof(basicConstraintsPolicyCheck) / sizeof(basicConstraintsPolicyCheck[0]);
     i++)
        checkChainPolicyStatus(CERT_CHAIN_POLICY_BASIC_CONSTRAINTS,
         &basicConstraintsPolicyCheck[i], i);
}

START_TEST(chain)
{
    testCreateCertChainEngine();
    testVerifyCertChainPolicy();
    testGetCertChain();
}
