/* Unit tests for treeview.
 *
 * Copyright 2005 Krzysztof Foltman
 * Copyright 2007 Christopher James Peterson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <assert.h>
#include <stdarg.h>

#include "windef.h"
#include "winbase.h"
#include "wingdi.h"
#include "winuser.h"
#include "winnls.h"
#include "winreg.h"
#include "commctrl.h" 

#include "wine/test.h"
#include "v6util.h"
#include "msg.h"

static const char *TEST_CALLBACK_TEXT = "callback_text";

#define NUM_MSG_SEQUENCES   2
#define TREEVIEW_SEQ_INDEX  0
#define PARENT_SEQ_INDEX    1

#define expect(expected, got) ok(got == expected, "Expected %d, got %d\n", expected, got)

static struct msg_sequence *sequences[NUM_MSG_SEQUENCES];
static struct msg_sequence *item_sequence[1];

static const struct message FillRootSeq[] = {
    { TVM_INSERTITEM, sent },
    { TVM_INSERTITEM, sent },
    { 0 }
};

static const struct message rootnone_select_seq[] = {
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { 0 }
};

static const struct message rootchild_select_seq[] = {
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { TVM_SELECTITEM, sent|wparam, 9 },
    { 0 }
};

static const struct message getitemtext_seq[] = {
    { TVM_INSERTITEMA, sent },
    { TVM_GETITEMA, sent },
    { TVM_DELETEITEM, sent },
    { 0 }
};

static const struct message focus_seq[] = {
    { TVM_INSERTITEM, sent },
    { TVM_INSERTITEM, sent },
    { TVM_SELECTITEM, sent|wparam, 9 },
    /* The following end up out of order in wine */
    { WM_WINDOWPOSCHANGING, sent|defwinproc },
    { WM_NCCALCSIZE, sent|wparam|defwinproc, TRUE },
    { WM_WINDOWPOSCHANGED, sent|defwinproc },
    { WM_SIZE, sent|defwinproc },
    { WM_PAINT, sent|defwinproc },
    { WM_NCPAINT, sent|wparam|defwinproc, 1 },
    { WM_ERASEBKGND, sent|defwinproc },
    { TVM_EDITLABEL, sent },
    { WM_COMMAND, sent|wparam|defwinproc, MAKEWPARAM(0, EN_UPDATE) },
    { WM_COMMAND, sent|wparam|defwinproc, MAKEWPARAM(0, EN_CHANGE) },
    { WM_PARENTNOTIFY, sent|wparam|defwinproc, MAKEWPARAM(WM_CREATE, 0) },
    { WM_KILLFOCUS, sent|defwinproc },
    { WM_PAINT, sent|defwinproc },
    { WM_IME_SETCONTEXT, sent|defwinproc|optional },
    { WM_COMMAND, sent|wparam|defwinproc, MAKEWPARAM(0, EN_SETFOCUS) },
    { WM_ERASEBKGND, sent|defwinproc|optional },
    { WM_CTLCOLOREDIT, sent|defwinproc|optional },
    { WM_CTLCOLOREDIT, sent|defwinproc|optional },
    { 0 }
};

static const struct message test_get_set_bkcolor_seq[] = {
    { TVM_GETBKCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETBKCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_GETBKCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETBKCOLOR, sent|wparam|lparam, 0, 0x00ffffff },
    { TVM_GETBKCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETBKCOLOR, sent|wparam|lparam, 0, -1 },
    { 0 }
};

static const struct message test_get_set_imagelist_seq[] = {
    { TVM_SETIMAGELIST, sent|wparam|lparam, 0, 0 },
    { TVM_GETIMAGELIST, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_indent_seq[] = {
    { TVM_SETINDENT, sent|wparam|lparam, 0, 0 },
    { TVM_GETINDENT, sent|wparam|lparam, 0, 0 },
    /* The actual amount to indent is dependent on the system for this message */
    { TVM_SETINDENT, sent },
    { TVM_GETINDENT, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_insertmarkcolor_seq[] = {
    { TVM_SETINSERTMARKCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_GETINSERTMARKCOLOR, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_item_seq[] = {
    { TVM_GETITEMA, sent },
    { TVM_SETITEMA, sent },
    { TVM_GETITEMA, sent },
    { TVM_SETITEMA, sent },
    { 0 }
};

static const struct message test_get_set_itemheight_seq[] = {
    { TVM_GETITEMHEIGHT, sent|wparam|lparam, 0, 0 },
    { TVM_SETITEMHEIGHT, sent|wparam|lparam, -1, 0 },
    { TVM_GETITEMHEIGHT, sent|wparam|lparam, 0, 0 },
    { TVM_SETITEMHEIGHT, sent|lparam, 0xcccccccc, 0 },
    { TVM_GETITEMHEIGHT, sent|wparam|lparam|optional, 0, 0 },
    { TVM_SETITEMHEIGHT, sent|wparam|lparam|optional, 9, 0 },
    { TVM_GETITEMHEIGHT, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_scrolltime_seq[] = {
    { TVM_SETSCROLLTIME, sent|wparam|lparam, 20, 0 },
    { TVM_GETSCROLLTIME, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_textcolor_seq[] = {
    { TVM_GETTEXTCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETTEXTCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_GETTEXTCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETTEXTCOLOR, sent|wparam|lparam, 0, RGB(255, 255, 255) },
    { TVM_GETTEXTCOLOR, sent|wparam|lparam, 0, 0 },
    { TVM_SETTEXTCOLOR, sent|wparam|lparam, 0, CLR_NONE },
    { 0 }
};

static const struct message test_get_set_tooltips_seq[] = {
    { WM_KILLFOCUS,    sent },
    { WM_IME_SETCONTEXT, sent|optional },
    { WM_IME_NOTIFY, sent|optional },
    { TVM_SETTOOLTIPS, sent|wparam|lparam, 0, 0 },
    { TVM_GETTOOLTIPS, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message test_get_set_unicodeformat_seq[] = {
    { TVM_SETUNICODEFORMAT, sent|wparam|lparam, TRUE, 0 },
    { TVM_GETUNICODEFORMAT, sent|wparam|lparam, 0, 0 },
    { TVM_SETUNICODEFORMAT, sent|wparam|lparam, 0, 0 },
    { TVM_GETUNICODEFORMAT, sent|wparam|lparam, 0, 0 },
    { TVM_SETUNICODEFORMAT, sent|wparam|lparam, 0, 0 },
    { 0 }
};

static const struct message parent_expand_seq[] = {
    { WM_NOTIFY, sent|id, 0, 0, TVN_ITEMEXPANDINGA },
    { WM_NOTIFY, sent|id, 0, 0, TVN_ITEMEXPANDEDA },
    { 0 }
};

static const struct message parent_singleexpand_seq[] = {
    { WM_NOTIFY, sent|id, 0, 0, TVN_SELCHANGINGA },
    { WM_NOTIFY, sent|id, 0, 0, TVN_SELCHANGEDA },
    { WM_NOTIFY, sent|id, 0, 0, TVN_SINGLEEXPAND },
    { WM_NOTIFY, sent|id, 0, 0, TVN_ITEMEXPANDINGA },
    { WM_NOTIFY, sent|id, 0, 0, TVN_ITEMEXPANDEDA },
    { 0 }
};

static const struct message empty_seq[] = {
    { 0 }
};

static HWND hMainWnd;

static HTREEITEM hRoot, hChild;

static int pos = 0;
static char sequence[256];

static void Clear(void)
{
    pos = 0;
    sequence[0] = '\0';
}

static void AddItem(char ch)
{
    sequence[pos++] = ch;
    sequence[pos] = '\0';
}

static void IdentifyItem(HTREEITEM hItem)
{
    if (hItem == hRoot) {
        AddItem('R');
        return;
    }
    if (hItem == hChild) {
        AddItem('C');
        return;
    }
    if (hItem == NULL) {
        AddItem('n');
        return;
    }
    AddItem('?');
}

/* This function hooks in and records all messages to the treeview control */
static LRESULT WINAPI TreeviewWndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static LONG defwndproc_counter = 0;
    LRESULT ret;
    struct message msg;
    WNDPROC lpOldProc = (WNDPROC)GetWindowLongPtrA(hwnd, GWLP_USERDATA);

    msg.message = message;
    msg.flags = sent|wparam|lparam;
    if (defwndproc_counter) msg.flags |= defwinproc;
    msg.wParam = wParam;
    msg.lParam = lParam;
    add_message(sequences, TREEVIEW_SEQ_INDEX, &msg);

    defwndproc_counter++;
    ret = CallWindowProcA(lpOldProc, hwnd, message, wParam, lParam);
    defwndproc_counter--;

    return ret;
}

static HWND create_treeview_control(void)
{
    WNDPROC pOldWndProc;
    HWND hTree;

    hTree = CreateWindowExA(WS_EX_CLIENTEDGE, WC_TREEVIEWA, NULL, WS_CHILD|WS_VISIBLE|
            TVS_LINESATROOT|TVS_HASLINES|TVS_HASBUTTONS|TVS_EDITLABELS,
            0, 0, 120, 100, hMainWnd, (HMENU)100, GetModuleHandleA(0), 0);

    SetFocus(hTree);

    /* Record the old WNDPROC so we can call it after recording the messages */
    pOldWndProc = (WNDPROC)SetWindowLongPtrA(hTree, GWLP_WNDPROC, (LONG_PTR)TreeviewWndProc);
    SetWindowLongPtrA(hTree, GWLP_USERDATA, (LONG_PTR)pOldWndProc);

    return hTree;
}

static void fill_tree(HWND hTree)
{
    TVINSERTSTRUCTA ins;
    static CHAR root[]  = "Root",
                child[] = "Child";

    ins.hParent = TVI_ROOT;
    ins.hInsertAfter = TVI_ROOT;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = root;
    hRoot = TreeView_InsertItem(hTree, &ins);

    ins.hParent = hRoot;
    ins.hInsertAfter = TVI_FIRST;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = child;
    hChild = TreeView_InsertItem(hTree, &ins);
}

static void test_fillroot(void)
{
    TVITEMA tvi;
    HWND hTree;

    hTree = create_treeview_control();

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    fill_tree(hTree);

    Clear();
    AddItem('A');
    assert(hRoot);
    AddItem('B');
    assert(hChild);
    AddItem('.');
    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, FillRootSeq, "FillRoot", FALSE);
    ok(!strcmp(sequence, "AB."), "Item creation\n");

    /* UMLPad 1.15 depends on this being not -1 (I_IMAGECALLBACK) */
    tvi.hItem = hRoot;
    tvi.mask = TVIF_IMAGE | TVIF_SELECTEDIMAGE;
    SendMessage( hTree, TVM_GETITEMA, 0, (LPARAM)&tvi );
    ok(tvi.iImage == 0, "tvi.iImage=%d\n", tvi.iImage);
    ok(tvi.iSelectedImage == 0, "tvi.iSelectedImage=%d\n", tvi.iSelectedImage);

    DestroyWindow(hTree);
}

static void test_callback(void)
{
    HTREEITEM hRoot;
    HTREEITEM hItem1, hItem2;
    TVINSERTSTRUCTA ins;
    TVITEMA tvi;
    CHAR test_string[] = "Test_string";
    CHAR buf[128];
    LRESULT ret;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    ret = TreeView_DeleteAllItems(hTree);
    ok(ret == TRUE, "ret\n");
    ins.hParent = TVI_ROOT;
    ins.hInsertAfter = TVI_ROOT;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = LPSTR_TEXTCALLBACK;
    hRoot = TreeView_InsertItem(hTree, &ins);
    assert(hRoot);

    tvi.hItem = hRoot;
    tvi.mask = TVIF_TEXT;
    tvi.pszText = buf;
    tvi.cchTextMax = sizeof(buf)/sizeof(buf[0]);
    ret = TreeView_GetItem(hTree, &tvi);
    ok(ret == 1, "ret\n");
    ok(strcmp(tvi.pszText, TEST_CALLBACK_TEXT) == 0, "Callback item text mismatch %s vs %s\n",
        tvi.pszText, TEST_CALLBACK_TEXT);

    ins.hParent = hRoot;
    ins.hInsertAfter = TVI_FIRST;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = test_string;
    hItem1 = TreeView_InsertItem(hTree, &ins);
    assert(hItem1);

    tvi.hItem = hItem1;
    ret = TreeView_GetItem(hTree, &tvi);
    ok(ret == TRUE, "ret\n");
    ok(strcmp(tvi.pszText, test_string) == 0, "Item text mismatch %s vs %s\n",
        tvi.pszText, test_string);

    /* undocumented: pszText of NULL also means LPSTR_CALLBACK: */
    tvi.pszText = NULL;
    ret = TreeView_SetItem(hTree, &tvi);
    ok(ret == 1, "Expected SetItem return 1, got %ld\n", ret);
    tvi.pszText = buf;
    ret = TreeView_GetItem(hTree, &tvi);
    ok(ret == TRUE, "Expected GetItem return TRUE, got %ld\n", ret);
    ok(strcmp(tvi.pszText, TEST_CALLBACK_TEXT) == 0, "Item text mismatch %s vs %s\n",
        tvi.pszText, TEST_CALLBACK_TEXT);

    U(ins).item.pszText = NULL;
    hItem2 = TreeView_InsertItem(hTree, &ins);
    assert(hItem2);
    tvi.hItem = hItem2;
    memset(buf, 0, sizeof(buf));
    ret = TreeView_GetItem(hTree, &tvi);
    ok(ret == TRUE, "Expected GetItem return TRUE, got %ld\n", ret);
    ok(strcmp(tvi.pszText, TEST_CALLBACK_TEXT) == 0, "Item text mismatch %s vs %s\n",
        tvi.pszText, TEST_CALLBACK_TEXT);

    DestroyWindow(hTree);
}

static void test_select(void)
{
    BOOL r;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    /* root-none select tests */
    flush_sequences(sequences, NUM_MSG_SEQUENCES);
    r = TreeView_SelectItem(hTree, NULL);
    expect(TRUE, r);
    Clear();
    AddItem('1');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('2');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('3');
    r = TreeView_SelectItem(hTree, NULL);
    expect(TRUE, r);
    AddItem('4');
    r = TreeView_SelectItem(hTree, NULL);
    expect(TRUE, r);
    AddItem('5');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('.');
    ok(!strcmp(sequence, "1(nR)nR23(Rn)Rn45(nR)nR."), "root-none select test\n");
    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, rootnone_select_seq,
                "root-none select seq", FALSE);

    /* root-child select tests */
    flush_sequences(sequences, NUM_MSG_SEQUENCES);
    r = TreeView_SelectItem(hTree, NULL);
    expect(TRUE, r);

    Clear();
    AddItem('1');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('2');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('3');
    r = TreeView_SelectItem(hTree, hChild);
    expect(TRUE, r);
    AddItem('4');
    r = TreeView_SelectItem(hTree, hChild);
    expect(TRUE, r);
    AddItem('5');
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    AddItem('.');
    ok(!strcmp(sequence, "1(nR)nR23(RC)RC45(CR)CR."), "root-child select test\n");
    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, rootchild_select_seq,
                "root-child select seq", FALSE);

    DestroyWindow(hTree);
}

static void test_getitemtext(void)
{
    TVINSERTSTRUCTA ins;
    HTREEITEM hChild;
    TVITEMA tvi;
    HWND hTree;

    CHAR szBuffer[80] = "Blah";
    int nBufferSize = sizeof(szBuffer)/sizeof(CHAR);

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* add an item without TVIF_TEXT mask and pszText == NULL */
    ins.hParent = hRoot;
    ins.hInsertAfter = TVI_ROOT;
    U(ins).item.mask = 0;
    U(ins).item.pszText = NULL;
    U(ins).item.cchTextMax = 0;
    hChild = TreeView_InsertItem(hTree, &ins);
    assert(hChild);

    /* retrieve it with TVIF_TEXT mask */
    tvi.hItem = hChild;
    tvi.mask = TVIF_TEXT;
    tvi.cchTextMax = nBufferSize;
    tvi.pszText = szBuffer;

    SendMessageA( hTree, TVM_GETITEMA, 0, (LPARAM)&tvi );
    ok(!strcmp(szBuffer, ""), "szBuffer=\"%s\", expected \"\"\n", szBuffer);
    ok(SendMessageA(hTree, TVM_DELETEITEM, 0, (LPARAM)hChild), "DeleteItem failed\n");
    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, getitemtext_seq, "get item text seq", FALSE);

    DestroyWindow(hTree);
}

static void test_focus(void)
{
    TVINSERTSTRUCTA ins;
    static CHAR child1[]  = "Edit",
                child2[]  = "A really long string";
    HTREEITEM hChild1, hChild2;
    HWND hTree;
    HWND hEdit;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* This test verifies that when a label is being edited, scrolling
     * the treeview does not cause the label to lose focus. To test
     * this, first some additional entries are added to generate
     * scrollbars.
     */
    ins.hParent = hRoot;
    ins.hInsertAfter = hChild;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = child1;
    hChild1 = TreeView_InsertItem(hTree, &ins);
    assert(hChild1);
    ins.hInsertAfter = hChild1;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = child2;
    hChild2 = TreeView_InsertItem(hTree, &ins);
    assert(hChild2);

    ShowWindow(hMainWnd,SW_SHOW);
    SendMessageA(hTree, TVM_SELECTITEM, TVGN_CARET, (LPARAM)hChild);
    hEdit = TreeView_EditLabel(hTree, hChild);
    ScrollWindowEx(hTree, -10, 0, NULL, NULL, NULL, NULL, SW_SCROLLCHILDREN);
    ok(GetFocus() == hEdit, "Edit control should have focus\n");
    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, focus_seq, "focus test", TRUE);

    DestroyWindow(hTree);
}

static void test_get_set_bkcolor(void)
{
    COLORREF crColor = RGB(0,0,0);
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* If the value is -1, the control is using the system color for the background color. */
    crColor = (COLORREF)SendMessage( hTree, TVM_GETBKCOLOR, 0, 0 );
    ok(crColor == -1, "Default background color reported as 0x%.8x\n", crColor);

    /* Test for black background */
    SendMessage( hTree, TVM_SETBKCOLOR, 0, RGB(0,0,0) );
    crColor = (COLORREF)SendMessage( hTree, TVM_GETBKCOLOR, 0, 0 );
    ok(crColor == RGB(0,0,0), "Black background color reported as 0x%.8x\n", crColor);

    /* Test for white background */
    SendMessage( hTree, TVM_SETBKCOLOR, 0, RGB(255,255,255) );
    crColor = (COLORREF)SendMessage( hTree, TVM_GETBKCOLOR, 0, 0 );
    ok(crColor == RGB(255,255,255), "White background color reported as 0x%.8x\n", crColor);

    /* Reset the default background */
    SendMessage( hTree, TVM_SETBKCOLOR, 0, -1 );

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_bkcolor_seq,
        "test get set bkcolor", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_imagelist(void)
{
    HIMAGELIST hImageList = NULL;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* Test a NULL HIMAGELIST */
    SendMessage( hTree, TVM_SETIMAGELIST, TVSIL_NORMAL, (LPARAM)hImageList );
    hImageList = (HIMAGELIST)SendMessage( hTree, TVM_GETIMAGELIST, TVSIL_NORMAL, 0 );
    ok(hImageList == NULL, "NULL image list, reported as 0x%p, expected 0.\n", hImageList);

    /* TODO: Test an actual image list */

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_imagelist_seq,
        "test get imagelist", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_indent(void)
{
    int ulIndent = -1;
    int ulMinIndent = -1;
    int ulMoreThanTwiceMin = -1;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* Finding the minimum indent */
    SendMessage( hTree, TVM_SETINDENT, 0, 0 );
    ulMinIndent = (int)SendMessage( hTree, TVM_GETINDENT, 0, 0 );

    /* Checking an indent that is more than twice the default indent */
    ulMoreThanTwiceMin = 2*ulMinIndent+1;
    SendMessage( hTree, TVM_SETINDENT, ulMoreThanTwiceMin, 0 );
    ulIndent = (DWORD)SendMessage( hTree, TVM_GETINDENT, 0, 0 );
    ok(ulIndent == ulMoreThanTwiceMin, "Indent reported as %d, expected %d\n", ulIndent, ulMoreThanTwiceMin);

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_indent_seq,
        "test get set indent", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_insertmark(void)
{
    COLORREF crColor = RGB(0,0,0);
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    SendMessage( hTree, TVM_SETINSERTMARKCOLOR, 0, crColor );
    crColor = (COLORREF)SendMessage( hTree, TVM_GETINSERTMARKCOLOR, 0, 0 );
    ok(crColor == RGB(0,0,0), "Insert mark color reported as 0x%.8x, expected 0x00000000\n", crColor);

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_insertmarkcolor_seq,
        "test get set insertmark color", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_item(void)
{
    TVITEMA tviRoot = {0};
    int nBufferSize = 80;
    char szBuffer[80] = {0};
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* Test the root item */
    tviRoot.hItem = hRoot;
    tviRoot.mask = TVIF_TEXT;
    tviRoot.cchTextMax = nBufferSize;
    tviRoot.pszText = szBuffer;
    SendMessage( hTree, TVM_GETITEMA, 0, (LPARAM)&tviRoot );
    ok(!strcmp("Root", szBuffer), "GetItem: szBuffer=\"%s\", expected \"Root\"\n", szBuffer);

    /* Change the root text */
    strncpy(szBuffer, "Testing123", nBufferSize);
    SendMessage( hTree, TVM_SETITEMA, 0, (LPARAM)&tviRoot );
    memset(szBuffer, 0, nBufferSize);
    SendMessage( hTree, TVM_GETITEMA, 0, (LPARAM)&tviRoot );
    ok(!strcmp("Testing123", szBuffer), "GetItem: szBuffer=\"%s\", expected \"Testing123\"\n", szBuffer);

    /* Reset the root text */
    memset(szBuffer, 0, nBufferSize);
    strncpy(szBuffer, "Root", nBufferSize);
    SendMessage( hTree, TVM_SETITEMA, 0, (LPARAM)&tviRoot );

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_item_seq,
        "test get set item", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_itemheight(void)
{
    int ulOldHeight = 0;
    int ulNewHeight = 0;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* Assuming default height to begin with */
    ulOldHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );

    /* Explicitly setting and getting the default height */
    SendMessage( hTree, TVM_SETITEMHEIGHT, -1, 0 );
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == ulOldHeight, "Default height not set properly, reported %d, expected %d\n", ulNewHeight, ulOldHeight);

    /* Explicitly setting and getting the height of twice the normal */
    SendMessage( hTree, TVM_SETITEMHEIGHT, 2*ulOldHeight, 0 );
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 2*ulOldHeight, "New height not set properly, reported %d, expected %d\n", ulNewHeight, 2*ulOldHeight);

    /* Assuming tree doesn't have TVS_NONEVENHEIGHT set, so a set of 9 will round down to 8 */
    SendMessage( hTree, TVM_SETITEMHEIGHT, 9, 0 );
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 8, "Uneven height not set properly, reported %d, expected %d\n", ulNewHeight, 8);

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_itemheight_seq,
        "test get set item height", FALSE);

    /* without TVS_NONEVENHEIGHT */
    SetWindowLong(hTree, GWL_STYLE, GetWindowLong(hTree, GWL_STYLE) & ~TVS_NONEVENHEIGHT);
    /* odd value */
    ulOldHeight = SendMessage( hTree, TVM_SETITEMHEIGHT, 3, 0);
    ok(ulOldHeight == 8, "got %d, expected %d\n", ulOldHeight, 8);
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 2, "got %d, expected %d\n", ulNewHeight, 2);

    ulOldHeight = SendMessage( hTree, TVM_SETITEMHEIGHT, 4, 0);
    ok(ulOldHeight == 2, "got %d, expected %d\n", ulOldHeight, 2);
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 4, "got %d, expected %d\n", ulNewHeight, 4);

    /* with TVS_NONEVENHEIGHT */
    SetWindowLong(hTree, GWL_STYLE, GetWindowLong(hTree, GWL_STYLE) | TVS_NONEVENHEIGHT);
    /* odd value */
    ulOldHeight = SendMessage( hTree, TVM_SETITEMHEIGHT, 3, 0);
    ok(ulOldHeight == 4, "got %d, expected %d\n", ulOldHeight, 4);
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 3, "got %d, expected %d\n", ulNewHeight, 3);
    /* even value */
    ulOldHeight = SendMessage( hTree, TVM_SETITEMHEIGHT, 10, 0);
    ok(ulOldHeight == 3, "got %d, expected %d\n", ulOldHeight, 3);
    ulNewHeight = (int) SendMessage( hTree, TVM_GETITEMHEIGHT, 0, 0 );
    ok(ulNewHeight == 10, "got %d, expected %d\n", ulNewHeight, 10);

    DestroyWindow(hTree);
}

static void test_get_set_scrolltime(void)
{
    int ulExpectedTime = 20;
    int ulTime = 0;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    SendMessage( hTree, TVM_SETSCROLLTIME, ulExpectedTime, 0 );
    ulTime = (int)SendMessage( hTree, TVM_GETSCROLLTIME, 0, 0 );
    ok(ulTime == ulExpectedTime, "Scroll time reported as %d, expected %d\n", ulTime, ulExpectedTime);

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_scrolltime_seq,
        "test get set scroll time", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_textcolor(void)
{
    /* If the value is -1, the control is using the system color for the text color. */
    COLORREF crColor = RGB(0,0,0);
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    crColor = (COLORREF)SendMessage( hTree, TVM_GETTEXTCOLOR, 0, 0 );
    ok(crColor == -1, "Default text color reported as 0x%.8x\n", crColor);

    /* Test for black text */
    SendMessage( hTree, TVM_SETTEXTCOLOR, 0, RGB(0,0,0) );
    crColor = (COLORREF)SendMessage( hTree, TVM_GETTEXTCOLOR, 0, 0 );
    ok(crColor == RGB(0,0,0), "Black text color reported as 0x%.8x\n", crColor);

    /* Test for white text */
    SendMessage( hTree, TVM_SETTEXTCOLOR, 0, RGB(255,255,255) );
    crColor = (COLORREF)SendMessage( hTree, TVM_GETTEXTCOLOR, 0, 0 );
    ok(crColor == RGB(255,255,255), "White text color reported as 0x%.8x\n", crColor);

    /* Reset the default text color */
    SendMessage( hTree, TVM_SETTEXTCOLOR, 0, CLR_NONE );

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_textcolor_seq,
        "test get set text color", FALSE);

    DestroyWindow(hTree);
}

static void test_get_set_tooltips(void)
{
    HWND hwndLastToolTip = NULL;
    HWND hPopupTreeView;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* show even WS_POPUP treeview don't send NM_TOOLTIPSCREATED */
    hPopupTreeView = CreateWindow(WC_TREEVIEW, NULL, WS_POPUP|WS_VISIBLE, 0, 0, 100, 100, hMainWnd, NULL, NULL, NULL);
    DestroyWindow(hPopupTreeView);

    /* Testing setting a NULL ToolTip */
    SendMessage( hTree, TVM_SETTOOLTIPS, 0, 0 );
    hwndLastToolTip = (HWND)SendMessage( hTree, TVM_GETTOOLTIPS, 0, 0 );
    ok(hwndLastToolTip == NULL, "NULL tool tip, reported as 0x%p, expected 0.\n", hwndLastToolTip);

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_tooltips_seq,
        "test get set tooltips", TRUE);

    /* TODO: Add a test of an actual tooltip */
    DestroyWindow(hTree);
}

static void test_get_set_unicodeformat(void)
{
    BOOL bPreviousSetting = 0;
    BOOL bNewSetting = 0;
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);

    /* Set to Unicode */
    bPreviousSetting = (BOOL)SendMessage( hTree, TVM_SETUNICODEFORMAT, 1, 0 );
    bNewSetting = (BOOL)SendMessage( hTree, TVM_GETUNICODEFORMAT, 0, 0 );
    ok(bNewSetting == 1, "Unicode setting did not work.\n");

    /* Set to ANSI */
    SendMessage( hTree, TVM_SETUNICODEFORMAT, 0, 0 );
    bNewSetting = (BOOL)SendMessage( hTree, TVM_GETUNICODEFORMAT, 0, 0 );
    ok(bNewSetting == 0, "ANSI setting did not work.\n");

    /* Revert to original setting */
    SendMessage( hTree, TVM_SETUNICODEFORMAT, bPreviousSetting, 0 );

    ok_sequence(sequences, TREEVIEW_SEQ_INDEX, test_get_set_unicodeformat_seq,
        "test get set unicode format", FALSE);

    DestroyWindow(hTree);
}

static TVITEMA g_item_expanding, g_item_expanded;
static BOOL g_get_from_expand;
static BOOL g_get_rect_in_expand;

static LRESULT CALLBACK parent_wnd_proc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static LONG defwndproc_counter = 0;
    struct message msg;
    LRESULT ret;
    RECT rect;
    HTREEITEM visibleItem;

    msg.message = message;
    msg.flags = sent|wparam|lparam;
    if (defwndproc_counter) msg.flags |= defwinproc;
    msg.wParam = wParam;
    msg.lParam = lParam;
    if (message == WM_NOTIFY && lParam) msg.id = ((NMHDR*)lParam)->code;

    /* log system messages, except for painting */
    if (message < WM_USER &&
        message != WM_PAINT &&
        message != WM_ERASEBKGND &&
        message != WM_NCPAINT &&
        message != WM_NCHITTEST &&
        message != WM_GETTEXT &&
        message != WM_GETICON &&
        message != WM_DEVICECHANGE)
    {
        trace("parent: %p, %04x, %08lx, %08lx\n", hWnd, message, wParam, lParam);
        add_message(sequences, PARENT_SEQ_INDEX, &msg);
    }

    switch(message) {
    case WM_NOTIFY:
    {
        NMHDR *pHdr = (NMHDR *)lParam;
    
        ok(pHdr->code != NM_TOOLTIPSCREATED, "Treeview should not send NM_TOOLTIPSCREATED\n");
        if (pHdr->idFrom == 100)
        {
            NMTREEVIEWA *pTreeView = (LPNMTREEVIEWA) lParam;
            switch(pHdr->code)
            {
            case TVN_SELCHANGINGA:
                AddItem('(');
                IdentifyItem(pTreeView->itemOld.hItem);
                IdentifyItem(pTreeView->itemNew.hItem);
                break;
            case TVN_SELCHANGEDA:
                AddItem(')');
                IdentifyItem(pTreeView->itemOld.hItem);
                IdentifyItem(pTreeView->itemNew.hItem);
                break;
            case TVN_GETDISPINFOA: {
                NMTVDISPINFOA *disp = (NMTVDISPINFOA *)lParam;
                if (disp->item.mask & TVIF_TEXT) {
                    lstrcpyn(disp->item.pszText, TEST_CALLBACK_TEXT, disp->item.cchTextMax);
                }
                break;
              }
            case TVN_ENDLABELEDIT: return TRUE;
            case TVN_ITEMEXPANDINGA:
                ok(pTreeView->itemNew.mask ==
                   (TVIF_HANDLE | TVIF_SELECTEDIMAGE | TVIF_IMAGE | TVIF_PARAM | TVIF_STATE),
                   "got wrong mask %x\n", pTreeView->itemNew.mask);
                ok((pTreeView->itemNew.state & TVIS_EXPANDED) == 0,
                   "got wrong state %x\n", pTreeView->itemNew.state);
                ok(pTreeView->itemOld.mask == 0,
                   "got wrong mask %x\n", pTreeView->itemOld.mask);

                if (g_get_from_expand)
                {
                  g_item_expanding.mask = TVIF_STATE;
                  g_item_expanding.hItem = hRoot;
                  ret = SendMessageA(pHdr->hwndFrom, TVM_GETITEMA, 0, (LPARAM)&g_item_expanding);
                  ok(ret == TRUE, "got %lu\n", ret);
                }
                break;
            case TVN_ITEMEXPANDEDA:
                ok(pTreeView->itemNew.mask & TVIF_STATE, "got wrong mask %x\n", pTreeView->itemNew.mask);
                ok(pTreeView->itemNew.state & (TVIS_EXPANDED|TVIS_EXPANDEDONCE),
                   "got wrong mask %x\n", pTreeView->itemNew.mask);
                ok(pTreeView->itemOld.mask == 0,
                   "got wrong mask %x\n", pTreeView->itemOld.mask);

                if (g_get_from_expand)
                {
                  g_item_expanded.mask = TVIF_STATE;
                  g_item_expanded.hItem = hRoot;
                  ret = SendMessageA(pHdr->hwndFrom, TVM_GETITEMA, 0, (LPARAM)&g_item_expanded);
                  ok(ret == TRUE, "got %lu\n", ret);
                }
                if (g_get_rect_in_expand)
                {
                  visibleItem = TreeView_GetNextItem(pHdr->hwndFrom, NULL, TVGN_FIRSTVISIBLE);
                  ok(pTreeView->itemNew.hItem == visibleItem, "expanded item == first visible item\n");
                  *(HTREEITEM*)&rect = visibleItem;
                  ok(SendMessage(pHdr->hwndFrom, TVM_GETITEMRECT, TRUE, (LPARAM)&rect), "Failed to get rect for first visible item.\n");
                  visibleItem = TreeView_GetNextItem(pHdr->hwndFrom, visibleItem, TVGN_NEXTVISIBLE);
                  *(HTREEITEM*)&rect = visibleItem;
                  ok(visibleItem != NULL, "There must be a visible item after the first visisble item.\n");
                  ok(SendMessage(pHdr->hwndFrom, TVM_GETITEMRECT, TRUE, (LPARAM)&rect), "Failed to get rect for second visible item.\n");
                }
                break;
            case TVN_DELETEITEMA:
            {
                struct message item;

                ok(pTreeView->itemNew.mask == 0, "got wrong mask 0x%x\n", pTreeView->itemNew.mask);

                ok(pTreeView->itemOld.mask == (TVIF_HANDLE | TVIF_PARAM), "got wrong mask 0x%x\n", pTreeView->itemOld.mask);
                ok(pTreeView->itemOld.hItem != NULL, "got %p\n", pTreeView->itemOld.hItem);

                memset(&item, 0, sizeof(item));
                item.lParam = (LPARAM)pTreeView->itemOld.hItem;
                add_message(item_sequence, 0, &item);

                break;
            }
            }
        }
    }

    case WM_DESTROY:
        PostQuitMessage(0);
        break;
    }

    defwndproc_counter++;
    ret = DefWindowProcA(hWnd, message, wParam, lParam);
    defwndproc_counter--;

    return ret;
}

static void test_expandinvisible(void)
{
    static CHAR nodeText[][5] = {"0", "1", "2", "3", "4"};
    TVINSERTSTRUCTA ins;
    HTREEITEM node[5];
    RECT dummyRect;
    BOOL nodeVisible;
    LRESULT ret;
    HWND hTree;

    hTree = create_treeview_control();

    /* The test builds the following tree and expands then node 1, while node 0 is collapsed.
     *
     * 0
     * |- 1
     * |  |- 2
     * |  |- 3
     * |- 4
     *
     */

    ret = TreeView_DeleteAllItems(hTree);
    ok(ret == TRUE, "ret\n");
    ins.hParent = TVI_ROOT;
    ins.hInsertAfter = TVI_ROOT;
    U(ins).item.mask = TVIF_TEXT;
    U(ins).item.pszText = nodeText[0];
    node[0] = TreeView_InsertItem(hTree, &ins);
    assert(node[0]);

    ins.hInsertAfter = TVI_LAST;
    U(ins).item.mask = TVIF_TEXT;
    ins.hParent = node[0];

    U(ins).item.pszText = nodeText[1];
    node[1] = TreeView_InsertItem(hTree, &ins);
    assert(node[1]);
    U(ins).item.pszText = nodeText[4];
    node[4] = TreeView_InsertItem(hTree, &ins);
    assert(node[4]);

    ins.hParent = node[1];

    U(ins).item.pszText = nodeText[2];
    node[2] = TreeView_InsertItem(hTree, &ins);
    assert(node[2]);
    U(ins).item.pszText = nodeText[3];
    node[3] = TreeView_InsertItem(hTree, &ins);
    assert(node[3]);


    nodeVisible = TreeView_GetItemRect(hTree, node[1], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 1 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[2], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 2 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[3], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 3 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[4], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 4 should not be visible.\n");

    ok(TreeView_Expand(hTree, node[1], TVE_EXPAND), "Expand of node 1 failed.\n");

    nodeVisible = TreeView_GetItemRect(hTree, node[1], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 1 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[2], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 2 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[3], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 3 should not be visible.\n");
    nodeVisible = TreeView_GetItemRect(hTree, node[4], &dummyRect, FALSE);
    ok(!nodeVisible, "Node 4 should not be visible.\n");

    DestroyWindow(hTree);
}

static void test_itemedit(void)
{
    DWORD r;
    HWND edit;
    TVITEMA item;
    CHAR buff[2];
    HWND hTree;

    hTree = create_treeview_control();
    fill_tree(hTree);

    /* try with null item */
    edit = (HWND)SendMessage(hTree, TVM_EDITLABEL, 0, 0);
    ok(!IsWindow(edit), "Expected valid handle\n");

    /* trigger edit */
    edit = (HWND)SendMessage(hTree, TVM_EDITLABEL, 0, (LPARAM)hRoot);
    ok(IsWindow(edit), "Expected valid handle\n");
    /* item shouldn't be selected automatically after TVM_EDITLABEL */
    r = SendMessage(hTree, TVM_GETITEMSTATE, (WPARAM)hRoot, TVIS_SELECTED);
    expect(0, r);
    /* try to cancel with wrong edit handle */
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), 0);
    expect(0, r);
    ok(IsWindow(edit), "Expected edit control to be valid\n");
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), (LPARAM)edit);
    expect(0, r);
    ok(!IsWindow(edit), "Expected edit control to be destroyed\n");
    /* try to cancel without creating edit */
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), 0);
    expect(0, r);

    /* try to cancel with wrong (not null) handle */
    edit = (HWND)SendMessage(hTree, TVM_EDITLABEL, 0, (LPARAM)hRoot);
    ok(IsWindow(edit), "Expected valid handle\n");
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), (LPARAM)hTree);
    expect(0, r);
    ok(IsWindow(edit), "Expected edit control to be valid\n");
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), (LPARAM)edit);
    expect(0, r);

    /* remove selection after starting edit */
    r = TreeView_SelectItem(hTree, hRoot);
    expect(TRUE, r);
    edit = (HWND)SendMessage(hTree, TVM_EDITLABEL, 0, (LPARAM)hRoot);
    ok(IsWindow(edit), "Expected valid handle\n");
    r = TreeView_SelectItem(hTree, NULL);
    expect(TRUE, r);
    /* alter text */
    strncpy(buff, "x", sizeof(buff)/sizeof(CHAR));
    r = SendMessage(edit, WM_SETTEXT, 0, (LPARAM)buff);
    expect(TRUE, r);
    r = SendMessage(hTree, WM_COMMAND, MAKEWPARAM(0, EN_KILLFOCUS), (LPARAM)edit);
    expect(0, r);
    ok(!IsWindow(edit), "Expected edit control to be destroyed\n");
    /* check that text is saved */
    item.mask = TVIF_TEXT;
    item.hItem = hRoot;
    item.pszText = buff;
    item.cchTextMax = sizeof(buff)/sizeof(CHAR);
    r = SendMessage(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    expect(TRUE, r);
    ok(!strcmp("x", buff), "Expected item text to change\n");

    DestroyWindow(hTree);
}

static void test_treeview_classinfo(void)
{
    WNDCLASSA cls;

    memset(&cls, 0, sizeof(cls));
    GetClassInfo(GetModuleHandleA("comctl32.dll"), WC_TREEVIEWA, &cls);
    ok(cls.hbrBackground == NULL, "Expected NULL background brush, got %p\n", cls.hbrBackground);
    ok(cls.style == (CS_GLOBALCLASS | CS_DBLCLKS), "Expected got %x\n", cls.style);
    expect(0, cls.cbClsExtra);
}

static void test_get_linecolor(void)
{
    COLORREF clr;
    HWND hTree;

    hTree = create_treeview_control();

    /* newly created control has default color */
    clr = (COLORREF)SendMessage(hTree, TVM_GETLINECOLOR, 0, 0);
    if (clr == 0)
        win_skip("TVM_GETLINECOLOR is not supported on comctl32 < 5.80\n");
    else
        expect(CLR_DEFAULT, clr);

    DestroyWindow(hTree);
}

static void test_get_insertmarkcolor(void)
{
    COLORREF clr;
    HWND hTree;

    hTree = create_treeview_control();

    /* newly created control has default color */
    clr = (COLORREF)SendMessage(hTree, TVM_GETINSERTMARKCOLOR, 0, 0);
    if (clr == 0)
        win_skip("TVM_GETINSERTMARKCOLOR is not supported on comctl32 < 5.80\n");
    else
        expect(CLR_DEFAULT, clr);

    DestroyWindow(hTree);
}

static void test_expandnotify(void)
{
    HWND hTree;
    BOOL ret;
    TVITEMA item;

    hTree = create_treeview_control();
    fill_tree(hTree);

    item.hItem = hRoot;
    item.mask = TVIF_STATE;

    item.state = TVIS_EXPANDED;
    ret = SendMessageA(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    ok(ret == TRUE, "got %d\n", ret);
    ok((item.state & TVIS_EXPANDED) == 0, "expected collapsed\n");

    /* preselect root node here */
    ret = SendMessageA(hTree, TVM_SELECTITEM, TVGN_CARET, (LPARAM)hRoot);
    ok(ret == TRUE, "got %d\n", ret);

    g_get_from_expand = TRUE;
    /* expand */
    flush_sequences(sequences, NUM_MSG_SEQUENCES);
    g_item_expanding.state = 0xdeadbeef;
    g_item_expanded.state = 0xdeadbeef;
    ret = SendMessageA(hTree, TVM_EXPAND, TVE_EXPAND, (LPARAM)hRoot);
    ok(ret == TRUE, "got %d\n", ret);
    ok(g_item_expanding.state == TVIS_SELECTED, "got state on TVN_ITEMEXPANDING 0x%08x\n",
       g_item_expanding.state);
    ok(g_item_expanded.state == (TVIS_SELECTED|TVIS_EXPANDED), "got state on TVN_ITEMEXPANDED 0x%08x\n",
       g_item_expanded.state);
    ok_sequence(sequences, PARENT_SEQ_INDEX, parent_expand_seq, "expand notifications", FALSE);
    g_get_from_expand = FALSE;

    /* check that it's expanded */
    item.state = TVIS_EXPANDED;
    ret = SendMessageA(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    ok(ret == TRUE, "got %d\n", ret);
    ok((item.state & TVIS_EXPANDED) == TVIS_EXPANDED, "expected expanded\n");

    /* collapse */
    flush_sequences(sequences, NUM_MSG_SEQUENCES);
    ret = SendMessageA(hTree, TVM_EXPAND, TVE_COLLAPSE, (LPARAM)hRoot);
    ok(ret == TRUE, "got %d\n", ret);
    item.state = TVIS_EXPANDED;
    ret = SendMessageA(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    ok(ret == TRUE, "got %d\n", ret);
    ok((item.state & TVIS_EXPANDED) == 0, "expected collapsed\n");
    /* all next collapse/expand attempts won't produce any notifications,
       the only way is to reset with all children removed */
    ok_sequence(sequences, PARENT_SEQ_INDEX, empty_seq, "collapse after expand notifications", FALSE);

    DestroyWindow(hTree);

    /* test TVM_GETITEMRECT inside TVN_ITEMEXPANDED notification */
    hTree = create_treeview_control();
    fill_tree(hTree);
    g_get_rect_in_expand = TRUE;
    ret = TreeView_Select(hTree, hChild, TVGN_CARET);
    g_get_rect_in_expand = FALSE;
    ok(ret, "got %d\n", ret);
    DestroyWindow(hTree);
}

static void test_expandedimage(void)
{
    TVITEMEXA item;
    HWND hTree;
    BOOL ret;

    hTree = create_treeview_control();
    fill_tree(hTree);

    item.mask = TVIF_EXPANDEDIMAGE;
    item.iExpandedImage = 1;
    item.hItem = hRoot;
    ret = SendMessageA(hTree, TVM_SETITEMA, 0, (LPARAM)&item);
    ok(ret, "got %d\n", ret);

    item.mask = TVIF_EXPANDEDIMAGE;
    item.iExpandedImage = -1;
    item.hItem = hRoot;
    ret = SendMessageA(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    ok(ret, "got %d\n", ret);

    if (item.iExpandedImage != 1)
    {
        win_skip("TVIF_EXPANDEDIMAGE not supported\n");
        DestroyWindow(hTree);
        return;
    }

    /* test for default iExpandedImage value */
    item.mask = TVIF_EXPANDEDIMAGE;
    item.iExpandedImage = -1;
    item.hItem = hChild;
    ret = SendMessageA(hTree, TVM_GETITEMA, 0, (LPARAM)&item);
    ok(ret, "got %d\n", ret);
    ok(item.iExpandedImage == (WORD)I_IMAGENONE, "got %d\n", item.iExpandedImage);

    DestroyWindow(hTree);
}

static void test_TVS_SINGLEEXPAND(void)
{
    HWND hTree;
    BOOL ret;

    hTree = create_treeview_control();
    SetWindowLongA(hTree, GWL_STYLE, GetWindowLong(hTree, GWL_STYLE) | TVS_SINGLEEXPAND);
    /* to avoid paiting related notifications */
    ShowWindow(hTree, SW_HIDE);
    fill_tree(hTree);

    flush_sequences(sequences, NUM_MSG_SEQUENCES);
    ret = SendMessageA(hTree, TVM_SELECTITEM, TVGN_CARET, (LPARAM)hRoot);
    ok(ret, "got %d\n", ret);
    ok_sequence(sequences, PARENT_SEQ_INDEX, parent_singleexpand_seq, "singleexpand notifications", FALSE);

    /* a workaround for NT4 that sends expanding notification when nothing is about to expand */
    ret = SendMessageA(hTree, TVM_DELETEITEM, 0, (LPARAM)hRoot);
    ok(ret, "got %d\n", ret);
    fill_tree(hTree);
    ret = SendMessageA(hTree, TVM_SELECTITEM, TVGN_CARET, 0);
    ok(ret, "got %d\n", ret);

    DestroyWindow(hTree);
}

static void test_WM_PAINT(void)
{
    HWND hTree;
    COLORREF clr;
    LONG ret;
    RECT rc;
    HDC hdc;

    hTree = create_treeview_control();

    clr = SendMessageA(hTree, TVM_SETBKCOLOR, 0, RGB(255, 0, 0));
    ok(clr == -1, "got %d, expected -1\n", clr);

    hdc = GetDC(hMainWnd);

    GetClientRect(hMainWnd, &rc);
    FillRect(hdc, &rc, GetStockObject(BLACK_BRUSH));

    clr = GetPixel(hdc, 1, 1);
    ok(clr == RGB(0, 0, 0), "got 0x%x\n", clr);

    ret = SendMessageA(hTree, WM_PAINT, (WPARAM)hdc, 0);
    ok(ret == 0, "got %d\n", ret);

    clr = GetPixel(hdc, 1, 1);
    ok(clr == RGB(255, 0, 0) || broken(clr == RGB(0, 0, 0)) /* win98 */,
        "got 0x%x\n", clr);

    ReleaseDC(hMainWnd, hdc);

    DestroyWindow(hTree);
}

static void test_delete_items(void)
{
    const struct message *msg;
    HWND hTree;
    INT ret;

    hTree = create_treeview_control();
    fill_tree(hTree);

    /* check delete order */
    flush_sequences(item_sequence, 1);
    ret = SendMessage(hTree, TVM_DELETEITEM, 0, 0);
    ok(ret == TRUE, "got %d\n", ret);

    msg = item_sequence[0]->sequence;
    ok(item_sequence[0]->count == 2, "expected 2 items, got %d\n", item_sequence[0]->count);

    if (item_sequence[0]->count == 2)
    {
      ok(msg[0].lParam == (LPARAM)hChild, "expected %p, got 0x%lx\n", hChild, msg[0].lParam);
      ok(msg[1].lParam == (LPARAM)hRoot, "expected %p, got 0x%lx\n", hRoot, msg[1].lParam);
    }

    ret = SendMessageA(hTree, TVM_GETCOUNT, 0, 0);
    ok(ret == 0, "got %d\n", ret);

    DestroyWindow(hTree);
}

START_TEST(treeview)
{
    HMODULE hComctl32;
    BOOL (WINAPI *pInitCommonControlsEx)(const INITCOMMONCONTROLSEX*);
    WNDCLASSA wc;
    MSG msg;

    ULONG_PTR ctx_cookie;
    HANDLE hCtx;
    HWND hwnd;
  
    hComctl32 = GetModuleHandleA("comctl32.dll");
    pInitCommonControlsEx = (void*)GetProcAddress(hComctl32, "InitCommonControlsEx");
    if (pInitCommonControlsEx)
    {
        INITCOMMONCONTROLSEX iccex;
        iccex.dwSize = sizeof(iccex);
        iccex.dwICC  = ICC_TREEVIEW_CLASSES;
        pInitCommonControlsEx(&iccex);
    }
    else
        InitCommonControls();

    init_msg_sequences(sequences, NUM_MSG_SEQUENCES);
    init_msg_sequences(item_sequence, 1);
  
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = GetModuleHandleA(NULL);
    wc.hIcon = NULL;
    wc.hCursor = LoadCursorA(NULL, IDC_IBEAM);
    wc.hbrBackground = GetSysColorBrush(COLOR_WINDOW);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = "MyTestWnd";
    wc.lpfnWndProc = parent_wnd_proc;
    RegisterClassA(&wc);

    hMainWnd = CreateWindowExA(0, "MyTestWnd", "Blah", WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, CW_USEDEFAULT, 130, 105, NULL, NULL, GetModuleHandleA(NULL), 0);

    ok(hMainWnd != NULL, "Failed to create parent window. Tests aborted.\n");
    if (!hMainWnd) return;

    test_fillroot();
    test_select();
    test_getitemtext();
    test_focus();
    test_get_set_bkcolor();
    test_get_set_imagelist();
    test_get_set_indent();
    test_get_set_insertmark();
    test_get_set_item();
    test_get_set_itemheight();
    test_get_set_scrolltime();
    test_get_set_textcolor();
    test_get_linecolor();
    test_get_insertmarkcolor();
    test_get_set_tooltips();
    test_get_set_unicodeformat();
    test_callback();
    test_expandinvisible();
    test_itemedit();
    test_treeview_classinfo();
    test_expandnotify();
    test_TVS_SINGLEEXPAND();
    test_WM_PAINT();
    test_delete_items();

    if (!load_v6_module(&ctx_cookie, &hCtx))
    {
        DestroyWindow(hMainWnd);
        return;
    }

    /* this is a XP SP3 failure workaround */
    hwnd = CreateWindowExA(0, WC_TREEVIEW, "foo",
                           WS_CHILD | WS_BORDER | WS_VISIBLE,
                           0, 0, 100, 100,
                           hMainWnd, NULL, GetModuleHandleA(NULL), NULL);
    if (!IsWindow(hwnd))
    {
        win_skip("FIXME: failed to create TreeView window.\n");
        unload_v6_module(ctx_cookie, hCtx);
        DestroyWindow(hMainWnd);
        return;
    }
    else
        DestroyWindow(hwnd);

    /* comctl32 version 6 tests start here */
    test_expandedimage();

    unload_v6_module(ctx_cookie, hCtx);

    PostMessageA(hMainWnd, WM_CLOSE, 0, 0);
    while(GetMessageA(&msg, 0, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessageA(&msg);
    }
}
