#!/bin/sh
#
# Script to automatically install all Wine Staging patches
#
# Copyright (C) 2015 Sebastian Lackner
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#

# Show usage information
usage()
{
	echo ""
	echo "Usage: ./patchinstall.sh [DESTDIR=path] [--all] [-W patchset] [patchset ...]"
	echo ""
	echo "Autogenerated script to apply all Wine Staging patches on your Wine"
	echo "source tree. This script replaces and enhances the old method of"
	echo "using a Makefile."
	echo ""
	echo "Configuration:"
	echo "  DESTDIR=path         Specify the path to the wine source tree"
	echo "  --all                Select all patches"
	echo "  --force-autoconf     Run autoreconf and tools/make_requests after each patch"
	echo "  --help               Display this help and exit"
	echo "  --no-patchlist       Do not apply patchlist (needed for 'wine --patches')"
	echo "  --no-autoconf        Do not run autoreconf and tools/make_requests"
	echo "  --version            Show version information"
	echo "  -W patchset          Exclude a specific patchset"
	echo ""
	echo "Backends:"
	echo "  --backend=patch      Use regular 'patch' utility to apply patches (default)"
	echo "  --backend=epatch     Use 'epatch' to apply patches (Gentoo only)"
	echo "  --backend=git-am     Use 'git am' to apply patches"
	echo "  --backend=git-apply  Use 'git apply' to apply patches"
	echo "  --backend=stg        Import the patches using stacked git"
	echo ""
}

# Show version information
version()
{
	echo "Wine Staging 1.7.47"
	echo "Copyright (C) 2014-2015 the Wine Staging project authors."
	echo ""
	echo "Patchset to be applied on upstream Wine:"
	echo "  commit 744f7b69bf4692265588e10650090ca2f5129ccd"
	echo ""
}

# Critical error, abort
abort()
{
	echo "ERROR: $1" >&2
	exit 1
}

# Show a warning
warning()
{
	echo "WARNING: $1" >&2
}

# Enable or disable all patchsets
patch_enable_all ()
{
	enable_Compiler_Warnings="$1"
	enable_Coverity="$1"
	enable_Exagear="$1"
	enable_Pipelight="$1"
	enable_Staging="$1"
	enable_advapi32_LsaLookupSids="$1"
	enable_browseui_Progress_Dialog="$1"
	enable_combase_String="$1"
	enable_comctl32_LoadIconMetric="$1"
	enable_configure_Absolute_RPATH="$1"
	enable_configure_Crosscompiling="$1"
	enable_crypt32_CMS_Certificates="$1"
	enable_d3d9_DesktopWindow="$1"
	enable_d3d9_Skip_Tests="$1"
	enable_d3d9_Surface_Refcount="$1"
	enable_d3dx9_24_ID3DXEffect="$1"
	enable_d3dx9_25_ID3DXEffect="$1"
	enable_d3dx9_26_ID3DXEffect="$1"
	enable_d3dx9_36_AnimationController="$1"
	enable_d3dx9_36_CloneEffect="$1"
	enable_d3dx9_36_D3DXStubs="$1"
	enable_d3dx9_36_DDS="$1"
	enable_d3dx9_36_DXTn="$1"
	enable_d3dx9_36_DrawText="$1"
	enable_d3dx9_36_Filter_Warnings="$1"
	enable_d3dx9_36_FindNextValidTechnique="$1"
	enable_d3dx9_36_GetShaderSemantics="$1"
	enable_d3dx9_36_Optimize_Inplace="$1"
	enable_d3dx9_36_Texture_Align="$1"
	enable_d3dx9_36_UpdateSkinnedMesh="$1"
	enable_dbghelp_Debug_Symbols="$1"
	enable_ddraw_EnumSurfaces="$1"
	enable_ddraw_IDirect3DTexture2_Load="$1"
	enable_ddraw_d3d_execute_buffer="$1"
	enable_dinput_Events="$1"
	enable_dsound_EAX="$1"
	enable_dsound_Fast_Mixer="$1"
	enable_dwrite_VDMX="$1"
	enable_dxgi_GetDesc="$1"
	enable_dxva2_Video_Decoder="$1"
	enable_fltmgr_Stub_SYS="$1"
	enable_fonts_Missing_Fonts="$1"
	enable_gdi32_Default_Palette="$1"
	enable_gdi32_MaxPixelFormats="$1"
	enable_gdi32_MultiMonitor="$1"
	enable_gdiplus_GIF_Encoder="$1"
	enable_gdiplus_GdipCreateEffect="$1"
	enable_imagehlp_BindImageEx="$1"
	enable_imagehlp_ImageLoad="$1"
	enable_inetcpl_Default_Home="$1"
	enable_iphlpapi_System_Ping="$1"
	enable_iphlpapi_TCP_Table="$1"
	enable_kernel32_CompareStringEx="$1"
	enable_kernel32_CopyFileEx="$1"
	enable_kernel32_GetFinalPathNameByHandle="$1"
	enable_kernel32_GetLogicalProcessorInformationEx="$1"
	enable_kernel32_InsertMode="$1"
	enable_kernel32_Named_Pipe="$1"
	enable_kernel32_NeedCurrentDirectoryForExePath="$1"
	enable_kernel32_Profile="$1"
	enable_kernel32_SetFileInformationByHandle="$1"
	enable_kernel32_TimezoneInformation_Registry="$1"
	enable_kernel32_VerifyVersionInfo="$1"
	enable_libs_Debug_Channel="$1"
	enable_libs_Unicode_Collation="$1"
	enable_makedep_PARENTSPEC="$1"
	enable_mfplat_MFTRegister="$1"
	enable_mmdevapi_AEV_Stubs="$1"
	enable_mountmgr_DosDevices="$1"
	enable_mscoree_CorValidateImage="$1"
	enable_msvcp90_basic_string_dtor="$1"
	enable_msvcrt_Math_Precision="$1"
	enable_msvfw32_Image_Size="$1"
	enable_ntdll_APC_Performance="$1"
	enable_ntdll_APC_Start_Process="$1"
	enable_ntdll_Activation_Context="$1"
	enable_ntdll_CLI_Images="$1"
	enable_ntdll_Coverity="$1"
	enable_ntdll_DOS_Attributes="$1"
	enable_ntdll_DeviceType_Systemroot="$1"
	enable_ntdll_DllRedirects="$1"
	enable_ntdll_Exception="$1"
	enable_ntdll_FileDispositionInformation="$1"
	enable_ntdll_FileFsFullSizeInformation="$1"
	enable_ntdll_FileFsVolumeInformation="$1"
	enable_ntdll_Fix_Alignment="$1"
	enable_ntdll_Fix_Free="$1"
	enable_ntdll_FreeBSD_Directory="$1"
	enable_ntdll_Heap_FreeLists="$1"
	enable_ntdll_Hide_Wine_Exports="$1"
	enable_ntdll_Junction_Points="$1"
	enable_ntdll_NtQueryEaFile="$1"
	enable_ntdll_NtQuerySection="$1"
	enable_ntdll_NtSetLdtEntries="$1"
	enable_ntdll_Pipe_SpecialCharacters="$1"
	enable_ntdll_Revert_Security_Cookie="$1"
	enable_ntdll_RtlIpStringToAddress="$1"
	enable_ntdll_ThreadTime="$1"
	enable_ntdll_Threading="$1"
	enable_ntdll_User_Shared_Data="$1"
	enable_ntdll_WRITECOPY="$1"
	enable_ntdll_WinSqm="$1"
	enable_ntdll_WriteWatches="$1"
	enable_ntoskrnl_DriverTest="$1"
	enable_ntoskrnl_Stubs="$1"
	enable_nvapi_Stub_DLL="$1"
	enable_nvcuda_CUDA_Support="$1"
	enable_nvcuvid_CUDA_Video_Support="$1"
	enable_nvencodeapi_Video_Encoder="$1"
	enable_opengl32_Revert_Disable_Ext="$1"
	enable_quartz_MediaSeeking_Positions="$1"
	enable_rasapi32_RasEnumDevicesA="$1"
	enable_regedit_Reg_Parser="$1"
	enable_riched20_IText_Interface="$1"
	enable_rpcrt4_Pipe_Transport="$1"
	enable_rpcrt4_Use_After_Free="$1"
	enable_secur32_ANSI_NTLM_Credentials="$1"
	enable_server_ACL_Compat="$1"
	enable_server_Address_List_Change="$1"
	enable_server_ClipCursor="$1"
	enable_server_CreateProcess_ACLs="$1"
	enable_server_Delete_On_Close="$1"
	enable_server_File_Permissions="$1"
	enable_server_Inherited_ACLs="$1"
	enable_server_JobObjects="$1"
	enable_server_Key_State="$1"
	enable_server_Misc_ACL="$1"
	enable_server_OpenProcess="$1"
	enable_server_PeekMessage="$1"
	enable_server_Realtime_Priority="$1"
	enable_server_RootDirectory_File="$1"
	enable_server_Shared_Memory="$1"
	enable_server_Stored_ACLs="$1"
	enable_server_Unexpected_Wakeup="$1"
	enable_setupapi_SetupDiSelectBestCompatDrv="$1"
	enable_setupapi_SetupDiSetDeviceInstallParamsW="$1"
	enable_setupapi_SetupPromptForDisk="$1"
	enable_shdocvw_ParseURLFromOutsideSource_Tests="$1"
	enable_shell32_Default_Path="$1"
	enable_shell32_File_Property_Dialog="$1"
	enable_shell32_Icons="$1"
	enable_shell32_Placeholder_Icons="$1"
	enable_shell32_Progress_Dialog="$1"
	enable_shell32_Quoted_ShellExecute="$1"
	enable_shell32_RunDLL_CallEntry16="$1"
	enable_shell32_Run_Dialog="$1"
	enable_shell32_SHCreateSessionKey="$1"
	enable_shell32_SHFileOperation="$1"
	enable_shell32_UnixFS="$1"
	enable_shlwapi_AssocGetPerceivedType="$1"
	enable_shlwapi_UrlCombine="$1"
	enable_urlmon_CoInternetSetFeatureEnabled="$1"
	enable_user32_Dialog_Paint_Event="$1"
	enable_user32_DrawTextExW="$1"
	enable_user32_GetSystemMetrics="$1"
	enable_user32_Invalidate_Key_State="$1"
	enable_user32_Mouse_Message_Hwnd="$1"
	enable_user32_Painting="$1"
	enable_user32_ScrollWindowEx="$1"
	enable_user32_WndProc="$1"
	enable_vcomp_Functions="$1"
	enable_version_VerQueryValue="$1"
	enable_wbemdisp_ISWbemSecurity="$1"
	enable_wbemdisp_Timeout="$1"
	enable_wiaservc_IEnumWIA_DEV_INFO="$1"
	enable_widl_Buffer_Overflow="$1"
	enable_windowscodecs_GIF_Decoder="$1"
	enable_windowscodecs_TIFF_Decoder="$1"
	enable_wine_inf_Performance="$1"
	enable_wine_inf_ProfileList_UserSID="$1"
	enable_wineboot_HKEY_DYN_DATA="$1"
	enable_wineboot_MachineGuid="$1"
	enable_wineboot_drivers_etc_Stubs="$1"
	enable_winecfg_Libraries="$1"
	enable_winecfg_Staging="$1"
	enable_winecfg_Unmounted_Devices="$1"
	enable_wineconsole_Insert_Mode="$1"
	enable_wined3d_Accounting="$1"
	enable_wined3d_CSMT_Helper="$1"
	enable_wined3d_CSMT_Main="$1"
	enable_wined3d_DXTn="$1"
	enable_wined3d_MESA_GPU_Info="$1"
	enable_wined3d_Multisampling="$1"
	enable_wined3d_Revert_PixelFormat="$1"
	enable_wined3d_UnhandledBlendFactor="$1"
	enable_wined3d_resource_check_usage="$1"
	enable_wined3d_wined3d_swapchain_present="$1"
	enable_winedevice_Fix_Relocation="$1"
	enable_winemenubuilder_Desktop_Icon_Path="$1"
	enable_winepulse_PulseAudio_Support="$1"
	enable_winex11_CandidateWindowPos="$1"
	enable_winex11_Clipboard_HTML="$1"
	enable_winex11_DragAndDrop="$1"
	enable_winex11_Thread_Data="$1"
	enable_winex11_Window_Groups="$1"
	enable_winex11_Window_Style="$1"
	enable_winex11_XEMBED="$1"
	enable_winex11_wglShareLists="$1"
	enable_winhttp_System_Proxy_Autoconfig="$1"
	enable_wininet_Cleanup="$1"
	enable_wininet_ParseX509EncodedCertificateForListBoxEntry="$1"
	enable_winmm_Delay_Import_Depends="$1"
	enable_winscard_SCardListReaders="$1"
	enable_winsta_WinStationEnumerateW="$1"
	enable_wpcap_Dynamic_Linking="$1"
	enable_ws2_32_APC_Performance="$1"
	enable_ws2_32_Connect_Time="$1"
	enable_ws2_32_TransmitFile="$1"
	enable_ws2_32_WSAPoll="$1"
	enable_ws2_32_WriteWatches="$1"
	enable_ws2_32_getaddrinfo="$1"
	enable_wtsapi32_EnumerateProcesses="$1"
}

# Enable or disable all categories
category_enable_all ()
{
	enable_category_stable="$1"
}

# Enable or disable a specific patchset/category
patch_enable ()
{
	case "$1" in
		Compiler_Warnings)
			enable_Compiler_Warnings="$2"
			;;
		Coverity)
			enable_Coverity="$2"
			;;
		Exagear)
			enable_Exagear="$2"
			;;
		Pipelight)
			enable_Pipelight="$2"
			;;
		Staging)
			enable_Staging="$2"
			;;
		advapi32-LsaLookupSids)
			enable_advapi32_LsaLookupSids="$2"
			;;
		browseui-Progress_Dialog)
			enable_browseui_Progress_Dialog="$2"
			;;
		category-stable)
			enable_category_stable="$2"
			;;
		combase-String)
			enable_combase_String="$2"
			;;
		comctl32-LoadIconMetric)
			enable_comctl32_LoadIconMetric="$2"
			;;
		configure-Absolute_RPATH)
			enable_configure_Absolute_RPATH="$2"
			;;
		configure-Crosscompiling)
			enable_configure_Crosscompiling="$2"
			;;
		crypt32-CMS_Certificates)
			enable_crypt32_CMS_Certificates="$2"
			;;
		d3d9-DesktopWindow)
			enable_d3d9_DesktopWindow="$2"
			;;
		d3d9-Skip_Tests)
			enable_d3d9_Skip_Tests="$2"
			;;
		d3d9-Surface_Refcount)
			enable_d3d9_Surface_Refcount="$2"
			;;
		d3dx9_24-ID3DXEffect)
			enable_d3dx9_24_ID3DXEffect="$2"
			;;
		d3dx9_25-ID3DXEffect)
			enable_d3dx9_25_ID3DXEffect="$2"
			;;
		d3dx9_26-ID3DXEffect)
			enable_d3dx9_26_ID3DXEffect="$2"
			;;
		d3dx9_36-AnimationController)
			enable_d3dx9_36_AnimationController="$2"
			;;
		d3dx9_36-CloneEffect)
			enable_d3dx9_36_CloneEffect="$2"
			;;
		d3dx9_36-D3DXStubs)
			enable_d3dx9_36_D3DXStubs="$2"
			;;
		d3dx9_36-DDS)
			enable_d3dx9_36_DDS="$2"
			;;
		d3dx9_36-DXTn)
			enable_d3dx9_36_DXTn="$2"
			;;
		d3dx9_36-DrawText)
			enable_d3dx9_36_DrawText="$2"
			;;
		d3dx9_36-Filter_Warnings)
			enable_d3dx9_36_Filter_Warnings="$2"
			;;
		d3dx9_36-FindNextValidTechnique)
			enable_d3dx9_36_FindNextValidTechnique="$2"
			;;
		d3dx9_36-GetShaderSemantics)
			enable_d3dx9_36_GetShaderSemantics="$2"
			;;
		d3dx9_36-Optimize_Inplace)
			enable_d3dx9_36_Optimize_Inplace="$2"
			;;
		d3dx9_36-Texture_Align)
			enable_d3dx9_36_Texture_Align="$2"
			;;
		d3dx9_36-UpdateSkinnedMesh)
			enable_d3dx9_36_UpdateSkinnedMesh="$2"
			;;
		dbghelp-Debug_Symbols)
			enable_dbghelp_Debug_Symbols="$2"
			;;
		ddraw-EnumSurfaces)
			enable_ddraw_EnumSurfaces="$2"
			;;
		ddraw-IDirect3DTexture2_Load)
			enable_ddraw_IDirect3DTexture2_Load="$2"
			;;
		ddraw-d3d_execute_buffer)
			enable_ddraw_d3d_execute_buffer="$2"
			;;
		dinput-Events)
			enable_dinput_Events="$2"
			;;
		dsound-EAX)
			enable_dsound_EAX="$2"
			;;
		dsound-Fast_Mixer)
			enable_dsound_Fast_Mixer="$2"
			;;
		dwrite-VDMX)
			enable_dwrite_VDMX="$2"
			;;
		dxgi-GetDesc)
			enable_dxgi_GetDesc="$2"
			;;
		dxva2-Video_Decoder)
			enable_dxva2_Video_Decoder="$2"
			;;
		fltmgr-Stub_SYS)
			enable_fltmgr_Stub_SYS="$2"
			;;
		fonts-Missing_Fonts)
			enable_fonts_Missing_Fonts="$2"
			;;
		gdi32-Default_Palette)
			enable_gdi32_Default_Palette="$2"
			;;
		gdi32-MaxPixelFormats)
			enable_gdi32_MaxPixelFormats="$2"
			;;
		gdi32-MultiMonitor)
			enable_gdi32_MultiMonitor="$2"
			;;
		gdiplus-GIF_Encoder)
			enable_gdiplus_GIF_Encoder="$2"
			;;
		gdiplus-GdipCreateEffect)
			enable_gdiplus_GdipCreateEffect="$2"
			;;
		imagehlp-BindImageEx)
			enable_imagehlp_BindImageEx="$2"
			;;
		imagehlp-ImageLoad)
			enable_imagehlp_ImageLoad="$2"
			;;
		inetcpl-Default_Home)
			enable_inetcpl_Default_Home="$2"
			;;
		iphlpapi-System_Ping)
			enable_iphlpapi_System_Ping="$2"
			;;
		iphlpapi-TCP_Table)
			enable_iphlpapi_TCP_Table="$2"
			;;
		kernel32-CompareStringEx)
			enable_kernel32_CompareStringEx="$2"
			;;
		kernel32-CopyFileEx)
			enable_kernel32_CopyFileEx="$2"
			;;
		kernel32-GetFinalPathNameByHandle)
			enable_kernel32_GetFinalPathNameByHandle="$2"
			;;
		kernel32-GetLogicalProcessorInformationEx)
			enable_kernel32_GetLogicalProcessorInformationEx="$2"
			;;
		kernel32-InsertMode)
			enable_kernel32_InsertMode="$2"
			;;
		kernel32-Named_Pipe)
			enable_kernel32_Named_Pipe="$2"
			;;
		kernel32-NeedCurrentDirectoryForExePath)
			enable_kernel32_NeedCurrentDirectoryForExePath="$2"
			;;
		kernel32-Profile)
			enable_kernel32_Profile="$2"
			;;
		kernel32-SetFileInformationByHandle)
			enable_kernel32_SetFileInformationByHandle="$2"
			;;
		kernel32-TimezoneInformation_Registry)
			enable_kernel32_TimezoneInformation_Registry="$2"
			;;
		kernel32-VerifyVersionInfo)
			enable_kernel32_VerifyVersionInfo="$2"
			;;
		libs-Debug_Channel)
			enable_libs_Debug_Channel="$2"
			;;
		libs-Unicode_Collation)
			enable_libs_Unicode_Collation="$2"
			;;
		makedep-PARENTSPEC)
			enable_makedep_PARENTSPEC="$2"
			;;
		mfplat-MFTRegister)
			enable_mfplat_MFTRegister="$2"
			;;
		mmdevapi-AEV_Stubs)
			enable_mmdevapi_AEV_Stubs="$2"
			;;
		mountmgr-DosDevices)
			enable_mountmgr_DosDevices="$2"
			;;
		mscoree-CorValidateImage)
			enable_mscoree_CorValidateImage="$2"
			;;
		msvcp90-basic_string_dtor)
			enable_msvcp90_basic_string_dtor="$2"
			;;
		msvcrt-Math_Precision)
			enable_msvcrt_Math_Precision="$2"
			;;
		msvfw32-Image_Size)
			enable_msvfw32_Image_Size="$2"
			;;
		ntdll-APC_Performance)
			enable_ntdll_APC_Performance="$2"
			;;
		ntdll-APC_Start_Process)
			enable_ntdll_APC_Start_Process="$2"
			;;
		ntdll-Activation_Context)
			enable_ntdll_Activation_Context="$2"
			;;
		ntdll-CLI_Images)
			enable_ntdll_CLI_Images="$2"
			;;
		ntdll-Coverity)
			enable_ntdll_Coverity="$2"
			;;
		ntdll-DOS_Attributes)
			enable_ntdll_DOS_Attributes="$2"
			;;
		ntdll-DeviceType_Systemroot)
			enable_ntdll_DeviceType_Systemroot="$2"
			;;
		ntdll-DllRedirects)
			enable_ntdll_DllRedirects="$2"
			;;
		ntdll-Exception)
			enable_ntdll_Exception="$2"
			;;
		ntdll-FileDispositionInformation)
			enable_ntdll_FileDispositionInformation="$2"
			;;
		ntdll-FileFsFullSizeInformation)
			enable_ntdll_FileFsFullSizeInformation="$2"
			;;
		ntdll-FileFsVolumeInformation)
			enable_ntdll_FileFsVolumeInformation="$2"
			;;
		ntdll-Fix_Alignment)
			enable_ntdll_Fix_Alignment="$2"
			;;
		ntdll-Fix_Free)
			enable_ntdll_Fix_Free="$2"
			;;
		ntdll-FreeBSD_Directory)
			enable_ntdll_FreeBSD_Directory="$2"
			;;
		ntdll-Heap_FreeLists)
			enable_ntdll_Heap_FreeLists="$2"
			;;
		ntdll-Hide_Wine_Exports)
			enable_ntdll_Hide_Wine_Exports="$2"
			;;
		ntdll-Junction_Points)
			enable_ntdll_Junction_Points="$2"
			;;
		ntdll-NtQueryEaFile)
			enable_ntdll_NtQueryEaFile="$2"
			;;
		ntdll-NtQuerySection)
			enable_ntdll_NtQuerySection="$2"
			;;
		ntdll-NtSetLdtEntries)
			enable_ntdll_NtSetLdtEntries="$2"
			;;
		ntdll-Pipe_SpecialCharacters)
			enable_ntdll_Pipe_SpecialCharacters="$2"
			;;
		ntdll-Revert_Security_Cookie)
			enable_ntdll_Revert_Security_Cookie="$2"
			;;
		ntdll-RtlIpStringToAddress)
			enable_ntdll_RtlIpStringToAddress="$2"
			;;
		ntdll-ThreadTime)
			enable_ntdll_ThreadTime="$2"
			;;
		ntdll-Threading)
			enable_ntdll_Threading="$2"
			;;
		ntdll-User_Shared_Data)
			enable_ntdll_User_Shared_Data="$2"
			;;
		ntdll-WRITECOPY)
			enable_ntdll_WRITECOPY="$2"
			;;
		ntdll-WinSqm)
			enable_ntdll_WinSqm="$2"
			;;
		ntdll-WriteWatches)
			enable_ntdll_WriteWatches="$2"
			;;
		ntoskrnl-DriverTest)
			enable_ntoskrnl_DriverTest="$2"
			;;
		ntoskrnl-Stubs)
			enable_ntoskrnl_Stubs="$2"
			;;
		nvapi-Stub_DLL)
			enable_nvapi_Stub_DLL="$2"
			;;
		nvcuda-CUDA_Support)
			enable_nvcuda_CUDA_Support="$2"
			;;
		nvcuvid-CUDA_Video_Support)
			enable_nvcuvid_CUDA_Video_Support="$2"
			;;
		nvencodeapi-Video_Encoder)
			enable_nvencodeapi_Video_Encoder="$2"
			;;
		opengl32-Revert_Disable_Ext)
			enable_opengl32_Revert_Disable_Ext="$2"
			;;
		quartz-MediaSeeking_Positions)
			enable_quartz_MediaSeeking_Positions="$2"
			;;
		rasapi32-RasEnumDevicesA)
			enable_rasapi32_RasEnumDevicesA="$2"
			;;
		regedit-Reg_Parser)
			enable_regedit_Reg_Parser="$2"
			;;
		riched20-IText_Interface)
			enable_riched20_IText_Interface="$2"
			;;
		rpcrt4-Pipe_Transport)
			enable_rpcrt4_Pipe_Transport="$2"
			;;
		rpcrt4-Use_After_Free)
			enable_rpcrt4_Use_After_Free="$2"
			;;
		secur32-ANSI_NTLM_Credentials)
			enable_secur32_ANSI_NTLM_Credentials="$2"
			;;
		server-ACL_Compat)
			enable_server_ACL_Compat="$2"
			;;
		server-Address_List_Change)
			enable_server_Address_List_Change="$2"
			;;
		server-ClipCursor)
			enable_server_ClipCursor="$2"
			;;
		server-CreateProcess_ACLs)
			enable_server_CreateProcess_ACLs="$2"
			;;
		server-Delete_On_Close)
			enable_server_Delete_On_Close="$2"
			;;
		server-File_Permissions)
			enable_server_File_Permissions="$2"
			;;
		server-Inherited_ACLs)
			enable_server_Inherited_ACLs="$2"
			;;
		server-JobObjects)
			enable_server_JobObjects="$2"
			;;
		server-Key_State)
			enable_server_Key_State="$2"
			;;
		server-Misc_ACL)
			enable_server_Misc_ACL="$2"
			;;
		server-OpenProcess)
			enable_server_OpenProcess="$2"
			;;
		server-PeekMessage)
			enable_server_PeekMessage="$2"
			;;
		server-Realtime_Priority)
			enable_server_Realtime_Priority="$2"
			;;
		server-RootDirectory_File)
			enable_server_RootDirectory_File="$2"
			;;
		server-Shared_Memory)
			enable_server_Shared_Memory="$2"
			;;
		server-Stored_ACLs)
			enable_server_Stored_ACLs="$2"
			;;
		server-Unexpected_Wakeup)
			enable_server_Unexpected_Wakeup="$2"
			;;
		setupapi-SetupDiSelectBestCompatDrv)
			enable_setupapi_SetupDiSelectBestCompatDrv="$2"
			;;
		setupapi-SetupDiSetDeviceInstallParamsW)
			enable_setupapi_SetupDiSetDeviceInstallParamsW="$2"
			;;
		setupapi-SetupPromptForDisk)
			enable_setupapi_SetupPromptForDisk="$2"
			;;
		shdocvw-ParseURLFromOutsideSource_Tests)
			enable_shdocvw_ParseURLFromOutsideSource_Tests="$2"
			;;
		shell32-Default_Path)
			enable_shell32_Default_Path="$2"
			;;
		shell32-File_Property_Dialog)
			enable_shell32_File_Property_Dialog="$2"
			;;
		shell32-Icons)
			enable_shell32_Icons="$2"
			;;
		shell32-Placeholder_Icons)
			enable_shell32_Placeholder_Icons="$2"
			;;
		shell32-Progress_Dialog)
			enable_shell32_Progress_Dialog="$2"
			;;
		shell32-Quoted_ShellExecute)
			enable_shell32_Quoted_ShellExecute="$2"
			;;
		shell32-RunDLL_CallEntry16)
			enable_shell32_RunDLL_CallEntry16="$2"
			;;
		shell32-Run_Dialog)
			enable_shell32_Run_Dialog="$2"
			;;
		shell32-SHCreateSessionKey)
			enable_shell32_SHCreateSessionKey="$2"
			;;
		shell32-SHFileOperation)
			enable_shell32_SHFileOperation="$2"
			;;
		shell32-UnixFS)
			enable_shell32_UnixFS="$2"
			;;
		shlwapi-AssocGetPerceivedType)
			enable_shlwapi_AssocGetPerceivedType="$2"
			;;
		shlwapi-UrlCombine)
			enable_shlwapi_UrlCombine="$2"
			;;
		urlmon-CoInternetSetFeatureEnabled)
			enable_urlmon_CoInternetSetFeatureEnabled="$2"
			;;
		user32-Dialog_Paint_Event)
			enable_user32_Dialog_Paint_Event="$2"
			;;
		user32-DrawTextExW)
			enable_user32_DrawTextExW="$2"
			;;
		user32-GetSystemMetrics)
			enable_user32_GetSystemMetrics="$2"
			;;
		user32-Invalidate_Key_State)
			enable_user32_Invalidate_Key_State="$2"
			;;
		user32-Mouse_Message_Hwnd)
			enable_user32_Mouse_Message_Hwnd="$2"
			;;
		user32-Painting)
			enable_user32_Painting="$2"
			;;
		user32-ScrollWindowEx)
			enable_user32_ScrollWindowEx="$2"
			;;
		user32-WndProc)
			enable_user32_WndProc="$2"
			;;
		vcomp-Functions)
			enable_vcomp_Functions="$2"
			;;
		version-VerQueryValue)
			enable_version_VerQueryValue="$2"
			;;
		wbemdisp-ISWbemSecurity)
			enable_wbemdisp_ISWbemSecurity="$2"
			;;
		wbemdisp-Timeout)
			enable_wbemdisp_Timeout="$2"
			;;
		wiaservc-IEnumWIA_DEV_INFO)
			enable_wiaservc_IEnumWIA_DEV_INFO="$2"
			;;
		widl-Buffer_Overflow)
			enable_widl_Buffer_Overflow="$2"
			;;
		windowscodecs-GIF_Decoder)
			enable_windowscodecs_GIF_Decoder="$2"
			;;
		windowscodecs-TIFF_Decoder)
			enable_windowscodecs_TIFF_Decoder="$2"
			;;
		wine.inf-Performance)
			enable_wine_inf_Performance="$2"
			;;
		wine.inf-ProfileList_UserSID)
			enable_wine_inf_ProfileList_UserSID="$2"
			;;
		wineboot-HKEY_DYN_DATA)
			enable_wineboot_HKEY_DYN_DATA="$2"
			;;
		wineboot-MachineGuid)
			enable_wineboot_MachineGuid="$2"
			;;
		wineboot-drivers_etc_Stubs)
			enable_wineboot_drivers_etc_Stubs="$2"
			;;
		winecfg-Libraries)
			enable_winecfg_Libraries="$2"
			;;
		winecfg-Staging)
			enable_winecfg_Staging="$2"
			;;
		winecfg-Unmounted_Devices)
			enable_winecfg_Unmounted_Devices="$2"
			;;
		wineconsole-Insert_Mode)
			enable_wineconsole_Insert_Mode="$2"
			;;
		wined3d-Accounting)
			enable_wined3d_Accounting="$2"
			;;
		wined3d-CSMT_Helper)
			enable_wined3d_CSMT_Helper="$2"
			;;
		wined3d-CSMT_Main)
			enable_wined3d_CSMT_Main="$2"
			;;
		wined3d-DXTn)
			enable_wined3d_DXTn="$2"
			;;
		wined3d-MESA_GPU_Info)
			enable_wined3d_MESA_GPU_Info="$2"
			;;
		wined3d-Multisampling)
			enable_wined3d_Multisampling="$2"
			;;
		wined3d-Revert_PixelFormat)
			enable_wined3d_Revert_PixelFormat="$2"
			;;
		wined3d-UnhandledBlendFactor)
			enable_wined3d_UnhandledBlendFactor="$2"
			;;
		wined3d-resource_check_usage)
			enable_wined3d_resource_check_usage="$2"
			;;
		wined3d-wined3d_swapchain_present)
			enable_wined3d_wined3d_swapchain_present="$2"
			;;
		winedevice-Fix_Relocation)
			enable_winedevice_Fix_Relocation="$2"
			;;
		winemenubuilder-Desktop_Icon_Path)
			enable_winemenubuilder_Desktop_Icon_Path="$2"
			;;
		winepulse-PulseAudio_Support)
			enable_winepulse_PulseAudio_Support="$2"
			;;
		winex11-CandidateWindowPos)
			enable_winex11_CandidateWindowPos="$2"
			;;
		winex11-Clipboard_HTML)
			enable_winex11_Clipboard_HTML="$2"
			;;
		winex11-DragAndDrop)
			enable_winex11_DragAndDrop="$2"
			;;
		winex11-Thread_Data)
			enable_winex11_Thread_Data="$2"
			;;
		winex11-Window_Groups)
			enable_winex11_Window_Groups="$2"
			;;
		winex11-Window_Style)
			enable_winex11_Window_Style="$2"
			;;
		winex11-XEMBED)
			enable_winex11_XEMBED="$2"
			;;
		winex11-wglShareLists)
			enable_winex11_wglShareLists="$2"
			;;
		winhttp-System_Proxy_Autoconfig)
			enable_winhttp_System_Proxy_Autoconfig="$2"
			;;
		wininet-Cleanup)
			enable_wininet_Cleanup="$2"
			;;
		wininet-ParseX509EncodedCertificateForListBoxEntry)
			enable_wininet_ParseX509EncodedCertificateForListBoxEntry="$2"
			;;
		winmm-Delay_Import_Depends)
			enable_winmm_Delay_Import_Depends="$2"
			;;
		winscard-SCardListReaders)
			enable_winscard_SCardListReaders="$2"
			;;
		winsta-WinStationEnumerateW)
			enable_winsta_WinStationEnumerateW="$2"
			;;
		wpcap-Dynamic_Linking)
			enable_wpcap_Dynamic_Linking="$2"
			;;
		ws2_32-APC_Performance)
			enable_ws2_32_APC_Performance="$2"
			;;
		ws2_32-Connect_Time)
			enable_ws2_32_Connect_Time="$2"
			;;
		ws2_32-TransmitFile)
			enable_ws2_32_TransmitFile="$2"
			;;
		ws2_32-WSAPoll)
			enable_ws2_32_WSAPoll="$2"
			;;
		ws2_32-WriteWatches)
			enable_ws2_32_WriteWatches="$2"
			;;
		ws2_32-getaddrinfo)
			enable_ws2_32_getaddrinfo="$2"
			;;
		wtsapi32-EnumerateProcesses)
			enable_wtsapi32_EnumerateProcesses="$2"
			;;
		*)
			return 1
			;;
	esac
	return 0
}

# Default settings
patch_enable_all 0
category_enable_all 0
enable_patchlist=1
enable_autoconf=1
patchlist="/dev/null"
backend="patch"

# Find location of patches
patchdir="$(dirname "$(readlink -f "$0")")"
if test ! -f "$patchdir/patchinstall.sh"; then
	if test -f ./patchinstall.sh; then
		patchdir="$(pwd)"
	else
		abort "Failed to find patch directory."
	fi
fi

# Parse commandline arguments
if test "$#" -eq 0; then
	abort "No commandline arguments given, don't know what to do."
fi

while test "$#" -gt 0; do
	case "$1" in
		DESTDIR=*)
			DESTDIR="${1#*=}"
			shift
			;;

		--all)
			patch_enable_all 1
			shift
			;;

		--backend=*)
			backend="${1#*=}"
			shift
			;;

		--force-autoconf)
			enable_autoconf=2
			shift
			;;

		--help)
			usage
			exit 0
			;;

		--no-patchlist)
			enable_patchlist=0
			shift
			;;

		--no-autoconf)
			enable_autoconf=0
			shift
			;;

		--version)
			version
			exit 0
			;;

		-W)
			# Disable patchset
			if ! patch_enable "$2" 2; then
				abort "Wrong usage of -W commandline argument, expected patchname."
			fi
			shift
			shift
			;;

		*)
			# Enable patchset
			if ! patch_enable "$1" 1; then
				abort "Unknown commandline argument $1."
			fi
			shift
			;;
	esac
done

# Determine DESTDIR if not explicitly specified
if test -z "$DESTDIR" -a -f ./tools/make_requests; then
	DESTDIR="$(pwd)"

elif test ! -f "$DESTDIR/tools/make_requests"; then
	abort "DESTDIR does not point to the Wine source tree."
fi

# Change directory to DESTDIR, epatch depends on that
if ! cd "$DESTDIR"; then
	abort "Unable to change directory to $DESTDIR."
fi

# Helper to update configure / the wineserver protocol if required
if ! command -v diff >/dev/null 2>&1 ||
   ! command -v grep >/dev/null 2>&1 ||
   ! command -v cmp >/dev/null 2>&1; then

	update_configure()
	{
		autoreconf -f
	}

	update_protocol()
	{
		./tools/make_requests
	}

else

	update_configure()
	{
		_file="./configure"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! autoreconf -f; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Shifting by 62 bits is undefined behaviour when off_t is 32-bit, see also
		# https://launchpad.net/ubuntu/+source/autoconf/2.69-6 - the bug is still
		# present in some other distros (including Archlinux).
		_large_off_old="^#define LARGE_OFF_T (((off_t) 1 << 62) - 1 + ((off_t) 1 << 62))$"
		_large_off_new="#define LARGE_OFF_T ((((off_t) 1 << 31) << 31) - 1 + (((off_t) 1 << 31) << 31))"
		sed -i'' -e "s|$_large_off_old|$_large_off_new|g" "$_file"
		unset _large_off_old _large_off_new

		# Restore original timestamp when nothing changed
		if ! cmp "$_file.old" "$_file" >/dev/null; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}

	update_protocol()
	{
		_file="./include/wine/server_protocol.h"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! ./tools/make_requests; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Restore original timestamp when nothing changed
		if diff -u "$_file.old" "$_file" |
		   grep -v "^[+-]#define SERVER_PROTOCOL_VERSION" |
		   grep -v "^\(+++\|---\)" | grep -q "^[+-]"; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}
fi


# Most backends will try to use git, either directly or indirectly.
# Unfortunately this does not work when "$DESTDIR" points to a
# subdirectory of a git tree, which has the effect that no patches
# are applied, but the exitcode is zero. To avoid broken builds we
# will workaround this issue or abort. For more information see
# https://github.com/wine-compholio/wine-staging/issues/7
test ! -d ".git" && git rev-parse --git-dir >/dev/null 2>&1
workaround_git_bug="$?"

# Apply the patches using gitapply.sh, a small wrapper around 'patch'
if test "$backend" = "patch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		echo "Applying $1"
		if ! "$patchdir/gitapply.sh" $gitapply_args < "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# 'epatch' backend - used on Gentoo
elif test "$backend" = "epatch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if ! command -v epatch >/dev/null 2>&1 || \
	   ! command -v ebegin >/dev/null 2>&1 || \
	   ! command -v eend   >/dev/null 2>&1 || \
	   ! command -v die    >/dev/null 2>&1; then
		abort "Shell functions epatch/ebegin/eend not found. You have to source this script from your ebuild."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		_shortname="$(basename "$1")"
		if grep -q "^GIT binary patch" "$1"; then
			ebegin "Applying $_shortname"
			if ! "$patchdir/gitapply.sh" $gitapply_args < "$1"; then
				die "Failed Patch: $1!"
			fi
			eend

		else
			epatch "$1" # epatch calls die upon failure
		fi
		unset _shortname
	}

# GIT backend - apply patches using 'git am'
elif test "$backend" = "git" -o "$backend" = "git-am"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-am' not possible when DESTDIR points to a git subdirectory."
	fi

	patch_apply_file()
	{
		echo "Applying $1"
		if ! git am "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
		if test "$enable_autoconf" -gt 1; then
			_do_commit=0

			# Run 'autoreconf -f' if required
			if git show --pretty=format: --name-only | grep -q "^\(configure.ac\|aclocal.m4\)$"; then
				if ! update_configure; then
					abort "'autoreconf -f' failed."
				fi
				git add ./configure
				git add ./include/config.h.in
				_do_commit=1
			fi

			# Run './tools/make_requests' if required
			if git show --pretty=format: --name-only | grep -q "^server/"; then
				if ! update_protocol; then
					abort "'./tools/make_requests' failed."
				fi
				git add ./include/wine/server_protocol.h
				git add ./server/trace.c
				git add ./server/request.h
				_do_commit=1
			fi

			if test "$_do_commit" -ne 0; then
				if ! git commit --amend --reuse-message HEAD; then
					abort "Failed to include autogenerated changes in commit."
				fi
			fi

			unset _do_commit
		fi
	}

# Git apply backend
elif test "$backend" = "git-apply"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-apply' not possible when DESTDIR points to a git subdirectory."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		echo "Applying $1"
		if ! git apply "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# Stacked GIT backend - import the patches (mainly for developers)
elif test "$backend" = "stg"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'stg' not possible when DESTDIR points to a git subdirectory."
	fi

	# Only import the regular patches, no autogenerated ones -
	# moreover, don't run autoreconf or ./tools/make_requests.
	enable_patchlist=0
	enable_autoconf=0

	patch_apply_file()
	{
		echo "Applying $1"
		_shortname="$(basename "$1")"
		if ! echo "staging/$_shortname" | cat - "$1" | stg import; then
			abort "Failed to apply patch, aborting!"
		fi
		unset _shortname
	}

else
	abort "Selected backend $backend not supported."
fi

patch_apply()
{
	patch_apply_file "$patchdir/$1"
}


if test "$enable_category_stable" -eq 1; then
	if test "$enable_Compiler_Warnings" -gt 1; then
		abort "Patchset Compiler_Warnings disabled, but category-stable depends on that."
	fi
	if test "$enable_Staging" -gt 1; then
		abort "Patchset Staging disabled, but category-stable depends on that."
	fi
	if test "$enable_combase_String" -gt 1; then
		abort "Patchset combase-String disabled, but category-stable depends on that."
	fi
	if test "$enable_configure_Absolute_RPATH" -gt 1; then
		abort "Patchset configure-Absolute_RPATH disabled, but category-stable depends on that."
	fi
	if test "$enable_d3d9_Skip_Tests" -gt 1; then
		abort "Patchset d3d9-Skip_Tests disabled, but category-stable depends on that."
	fi
	if test "$enable_d3d9_Surface_Refcount" -gt 1; then
		abort "Patchset d3d9-Surface_Refcount disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_24_ID3DXEffect" -gt 1; then
		abort "Patchset d3dx9_24-ID3DXEffect disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_25_ID3DXEffect" -gt 1; then
		abort "Patchset d3dx9_25-ID3DXEffect disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_26_ID3DXEffect" -gt 1; then
		abort "Patchset d3dx9_26-ID3DXEffect disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_AnimationController" -gt 1; then
		abort "Patchset d3dx9_36-AnimationController disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_D3DXStubs" -gt 1; then
		abort "Patchset d3dx9_36-D3DXStubs disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_FindNextValidTechnique" -gt 1; then
		abort "Patchset d3dx9_36-FindNextValidTechnique disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_Optimize_Inplace" -gt 1; then
		abort "Patchset d3dx9_36-Optimize_Inplace disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_Texture_Align" -gt 1; then
		abort "Patchset d3dx9_36-Texture_Align disabled, but category-stable depends on that."
	fi
	if test "$enable_d3dx9_36_UpdateSkinnedMesh" -gt 1; then
		abort "Patchset d3dx9_36-UpdateSkinnedMesh disabled, but category-stable depends on that."
	fi
	if test "$enable_dbghelp_Debug_Symbols" -gt 1; then
		abort "Patchset dbghelp-Debug_Symbols disabled, but category-stable depends on that."
	fi
	if test "$enable_ddraw_EnumSurfaces" -gt 1; then
		abort "Patchset ddraw-EnumSurfaces disabled, but category-stable depends on that."
	fi
	if test "$enable_dinput_Events" -gt 1; then
		abort "Patchset dinput-Events disabled, but category-stable depends on that."
	fi
	if test "$enable_dxgi_GetDesc" -gt 1; then
		abort "Patchset dxgi-GetDesc disabled, but category-stable depends on that."
	fi
	if test "$enable_fonts_Missing_Fonts" -gt 1; then
		abort "Patchset fonts-Missing_Fonts disabled, but category-stable depends on that."
	fi
	if test "$enable_gdi32_MaxPixelFormats" -gt 1; then
		abort "Patchset gdi32-MaxPixelFormats disabled, but category-stable depends on that."
	fi
	if test "$enable_gdiplus_GdipCreateEffect" -gt 1; then
		abort "Patchset gdiplus-GdipCreateEffect disabled, but category-stable depends on that."
	fi
	if test "$enable_kernel32_CompareStringEx" -gt 1; then
		abort "Patchset kernel32-CompareStringEx disabled, but category-stable depends on that."
	fi
	if test "$enable_kernel32_Named_Pipe" -gt 1; then
		abort "Patchset kernel32-Named_Pipe disabled, but category-stable depends on that."
	fi
	if test "$enable_libs_Debug_Channel" -gt 1; then
		abort "Patchset libs-Debug_Channel disabled, but category-stable depends on that."
	fi
	if test "$enable_libs_Unicode_Collation" -gt 1; then
		abort "Patchset libs-Unicode_Collation disabled, but category-stable depends on that."
	fi
	if test "$enable_makedep_PARENTSPEC" -gt 1; then
		abort "Patchset makedep-PARENTSPEC disabled, but category-stable depends on that."
	fi
	if test "$enable_mfplat_MFTRegister" -gt 1; then
		abort "Patchset mfplat-MFTRegister disabled, but category-stable depends on that."
	fi
	if test "$enable_mountmgr_DosDevices" -gt 1; then
		abort "Patchset mountmgr-DosDevices disabled, but category-stable depends on that."
	fi
	if test "$enable_mscoree_CorValidateImage" -gt 1; then
		abort "Patchset mscoree-CorValidateImage disabled, but category-stable depends on that."
	fi
	if test "$enable_msvcp90_basic_string_dtor" -gt 1; then
		abort "Patchset msvcp90-basic_string_dtor disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_APC_Start_Process" -gt 1; then
		abort "Patchset ntdll-APC_Start_Process disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_CLI_Images" -gt 1; then
		abort "Patchset ntdll-CLI_Images disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_DllRedirects" -gt 1; then
		abort "Patchset ntdll-DllRedirects disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_FileFsFullSizeInformation" -gt 1; then
		abort "Patchset ntdll-FileFsFullSizeInformation disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_Fix_Alignment" -gt 1; then
		abort "Patchset ntdll-Fix_Alignment disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_FreeBSD_Directory" -gt 1; then
		abort "Patchset ntdll-FreeBSD_Directory disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_Heap_FreeLists" -gt 1; then
		abort "Patchset ntdll-Heap_FreeLists disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_NtSetLdtEntries" -gt 1; then
		abort "Patchset ntdll-NtSetLdtEntries disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_Pipe_SpecialCharacters" -gt 1; then
		abort "Patchset ntdll-Pipe_SpecialCharacters disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_RtlIpStringToAddress" -gt 1; then
		abort "Patchset ntdll-RtlIpStringToAddress disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_Threading" -gt 1; then
		abort "Patchset ntdll-Threading disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_User_Shared_Data" -gt 1; then
		abort "Patchset ntdll-User_Shared_Data disabled, but category-stable depends on that."
	fi
	if test "$enable_ntdll_WriteWatches" -gt 1; then
		abort "Patchset ntdll-WriteWatches disabled, but category-stable depends on that."
	fi
	if test "$enable_opengl32_Revert_Disable_Ext" -gt 1; then
		abort "Patchset opengl32-Revert_Disable_Ext disabled, but category-stable depends on that."
	fi
	if test "$enable_server_Address_List_Change" -gt 1; then
		abort "Patchset server-Address_List_Change disabled, but category-stable depends on that."
	fi
	if test "$enable_server_ClipCursor" -gt 1; then
		abort "Patchset server-ClipCursor disabled, but category-stable depends on that."
	fi
	if test "$enable_server_CreateProcess_ACLs" -gt 1; then
		abort "Patchset server-CreateProcess_ACLs disabled, but category-stable depends on that."
	fi
	if test "$enable_server_Delete_On_Close" -gt 1; then
		abort "Patchset server-Delete_On_Close disabled, but category-stable depends on that."
	fi
	if test "$enable_server_Unexpected_Wakeup" -gt 1; then
		abort "Patchset server-Unexpected_Wakeup disabled, but category-stable depends on that."
	fi
	if test "$enable_setupapi_SetupDiSetDeviceInstallParamsW" -gt 1; then
		abort "Patchset setupapi-SetupDiSetDeviceInstallParamsW disabled, but category-stable depends on that."
	fi
	if test "$enable_shell32_RunDLL_CallEntry16" -gt 1; then
		abort "Patchset shell32-RunDLL_CallEntry16 disabled, but category-stable depends on that."
	fi
	if test "$enable_shell32_SHFileOperation" -gt 1; then
		abort "Patchset shell32-SHFileOperation disabled, but category-stable depends on that."
	fi
	if test "$enable_urlmon_CoInternetSetFeatureEnabled" -gt 1; then
		abort "Patchset urlmon-CoInternetSetFeatureEnabled disabled, but category-stable depends on that."
	fi
	if test "$enable_user32_DrawTextExW" -gt 1; then
		abort "Patchset user32-DrawTextExW disabled, but category-stable depends on that."
	fi
	if test "$enable_user32_WndProc" -gt 1; then
		abort "Patchset user32-WndProc disabled, but category-stable depends on that."
	fi
	if test "$enable_windowscodecs_GIF_Decoder" -gt 1; then
		abort "Patchset windowscodecs-GIF_Decoder disabled, but category-stable depends on that."
	fi
	if test "$enable_wine_inf_Performance" -gt 1; then
		abort "Patchset wine.inf-Performance disabled, but category-stable depends on that."
	fi
	if test "$enable_wine_inf_ProfileList_UserSID" -gt 1; then
		abort "Patchset wine.inf-ProfileList_UserSID disabled, but category-stable depends on that."
	fi
	if test "$enable_wineboot_HKEY_DYN_DATA" -gt 1; then
		abort "Patchset wineboot-HKEY_DYN_DATA disabled, but category-stable depends on that."
	fi
	if test "$enable_winecfg_Libraries" -gt 1; then
		abort "Patchset winecfg-Libraries disabled, but category-stable depends on that."
	fi
	if test "$enable_wined3d_Multisampling" -gt 1; then
		abort "Patchset wined3d-Multisampling disabled, but category-stable depends on that."
	fi
	if test "$enable_wined3d_Revert_PixelFormat" -gt 1; then
		abort "Patchset wined3d-Revert_PixelFormat disabled, but category-stable depends on that."
	fi
	if test "$enable_wined3d_UnhandledBlendFactor" -gt 1; then
		abort "Patchset wined3d-UnhandledBlendFactor disabled, but category-stable depends on that."
	fi
	if test "$enable_wined3d_resource_check_usage" -gt 1; then
		abort "Patchset wined3d-resource_check_usage disabled, but category-stable depends on that."
	fi
	if test "$enable_wined3d_wined3d_swapchain_present" -gt 1; then
		abort "Patchset wined3d-wined3d_swapchain_present disabled, but category-stable depends on that."
	fi
	if test "$enable_winemenubuilder_Desktop_Icon_Path" -gt 1; then
		abort "Patchset winemenubuilder-Desktop_Icon_Path disabled, but category-stable depends on that."
	fi
	if test "$enable_winepulse_PulseAudio_Support" -gt 1; then
		abort "Patchset winepulse-PulseAudio_Support disabled, but category-stable depends on that."
	fi
	if test "$enable_winex11_Thread_Data" -gt 1; then
		abort "Patchset winex11-Thread_Data disabled, but category-stable depends on that."
	fi
	if test "$enable_winex11_Window_Style" -gt 1; then
		abort "Patchset winex11-Window_Style disabled, but category-stable depends on that."
	fi
	if test "$enable_winex11_XEMBED" -gt 1; then
		abort "Patchset winex11-XEMBED disabled, but category-stable depends on that."
	fi
	if test "$enable_winex11_wglShareLists" -gt 1; then
		abort "Patchset winex11-wglShareLists disabled, but category-stable depends on that."
	fi
	if test "$enable_wininet_ParseX509EncodedCertificateForListBoxEntry" -gt 1; then
		abort "Patchset wininet-ParseX509EncodedCertificateForListBoxEntry disabled, but category-stable depends on that."
	fi
	if test "$enable_winmm_Delay_Import_Depends" -gt 1; then
		abort "Patchset winmm-Delay_Import_Depends disabled, but category-stable depends on that."
	fi
	if test "$enable_wpcap_Dynamic_Linking" -gt 1; then
		abort "Patchset wpcap-Dynamic_Linking disabled, but category-stable depends on that."
	fi
	if test "$enable_ws2_32_Connect_Time" -gt 1; then
		abort "Patchset ws2_32-Connect_Time disabled, but category-stable depends on that."
	fi
	if test "$enable_ws2_32_WriteWatches" -gt 1; then
		abort "Patchset ws2_32-WriteWatches disabled, but category-stable depends on that."
	fi
	enable_Compiler_Warnings=1
	enable_Staging=1
	enable_combase_String=1
	enable_configure_Absolute_RPATH=1
	enable_d3d9_Skip_Tests=1
	enable_d3d9_Surface_Refcount=1
	enable_d3dx9_24_ID3DXEffect=1
	enable_d3dx9_25_ID3DXEffect=1
	enable_d3dx9_26_ID3DXEffect=1
	enable_d3dx9_36_AnimationController=1
	enable_d3dx9_36_D3DXStubs=1
	enable_d3dx9_36_FindNextValidTechnique=1
	enable_d3dx9_36_Optimize_Inplace=1
	enable_d3dx9_36_Texture_Align=1
	enable_d3dx9_36_UpdateSkinnedMesh=1
	enable_dbghelp_Debug_Symbols=1
	enable_ddraw_EnumSurfaces=1
	enable_dinput_Events=1
	enable_dxgi_GetDesc=1
	enable_fonts_Missing_Fonts=1
	enable_gdi32_MaxPixelFormats=1
	enable_gdiplus_GdipCreateEffect=1
	enable_kernel32_CompareStringEx=1
	enable_kernel32_Named_Pipe=1
	enable_libs_Debug_Channel=1
	enable_libs_Unicode_Collation=1
	enable_makedep_PARENTSPEC=1
	enable_mfplat_MFTRegister=1
	enable_mountmgr_DosDevices=1
	enable_mscoree_CorValidateImage=1
	enable_msvcp90_basic_string_dtor=1
	enable_ntdll_APC_Start_Process=1
	enable_ntdll_CLI_Images=1
	enable_ntdll_DOS_Attributes=1
	enable_ntdll_DllRedirects=1
	enable_ntdll_FileFsFullSizeInformation=1
	enable_ntdll_Fix_Alignment=1
	enable_ntdll_FreeBSD_Directory=1
	enable_ntdll_Heap_FreeLists=1
	enable_ntdll_NtSetLdtEntries=1
	enable_ntdll_Pipe_SpecialCharacters=1
	enable_ntdll_RtlIpStringToAddress=1
	enable_ntdll_Threading=1
	enable_ntdll_User_Shared_Data=1
	enable_ntdll_WriteWatches=1
	enable_opengl32_Revert_Disable_Ext=1
	enable_server_Address_List_Change=1
	enable_server_ClipCursor=1
	enable_server_CreateProcess_ACLs=1
	enable_server_Delete_On_Close=1
	enable_server_Unexpected_Wakeup=1
	enable_setupapi_SetupDiSetDeviceInstallParamsW=1
	enable_shell32_RunDLL_CallEntry16=1
	enable_shell32_SHFileOperation=1
	enable_urlmon_CoInternetSetFeatureEnabled=1
	enable_user32_DrawTextExW=1
	enable_user32_WndProc=1
	enable_windowscodecs_GIF_Decoder=1
	enable_wine_inf_Performance=1
	enable_wine_inf_ProfileList_UserSID=1
	enable_wineboot_HKEY_DYN_DATA=1
	enable_winecfg_Libraries=1
	enable_wined3d_Multisampling=1
	enable_wined3d_Revert_PixelFormat=1
	enable_wined3d_UnhandledBlendFactor=1
	enable_wined3d_resource_check_usage=1
	enable_wined3d_wined3d_swapchain_present=1
	enable_winemenubuilder_Desktop_Icon_Path=1
	enable_winepulse_PulseAudio_Support=1
	enable_winex11_Thread_Data=1
	enable_winex11_Window_Style=1
	enable_winex11_XEMBED=1
	enable_winex11_wglShareLists=1
	enable_wininet_ParseX509EncodedCertificateForListBoxEntry=1
	enable_winmm_Delay_Import_Depends=1
	enable_wpcap_Dynamic_Linking=1
	enable_ws2_32_Connect_Time=1
	enable_ws2_32_WriteWatches=1
fi

if test "$enable_wined3d_CSMT_Main" -eq 1; then
	if test "$enable_wined3d_CSMT_Helper" -gt 1; then
		abort "Patchset wined3d-CSMT_Helper disabled, but wined3d-CSMT_Main depends on that."
	fi
	enable_wined3d_CSMT_Helper=1
fi

if test "$enable_wined3d_MESA_GPU_Info" -eq 1; then
	if test "$enable_wined3d_Accounting" -gt 1; then
		abort "Patchset wined3d-Accounting disabled, but wined3d-MESA_GPU_Info depends on that."
	fi
	enable_wined3d_Accounting=1
fi

if test "$enable_wined3d_CSMT_Helper" -eq 1; then
	if test "$enable_makedep_PARENTSPEC" -gt 1; then
		abort "Patchset makedep-PARENTSPEC disabled, but wined3d-CSMT_Helper depends on that."
	fi
	if test "$enable_ntdll_DllRedirects" -gt 1; then
		abort "Patchset ntdll-DllRedirects disabled, but wined3d-CSMT_Helper depends on that."
	fi
	if test "$enable_wined3d_DXTn" -gt 1; then
		abort "Patchset wined3d-DXTn disabled, but wined3d-CSMT_Helper depends on that."
	fi
	enable_makedep_PARENTSPEC=1
	enable_ntdll_DllRedirects=1
	enable_wined3d_DXTn=1
fi

if test "$enable_shell32_SHFileOperation" -eq 1; then
	if test "$enable_shell32_Progress_Dialog" -gt 1; then
		abort "Patchset shell32-Progress_Dialog disabled, but shell32-SHFileOperation depends on that."
	fi
	enable_shell32_Progress_Dialog=1
fi

if test "$enable_shell32_Progress_Dialog" -eq 1; then
	if test "$enable_kernel32_CopyFileEx" -gt 1; then
		abort "Patchset kernel32-CopyFileEx disabled, but shell32-Progress_Dialog depends on that."
	fi
	enable_kernel32_CopyFileEx=1
fi

if test "$enable_server_Shared_Memory" -eq 1; then
	if test "$enable_dinput_Events" -gt 1; then
		abort "Patchset dinput-Events disabled, but server-Shared_Memory depends on that."
	fi
	if test "$enable_ntdll_Threading" -gt 1; then
		abort "Patchset ntdll-Threading disabled, but server-Shared_Memory depends on that."
	fi
	if test "$enable_server_ClipCursor" -gt 1; then
		abort "Patchset server-ClipCursor disabled, but server-Shared_Memory depends on that."
	fi
	if test "$enable_server_Key_State" -gt 1; then
		abort "Patchset server-Key_State disabled, but server-Shared_Memory depends on that."
	fi
	if test "$enable_server_PeekMessage" -gt 1; then
		abort "Patchset server-PeekMessage disabled, but server-Shared_Memory depends on that."
	fi
	enable_dinput_Events=1
	enable_ntdll_Threading=1
	enable_server_ClipCursor=1
	enable_server_Key_State=1
	enable_server_PeekMessage=1
fi

if test "$enable_server_JobObjects" -eq 1; then
	if test "$enable_server_Misc_ACL" -gt 1; then
		abort "Patchset server-Misc_ACL disabled, but server-JobObjects depends on that."
	fi
	if test "$enable_server_OpenProcess" -gt 1; then
		abort "Patchset server-OpenProcess disabled, but server-JobObjects depends on that."
	fi
	enable_server_Misc_ACL=1
	enable_server_OpenProcess=1
fi

if test "$enable_server_ACL_Compat" -eq 1; then
	if test "$enable_server_Inherited_ACLs" -gt 1; then
		abort "Patchset server-Inherited_ACLs disabled, but server-ACL_Compat depends on that."
	fi
	enable_server_Inherited_ACLs=1
fi

if test "$enable_server_Inherited_ACLs" -eq 1; then
	if test "$enable_server_Stored_ACLs" -gt 1; then
		abort "Patchset server-Stored_ACLs disabled, but server-Inherited_ACLs depends on that."
	fi
	enable_server_Stored_ACLs=1
fi

if test "$enable_server_Stored_ACLs" -eq 1; then
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but server-Stored_ACLs depends on that."
	fi
	if test "$enable_server_File_Permissions" -gt 1; then
		abort "Patchset server-File_Permissions disabled, but server-Stored_ACLs depends on that."
	fi
	if test "$enable_server_RootDirectory_File" -gt 1; then
		abort "Patchset server-RootDirectory_File disabled, but server-Stored_ACLs depends on that."
	fi
	enable_ntdll_DOS_Attributes=1
	enable_server_File_Permissions=1
	enable_server_RootDirectory_File=1
fi

if test "$enable_server_RootDirectory_File" -eq 1; then
	if test "$enable_ntdll_FileDispositionInformation" -gt 1; then
		abort "Patchset ntdll-FileDispositionInformation disabled, but server-RootDirectory_File depends on that."
	fi
	enable_ntdll_FileDispositionInformation=1
fi

if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	if test "$enable_nvcuvid_CUDA_Video_Support" -gt 1; then
		abort "Patchset nvcuvid-CUDA_Video_Support disabled, but nvencodeapi-Video_Encoder depends on that."
	fi
	enable_nvcuvid_CUDA_Video_Support=1
fi

if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	if test "$enable_nvapi_Stub_DLL" -gt 1; then
		abort "Patchset nvapi-Stub_DLL disabled, but nvcuvid-CUDA_Video_Support depends on that."
	fi
	enable_nvapi_Stub_DLL=1
fi

if test "$enable_nvapi_Stub_DLL" -eq 1; then
	if test "$enable_nvcuda_CUDA_Support" -gt 1; then
		abort "Patchset nvcuda-CUDA_Support disabled, but nvapi-Stub_DLL depends on that."
	fi
	enable_nvcuda_CUDA_Support=1
fi

if test "$enable_ntdll_WriteWatches" -eq 1; then
	if test "$enable_kernel32_Named_Pipe" -gt 1; then
		abort "Patchset kernel32-Named_Pipe disabled, but ntdll-WriteWatches depends on that."
	fi
	if test "$enable_ws2_32_WriteWatches" -gt 1; then
		abort "Patchset ws2_32-WriteWatches disabled, but ntdll-WriteWatches depends on that."
	fi
	enable_kernel32_Named_Pipe=1
	enable_ws2_32_WriteWatches=1
fi

if test "$enable_ntdll_Junction_Points" -eq 1; then
	if test "$enable_ntdll_Fix_Free" -gt 1; then
		abort "Patchset ntdll-Fix_Free disabled, but ntdll-Junction_Points depends on that."
	fi
	if test "$enable_ntdll_NtQueryEaFile" -gt 1; then
		abort "Patchset ntdll-NtQueryEaFile disabled, but ntdll-Junction_Points depends on that."
	fi
	enable_ntdll_Fix_Free=1
	enable_ntdll_NtQueryEaFile=1
fi

if test "$enable_ntdll_CLI_Images" -eq 1; then
	if test "$enable_mscoree_CorValidateImage" -gt 1; then
		abort "Patchset mscoree-CorValidateImage disabled, but ntdll-CLI_Images depends on that."
	fi
	enable_mscoree_CorValidateImage=1
fi

if test "$enable_kernel32_Named_Pipe" -eq 1; then
	if test "$enable_rpcrt4_Pipe_Transport" -gt 1; then
		abort "Patchset rpcrt4-Pipe_Transport disabled, but kernel32-Named_Pipe depends on that."
	fi
	enable_rpcrt4_Pipe_Transport=1
fi

if test "$enable_kernel32_CopyFileEx" -eq 1; then
	if test "$enable_kernel32_SetFileInformationByHandle" -gt 1; then
		abort "Patchset kernel32-SetFileInformationByHandle disabled, but kernel32-CopyFileEx depends on that."
	fi
	if test "$enable_ntdll_FileDispositionInformation" -gt 1; then
		abort "Patchset ntdll-FileDispositionInformation disabled, but kernel32-CopyFileEx depends on that."
	fi
	enable_kernel32_SetFileInformationByHandle=1
	enable_ntdll_FileDispositionInformation=1
fi

if test "$enable_ntdll_FileDispositionInformation" -eq 1; then
	if test "$enable_server_File_Permissions" -gt 1; then
		abort "Patchset server-File_Permissions disabled, but ntdll-FileDispositionInformation depends on that."
	fi
	enable_server_File_Permissions=1
fi

if test "$enable_dxva2_Video_Decoder" -eq 1; then
	if test "$enable_winecfg_Staging" -gt 1; then
		abort "Patchset winecfg-Staging disabled, but dxva2-Video_Decoder depends on that."
	fi
	enable_winecfg_Staging=1
fi

if test "$enable_dsound_EAX" -eq 1; then
	if test "$enable_dsound_Fast_Mixer" -gt 1; then
		abort "Patchset dsound-Fast_Mixer disabled, but dsound-EAX depends on that."
	fi
	enable_dsound_Fast_Mixer=1
fi

if test "$enable_d3dx9_36_AnimationController" -eq 1; then
	if test "$enable_d3dx9_36_DXTn" -gt 1; then
		abort "Patchset d3dx9_36-DXTn disabled, but d3dx9_36-AnimationController depends on that."
	fi
	enable_d3dx9_36_DXTn=1
fi

if test "$enable_d3dx9_36_DXTn" -eq 1; then
	if test "$enable_wined3d_DXTn" -gt 1; then
		abort "Patchset wined3d-DXTn disabled, but d3dx9_36-DXTn depends on that."
	fi
	enable_wined3d_DXTn=1
fi

if test "$enable_d3dx9_24_ID3DXEffect" -eq 1; then
	if test "$enable_d3dx9_25_ID3DXEffect" -gt 1; then
		abort "Patchset d3dx9_25-ID3DXEffect disabled, but d3dx9_24-ID3DXEffect depends on that."
	fi
	enable_d3dx9_25_ID3DXEffect=1
fi

if test "$enable_advapi32_LsaLookupSids" -eq 1; then
	if test "$enable_server_CreateProcess_ACLs" -gt 1; then
		abort "Patchset server-CreateProcess_ACLs disabled, but advapi32-LsaLookupSids depends on that."
	fi
	if test "$enable_server_Misc_ACL" -gt 1; then
		abort "Patchset server-Misc_ACL disabled, but advapi32-LsaLookupSids depends on that."
	fi
	enable_server_CreateProcess_ACLs=1
	enable_server_Misc_ACL=1
fi

if test "$enable_Exagear" -eq 1; then
	if test "$enable_ntdll_WRITECOPY" -gt 1; then
		abort "Patchset ntdll-WRITECOPY disabled, but Exagear depends on that."
	fi
	enable_ntdll_WRITECOPY=1
fi

if test "$enable_ntdll_WRITECOPY" -eq 1; then
	if test "$enable_ws2_32_WriteWatches" -gt 1; then
		abort "Patchset ws2_32-WriteWatches disabled, but ntdll-WRITECOPY depends on that."
	fi
	enable_ws2_32_WriteWatches=1
fi


# If autoupdate is enabled then create a tempfile to keep track of all patches
if test "$enable_patchlist" -eq 1; then
	if test "$enable_Staging" -eq 1; then
		patchlist=$(mktemp)
		if test ! -f "$patchlist"; then
			abort "Unable to create temporary file for patchlist."
		fi
	else
		warning "Skipping generation of patchlist because 'Staging' patchset is disabled."
		enable_patchlist=0
	fi
fi


# Patchset Compiler_Warnings
# |
# | Modified files:
# |   *	dlls/d3d9/tests/visual.c, dlls/netapi32/netapi32.c, dlls/winealsa.drv/mmdevdrv.c, dlls/wined3d/glsl_shader.c,
# | 	tools/makedep.c
# |
if test "$enable_Compiler_Warnings" -eq 1; then
	patch_apply Compiler_Warnings/0001-Appease-the-blessed-version-of-gcc-4.5-when-Werror-i.patch
	(
		echo '+    { "Erich E. Hoover", "Appease the blessed version of gcc (4.5) when -Werror is enabled.", 1 },';
	) >> "$patchlist"
fi

# Patchset Coverity
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c
# |
if test "$enable_Coverity" -eq 1; then
	patch_apply Coverity/0001-ws2_32-Fix-uninitialized-memory-access-in-do_poll-Co.patch
	(
		echo '+    { "Sebastian Lackner", "ws2_32: Fix uninitialized memory access in do_poll (Coverity).", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-WriteWatches
# |
# | Modified files:
# |   *	dlls/ntdll/ntdll.spec, dlls/ntdll/ntdll_misc.h, dlls/ntdll/signal_i386.c, dlls/ntdll/virtual.c, dlls/ws2_32/socket.c,
# | 	include/winternl.h
# |
if test "$enable_ws2_32_WriteWatches" -eq 1; then
	patch_apply ws2_32-WriteWatches/0001-ntdll-Expose-wine_uninterrupted_-read-write-_memory-.patch
	patch_apply ws2_32-WriteWatches/0002-ws2_32-Avoid-race-conditions-of-async-WSARecv-operat.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Expose wine_uninterrupted_[read|write]_memory as exports.", 1 },';
		echo '+    { "Sebastian Lackner", "ws2_32: Avoid race-conditions of async WSARecv() operations with write watches.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-WRITECOPY
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29384] Voobly expects correct handling of WRITECOPY memory protection
# |   *	[#35561] MSYS2 expects correct handling of WRITECOPY memory protection
# |
# | Modified files:
# |   *	dlls/advapi32/crypt.c, dlls/kernel32/tests/virtual.c, dlls/ntdll/ntdll_misc.h, dlls/ntdll/server.c,
# | 	dlls/ntdll/signal_arm.c, dlls/ntdll/signal_arm64.c, dlls/ntdll/signal_i386.c, dlls/ntdll/signal_powerpc.c,
# | 	dlls/ntdll/signal_x86_64.c, dlls/ntdll/thread.c, dlls/ntdll/virtual.c
# |
if test "$enable_ntdll_WRITECOPY" -eq 1; then
	patch_apply ntdll-WRITECOPY/0001-ntdll-Trigger-write-watches-before-passing-userdata-.patch
	patch_apply ntdll-WRITECOPY/0002-advapi-Trigger-write-watches-before-passing-userdata.patch
	patch_apply ntdll-WRITECOPY/0003-ntdll-Setup-a-temporary-signal-handler-during-proces.patch
	patch_apply ntdll-WRITECOPY/0004-ntdll-Properly-handle-PAGE_WRITECOPY-protection.-try.patch
	patch_apply ntdll-WRITECOPY/0005-ntdll-Only-enable-true-WRITECOPY-protection-when-a-s.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Trigger write watches before passing userdata pointer to wait_reply.", 1 },';
		echo '+    { "Sebastian Lackner", "advapi: Trigger write watches before passing userdata pointer to read syscall.", 1 },';
		echo '+    { "Michael Müller", "ntdll: Setup a temporary signal handler during process startup to handle page faults.", 2 },';
		echo '+    { "Michael Müller", "ntdll: Properly handle PAGE_WRITECOPY protection.", 5 },';
		echo '+    { "Michael Müller", "ntdll: Only enable true WRITECOPY protection when a special environment variable is set.", 1 },';
	) >> "$patchlist"
fi

# Patchset Exagear
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/signal_i386.c, dlls/ntdll/virtual.c, server/ptrace.c
# |
if test "$enable_Exagear" -eq 1; then
	patch_apply Exagear/0001-ntdll-Implement-emulation-of-SIDT-instruction-when-u.patch
	patch_apply Exagear/0002-ntdll-Fix-issues-with-write-watches-when-using-Exage.patch
	patch_apply Exagear/0003-server-Don-t-attempt-to-use-ptrace-when-running-with.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Implement emulation of SIDT instruction when using Exagear.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Fix issues with write watches when using Exagear.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Don'\''t attempt to use ptrace when running with Exagear.", 1 },';
	) >> "$patchlist"
fi

# Patchset Pipelight
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/wined3d/wined3d.spec, dlls/wined3d/wined3d_main.c, dlls/winex11.drv/init.c,
# | 	dlls/winex11.drv/opengl.c, dlls/winex11.drv/x11drv.h
# |
if test "$enable_Pipelight" -eq 1; then
	patch_apply Pipelight/0001-winex11-Implement-X11DRV_FLUSH_GDI_DISPLAY-ExtEscape-c.patch
	patch_apply Pipelight/0002-user32-Decrease-minimum-SetTimer-interval-to-5-ms.patch
	patch_apply Pipelight/0003-wined3d-allow-changing-strict-drawing-through-an-exp.patch
	patch_apply Pipelight/0004-winex11.drv-Indicate-direct-rendering-through-OpenGL.patch
	(
		echo '+    { "Sebastian Lackner", "winex11: Implement X11DRV_FLUSH_GDI_DISPLAY ExtEscape command.", 1 },';
		echo '+    { "Michael Müller", "user32: Decrease minimum SetTimer interval to 5 ms.", 2 },';
		echo '+    { "Michael Müller", "wined3d: allow changing strict drawing through an exported function.", 1 },';
		echo '+    { "Michael Müller", "winex11.drv: Indicate direct rendering through OpenGL extension.", 1 },';
	) >> "$patchlist"
fi

# Patchset Staging
# |
# | Modified files:
# |   *	dlls/kernel32/process.c, dlls/ntdll/misc.c, dlls/ntdll/ntdll.spec, include/wine/library.h, libs/wine/Makefile.in,
# | 	libs/wine/config.c, libs/wine/loader.c, libs/wine/wine.def, libs/wine/wine.map, loader/main.c, po/ar.po, po/bg.po,
# | 	po/ca.po, po/cs.po, po/da.po, po/de.po, po/el.po, po/en.po, po/en_US.po, po/eo.po, po/es.po, po/fa.po, po/fi.po,
# | 	po/fr.po, po/he.po, po/hi.po, po/hr.po, po/hu.po, po/it.po, po/ja.po, po/ko.po, po/lt.po, po/ml.po, po/nb_NO.po,
# | 	po/nl.po, po/or.po, po/pa.po, po/pl.po, po/pt_BR.po, po/pt_PT.po, po/rm.po, po/ro.po, po/ru.po, po/sk.po, po/sl.po,
# | 	po/sr_RS@cyrillic.po, po/sr_RS@latin.po, po/sv.po, po/te.po, po/th.po, po/tr.po, po/uk.po, po/wa.po, po/wine.pot,
# | 	po/zh_CN.po, po/zh_TW.po, programs/winedbg/winedbg.rc
# |
if test "$enable_Staging" -eq 1; then
	patch_apply Staging/0001-kernel32-Add-winediag-message-to-show-warning-that-t.patch
	patch_apply Staging/0002-winedbg-Change-bug-reporting-URL-to-Wine-Staging.patch
	patch_apply Staging/0003-winelib-Append-Staging-at-the-end-of-the-version-s.patch
	patch_apply Staging/0004-loader-Add-commandline-option-patches-to-show-the-pa.patch
	patch_apply Staging/0005-loader-Add-commandline-option-check-libs.patch
	patch_apply Staging/0006-loader-Print-library-paths-for-check-libs-on-Mac-OS-.patch
	(
		echo '+    { "Sebastian Lackner", "kernel32: Add winediag message to show warning, that this isn'\''t vanilla wine.", 1 },';
		echo '+    { "Sebastian Lackner", "winedbg: Change bug reporting URL to Wine Staging.", 1 },';
		echo '+    { "Sebastian Lackner", "winelib: Append '\''(Staging)'\'' at the end of the version string.", 1 },';
		echo '+    { "Sebastian Lackner", "loader: Add commandline option --patches to show the patch list.", 1 },';
		echo '+    { "Michael Müller", "loader: Add commandline option --check-libs.", 1 },';
		echo '+    { "Michael Müller", "loader: Print library paths for --check-libs on Mac OS X.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-Misc_ACL
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#15980] GetSecurityInfo returns NULL DACL for process object
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, server/process.c, server/security.h, server/token.c
# |
if test "$enable_server_Misc_ACL" -eq 1; then
	patch_apply server-Misc_ACL/0001-server-Add-default-security-descriptor-ownership-for.patch
	patch_apply server-Misc_ACL/0002-server-Add-default-security-descriptor-DACL-for-proc.patch
	(
		echo '+    { "Erich E. Hoover", "server: Add default security descriptor ownership for processes.", 1 },';
		echo '+    { "Erich E. Hoover", "server: Add default security descriptor DACL for processes.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-CreateProcess_ACLs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22006] Support for process ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/kernel32/process.c, server/process.c, server/protocol.def
# |
if test "$enable_server_CreateProcess_ACLs" -eq 1; then
	patch_apply server-CreateProcess_ACLs/0001-server-Support-for-thread-and-process-security-descr.patch
	patch_apply server-CreateProcess_ACLs/0002-kernel32-Implement-passing-security-descriptors-from.patch
	patch_apply server-CreateProcess_ACLs/0003-advapi32-tests-Add-additional-tests-for-passing-a-th.patch
	(
		echo '+    { "Sebastian Lackner", "server: Support for thread and process security descriptors in new_process wineserver call.", 2 },';
		echo '+    { "Sebastian Lackner", "kernel32: Implement passing security descriptors from CreateProcess to the wineserver.", 2 },';
		echo '+    { "Joris van der Wel", "advapi32/tests: Add additional tests for passing a thread sd to CreateProcess.", 1 },';
	) >> "$patchlist"
fi

# Patchset advapi32-LsaLookupSids
# |
# | Modified files:
# |   *	dlls/advapi32/lsa.c, dlls/advapi32/security.c, dlls/advapi32/tests/security.c, server/token.c
# |
if test "$enable_advapi32_LsaLookupSids" -eq 1; then
	patch_apply advapi32-LsaLookupSids/0001-advapi32-Initialize-buffer-length-to-zero-in-LsaLook.patch
	patch_apply advapi32-LsaLookupSids/0002-advapi32-Prepend-a-hidden-LSA_TRUST_INFORMATION-in-L.patch
	patch_apply advapi32-LsaLookupSids/0003-advapi32-Prepend-a-hidden-LSA_TRUST_INFORMATION-in-L.patch
	patch_apply advapi32-LsaLookupSids/0004-advapi32-Fallback-to-Sid-string-when-LookupAccountSi.patch
	patch_apply advapi32-LsaLookupSids/0005-advapi32-tests-Test-prefix-and-use-of-TokenPrimaryGr.patch
	patch_apply advapi32-LsaLookupSids/0006-server-Create-primary-group-using-DOMAIN_GROUP_RID_U.patch
	patch_apply advapi32-LsaLookupSids/0007-advapi32-Fix-name-and-use-of-DOMAIN_GROUP_RID_USERS.patch
	(
		echo '+    { "Qian Hong", "advapi32: Initialize buffer length to zero in LsaLookupSids to prevent crash.", 2 },';
		echo '+    { "Qian Hong", "advapi32: Prepend a hidden LSA_TRUST_INFORMATION in LsaLookupSids to avoid crash when Domains[-1] incorrectly accessed by application.", 2 },';
		echo '+    { "Qian Hong", "advapi32: Prepend a hidden LSA_TRUST_INFORMATION in LsaLookupNames2 to avoid crash when Domains[-1] incorrectly accessed by application.", 2 },';
		echo '+    { "Qian Hong", "advapi32: Fallback to Sid string when LookupAccountSid fails.", 1 },';
		echo '+    { "Qian Hong", "advapi32/tests: Test prefix and use of TokenPrimaryGroup Sid.", 1 },';
		echo '+    { "Qian Hong", "server: Create primary group using DOMAIN_GROUP_RID_USERS.", 1 },';
		echo '+    { "Qian Hong", "advapi32: Fix name and use of DOMAIN_GROUP_RID_USERS.", 1 },';
	) >> "$patchlist"
fi

# Patchset browseui-Progress_Dialog
# |
# | Modified files:
# |   *	dlls/browseui/browseui.rc, dlls/browseui/progressdlg.c, dlls/browseui/resids.h
# |
if test "$enable_browseui_Progress_Dialog" -eq 1; then
	patch_apply browseui-Progress_Dialog/0001-browseui-Implement-IProgressDialog-SetAnimation.patch
	patch_apply browseui-Progress_Dialog/0002-browseui-Implement-PROGDLG_AUTOTIME-flag-for-IProgre.patch
	(
		echo '+    { "Michael Müller", "browseui: Implement IProgressDialog::SetAnimation.", 1 },';
		echo '+    { "Michael Müller", "browseui: Implement PROGDLG_AUTOTIME flag for IProgressDialog.", 1 },';
	) >> "$patchlist"
fi

# Patchset combase-String
# |
# | Modified files:
# |   *	dlls/api-ms-win-core-winrt-string-l1-1-0/api-ms-win-core-winrt-string-l1-1-0.spec, dlls/combase/combase.spec,
# | 	dlls/combase/string.c, dlls/combase/tests/string.c, include/winerror.h
# |
if test "$enable_combase_String" -eq 1; then
	patch_apply combase-String/0001-combase-implement-WindowsSubstring-try-2.patch
	(
		echo '+    { "Thomas Pointhuber", "combase: implement WindowsSubstring.", 2 },';
	) >> "$patchlist"
fi

# Patchset comctl32-LoadIconMetric
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35375] Support for LoadIconMetric
# |
# | Modified files:
# |   *	dlls/comctl32/Makefile.in, dlls/comctl32/comctl32.spec, dlls/comctl32/icon.c, dlls/comctl32/tests/misc.c,
# | 	include/commctrl.h
# |
if test "$enable_comctl32_LoadIconMetric" -eq 1; then
	patch_apply comctl32-LoadIconMetric/0001-comctl32-Implement-LoadIconMetric-function.patch
	patch_apply comctl32-LoadIconMetric/0002-comctl32-tests-Add-tests-for-LoadIconMetric-function.patch
	(
		echo '+    { "Michael Müller", "comctl32: Implement LoadIconMetric function.", 1 },';
		echo '+    { "Michael Müller", "comctl32/tests: Add tests for LoadIconMetric function.", 1 },';
	) >> "$patchlist"
fi

# Patchset configure-Absolute_RPATH
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26256] Support for setcap on wine-preloader
# |
# | Modified files:
# |   *	configure.ac
# |
if test "$enable_configure_Absolute_RPATH" -eq 1; then
	patch_apply configure-Absolute_RPATH/0001-configure-Also-add-the-absolute-RPATH-when-linking-a.patch
	(
		echo '+    { "Sebastian Lackner", "configure: Also add the absolute RPATH when linking against libwine.", 1 },';
	) >> "$patchlist"
fi

# Patchset configure-Crosscompiling
# |
# | Modified files:
# |   *	aclocal.m4, configure.ac
# |
if test "$enable_configure_Crosscompiling" -eq 1; then
	patch_apply configure-Crosscompiling/0001-configure.ac-Search-for-otool-using-AC_CHECK_TOOL.patch
	patch_apply configure-Crosscompiling/0002-configure.ac-Remove-check-for-strength-reduce-bug.patch
	(
		echo '+    { "Michael Müller", "configure.ac: Search for otool using AC_CHECK_TOOL.", 2 },';
		echo '+    { "Michael Müller", "configure.ac: Remove check for strength-reduce bug.", 1 },';
	) >> "$patchlist"
fi

# Patchset crypt32-CMS_Certificates
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34388] Skip unknown item when decoding a CMS certificate
# |
# | Modified files:
# |   *	dlls/crypt32/decode.c
# |
if test "$enable_crypt32_CMS_Certificates" -eq 1; then
	patch_apply crypt32-CMS_Certificates/0001-crypt32-Skip-unknown-item-when-decoding-a-CMS-certif.patch
	(
		echo '+    { "Charles Davis", "crypt32: Skip unknown item when decoding a CMS certificate.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3d9-DesktopWindow
# |
# | Modified files:
# |   *	dlls/d3d9/tests/device.c, dlls/winex11.drv/opengl.c
# |
if test "$enable_d3d9_DesktopWindow" -eq 1; then
	patch_apply d3d9-DesktopWindow/0001-winex11.drv-Allow-changing-the-opengl-pixel-format-o.patch
	(
		echo '+    { "Michael Müller", "winex11.drv: Allow changing the opengl pixel format on the desktop window.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3d9-Skip_Tests
# |
# | Modified files:
# |   *	dlls/d3d9/tests/device.c
# |
if test "$enable_d3d9_Skip_Tests" -eq 1; then
	patch_apply d3d9-Skip_Tests/0001-d3d9-tests-Avoid-crash-when-surface-and-texture-crea.patch
	(
		echo '+    { "Christian Costa", "d3d9/tests: Avoid crash when surface and texture creation fails.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3d9-Surface_Refcount
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18477] Avoid crashing when broken app tries to release surface although refcount is zero
# |
# | Modified files:
# |   *	dlls/d3d9/surface.c
# |
if test "$enable_d3d9_Surface_Refcount" -eq 1; then
	patch_apply d3d9-Surface_Refcount/0001-d3d9-Don-t-decrease-surface-refcount-when-its-alread.patch
	(
		echo '+    { "Henri Verbeet", "d3d9: Don'\''t decrease surface refcount when its already zero.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_25-ID3DXEffect
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#25138] Fix wrong version of ID3DXEffect interface for d3dx9_25
# |
# | Modified files:
# |   *	dlls/d3dx9_25/Makefile.in, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_25/effect.c, include/d3dx9effect.h
# |
if test "$enable_d3dx9_25_ID3DXEffect" -eq 1; then
	patch_apply d3dx9_25-ID3DXEffect/0001-d3dx9_25-Add-an-interface-wrapper-for-different-vers.patch
	(
		echo '+    { "Sebastian Lackner", "d3dx9_25: Add an interface wrapper for different version of ID3DXEffect.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_24-ID3DXEffect
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec
# |
if test "$enable_d3dx9_24_ID3DXEffect" -eq 1; then
	patch_apply d3dx9_24-ID3DXEffect/0001-d3dx9_24-Add-an-interface-wrapper-for-different-vers.patch
	(
		echo '+    { "Sebastian Lackner", "d3dx9_24: Add an interface wrapper for different version of ID3DXEffect.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_26-ID3DXEffect
# |
# | Modified files:
# |   *	dlls/d3dx9_36/effect.c
# |
if test "$enable_d3dx9_26_ID3DXEffect" -eq 1; then
	patch_apply d3dx9_26-ID3DXEffect/0001-d3dx9_36-Allow-to-query-for-d3dx9_26-specific-ID3DXE.patch
	(
		echo '+    { "Sebastian Lackner", "d3dx9_36: Allow to query for d3dx9_26 specific ID3DXEffect interface.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-DXTn
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#25486] Lego Stunt Rally requires DXTn software de/encoding support
# |   *	[#29586] Tumblebugs 2 requires DXTn software encoding support
# |   *	[#14939] Black & White needs DXTn software decoding support
# |   *	[#17913] Port Royale doesn't display ocean correctly
# |   *	[#29598] eRacer Demo doesn't correctly display text
# |
# | Modified files:
# |   *	configure.ac, dlls/wined3d/Makefile.in, dlls/wined3d/dxtn.c, dlls/wined3d/surface.c, dlls/wined3d/wined3d.spec,
# | 	dlls/wined3d/wined3d_main.c, dlls/wined3d/wined3d_private.h, include/wine/wined3d.h
# |
if test "$enable_wined3d_DXTn" -eq 1; then
	patch_apply wined3d-DXTn/0001-wined3d-Add-support-for-DXTn-software-decoding-throu.patch
	patch_apply wined3d-DXTn/0002-wined3d-Improve-DXTn-support-and-export-conversion-f.patch
	patch_apply wined3d-DXTn/0003-wined3d-add-DXT1-to-B4G4R4A4-DXT1-to-B5G5R5A1-and-DX.patch
	patch_apply wined3d-DXTn/0004-wined3d-Load-dxtn-dylib-library-on-Mac-OS-X.patch
	(
		echo '+    { "Michael Müller", "wined3d: Add support for DXTn software decoding through libtxc_dxtn.", 3 },';
		echo '+    { "Christian Costa", "wined3d: Improve DXTn support and export conversion functions for d3dx9_36.", 1 },';
		echo '+    { "Michael Müller", "wined3d: add DXT1 to B4G4R4A4, DXT1 to B5G5R5A1 and DXT3 to B4G4R4A4 conversion.", 1 },';
		echo '+    { "Michael Müller", "wined3d: Load dxtn dylib library on Mac OS X.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-DXTn
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33768] Fix texture corruption in CSI: Fatal Conspiracy
# |   *	[#37391] Exception during start of fr-043 caused by missing DXTn support
# |   *	[#34692] Fix wrong colors in Wolfenstein (2009)
# |   *	[#24983] Fix crash in Space Rangers2 caused by missing DXTn support
# |
# | Modified files:
# |   *	dlls/d3dx9_36/Makefile.in, dlls/d3dx9_36/surface.c, dlls/d3dx9_36/tests/surface.c
# |
if test "$enable_d3dx9_36_DXTn" -eq 1; then
	patch_apply d3dx9_36-DXTn/0001-d3dx9_36-Add-dxtn-support.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Add dxtn support.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-AnimationController
# |
# | Modified files:
# |   *	dlls/d3dx9_36/Makefile.in, dlls/d3dx9_36/animation.c, dlls/d3dx9_36/d3dx9_36.spec, include/d3dx9anim.h
# |
if test "$enable_d3dx9_36_AnimationController" -eq 1; then
	patch_apply d3dx9_36-AnimationController/0001-d3dx9_36-Implement-D3DXCreateAnimationController-wit.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Implement D3DXCreateAnimationController with a stubbed ID3DXAnimationController interface.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-CloneEffect
# |
# | Modified files:
# |   *	dlls/d3dx9_36/effect.c
# |
if test "$enable_d3dx9_36_CloneEffect" -eq 1; then
	patch_apply d3dx9_36-CloneEffect/0001-d3dx9_36-Improve-stub-for-ID3DXEffectImpl_CloneEffec.patch
	(
		echo '+    { "Sebastian Lackner", "d3dx9_36: Improve stub for ID3DXEffectImpl_CloneEffect.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-D3DXStubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31984] Add stub for D3DXComputeTangentFrameEx
# |   *	[#26379] Support for D3DXComputeNormals
# |   *	[#38334] Add stub for D3DXFrameFind
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/mesh.c, dlls/d3dx9_37/d3dx9_37.spec, dlls/d3dx9_38/d3dx9_38.spec,
# | 	dlls/d3dx9_39/d3dx9_39.spec, dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec, dlls/d3dx9_42/d3dx9_42.spec,
# | 	dlls/d3dx9_43/d3dx9_43.spec
# |
if test "$enable_d3dx9_36_D3DXStubs" -eq 1; then
	patch_apply d3dx9_36-D3DXStubs/0001-d3dx9_36-Add-stub-for-D3DXComputeTangentFrameEx.patch
	patch_apply d3dx9_36-D3DXStubs/0002-d3dx9_36-Add-stub-for-D3DXIntersect.patch
	patch_apply d3dx9_36-D3DXStubs/0003-d3dx9_36-Implement-D3DXComputeNormals.patch
	patch_apply d3dx9_36-D3DXStubs/0004-d3dx9_36-Add-stub-for-D3DXComputeNormalMap.patch
	patch_apply d3dx9_36-D3DXStubs/0005-d3dx9_36-Add-D3DXFrameFind-stub.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Add stub for D3DXComputeTangentFrameEx.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Add stub for D3DXIntersect.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Implement D3DXComputeNormals.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Add stub for D3DXComputeNormalMap.", 1 },';
		echo '+    { "Andrey Gusev", "d3dx9_36: Add D3DXFrameFind stub.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-DDS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26898] Support for DDS file format in D3DXSaveTextureToFileInMemory
# |
# | Modified files:
# |   *	dlls/d3dx9_36/d3dx9_36_private.h, dlls/d3dx9_36/surface.c, dlls/d3dx9_36/texture.c
# |
if test "$enable_d3dx9_36_DDS" -eq 1; then
	patch_apply d3dx9_36-DDS/0001-d3dx9_36-Fix-several-issues-in-save_dds_surface_to_m.patch
	patch_apply d3dx9_36-DDS/0002-d3dx9_36-Add-support-for-FOURCC-surface-to-save_dds_.patch
	patch_apply d3dx9_36-DDS/0003-d3dx9_36-Improve-D3DXSaveTextureToFile-to-save-simpl.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Fix several issues in save_dds_surface_to_memory.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Add support for FOURCC surface to save_dds_surface_to_memory.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Improve D3DXSaveTextureToFile to save simple texture to dds file.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-DrawText
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24754] Support for ID3DXFont::DrawTextA/W
# |
# | Modified files:
# |   *	dlls/d3dx9_36/font.c
# |
if test "$enable_d3dx9_36_DrawText" -eq 1; then
	patch_apply d3dx9_36-DrawText/0001-d3dx9_36-Implement-ID3DXFontImpl_DrawText.patch
	patch_apply d3dx9_36-DrawText/0002-d3dx9_36-Fix-horizontal-centering-in-ID3DXFont_DrawT.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Implement ID3DXFontImpl_DrawText.", 1 },';
		echo '+    { "Christian Costa", "d3dx9_36: Fix horizontal centering in ID3DXFont_DrawText.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-Filter_Warnings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33770] D3DCompileShader should filter specific warning messages
# |
# | Modified files:
# |   *	dlls/d3dx9_36/shader.c
# |
if test "$enable_d3dx9_36_Filter_Warnings" -eq 1; then
	patch_apply d3dx9_36-Filter_Warnings/0001-d3dx9_36-Filter-out-D3DCompile-warning-messages-that.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Filter out D3DCompile warning messages that are not present with D3DCompileShader.", 4 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-FindNextValidTechnique
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34101] Implement ID3DXEffect::FindNextValidTechnique
# |
# | Modified files:
# |   *	dlls/d3dx9_36/effect.c, dlls/d3dx9_36/tests/effect.c
# |
if test "$enable_d3dx9_36_FindNextValidTechnique" -eq 1; then
	patch_apply d3dx9_36-FindNextValidTechnique/0001-d3dx9_36-Implement-ID3DXEffect_FindNextValidTechniqu.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Implement ID3DXEffect_FindNextValidTechnique + add tests.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-GetShaderSemantics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22682] Support for D3DXGetShaderInputSemantics
# |
# | Modified files:
# |   *	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/shader.c, dlls/d3dx9_36/tests/shader.c, include/d3dx9shader.h
# |
if test "$enable_d3dx9_36_GetShaderSemantics" -eq 1; then
	patch_apply d3dx9_36-GetShaderSemantics/0001-d3dx9_36-Implement-D3DXGetShaderInputSemantics-tests.patch
	patch_apply d3dx9_36-GetShaderSemantics/0002-d3dx9_36-Implement-D3DXGetShaderOutputSemantics.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Implement D3DXGetShaderInputSemantics + tests.", 2 },';
		echo '+    { "Christian Costa", "d3dx9_36: Implement D3DXGetShaderOutputSemantics.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-Optimize_Inplace
# |
# | Modified files:
# |   *	dlls/d3dx9_36/mesh.c
# |
if test "$enable_d3dx9_36_Optimize_Inplace" -eq 1; then
	patch_apply d3dx9_36-Optimize_Inplace/0001-d3dx9_36-No-need-to-fail-if-we-don-t-support-vertice.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: No need to fail if we don'\''t support vertices reordering in D3DXMESHOPT_ATTRSORT.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-Texture_Align
# |
# | Modified files:
# |   *	dlls/d3dx9_36/tests/texture.c, dlls/d3dx9_36/texture.c
# |
if test "$enable_d3dx9_36_Texture_Align" -eq 1; then
	patch_apply d3dx9_36-Texture_Align/0001-d3dx9_36-Align-texture-dimensions-to-block-size-for-.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Align texture dimensions to block size for compressed textures in D3DXCheckTextureRequirements.", 1 },';
	) >> "$patchlist"
fi

# Patchset d3dx9_36-UpdateSkinnedMesh
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32572] Support for ID3DXSkinInfoImpl_UpdateSkinnedMesh
# |
# | Modified files:
# |   *	dlls/d3dx9_36/skin.c, dlls/d3dx9_36/tests/mesh.c
# |
if test "$enable_d3dx9_36_UpdateSkinnedMesh" -eq 1; then
	patch_apply d3dx9_36-UpdateSkinnedMesh/0001-d3dx9_36-Implement-ID3DXSkinInfoImpl_UpdateSkinnedMe.patch
	(
		echo '+    { "Christian Costa", "d3dx9_36: Implement ID3DXSkinInfoImpl_UpdateSkinnedMesh.", 1 },';
	) >> "$patchlist"
fi

# Patchset dbghelp-Debug_Symbols
# |
# | Modified files:
# |   *	dlls/dbghelp/Makefile.in, dlls/dbghelp/elf_module.c
# |
if test "$enable_dbghelp_Debug_Symbols" -eq 1; then
	patch_apply dbghelp-Debug_Symbols/0001-dbghelp-Always-check-for-debug-symbols-in-BINDIR.patch
	(
		echo '+    { "Sebastian Lackner", "dbghelp: Always check for debug symbols in BINDIR.", 1 },';
	) >> "$patchlist"
fi

# Patchset ddraw-EnumSurfaces
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#17233] Implement DDENUMSURFACES_CANBECREATED in IDirectDraw7::EnumSurfaces
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c, dlls/ddraw/main.c, dlls/ddraw/tests/d3d.c
# |
if test "$enable_ddraw_EnumSurfaces" -eq 1; then
	patch_apply ddraw-EnumSurfaces/0001-ddraw-Fix-arguments-to-IDirectDraw7-EnumSurfaces-in-.patch
	patch_apply ddraw-EnumSurfaces/0002-ddraw-Implement-DDENUMSURFACES_CANBECREATED-in-IDire.patch
	(
		echo '+    { "Michael Müller", "ddraw: Fix arguments to IDirectDraw7::EnumSurfaces in DllMain.", 1 },';
		echo '+    { "Michael Müller", "ddraw: Implement DDENUMSURFACES_CANBECREATED in IDirectDraw7::EnumSurfaces and fix some bugs.", 1 },';
	) >> "$patchlist"
fi

# Patchset ddraw-IDirect3DTexture2_Load
# |
# | Modified files:
# |   *	dlls/ddraw/surface.c, dlls/ddraw/tests/d3d.c, dlls/ddraw/tests/ddraw2.c
# |
if test "$enable_ddraw_IDirect3DTexture2_Load" -eq 1; then
	patch_apply ddraw-IDirect3DTexture2_Load/0001-ddraw-Allow-size-and-format-conversions-in-IDirect3D.patch
	patch_apply ddraw-IDirect3DTexture2_Load/0002-ddraw-tests-Add-more-tests-for-IDirect3DTexture2-Loa.patch
	(
		echo '+    { "Michael Müller", "ddraw: Allow size and format conversions in IDirect3DTexture2::Load.", 1 },';
		echo '+    { "Michael Müller", "ddraw/tests: Add more tests for IDirect3DTexture2::Load.", 1 },';
	) >> "$patchlist"
fi

# Patchset ddraw-d3d_execute_buffer
# |
# | Modified files:
# |   *	dlls/ddraw/executebuffer.c
# |
if test "$enable_ddraw_d3d_execute_buffer" -eq 1; then
	patch_apply ddraw-d3d_execute_buffer/0001-ddraw-Don-t-call-IDirect3DDevice7_DrawIndexedPrimiti.patch
	(
		echo '+    { "Christian Costa", "ddraw: Don'\''t call IDirect3DDevice7_DrawIndexedPrimitive if there is no primitive.", 1 },';
	) >> "$patchlist"
fi

# Patchset dinput-Events
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8854] Ensure X11 input events are handled even without explicit message loop
# |   *	[#34559] Scrolling causes mouse and screen to lock in Call to Power II
# |
# | Modified files:
# |   *	dlls/dinput/device.c, dlls/dinput/keyboard.c, dlls/dinput/mouse.c, dlls/user32/input.c, dlls/user32/user32.spec,
# | 	include/winuser.h
# |
if test "$enable_dinput_Events" -eq 1; then
	patch_apply dinput-Events/0001-dinput-Ensure-X11-input-events-are-handled-even-with.patch
	patch_apply dinput-Events/0002-dinput-Skip-Wine-specific-__wine_check_for_events-ca.patch
	(
		echo '+    { "Sebastian Lackner", "dinput: Ensure X11 input events are handled even without explicit message loop.", 3 },';
		echo '+    { "Amine Khaldi", "dinput: Skip Wine specific __wine_check_for_events calls in ReactOS.", 1 },';
	) >> "$patchlist"
fi

# Patchset dsound-Fast_Mixer
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30639] Audio stuttering and performance drops in multiple applications
# |
# | Modified files:
# |   *	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/mixer.c
# |
if test "$enable_dsound_Fast_Mixer" -eq 1; then
	patch_apply dsound-Fast_Mixer/0001-dsound-Add-a-linear-resampler-for-use-with-a-large-n.patch
	(
		echo '+    { "Alexander E. Patrakov", "dsound: Add a linear resampler for use with a large number of mixing buffers.", 2 },';
	) >> "$patchlist"
fi

# Patchset dsound-EAX
# |
# | Modified files:
# |   *	dlls/dsound/Makefile.in, dlls/dsound/buffer.c, dlls/dsound/dsound.c, dlls/dsound/dsound_eax.h,
# | 	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/eax.c, dlls/dsound/mixer.c
# |
if test "$enable_dsound_EAX" -eq 1; then
	patch_apply dsound-EAX/0001-dsound-Apply-filters-before-sound-is-multiplied-to-s.patch
	patch_apply dsound-EAX/0002-dsound-Add-EAX-v1-constants-and-structs.patch
	patch_apply dsound-EAX/0003-dsound-Report-that-we-support-EAX-v1.patch
	patch_apply dsound-EAX/0004-dsound-Add-EAX-propset-stubs.patch
	patch_apply dsound-EAX/0005-dsound-Add-EAX-presets.patch
	patch_apply dsound-EAX/0006-dsound-Support-getting-and-setting-EAX-properties.patch
	patch_apply dsound-EAX/0007-dsound-Support-getting-and-setting-EAX-buffer-proper.patch
	patch_apply dsound-EAX/0008-dsound-Add-EAX-init-and-free-stubs.patch
	patch_apply dsound-EAX/0009-dsound-Feed-data-through-EAX-function.patch
	patch_apply dsound-EAX/0010-dsound-Allocate-EAX-delay-lines.patch
	patch_apply dsound-EAX/0011-dsound-Add-EAX-VerbPass-stub.patch
	patch_apply dsound-EAX/0012-dsound-Implement-EAX-lowpass-filter.patch
	patch_apply dsound-EAX/0013-dsound-Add-delay-line-EAX-functions.patch
	patch_apply dsound-EAX/0014-dsound-Implement-EAX-early-reflections.patch
	patch_apply dsound-EAX/0015-dsound-Implement-EAX-decorrelator.patch
	patch_apply dsound-EAX/0016-dsound-Implement-EAX-late-reverb.patch
	patch_apply dsound-EAX/0017-dsound-Implement-EAX-late-all-pass-filter.patch
	patch_apply dsound-EAX/0018-dsound-Various-improvements-to-EAX-support.patch
	patch_apply dsound-EAX/0019-dsound-Allow-disabling-of-EAX-support-in-the-registr.patch
	(
		echo '+    { "Sebastian Lackner", "dsound: Apply filters before sound is multiplied to speakers.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add EAX v1 constants and structs.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Report that we support EAX.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add EAX propset stubs.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add EAX presets.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Support getting and setting EAX properties.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Support getting and setting EAX buffer properties.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add EAX init and free stubs.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Feed data through EAX function.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Allocate EAX delay lines.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add EAX VerbPass stub.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Implement EAX lowpass filter.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Add delay line EAX functions.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Implement EAX early reflections.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Implement EAX decorrelator.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Implement EAX late reverb.", 1 },';
		echo '+    { "Mark Harmstone", "dsound: Implement EAX late all-pass filter.", 1 },';
		echo '+    { "Sebastian Lackner", "dsound: Various improvements to EAX support.", 1 },';
		echo '+    { "Sebastian Lackner", "dsound: Allow disabling of EAX support in the registry.", 1 },';
	) >> "$patchlist"
fi

# Patchset dwrite-VDMX
# |
# | Modified files:
# |   *	dlls/dwrite/opentype.c
# |
if test "$enable_dwrite_VDMX" -eq 1; then
	patch_apply dwrite-VDMX/0001-dwrite-Avoid-dereferencing-NULL-pointer-for-fonts-wi.patch
	(
		echo '+    { "Sebastian Lackner", "dwrite: Avoid dereferencing NULL pointer for fonts without VDMX.", 1 },';
	) >> "$patchlist"
fi

# Patchset dxgi-GetDesc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31998] Avseq crashes when multisampling is enabled
# |
# | Modified files:
# |   *	dlls/dxgi/output.c
# |
if test "$enable_dxgi_GetDesc" -eq 1; then
	patch_apply dxgi-GetDesc/0001-dxgi-Implement-IDXGIOutput-GetDesc.patch
	(
		echo '+    { "Michael Müller", "dxgi: Implement IDXGIOutput::GetDesc.", 1 },';
	) >> "$patchlist"
fi

# Patchset winecfg-Staging
# |
# | Modified files:
# |   *	programs/winecfg/Makefile.in, programs/winecfg/main.c, programs/winecfg/resource.h, programs/winecfg/staging.c,
# | 	programs/winecfg/winecfg.h, programs/winecfg/winecfg.rc
# |
if test "$enable_winecfg_Staging" -eq 1; then
	patch_apply winecfg-Staging/0001-winecfg-Add-staging-tab-for-CSMT.patch
	patch_apply winecfg-Staging/0002-winecfg-Add-checkbox-to-enable-disable-vaapi-GPU-dec.patch
	patch_apply winecfg-Staging/0003-winecfg-Add-checkbox-to-enable-disable-EAX-support.patch
	patch_apply winecfg-Staging/0004-winecfg-Add-checkbox-to-enable-disable-HideWineExpor.patch
	(
		echo '+    { "Michael Müller", "winecfg: Add staging tab for CSMT.", 1 },';
		echo '+    { "Sebastian Lackner", "winecfg: Add checkbox to enable/disable vaapi GPU decoder.", 1 },';
		echo '+    { "Mark Harmstone", "winecfg: Add checkbox to enable/disable EAX support.", 1 },';
		echo '+    { "Sebastian Lackner", "winecfg: Add checkbox to enable/disable HideWineExports registry key.", 1 },';
	) >> "$patchlist"
fi

# Patchset dxva2-Video_Decoder
# |
# | Modified files:
# |   *	configure.ac, dlls/dxva2/Makefile.in, dlls/dxva2/backend.idl, dlls/dxva2/devicemanager.c, dlls/dxva2/dxva2_private.h,
# | 	dlls/dxva2/genericdecoder.c, dlls/dxva2/main.c, dlls/dxva2/softwareprocessor.c, dlls/dxva2/tests/Makefile.in,
# | 	dlls/dxva2/tests/dxva2.c, dlls/dxva2/vaapi-h264.c, dlls/dxva2/vaapi-mpeg2.c, dlls/dxva2/vaapi.c,
# | 	dlls/dxva2/videoservices.c, include/Makefile.in, include/dxva.h, include/dxva2api.idl
# |
if test "$enable_dxva2_Video_Decoder" -eq 1; then
	patch_apply dxva2-Video_Decoder/0001-dxva2-Implement-semi-stub-for-Direct3DDeviceManager9.patch
	patch_apply dxva2-Video_Decoder/0002-dxva2-Implement-stubbed-interfaces-for-IDirectXVideo.patch
	patch_apply dxva2-Video_Decoder/0003-include-Fix-an-invalid-UUID-in-dxva2api.idl.patch
	patch_apply dxva2-Video_Decoder/0004-dxva2-Implement-stubbed-DirectX-Software-VideoProces.patch
	patch_apply dxva2-Video_Decoder/0005-include-Add-dxva.h-header-file.patch
	patch_apply dxva2-Video_Decoder/0006-dxva2-tests-Add-tests-for-dxva2-decoder.patch
	patch_apply dxva2-Video_Decoder/0007-dxva2-Initial-implementation-of-MPEG2-decoder-using-.patch
	patch_apply dxva2-Video_Decoder/0008-dxva2-Implement-h264-decoder.patch
	patch_apply dxva2-Video_Decoder/0009-dxva2-Add-DRM-mode-for-vaapi.patch
	patch_apply dxva2-Video_Decoder/0010-dxva2-Fill-h264-luma-and-chroma-weights-offsets-with.patch
	patch_apply dxva2-Video_Decoder/0011-dxva2-Always-destroy-buffers-when-calling-vaRenderPi.patch
	(
		echo '+    { "Sebastian Lackner", "dxva2: Implement semi-stub for Direct3DDeviceManager9 interface.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Implement stubbed interfaces for IDirectXVideo{Acceleration,Decoder,Processor}Service.", 1 },';
		echo '+    { "Michael Müller", "include: Fix an invalid UUID in dxva2api.idl.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Implement stubbed DirectX Software VideoProcessor interface.", 1 },';
		echo '+    { "Michael Müller", "include: Add dxva.h header file.", 1 },';
		echo '+    { "Michael Müller", "dxva2/tests: Add tests for dxva2 decoder.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Initial implementation of MPEG2 decoder using vaapi backend.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Implement h264 decoder.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Add DRM mode for vaapi.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Fill h264 luma and chroma weights / offsets with default values in case they are not specified.", 1 },';
		echo '+    { "Michael Müller", "dxva2: Always destroy buffers when calling vaRenderPicture.", 1 },';
	) >> "$patchlist"
fi

# Patchset fltmgr-Stub_SYS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23583] Add stub fltmgr.sys (filter manager driver)
# |
# | Modified files:
# |   *	configure.ac, dlls/fltmgr.sys/Makefile.in, dlls/fltmgr.sys/fltmgr.sys.spec, dlls/fltmgr.sys/main.c, loader/wine.inf.in
# |
if test "$enable_fltmgr_Stub_SYS" -eq 1; then
	patch_apply fltmgr-Stub_SYS/0001-fltmgr.sys-add-stub-dll-try-3.patch
	(
		echo '+    { "Austin English", "fltmgr.sys: add stub dll.", 4 },';
	) >> "$patchlist"
fi

# Patchset fonts-Missing_Fonts
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32323] Implement an Arial replacement font
# |   *	[#13829] Implement a Microsoft Yahei replacement font
# |
# | Modified files:
# |   *	COPYING.arial, COPYING.cour, COPYING.msyh, LICENSE, fonts/Makefile.in, fonts/arial.sfd, fonts/arial.ttf, fonts/cour.sfd,
# | 	fonts/cour.ttf, fonts/msyh.sfd, fonts/msyh.ttf
# |
if test "$enable_fonts_Missing_Fonts" -eq 1; then
	patch_apply fonts-Missing_Fonts/0001-fonts-Add-Liberation-Sans-as-an-Arial-replacement.patch
	patch_apply fonts-Missing_Fonts/0002-fonts-Add-WenQuanYi-Micro-Hei-as-a-Microsoft-Yahei-r.patch
	patch_apply fonts-Missing_Fonts/0003-fonts-Add-Courier-Prime-as-a-Courier-New-replacement.patch
	patch_apply fonts-Missing_Fonts/0004-Add-licenses-for-fonts-as-separate-files.patch
	(
		echo '+    { "Torsten Kurbad", "fonts: Add Liberation Sans as an Arial replacement.", 2 },';
		echo '+    { "Erich E. Hoover", "fonts: Add WenQuanYi Micro Hei as a Microsoft Yahei replacement.", 1 },';
		echo '+    { "Erich E. Hoover", "fonts: Add Courier Prime as a Courier New replacement.", 1 },';
		echo '+    { "Michael Müller", "Add licenses for fonts as separate files.", 1 },';
	) >> "$patchlist"
fi

# Patchset gdi32-Default_Palette
# |
# | Modified files:
# |   *	dlls/gdi32/palette.c, dlls/gdi32/tests/palette.c
# |
if test "$enable_gdi32_Default_Palette" -eq 1; then
	patch_apply gdi32-Default_Palette/0001-gdi32-Return-default-palette-entries-from-GetSystemP.patch
	(
		echo '+    { "Anton Baskanov", "gdi32: Return default palette entries from GetSystemPaletteEntries for non-palette-based devices.", 1 },';
	) >> "$patchlist"
fi

# Patchset gdi32-MaxPixelFormats
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#6176] wglDescribePixelFormat should return max index for NULL descriptor
# |
# | Modified files:
# |   *	dlls/gdi32/dibdrv/opengl.c
# |
if test "$enable_gdi32_MaxPixelFormats" -eq 1; then
	patch_apply gdi32-MaxPixelFormats/0001-gdi32-Return-maximum-number-of-pixel-formats-when-NU.patch
	(
		echo '+    { "Sebastian Lackner", "gdi32: Return maximum number of pixel formats when NULL pointer is passed to wglDescribePixelFormat.", 1 },';
	) >> "$patchlist"
fi

# Patchset gdi32-MultiMonitor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34978] Multiple applications need EnumDisplayDevicesW implementation
# |   *	[#37709] GetMonitorInfo returns the same name for all monitors
# |
# | Modified files:
# |   *	dlls/gdi32/driver.c, dlls/user32/misc.c, dlls/winex11.drv/xinerama.c
# |
if test "$enable_gdi32_MultiMonitor" -eq 1; then
	patch_apply gdi32-MultiMonitor/0001-gdi32-Also-accept-.-DISPLAY-n-devices-names-with-n-o.patch
	patch_apply gdi32-MultiMonitor/0002-winex11-Make-GetMonitorInfo-give-a-different-device-.patch
	patch_apply gdi32-MultiMonitor/0003-user32-Implement-EnumDisplayDevicesW-based-on-EnumDi.patch
	(
		echo '+    { "Ken Thomases", "gdi32: Also accept \"\\\\\\\\.\\\\DISPLAY<n>\" devices names with <n> other than 1 as display devices.", 1 },';
		echo '+    { "Ken Thomases", "winex11: Make GetMonitorInfo() give a different device name (\\\\.\\\\DISPLAY<n>) to each monitor.", 1 },';
		echo '+    { "Ken Thomases", "user32: Implement EnumDisplayDevicesW() based on EnumDisplayMonitors() and GetMonitorInfoW().", 1 },';
	) >> "$patchlist"
fi

# Patchset gdiplus-GIF_Encoder
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34356] Forward GIF encoder requests to windowscodecs
# |
# | Modified files:
# |   *	dlls/gdiplus/image.c, dlls/gdiplus/tests/image.c
# |
if test "$enable_gdiplus_GIF_Encoder" -eq 1; then
	patch_apply gdiplus-GIF_Encoder/0001-gdiplus-Forward-GIF-encoder-requests-to-windowscodec.patch
	(
		echo '+    { "Dmitry Timoshkov", "gdiplus: Forward GIF encoder requests to windowscodecs.", 1 },';
	) >> "$patchlist"
fi

# Patchset gdiplus-GdipCreateEffect
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32163] Add stub for gdiplus.GdipCreateEffect
# |
# | Modified files:
# |   *	dlls/gdiplus/gdiplus.spec, dlls/gdiplus/image.c
# |
if test "$enable_gdiplus_GdipCreateEffect" -eq 1; then
	patch_apply gdiplus-GdipCreateEffect/0001-gdiplus-Add-stub-for-GdipCreateEffect.patch
	(
		echo '+    { "David Hedberg", "gdiplus: Add stub for GdipCreateEffect.", 1 },';
	) >> "$patchlist"
fi

# Patchset imagehlp-BindImageEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#3591] Support for BindImageEx
# |
# | Modified files:
# |   *	dlls/imagehlp/modify.c, dlls/imagehlp/tests/image.c
# |
if test "$enable_imagehlp_BindImageEx" -eq 1; then
	patch_apply imagehlp-BindImageEx/0001-imagehlp-Implement-parts-of-BindImageEx-to-make-free.patch
	(
		echo '+    { "Bernhard Reiter", "imagehlp: Implement parts of BindImageEx to make freezing Python scripts work.", 1 },';
	) >> "$patchlist"
fi

# Patchset imagehlp-ImageLoad
# |
# | Modified files:
# |   *	dlls/imagehlp/tests/image.c
# |
if test "$enable_imagehlp_ImageLoad" -eq 1; then
	patch_apply imagehlp-ImageLoad/0001-imagehlp-tests-Add-tests-for-ImageLoad-ImageUnload-G.patch
	patch_apply imagehlp-ImageLoad/0002-imagehlp-tests-msvc-compatibility-fixes.patch
	(
		echo '+    { "Mark Jansen", "imagehlp/tests: Add tests for ImageLoad, ImageUnload, GetImageUnusedHeaderBytes.", 1 },';
		echo '+    { "Mark Jansen", "imagehlp/tests: msvc compatibility fixes.", 1 },';
	) >> "$patchlist"
fi

# Patchset inetcpl-Default_Home
# |
# | Modified files:
# |   *	dlls/inetcpl.cpl/general.c
# |
if test "$enable_inetcpl_Default_Home" -eq 1; then
	patch_apply inetcpl-Default_Home/0001-inetcpl-Implement-default-page-button.patch
	(
		echo '+    { "Jared Smudde", "inetcpl: Implement default page button.", 1 },';
	) >> "$patchlist"
fi

# Patchset iphlpapi-System_Ping
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8332] Fallback to system ping command when CAP_NET_RAW is not available
# |
# | Modified files:
# |   *	dlls/iphlpapi/icmp.c
# |
if test "$enable_iphlpapi_System_Ping" -eq 1; then
	patch_apply iphlpapi-System_Ping/0001-iphlpapi-Fallback-to-system-ping-when-ICMP-permissio.patch
	(
		echo '+    { "Sebastian Lackner", "iphlpapi: Fallback to system ping when ICMP permissions are not present.", 1 },';
	) >> "$patchlist"
fi

# Patchset iphlpapi-TCP_Table
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34372] Support for AllocateAndGetTcpExTableFromStack
# |
# | Modified files:
# |   *	dlls/iphlpapi/iphlpapi.spec, dlls/iphlpapi/ipstats.c, dlls/iphlpapi/ipstats.h
# |
if test "$enable_iphlpapi_TCP_Table" -eq 1; then
	patch_apply iphlpapi-TCP_Table/0001-iphlpapi-Implement-AllocateAndGetTcpExTableFromStack.patch
	(
		echo '+    { "Erich E. Hoover", "iphlpapi: Implement AllocateAndGetTcpExTableFromStack.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-CompareStringEx
# |
# | Modified files:
# |   *	dlls/kernel32/locale.c
# |
if test "$enable_kernel32_CompareStringEx" -eq 1; then
	patch_apply kernel32-CompareStringEx/0001-kernel32-Silence-repeated-CompareStringEx-FIXME.patch
	(
		echo '+    { "Sebastian Lackner", "kernel32: Silence repeated CompareStringEx FIXME.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-File_Permissions
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/ntdll/tests/file.c, server/fd.c
# |
if test "$enable_server_File_Permissions" -eq 1; then
	patch_apply server-File_Permissions/0001-server-Allow-to-open-files-without-any-permission-bi.patch
	patch_apply server-File_Permissions/0002-server-When-creating-new-directories-temporarily-giv.patch
	patch_apply server-File_Permissions/0003-advapi32-tests-Add-tests-for-ACL-inheritance-in-Crea.patch
	patch_apply server-File_Permissions/0004-advapi32-tests-Add-ACL-inheritance-tests-for-creatin.patch
	patch_apply server-File_Permissions/0005-ntdll-tests-Added-tests-for-open-behaviour-on-readon.patch
	patch_apply server-File_Permissions/0006-server-FILE_WRITE_ATTRIBUTES-should-succeed-for-read.patch
	(
		echo '+    { "Sebastian Lackner", "server: Allow to open files without any permission bits.", 2 },';
		echo '+    { "Sebastian Lackner", "server: When creating new directories temporarily give read-permissions until they are opened.", 1 },';
		echo '+    { "Sebastian Lackner", "advapi32/tests: Add tests for ACL inheritance in CreateDirectoryA.", 1 },';
		echo '+    { "Sebastian Lackner", "advapi32/tests: Add ACL inheritance tests for creating subdirectories with NtCreateFile.", 1 },';
		echo '+    { "Qian Hong", "ntdll/tests: Added tests for open behaviour on readonly files.", 1 },';
		echo '+    { "Sebastian Lackner", "server: FILE_WRITE_ATTRIBUTES should succeed for readonly files.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-FileDispositionInformation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30397] Support for NtSetInformationFile class FileDispositionInformation
# |   *	[#30399] Support for NtSetInformationFile class FileRenameInformation
# |
# | Modified files:
# |   *	dlls/kernel32/tests/file.c, dlls/ntdll/file.c, dlls/ntdll/tests/file.c, include/winternl.h, server/fd.c, server/file.c,
# | 	server/file.h, server/protocol.def
# |
if test "$enable_ntdll_FileDispositionInformation" -eq 1; then
	patch_apply ntdll-FileDispositionInformation/0001-server-Keep-a-pointer-to-parent-s-fd-unix_name-in-th.patch
	patch_apply ntdll-FileDispositionInformation/0002-server-Add-support-for-setting-file-disposition-info.patch
	patch_apply ntdll-FileDispositionInformation/0003-server-Do-not-permit-FileDispositionInformation-to-d.patch
	patch_apply ntdll-FileDispositionInformation/0004-ntdll-tests-Added-tests-to-set-disposition-on-file-w.patch
	patch_apply ntdll-FileDispositionInformation/0005-server-Do-not-allow-to-set-disposition-on-file-which.patch
	patch_apply ntdll-FileDispositionInformation/0006-ntdll-tests-Add-tests-for-FileRenameInformation.patch
	patch_apply ntdll-FileDispositionInformation/0007-ntdll-tests-Add-additional-tests-for-FileRenameInfor.patch
	patch_apply ntdll-FileDispositionInformation/0008-ntdll-tests-Add-tests-for-FileRenameInformation-with.patch
	patch_apply ntdll-FileDispositionInformation/0009-ntdll-Implement-FileRenameInformation-support.patch
	patch_apply ntdll-FileDispositionInformation/0010-server-When-combining-root-and-name-make-sure-there-.patch
	patch_apply ntdll-FileDispositionInformation/0011-include-Add-declaration-for-FILE_LINK_INFORMATION.patch
	patch_apply ntdll-FileDispositionInformation/0012-ntdll-tests-Add-tests-for-FileLinkInformation-class.patch
	patch_apply ntdll-FileDispositionInformation/0013-server-Implement-support-for-FileLinkInformation-cla.patch
	(
		echo '+    { "Dmitry Timoshkov", "server: Keep a pointer to parent'\''s fd unix_name in the closed_fd structure.", 1 },';
		echo '+    { "Dmitry Timoshkov", "server: Add support for setting file disposition information.", 1 },';
		echo '+    { "Erich E. Hoover", "server: Do not permit FileDispositionInformation to delete a file without write access.", 1 },';
		echo '+    { "Qian Hong", "ntdll/tests: Added tests to set disposition on file which is mapped to memory.", 1 },';
		echo '+    { "Qian Hong", "server: Do not allow to set disposition on file which has a file mapping.", 1 },';
		echo '+    { "Jianqiu Zhang", "ntdll/tests: Add tests for FileRenameInformation.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll/tests: Add additional tests for FileRenameInformation.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll/tests: Add tests for FileRenameInformation with nonzero RootDir.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Implement FileRenameInformation support.", 1 },';
		echo '+    { "Sebastian Lackner", "server: When combining root and name, make sure there is only one slash.", 2 },';
		echo '+    { "Zhaonan Liang", "include: Add declaration for FILE_LINK_INFORMATION.", 1 },';
		echo '+    { "Qian Hong", "ntdll/tests: Add tests for FileLinkInformation class.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Implement support for FileLinkInformation class in NtSetInformationFile.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-SetFileInformationByHandle
# |
# | Modified files:
# |   *	include/winbase.h
# |
if test "$enable_kernel32_SetFileInformationByHandle" -eq 1; then
	patch_apply kernel32-SetFileInformationByHandle/0001-include-Declare-a-couple-more-file-information-class.patch
	(
		echo '+    { "Michael Müller", "include: Declare a couple more file information class structures.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-CopyFileEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22692] Add support for CopyFileEx progress callback
# |   *	[#22690] Allow to cancel a file operation via progress callback
# |
# | Modified files:
# |   *	dlls/kernel32/path.c, dlls/kernel32/tests/file.c
# |
if test "$enable_kernel32_CopyFileEx" -eq 1; then
	patch_apply kernel32-CopyFileEx/0001-kernel32-Add-support-for-progress-callback-in-CopyFi.patch
	(
		echo '+    { "Michael Müller", "kernel32: Add support for progress callback in CopyFileEx.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-GetFinalPathNameByHandle
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34851] Support for GetFinalPathNameByHandle
# |
# | Modified files:
# |   *	dlls/api-ms-win-core-file-l1-1-0/api-ms-win-core-file-l1-1-0.spec, dlls/api-ms-win-core-file-l1-2-0/api-ms-win-core-
# | 	file-l1-2-0.spec, dlls/kernel32/file.c, dlls/kernel32/kernel32.spec
# |
if test "$enable_kernel32_GetFinalPathNameByHandle" -eq 1; then
	patch_apply kernel32-GetFinalPathNameByHandle/0001-kernel32-Implement-GetFinalPathNameByHandle.patch
	(
		echo '+    { "Michael Müller", "kernel32: Implement GetFinalPathNameByHandle.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-GetLogicalProcessorInformationEx
# |
# | Modified files:
# |   *	dlls/kernel32/process.c
# |
if test "$enable_kernel32_GetLogicalProcessorInformationEx" -eq 1; then
	patch_apply kernel32-GetLogicalProcessorInformationEx/0001-kernel32-Make-GetLogicalProcessorInformationEx-a-stu.patch
	(
		echo '+    { "Sebastian Lackner", "kernel32: Make GetLogicalProcessorInformationEx a stub which returns TRUE.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-InsertMode
# |
# | Modified files:
# |   *	dlls/kernel32/editline.c
# |
if test "$enable_kernel32_InsertMode" -eq 1; then
	patch_apply kernel32-InsertMode/0001-kernel32-Set-console-InsertMode-immediately.patch
	(
		echo '+    { "Hugh McMaster", "kernel32: Set console InsertMode immediately.", 1 },';
	) >> "$patchlist"
fi

# Patchset rpcrt4-Pipe_Transport
# |
# | Modified files:
# |   *	dlls/rpcrt4/rpc_transport.c
# |
if test "$enable_rpcrt4_Pipe_Transport" -eq 1; then
	patch_apply rpcrt4-Pipe_Transport/0001-rpcrt4-Restore-original-error-code-when-ReadFile-fai.patch
	(
		echo '+    { "Sebastian Lackner", "rpcrt4: Restore original error code when ReadFile fails with ERROR_MORE_DATA.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-Named_Pipe
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#16550] Fix for ConnectNamedPort return value in overlapped mode
# |   *	[#17195] Support for named pipe message mode (Linux only)
# |
# | Modified files:
# |   *	dlls/kernel32/file.c, dlls/kernel32/sync.c, dlls/kernel32/tests/pipe.c, dlls/ntdll/file.c, include/winternl.h,
# | 	server/named_pipe.c, server/protocol.def, server/sock.c, server/sock.h
# |
if test "$enable_kernel32_Named_Pipe" -eq 1; then
	patch_apply kernel32-Named_Pipe/0001-kernel32-ConnectNamedPort-should-return-FALSE-and-se.patch
	patch_apply kernel32-Named_Pipe/0002-kernel32-tests-Add-tests-for-PeekNamedPipe-with-part.patch
	patch_apply kernel32-Named_Pipe/0003-kernel32-tests-Add-tests-for-sending-and-receiving-l.patch
	patch_apply kernel32-Named_Pipe/0004-kernel32-tests-Add-tests-for-closing-named-pipes.patch
	patch_apply kernel32-Named_Pipe/0005-server-Show-warning-if-message-mode-is-not-supported.patch
	patch_apply kernel32-Named_Pipe/0006-ntdll-Unify-similar-code-in-NtReadFile-and-FILE_Asyn.patch
	patch_apply kernel32-Named_Pipe/0007-ntdll-Move-logic-to-check-for-broken-pipe-into-a-sep.patch
	patch_apply kernel32-Named_Pipe/0008-ntdll-Unify-similar-code-in-NtWriteFile-and-FILE_Asy.patch
	patch_apply kernel32-Named_Pipe/0009-server-Use-SOCK_SEQPACKET-socket-in-combination-with.patch
	patch_apply kernel32-Named_Pipe/0010-ntdll-Add-handling-for-partially-received-messages-i.patch
	patch_apply kernel32-Named_Pipe/0011-kernel32-tests-Add-more-tests-with-overlapped-IO-and.patch
	patch_apply kernel32-Named_Pipe/0012-ntdll-Fix-some-tests-for-overlapped-partial-reads.patch
	patch_apply kernel32-Named_Pipe/0013-kernel32-tests-Test-sending-peeking-and-receiving-an.patch
	patch_apply kernel32-Named_Pipe/0014-ntdll-Add-support-for-nonblocking-pipes.patch
	patch_apply kernel32-Named_Pipe/0015-kernel32-tests-Add-tests-for-PIPE_NOWAIT-in-message-.patch
	patch_apply kernel32-Named_Pipe/0016-ntdll-Allow-to-set-PIPE_NOWAIT-on-byte-mode-pipes.patch
	patch_apply kernel32-Named_Pipe/0017-kernel32-tests-Add-additional-tests-for-PIPE_NOWAIT-.patch
	patch_apply kernel32-Named_Pipe/0018-ntdll-Improve-ReadDataAvailable-handling-in-FilePipe.patch
	patch_apply kernel32-Named_Pipe/0019-ntdll-Set-NamedPipeState-to-FILE_PIPE_CLOSING_STATE-.patch
	patch_apply kernel32-Named_Pipe/0020-kernel32-tests-Add-tests-for-behaviour-of-WriteFile-.patch
	patch_apply kernel32-Named_Pipe/0021-server-Return-correct-error-codes-for-NtWriteFile-wh.patch
	patch_apply kernel32-Named_Pipe/0022-ntdll-Pre-cache-file-descriptors-after-opening-a-fil.patch
	patch_apply kernel32-Named_Pipe/0023-server-Fix-wineserver-crash-when-pipe-server-object-.patch
	(
		echo '+    { "Dan Kegel", "kernel32: ConnectNamedPort should return FALSE and set ERROR_PIPE_CONNECTED on success in overlapped mode.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add tests for PeekNamedPipe with partial received messages.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add tests for sending and receiving large messages.", 1 },';
		echo '+    { "Adam Martinson", "kernel32/tests: Add tests for closing named pipes.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Show warning if message mode is not supported.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Unify similar code in NtReadFile and FILE_AsyncReadService.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Move logic to check for broken pipe into a separate function.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Unify similar code in NtWriteFile and FILE_AsyncWriteService.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Use SOCK_SEQPACKET socket in combination with SO_PEEK_OFF to implement message mode on Unix.", 6 },';
		echo '+    { "Sebastian Lackner", "ntdll: Add handling for partially received messages in NtReadFile.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add more tests with overlapped IO and partial reads from named pipes.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Fix some tests for overlapped partial reads.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Test sending, peeking and receiving an empty message.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Add support for nonblocking pipes.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add tests for PIPE_NOWAIT in message mode.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Allow to set PIPE_NOWAIT on byte-mode pipes.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add additional tests for PIPE_NOWAIT in overlapped mode.", 1 },';
		echo '+    { "Qian Hong", "ntdll: Improve ReadDataAvailable handling in FilePipeLocalInformation class support.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Set NamedPipeState to FILE_PIPE_CLOSING_STATE on broken pipe in NtQueryInformationFile.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add tests for behaviour of WriteFile on closed pipe.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Return correct error codes for NtWriteFile when pipes are closed without disconnecting.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Pre-cache file descriptors after opening a file.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Fix wineserver crash when pipe server object is destroyed before client.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-NeedCurrentDirectoryForExePath
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23934] CreateProcess does not prioritize the working directory over the system search path
# |
# | Modified files:
# |   *	dlls/kernel32/path.c, dlls/kernel32/process.c, dlls/kernel32/tests/path.c
# |
if test "$enable_kernel32_NeedCurrentDirectoryForExePath" -eq 1; then
	patch_apply kernel32-NeedCurrentDirectoryForExePath/0001-kernel32-Add-SearchPath-test-demonstrating-the-prior.patch
	patch_apply kernel32-NeedCurrentDirectoryForExePath/0002-kernel32-NeedCurrentDirectoryForExePath-does-not-use.patch
	patch_apply kernel32-NeedCurrentDirectoryForExePath/0003-kernel32-Consider-the-working-directory-first-when-l.patch
	(
		echo '+    { "Erich E. Hoover", "kernel32: Add SearchPath test demonstrating the priority of the working directory.", 1 },';
		echo '+    { "Erich E. Hoover", "kernel32: NeedCurrentDirectoryForExePath does not use the registry.", 1 },';
		echo '+    { "Erich E. Hoover", "kernel32: Consider the working directory first when launching executables with CreateProcess.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-Profile
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8036] Fix handling of empty section and key name for profile files.
# |   *	[#18099] Super Mario 3: Mario Forever fails to load keyboard mapping from profile files.
# |
# | Modified files:
# |   *	dlls/kernel32/profile.c, dlls/kernel32/tests/profile.c
# |
if test "$enable_kernel32_Profile" -eq 1; then
	patch_apply kernel32-Profile/0001-kernel32-Allow-empty-profile-section-and-key-name-st.patch
	(
		echo '+    { "Claudio Fontana", "kernel32: Allow empty profile section and key name strings.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-TimezoneInformation_Registry
# |
# | Modified files:
# |   *	dlls/kernel32/kernel_main.c, dlls/kernel32/kernel_private.h, dlls/kernel32/time.c
# |
if test "$enable_kernel32_TimezoneInformation_Registry" -eq 1; then
	patch_apply kernel32-TimezoneInformation_Registry/0001-kernel32-Init-TimezoneInformation-registry.patch
	(
		echo '+    { "Qian Hong", "kernel32: Init TimezoneInformation registry.", 1 },';
	) >> "$patchlist"
fi

# Patchset kernel32-VerifyVersionInfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36143] Fix condition mask handling in RtlVerifyVersionInfo
# |
# | Modified files:
# |   *	dlls/kernel32/tests/version.c, dlls/ntdll/version.c
# |
if test "$enable_kernel32_VerifyVersionInfo" -eq 1; then
	patch_apply kernel32-VerifyVersionInfo/0001-kernel32-tests-Add-additional-tests-for-condition-ma.patch
	patch_apply kernel32-VerifyVersionInfo/0002-ntdll-Fix-condition-mask-handling-in-RtlVerifyVersio.patch
	(
		echo '+    { "Sebastian Lackner", "kernel32/tests: Add additional tests for condition mask of VerifyVersionInfoA.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Fix condition mask handling in RtlVerifyVersionInfo.", 1 },';
	) >> "$patchlist"
fi

# Patchset libs-Debug_Channel
# |
# | Modified files:
# |   *	libs/wine/debug.c
# |
if test "$enable_libs_Debug_Channel" -eq 1; then
	patch_apply libs-Debug_Channel/0001-libwine-Add-process-specific-debug-channels.patch
	(
		echo '+    { "Michael Müller", "libwine: Add process specific debug channels.", 1 },';
	) >> "$patchlist"
fi

# Patchset libs-Unicode_Collation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#10767] Fix comparison of punctuation characters in lstrcmp
# |   *	[#32490] Graphical issues in Inquisitor
# |
# | Modified files:
# |   *	dlls/kernel32/tests/locale.c, libs/wine/collation.c
# |
if test "$enable_libs_Unicode_Collation" -eq 1; then
	patch_apply libs-Unicode_Collation/0001-libs-Fix-most-problems-with-CompareString.patch
	(
		echo '+    { "Dmitry Timoshkov", "libs: Fix most problems with CompareString.", 1 },';
	) >> "$patchlist"
fi

# Patchset makedep-PARENTSPEC
# |
# | Modified files:
# |   *	tools/makedep.c
# |
if test "$enable_makedep_PARENTSPEC" -eq 1; then
	patch_apply makedep-PARENTSPEC/0001-makedep-Add-support-for-PARENTSPEC-Makefile-variable.patch
	(
		echo '+    { "Sebastian Lackner", "makedep: Add support for PARENTSPEC Makefile variable.", 1 },';
	) >> "$patchlist"
fi

# Patchset mfplat-MFTRegister
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37811] Add implementation for mfplat.MFTRegister
# |
# | Modified files:
# |   *	dlls/mfplat/Makefile.in, dlls/mfplat/main.c, dlls/mfplat/mfplat.spec, include/mfapi.h, loader/wine.inf.in
# |
if test "$enable_mfplat_MFTRegister" -eq 1; then
	patch_apply mfplat-MFTRegister/0001-mfplat-Implement-MFTRegister.patch
	(
		echo '+    { "Michael Müller", "mfplat: Implement MFTRegister.", 1 },';
	) >> "$patchlist"
fi

# Patchset mmdevapi-AEV_Stubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35658] Improve stub for AEV_GetVolumeRange
# |
# | Modified files:
# |   *	dlls/mmdevapi/audiovolume.c
# |
if test "$enable_mmdevapi_AEV_Stubs" -eq 1; then
	patch_apply mmdevapi-AEV_Stubs/0001-mmdevapi-Improve-AEV_GetVolumeRange-stub.patch
	patch_apply mmdevapi-AEV_Stubs/0002-mmdevapi-Improve-AEV_SetMasterVolumeLevel-and-AEV_Ge.patch
	patch_apply mmdevapi-AEV_Stubs/0003-mmdevapi-Improve-AEV_SetMute-and-AEV_GetMute-stubs.patch
	(
		echo '+    { "Christian Costa", "mmdevapi: Improve AEV_GetVolumeRange stub.", 1 },';
		echo '+    { "Christian Costa", "mmdevapi: Improve AEV_SetMasterVolumeLevel and AEV_GetMasterVolumeLevel stubs.", 1 },';
		echo '+    { "Christian Costa", "mmdevapi: Improve AEV_SetMute and AEV_GetMute stubs.", 1 },';
	) >> "$patchlist"
fi

# Patchset mountmgr-DosDevices
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38235] Fix device paths in HKLM\SYSTEM\MountedDevices
# |
# | Modified files:
# |   *	dlls/mountmgr.sys/device.c, dlls/mountmgr.sys/mountmgr.c, dlls/mountmgr.sys/mountmgr.h
# |
if test "$enable_mountmgr_DosDevices" -eq 1; then
	patch_apply mountmgr-DosDevices/0001-mountmgr.sys-Write-usable-device-paths-into-HKLM-SYS.patch
	(
		echo '+    { "Michael Müller", "mountmgr.sys: Write usable device paths into HKLM\\\\SYSTEM\\\\MountedDevices.", 1 },';
	) >> "$patchlist"
fi

# Patchset mscoree-CorValidateImage
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38662] Implement mscoree._CorValidateImage for mono runtime
# |
# | Modified files:
# |   *	dlls/mscoree/mscoree_main.c
# |
if test "$enable_mscoree_CorValidateImage" -eq 1; then
	patch_apply mscoree-CorValidateImage/0001-mscoree-Implement-_CorValidateImage.patch
	(
		echo '+    { "Michael Müller", "mscoree: Implement _CorValidateImage.", 1 },';
	) >> "$patchlist"
fi

# Patchset msvcp90-basic_string_dtor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37358] FEAR 1 installer expects basic_string_wchar_dtor to return NULL
# |
# | Modified files:
# |   *	dlls/msvcp90/msvcp90.h, dlls/msvcp90/string.c, dlls/msvcp90/tests/string.c
# |
if test "$enable_msvcp90_basic_string_dtor" -eq 1; then
	patch_apply msvcp90-basic_string_dtor/0001-msvcp90-basic_string_wchar_dtor-needs-to-return-NULL.patch
	patch_apply msvcp90-basic_string_dtor/0002-msvcp90-tests-Add-tests-to-check-that-basic_string_w.patch
	(
		echo '+    { "Michael Müller", "msvcp90: basic_string_wchar_dtor needs to return NULL.", 1 },';
		echo '+    { "Michael Müller", "msvcp90/tests: Add tests to check that basic_string_wchar_dtor returns NULL.", 1 },';
	) >> "$patchlist"
fi

# Patchset msvcrt-Math_Precision
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37149] Calculate msvcrt exponential math operations with higher precision
# |
# | Modified files:
# |   *	dlls/msvcrt/math.c
# |
if test "$enable_msvcrt_Math_Precision" -eq 1; then
	patch_apply msvcrt-Math_Precision/0001-msvcrt-Calculate-sinh-cosh-exp-pow-with-higher-preci.patch
	(
		echo '+    { "Sebastian Lackner", "msvcrt: Calculate sinh/cosh/exp/pow with higher precision.", 2 },';
	) >> "$patchlist"
fi

# Patchset msvfw32-Image_Size
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#27595] Fix NULL dereference in ICSeqCompressFrameStart
# |
# | Modified files:
# |   *	dlls/msvfw32/msvideo_main.c
# |
if test "$enable_msvfw32_Image_Size" -eq 1; then
	patch_apply msvfw32-Image_Size/0001-msvfw32-Derive-image-size-from-input-image-to-avoid-.patch
	(
		echo '+    { "Bruno Jesus", "msvfw32: Derive image size from input image to avoid NULL dereference.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-APC_Performance
# |
# | Modified files:
# |   *	dlls/ntdll/file.c
# |
if test "$enable_ntdll_APC_Performance" -eq 1; then
	patch_apply ntdll-APC_Performance/0001-ntdll-Reuse-old-async-fileio-structures-if-possible.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Reuse old async fileio structures if possible.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-APC_Start_Process
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c
# |
if test "$enable_ntdll_APC_Start_Process" -eq 1; then
	patch_apply ntdll-APC_Start_Process/0001-ntdll-Process-APC-calls-before-starting-process.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Process APC calls before starting process.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Activation_Context
# |
# | Modified files:
# |   *	dlls/kernel32/tests/actctx.c, dlls/ntdll/actctx.c
# |
if test "$enable_ntdll_Activation_Context" -eq 1; then
	patch_apply ntdll-Activation_Context/0001-ntdll-Fix-return-value-for-missing-ACTIVATION_CONTEX.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Fix return value for missing ACTIVATION_CONTEXT_SECTION_ASSEMBLY_INFORMATION key.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-CLI_Images
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38661] Implement proper handling of CLI .NET images in Wine library loader
# |
# | Modified files:
# |   *	dlls/kernel32/process.c, dlls/ntdll/loader.c
# |
if test "$enable_ntdll_CLI_Images" -eq 1; then
	patch_apply ntdll-CLI_Images/0001-ntdll-Load-CLI-.NET-images-in-the-same-way-as-Window.patch
	(
		echo '+    { "Michael Müller", "ntdll: Load CLI/.NET images in the same way as Windows XP and above.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Coverity
# |
# | Modified files:
# |   *	dlls/ntdll/threadpool.c
# |
if test "$enable_ntdll_Coverity" -eq 1; then
	patch_apply ntdll-Coverity/0001-ntdll-Fix-incorrect-assignment-in-assert-statement-C.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Fix incorrect assignment in assert statement (Coverity).", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-DOS_Attributes
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#9158] Support for DOS hidden/system file attributes
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/directory.c, dlls/ntdll/file.c, dlls/ntdll/ntdll_misc.h, dlls/ntdll/tests/directory.c,
# | 	dlls/ntdll/tests/file.c, include/wine/port.h, libs/port/Makefile.in, libs/port/xattr.c
# |
if test "$enable_ntdll_DOS_Attributes" -eq 1; then
	patch_apply ntdll-DOS_Attributes/0001-ntdll-Implement-retrieving-DOS-attributes-in-NtQuery.patch
	patch_apply ntdll-DOS_Attributes/0002-ntdll-Implement-retrieving-DOS-attributes-in-NtQuery.patch
	patch_apply ntdll-DOS_Attributes/0003-ntdll-Implement-storing-DOS-attributes-in-NtSetInfor.patch
	patch_apply ntdll-DOS_Attributes/0004-ntdll-Implement-storing-DOS-attributes-in-NtCreateFi.patch
	patch_apply ntdll-DOS_Attributes/0005-libport-Add-support-for-Mac-OS-X-style-extended-attr.patch
	patch_apply ntdll-DOS_Attributes/0006-libport-Add-support-for-FreeBSD-style-extended-attri.patch
	patch_apply ntdll-DOS_Attributes/0007-ntdll-Perform-the-Unix-style-hidden-file-check-withi.patch
	(
		echo '+    { "Erich E. Hoover", "ntdll: Implement retrieving DOS attributes in NtQueryInformationFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Implement retrieving DOS attributes in NtQuery[Full]AttributesFile and NtQueryDirectoryFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Implement storing DOS attributes in NtSetInformationFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Implement storing DOS attributes in NtCreateFile.", 1 },';
		echo '+    { "Erich E. Hoover", "libport: Add support for Mac OS X style extended attributes.", 1 },';
		echo '+    { "Erich E. Hoover", "libport: Add support for FreeBSD style extended attributes.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Perform the Unix-style hidden file check within the unified file info grabbing routine.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-DeviceType_Systemroot
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36546] Return fake device type when systemroot is located on virtual disk
# |
# | Modified files:
# |   *	dlls/ntdll/file.c
# |
if test "$enable_ntdll_DeviceType_Systemroot" -eq 1; then
	patch_apply ntdll-DeviceType_Systemroot/0001-ntdll-Return-fake-device-type-when-systemroot-is-loc.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Return fake device type when systemroot is located on virtual disk.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-DllRedirects
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, dlls/ntdll/loadorder.c, dlls/ntdll/ntdll_misc.h
# |
if test "$enable_ntdll_DllRedirects" -eq 1; then
	patch_apply ntdll-DllRedirects/0001-ntdll-Move-logic-to-determine-loadorder-HKCU-app-key.patch
	patch_apply ntdll-DllRedirects/0002-ntdll-Move-logic-to-read-loadorder-registry-values-i.patch
	patch_apply ntdll-DllRedirects/0003-ntdll-Move-code-to-determine-module-basename-into-se.patch
	patch_apply ntdll-DllRedirects/0004-ntdll-Implement-get_redirect-function.patch
	patch_apply ntdll-DllRedirects/0005-ntdll-Implement-loader-redirection-scheme.patch
	(
		echo '+    { "Michael Müller", "ntdll: Move logic to determine loadorder HKCU/app key into separate functions.", 1 },';
		echo '+    { "Michael Müller", "ntdll: Move logic to read loadorder registry values into separate function.", 1 },';
		echo '+    { "Michael Müller", "ntdll: Move code to determine module basename into separate function.", 1 },';
		echo '+    { "Michael Müller", "ntdll: Implement get_redirect function.", 1 },';
		echo '+    { "Michael Müller", "ntdll: Implement loader redirection scheme.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Exception
# |
# | Modified files:
# |   *	dlls/kernel32/debugger.c, dlls/ntdll/om.c, dlls/ntdll/tests/exception.c
# |
if test "$enable_ntdll_Exception" -eq 1; then
	patch_apply ntdll-Exception/0001-ntdll-Throw-exception-if-invalid-handle-is-passed-to.patch
	patch_apply ntdll-Exception/0002-ntdll-OutputDebugString-should-throw-the-exception-a.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Throw exception if invalid handle is passed to NtClose and debugger enabled.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: OutputDebugString should throw the exception a second time, if a debugger is attached.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-FileFsFullSizeInformation
# |
# | Modified files:
# |   *	dlls/ntdll/file.c, dlls/ntdll/tests/file.c
# |
if test "$enable_ntdll_FileFsFullSizeInformation" -eq 1; then
	patch_apply ntdll-FileFsFullSizeInformation/0001-ntdll-Add-support-for-FileFsFullSizeInformation-clas.patch
	(
		echo '+    { "Jianqiu Zhang", "ntdll: Add support for FileFsFullSizeInformation class in NtQueryVolumeInformationFile.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-FileFsVolumeInformation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#21466] Add semi-stub for FileFsVolumeInformation information class
# |
# | Modified files:
# |   *	dlls/ntdll/file.c, dlls/ntdll/tests/file.c
# |
if test "$enable_ntdll_FileFsVolumeInformation" -eq 1; then
	patch_apply ntdll-FileFsVolumeInformation/0001-ntdll-Add-semi-stub-for-FileFsVolumeInformation-info.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Add semi-stub for FileFsVolumeInformation information class.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Fix_Alignment
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33162] Ensure NtProtectVirtualMemory and NtCreateSection are on separate pages
# |
# | Modified files:
# |   *	dlls/ntdll/virtual.c
# |
if test "$enable_ntdll_Fix_Alignment" -eq 1; then
	patch_apply ntdll-Fix_Alignment/0001-ntdll-Move-NtProtectVirtualMemory-and-NtCreateSectio.patch
	(
		echo '+    { "Michael Müller", "ntdll: Move NtProtectVirtualMemory and NtCreateSection to separate pages on x86.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Fix_Free
# |
# | Modified files:
# |   *	dlls/kernel32/path.c
# |
if test "$enable_ntdll_Fix_Free" -eq 1; then
	patch_apply ntdll-Fix_Free/0001-kernel32-Fix-leaking-directory-handle-in-RemoveDirec.patch
	(
		echo '+    { "Sebastian Lackner", "kernel32: Fix leaking directory handle in RemoveDirectoryW.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-FreeBSD_Directory
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35397] Use POSIX implementation to enumerate directory content on FreeBSD
# |
# | Modified files:
# |   *	dlls/ntdll/directory.c
# |
if test "$enable_ntdll_FreeBSD_Directory" -eq 1; then
	patch_apply ntdll-FreeBSD_Directory/0001-ntdll-Use-POSIX-implementation-to-enumerate-director.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Use POSIX implementation to enumerate directory content.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Heap_FreeLists
# |
# | Modified files:
# |   *	dlls/ntdll/heap.c
# |
if test "$enable_ntdll_Heap_FreeLists" -eq 1; then
	patch_apply ntdll-Heap_FreeLists/0001-ntdll-Improve-heap-allocation-performance-by-using-m.patch
	(
		echo '+    { "Steaphan Greene", "ntdll: Improve heap allocation performance by using more fine-grained free lists.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Hide_Wine_Exports
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38656] Add support for hiding wine version information from applications
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, dlls/ntdll/ntdll_misc.h
# |
if test "$enable_ntdll_Hide_Wine_Exports" -eq 1; then
	patch_apply ntdll-Hide_Wine_Exports/0001-ntdll-Add-support-for-hiding-wine-version-informatio.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Add support for hiding wine version information from applications.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-NtQueryEaFile
# |
# | Modified files:
# |   *	dlls/ntdll/file.c, dlls/ntdll/tests/file.c
# |
if test "$enable_ntdll_NtQueryEaFile" -eq 1; then
	patch_apply ntdll-NtQueryEaFile/0001-ntdll-Improve-stub-of-NtQueryEaFile.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Improve stub of NtQueryEaFile.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Junction_Points
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12401] Support for Junction Points
# |
# | Modified files:
# |   *	dlls/kernel32/path.c, dlls/kernel32/volume.c, dlls/ntdll/file.c, dlls/ntdll/tests/file.c, include/Makefile.in,
# | 	include/ntifs.h
# |
if test "$enable_ntdll_Junction_Points" -eq 1; then
	patch_apply ntdll-Junction_Points/0001-ntdll-Add-support-for-junction-point-creation.patch
	patch_apply ntdll-Junction_Points/0002-ntdll-Add-support-for-reading-junction-points.patch
	patch_apply ntdll-Junction_Points/0003-ntdll-Add-support-for-deleting-junction-points.patch
	patch_apply ntdll-Junction_Points/0004-ntdll-Add-a-test-for-junction-point-advertisement.patch
	patch_apply ntdll-Junction_Points/0005-kernel32-ntdll-Add-support-for-deleting-junction-poi.patch
	patch_apply ntdll-Junction_Points/0006-kernel32-Advertise-junction-point-support.patch
	patch_apply ntdll-Junction_Points/0007-ntdll-tests-Add-test-for-deleting-junction-point-tar.patch
	(
		echo '+    { "Erich E. Hoover", "ntdll: Add support for junction point creation.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Add support for reading junction points.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Add support for deleting junction points.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll: Add a test for junction point advertisement.", 1 },';
		echo '+    { "Erich E. Hoover", "kernel32,ntdll: Add support for deleting junction points with RemoveDirectory.", 1 },';
		echo '+    { "Erich E. Hoover", "kernel32: Advertise junction point support.", 1 },';
		echo '+    { "Erich E. Hoover", "ntdll/tests: Add test for deleting junction point target.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-NtQuerySection
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37338] Support for NtQuerySection
# |
# | Modified files:
# |   *	dlls/kernel32/tests/virtual.c, dlls/ntdll/nt.c, dlls/ntdll/ntdll.spec, dlls/ntdll/virtual.c,
# | 	dlls/ntoskrnl.exe/ntoskrnl.exe.spec, server/mapping.c, server/protocol.def
# |
if test "$enable_ntdll_NtQuerySection" -eq 1; then
	patch_apply ntdll-NtQuerySection/0001-ntdll-Implement-NtQuerySection.patch
	patch_apply ntdll-NtQuerySection/0002-kernel32-tests-Add-tests-for-NtQuerySection.patch
	(
		echo '+    { "Dmitry Timoshkov", "ntdll: Implement NtQuerySection.", 2 },';
		echo '+    { "Dmitry Timoshkov", "kernel32/tests: Add tests for NtQuerySection.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-NtSetLdtEntries
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26268] Add stub for NtSetLdtEntries/ZwSetLdtEntries
# |
# | Modified files:
# |   *	dlls/ntdll/nt.c, dlls/ntdll/ntdll.spec, include/ddk/wdm.h, include/winternl.h
# |
if test "$enable_ntdll_NtSetLdtEntries" -eq 1; then
	patch_apply ntdll-NtSetLdtEntries/0001-ntdll-add-NtSetLdtEntries-ZwSetLdtEntries-stub-try-2.patch
	(
		echo '+    { "Austin English", "ntdll: add NtSetLdtEntries/ZwSetLdtEntries stub.", 2 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Pipe_SpecialCharacters
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28995] Allow special characters in pipe names
# |
# | Modified files:
# |   *	dlls/kernel32/tests/pipe.c, dlls/ntdll/directory.c
# |
if test "$enable_ntdll_Pipe_SpecialCharacters" -eq 1; then
	patch_apply ntdll-Pipe_SpecialCharacters/0001-ntdll-Allow-special-characters-in-pipe-names.patch
	(
		echo '+    { "Michael Müller", "ntdll: Allow special characters in pipe names.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Revert_Security_Cookie
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38895] Revert patch causing crash in IrfanView when using the WebP plugin
# |
# | Modified files:
# |   *	dlls/ntdll/virtual.c
# |
if test "$enable_ntdll_Revert_Security_Cookie" -eq 1; then
	patch_apply ntdll-Revert_Security_Cookie/0001-Revert-ntdll-Randomize-security-cookie-when-availabl.patch
	(
		echo '+    { "Sebastian Lackner", "Revert \"ntdll: Randomize security cookie when available.\".", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-RtlIpStringToAddress
# |
# | Modified files:
# |   *	dlls/ntdll/tests/rtl.c
# |
if test "$enable_ntdll_RtlIpStringToAddress" -eq 1; then
	patch_apply ntdll-RtlIpStringToAddress/0001-ntdll-tests-Tests-for-RtlIpv6StringToAddress-try-6.patch
	patch_apply ntdll-RtlIpStringToAddress/0002-ntdll-tests-Tests-for-RtlIpv6StringToAddressEx-try-6.patch
	patch_apply ntdll-RtlIpStringToAddress/0003-ntdll-tests-Tests-for-RtlIpv4StringToAddressEx-try-5.patch
	patch_apply ntdll-RtlIpStringToAddress/0004-ntdll-tests-Add-tests-for-RtlIpv6AddressToString-and.patch
	(
		echo '+    { "Mark Jansen", "ntdll/tests: Tests for RtlIpv6StringToAddress.", 6 },';
		echo '+    { "Mark Jansen", "ntdll/tests: Tests for RtlIpv6StringToAddressEx.", 6 },';
		echo '+    { "Mark Jansen", "ntdll/tests: Tests for RtlIpv4StringToAddressEx (try 5, resend).", 1 },';
		echo '+    { "Mark Jansen", "ntdll/tests: Add tests for RtlIpv6AddressToString and RtlIpv6AddressToStringEx.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-ThreadTime
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#20230] Return correct values for GetThreadTimes function
# |
# | Modified files:
# |   *	dlls/ntdll/thread.c, server/protocol.def, server/thread.c
# |
if test "$enable_ntdll_ThreadTime" -eq 1; then
	patch_apply ntdll-ThreadTime/0001-ntdll-Return-correct-values-in-GetThreadTimes-for-al.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Return correct values in GetThreadTimes() for all threads.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-Threading
# |
# | Modified files:
# |   *	dlls/ntdll/process.c, dlls/ntdll/thread.c
# |
if test "$enable_ntdll_Threading" -eq 1; then
	patch_apply ntdll-Threading/0001-ntdll-Fix-race-condition-when-threads-are-killed-dur.patch
	patch_apply ntdll-Threading/0002-ntdll-Avoid-deadlock-by-using-_exit-in-NtTerminatePr.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Fix race-condition when threads are killed during shutdown.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Avoid deadlock by using _exit() in NtTerminateProcess.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-User_Shared_Data
# |
# | Modified files:
# |   *	dlls/ntdll/ntdll.spec, dlls/ntdll/thread.c, dlls/ntoskrnl.exe/instr.c
# |
if test "$enable_ntdll_User_Shared_Data" -eq 1; then
	patch_apply ntdll-User_Shared_Data/0001-ntdll-Move-code-to-update-user-shared-data-into-a-se.patch
	patch_apply ntdll-User_Shared_Data/0002-ntoskrnl-Update-USER_SHARED_DATA-before-accessing-me.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Move code to update user shared data into a separate function.", 1 },';
		echo '+    { "Sebastian Lackner", "ntoskrnl: Update USER_SHARED_DATA before accessing memory.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-WinSqm
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31971] ntdll is missing WinSqm[Start|End]Session implementation
# |
# | Modified files:
# |   *	dlls/ntdll/misc.c, dlls/ntdll/ntdll.spec, dlls/ntdll/tests/rtl.c
# |
if test "$enable_ntdll_WinSqm" -eq 1; then
	patch_apply ntdll-WinSqm/0001-ntdll-Add-stubs-for-WinSqmStartSession-WinSqmEndSess.patch
	(
		echo '+    { "Erich E. Hoover", "ntdll: Add stubs for WinSqmStartSession / WinSqmEndSession.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntdll-WriteWatches
# |
# | Modified files:
# |   *	dlls/ntdll/file.c
# |
if test "$enable_ntdll_WriteWatches" -eq 1; then
	patch_apply ntdll-WriteWatches/0001-ntdll-Avoid-race-conditions-with-write-watches-in-Nt.patch
	(
		echo '+    { "Dmitry Timoshkov", "ntdll: Avoid race-conditions with write watches in NtReadFile.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntoskrnl-DriverTest
# |
# | Modified files:
# |   *	aclocal.m4, configure.ac, dlls/ntoskrnl.exe/ntoskrnl.exe.spec, dlls/ntoskrnl.exe/tests/Makefile.in,
# | 	dlls/ntoskrnl.exe/tests/driver.sys/Makefile.in, dlls/ntoskrnl.exe/tests/driver.sys/driver.c,
# | 	dlls/ntoskrnl.exe/tests/driver.sys/driver.h, dlls/ntoskrnl.exe/tests/driver.sys/driver.sys.spec,
# | 	dlls/ntoskrnl.exe/tests/driver.sys/test.c, dlls/ntoskrnl.exe/tests/driver.sys/test.h,
# | 	dlls/ntoskrnl.exe/tests/driver.sys/util.h, dlls/ntoskrnl.exe/tests/ntoskrnl.c, include/wine/test.h,
# | 	tools/make_makefiles, tools/makedep.c
# |
if test "$enable_ntoskrnl_DriverTest" -eq 1; then
	patch_apply ntoskrnl-DriverTest/0001-ntoskrnl.exe-tests-Add-initial-driver-testing-framew.patch
	patch_apply ntoskrnl-DriverTest/0002-ntoskrnl.exe-tests-Add-kernel-compliant-test-functio.patch
	(
		echo '+    { "Sebastian Lackner", "ntoskrnl.exe/tests: Add initial driver testing framework and corresponding changes to Makefile system.", 2 },';
		echo '+    { "Michael Müller", "ntoskrnl.exe/tests: Add kernel compliant test functions.", 1 },';
	) >> "$patchlist"
fi

# Patchset ntoskrnl-Stubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38103] Process Hacker 2.x needs ntoskrnl.ProbeForRead
# |   *	[#21448] SecuROM 5.x media validation fails
# |
# | Modified files:
# |   *	dlls/ntoskrnl.exe/ntoskrnl.c, dlls/ntoskrnl.exe/ntoskrnl.exe.spec, include/ddk/wdm.h
# |
if test "$enable_ntoskrnl_Stubs" -eq 1; then
	patch_apply ntoskrnl-Stubs/0001-ntoskrnl.exe-add-KeWaitForMultipleObjects-stub.patch
	patch_apply ntoskrnl-Stubs/0002-ntoskrnl.exe-Add-stub-for-IoGetAttachedDeviceReferen.patch
	patch_apply ntoskrnl-Stubs/0003-ntoskrnl.exe-Add-stubs-for-ExAcquireFastMutexUnsafe-.patch
	patch_apply ntoskrnl-Stubs/0004-ntoskrnl.exe-Add-stubs-for-ObReferenceObjectByPointe.patch
	patch_apply ntoskrnl-Stubs/0005-ntoskrnl.exe-Add-stub-for-KeDelayExecutionThread.patch
	patch_apply ntoskrnl-Stubs/0006-ntoskrnl.exe-Improve-KeReleaseMutex-stub.patch
	patch_apply ntoskrnl-Stubs/0007-ntoskrnl.exe-Improve-KeInitializeSemaphore-stub.patch
	patch_apply ntoskrnl-Stubs/0008-ntoskrnl.exe-Improve-KeInitializeTimerEx-stub.patch
	patch_apply ntoskrnl-Stubs/0009-ntoskrnl.exe-Fix-IoReleaseCancelSpinLock-argument.patch
	patch_apply ntoskrnl-Stubs/0010-ntoskrnl.exe-Add-stub-for-ExAcquireResourceExclusive.patch
	patch_apply ntoskrnl-Stubs/0011-ntoskrnl.exe-Add-stub-for-ExReleaseResourceForThread.patch
	patch_apply ntoskrnl-Stubs/0012-ntoskrnl.exe-Add-stub-for-ExDeleteResourceLite.patch
	patch_apply ntoskrnl-Stubs/0013-ntoskrnl.exe-Implement-MmMapLockedPages-and-MmUnmapL.patch
	patch_apply ntoskrnl-Stubs/0014-ntoskrnl.exe-Implement-KeInitializeMutex.patch
	patch_apply ntoskrnl-Stubs/0015-ntoskrnl.exe-Add-stub-for-ProbeForRead.patch
	patch_apply ntoskrnl-Stubs/0016-ntoskrnl.exe-Add-stub-for-ProbeForWrite.patch
	patch_apply ntoskrnl-Stubs/0017-ntoskrnl.exe-Add-stub-for-PsRemoveLoadImageNotifyRou.patch
	patch_apply ntoskrnl-Stubs/0018-ntoskrnl.exe-Improve-IoGetDeviceObjectPointer-stub-t.patch
	(
		echo '+    { "Austin English", "ntoskrnl.exe: add KeWaitForMultipleObjects stub.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Add stub for IoGetAttachedDeviceReference.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Add stubs for ExAcquireFastMutexUnsafe and ExReleaseFastMutexUnsafe.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Add stubs for ObReferenceObjectByPointer and ObDereferenceObject.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Add stub for KeDelayExecutionThread.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Improve KeReleaseMutex stub.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Improve KeInitializeSemaphore stub.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Improve KeInitializeTimerEx stub.", 1 },';
		echo '+    { "Christian Costa", "ntoskrnl.exe: Fix IoReleaseCancelSpinLock argument.", 1 },';
		echo '+    { "Christian Costa", "ntoskrnl.exe: Add stub for ExAcquireResourceExclusiveLite.", 1 },';
		echo '+    { "Christian Costa", "ntoskrnl.exe: Add stub for ExReleaseResourceForThreadLite.", 1 },';
		echo '+    { "Christian Costa", "ntoskrnl.exe: Add stub for ExDeleteResourceLite.", 1 },';
		echo '+    { "Christian Costa", "ntoskrnl.exe: Implement MmMapLockedPages and MmUnmapLockedPages.", 1 },';
		echo '+    { "Alexander Morozov", "ntoskrnl.exe: Implement KeInitializeMutex.", 1 },';
		echo '+    { "Austin English", "ntoskrnl.exe: Add stub for ProbeForRead.", 1 },';
		echo '+    { "Sebastian Lackner", "ntoskrnl.exe: Add stub for ProbeForWrite.", 1 },';
		echo '+    { "Michael Müller", "ntoskrnl.exe: Add stub for PsRemoveLoadImageNotifyRoutine.", 1 },';
		echo '+    { "Erich E. Hoover", "ntoskrnl.exe: Improve IoGetDeviceObjectPointer stub to appease SecuROM 5.x.", 1 },';
	) >> "$patchlist"
fi

# Patchset nvcuda-CUDA_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37664] MediaCoder needs CUDA for video encoding
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuda/Makefile.in, dlls/nvcuda/internal.c, dlls/nvcuda/nvcuda.c, dlls/nvcuda/nvcuda.h,
# | 	dlls/nvcuda/nvcuda.rc, dlls/nvcuda/nvcuda.spec, dlls/nvcuda/tests/Makefile.in, dlls/nvcuda/tests/nvcuda.c,
# | 	include/Makefile.in, include/cuda.h
# |
if test "$enable_nvcuda_CUDA_Support" -eq 1; then
	patch_apply nvcuda-CUDA_Support/0001-include-Add-cuda.h.h.patch
	patch_apply nvcuda-CUDA_Support/0002-nvcuda-Add-stub-dll.patch
	patch_apply nvcuda-CUDA_Support/0003-nvcuda-First-implementation.patch
	patch_apply nvcuda-CUDA_Support/0004-nvcuda-Implement-new-functions-added-in-CUDA-6.5.patch
	patch_apply nvcuda-CUDA_Support/0005-nvcuda-Properly-wrap-undocumented-ContextStorage-int.patch
	patch_apply nvcuda-CUDA_Support/0006-nvcuda-Emulate-two-d3d9-initialization-functions.patch
	patch_apply nvcuda-CUDA_Support/0007-nvcuda-Properly-wrap-stream-callbacks-by-forwarding-.patch
	patch_apply nvcuda-CUDA_Support/0008-nvcuda-Add-support-for-CUDA-7.0.patch
	patch_apply nvcuda-CUDA_Support/0009-nvcuda-Implement-cuModuleLoad-wrapper-function.patch
	patch_apply nvcuda-CUDA_Support/0010-nvcuda-Search-for-dylib-library-on-Mac-OS-X.patch
	(
		echo '+    { "Sebastian Lackner", "include: Add cuda.h.h.", 1 },';
		echo '+    { "Sebastian Lackner", "nvcuda: Add stub dll.", 1 },';
		echo '+    { "Michael Müller", "nvcuda: First implementation.", 2 },';
		echo '+    { "Sebastian Lackner", "nvcuda: Implement new functions added in CUDA 6.5.", 1 },';
		echo '+    { "Michael Müller", "nvcuda: Properly wrap undocumented '\''ContextStorage'\'' interface and add tests.", 1 },';
		echo '+    { "Michael Müller", "nvcuda: Emulate two d3d9 initialization functions.", 1 },';
		echo '+    { "Sebastian Lackner", "nvcuda: Properly wrap stream callbacks by forwarding them to a worker thread.", 1 },';
		echo '+    { "Sebastian Lackner", "nvcuda: Add support for CUDA 7.0.", 1 },';
		echo '+    { "Sebastian Lackner", "nvcuda: Implement cuModuleLoad wrapper function.", 1 },';
		echo '+    { "Michael Müller", "nvcuda: Search for dylib library on Mac OS X.", 1 },';
	) >> "$patchlist"
fi

# Patchset nvapi-Stub_DLL
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35062] Fix graphical corruption in FarCry 3 with NVIDIA drivers
# |
# | Modified files:
# |   *	configure.ac, dlls/nvapi/Makefile.in, dlls/nvapi/nvapi.c, dlls/nvapi/nvapi.spec, dlls/nvapi/tests/Makefile.in,
# | 	dlls/nvapi/tests/nvapi.c, dlls/nvapi64/Makefile.in, dlls/nvapi64/nvapi64.spec, include/Makefile.in, include/nvapi.h
# |
if test "$enable_nvapi_Stub_DLL" -eq 1; then
	patch_apply nvapi-Stub_DLL/0001-nvapi-First-implementation.patch
	patch_apply nvapi-Stub_DLL/0002-nvapi-Add-stubs-for-NvAPI_EnumLogicalGPUs-and-undocu.patch
	patch_apply nvapi-Stub_DLL/0003-nvapi-Add-NvAPI_GetPhysicalGPUsFromLogicalGPU.patch
	patch_apply nvapi-Stub_DLL/0004-nvapi-Add-stub-for-NvAPI_EnumPhysicalGPUs.patch
	patch_apply nvapi-Stub_DLL/0005-nvapi-Add-stubs-for-NvAPI_GPU_GetFullName.patch
	patch_apply nvapi-Stub_DLL/0006-nvapi-Explicity-return-NULL-for-0x33c7358c-and-0x593.patch
	patch_apply nvapi-Stub_DLL/0007-nvapi-Add-stub-for-NvAPI_DISP_GetGDIPrimaryDisplayId.patch
	patch_apply nvapi-Stub_DLL/0008-nvapi-Add-stub-for-EnumNvidiaDisplayHandle.patch
	patch_apply nvapi-Stub_DLL/0009-nvapi-Add-stub-for-NvAPI_SYS_GetDriverAndBranchVersi.patch
	patch_apply nvapi-Stub_DLL/0010-nvapi-Add-stub-for-NvAPI_Unload.patch
	patch_apply nvapi-Stub_DLL/0011-nvapi-Add-stub-for-NvAPI_D3D_GetCurrentSLIState.patch
	patch_apply nvapi-Stub_DLL/0012-nvapi-Print-fixme-message-for-NvAPI_D3D9_StretchRect.patch
	patch_apply nvapi-Stub_DLL/0013-nvapi-tests-Use-structure-to-list-imports.patch
	patch_apply nvapi-Stub_DLL/0014-nvapi-Add-stub-for-NvAPI_GetLogicalGPUFromDisplay.patch
	patch_apply nvapi-Stub_DLL/0015-nvapi-Add-stub-for-NvAPI_D3D_GetObjectHandleForResou.patch
	patch_apply nvapi-Stub_DLL/0016-nvapi-Add-stub-for-NvAPI_D3D9_RegisterResource.patch
	(
		echo '+    { "Michael Müller", "nvapi: First implementation.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stubs for NvAPI_EnumLogicalGPUs and undocumented equivalent.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add NvAPI_GetPhysicalGPUsFromLogicalGPU.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_EnumPhysicalGPUs.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stubs for NvAPI_GPU_GetFullName.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Explicity return NULL for 0x33c7358c and 0x593e8644.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_DISP_GetGDIPrimaryDisplayId.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for EnumNvidiaDisplayHandle.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_SYS_GetDriverAndBranchVersion.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_Unload.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_D3D_GetCurrentSLIState.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Print fixme message for NvAPI_D3D9_StretchRectEx.", 1 },';
		echo '+    { "Michael Müller", "nvapi/tests: Use structure to list imports.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_GetLogicalGPUFromDisplay.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_D3D_GetObjectHandleForResource.", 1 },';
		echo '+    { "Michael Müller", "nvapi: Add stub for NvAPI_D3D9_RegisterResource.", 1 },';
	) >> "$patchlist"
fi

# Patchset nvcuvid-CUDA_Video_Support
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuvid/Makefile.in, dlls/nvcuvid/nvcuvid.c, dlls/nvcuvid/nvcuvid.spec, include/Makefile.in,
# | 	include/cuviddec.h, include/nvcuvid.h
# |
if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	patch_apply nvcuvid-CUDA_Video_Support/0001-nvcuvid-First-implementation.patch
	(
		echo '+    { "Michael Müller", "nvcuvid: First implementation.", 2 },';
	) >> "$patchlist"
fi

# Patchset nvencodeapi-Video_Encoder
# |
# | Modified files:
# |   *	configure.ac, dlls/nvencodeapi/Makefile.in, dlls/nvencodeapi/nvencodeapi.c, dlls/nvencodeapi/nvencodeapi.spec,
# | 	dlls/nvencodeapi64/Makefile.in, dlls/nvencodeapi64/nvencodeapi64.spec, include/Makefile.in, include/nvencodeapi.h
# |
if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	patch_apply nvencodeapi-Video_Encoder/0001-nvencodeapi-First-implementation.patch
	(
		echo '+    { "Michael Müller", "nvencodeapi: First implementation.", 1 },';
	) >> "$patchlist"
fi

# Patchset opengl32-Revert_Disable_Ext
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38480] Fix regression caused by blacklisting supported OpenGL extensions
# |
# | Modified files:
# |   *	dlls/opengl32/wgl.c
# |
if test "$enable_opengl32_Revert_Disable_Ext" -eq 1; then
	patch_apply opengl32-Revert_Disable_Ext/0001-Revert-opengl32-Return-a-NULL-pointer-for-functions-.patch
	(
		echo '+    { "Sebastian Lackner", "Revert \"opengl32: Return a NULL pointer for functions requiring unsupported or disabled extensions.\".", 1 },';
	) >> "$patchlist"
fi

# Patchset quartz-MediaSeeking_Positions
# |
# | Modified files:
# |   *	dlls/quartz/filtergraph.c, dlls/strmbase/pospass.c
# |
if test "$enable_quartz_MediaSeeking_Positions" -eq 1; then
	patch_apply quartz-MediaSeeking_Positions/0001-strmbase-Fix-MediaSeekingPassThru_GetPositions-retur.patch
	patch_apply quartz-MediaSeeking_Positions/0002-quartz-Include-the-stream-position-in-addition-to-th.patch
	patch_apply quartz-MediaSeeking_Positions/0003-quartz-Implement-MediaSeeking_GetCurrentPosition-on-.patch
	patch_apply quartz-MediaSeeking_Positions/0004-quartz-Implement-MediaSeeking_GetStopPosition-on-top.patch
	patch_apply quartz-MediaSeeking_Positions/0005-quartz-Remove-unused-cache-of-MediaSeeking-stop-posi.patch
	(
		echo '+    { "Erich E. Hoover", "strmbase: Fix MediaSeekingPassThru_GetPositions return when the pins are unconnected.", 1 },';
		echo '+    { "Erich E. Hoover", "quartz: Include the stream position in addition to the reference clock offset in the time returned by MediaSeeking_GetPositions.", 1 },';
		echo '+    { "Erich E. Hoover", "quartz: Implement MediaSeeking_GetCurrentPosition on top of MediaSeeking_GetPositions.", 1 },';
		echo '+    { "Erich E. Hoover", "quartz: Implement MediaSeeking_GetStopPosition on top of MediaSeeking_GetPositions.", 1 },';
		echo '+    { "Erich E. Hoover", "quartz: Remove unused cache of MediaSeeking stop position.", 1 },';
	) >> "$patchlist"
fi

# Patchset rasapi32-RasEnumDevicesA
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30378] Initialize *lpcDevices in RasEnumDevicesA
# |
# | Modified files:
# |   *	dlls/rasapi32/rasapi.c, dlls/rasapi32/tests/rasapi.c
# |
if test "$enable_rasapi32_RasEnumDevicesA" -eq 1; then
	patch_apply rasapi32-RasEnumDevicesA/0001-rasapi32-Set-lpcDevices-in-RasEnumDevicesA.patch
	(
		echo '+    { "Sebastian Lackner", "rasapi32: Set *lpcDevices in RasEnumDevicesA.", 1 },';
	) >> "$patchlist"
fi

# Patchset regedit-Reg_Parser
# |
# | Modified files:
# |   *	programs/regedit/regproc.c
# |
if test "$enable_regedit_Reg_Parser" -eq 1; then
	patch_apply regedit-Reg_Parser/0001-regedit-Need-3-bytes-of-room-at-end-of-buffer-for-r-.patch
	(
		echo '+    { "Jiaxing Wang", "regedit: Need 3 bytes of room at end of buffer for \\\\r\\\\n\\\\0 to avoid endless loop.", 1 },';
	) >> "$patchlist"
fi

# Patchset riched20-IText_Interface
# |
# | Modified files:
# |   *	dlls/riched20/richole.c, dlls/riched20/run.c, dlls/riched20/tests/richole.c
# |
if test "$enable_riched20_IText_Interface" -eq 1; then
	patch_apply riched20-IText_Interface/0001-riched20-Implement-IText-Selection-Range-Set-Start-E.patch
	patch_apply riched20-IText_Interface/0002-riched20-Stub-for-ITextFont-interface-and-implement-.patch
	patch_apply riched20-IText_Interface/0003-riched20-Stub-for-ITextPara-interface-and-implement-.patch
	patch_apply riched20-IText_Interface/0004-riched20-Fix-ME_RunOfsFromCharOfs-when-nCharOfs-strl.patch
	patch_apply riched20-IText_Interface/0005-riched20-Implement-ITextRange-GetText.patch
	patch_apply riched20-IText_Interface/0006-riched20-Implement-ITextRange-SetRange.patch
	patch_apply riched20-IText_Interface/0007-riched20-Implement-ITextRange-IsEqual.patch
	patch_apply riched20-IText_Interface/0008-riched20-Implement-ITextRange-GetStoryLength.patch
	patch_apply riched20-IText_Interface/0009-riched20-Implement-ITextSelection-GetStoryLength.patch
	patch_apply riched20-IText_Interface/0010-riched20-Silence-repeated-FIXMEs-triggered-by-Adobe-.patch
	(
		echo '+    { "Jactry Zeng", "riched20: Implement IText{Selection, Range}::Set{Start, End}.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Stub for ITextFont interface and implement ITextRange::GetFont and ITextSelection::GetFont.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Stub for ITextPara interface and implement ITextRange::GetPara.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Fix ME_RunOfsFromCharOfs() when nCharOfs > strlen().", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Implement ITextRange::GetText.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Implement ITextRange::SetRange.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Implement ITextRange::IsEqual.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Implement ITextRange::GetStoryLength.", 1 },';
		echo '+    { "Jactry Zeng", "riched20: Implement ITextSelection::GetStoryLength.", 1 },';
		echo '+    { "Sebastian Lackner", "riched20: Silence repeated FIXMEs triggered by Adobe Reader.", 1 },';
	) >> "$patchlist"
fi

# Patchset rpcrt4-Use_After_Free
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36743] Free RPC parameters allocated by application before anything else
# |
# | Modified files:
# |   *	dlls/rpcrt4/ndr_stubless.c
# |
if test "$enable_rpcrt4_Use_After_Free" -eq 1; then
	patch_apply rpcrt4-Use_After_Free/0001-rpcrt4-Free-parameters-allocated-by-application-befo.patch
	(
		echo '+    { "Jérôme Gardou", "rpcrt4: Free parameters allocated by application before anything else.", 1 },';
	) >> "$patchlist"
fi

# Patchset secur32-ANSI_NTLM_Credentials
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37063] Fix handling of ANSI NTLM credentials
# |
# | Modified files:
# |   *	dlls/secur32/ntlm.c
# |
if test "$enable_secur32_ANSI_NTLM_Credentials" -eq 1; then
	patch_apply secur32-ANSI_NTLM_Credentials/0001-secur32-Fix-handling-of-ANSI-NTLM-credentials.patch
	(
		echo '+    { "David Woodhouse", "secur32: Fix handling of ANSI NTLM credentials.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-RootDirectory_File
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, server/fd.c, server/file.c, server/file.h
# |
if test "$enable_server_RootDirectory_File" -eq 1; then
	patch_apply server-RootDirectory_File/0001-server-Fix-handling-of-opening-a-file-with-RootDirec.patch
	(
		echo '+    { "Sebastian Lackner", "server: Fix handling of opening a file with RootDirectory pointing to a file handle.", 3 },';
	) >> "$patchlist"
fi

# Patchset server-Stored_ACLs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33576] Support for stored file ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, include/wine/port.h, server/change.c, server/file.c, server/file.h, server/object.c,
# | 	server/object.h
# |
if test "$enable_server_Stored_ACLs" -eq 1; then
	patch_apply server-Stored_ACLs/0001-server-Unify-the-storage-of-security-attributes-for-.patch
	patch_apply server-Stored_ACLs/0002-server-Unify-the-retrieval-of-security-attributes-fo.patch
	patch_apply server-Stored_ACLs/0003-server-Add-a-helper-function-set_sd_from_token_inter.patch
	patch_apply server-Stored_ACLs/0004-server-Temporarily-store-the-full-security-descripto.patch
	patch_apply server-Stored_ACLs/0005-server-Store-file-security-attributes-with-extended-.patch
	patch_apply server-Stored_ACLs/0006-server-Convert-return-of-file-security-masks-with-ge.patch
	patch_apply server-Stored_ACLs/0007-server-Retrieve-file-security-attributes-with-extend.patch
	(
		echo '+    { "Erich E. Hoover", "server: Unify the storage of security attributes for files and directories.", 7 },';
		echo '+    { "Erich E. Hoover", "server: Unify the retrieval of security attributes for files and directories.", 7 },';
		echo '+    { "Sebastian Lackner", "server: Add a helper function set_sd_from_token_internal to merge two security descriptors.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Temporarily store the full security descriptor for file objects.", 1 },';
		echo '+    { "Erich E. Hoover", "server: Store file security attributes with extended file attributes.", 8 },';
		echo '+    { "Erich E. Hoover", "server: Convert return of file security masks with generic access mappings.", 7 },';
		echo '+    { "Erich E. Hoover", "server: Retrieve file security attributes with extended file attributes.", 7 },';
	) >> "$patchlist"
fi

# Patchset server-Inherited_ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, server/file.c
# |
if test "$enable_server_Inherited_ACLs" -eq 1; then
	patch_apply server-Inherited_ACLs/0001-server-Inherit-security-attributes-from-parent-direc.patch
	(
		echo '+    { "Erich E. Hoover", "server: Inherit security attributes from parent directories on creation.", 7 },';
	) >> "$patchlist"
fi

# Patchset server-ACL_Compat
# |
# | Modified files:
# |   *	server/file.c
# |
if test "$enable_server_ACL_Compat" -eq 1; then
	patch_apply server-ACL_Compat/0001-server-Add-compatibility-code-for-handling-the-old-m.patch
	(
		echo '+    { "Erich E. Hoover", "server: Add compatibility code for handling the old method of storing ACLs.", 6 },';
	) >> "$patchlist"
fi

# Patchset server-Address_List_Change
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38062] Support for non-blocking SIO_ADDRESS_LIST_CHANGE requests
# |
# | Modified files:
# |   *	server/sock.c
# |
if test "$enable_server_Address_List_Change" -eq 1; then
	patch_apply server-Address_List_Change/0001-server-Return-STATUS_CANT_WAIT-WSAEWOULDBLOCK-for-no.patch
	(
		echo '+    { "Erich E. Hoover", "server: Return STATUS_CANT_WAIT/WSAEWOULDBLOCK for non-overlapped SIO_ADDRESS_LIST_CHANGE requests on non-blocking sockets.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-ClipCursor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33479] Fix cursor clip regression / broken raw input in multiple games
# |   *	[#38087] Fix multithreading issues with fullscreen clipping
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/winex11.drv/mouse.c, server/queue.c
# |
if test "$enable_server_ClipCursor" -eq 1; then
	patch_apply server-ClipCursor/0001-server-Only-send-WM_WINE_CLIPCURSOR-for-forced-clip-.patch
	patch_apply server-ClipCursor/0002-winex11-Forward-all-clipping-requests-to-the-right-t.patch
	(
		echo '+    { "Sebastian Lackner", "server: Only send WM_WINE_CLIPCURSOR for forced clip resets.", 1 },';
		echo '+    { "Sebastian Lackner", "winex11: Forward all clipping requests to the right thread (including fullscreen clipping).", 1 },';
	) >> "$patchlist"
fi

# Patchset server-Delete_On_Close
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38417] Fix handling of opening read-only files for FILE_DELETE_ON_CLOSE
# |
# | Modified files:
# |   *	dlls/kernel32/file.c, dlls/kernel32/tests/file.c, server/fd.c
# |
if test "$enable_server_Delete_On_Close" -eq 1; then
	patch_apply server-Delete_On_Close/0001-server-Fix-handling-of-opening-read-only-files-with-.patch
	(
		echo '+    { "Sebastian Lackner", "server: Fix handling of opening read-only files with FILE_DELETE_ON_CLOSE.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-OpenProcess
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37087] Return an error when trying to open a terminated process
# |
# | Modified files:
# |   *	server/process.c, server/process.h
# |
if test "$enable_server_OpenProcess" -eq 1; then
	patch_apply server-OpenProcess/0001-server-Return-error-when-opening-a-terminating-proce.patch
	(
		echo '+    { "Michael Müller", "server: Return error when opening a terminating process.", 3 },';
	) >> "$patchlist"
fi

# Patchset server-JobObjects
# |
# | Modified files:
# |   *	dlls/kernel32/tests/process.c, dlls/ntdll/sync.c, server/process.c
# |
if test "$enable_server_JobObjects" -eq 1; then
	patch_apply server-JobObjects/0001-server-Implement-JOB_OBJECT_LIMIT_KILL_ON_JOB_CLOSE.patch
	patch_apply server-JobObjects/0002-ntdll-Implement-NtQueryInformationJobObject-stub-fun.patch
	(
		echo '+    { "Andrew Cook", "server: Implement JOB_OBJECT_LIMIT_KILL_ON_JOB_CLOSE.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Implement NtQueryInformationJobObject stub function.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-Key_State
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31899] Implement locking and synchronization of key states
# |   *	[#35907] Fix caps lock state issues with multiple processes
# |
# | Modified files:
# |   *	dlls/comctl32/tests/listview.c, server/queue.c
# |
if test "$enable_server_Key_State" -eq 1; then
	patch_apply server-Key_State/0001-server-Introduce-a-helper-function-to-update-the-thr.patch
	patch_apply server-Key_State/0002-server-Implement-locking-and-synchronization-of-keys.patch
	patch_apply server-Key_State/0003-server-Introduce-a-shadow-keystate-array-to-sync-key.patch
	(
		echo '+    { "Sebastian Lackner", "server: Introduce a helper function to update the thread_input key state.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Implement locking and synchronization of keystate buffer.", 2 },';
		echo '+    { "Sebastian Lackner", "server: Introduce a shadow keystate array to sync keystates only on changes.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-PeekMessage
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28884] GetMessage should remove already seen messages with higher priority
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, server/queue.c
# |
if test "$enable_server_PeekMessage" -eq 1; then
	patch_apply server-PeekMessage/0001-server-Fix-handling-of-GetMessage-after-previous-Pee.patch
	(
		echo '+    { "Sebastian Lackner", "server: Fix handling of GetMessage after previous PeekMessage call.", 2 },';
	) >> "$patchlist"
fi

# Patchset server-Realtime_Priority
# |
# | Modified files:
# |   *	server/Makefile.in, server/main.c, server/scheduler.c, server/thread.c, server/thread.h
# |
if test "$enable_server_Realtime_Priority" -eq 1; then
	patch_apply server-Realtime_Priority/0001-wineserver-Draft-to-implement-priority-levels-throug.patch
	(
		echo '+    { "Joakim Hernberg", "wineserver: Draft to implement priority levels through POSIX scheduling policies on linux.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-Shared_Memory
# |
# | Modified files:
# |   *	dlls/ntdll/ntdll_misc.h, dlls/ntdll/server.c, dlls/ntdll/thread.c, dlls/ntdll/virtual.c, dlls/user32/focus.c,
# | 	dlls/user32/input.c, dlls/user32/message.c, dlls/user32/user_private.h, include/wine/server.h, include/winternl.h,
# | 	server/fd.c, server/file.h, server/main.c, server/mapping.c, server/protocol.def, server/queue.c, server/thread.c,
# | 	server/thread.h
# |
if test "$enable_server_Shared_Memory" -eq 1; then
	patch_apply server-Shared_Memory/0001-ntdll-Implement-virtual_map_shared_memory.patch
	patch_apply server-Shared_Memory/0002-server-Implement-support-for-global-and-local-shared.patch
	patch_apply server-Shared_Memory/0003-user32-Get-rid-of-wineserver-call-for-GetInputState.patch
	patch_apply server-Shared_Memory/0004-user32-Avoid-unnecessary-wineserver-calls-in-PeekMes.patch
	patch_apply server-Shared_Memory/0005-user32-Get-rid-of-wineserver-call-for-GetLastInputIn.patch
	patch_apply server-Shared_Memory/0006-ntdll-Only-enable-wineserver-shared-memory-communica.patch
	patch_apply server-Shared_Memory/0007-server-Store-a-list-of-associated-queues-for-each-th.patch
	patch_apply server-Shared_Memory/0008-user32-Get-rid-of-wineserver-call-for-GetActiveWindo.patch
	patch_apply server-Shared_Memory/0009-user32-Cache-the-result-of-GetForegroundWindow.patch
	(
		echo '+    { "Sebastian Lackner", "ntdll: Implement virtual_map_shared_memory.", 1 },';
		echo '+    { "Michael Müller", "server: Implement support for global and local shared memory blocks based on memfd.", 1 },';
		echo '+    { "Sebastian Lackner", "user32: Get rid of wineserver call for GetInputState.", 1 },';
		echo '+    { "Sebastian Lackner", "user32: Avoid unnecessary wineserver calls in PeekMessage/GetMessage.", 1 },';
		echo '+    { "Michael Müller", "user32: Get rid of wineserver call for GetLastInputInfo.", 1 },';
		echo '+    { "Sebastian Lackner", "ntdll: Only enable wineserver shared memory communication when a special environment variable is set.", 1 },';
		echo '+    { "Sebastian Lackner", "server: Store a list of associated queues for each thread input.", 1 },';
		echo '+    { "Sebastian Lackner", "user32: Get rid of wineserver call for GetActiveWindow, GetFocus, GetCapture.", 1 },';
		echo '+    { "Sebastian Lackner", "user32: Cache the result of GetForegroundWindow.", 1 },';
	) >> "$patchlist"
fi

# Patchset server-Unexpected_Wakeup
# |
# | Modified files:
# |   *	dlls/kernel32/tests/sync.c, server/thread.c
# |
if test "$enable_server_Unexpected_Wakeup" -eq 1; then
	patch_apply server-Unexpected_Wakeup/0001-server-Avoid-sending-unexpected-wakeup-with-uninitia.patch
	patch_apply server-Unexpected_Wakeup/0002-kernel32-tests-Repeat-test-for-SignalObjectAndWait-m.patch
	(
		echo '+    { "Sebastian Lackner", "server: Avoid sending unexpected wakeup with uninitialized cookie value.", 1 },';
		echo '+    { "Sebastian Lackner", "kernel32/tests: Repeat test for SignalObjectAndWait multiple times to test wineserver wakeup cookie management.", 1 },';
	) >> "$patchlist"
fi

# Patchset setupapi-SetupDiSelectBestCompatDrv
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32088] Add stub for setupapi.SetupDiSelectBestCompatDrv
# |
# | Modified files:
# |   *	dlls/setupapi/setupapi.spec, dlls/setupapi/stubs.c
# |
if test "$enable_setupapi_SetupDiSelectBestCompatDrv" -eq 1; then
	patch_apply setupapi-SetupDiSelectBestCompatDrv/0001-setupapi-Add-stub-for-SetupDiSelectBestCompatDrv.patch
	(
		echo '+    { "Austin English", "setupapi: Add stub for SetupDiSelectBestCompatDrv.", 1 },';
	) >> "$patchlist"
fi

# Patchset setupapi-SetupDiSetDeviceInstallParamsW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29903] Add stub for setupapi.SetupDiSetDeviceInstallParamsW
# |
# | Modified files:
# |   *	dlls/setupapi/devinst.c, dlls/setupapi/setupapi.spec
# |
if test "$enable_setupapi_SetupDiSetDeviceInstallParamsW" -eq 1; then
	patch_apply setupapi-SetupDiSetDeviceInstallParamsW/0001-setupapi-add-SetupDiSetDeviceInstallParamsW-stub.patch
	(
		echo '+    { "Austin English", "setupapi: add SetupDiSetDeviceInstallParamsW stub.", 1 },';
	) >> "$patchlist"
fi

# Patchset setupapi-SetupPromptForDisk
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#20465] Wine ignores IDF_CHECKFIRST flag in SetupPromptForDisk
# |
# | Modified files:
# |   *	dlls/setupapi/dialog.c, dlls/setupapi/tests/Makefile.in, dlls/setupapi/tests/dialog.c
# |
if test "$enable_setupapi_SetupPromptForDisk" -eq 1; then
	patch_apply setupapi-SetupPromptForDisk/0001-setupapi-Add-support-for-IDF_CHECKFIRST-flag-in-Setu.patch
	patch_apply setupapi-SetupPromptForDisk/0002-setupapi-tests-Add-test-for-IDF_CHECKFIRST-and-Setup.patch
	(
		echo '+    { "Michael Müller", "setupapi: Add support for IDF_CHECKFIRST flag in SetupPromptForDiskW.", 1 },';
		echo '+    { "Michael Müller", "setupapi/tests: Add test for IDF_CHECKFIRST and SetupPromptForDiskA/W.", 1 },';
	) >> "$patchlist"
fi

# Patchset shdocvw-ParseURLFromOutsideSource_Tests
# |
# | Modified files:
# |   *	dlls/shdocvw/shdocvw_main.c, dlls/shdocvw/tests/shdocvw.c
# |
if test "$enable_shdocvw_ParseURLFromOutsideSource_Tests" -eq 1; then
	patch_apply shdocvw-ParseURLFromOutsideSource_Tests/0001-shdocvw-Check-precisely-ParseURLFromOutsideSourceX-r.patch
	(
		echo '+    { "Christian Costa", "shdocvw: Check precisely ParseURLFromOutsideSourceX returned values in tests and make code clearer about that.", 3 },';
	) >> "$patchlist"
fi

# Patchset shell32-Default_Path
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30385] Support for KF_FLAG_DEFAULT_PATH in SHGetKnownFolderPath
# |
# | Modified files:
# |   *	dlls/shell32/shellpath.c, dlls/shell32/tests/shellpath.c
# |
if test "$enable_shell32_Default_Path" -eq 1; then
	patch_apply shell32-Default_Path/0001-shell32-Implement-KF_FLAG_DEFAULT_PATH-flag-for-SHGe.patch
	(
		echo '+    { "Sebastian Lackner", "shell32: Implement KF_FLAG_DEFAULT_PATH flag for SHGetKnownFolderPath.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-File_Property_Dialog
# |
# | Modified files:
# |   *	dlls/shell32/shell32.rc, dlls/shell32/shlview_cmenu.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_File_Property_Dialog" -eq 1; then
	patch_apply shell32-File_Property_Dialog/0001-shell32-Add-general-tab-in-file-property-dialog.patch
	(
		echo '+    { "Michael Müller", "shell32: Add general tab in file property dialog.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-Icons
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24721] Support for extra large and jumbo icon lists in shell32
# |
# | Modified files:
# |   *	dlls/shell32/iconcache.c, dlls/shell32/shell32_main.h, dlls/shell32/shellord.c
# |
if test "$enable_shell32_Icons" -eq 1; then
	patch_apply shell32-Icons/0001-shell32-Add-support-for-extra-large-and-jumbo-icon-l.patch
	(
		echo '+    { "Michael Müller", "shell32: Add support for extra large and jumbo icon lists.", 2 },';
	) >> "$patchlist"
fi

# Patchset shell32-Placeholder_Icons
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30185] Add shell32 placeholder icons to match offsets with Windows
# |
# | Modified files:
# |   *	dlls/shell32/placeholder.ico, dlls/shell32/shell32.rc, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_Placeholder_Icons" -eq 1; then
	patch_apply shell32-Placeholder_Icons/0001-shell32-Add-placeholder-icons-to-match-icon-offset-w.patch
	(
		echo '+    { "Michael Müller", "shell32: Add placeholder icons to match icon offset with XP.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-Progress_Dialog
# |
# | Modified files:
# |   *	dlls/shell32/shell32.rc, dlls/shell32/shlfileop.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_Progress_Dialog" -eq 1; then
	patch_apply shell32-Progress_Dialog/0001-shell32-Correct-indentation-in-shfileop.c.patch
	patch_apply shell32-Progress_Dialog/0002-shell32-Pass-FILE_INFORMATION-into-SHNotify-function.patch
	patch_apply shell32-Progress_Dialog/0003-shell32-Implement-file-operation-progress-dialog.patch
	patch_apply shell32-Progress_Dialog/0004-shell32-Show-animation-during-SHFileOperation.patch
	(
		echo '+    { "Michael Müller", "shell32: Correct indentation in shfileop.c.", 1 },';
		echo '+    { "Michael Müller", "shell32: Pass FILE_INFORMATION into SHNotify* functions.", 1 },';
		echo '+    { "Michael Müller", "shell32: Implement file operation progress dialog.", 1 },';
		echo '+    { "Michael Müller", "shell32: Show animation during SHFileOperation.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-Quoted_ShellExecute
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#19666] Multiple applications start wrong executable if whitespace present in name
# |
# | Modified files:
# |   *	dlls/shell32/shlexec.c, dlls/shell32/tests/shlexec.c
# |
if test "$enable_shell32_Quoted_ShellExecute" -eq 1; then
	patch_apply shell32-Quoted_ShellExecute/0001-shell32-Quote-program-name-in-ShellExecuteEx-if-it-c.patch
	(
		echo '+    { "Stefan Leichter", "shell32: Quote program name in ShellExecuteEx if it contains spaces.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-RunDLL_CallEntry16
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23033] Use manual relay for RunDLL_CallEntry16 in shell32
# |
# | Modified files:
# |   *	dlls/shell32/control.c, dlls/shell32/shell32.spec
# |
if test "$enable_shell32_RunDLL_CallEntry16" -eq 1; then
	patch_apply shell32-RunDLL_CallEntry16/0001-shell32-Use-manual-redirection-for-RunDLL_CallEntry1.patch
	(
		echo '+    { "Michael Müller", "shell32: Use manual redirection for RunDLL_CallEntry16.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-Run_Dialog
# |
# | Modified files:
# |   *	dlls/shell32/shell32.rc
# |
if test "$enable_shell32_Run_Dialog" -eq 1; then
	patch_apply shell32-Run_Dialog/0001-shell32-Add-caption-to-Run-dialog.patch
	(
		echo '+    { "Jared Smudde", "shell32: Add caption to Run dialog.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-SHCreateSessionKey
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35630] Support for SHCreateSessionKey
# |
# | Modified files:
# |   *	dlls/shell32/shell32.spec, dlls/shell32/shellreg.c
# |
if test "$enable_shell32_SHCreateSessionKey" -eq 1; then
	patch_apply shell32-SHCreateSessionKey/0001-shell32-Implement-SHCreateSessionKey.patch
	(
		echo '+    { "Dmitry Timoshkov", "shell32: Implement SHCreateSessionKey.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-SHFileOperation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37916] Anno 1602 installer depends on Windows 98 behavior of SHFileOperationW
# |
# | Modified files:
# |   *	dlls/shell32/shlfileop.c
# |
if test "$enable_shell32_SHFileOperation" -eq 1; then
	patch_apply shell32-SHFileOperation/0001-shell32-Choose-return-value-for-SHFileOperationW-dep.patch
	(
		echo '+    { "Michael Müller", "shell32: Choose return value for SHFileOperationW depending on windows version.", 1 },';
	) >> "$patchlist"
fi

# Patchset shell32-UnixFS
# |
# | Modified files:
# |   *	dlls/shell32/shfldr_desktop.c, dlls/shell32/tests/shlfolder.c
# |
if test "$enable_shell32_UnixFS" -eq 1; then
	patch_apply shell32-UnixFS/0001-shell32-Do-not-use-unixfs-for-devices-without-mountp.patch
	(
		echo '+    { "Michael Müller", "shell32: Do not use unixfs for devices without mountpoint.", 1 },';
	) >> "$patchlist"
fi

# Patchset shlwapi-AssocGetPerceivedType
# |
# | Modified files:
# |   *	dlls/shlwapi/assoc.c, dlls/shlwapi/tests/assoc.c
# |
if test "$enable_shlwapi_AssocGetPerceivedType" -eq 1; then
	patch_apply shlwapi-AssocGetPerceivedType/0001-shlwapi-tests-Add-tests-for-AssocGetPerceivedType.patch
	patch_apply shlwapi-AssocGetPerceivedType/0002-shlwapi-Implement-AssocGetPerceivedType.patch
	(
		echo '+    { "Mark Jansen", "shlwapi/tests: Add tests for AssocGetPerceivedType.", 1 },';
		echo '+    { "Mark Jansen", "shlwapi: Implement AssocGetPerceivedType.", 2 },';
	) >> "$patchlist"
fi

# Patchset shlwapi-UrlCombine
# |
# | Modified files:
# |   *	dlls/shlwapi/tests/url.c, dlls/shlwapi/url.c
# |
if test "$enable_shlwapi_UrlCombine" -eq 1; then
	patch_apply shlwapi-UrlCombine/0001-shlwapi-tests-Add-additional-tests-for-UrlCombine-and-.patch
	patch_apply shlwapi-UrlCombine/0002-shlwapi-UrlCombineW-workaround-for-relative-paths.patch
	(
		echo '+    { "Sebastian Lackner", "shlwapi/tests: Add additional tests for UrlCombine and UrlCanonicalize.", 1 },';
		echo '+    { "Sebastian Lackner", "shlwapi: UrlCombineW workaround for relative paths.", 1 },';
	) >> "$patchlist"
fi

# Patchset urlmon-CoInternetSetFeatureEnabled
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35197] Ignore unsupported flags for CoInternetSetFeatureEnabled
# |
# | Modified files:
# |   *	dlls/urlmon/internet.c
# |
if test "$enable_urlmon_CoInternetSetFeatureEnabled" -eq 1; then
	patch_apply urlmon-CoInternetSetFeatureEnabled/0001-urlmon-Ignore-unsupported-flags-for-CoInternetSetFea.patch
	(
		echo '+    { "Bruno Jesus", "urlmon: Ignore unsupported flags for CoInternetSetFeatureEnabled.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-Dialog_Paint_Event
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35652] Send WM_PAINT event during dialog creation
# |
# | Modified files:
# |   *	dlls/user32/dialog.c
# |
if test "$enable_user32_Dialog_Paint_Event" -eq 1; then
	patch_apply user32-Dialog_Paint_Event/0001-user32-Call-UpdateWindow-during-DIALOG_CreateIndirec.patch
	(
		echo '+    { "Sebastian Lackner", "user32: Call UpdateWindow() during DIALOG_CreateIndirect.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-DrawTextExW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22109] Fix handling of invert_y in DrawTextExW
# |
# | Modified files:
# |   *	dlls/user32/text.c
# |
if test "$enable_user32_DrawTextExW" -eq 1; then
	patch_apply user32-DrawTextExW/0001-user32-Fix-handling-of-invert_y-in-DrawTextExW.patch
	(
		echo '+    { "Sebastian Lackner", "user32: Fix handling of invert_y in DrawTextExW.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-GetSystemMetrics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18732] Make it possible to change media center / tablet pc status
# |
# | Modified files:
# |   *	dlls/user32/sysparams.c
# |
if test "$enable_user32_GetSystemMetrics" -eq 1; then
	patch_apply user32-GetSystemMetrics/0001-user32-Allow-changing-the-tablet-media-center-status.patch
	(
		echo '+    { "Michael Müller", "user32: Allow changing the tablet / media center status via wine registry key.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-Invalidate_Key_State
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29871] Globally invalidate key state on changes in other threads
# |
# | Modified files:
# |   *	dlls/user32/input.c
# |
if test "$enable_user32_Invalidate_Key_State" -eq 1; then
	patch_apply user32-Invalidate_Key_State/0001-user32-Globally-invalidate-key-state-on-changes-in-o.patch
	(
		echo '+    { "Sebastian Lackner", "user32: Globally invalidate key state on changes in other threads.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-Mouse_Message_Hwnd
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12007] Fix issues with dragging layers between images in Adobe Photoshop 7.0
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/user32/tests/input.c
# |
if test "$enable_user32_Mouse_Message_Hwnd" -eq 1; then
	patch_apply user32-Mouse_Message_Hwnd/0001-user32-Try-harder-to-find-a-target-for-mouse-message.patch
	(
		echo '+    { "Dmitry Timoshkov", "user32: Try harder to find a target for mouse messages.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-Painting
# |
# | Modified files:
# |   *	dlls/user32/painting.c, dlls/user32/win.c
# |
if test "$enable_user32_Painting" -eq 1; then
	patch_apply user32-Painting/0001-user32-Enforce-that-surfaces-are-flushed-after-Relea.patch
	(
		echo '+    { "Sebastian Lackner", "user32: Enforce that surfaces are flushed after ReleaseDC.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-ScrollWindowEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37706] Fix return value of ScrollWindowEx for invisible windows
# |
# | Modified files:
# |   *	dlls/user32/painting.c
# |
if test "$enable_user32_ScrollWindowEx" -eq 1; then
	patch_apply user32-ScrollWindowEx/0001-user32-Fix-return-value-of-ScrollWindowEx-for-invisi.patch
	(
		echo '+    { "Dmitry Timoshkov", "user32: Fix return value of ScrollWindowEx for invisible windows.", 1 },';
	) >> "$patchlist"
fi

# Patchset user32-WndProc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32451] Fix for programs leaking wndproc slots
# |
# | Modified files:
# |   *	dlls/user.exe16/message.c, dlls/user32/winproc.c
# |
if test "$enable_user32_WndProc" -eq 1; then
	patch_apply user32-WndProc/0001-user32-Increase-MAX_WINPROCS-to-16384.patch
	(
		echo '+    { "Sebastian Lackner", "user32: Increase MAX_WINPROCS to 16384.", 2 },';
	) >> "$patchlist"
fi

# Patchset vcomp-Functions
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31640] Implement various vcomp functions
# |
# | Modified files:
# |   *	configure.ac, dlls/vcomp/main.c, dlls/vcomp/tests/Makefile.in, dlls/vcomp/tests/vcomp.c,
# | 	dlls/vcomp/tests/vcomp.manifest, dlls/vcomp/tests/vcomp.rc, dlls/vcomp/vcomp.spec, dlls/vcomp100/vcomp100.spec,
# | 	dlls/vcomp90/vcomp90.spec
# |
if test "$enable_vcomp_Functions" -eq 1; then
	patch_apply vcomp-Functions/0001-vcomp-Implement-stub-for-_vcomp_fork.patch
	patch_apply vcomp-Functions/0002-vcomp-Add-basic-worker-thread-infrastructure.patch
	patch_apply vcomp-Functions/0003-vcomp-tests-Add-initial-tests-for-_vcomp_fork.patch
	patch_apply vcomp-Functions/0004-vcomp-Implement-_vcomp_for_static_simple_init-and-_v.patch
	patch_apply vcomp-Functions/0005-vcomp-tests-Add-tests-for-_vcomp_for_static_simple_i.patch
	patch_apply vcomp-Functions/0006-vcomp-Implement-_vcomp_for_static_init.patch
	patch_apply vcomp-Functions/0007-vcomp-tests-Add-tests-for-_vcomp_for_static_init.patch
	patch_apply vcomp-Functions/0008-vcomp-Implement-_vcomp_barrier.patch
	patch_apply vcomp-Functions/0009-vcomp-Implement-omp_in_parallel.patch
	patch_apply vcomp-Functions/0010-vcomp-Implement-_vcomp_sections_init-and-_vcomp_sect.patch
	patch_apply vcomp-Functions/0011-vcomp-Implement-_vcomp_for_dynamic_init-and-_vcomp_f.patch
	(
		echo '+    { "Dan Kegel", "vcomp: Implement stub for _vcomp_fork.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Add basic worker thread infrastructure.", 1 },';
		echo '+    { "Dan Kegel", "vcomp/tests: Add initial tests for _vcomp_fork.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement _vcomp_for_static_simple_init and _vcomp_for_static_end.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp/tests: Add tests for _vcomp_for_static_simple_init.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement _vcomp_for_static_init.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp/tests: Add tests for _vcomp_for_static_init.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement _vcomp_barrier.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement omp_in_parallel.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement _vcomp_sections_init and _vcomp_sections_next and add tests.", 1 },';
		echo '+    { "Sebastian Lackner", "vcomp: Implement _vcomp_for_dynamic_init and _vcomp_for_dynamic_next and add tests.", 1 },';
	) >> "$patchlist"
fi

# Patchset version-VerQueryValue
# |
# | Modified files:
# |   *	dlls/version/tests/info.c
# |
if test "$enable_version_VerQueryValue" -eq 1; then
	patch_apply version-VerQueryValue/0001-version-Test-for-VerQueryValueA-try-2.patch
	(
		echo '+    { "Mark Jansen", "version: Test for VerQueryValueA.", 2 },';
	) >> "$patchlist"
fi

# Patchset wbemdisp-ISWbemSecurity
# |
# | Modified files:
# |   *	dlls/wbemdisp/locator.c
# |
if test "$enable_wbemdisp_ISWbemSecurity" -eq 1; then
	patch_apply wbemdisp-ISWbemSecurity/0001-wbemdisp-Add-ISWbemSecurity-stub-interface.patch
	(
		echo '+    { "Michael Müller", "wbemdisp: Add ISWbemSecurity stub interface.", 1 },';
	) >> "$patchlist"
fi

# Patchset wbemdisp-Timeout
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37618] Silence repeated wbemprox "timeout not supported" fixme
# |
# | Modified files:
# |   *	dlls/wbemprox/class.c
# |
if test "$enable_wbemdisp_Timeout" -eq 1; then
	patch_apply wbemdisp-Timeout/0001-wbemprox-Silence-repeated-timeout-not-supported-fixm.patch
	(
		echo '+    { "Jarkko Korpi", "wbemprox: Silence repeated \"timeout not supported\" fixme.", 1 },';
	) >> "$patchlist"
fi

# Patchset wiaservc-IEnumWIA_DEV_INFO
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#27775] Implement empty enumerator for IWiaDevMgr::EnumDeviceInfo
# |
# | Modified files:
# |   *	dlls/wiaservc/Makefile.in, dlls/wiaservc/enumwiadevinfo.c, dlls/wiaservc/wiadevmgr.c, dlls/wiaservc/wiaservc_private.h
# |
if test "$enable_wiaservc_IEnumWIA_DEV_INFO" -eq 1; then
	patch_apply wiaservc-IEnumWIA_DEV_INFO/0001-wiaservc-Implement-IWiaDevMgr-EnumDeviceInfo-by-retu.patch
	(
		echo '+    { "Mikael Ståldal", "wiaservc: Implement IWiaDevMgr::EnumDeviceInfo by returning an empty enumeration of devices.", 1 },';
	) >> "$patchlist"
fi

# Patchset widl-Buffer_Overflow
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37129] Increase buffer size in widl/typegen.c to avoid buffer overflow
# |
# | Modified files:
# |   *	tools/widl/typegen.c
# |
if test "$enable_widl_Buffer_Overflow" -eq 1; then
	patch_apply widl-Buffer_Overflow/0001-widl-Increase-buffer-size-in-typegen.c.patch
	(
		echo '+    { "Jarkko Korpi", "widl: Increase buffer size in typegen.c.", 1 },';
	) >> "$patchlist"
fi

# Patchset windowscodecs-GIF_Decoder
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32227] Ignore garbage after decoding gif lines
# |
# | Modified files:
# |   *	dlls/windowscodecs/ungif.c
# |
if test "$enable_windowscodecs_GIF_Decoder" -eq 1; then
	patch_apply windowscodecs-GIF_Decoder/0001-windowscodecs-Don-t-fail-to-decode-GIF-if-an-image-h.patch
	(
		echo '+    { "Dmitry Timoshkov", "windowscodecs: Don'\''t fail to decode GIF if an image has been already loaded.", 1 },';
	) >> "$patchlist"
fi

# Patchset windowscodecs-TIFF_Decoder
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38027] Support for 8bpp grayscale TIFF images with 8bpp alpha channel
# |
# | Modified files:
# |   *	dlls/windowscodecs/tests/tiffformat.c, dlls/windowscodecs/tiffformat.c
# |
if test "$enable_windowscodecs_TIFF_Decoder" -eq 1; then
	patch_apply windowscodecs-TIFF_Decoder/0001-windowscodecs-Add-support-for-8bpp-grayscale-TIFF-im.patch
	(
		echo '+    { "Dmitry Timoshkov", "windowscodecs: Add support for 8bpp grayscale TIFF images with 8bpp alpha channel.", 1 },';
	) >> "$patchlist"
fi

# Patchset wine.inf-Performance
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33661] Add performance library registry keys needed by MS SQL Server Management Studio Express 2008 R2
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_Performance" -eq 1; then
	patch_apply wine.inf-Performance/0001-wine.inf-Add-registry-keys-for-Windows-Performance-L.patch
	(
		echo '+    { "Daniel Jelinski", "wine.inf: Add registry keys for Windows Performance Library.", 1 },';
	) >> "$patchlist"
fi

# Patchset wine.inf-ProfileList_UserSID
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#15670] Add a ProfileList\<UserSID> registry subkey
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_ProfileList_UserSID" -eq 1; then
	patch_apply wine.inf-ProfileList_UserSID/0001-wine.inf-Add-a-ProfileList-UserSID-registry-subkey.patch
	(
		echo '+    { "Sebastian Lackner", "wine.inf: Add a ProfileList\\\\<UserSID> registry subkey.", 1 },';
	) >> "$patchlist"
fi

# Patchset wineboot-HKEY_DYN_DATA
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7115] Need for Speed 3 installer requires devices in HKEY_DYN_DATA
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_HKEY_DYN_DATA" -eq 1; then
	patch_apply wineboot-HKEY_DYN_DATA/0001-wineboot-Add-some-generic-hardware-in-HKEY_DYN_DATA-.patch
	(
		echo '+    { "Michael Müller", "wineboot: Add some generic hardware in HKEY_DYN_DATA\\\\Config Manager\\\\Enum.", 1 },';
	) >> "$patchlist"
fi

# Patchset wineboot-MachineGuid
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38508] Create HKLM\Software\Microsoft\Cryptography\MachineGuid registry key
# |   *	[#17823] Assign a drive serial number during prefix creation/update
# |
# | Modified files:
# |   *	programs/wineboot/Makefile.in, programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_MachineGuid" -eq 1; then
	patch_apply wineboot-MachineGuid/0001-wineboot-Create-MachineGuid-registry-value.patch
	patch_apply wineboot-MachineGuid/0002-wineboot-Assign-a-drive-serial-number-during-prefix-.patch
	(
		echo '+    { "Michael Müller", "wineboot: Create MachineGuid registry value.", 1 },';
		echo '+    { "Sebastian Lackner", "wineboot: Assign a drive serial number during prefix creation/update.", 1 },';
	) >> "$patchlist"
fi

# Patchset wineboot-drivers_etc_Stubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12076] Create stub files for system32/drivers/etc/{services,hosts,networks,protocol}
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_drivers_etc_Stubs" -eq 1; then
	patch_apply wineboot-drivers_etc_Stubs/0001-wineboot-Init-system32-drivers-etc-host-networks-pro.patch
	(
		echo '+    { "Sebastian Lackner", "wineboot: Init system32/drivers/etc/{host,networks,protocol,services}.", 1 },';
	) >> "$patchlist"
fi

# Patchset winecfg-Libraries
# |
# | Modified files:
# |   *	programs/winecfg/libraries.c
# |
if test "$enable_winecfg_Libraries" -eq 1; then
	patch_apply winecfg-Libraries/0001-winecfg-Double-click-in-dlls-list-to-edit-item-s-ove.patch
	(
		echo '+    { "Hao Peng", "winecfg: Double click in dlls list to edit item'\''s overides.", 3 },';
	) >> "$patchlist"
fi

# Patchset winecfg-Unmounted_Devices
# |
# | Modified files:
# |   *	programs/winecfg/drive.c, programs/winecfg/driveui.c
# |
if test "$enable_winecfg_Unmounted_Devices" -eq 1; then
	patch_apply winecfg-Unmounted_Devices/0001-winecfg-Show-unmounted-devices-and-allow-changing-th.patch
	(
		echo '+    { "Michael Müller", "winecfg: Show unmounted devices and allow changing the device value.", 1 },';
	) >> "$patchlist"
fi

# Patchset wineconsole-Insert_Mode
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38697] Allow to enable/disable InsertMode in wineconsole settings
# |
# | Modified files:
# |   *	programs/wineconsole/dialog.c, programs/wineconsole/wineconsole.c, programs/wineconsole/wineconsole.rc,
# | 	programs/wineconsole/wineconsole_res.h
# |
if test "$enable_wineconsole_Insert_Mode" -eq 1; then
	patch_apply wineconsole-Insert_Mode/0001-wineconsole-Rearrange-user-dialog-to-make-space-for-.patch
	patch_apply wineconsole-Insert_Mode/0002-wineconsole-Improve-semantics-of-some-poorly-worded-.patch
	patch_apply wineconsole-Insert_Mode/0003-wineconsole-Add-InsertMode-to-the-user-dialog.patch
	(
		echo '+    { "Hugh McMaster", "wineconsole: Rearrange user dialog to make space for InsertMode checkbox.", 1 },';
		echo '+    { "Hugh McMaster", "wineconsole: Improve semantics of some poorly-worded resource strings.", 1 },';
		echo '+    { "Hugh McMaster", "wineconsole: Add InsertMode to the user dialog.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-Accounting
# |
# | Modified files:
# |   *	dlls/wined3d/device.c, dlls/wined3d/directx.c, dlls/wined3d/wined3d_gl.h
# |
if test "$enable_wined3d_Accounting" -eq 1; then
	patch_apply wined3d-Accounting/0001-wined3d-Use-real-values-for-memory-accounting-on-NVI.patch
	(
		echo '+    { "Michael Müller", "wined3d: Use real values for memory accounting on NVIDIA cards.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-CSMT_Helper
# |
# | Modified files:
# |   *	configure.ac, dlls/ddraw/surface.c, dlls/wined3d-csmt/Makefile.in, dlls/wined3d-csmt/version.rc,
# | 	dlls/wined3d/resource.c, dlls/wined3d/surface.c, dlls/wined3d/texture.c, dlls/wined3d/volume.c,
# | 	dlls/wined3d/wined3d.spec, dlls/wined3d/wined3d_private.h, include/wine/wined3d.h
# |
if test "$enable_wined3d_CSMT_Helper" -eq 1; then
	patch_apply wined3d-CSMT_Helper/0001-wined3d-Merge-get_pitch-functions.patch
	patch_apply wined3d-CSMT_Helper/0002-wined3d-Add-second-dll-with-STAGING_CSMT-definition-.patch
	(
		echo '+    { "Stefan Dösinger", "wined3d: Merge get_pitch functions.", 1 },';
		echo '+    { "Sebastian Lackner", "wined3d: Add second dll with STAGING_CSMT definition set.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-MESA_GPU_Info
# |
# | Modified files:
# |   *	dlls/wined3d/directx.c, dlls/wined3d/wined3d_gl.h, dlls/winex11.drv/opengl.c, include/wine/wgl_driver.h
# |
if test "$enable_wined3d_MESA_GPU_Info" -eq 1; then
	patch_apply wined3d-MESA_GPU_Info/0001-wined3d-Use-pci-and-memory-information-from-MESA-if-.patch
	(
		echo '+    { "Michael Müller", "wined3d: Use pci and memory information from MESA if possible.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-Multisampling
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12652] Allow to override number of quality levels for D3DMULTISAMPLE_NONMASKABLE.
# |
# | Modified files:
# |   *	dlls/wined3d/directx.c, dlls/wined3d/wined3d_main.c, dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_Multisampling" -eq 1; then
	patch_apply wined3d-Multisampling/0001-wined3d-Allow-to-specify-multisampling-AA-quality-le.patch
	(
		echo '+    { "Austin English", "wined3d: Allow to specify multisampling AA quality levels via registry.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-Revert_PixelFormat
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35655] Fix wined3d performance drop introduced by pixelformat changes.
# |   *	[#35718] Fix flickering introduced by pixelformat changes.
# |   *	[#35975] Fix gray screen on startup introduced by pixelformat changes.
# |   *	[#36900] Fix missing video introduced by pixelformat changes.
# |
# | Modified files:
# |   *	dlls/d3d8/tests/device.c, dlls/d3d9/tests/device.c, dlls/ddraw/tests/ddraw1.c, dlls/ddraw/tests/ddraw2.c,
# | 	dlls/ddraw/tests/ddraw4.c, dlls/ddraw/tests/ddraw7.c, dlls/wined3d/context.c, dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_Revert_PixelFormat" -eq 1; then
	patch_apply wined3d-Revert_PixelFormat/0001-Revert-wined3d-Track-if-a-context-s-private-hdc-has-.patch
	patch_apply wined3d-Revert_PixelFormat/0002-Revert-wined3d-Track-if-a-context-s-hdc-is-private-s.patch
	patch_apply wined3d-Revert_PixelFormat/0003-Revert-wined3d-When-restoring-pixel-format-in-contex.patch
	patch_apply wined3d-Revert_PixelFormat/0004-Revert-wined3d-Don-t-call-GetPixelFormat-to-set-a-fl.patch
	patch_apply wined3d-Revert_PixelFormat/0005-Revert-wined3d-Restore-the-pixel-format-of-the-windo.patch
	patch_apply wined3d-Revert_PixelFormat/0006-d3d8-Mark-tests-which-no-longer-pass-due-to-reverts-.patch
	patch_apply wined3d-Revert_PixelFormat/0007-d3d9-Mark-tests-which-no-longer-pass-due-to-reverts-.patch
	patch_apply wined3d-Revert_PixelFormat/0008-ddraw-Mark-tests-which-no-longer-pass-due-to-reverts.patch
	(
		echo '+    { "Ken Thomases", "Revert \"wined3d: Track if a context'\''s private hdc has had its pixel format set, so we don'\''t need to check it.\".", 1 },';
		echo '+    { "Ken Thomases", "Revert \"wined3d: Track if a context'\''s hdc is private so we never need to restore its pixel format.\".", 1 },';
		echo '+    { "Ken Thomases", "Revert \"wined3d: When restoring pixel format in context_release(), mark the context as needing to be set on the next context_acquire().\".", 1 },';
		echo '+    { "Ken Thomases", "Revert \"wined3d: Don'\''t call GetPixelFormat() to set a flag that'\''s already set.\".", 1 },';
		echo '+    { "Ken Thomases", "Revert \"wined3d: Restore the pixel format of the window whose pixel format was actually changed.\".", 1 },';
		echo '+    { "Ken Thomases", "d3d8: Mark tests which no longer pass due to reverts as todo_wine.", 1 },';
		echo '+    { "Ken Thomases", "d3d9: Mark tests which no longer pass due to reverts as todo_wine.", 1 },';
		echo '+    { "Ken Thomases", "ddraw: Mark tests which no longer pass due to reverts as todo_wine.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-UnhandledBlendFactor
# |
# | Modified files:
# |   *	dlls/wined3d/state.c
# |
if test "$enable_wined3d_UnhandledBlendFactor" -eq 1; then
	patch_apply wined3d-UnhandledBlendFactor/0001-wined3d-Silence-repeated-Unhandled-blend-factor-0-me.patch
	(
		echo '+    { "Sebastian Lackner", "wined3d: Silence repeated '\''Unhandled blend factor 0'\'' messages.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-resource_check_usage
# |
# | Modified files:
# |   *	dlls/wined3d/resource.c
# |
if test "$enable_wined3d_resource_check_usage" -eq 1; then
	patch_apply wined3d-resource_check_usage/0001-wined3d-Silence-repeated-resource_check_usage-FIXME.patch
	(
		echo '+    { "Erich E. Hoover", "wined3d: Silence repeated resource_check_usage FIXME.", 2 },';
	) >> "$patchlist"
fi

# Patchset wined3d-wined3d_swapchain_present
# |
# | Modified files:
# |   *	dlls/wined3d/swapchain.c
# |
if test "$enable_wined3d_wined3d_swapchain_present" -eq 1; then
	patch_apply wined3d-wined3d_swapchain_present/0001-wined3d-Silence-repeated-wined3d_swapchain_present-F.patch
	(
		echo '+    { "Sebastian Lackner", "wined3d: Silence repeated wined3d_swapchain_present FIXME.", 1 },';
	) >> "$patchlist"
fi

# Patchset wined3d-CSMT_Main
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11674] Support for CSMT (command stream) to increase graphic performance
# |
# | Modified files:
# |   *	dlls/d3d8/tests/visual.c, dlls/d3d9/tests/visual.c, dlls/wined3d/arb_program_shader.c, dlls/wined3d/buffer.c,
# | 	dlls/wined3d/context.c, dlls/wined3d/cs.c, dlls/wined3d/device.c, dlls/wined3d/directx.c, dlls/wined3d/drawprim.c,
# | 	dlls/wined3d/glsl_shader.c, dlls/wined3d/query.c, dlls/wined3d/resource.c, dlls/wined3d/shader.c, dlls/wined3d/state.c,
# | 	dlls/wined3d/stateblock.c, dlls/wined3d/surface.c, dlls/wined3d/swapchain.c, dlls/wined3d/texture.c,
# | 	dlls/wined3d/utils.c, dlls/wined3d/vertexdeclaration.c, dlls/wined3d/view.c, dlls/wined3d/volume.c,
# | 	dlls/wined3d/wined3d_main.c, dlls/wined3d/wined3d_private.h, dlls/winex11.drv/opengl.c
# |
if test "$enable_wined3d_CSMT_Main" -eq 1; then
	patch_apply wined3d-CSMT_Main/0001-wined3d-Pass-a-context-to-surface_load_sysmem.patch
	patch_apply wined3d-CSMT_Main/0002-wined3d-Pass-a-context-to-read_from_framebuffer.patch
	patch_apply wined3d-CSMT_Main/0003-wined3d-Pass-a-context-to-surface_load_drawable-and-.patch
	patch_apply wined3d-CSMT_Main/0004-wined3d-Pass-a-context-to-surface_blt_fbo.patch
	patch_apply wined3d-CSMT_Main/0005-wined3d-Pass-a-context-to-surface_multisample_resolv.patch
	patch_apply wined3d-CSMT_Main/0006-wined3d-Pass-a-context-to-surface_load_texture.patch
	patch_apply wined3d-CSMT_Main/0007-wined3d-Pass-a-context-to-surface_load_location.patch
	patch_apply wined3d-CSMT_Main/0008-wined3d-Make-surface_load_location-return-nothing.patch
	patch_apply wined3d-CSMT_Main/0009-wined3d-Store-volume-locations-in-the-resource.patch
	patch_apply wined3d-CSMT_Main/0010-wined3d-Move-validate_location-to-resource.c.patch
	patch_apply wined3d-CSMT_Main/0011-wined3d-Move-surface-locations-into-the-resource.patch
	patch_apply wined3d-CSMT_Main/0012-wined3d-Remove-surface_validate_location.patch
	patch_apply wined3d-CSMT_Main/0013-wined3d-Move-invalidate_location-to-resource.c.patch
	patch_apply wined3d-CSMT_Main/0014-wined3d-Invalidate-containers-via-callback.patch
	patch_apply wined3d-CSMT_Main/0015-wined3d-Remove-surface_invalidate_location.patch
	patch_apply wined3d-CSMT_Main/0016-wined3d-Move-bitmap_data-and-user_memory-into-the-re.patch
	patch_apply wined3d-CSMT_Main/0017-wined3d-Move-load_location-into-the-resource.patch
	patch_apply wined3d-CSMT_Main/0018-wined3d-Replace-surface_load_location-with-resource_.patch
	patch_apply wined3d-CSMT_Main/0019-wined3d-Introduce-helper-functions-for-mapping-volum.patch
	patch_apply wined3d-CSMT_Main/0020-wined3d-Move-volume-PBO-infrastructure-into-the-reso.patch
	patch_apply wined3d-CSMT_Main/0021-wined3d-Remove-surface-pbo.patch
	patch_apply wined3d-CSMT_Main/0022-wined3d-Use-resource-buffer-mapping-facilities-in-su.patch
	patch_apply wined3d-CSMT_Main/0023-wined3d-Move-buffer-creation-into-the-resource.patch
	patch_apply wined3d-CSMT_Main/0024-wined3d-Handle-WINED3D_LOCATION_DISCARDED-in-surface.patch
	patch_apply wined3d-CSMT_Main/0025-wined3d-Handle-LOCATION_DISCARDED-in-surface_load_dr.patch
	patch_apply wined3d-CSMT_Main/0026-wined3d-Handle-WINED3D_LOCATION_DISCARDED-for-sysmem.patch
	patch_apply wined3d-CSMT_Main/0027-wined3d-Discard-implicit-surfaces-on-unload.patch
	patch_apply wined3d-CSMT_Main/0028-wined3d-Don-t-try-to-flip-sysmem-copies-in-swapchain.patch
	patch_apply wined3d-CSMT_Main/0029-wined3d-Discard-the-backbuffer-in-discard-presents.patch
	patch_apply wined3d-CSMT_Main/0030-wined3d-Allocate-sysmem-for-client-storage-if-it-doe.patch
	patch_apply wined3d-CSMT_Main/0031-wined3d-Introduce-a-function-to-retrieve-resource-me.patch
	patch_apply wined3d-CSMT_Main/0032-wined3d-Make-surface_ops-unmap-specific-for-front-bu.patch
	patch_apply wined3d-CSMT_Main/0033-wined3d-Move-check_block_align-to-resource.c.patch
	patch_apply wined3d-CSMT_Main/0034-wined3d-Replace-surface-alloc-functions-with-resourc.patch
	patch_apply wined3d-CSMT_Main/0035-wined3d-Don-t-delete-the-buffer-in-surface_cleanup.patch
	patch_apply wined3d-CSMT_Main/0036-wined3d-Use-resource-facilities-to-destroy-PBOs.patch
	patch_apply wined3d-CSMT_Main/0037-wined3d-Move-simple-location-copying-to-the-resource.patch
	patch_apply wined3d-CSMT_Main/0038-wined3d-Move-most-of-volume_map-to-resource.c.patch
	patch_apply wined3d-CSMT_Main/0039-wined3d-Use-resource_map-for-surface_map.patch
	patch_apply wined3d-CSMT_Main/0040-wined3d-Use-client-storage-with-DIB-sections.patch
	patch_apply wined3d-CSMT_Main/0041-wined3d-Don-t-call-the-public-map-function-in-surfac.patch
	patch_apply wined3d-CSMT_Main/0042-wined3d-Don-t-call-the-public-map-function-in-surfac.patch
	patch_apply wined3d-CSMT_Main/0043-wined3d-Move-the-framebuffer-into-wined3d_state.patch
	patch_apply wined3d-CSMT_Main/0044-wined3d-Get-rid-of-state-access-in-shader_generate_g.patch
	patch_apply wined3d-CSMT_Main/0045-wined3d-Preload-buffers-if-streamsrc-is-not-dirty.patch
	patch_apply wined3d-CSMT_Main/0046-wined3d-Hackily-introduce-a-multithreaded-command-st.patch
	patch_apply wined3d-CSMT_Main/0047-wined3d-Wait-for-resource-updates-to-finish-when-usi.patch
	patch_apply wined3d-CSMT_Main/0048-wined3d-Don-t-store-pointers-in-struct-wined3d_cs_pr.patch
	patch_apply wined3d-CSMT_Main/0049-wined3d-Don-t-put-rectangle-pointers-into-wined3d_cs.patch
	patch_apply wined3d-CSMT_Main/0050-wined3d-Store-the-color-in-clear-ops-instead-of-a-po.patch
	patch_apply wined3d-CSMT_Main/0051-wined3d-Pass-the-state-to-draw_primitive.patch
	patch_apply wined3d-CSMT_Main/0052-wined3d-Wait-for-the-cs-before-destroying-objects.patch
	patch_apply wined3d-CSMT_Main/0053-wined3d-Give-the-cs-its-own-state.patch
	patch_apply wined3d-CSMT_Main/0054-wined3d-Send-float-constant-updates-through-the-comm.patch
	patch_apply wined3d-CSMT_Main/0055-wined3d-Request-a-glFinish-before-modifying-resource.patch
	patch_apply wined3d-CSMT_Main/0056-wined3d-Finish-the-cs-before-changing-the-texture-lo.patch
	patch_apply wined3d-CSMT_Main/0057-wined3d-Don-t-call-glFinish-after-clears.patch
	patch_apply wined3d-CSMT_Main/0058-wined3d-Don-t-call-glFinish-after-draws.patch
	patch_apply wined3d-CSMT_Main/0059-wined3d-Shadow-device-offscreenBuffer-in-the-context.patch
	patch_apply wined3d-CSMT_Main/0060-wined3d-Don-t-access-the-stateblock-in-find_draw_buf.patch
	patch_apply wined3d-CSMT_Main/0061-wined3d-Pass-the-depth-stencil-to-swapchain-present.patch
	patch_apply wined3d-CSMT_Main/0062-wined3d-Don-t-store-viewport-pointers-in-the-command.patch
	patch_apply wined3d-CSMT_Main/0063-wined3d-Keep-track-of-the-onscreen-depth-stencil-in-.patch
	patch_apply wined3d-CSMT_Main/0064-wined3d-Send-base-vertex-index-updates-through-the-c.patch
	patch_apply wined3d-CSMT_Main/0065-wined3d-Send-primitive-type-updates-through-the-comm.patch
	patch_apply wined3d-CSMT_Main/0066-wined3d-Send-bool-constant-updates-through-the-comma.patch
	patch_apply wined3d-CSMT_Main/0067-wined3d-Send-int-constant-updates-through-the-comman.patch
	patch_apply wined3d-CSMT_Main/0068-wined3d-Send-light-updates-through-the-command-strea.patch
	patch_apply wined3d-CSMT_Main/0069-wined3d-Prevent-the-command-stream-from-running-ahea.patch
	patch_apply wined3d-CSMT_Main/0070-wined3d-Wait-for-the-cs-to-finish-before-destroying-.patch
	patch_apply wined3d-CSMT_Main/0071-wined3d-Run-the-cs-asynchronously.patch
	patch_apply wined3d-CSMT_Main/0072-wined3d-Send-blits-through-the-command-stream.patch
	patch_apply wined3d-CSMT_Main/0073-wined3d-Put-update_surface-checks-back-in-place.patch
	patch_apply wined3d-CSMT_Main/0074-wined3d-Get-rid-of-WINED3D_BUFFER_FLUSH.patch
	patch_apply wined3d-CSMT_Main/0075-wined3d-Add-cs-waiting-debug-code.patch
	patch_apply wined3d-CSMT_Main/0076-wined3d-Don-t-force-strict-draw-ordering-for-multith.patch
	patch_apply wined3d-CSMT_Main/0077-wined3d-Send-render-target-view-clears-through-the-c.patch
	patch_apply wined3d-CSMT_Main/0078-wined3d-Wait-for-the-CS-in-GetDC.patch
	patch_apply wined3d-CSMT_Main/0079-wined3d-send-resource-maps-through-the-command-strea.patch
	patch_apply wined3d-CSMT_Main/0080-wined3d-Get-rid-of-the-end_scene-flush-and-finish.patch
	patch_apply wined3d-CSMT_Main/0081-wined3d-Replace-the-linked-lists-with-a-ringbuffer.patch
	patch_apply wined3d-CSMT_Main/0082-wined3d-Don-t-preload-buffers-on-unmap.patch
	patch_apply wined3d-CSMT_Main/0083-wined3d-Don-t-call-glFinish-before-swapping.patch
	patch_apply wined3d-CSMT_Main/0084-wined3d-wined3d_-_query_issue-never-fails.patch
	patch_apply wined3d-CSMT_Main/0085-wined3d-Add-query-support-to-the-command-stream.patch
	patch_apply wined3d-CSMT_Main/0086-wined3d-Check-our-CS-state-to-find-out-if-a-query-is.patch
	patch_apply wined3d-CSMT_Main/0087-wined3d-Poll-queries-automatically-in-the-CS.patch
	patch_apply wined3d-CSMT_Main/0088-wined3d-Introduce-a-separate-queue-for-priority-comm.patch
	patch_apply wined3d-CSMT_Main/0089-wined3d-Destroy-queries-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0090-wined3d-Separate-main-and-worker-thread-query-state.patch
	patch_apply wined3d-CSMT_Main/0091-wined3d-Don-t-poll-queries-that-failed-to-start.patch
	patch_apply wined3d-CSMT_Main/0092-wined3d-Remove-restated-queries-from-the-poll-list.patch
	patch_apply wined3d-CSMT_Main/0093-wined3d-Don-t-reset-the-query-state-if-it-doesn-t-ha.patch
	patch_apply wined3d-CSMT_Main/0094-wined3d-Put-this-into-the-query-poll-patch.patch
	patch_apply wined3d-CSMT_Main/0095-wined3d-Send-update_surface-commands-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0096-wined3d-Send-texture-preloads-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0097-wined3d-Send-surface-preloads-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0098-wined3d-Send-update_texture-calls-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0099-wined3d-Get-rid-of-the-surface_upload_data-glFinish.patch
	patch_apply wined3d-CSMT_Main/0100-wined3d-Don-t-lock-the-src-volume-in-device_update_v.patch
	patch_apply wined3d-CSMT_Main/0101-wined3d-Handle-evit_managed_resources-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0102-wined3d-Send-flips-through-the-command-stream.patch
	patch_apply wined3d-CSMT_Main/0103-wined3d-Introduce-resource-fencing.patch
	patch_apply wined3d-CSMT_Main/0104-wined3d-Fence-update_texture-and-update_surface-call.patch
	patch_apply wined3d-CSMT_Main/0105-wined3d-Dirtify-resources-on-unmap.patch
	patch_apply wined3d-CSMT_Main/0106-wined3d-Fence-texture-reads-in-draws.patch
	patch_apply wined3d-CSMT_Main/0107-wined3d-Fence-render-targets-and-depth-stencils.patch
	patch_apply wined3d-CSMT_Main/0108-wined3d-Fence-blit-operations.patch
	patch_apply wined3d-CSMT_Main/0109-wined3d-Fence-color_fill-operations.patch
	patch_apply wined3d-CSMT_Main/0110-wined3d-Fence-clear-calls.patch
	patch_apply wined3d-CSMT_Main/0111-wined3d-Fence-present-calls.patch
	patch_apply wined3d-CSMT_Main/0112-wined3d-Make-resource-maps-and-unmaps-a-priority-com.patch
	patch_apply wined3d-CSMT_Main/0113-wined3d-Dirtify-changed-textures-through-the-command.patch
	patch_apply wined3d-CSMT_Main/0114-wined3d-Wrap-GL-BOs-in-a-structure.patch
	patch_apply wined3d-CSMT_Main/0115-wined3d-Separate-resource-map-and-draw-buffers.patch
	patch_apply wined3d-CSMT_Main/0116-wined3d-Implement-DISCARD-resource-maps-with-buffers.patch
	patch_apply wined3d-CSMT_Main/0117-wined3d-Implement-DISCARD-resource-maps-with-heap-me.patch
	patch_apply wined3d-CSMT_Main/0118-wined3d-Unset-some-objects-in-state_init_default.patch
	patch_apply wined3d-CSMT_Main/0119-wined3d-Don-t-request-the-frontbuffer-to-create-dumm.patch
	patch_apply wined3d-CSMT_Main/0120-wined3d-Use-double-buffered-buffers-for-multithreade.patch
	patch_apply wined3d-CSMT_Main/0121-wined3d-Don-t-synchronize-NOOVERWRITE-buffer-maps.patch
	patch_apply wined3d-CSMT_Main/0122-wined3d-Separate-buffer-map-write-and-draw-read-memo.patch
	patch_apply wined3d-CSMT_Main/0123-wined3d-Accelerate-DISCARD-buffer-maps.patch
	patch_apply wined3d-CSMT_Main/0124-wined3d-Accelerate-READONLY-buffer-maps.patch
	patch_apply wined3d-CSMT_Main/0125-wined3d-Access-the-buffer-dirty-areas-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0126-wined3d-Ignore-buffer-resource.map_count-in-the-CS.patch
	patch_apply wined3d-CSMT_Main/0127-wined3d-Send-buffer-preloads-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0128-wined3d-Use-glBufferSubData-instead-of-glMapBufferRa.patch
	patch_apply wined3d-CSMT_Main/0129-wined3d-Separate-GL-buffer-discard-control-from-igno.patch
	patch_apply wined3d-CSMT_Main/0130-wined3d-Create-buffers-before-mapping-them.patch
	patch_apply wined3d-CSMT_Main/0131-wined3d-Destroy-views-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0132-wined3d-Remove-another-glFinish.patch
	patch_apply wined3d-CSMT_Main/0133-wined3d-Destroy-vertex-declarations-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0134-wined3d-Destroy-shaders-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0135-wined3d-Create-VBOs-through-the-command-stream.patch
	patch_apply wined3d-CSMT_Main/0136-wined3d-Clean-up-resource-data-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0137-wined3d-Clean-up-buffer-resource-data-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0138-wined3d-Clean-up-volume-resource-data-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0139-wined3d-Clean-up-surfaces-through-the-cs.patch
	patch_apply wined3d-CSMT_Main/0140-wined3d-Clean-up-texture-resources-through-the-cs.patch
	patch_apply wined3d-CSMT_Main/0141-wined3d-Unload-resources-through-the-CS-in-uninit_3d.patch
	patch_apply wined3d-CSMT_Main/0142-wined3d-Unload-resources-through-the-CS-in-device_re.patch
	patch_apply wined3d-CSMT_Main/0143-wined3d-Don-t-glFinish-after-a-depth-buffer-blit.patch
	patch_apply wined3d-CSMT_Main/0144-wined3d-Remove-software-cursor-support.patch
	patch_apply wined3d-CSMT_Main/0145-wined3d-Create-dummy-textures-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0146-wined3d-Create-the-initial-context-through-the-CS.patch
	patch_apply wined3d-CSMT_Main/0147-wined3d-Recreate-ctx-and-dummy-textures-through-the-.patch
	patch_apply wined3d-CSMT_Main/0148-wined3d-Ignore-WINED3D_MAP_NO_DIRTY_UPDATE-in-resour.patch
	patch_apply wined3d-CSMT_Main/0149-wined3d-Delete-GL-contexts-through-the-CS-in-reset.patch
	patch_apply wined3d-CSMT_Main/0150-wined3d-Delete-GL-contexts-through-the-CS-in-uninit_.patch
	patch_apply wined3d-CSMT_Main/0151-wined3d-Invoke-surface_unload-through-the-CS-in-wine.patch
	patch_apply wined3d-CSMT_Main/0152-wined3d-Use-an-event-to-block-the-worker-thread-when.patch
	patch_apply wined3d-CSMT_Main/0153-wined3d-Fence-preload-operations.patch
	patch_apply wined3d-CSMT_Main/0154-d3d8-tests-D3DLOCK_NO_DIRTY_UPDATE-on-managed-textur.patch
	patch_apply wined3d-CSMT_Main/0155-d3d9-tests-D3DLOCK_NO_DIRTY_UPDATE-on-managed-textur.patch
	patch_apply wined3d-CSMT_Main/0156-wined3d-Completely-reset-the-state-on-reset.patch
	patch_apply wined3d-CSMT_Main/0157-wined3d-Send-getdc-and-releasedc-through-the-command.patch
	patch_apply wined3d-CSMT_Main/0158-wined3d-Set-map_heap_memory-NULL-when-allocating-a-P.patch
	patch_apply wined3d-CSMT_Main/0159-wined3d-Wait-only-for-the-buffer-to-be-idle.patch
	patch_apply wined3d-CSMT_Main/0160-wined3d-Add-a-comment-about-worker-thread-lag.patch
	patch_apply wined3d-CSMT_Main/0161-wined3d-Remove-the-texture-destroy-glFinish.patch
	patch_apply wined3d-CSMT_Main/0162-wined3d-Move-FBO-destruction-into-the-worker-thread.patch
	patch_apply wined3d-CSMT_Main/0163-wined3d-Don-t-incref-decref-textures-in-color-depth-.patch
	patch_apply wined3d-CSMT_Main/0164-Winex11-complain-about-glfinish.patch
	patch_apply wined3d-CSMT_Main/0165-wined3d-Make-sure-the-new-window-is-set-up-before-se.patch
	patch_apply wined3d-CSMT_Main/0166-wined3d-Remove-the-device_reset-CS-sync-fixme.patch
	patch_apply wined3d-CSMT_Main/0167-wined3d-Put-GL_APPLE_flush_buffer_range-syncing-back.patch
	patch_apply wined3d-CSMT_Main/0168-wined3d-Wait-for-the-resource-to-be-idle-when-destro.patch
	patch_apply wined3d-CSMT_Main/0169-wined3d-Don-t-sync-on-redundant-discard-calls.patch
	patch_apply wined3d-CSMT_Main/0170-wined3d-Don-t-discard-new-buffers.patch
	patch_apply wined3d-CSMT_Main/0171-wined3d-Don-t-try-to-sync-VBOs-manually-on-OSX-with-.patch
	patch_apply wined3d-CSMT_Main/0172-wined3d-Render-target-lock-hack.patch
	patch_apply wined3d-CSMT_Main/0173-wined3d-Avoid-calling-wined3d_surface_blt-from-surfa.patch
	patch_apply wined3d-CSMT_Main/0174-wined3d-Enable-CSMT-by-default-print-a-winediag-mess.patch
	patch_apply wined3d-CSMT_Main/9999-IfDefined.patch
	(
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_load_sysmem.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to read_from_framebuffer.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_load_drawable and surface_blt_to_drawable.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_blt_fbo.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_multisample_resolve.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_load_texture.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass a context to surface_load_location.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Make surface_load_location return nothing.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Store volume locations in the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move validate_location to resource.c.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move surface locations into the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove surface_validate_location.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move invalidate_location to resource.c.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Invalidate containers via callback.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove surface_invalidate_location.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move bitmap_data and user_memory into the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move load_location into the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Replace surface_load_location with resource_load_location.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Introduce helper functions for mapping volumes.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move volume PBO infrastructure into the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove surface->pbo.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use resource buffer mapping facilities in surfaces.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move buffer creation into the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Handle WINED3D_LOCATION_DISCARDED in surface_load_texture.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Handle LOCATION_DISCARDED in surface_load_drawable.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Handle WINED3D_LOCATION_DISCARDED for sysmem loads.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Discard implicit surfaces on unload.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t try to flip sysmem copies in swapchain_present.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Discard the backbuffer in discard presents.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Allocate sysmem for client storage if it doesn'\''t exist already.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Introduce a function to retrieve resource memory.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Make surface_ops->unmap specific for front buffers.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move check_block_align to resource.c.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Replace surface alloc functions with resource ones.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t delete the buffer in surface_cleanup.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use resource facilities to destroy PBOs.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move simple location copying to the resource.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move most of volume_map to resource.c.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use resource_map for surface_map.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use client storage with DIB sections.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t call the public map function in surface_convert_format.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t call the public map function in surface_cpu_blt.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move the framebuffer into wined3d_state.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Get rid of state access in shader_generate_glsl_declarations.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Preload buffers if streamsrc is not dirty.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Hackily introduce a multithreaded command stream.", 1 },';
		echo '+    { "Henri Verbeet", "wined3d: Wait for resource updates to finish when using the multithreaded command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t store pointers in struct wined3d_cs_present.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t put rectangle pointers into wined3d_cs_clear.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Store the color in clear ops instead of a pointer.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass the state to draw_primitive.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wait for the cs before destroying objects.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Give the cs its own state.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send float constant updates through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Request a glFinish before modifying resources outside the cs.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Finish the cs before changing the texture lod.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t call glFinish after clears.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t call glFinish after draws.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Shadow device->offscreenBuffer in the context.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t access the stateblock in find_draw_buffers_mask.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Pass the depth stencil to swapchain->present.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t store viewport pointers in the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Keep track of the onscreen depth stencil in the command stream instead of the device.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send base vertex index updates through the cs.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send primitive type updates through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send bool constant updates through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send int constant updates through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send light updates through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Prevent the command stream from running ahead too far.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wait for the cs to finish before destroying the device.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Run the cs asynchronously.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send blits through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Put update_surface checks back in place.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Get rid of WINED3D_BUFFER_FLUSH.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Add cs waiting debug code.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t force strict draw ordering for multithreaded CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send render target view clears through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wait for the CS in GetDC.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: send resource maps through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Get rid of the end_scene flush and finish.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Replace the linked lists with a ringbuffer.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t preload buffers on unmap.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t call glFinish before swapping.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: wined3d_*_query_issue never fails.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Add query support to the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Check our CS state to find out if a query is done.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Poll queries automatically in the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Introduce a separate queue for priority commands.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Destroy queries through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Separate main and worker thread query state.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t poll queries that failed to start.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove restated queries from the poll list.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t reset the query state if it doesn'\''t have a ctx.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Put this into the query poll patch.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send update_surface commands through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send texture preloads through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send surface preloads through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send update_texture calls through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Get rid of the surface_upload_data glFinish.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t lock the src volume in device_update_volume.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Handle evit_managed_resources through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send flips through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Introduce resource fencing.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence update_texture and update_surface calls.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Dirtify resources on unmap.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence texture reads in draws.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence render targets and depth stencils.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence blit operations.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence color_fill operations.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence clear calls.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence present calls.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Make resource maps and unmaps a priority command.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Dirtify changed textures through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wrap GL BOs in a structure.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Separate resource map and draw buffers.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Implement DISCARD resource maps with buffers.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Implement DISCARD resource maps with heap memory.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Unset some objects in state_init_default.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t request the frontbuffer to create dummy textures.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use double-buffered buffers for multithreaded CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t synchronize NOOVERWRITE buffer maps.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Separate buffer map write and draw read memory pointers.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Accelerate DISCARD buffer maps.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Accelerate READONLY buffer maps.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Access the buffer dirty areas through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Ignore buffer->resource.map_count in the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send buffer preloads through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use glBufferSubData instead of glMapBufferRange.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Separate GL buffer discard control from ignoring MAP_DISCARD.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Create buffers before mapping them.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Destroy views through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove another glFinish.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Destroy vertex declarations through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Destroy shaders through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Create VBOs through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Clean up resource data through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Clean up buffer resource data through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Clean up volume resource data through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Clean up surfaces through the cs.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Clean up texture resources through the cs.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Unload resources through the CS in uninit_3d.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Unload resources through the CS in device_reset.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t glFinish after a depth buffer blit.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove software cursor support.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Create dummy textures through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Create the initial context through the CS.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Recreate ctx and dummy textures through the CS after resets.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Ignore WINED3D_MAP_NO_DIRTY_UPDATE in resource_map.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Delete GL contexts through the CS in reset.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Delete GL contexts through the CS in uninit_3d.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Invoke surface_unload through the CS in wined3d_surface_update_desc.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Use an event to block the worker thread when it is idle.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Fence preload operations.", 1 },';
		echo '+    { "Stefan Dösinger", "d3d8/tests: D3DLOCK_NO_DIRTY_UPDATE on managed textures is temporarily broken.", 1 },';
		echo '+    { "Stefan Dösinger", "d3d9/tests: D3DLOCK_NO_DIRTY_UPDATE on managed textures is temporarily broken.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Completely reset the state on reset.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Send getdc and releasedc through the command stream.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Set map_heap_memory = NULL when allocating a PBO.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wait only for the buffer to be idle.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Add a comment about worker thread lag.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove the texture destroy glFinish.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Move FBO destruction into the worker thread.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t incref / decref textures in color / depth fill blits.", 1 },';
		echo '+    { "Stefan Dösinger", "Winex11: complain about glfinish.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Make sure the new window is set up before setting up a context.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Remove the device_reset CS sync fixme.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Put GL_APPLE_flush_buffer_range syncing back in place.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Wait for the resource to be idle when destroying user memory surfaces.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t sync on redundant discard calls.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t discard new buffers.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Don'\''t try to sync VBOs manually on OSX with CSMT.", 1 },';
		echo '+    { "Stefan Dösinger", "wined3d: Render target lock hack.", 1 },';
		echo '+    { "Matteo Bruni", "wined3d: Avoid calling wined3d_surface_blt() from surface_upload_from_surface().", 1 },';
		echo '+    { "Sebastian Lackner", "wined3d: Enable CSMT by default, print a winediag message informing about this patchset.", 1 },';
		echo '+    { "Wine Staging Team", "Autogenerated #ifdef patch for wined3d-CSMT_Main.", 1 },';
	) >> "$patchlist"
fi

# Patchset winedevice-Fix_Relocation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28254] Fix crash of winedevice when relocation entry crosses page boundary
# |
# | Modified files:
# |   *	programs/winedevice/device.c
# |
if test "$enable_winedevice_Fix_Relocation" -eq 1; then
	patch_apply winedevice-Fix_Relocation/0001-winedevice-Avoid-invalid-memory-access-when-relocati.patch
	(
		echo '+    { "Sebastian Lackner", "winedevice: Avoid invalid memory access when relocation block addresses memory outside of the current page.", 1 },';
	) >> "$patchlist"
fi

# Patchset winemenubuilder-Desktop_Icon_Path
# |
# | Modified files:
# |   *	programs/winemenubuilder/Makefile.in, programs/winemenubuilder/winemenubuilder.c
# |
if test "$enable_winemenubuilder_Desktop_Icon_Path" -eq 1; then
	patch_apply winemenubuilder-Desktop_Icon_Path/0001-winemenubuilder-Create-desktop-shortcuts-with-absolu.patch
	(
		echo '+    { "Sebastian Lackner", "winemenubuilder: Create desktop shortcuts with absolute wine path.", 1 },';
	) >> "$patchlist"
fi

# Patchset winepulse-PulseAudio_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#10495] Support for PulseAudio backend for audio
# |   *	[#37042] Implement exclusive mode in PulseAudio backend
# |
# | Modified files:
# |   *	configure.ac, dlls/mmdevapi/main.c, dlls/mmdevapi/tests/render.c, dlls/winepulse.drv/Makefile.in,
# | 	dlls/winepulse.drv/mmdevdrv.c, dlls/winepulse.drv/winepulse.drv.spec
# |
if test "$enable_winepulse_PulseAudio_Support" -eq 1; then
	patch_apply winepulse-PulseAudio_Support/0001-winepulse-Add-initial-stub-for-pulseaudio-support.patch
	patch_apply winepulse-PulseAudio_Support/0002-winepulse-Add-format-and-period-probing.patch
	patch_apply winepulse-PulseAudio_Support/0003-winepulse-Add-audioclient.patch
	patch_apply winepulse-PulseAudio_Support/0004-winepulse-Add-IAudioRenderClient-and-IAudioCaptureCl.patch
	patch_apply winepulse-PulseAudio_Support/0005-winepulse-Add-IAudioClock-and-IAudioClock2.patch
	patch_apply winepulse-PulseAudio_Support/0006-winepulse-Add-audiostreamvolume.patch
	patch_apply winepulse-PulseAudio_Support/0007-winepulse-Add-session-support.patch
	patch_apply winepulse-PulseAudio_Support/0008-fix-fdels-trailing-whitespaces.patch
	patch_apply winepulse-PulseAudio_Support/0009-winepulse-v12.patch
	patch_apply winepulse-PulseAudio_Support/0010-winepulse-v15-Add-support-for-missing-formats-and-si.patch
	patch_apply winepulse-PulseAudio_Support/0011-winepulse-v16-Add-official-warning-wine-doesn-t-want.patch
	patch_apply winepulse-PulseAudio_Support/0012-winepulse-v17-Fix-winmm-tests.patch
	patch_apply winepulse-PulseAudio_Support/0013-winepulse-v18-Latency-and-compilation-improvements.patch
	patch_apply winepulse-PulseAudio_Support/0014-winepulse-API-Compatibility-with-1.5.2-onward-v2.patch
	patch_apply winepulse-PulseAudio_Support/0015-winepulse-Fix-low-latency-support.patch
	patch_apply winepulse-PulseAudio_Support/0016-winepulse-drop-realtime-priority-before-thread-destr.patch
	patch_apply winepulse-PulseAudio_Support/0017-winepulse-remove-bogus-SetEvent-from-pulse_started_c.patch
	patch_apply winepulse-PulseAudio_Support/0018-winepulse-disable-the-setevent-part-of-the-latency-h.patch
	patch_apply winepulse-PulseAudio_Support/0019-winepulse-v20-fix-the-checks-in-IsFormatSupported.patch
	patch_apply winepulse-PulseAudio_Support/0020-winepulse-fixup-IsFormatSupported-calls.patch
	patch_apply winepulse-PulseAudio_Support/0021-winepulse-v21-return-early-if-padding-didn-t-update.patch
	patch_apply winepulse-PulseAudio_Support/0022-winepulse-fix-unneeded-free-in-write.patch
	patch_apply winepulse-PulseAudio_Support/0023-winepulse-v23-fixup-a-invalid-free-in-mmdevapi.patch
	patch_apply winepulse-PulseAudio_Support/0024-winepulse-use-a-pi-mutex-for-serialization.patch
	patch_apply winepulse-PulseAudio_Support/0025-winepulse-add-support-for-IMarshal.patch
	patch_apply winepulse-PulseAudio_Support/0026-winepulse-handle-stream-create-failing-correctly.patch
	patch_apply winepulse-PulseAudio_Support/0027-winepulse-Trivial-cleanups-and-changes-for-consisten.patch
	patch_apply winepulse-PulseAudio_Support/0028-winepulse-Sync-default-channel-masks-with-other-driv.patch
	patch_apply winepulse-PulseAudio_Support/0029-winepulse-In-Shared-mode-track-device-position-in-by.patch
	patch_apply winepulse-PulseAudio_Support/0030-winepulse-Always-mute-buffer.patch
	patch_apply winepulse-PulseAudio_Support/0031-winepulse-Remove-volume-support.patch
	patch_apply winepulse-PulseAudio_Support/0032-winepulse-Forward-winmm-functions-to-winealsa.patch
	patch_apply winepulse-PulseAudio_Support/0033-winepulse-expose-audio-devices-directly-to-programs.patch
	patch_apply winepulse-PulseAudio_Support/0034-winepulse-implement-exclusive-mode.patch
	patch_apply winepulse-PulseAudio_Support/0035-winepulse-fix-segfault-in-pulse_rd_loop.patch
	patch_apply winepulse-PulseAudio_Support/0036-winepulse-implement-GetPropValue.patch
	patch_apply winepulse-PulseAudio_Support/0037-winepulse-fetch-actual-program-name-if-possible.patch
	patch_apply winepulse-PulseAudio_Support/0038-winepulse-return-PKEY_AudioEndpoint_PhysicalSpeakers.patch
	(
		echo '+    { "Maarten Lankhorst", "winepulse: Add initial stub for pulseaudio support.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add format and period probing.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add audioclient.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add IAudioRenderClient and IAudioCaptureClient.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add IAudioClock and IAudioClock2.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add audiostreamvolume.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add session support.", 1 },';
		echo '+    { "Maarten Lankhorst", "fix fdels trailing whitespaces.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse.", 12 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add support for missing formats, and silence an error for missing format tags.", 15 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Add official warning wine doesn'\''t want to support winepulse.", 16 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Fix winmm tests.", 17 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Latency and compilation improvements.", 18 },';
		echo '+    { "Juergen Tretthahn", "winepulse: API Compatibility with 1.5.2 onward.", 2 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Fix low latency support.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: drop realtime priority before thread destruction.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: remove bogus SetEvent from pulse_started_callback.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: disable the setevent part of the latency hack.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: fix the checks in IsFormatSupported.", 20 },';
		echo '+    { "Maarten Lankhorst", "winepulse: fixup IsFormatSupported calls.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: return early if padding didn'\''t update.", 21 },';
		echo '+    { "Maarten Lankhorst", "winepulse: fix unneeded free in write.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: fixup a invalid free in mmdevapi.", 23 },';
		echo '+    { "Maarten Lankhorst", "winepulse: use a pi-mutex for serialization.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: add support for IMarshal.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: handle stream create failing correctly.", 1 },';
		echo '+    { "Andrew Eikum", "winepulse: Trivial cleanups and changes for consistency with other drivers.", 1 },';
		echo '+    { "Andrew Eikum", "winepulse: Sync default channel masks with other drivers.", 1 },';
		echo '+    { "Andrew Eikum", "winepulse: In Shared mode, track device position in bytes.", 1 },';
		echo '+    { "Andrew Eikum", "winepulse: Always mute buffer.", 1 },';
		echo '+    { "Andrew Eikum", "winepulse: Remove volume support.", 1 },';
		echo '+    { "Maarten Lankhorst", "winepulse: Forward winmm functions to winealsa.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: expose audio devices directly to programs.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: implement exclusive mode.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: fix segfault in pulse_rd_loop.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: implement GetPropValue.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: fetch actual program name if possible.", 1 },';
		echo '+    { "Mark Harmstone", "winepulse: return PKEY_AudioEndpoint_PhysicalSpeakers device prop.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-CandidateWindowPos
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30938] Update a XIM candidate position when cursor location changes
# |
# | Modified files:
# |   *	dlls/user32/caret.c, dlls/user32/driver.c, dlls/user32/user_private.h, dlls/winex11.drv/winex11.drv.spec,
# | 	dlls/winex11.drv/xim.c
# |
if test "$enable_winex11_CandidateWindowPos" -eq 1; then
	patch_apply winex11-CandidateWindowPos/0001-winex11.drv-Update-a-candidate-window-s-position-wit.patch
	(
		echo '+    { "Felix Yan", "winex11.drv: Update a candidate window'\''s position with over-the-spot style.", 2 },';
	) >> "$patchlist"
fi

# Patchset winex11-Clipboard_HTML
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7372] Support for pasting HTML from Unix applications
# |
# | Modified files:
# |   *	dlls/winex11.drv/clipboard.c
# |
if test "$enable_winex11_Clipboard_HTML" -eq 1; then
	patch_apply winex11-Clipboard_HTML/0001-winex11.drv-Import-X11-s-text-html-as-HTML-Format.patch
	(
		echo '+    { "Damjan Jovanovic", "winex11.drv: Import X11'\''s \"text/html\" as \"HTML Format\".", 3 },';
	) >> "$patchlist"
fi

# Patchset winex11-DragAndDrop
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29081] Only send WM_DROPFILES when OLE dnd fails
# |
# | Modified files:
# |   *	dlls/winex11.drv/xdnd.c
# |
if test "$enable_winex11_DragAndDrop" -eq 1; then
	patch_apply winex11-DragAndDrop/0001-winex11.drv-Only-send-WM_DROPFILES-when-OLE-dnd-fail.patch
	(
		echo '+    { "Damjan Jovanovic", "winex11.drv: Only send WM_DROPFILES when OLE dnd fails.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-Thread_Data
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36915] Fix crash in clip_cursor_notify caused by uninitialized TLS
# |
# | Modified files:
# |   *	dlls/winex11.drv/mouse.c
# |
if test "$enable_winex11_Thread_Data" -eq 1; then
	patch_apply winex11-Thread_Data/0001-winex11-Initialize-thread-data-in-clip_cursor_notify.patch
	(
		echo '+    { "Sebastian Lackner", "winex11: Initialize thread data in clip_cursor_notify.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-Window_Groups
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32699] Prevent window managers from grouping all wine programs together
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_Window_Groups" -eq 1; then
	patch_apply winex11-Window_Groups/0001-winex11-Prevent-window-managers-from-grouping-all-wi.patch
	(
		echo '+    { "Michael Müller", "winex11: Prevent window managers from grouping all wine programs together.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-Window_Style
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37876] Fix handling of window attributes for WS_EX_LAYERED | WS_EX_COMPOSITED
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_Window_Style" -eq 1; then
	patch_apply winex11-Window_Style/0001-winex11-Fix-handling-of-window-attributes-for-WS_EX_.patch
	(
		echo '+    { "Dmitry Timoshkov", "winex11: Fix handling of window attributes for WS_EX_LAYERED | WS_EX_COMPOSITED.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-XEMBED
# |
# | Modified files:
# |   *	dlls/winex11.drv/event.c
# |
if test "$enable_winex11_XEMBED" -eq 1; then
	patch_apply winex11-XEMBED/0001-winex11-Enable-disable-windows-when-they-are-un-mapped.patch
	(
		echo '+    { "Sebastian Lackner", "winex11: Enable/disable windows when they are (un)mapped by foreign applications.", 1 },';
	) >> "$patchlist"
fi

# Patchset winex11-wglShareLists
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11436] Do not fail when a used context is passed to wglShareLists
# |
# | Modified files:
# |   *	dlls/winex11.drv/opengl.c
# |
if test "$enable_winex11_wglShareLists" -eq 1; then
	patch_apply winex11-wglShareLists/0001-winex11.drv-Only-warn-about-used-contexts-in-wglShar.patch
	(
		echo '+    { "Michael Müller", "winex11.drv: Only warn about used contexts in wglShareLists.", 1 },';
	) >> "$patchlist"
fi

# Patchset winhttp-System_Proxy_Autoconfig
# |
# | Modified files:
# |   *	dlls/winhttp/session.c
# |
if test "$enable_winhttp_System_Proxy_Autoconfig" -eq 1; then
	patch_apply winhttp-System_Proxy_Autoconfig/0001-winhttp-Silence-repeated-no-support-on-this-platform.patch
	(
		echo '+    { "Jarkko Korpi", "winhttp: Silence repeated \"no support on this platform\" message.", 1 },';
	) >> "$patchlist"
fi

# Patchset wininet-Cleanup
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28911] Add HTTP Host header in HttpSendRequest instead of HttpOpenRequest
# |
# | Modified files:
# |   *	dlls/wininet/http.c, dlls/wininet/tests/http.c
# |
if test "$enable_wininet_Cleanup" -eq 1; then
	patch_apply wininet-Cleanup/0001-wininet-tests-Add-more-tests-for-cookies.patch
	patch_apply wininet-Cleanup/0002-wininet-tests-Test-auth-credential-reusage-with-host.patch
	patch_apply wininet-Cleanup/0003-wininet-tests-Check-cookie-behaviour-when-overriding.patch
	patch_apply wininet-Cleanup/0004-wininet-Use-request-server-name-when-processing-cook.patch
	patch_apply wininet-Cleanup/0005-wininet-Delay-setting-the-http-host-header.patch
	patch_apply wininet-Cleanup/0006-wininet-Use-request-server-canon_host_port-in-authen.patch
	patch_apply wininet-Cleanup/0007-wininet-Use-request-server-canon_host_port-when-quer.patch
	patch_apply wininet-Cleanup/0008-wininet-Strip-filename-if-no-path-is-set-in-cookie.patch
	patch_apply wininet-Cleanup/0009-wininet-Replacing-header-fields-should-fail-if-they-.patch
	(
		echo '+    { "Michael Müller", "wininet/tests: Add more tests for cookies.", 1 },';
		echo '+    { "Michael Müller", "wininet/tests: Test auth credential reusage with host override.", 1 },';
		echo '+    { "Michael Müller", "wininet/tests: Check cookie behaviour when overriding host.", 1 },';
		echo '+    { "Michael Müller", "wininet: Use request->server->name when processing cookies instead of Host header field.", 1 },';
		echo '+    { "Michael Müller", "wininet: Delay setting the http host header.", 1 },';
		echo '+    { "Michael Müller", "wininet: Use request->server->canon_host_port in authentication process.", 1 },';
		echo '+    { "Michael Müller", "wininet: Use request->server->canon_host_port when querying for INTERNET_OPTION_URL.", 1 },';
		echo '+    { "Michael Müller", "wininet: Strip filename if no path is set in cookie.", 1 },';
		echo '+    { "Michael Müller", "wininet: Replacing header fields should fail if they do not exist yet.", 1 },';
	) >> "$patchlist"
fi

# Patchset wininet-ParseX509EncodedCertificateForListBoxEntry
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29842] Add stub for wininet.ParseX509EncodedCertificateForListBoxEntry
# |
# | Modified files:
# |   *	dlls/wininet/dialogs.c, dlls/wininet/wininet.spec, include/winineti.h
# |
if test "$enable_wininet_ParseX509EncodedCertificateForListBoxEntry" -eq 1; then
	patch_apply wininet-ParseX509EncodedCertificateForListBoxEntry/0001-wininet-add-ParseX509EncodedCertificateForListBoxEnt.patch
	(
		echo '+    { "Austin English", "wininet: add ParseX509EncodedCertificateForListBoxEntry stub.", 2 },';
	) >> "$patchlist"
fi

# Patchset winmm-Delay_Import_Depends
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37983] Jedi Knight: Dark Forces II crashes with winmm set to native
# |
# | Modified files:
# |   *	dlls/winmm/Makefile.in
# |
if test "$enable_winmm_Delay_Import_Depends" -eq 1; then
	patch_apply winmm-Delay_Import_Depends/0001-winmm-Delay-import-ole32-msacm32-to-workaround-bug-w.patch
	(
		echo '+    { "Michael Müller", "winmm: Delay import ole32 msacm32 to workaround bug when loading multiple winmm versions.", 1 },';
	) >> "$patchlist"
fi

# Patchset winscard-SCardListReaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26978] Add stub for winscard.SCardListReadersA/W
# |
# | Modified files:
# |   *	dlls/winscard/winscard.c, dlls/winscard/winscard.spec
# |
if test "$enable_winscard_SCardListReaders" -eq 1; then
	patch_apply winscard-SCardListReaders/0001-winscard-add-stubs-for-SCardListReadersA-W.patch
	(
		echo '+    { "Austin English", "winscard: add stubs for SCardListReadersA/W.", 1 },';
	) >> "$patchlist"
fi

# Patchset winsta-WinStationEnumerateW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38102] Add stub for winsta.WinStationEnumerateW
# |
# | Modified files:
# |   *	dlls/winsta/main.c, dlls/winsta/winsta.spec
# |
if test "$enable_winsta_WinStationEnumerateW" -eq 1; then
	patch_apply winsta-WinStationEnumerateW/0001-winsta-Add-stub-for-WinStationEnumerateW.patch
	(
		echo '+    { "Austin English", "winsta: Add stub for WinStationEnumerateW.", 1 },';
	) >> "$patchlist"
fi

# Patchset wpcap-Dynamic_Linking
# |
# | Modified files:
# |   *	configure.ac, dlls/wpcap/Makefile.in, dlls/wpcap/wpcap.c
# |
if test "$enable_wpcap_Dynamic_Linking" -eq 1; then
	patch_apply wpcap-Dynamic_Linking/0001-wpcap-Load-libpcap-dynamically.patch
	(
		echo '+    { "André Hentschel", "wpcap: Load libpcap dynamically.", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-APC_Performance
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c
# |
if test "$enable_ws2_32_APC_Performance" -eq 1; then
	patch_apply ws2_32-APC_Performance/0001-ws2_32-Reuse-old-async-ws2_async_io-structures-if-po.patch
	(
		echo '+    { "Sebastian Lackner", "ws2_32: Reuse old async ws2_async_io structures if possible.", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-Connect_Time
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c, server/protocol.def, server/sock.c
# |
if test "$enable_ws2_32_Connect_Time" -eq 1; then
	patch_apply ws2_32-Connect_Time/0001-ws2_32-Implement-returning-the-proper-time-with-SO_C.patch
	(
		echo '+    { "Sebastian Lackner", "ws2_32: Implement returning the proper time with SO_CONNECT_TIME.", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-TransmitFile
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#5048] Support for TransmitFile
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c, include/winsock.h, server/protocol.def, server/sock.c
# |
if test "$enable_ws2_32_TransmitFile" -eq 1; then
	patch_apply ws2_32-TransmitFile/0001-ws2_32-Add-stub-for-TransmitFile.patch
	patch_apply ws2_32-TransmitFile/0002-ws2_32-Check-for-invalid-parameters-in-TransmitFile.patch
	patch_apply ws2_32-TransmitFile/0003-ws2_32-Implement-a-basic-synchronous-TransmitFile.-r.patch
	patch_apply ws2_32-TransmitFile/0004-ws2_32-Add-support-for-TransmitFile-headers-and-foot.patch
	patch_apply ws2_32-TransmitFile/0005-ws2_32-Add-asynchronous-support-for-TransmitFile.-re.patch
	patch_apply ws2_32-TransmitFile/0006-ws2_32-Add-support-for-TF_DISCONNECT-to-TransmitFile.patch
	patch_apply ws2_32-TransmitFile/0007-ws2_32-Add-support-for-TF_REUSE_SOCKET-to-TransmitFi.patch
	(
		echo '+    { "Erich E. Hoover", "ws2_32: Add stub for TransmitFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Check for invalid parameters in TransmitFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Implement a basic synchronous TransmitFile.", 2 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Add support for TransmitFile headers and footers.", 1 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Add asynchronous support for TransmitFile.", 2 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Add support for TF_DISCONNECT to TransmitFile.", 1 },';
		echo '+    { "Erich E. Hoover", "ws2_32: Add support for TF_REUSE_SOCKET to TransmitFile.", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-WSAPoll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38601] Support for ws2_32.dll.WSAPoll
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c, dlls/ws2_32/ws2_32.spec, include/winsock2.h
# |
if test "$enable_ws2_32_WSAPoll" -eq 1; then
	patch_apply ws2_32-WSAPoll/0001-include-Add-stuff-related-to-WSAPoll-try-2.patch
	patch_apply ws2_32-WSAPoll/0002-ws2_32-tests-Add-WSAPoll-tests.patch
	patch_apply ws2_32-WSAPoll/0003-ws2_32-Add-WSAPoll-implementation.patch
	(
		echo '+    { "Bruno Jesus", "include: Add stuff related to WSAPoll().", 2 },';
		echo '+    { "Bruno Jesus", "ws2_32/tests: Add WSAPoll() tests.", 1 },';
		echo '+    { "Bruno Jesus", "ws2_32: Add WSAPoll() implementation.", 1 },';
	) >> "$patchlist"
fi

# Patchset ws2_32-getaddrinfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29609] Try harder to get the host name address in getaddrinfo()
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c
# |
if test "$enable_ws2_32_getaddrinfo" -eq 1; then
	patch_apply ws2_32-getaddrinfo/0001-ws2_32-Try-harder-to-get-the-host-name-address-in-ge.patch
	(
		echo '+    { "Bruno Jesus", "ws2_32: Try harder to get the host name address in getaddrinfo().", 1 },';
	) >> "$patchlist"
fi

# Patchset wtsapi32-EnumerateProcesses
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29903] Support for WTSEnumerateProcessesW
# |
# | Modified files:
# |   *	dlls/wtsapi32/tests/wtsapi.c, dlls/wtsapi32/wtsapi32.c
# |
if test "$enable_wtsapi32_EnumerateProcesses" -eq 1; then
	patch_apply wtsapi32-EnumerateProcesses/0001-wtsapi32-Partial-implementation-of-WTSEnumerateProce.patch
	(
		echo '+    { "Sebastian Lackner", "wtsapi32: Partial implementation of WTSEnumerateProcessesW.", 1 },';
	) >> "$patchlist"
fi


if test "$enable_patchlist" -eq 1; then

	# Generate a temporary patch containing the patchlist and apply it
	patch_data=$(cat "$patchlist" | sort)
	if test ! -z "$patch_data"; then
		patch_lines=$(echo "$patch_data" | wc -l)
		patch_lines=$((${patch_lines}+21))
		cat > "$patchlist" <<EOF
From: Wine Staging Team <webmaster@fds-team.de>
Subject: Autogenerated patch list.

diff --git a/libs/wine/config.c b/libs/wine/config.c
index 5262c76..0a3182f 100644
--- a/libs/wine/config.c
+++ b/libs/wine/config.c
@@ -478,10 +478,${patch_lines} @@ const char *wine_get_version(void)
     return PACKAGE_VERSION;
 }
 
+static const struct
+{
+    const char *author;
+    const char *subject;
+    int revision;
+}
+wine_patch_data[] =
+{
${patch_data}
+    { NULL, NULL, 0 }
+};
+
 /* return the applied non-standard patches */
 const void *wine_get_patches(void)
 {
-    return NULL;
+    return &wine_patch_data[0];
 }
 
 /* return the build id string */
EOF
		patch_apply_file "$patchlist"
	fi
	rm "$patchlist"
fi

if test "$enable_autoconf" -eq 1; then
	if ! update_configure; then
		abort "'autoreconf -f' failed."
	fi
	if ! update_protocol; then
		abort "'./tools/make_requests' failed."
	fi
fi
# Success
exit 0
