#!/usr/bin/env python

# This file is part of Window-Switch.
# Copyright (c) 2009-2012 Antoine Martin <antoine@nagafix.co.uk>
# Window-Switch is released under the terms of the GNU GPL v3

import gtk
import sys

from winswitch.consts import SPEED_NAMES, DEFAULT_LAN_SPEED
from winswitch.util.common import get_bool
from winswitch.globals import WIN32, OSX
from winswitch.virt.xpra_client_base import XpraClientBase
from winswitch.virt.client_options_base import ClientOptionsBase
from winswitch.virt.options_common import ENCODING, KEYBOARD_SYNC, MMAP
from winswitch.objects.server_command import ServerCommand

XPRA_DEBUG_OPTION="--debug-xpra" in sys.argv

# enable options to choose bandwidth constraints (auto adjust jpeg)
ADAPTIVE_BANDWIDTH = True

class	XpraClientUtil(XpraClientBase, ClientOptionsBase):

	LOSSY_10 = "JPEG lowest quality (10%)"
	LOSSY_25 = "JPEG low quality (25%)"
	LOSSY_50 = "JPEG average quality (50%)"
	LOSSY_90 = "JPEG best lossy quality (90%)"

	JPEG_COMPRESSION_OPTIONS = [LOSSY_10, LOSSY_25, LOSSY_50, LOSSY_90]
	JPEG_COMPRESSION_OPTIONS_PAIRS = [
							(LOSSY_10, 10),
							(LOSSY_25, 25),
							(LOSSY_50, 50),
							(LOSSY_90, 90)
							]

	def	__init__(self, update_session_status, notify_callback):
		XpraClientBase.__init__(self, update_session_status, notify_callback)


	def get_encoding_options(self, server):
		""" returns the list of encoding options """
		options = []
		if not server.supports_xpra_encodings:
			return	options
		if "png" in server.supports_xpra_encodings:
			options.append(("PNG compression (lossless)", "png"))
		if "rgb24" in server.supports_xpra_encodings:
			options.append(("No compression (raw)", "rgb24"))
		if "x264" in server.supports_xpra_encodings:
			options.append(("x264 video compression (lossy)", "x264"))
		if "vpx" in server.supports_xpra_encodings:
			options.append(("vpx video compression (lossy)", "vpx"))
		if "jpeg" in server.supports_xpra_encodings:
			for name, value in XpraClientUtil.JPEG_COMPRESSION_OPTIONS_PAIRS:
				options += [(name, "jpeg:%s" % value)]
			if ADAPTIVE_BANDWIDTH:
				speeds = SPEED_NAMES.keys()		#list of speeds in bytes per second
				speeds.sort()
				for speed in speeds:
					if speed>server.line_speed or speed>=DEFAULT_LAN_SPEED:
						break
					options += [("JPEG adaptive - %s" % SPEED_NAMES.get(speed), "jpeg:%s" % (-speed/1000))]
		return	options

	def get_encoding_widget(self, server, server_command, current_options, change_cb, shown=True):
		""" A combo box where users can select the picture encoding	"""
		if not shown:
			return
		cb = gtk.combo_box_new_text()
		#find the current value:
		current_encoding_value = self.get_default_compression(server, server_command)
		if current_options and ENCODING in current_options:
			try:
				current_encoding_value = int(current_options[ENCODING])
			except:
				pass
		#populate drop down:
		index = 0
		active = 0
		options = self.get_encoding_options(server)
		for text,value in options:
			cb.append_text(text)
			if current_encoding_value!=0 and current_encoding_value==value:
				active = index
			index += 1
		if active>=0:
			cb.set_active(active)
		if change_cb:
			def encoding_changed(widget):
				selected = widget.get_active_text()
				comp = None
				for _text,_value in options:
					if selected==_text:
						comp = _value
				change_cb(ENCODING, comp)
			cb.connect("changed", encoding_changed)
		return cb

	def get_keyboard_sync_widget(self, server, current_options, change_cb, shown=True, enabled=True):
		""" A checkbox to enable keyboard sync"""
		ks = None
		if shown:
			ks = gtk.CheckButton()
			ks.set_active(enabled and get_bool(current_options.get(KEYBOARD_SYNC, True)))
			ks.set_sensitive(enabled)
			self.setup_cb_callback(ks, KEYBOARD_SYNC, change_cb)
		return ks

	def get_mmap_widget(self, server, current_options, change_cb, enabled=True):
		mmap = gtk.CheckButton()
		mmap.set_active(get_bool(current_options.get(MMAP, True)))
		mmap.set_sensitive(enabled)
		self.setup_cb_callback(mmap, MMAP, change_cb)
		return	mmap

	def get_options_widgets(self, server, server_command, shadowed_session=None, current_options=None, default_options=None, change_cb=None):
		""" xpra options: encoding compression (all) and clipboard (not on win32) """
		options = self.get_options(current_options, default_options)
		widgets = []
		has_pil = False
		try:
			import Image
			has_pil = Image is not None
		except:
			pass
		jw = self.get_encoding_widget(server, server_command, options, change_cb, has_pil)
		if jw:
			widgets.append(("Image Compression", jw))
		cw = self.get_clipboard_widget(server, options, change_cb, True, not WIN32 and not OSX)
		if cw:
			widgets.append(("Clipboard Sharing", cw))
		show_ks = self.settings.xpra_version>=[0,0,7,31] and self.settings.advanced
		ks = self.get_keyboard_sync_widget(server, options, change_cb, show_ks, True)
		if ks:
			widgets.append(("Keyboard Synchronized", ks))
		if self.settings.xpra_version>=[0,0,7,33]:
			widgets.append(("Read Only Mode", self.get_readonly_widget(server, options, change_cb, True)))
		if self.settings.xpra_version>=[0,0,7,31] and self.settings.advanced and not WIN32:
			widgets.append(("Enable MMAP", self.get_mmap_widget(server, options, change_cb, server.local or server.embedded_server is not None)))
		if default_options is not None:	#not edit mode
			self.add_timezone_locale_widgets(widgets, self.settings, server, options, change_cb)
			if server_command and server_command.type==ServerCommand.DESKTOP:
				self.add_screen_size_widget(widgets, server, server_command, shadowed_session, options, change_cb)
		return	widgets
