#!/usr/bin/env python

# This file is part of Window-Switch.
# Copyright (c) 2009-2012 Antoine Martin <antoine@nagafix.co.uk>
# Window-Switch is released under the terms of the GNU GPL v3

import sys
import traceback
import dbus
import avahi

from winswitch.consts import MDNS_TYPE
from winswitch.util.simple_logger import Logger

DEBUG = "--debug-mdns" in sys.argv

class AvahiListener:
	def __init__(self, service_type, mdns_found=None, mdns_add=None, mdns_remove=None):
		Logger(self, log_colour=Logger.YELLOW)
		self.sdref = None
		self.readers = []
		self.resolvers = []
		self.service_type = service_type
		self.mdns_found = mdns_found
		self.mdns_add = mdns_add
		self.mdns_remove = mdns_remove
		self.server = None

	def resolve_error(self, *args):
		self.serror(None, *args)

	def service_resolved(self, interface, protocol, name, stype, domain, host, x, address, port, text_array, v):
		self.slog(None, interface, protocol, name, stype, domain, host, x, address, port, "..", v)
		if self.mdns_add:
			#parse text data:
			text = {}
			try:
				for text_line in text_array:
					line = ""
					for b in text_line:
						line += chr(b.real)
					parts = line.split("=", 1)
					if len(parts)==2:
						text[parts[0]] = parts[1]
				self.slog("text=%s" % text, interface, protocol, name, stype, domain, host, x, address, port, "..", v)
			except Exception, e:
				self.serr(None, e, interface, protocol, name, stype, domain, host, x, address, port, text_array, v)
			self.mdns_add(interface, name, domain, host, address, port, text)

	def service_found(self, interface, protocol, name, stype, domain, flags):
		self.slog(None, interface, protocol, name, stype, domain, flags)
		if flags & avahi.LOOKUP_RESULT_LOCAL:
			# local service, skip
			pass
		if self.mdns_found:
			self.mdns_found(interface, name)
		self.server.ResolveService(interface, protocol, name, stype,
				domain, avahi.PROTO_UNSPEC, dbus.UInt32(0), 
				reply_handler=self.service_resolved, error_handler=self.resolve_error)

	def start(self):
		from dbus.mainloop.glib import DBusGMainLoop
		loop = DBusGMainLoop()
		bus = dbus.SystemBus(mainloop=loop)
		self.server = dbus.Interface(bus.get_object(avahi.DBUS_NAME, '/'), 'org.freedesktop.Avahi.Server')

		self.sbrowser = dbus.Interface(bus.get_object(avahi.DBUS_NAME,
								self.server.ServiceBrowserNew(avahi.IF_UNSPEC,
								avahi.PROTO_UNSPEC, MDNS_TYPE, 'local', dbus.UInt32(0))),
								avahi.DBUS_INTERFACE_SERVICE_BROWSER)
		self.sbrowser.connect_to_signal("ItemNew", self.service_found)
		
	def stop(self):
		#FIXME: how do we tell dbus we are no longer interested?
		pass

def main():
	from winswitch.util.main_loop import loop_init, loop_run, callWhenRunning
	listener = None
	try :
		def mdns_found(*args):
			print("mdns_found")
		def mdns_add(*args):
			print("mdns_add")
		def mdns_remove(*args):
			print("mdns_remove")
		listener = AvahiListener(MDNS_TYPE, mdns_found, mdns_add, mdns_remove)
		
		loop_init(False)
		callWhenRunning(listener.start)
		loop_run()

	except Exception, e:
		print('*** Caught exception: %s' % e)
		traceback.print_exc()
		exit()

if __name__ == "__main__":
	main()
