/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme;

import javax.xml.bind.JAXBException;
import javax.xml.bind.DatatypeConverterInterface;
import javax.xml.namespace.QName;

import org.apache.ws.jaxme.impl.JAXBContextImpl;
import org.apache.ws.jaxme.util.NamespaceSupport;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;


/** <p>An XML Serializer is used to marshal a fixed subclass
 * of JMElement. It is unable to marshal generic JMElements.
 * The XML Serializer is typically created by the Manager.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: JMXmlSerializer.java,v 1.3.2.1 2004/08/16 08:29:16 jochen Exp $
 */
public interface JMXmlSerializer {
  public interface Data {
    /** <p>Returns the Marshaller controlling the marshalling process.</p>
     */
    public JMMarshaller getJMMarshaller();

    /** <p>Returns the ContentHandler which is the target of the
     * marshalling process.</p>
     */
    public ContentHandler getContentHandler();

    /** <p>Returns the outer JMXmlSerializer which created the
     * JMXmlSerializerData instance through its <code>getData()</code>
     * method.</p>
     */
    public JMXmlSerializer getJMXmlSerializer();

    /** <p>Returns an instance of NamespaceSupport.</p>
     */
    public NamespaceSupport getNamespaceContext();

    /** <p>Returns a new prefix for the given URI which has
     * not yet been declared.</p>
     * @param pURI A namespace URI being declared.
     * @param pPrefix A suggested prefix or null, if none is available.
     */
    public String getNewPrefix(String pURI, String pPrefix);

    /** <p>Returns the {@link javax.xml.bind.DatatypeConverterInterface}.</p>
     */
    public DatatypeConverterInterface getDatatypeConverter();

    /** <p>Returns whether the root element has already been generated.
     * If this isn't the case, then the caller must take care of
     * attributes like xsi:schemaLocation, and the like.</p>
     */
    public boolean isRootElementCreated();

    /** <p>Sets whether the root element has already been generated.
     * If this isn't the case, then the caller must take care of
     * attributes like xsi:schemaLocation, and the like.</p>
     */
    public void setRootElementCreated(boolean pCreated);
  }
  /** <p>Initializes the JMXmlSerializer; called by the JAXBContextImpl
   * before the serializer is returned to the Marshaller. This method
   * must be called only once.</p>
   */
  public void init(JAXBContextImpl pFactory) throws JAXBException;

  /** <p>Returns a suggested prefix for the given URI.</p>
   * @return Prefix suggestion or null, in which case a default
   *   selection will happen.
   */
  public String getPreferredPrefix(String pURI);

  /** <p>Initializes the marshalling stage by creating an
   * instance of JMXmlSerializerData.</p>
   *
   * @param pMarshaller The Marshaller controlling the marshalling
   *   process. In particular it will be used as an object factory
   *   (via its <code>getContext()</code> method) and for reading
   *   settings like indentation and the like.
   * @param pHandler The ContentHandler that 
   */
  public Data getData(JMMarshaller pMarshaller,
                       ContentHandler pHandler);

  /** <p>Marshals the given JMElement.</p>
   *
   * @param pData An instance of JMXmlSerializerData that was
   *    obtained by calling <code>getData(JMMarshaller,
   *    ContentHandler)</code>.
   * @param pElement The element being marshalled. It must be
   *    an instance of the class associated to this specific
   *    JMXmlSerializer.
   */
  public void marshal(Data pData, QName pName, Object pElement) throws SAXException;
}
