/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.impl;

import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.ws.jaxme.JMManager;
import org.xml.sax.SAXException;



/** <p>An instance of this class represents a config file.
 * A JAXBContext requires an associated Configuration which
 * is located through the classpath.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: Configuration.java,v 1.5 2004/02/16 23:39:57 jochen Exp $
 */
public class Configuration {
  JAXBContextImpl context;
  Manager currentManager;
  Map managers = new HashMap();
  private Class jmMarshallerClass = JMMarshallerImpl.class;
  private Class jmUnmarshallerClass = JMUnmarshallerImpl.class;
  private Class jmValidatorClass = JMValidatorImpl.class;

  public Configuration(JAXBContextImpl pContext) {
    context = pContext;
  }

  public class Manager implements JMManager {
    public class Property {
      private String managerName;
      private String value;

      public String getName() { return managerName; }
      public void setName(String pName) { managerName = pName; }
      public String getValue() { return value; }
      public void setValue(String pValue) { value = pValue; }
      public void finish() throws SAXException {
        if (managerName == null) {
          throw new NullPointerException("Missing 'name' attribute in 'property' element.");
        }
        if (value == null) {
          throw new NullPointerException("Missing 'value' attribute in 'property' element.");
        }
        if (properties == null) {
          properties = new HashMap();
        }
        if (properties.put(managerName, value) != null) {
          throw new IllegalStateException("The property " + managerName + " was specified more than once.");
        }
      }
    }

    private QName name;
    private Class elementInterface;
    private Class elementClass;
    private Class handlerClass;
    private Class marshallerClass;
    private Class pmClass;
    private Map properties;
    public void setQName(QName pName) { name = pName; }
    public QName getQName() { return name; }
    public void setElementClass(String pElementClass) throws ClassNotFoundException {
        elementClass = context.getClassLoader().loadClass(pElementClass);
    }
    public Class getElementClass() { return elementClass; }
    public void setElementInterface(String pElementInterface) throws ClassNotFoundException {
        elementInterface = context.getClassLoader().loadClass(pElementInterface);
    }
    public Class getElementInterface() { return elementInterface; }
    public void setHandlerClass(String pHandlerClass) throws ClassNotFoundException {
        handlerClass = context.getClassLoader().loadClass(pHandlerClass);
    }
    public Class getHandlerClass() { return handlerClass; }
    public void setMarshallerClass(String pMarshallerClass) throws ClassNotFoundException {
        marshallerClass = context.getClassLoader().loadClass(pMarshallerClass);
    }
    public Class getMarshallerClass() { return marshallerClass; }
    public void setPmClass(String pPersistencyClass) throws ClassNotFoundException {
        pmClass = context.getClassLoader().loadClass(pPersistencyClass);
    }
    public Class getPmClass() { return pmClass; }
    public JAXBContextImpl getFactory() { return context; }
    public Property createProperty() {
      return new Property();
    }
    public String getProperty(String pName) {
      if (pName == null) {
        throw new IllegalArgumentException("The property name must not be null.");
      }
      if (properties == null) {
        return null;
      }
      return (String) properties.get(pName);
    }
    public void finish() throws SAXException {
      if (currentManager != this) {
        throw new IllegalStateException("currentManager != this");
      }
      try {
        context.addManager(currentManager);
        currentManager = null;
      } catch (Exception e) {
        throw new SAXException(e.getMessage(), e);
      }
    }
  }

  /** <p>Creates a new Manager.</p>
   */
  public Manager createManager() {
    if (currentManager != null) {
      throw new IllegalStateException("currentManager != null");
    }
    currentManager = new Manager();
    return currentManager;
  }

  /** <p>Sets the JMMarshaller class.</p>
   */
  public void setJMMarshallerClass(Class pJMMarshallerClass) {
    jmMarshallerClass = pJMMarshallerClass;
  }

  /** <p>Returns the JMMarshaller class.</p>
   */
  public Class getJMMarshallerClass() {
    return jmMarshallerClass;
  }

  /** <p>Sets the JMUnmarshaller class.</p>
   */
  public void setJMUnmarshallerClass(Class pJMUnmarshallerClass) {
    jmUnmarshallerClass = pJMUnmarshallerClass;
  }

  /** <p>Returns the JMUnmarshaller class.</p>
   */
  public Class getJMUnmarshallerClass() {
    return jmUnmarshallerClass;
  }

  /** <p>Sets the JMValidator class.</p>
   */
  public void setJMValidatorClass(Class pJMValidatorClass) {
    jmValidatorClass = pJMValidatorClass;
  }

  /** <p>Returns the JMValidator class.</p>
   */
  public Class getJMValidatorClass() {
    return jmValidatorClass;
  }
}
