/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.impl;

import java.text.Format;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.DatatypeConverterInterface;
import javax.xml.bind.PropertyException;
import javax.xml.bind.ValidationEventHandler;

import org.apache.ws.jaxme.xs.util.XsDateFormat;
import org.apache.ws.jaxme.xs.util.XsDateTimeFormat;
import org.apache.ws.jaxme.xs.util.XsTimeFormat;



/** <p>Common subclass for JMMarshallerImpl, JMUnmarshallerImpl and
 * JMValidatorImpl.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: JMControllerImpl.java,v 1.3 2004/02/16 23:39:57 jochen Exp $
 */
public class JMControllerImpl {
  /** <p>Property prefix for users private settings:
   * "jaxme.private."; if a property
   * name starts with this prefix, then the property value is
   * stored in an internal Map.</p>
   */
  public static final String JAXME_PRIVATE = "jaxme.private.";

  /** <p>Name of the property for setting the
   * {@link javax.xml.bind.DatatypeConverterInterface}: "jaxme.datatypeConverter"</p>
   */
  public static final String JAXME_DATATYPE_CONVERTER = "jaxme.datatypeConverter";

  /** <p>Property for setting an instance of {@link java.text.Format},
   * which is being used for parsing and printing <code>xs:dateTime</code>
   * values. Defaults to an instance of
   * {@link org.apache.ws.jaxme.xs.util.XsDateTimeFormat}.</p>
   */
  public static final String JAXME_FORMAT_DATETIME = "jaxme.format.dateTime";

  /** <p>Property for setting an instance of {@link java.text.Format},
   * which is being used for parsing and printing <code>xs:date</code>
   * values. Defaults to an instance of
   * {@link org.apache.ws.jaxme.xs.util.XsDateFormat}.</p>
   */
  public static final String JAXME_FORMAT_DATE = "jaxme.format.date";

  /** <p>Property for setting an instance of {@link java.text.Format},
   * which is being used for parsing and printing <code>xs:time</code>
   * values. Defaults to an instance of
   * {@link org.apache.ws.jaxme.xs.util.XsTimeFormat}.</p>
   */
  public static final String JAXME_FORMAT_TIME = "jaxme.format.time";

  private Map privateMap;
  private JAXBContextImpl context;
  private DatatypeConverterInterface datatypeConverter = new DatatypeConverterImpl();

  public void setProperty(String pProperty, Object pValue)
      throws PropertyException {
    if (pProperty.startsWith(JAXME_PRIVATE)) {
      if (privateMap == null) {
        privateMap = new HashMap();
      }
      privateMap.put(pProperty, pValue);
      return;
    } else if (pProperty.equals(JAXME_DATATYPE_CONVERTER)) {
      datatypeConverter = (DatatypeConverterInterface) pValue;
      return;
    } else if (JAXME_FORMAT_DATETIME.equals(pProperty)) {
      setDateTimeFormat((Format) pValue);
    } else if (JAXME_FORMAT_DATE.equals(pProperty)) {
      setDateFormat((Format) pValue);
    } else if (JAXME_FORMAT_TIME.equals(pProperty)) {
      setTimeFormat((Format) pValue);
    }
    
    throw new PropertyException("Unknown property: " + pProperty);
  }

  public Object getProperty(String pProperty) throws PropertyException {
    if (pProperty.startsWith(JAXME_PRIVATE)) {
      if (privateMap == null) { return null; }
      return privateMap.get(pProperty);
    } else if (pProperty.equals(JAXME_DATATYPE_CONVERTER)) {
      return datatypeConverter;
    } else if (JAXME_FORMAT_DATETIME.equals(pProperty)) {
      return getDateTimeFormat();
    } else if (JAXME_FORMAT_DATE.equals(pProperty)) {
      return getDateFormat();
    } else if (JAXME_FORMAT_TIME.equals(pProperty)) {
      return getTimeFormat();
    }

    throw new PropertyException("Unknown property: " + pProperty);
  }

  protected ValidationEventHandler eventHandler;

  public ValidationEventHandler getEventHandler() { return eventHandler; }

  public void setEventHandler(ValidationEventHandler pEventHandler) { eventHandler = pEventHandler; }

  public void setJAXBContextImpl(JAXBContextImpl pContext) { context = pContext; }
  public JAXBContextImpl getJAXBContextImpl() { return context; }

  public void setDatatypeConverter(DatatypeConverterInterface pConverter) { datatypeConverter = pConverter; }
  public DatatypeConverterInterface getDatatypeConverter() { return datatypeConverter; }

private Format dateTimeFormat;

private Format dateFormat;

private Format timeFormat;

/** <p>Sets the {@link java.text.Format} for parsing and printing
   * <code>xs:dateTime</code> values.</p>
   * @param pFormat An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsDateTimeFormat}
   *   (default).
   */
public void setDateTimeFormat(Format pFormat) {
      dateTimeFormat = pFormat;
  }

/** <p>Returns the {@link java.text.Format} for parsing and printing
   * <code>xs:dateTime</code> values.</p>
   * @return An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsDateTimeFormat}
   *   (default).
   */
public Format getDateTimeFormat() {
      if (dateTimeFormat == null) {
          dateTimeFormat = new XsDateTimeFormat();
      }
      return dateTimeFormat;
  }

/** <p>Sets the {@link java.text.Format} for parsing and printing
   * <code>xs:date</code> values.</p>
   * @param pFormat An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsDateFormat}
   *   (default).
   */
public void setDateFormat(Format pFormat) {
      dateFormat = pFormat;
  }

/** <p>Returns the {@link java.text.Format} for parsing and printing
   * <code>xs:date</code> values.</p>
   * @return An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsDateFormat}
   *   (default).
   */
public Format getDateFormat() {
      if (dateFormat == null) {
          dateFormat = new XsDateFormat();
      }
      return dateFormat;
  }

/** <p>Sets the {@link java.text.Format} for parsing and printing
   * <code>xs:date</code> values.</p>
   * @param pFormat An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsDateFormat}
   *   (default).
   */
public void setTimeFormat(Format pFormat) {
      timeFormat = pFormat;
  }

/** <p>Returns the {@link java.text.Format} for parsing and printing
   * <code>xs:time</code> values.</p>
   * @return An instance of {@link java.text.DateFormat} or an
   *   instance of {@link org.apache.ws.jaxme.xs.util.XsTimeFormat}
   *   (default).
   */
public Format getTimeFormat() {
      if (timeFormat == null) {
          timeFormat = new XsTimeFormat();
      }
      return timeFormat;
  }
}
