/*
 * Copyright 2003,2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.impl;

import javax.xml.bind.DatatypeConverterInterface;
import javax.xml.bind.JAXBException;
import javax.xml.bind.ValidationEvent;
import javax.xml.bind.ValidationEventHandler;
import javax.xml.bind.ValidationEventLocator;
import javax.xml.bind.helpers.ValidationEventLocatorImpl;

import org.apache.ws.jaxme.JMHandler;
import org.apache.ws.jaxme.JMUnmarshaller;
import org.apache.ws.jaxme.Observer;
import org.apache.ws.jaxme.ValidationEvents;
import org.apache.ws.jaxme.util.NamespaceSupport;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;


/** <p>A common base class for the AtomicHandler and the
 * JMHandlerImpl.</p>
 * 
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: JMHandlerBase.java,v 1.4 2004/02/16 23:39:57 jochen Exp $
 */
public abstract class JMHandlerBase implements JMHandler {
  private Locator locator;
  private JMHandler.Data data;
  private Observer observer;

  protected class DataImpl implements JMHandler.Data {
    private JMUnmarshaller unmarshaller;
    private AtomicHandler atomicHandler;
    private NamespaceSupport namespaceSupport = new NamespaceSupport();

    public DataImpl(JMUnmarshaller pUnmarshaller) {
      unmarshaller = pUnmarshaller;
    }
    public JMUnmarshaller getUnmarshaller() {
      return unmarshaller;
    }
    public JAXBContextImpl getFactory() {
      return unmarshaller.getJAXBContextImpl();
    }
    public JMHandler getAtomicHandler() {
      if (atomicHandler == null) {
        atomicHandler = new AtomicHandler();
      }
      return atomicHandler;
    }
    public DatatypeConverterInterface getDatatypeConverter() {
      return unmarshaller.getDatatypeConverter();
    }

    public NamespaceSupport getNamespaceContext() {
      return namespaceSupport;
    }
  }

  public void init(JMUnmarshaller pUnmarshaller) throws JAXBException {
    init(new DataImpl(pUnmarshaller));
  }

  public void init(JMHandler.Data pData) throws JAXBException {
    data = pData;
  }

  public JMHandler.Data getData() {
    return data;
  }

  public void setDocumentLocator(Locator pLocator) {
    locator = pLocator;
  }

  /** <p>Returns the Locator previously set by <code>setDocumentLocator()</code>,
   * if any.</p>
   */
  public Locator getDocumentLocator() {
    return locator;
  }

  public void setObserver(Observer pObserver) {
    observer = pObserver;
  }

  public Observer getObserver() {
    return observer;
  }

  protected void validationEvent(int pSeverity, String pMsg, String pErrorCode)
      throws SAXException {
    ValidationEventHandler eventHandler;
    try {
      eventHandler = getData().getUnmarshaller().getEventHandler();
    } catch (JAXBException e) {
      /* Interesting question: Which event to report? The JAXBException or the
       * validation event? We choose the former.
       */
       throw new SAXException(e);
    }
    if (eventHandler == null) {
      throw new SAXParseException(pMsg, getDocumentLocator());
    } else {
      ValidationEventLocator myLocator = new ValidationEventLocatorImpl(getDocumentLocator());
      ValidationEventImpl event = new ValidationEventImpl(pSeverity,
                                                          pErrorCode + ": " + pMsg,
                                                          myLocator);
      event.setErrorCode(pErrorCode);
      eventHandler.handleEvent(event);
    }
  }

  protected void validationEvent(int pSeverity, String pMsg, String pErrorCode,
                                   Exception pException)
      throws SAXException {
    ValidationEventHandler eventHandler;
    try {
      eventHandler = getData().getUnmarshaller().getEventHandler();
    } catch (JAXBException e) {
      /* Interesting question: Which event to report? The JAXBException or the
       * validation event? We decide against the former.
       */
       throw new SAXException(e);
    }
    if (eventHandler == null) {
      throw new SAXParseException(pMsg, getDocumentLocator(), pException);
    } else {
      ValidationEventLocator myLocator = new ValidationEventLocatorImpl(getDocumentLocator());
      ValidationEventImpl event = new ValidationEventImpl(pSeverity,
                                                          pErrorCode + ": " + pMsg,
                                                          myLocator,
                                                          pException);
      event.setErrorCode(pErrorCode);
      eventHandler.handleEvent(event);
    }
  }

  public void processingInstruction(String pTarget, String pData)
      throws SAXException {
    validationEvent(ValidationEvent.WARNING,
                    "Don't know how to handle processing instructions.",
                    ValidationEvents.EVENT_PROCESSING_INSTRUCTION);
  }

  public void skippedEntity(String name) throws SAXException {
    validationEvent(ValidationEvent.WARNING,
                    "Don't know how to handle skipped entities.",
                    ValidationEvents.EVENT_SKIPPED_ENTITY);
  }

  public void startPrefixMapping(String pPrefix, String pURI)
      throws SAXException {
    getData().getNamespaceContext().declarePrefix(pPrefix, pURI);
  }

  public void endPrefixMapping(String pPrefix) throws SAXException {
    getData().getNamespaceContext().undeclarePrefix(pPrefix);
  }

  public void endDocument() {
    Observer o = getObserver();
    if (o != null) {
      o.notify(getResult());
    }
  }
}
