/*
 * Copyright 2003,2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.impl;

import java.util.ArrayList;
import java.util.List;
import javax.xml.bind.JAXBException;
import javax.xml.bind.UnmarshallerHandler;
import javax.xml.namespace.QName;

import org.apache.ws.jaxme.JMHandler;
import org.apache.ws.jaxme.JMUnmarshaller;
import org.apache.ws.jaxme.Observer;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/** <p>Implementation of a JMUnmarshallerHandler; the
 * UnmarshallerHandler receives SAX events which he silently
 * discards, as long as the first <code>startElement</code>
 * event is seen. Depending on namespace URI and local name,
 * the Unmarshallerhandler creates a new instance of JMHandler
 * and from now on forwards all SAX events to the JMHandler.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: JMUnmarshallerHandlerImpl.java,v 1.4 2004/02/16 23:39:57 jochen Exp $
 */
public class JMUnmarshallerHandlerImpl implements UnmarshallerHandler,
    Observer {
  private JMUnmarshaller unmarshaller;
  private JMHandler myHandler;
  private Object result;
  private Locator locator;
  private List prefixes = new ArrayList();

  public void setJMUnmarshaller(JMUnmarshaller pUnmarshaller) {
    unmarshaller = pUnmarshaller;
  }

  public JMUnmarshaller getJMUnmarshaller() {
    return unmarshaller;
  }

  public void notify(Object pObject) {
    result = pObject;
  }

  public Object getResult() throws JAXBException, IllegalStateException {
    if (myHandler == null) {
      throw new IllegalStateException("No startElement event has been seen.");
    }
    if (result == null) {
      throw new IllegalStateException("Parsing is not yet finished.");
    }
    return result;
  }

  public void setDocumentLocator(Locator pLocator) {
    locator = pLocator;
    if (myHandler != null) { myHandler.setDocumentLocator(pLocator); }
  }

  public void startDocument() throws SAXException {
    result = null;
    myHandler = null;
  }

  public void endDocument() throws SAXException {
    if (myHandler != null) {
      myHandler.endDocument();
    }
  }

  public void startPrefixMapping(String pPrefix, String pURI)
      throws SAXException {
    if (myHandler != null) {
      myHandler.startPrefixMapping(pPrefix, pURI);
    } else {
      prefixes.add(pPrefix);
      prefixes.add(pURI);
    }
  }

  public void endPrefixMapping(String pPrefix) throws SAXException {
    if (myHandler != null) {
      myHandler.endPrefixMapping(pPrefix);
    } else {
      for (int i = prefixes.size()-2;  i >= 0;  i -= 2) {
          if (pPrefix.equals(prefixes.get(i))) {
              prefixes.remove(i);
              prefixes.remove(i);
              return;
          }
      }
      throw new SAXException("Undeclared namespace prefix: " + pPrefix);
    }
  }

  public void startElement(String pNamespaceURI, String pLocalName,
                            String pQName, Attributes pAttr) throws SAXException {
    if (myHandler == null) {
      QName name = new QName(pNamespaceURI, pLocalName);
      try {
        myHandler = unmarshaller.getJAXBContextImpl().getJMHandler(name);
        myHandler.init(unmarshaller);
      } catch (JAXBException e) {
        throw new SAXException(e);
      }
      if (locator != null) {
        myHandler.setDocumentLocator(locator);
      }
      myHandler.setObserver(this);
      myHandler.startDocument();
      if (prefixes.size() > 0) {
        for (int i = 0;  i < prefixes.size();  i+=2) {
          myHandler.startPrefixMapping(prefixes.get(i).toString(),
                                       prefixes.get(i+1).toString());
        }
        prefixes.clear();
      }
    }
    myHandler.startElement(pNamespaceURI, pLocalName, pQName, pAttr);
  }

  public void endElement(String pNamespaceURI, String pLocalName, String pQName)
      throws SAXException {
    myHandler.endElement(pNamespaceURI, pLocalName, pQName);
  }

  public void characters(char[] pChars, int pStart, int pLen)
      throws SAXException {
    myHandler.characters(pChars, pStart, pLen);
  }

  public void ignorableWhitespace(char[] pChars, int pStart, int pLen)
      throws SAXException {
    myHandler.ignorableWhitespace(pChars, pStart, pLen);
  }

  public void processingInstruction(String pTarget, String pData)
      throws SAXException {
    if (myHandler != null) {
      myHandler.processingInstruction(pTarget, pData);
    }
  }

  public void skippedEntity(String pName) throws SAXException {
    if (myHandler != null) {
      myHandler.skippedEntity(pName);
    }
  }
}
