/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.junit;

import java.io.StringReader;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Calendar;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.ws.jaxme.JMElement;
import org.apache.ws.jaxme.JMMarshaller;
import org.apache.ws.jaxme.JMUnmarshaller;
import org.apache.ws.jaxme.JMXmlSerializer;
import org.apache.ws.jaxme.XMLWriter;
import org.apache.ws.jaxme.impl.JAXBContextImpl;
import org.apache.ws.jaxme.impl.XMLWriterImpl;
import org.apache.ws.jaxme.test.misc.enumeration.AllSimpleTypes;
import org.apache.ws.jaxme.test.misc.enumeration.AllTypesElement;
import org.apache.ws.jaxme.test.misc.enumeration.MyDoubleTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.MyFloatTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.MyIntTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.MyLongTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.MyShortTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.MyStringTypeClass;
import org.apache.ws.jaxme.test.misc.enumeration.impl.AllSimpleTypesHandler;
import org.apache.ws.jaxme.test.misc.enumeration.impl.AllSimpleTypesImpl;
import org.apache.ws.jaxme.test.misc.enumeration.impl.AllSimpleTypesSerializer;
import org.apache.ws.jaxme.test.misc.enumeration.impl.AllTypesElementImpl;
import org.apache.ws.jaxme.test.misc.enumeration.impl.AllTypesElementTypeSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;


/**
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: EnumerationTest.java,v 1.5.2.1 2004/08/20 12:22:55 jochen Exp $
 */
public class EnumerationTest extends BaseTestCase {
  private JAXBContextImpl factory;

  public EnumerationTest(String arg) { super(arg); }

  public void setUp() throws JAXBException {
    factory = (JAXBContextImpl) JAXBContext.newInstance("org.apache.ws.jaxme.test.misc.enumeration");
  }

  public JAXBContextImpl getFactory() {
    return factory;
  }

  public AllSimpleTypes getAllSimpleTypesElement() {
    AllSimpleTypes element = new AllSimpleTypesImpl();
    element.setStringElem(MyStringTypeClass.FOO);
    element.setIntElem(MyIntTypeClass.INT3);
    element.setLongElem(MyLongTypeClass.LONG_NEGATIVE);
    element.setShortElem(MyShortTypeClass.SHORT_POSITIVE);
    element.setDoubleElem(MyDoubleTypeClass.DOUBLE_POSITIVE);
    element.setFloatElem(MyFloatTypeClass.FLOAT_NEGATIVE);
    return element;
  }

  public AllTypesElement getAllTypesElement() throws JAXBException {
    AllTypesElement element = (AllTypesElement) getFactory().getElement(AllTypesElement.class);
    element.setAllSimpleTypesElement(getAllSimpleTypesElement());
    return element;
  }

  public String getDateTime() {
    Calendar result = Calendar.getInstance();
    result.set(Calendar.YEAR, 2002);
    result.set(Calendar.MONTH, 12-1);
    result.set(Calendar.DAY_OF_MONTH, 17);
    result.set(Calendar.HOUR_OF_DAY, 12);
    result.set(Calendar.MINUTE, 23);
    result.set(Calendar.SECOND, 11);
    return DateFormat.getDateTimeInstance().format(result.getTime());
  }

  public String getDate() {
    Calendar result = Calendar.getInstance();
    result.set(Calendar.YEAR, 2002);
    result.set(Calendar.MONTH, 12-1);
    result.set(Calendar.DAY_OF_MONTH, 16);
    return DateFormat.getDateInstance().format(result.getTime());
  }

  public String getTime() {
    Calendar result = Calendar.getInstance();
    result.set(Calendar.HOUR_OF_DAY, 12);
    result.set(Calendar.MINUTE, 00);
    result.set(Calendar.SECOND, 11);
    return DateFormat.getTimeInstance().format(result.getTime());
  }

  public String getAllSimpleTypesElementString() {
    AllTypesElementImpl elem = new AllTypesElementImpl();
    String uri = elem.getQName().getNamespaceURI();
    return
      "<ex:AllSimpleTypesElement xmlns:ex=\"" + uri + "\">" +
        "<ex:StringElem>FOO</ex:StringElem>" +
        "<ex:IntElem>3</ex:IntElem>" +
        "<ex:LongElem>-23987982739273989</ex:LongElem>" +
        "<ex:ShortElem>3468</ex:ShortElem>" +
        "<ex:DoubleElem>3249239847982.234</ex:DoubleElem>" +
        "<ex:FloatElem>-24234.234</ex:FloatElem>" +
      "</ex:AllSimpleTypesElement>";
  }

  public String getAllTypesElementString() {
    AllTypesElementImpl elem = new AllTypesElementImpl();
    String uri = elem.getQName().getNamespaceURI();
    return
      "<ex:AllTypesElement xmlns:ex=\"" + uri + "\">\n" +
      "  <ex:AllSimpleTypesElement>\n" +
      "    <ex:StringElem>FOO</ex:StringElem>\n" +
      "    <ex:IntElem>3</ex:IntElem>\n" +
      "    <ex:LongElem>-23987982739273989</ex:LongElem>\n" +
      "    <ex:ShortElem>3468</ex:ShortElem>\n" +
      "    <ex:DoubleElem>3249239847982.234</ex:DoubleElem>\n" +
      "    <ex:FloatElem>-24234.234</ex:FloatElem>\n" +
      "  </ex:AllSimpleTypesElement>\n" +
      "</ex:AllTypesElement>";
  }

  public XMLReader getXMLReader() throws ParserConfigurationException, SAXException {
    // Verify whether we can create a SAX Parser; it's better to detect this here
    // than within JAXB
    SAXParserFactory spf = SAXParserFactory.newInstance();
    spf.setNamespaceAware(true);
    spf.setValidating(false);
    SAXParser sp = spf.newSAXParser();
    return sp.getXMLReader();
  }

  public void verifyAllSimpleTypesElement(AllSimpleTypes pElement) {
    assertEquals(MyStringTypeClass.FOO, pElement.getStringElem());
    assertEquals("FOO", pElement.getStringElem().getValue());
    assertEquals("FOO", pElement.getStringElem().toString());
    assertEquals(MyIntTypeClass.INT3, pElement.getIntElem());
    assertEquals(3, pElement.getIntElem().getValue());
    assertEquals("INT3", pElement.getIntElem().getName());
    assertEquals("3", pElement.getIntElem().toString());
    assertEquals(MyLongTypeClass.LONG_NEGATIVE, pElement.getLongElem());
    assertEquals(-23987982739273989L, pElement.getLongElem().getValue());
    assertEquals("LONG_NEGATIVE", pElement.getLongElem().getName());
    assertEquals("-23987982739273989", pElement.getLongElem().toString());
    assertEquals(MyShortTypeClass.SHORT_POSITIVE, pElement.getShortElem());
    assertEquals(3468, pElement.getShortElem().getValue());
    assertEquals("SHORT_POSITIVE", pElement.getShortElem().getName());
    assertEquals("3468", pElement.getShortElem().toString());
    assertEquals(MyDoubleTypeClass.DOUBLE_POSITIVE, pElement.getDoubleElem());
    assertTrue(3249239847982.234 == pElement.getDoubleElem().getValue());
    assertEquals("DOUBLE_POSITIVE", pElement.getDoubleElem().getName());
    assertEquals("3249239847982.234", pElement.getDoubleElem().toString());
    assertEquals(MyFloatTypeClass.FLOAT_NEGATIVE, pElement.getFloatElem());
    assertTrue(Float.parseFloat("-24234.234") == pElement.getFloatElem().getValue());
    assertEquals("FLOAT_NEGATIVE", pElement.getFloatElem().getName());
    assertEquals("-24234.234", pElement.getFloatElem().toString());
  }

  public void verifyAllTypesElement(AllTypesElement pElement) {
    verifyAllSimpleTypesElement(pElement.getAllSimpleTypesElement());
  }

  public void testUnmarshalSimpleElements() throws Exception {
    XMLReader xr = getXMLReader();
    AllSimpleTypesHandler unmarshaller = new AllSimpleTypesHandler();
    JMUnmarshaller jmUnmarshaller = (JMUnmarshaller) getFactory().createUnmarshaller();
    unmarshaller.init(jmUnmarshaller);
    xr.setContentHandler(unmarshaller);
    StringReader sr = new StringReader(getAllSimpleTypesElementString());
    xr.parse(new InputSource(sr));
    AllSimpleTypes result = (AllSimpleTypes) unmarshaller.getResult();
    verifyAllSimpleTypesElement(result);
  }

  public void testUnmarshalComplexElements() throws Exception {
    JAXBContext myFactory = getFactory();
    Unmarshaller unmarshaller = myFactory.createUnmarshaller();
    StringReader sr = new StringReader(getAllTypesElementString());
    AllTypesElement result = (AllTypesElement) unmarshaller.unmarshal(new InputSource(sr));
    verifyAllTypesElement(result);
  }

  public void testMarshalSimpleElements() throws Exception {
    AllSimpleTypesSerializer marshaller = new AllSimpleTypesSerializer();
    marshaller.init(getFactory());
    XMLWriter xw = new XMLWriterImpl();
    StringWriter sw = new StringWriter();
    xw.setWriter(sw);
    JMXmlSerializer.Data data = marshaller.getData((JMMarshaller) factory.createMarshaller(), xw);
    AllTypesElementImpl element = new AllTypesElementImpl();
    marshaller.marshal(data, new QName(element.getQName().getNamespaceURI(), "AllSimpleTypesElement"),
                       getAllSimpleTypesElement());
    String expected = getAllSimpleTypesElementString();
    String got = sw.toString();
    assertEquals(expected, got);
  }

  public void testMarshalComplexElements() throws Exception {
    JAXBContext myFactory = getFactory();
    Marshaller marshaller = myFactory.createMarshaller();
    StringWriter sw = new StringWriter();
    JMElement jmElement = (JMElement) getAllTypesElement();
    assertNotNull(jmElement.getQName());
    marshaller.marshal(jmElement, sw);
    assertEquals(getAllTypesElementString(), sw.toString());
  }

  public void testPrefixes() throws Exception {
    AllTypesElementImpl el = new AllTypesElementImpl();
    JMXmlSerializer ser = new AllTypesElementTypeSerializer();
    assertEquals("ex", ser.getPreferredPrefix(el.getQName().getNamespaceURI()));
    assertNull(ser.getPreferredPrefix("dummyURI"));
  }
}
