/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.js;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.ws.jaxme.js.JavaSource.Protection;


/**
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 */
public abstract class AbstractJavaMethod
    extends ConditionalIndentationJavaSourceObject {
  private List exceptions = new ArrayList();
  private List params = new ArrayList();
  public AbstractJavaMethod(String pName, JavaQName pType,
                            Protection pProtection) {
    super(pName, pType, pProtection);
  }

  /** <p>Returns whether the method is throwing the given exception.
   * Note that this method doesn't care for inheritance. For example,
   * if the method declares to be throwing an {@link java.io.IOException},
   * then the value <code>isThrowing(java.net.MalformedURLException.class)</code>
   * is still false.</p>
   */
  public boolean isThrowing(JavaQName e) {
    if (e == null) {
      throw new NullPointerException("The exception argument must not be null.");
    }
    for (Iterator iter = exceptions.iterator();  iter.hasNext();  ) {
      if (e.equals(iter.next())) {
        return true;
      }
    }
    return false;
  }

  /** <p>Returns whether the method is throwing the given exception.
   * Note that this method doesn't care for inheritance. For example,
   * if the method declares to be throwing an {@link java.io.IOException},
   * then the value <code>isThrowing(java.net.MalformedURLException.class)</code>
   * is still false.</p>
   */
  public boolean isThrowing(Class e) {
    if (e == null) {
      throw new NullPointerException("The exception argument must not be null.");
    }
    return isThrowing(JavaQNameImpl.getInstance(e));
  }

  /** <p>Adds an exception to this methods list of exceptions.</p>
   *
   * @see #getExceptions
   */
  public void addThrows(JavaQName e) {
    if (e == null) {
      throw new NullPointerException("The exception argument must not be null.");
    }
    exceptions.add(e);
  }

  /** <p>Adds an exception to this methods list of exceptions.</p>
   *
   * @see #getExceptions
   */
  public void addThrows(Class e) {
    if (e == null) {
      throw new NullPointerException("The exception argument must not be null.");
    }
    exceptions.add(JavaQNameImpl.getInstance(e));
  }

  /** <p>Adds an exception to this methods list of exceptions.</p>
   *
   * @see #getExceptions
   * @deprecated Use {@link #addThrows(JavaQName)}
   */
  public void addThrows(String e) {
    exceptions.add(JavaQNameImpl.getInstance(e));
    if (e == null) {
      throw new NullPointerException("The exception argument must not be null.");
    }
  }


  /** <p>Adds a parameter that this method takes.</p>
   *
   * @see #getParams
   * @deprecated Use {@link #addParam(JavaQName, String)}
   */
  public void addParam(String p) {
    if (p == null) {
      throw new NullPointerException("param argument must not be null");
    }
    StringTokenizer st = new StringTokenizer(p);
    if (!st.hasMoreTokens()) {
      throw new IllegalArgumentException("param argument must have two tokens: type name");
    }
    String type = st.nextToken();
    if (!st.hasMoreTokens()) {
      throw new IllegalArgumentException("param argument must have two tokens: type name");
    }
    String name = st.nextToken();
    if (st.hasMoreTokens()) {
      throw new IllegalArgumentException("param argument must have exactly two tokens: type name");
    }
    addParam(type, name);
  }

  /** <p>Adds a parameter that this method takes.</p>
   *
   * @see #getParams
   * @deprecated Use {@link #addParam(JavaQName, String)}
   */
  public void addParam(String p, String v) {
    if (p == null) {
      throw new NullPointerException("param argument must not be null");
    }
    p = p.trim();
    int offset = p.indexOf('.');
    if (offset == -1) {
      addParam(JavaQNameImpl.getInstance(p), v);
    } else {
      addParam(JavaQNameImpl.getInstance(p.substring(0, offset),
                                         p.substring(offset+1)), v);
    }
  }

  /** <p>Adds a parameter that this method takes.</p>
   *
   * @see #getParams
   * @return An object to use for referencing the parameter inside the method.
   */
  public Parameter addParam(Class p, String v) {
    return addParam(JavaQNameImpl.getInstance(p), v);
  }

  /** <p>Adds a parameter that this method takes.</p>
   *
   * @see #getParams
   * @return An object to use for referencing the parameter inside the method.
   */
  public Parameter addParam(JavaQName pType, String pName) {
    if (pType == null) {
      throw new NullPointerException("Type argument must not be null");
    }
    if (pName == null) {
      throw new NullPointerException("Parameter name argument must not be null");
    }
    Parameter p = new Parameter(pType, pName);
    params.add(p);
    return p;
  }

  /** <p>Adds a parameter that this method takes.</p>
   *
   * @see #getParams
   * @return An object to use for referencing the parameter inside the method.
   */
  public Parameter addParam(Parameter pParam) {
    return addParam(pParam.getType(), pParam.getName());
  }

  /** <p>Clears the list of parameters.</p>
   */
  public void clearParams() {
    params.clear();
  }

  /** <p>Returns the list of exceptions thrown by this method.</p>
   *
   * @see #addThrows(JavaQName)
   */
  public JavaQName[] getExceptions() {
    return (JavaQName[]) exceptions.toArray(new JavaQName[exceptions.size()]);
  }


  /** <p>Returns the list of parameters that this method takes. Any element
   * in the list is an instance of {@link Parameter}.</p>
   *
   * @see #addParam(JavaQName, String)
   */
  public Parameter[] getParams() {
         return (Parameter[]) params.toArray(new Parameter[params.size()]);
  }
}
