/*
 * Copyright 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.js;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;


/** <p>A class representing a Java source file.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 */
public class JavaSource extends IndentationEngineImpl {
  public static class Type implements Serializable {
  	 private String name;
  	 Type(String pName) {
  	 	name = pName;
  	 }
  	 public String toString() { return name; }
  	 public static Type valueOf(String pType) {
  	 	if ("class".equals(pType)) {
  	 		return CLASS;
  	 	} else if ("interface".equals(pType)) {
  	 		return INTERFACE;
  	 	} else {
  	 		throw new IllegalArgumentException("Type must be either 'class' or 'interface'.");
  	 	}
  	 }
  }

  public static class Protection implements Serializable {
  	 private String name;
  	 Protection(String pName) {
  	 	name = pName;
  	 }
  	 public String toString() { return name; }
  	 public static Protection valueOf(String pProtection) {
  	 	if ("public".equals(pProtection)) {
  	 		return PUBLIC;
  	 	} else if ("protected".equals(pProtection)) {
  	 		return PROTECTED;
  	 	} else if ("private".equals(pProtection)) {
  	 		return PRIVATE;
  	 	} else if (pProtection == null  ||  "".equals(pProtection)) {
  	 		return DEFAULT_PROTECTION;
  	 	} else {
  	 		throw new IllegalArgumentException("Protection must be either 'public', 'protected', 'private', null or '' (default protection).");
  	 	}
  	 }
	 /** <p>Returns an instance of Protection by using the methods
	  * {@link Modifier#isPublic(int)}, {@link Modifier#isProtected(int)} and
	  * {@link Modifier#isPrivate(int)} on the argument <code>pModifiers</code>.
	  * If neither returns true, assumes {@link JavaSource#DEFAULT_PROTECTION}.</p>
	  */
  	 public static Protection valueOf(int pModifiers) {
  	 	if (Modifier.isPublic(pModifiers)) {
  	 		return PUBLIC;
  	 	} else if (Modifier.isProtected(pModifiers)) {
  	 		return PROTECTED;
  	 	} else if (Modifier.isPrivate(pModifiers)) {
  	 		return PRIVATE;
  	 	} else {
  	 		return DEFAULT_PROTECTION;
  	 	}
  	 }
  }

  /** <p>Type of a JavaSource class.</p>
   * @see #INTERFACE
   * @see #getType
   * @see #setType
   */
  public static final Type CLASS = new Type("class");

  /** <p>Type of a JavaSource interface.</p>
   * @see #CLASS
   * @see #getType
   * @see #setType
   */
  public static final Type INTERFACE = new Type("interface");

  /** <p>Protection of a class, field or method: public</p>
	*/
  public static final Protection PUBLIC = new Protection("public");

  /** <p>Protection of a class, field or method: protected</p>
	*/
  public static final Protection PROTECTED = new Protection("protected");

  /** <p>Protection of a class, field or method: private</p>
	*/
  public static final Protection PRIVATE = new Protection("private");

  /** <p>Default protection of a class, field or method</p>
	*/
  public static final Protection DEFAULT_PROTECTION = new Protection("");

  /** <p>Creates a new instance of JavaSource with the given protection.</p>
   *
   * @param pFactory The {@link JavaSourceFactory factory} creating this
   *   instance of JavaSource.
   * @param pName The class or interface name
   * @param pProtection null, "public", "protected" or "private"
   */
  JavaSource(JavaSourceFactory pFactory, JavaQName pName, Protection pProtection) {
    factory = pFactory;
    setQName(pName);
    setProtection(pProtection);
  }

  List myObjects = new ArrayList();

  /** <p>Returns the static class initializers.</p>
   */
  public JavaClassInitializer[] getClassInitializers() {
     List result = new ArrayList(myObjects);
     for (Iterator iter = myObjects.iterator();  iter.hasNext();  ) {
        ConditionalIndentationJavaSourceObject object =
         (ConditionalIndentationJavaSourceObject) iter.next();
        if (object instanceof JavaClassInitializer) {
          result.add(object);
        }
     }
     return (JavaClassInitializer[]) result.toArray(new JavaClassInitializer[result.size()]);
  }

  private JavaSourceFactory factory;
  /** <p>Returns the {@link JavaSourceFactory} that created this instance of
   * JavaSource.</p>
   */
  public JavaSourceFactory getFactory() {
    return factory;
  }

  private JavaQName myQName;
  /** <p>Returns the JavaSource's JavaQName.</p>
   */
  public JavaQName getQName() {
    return myQName;
  }
  /** <p>Sets the JavaSource's JavaQName.</p>
   */
  public void setQName(JavaQName pQName) {
    myQName = pQName;
  }
  /** <p>Returns the class or interface name.</p>
   *
   * @see #setQName
   */
  public String getClassName() { return myQName.getClassName(); }
  /** <p>Returns the package name. The empty String represents the
   * default package.</p>
   */
  public String getPackageName() { return myQName.getPackageName(); }

  private Protection myProtection;
  /** <p>Returns the protection.</p>
   *
   * @see #setProtection
   */
  public Protection getProtection() { return myProtection; }
  /** <p>Sets the protection; use null for default protection.</p>
   *
   * @see #getProtection
   */
  public void setProtection(Protection protection) {
  	 myProtection = (protection == null) ? DEFAULT_PROTECTION : protection;
  }

  private Type type = CLASS;
  /** <p>Returns the JavaSource type.</p>
   *
   * @return "class" or "interface"
   * @see #setType
   */
  public Type getType() { return type; }
  /** <p>Sets the JavaSource type.</p>
   *
   * @param pType "class" or "interface"
   * @see #getType
   */
  public void setType(Type pType) {
  	 this.type = (pType == null) ? CLASS : pType;
  }

  private JavaComment comment;
  /** <p>Returns the comment describing this class or interface.</p>
   *
   * @see #newComment
   */
  public JavaComment getComment() { return comment; }
  /** <p>Creates a new Javadoc comment describing this class or interface.</p>
   *
   * @see #getComment
   */
  public JavaComment newComment() {
  	 if (comment == null) {
  	 	comment = new JavaComment();
  	 	return comment;
  	 } else {
  	 	throw new IllegalStateException("A Javadoc comment has already been created for this object.");
  	 }
  }

  private List extendedClasses;
  /** <p>Clears the list of extended classes or interfaces.</p>
   */
  public void clearExtends() {
    extendedClasses = null;
  }
  /** <p>Returns the class or interface extended by this class or interface.</p>
   *
   * @see #addExtends(JavaQName)
   */
  public JavaQName[] getExtends() {
  	 if (extendedClasses == null) {
  	 	return new JavaQName[0];
  	 } else {
  	 	return (JavaQName[]) extendedClasses.toArray(new JavaQName[extendedClasses.size()]);
  	 }
  }
  /** <p>Sets the class or interface extended by this class or interface.
   * Null or the empty string disable the "extends" clause.</p>
   *
   * @see #getExtends
   */
  public void addExtends(JavaQName pExtends) {
  	 if (extendedClasses == null) {
  	 	extendedClasses = new ArrayList();
  	 } else if ("class".equals(getType())) {
  	 	throw new IllegalStateException("Only interfaces may extend multiple classes.");
  	 }
    extendedClasses.add(pExtends);
  }
  /** <p>Sets the class or interface extended by this class or interface.
   * Null or the empty string disable the "extends" clause.</p>
   *
   * @see #getExtends
   */
  public void addExtends(Class pExtends) {
  	 addExtends(JavaQNameImpl.getInstance(pExtends));
  }

  /** <p>Returns whether the class is extending the given super class or interface.</p>
   */
  public boolean isExtending(JavaQName pClass) {
    for (Iterator iter = extendedClasses.iterator();  iter.hasNext();  ) {
      if (iter.next().equals(pClass)) {
        return true;
      }
    }
    return false;
  }

  /** <p>Returns whether the class is extending the given super class or interface.</p>
   */
  public boolean isExtending(Class pClass) {
    return isExtending(JavaQNameImpl.getInstance(pClass));
  }

  private ArrayList imports = new ArrayList();
  /** <p>Returns the list of packages and classes being imported.</p>
   *
   * @see #addImport
   */
  public JavaQName[] getImports() {
    return (JavaQName[]) imports.toArray(new JavaQName[imports.size()]);
  }

  /** <p>Adds a package or class to the list of packages and classes being imported.</p>
   *
   * @see #addImport
   */
  public void addImport(JavaQName s) {
    if (s.isArray()) {
      throw new IllegalArgumentException("Arrays cannot be imported");
    }
    imports.add(s);
  }

  /** <p>Adds a package or class to the list of packages and classes being imported.</p>
   *
   * @see #addImport
   */
  public void addImport(Class s) { imports.add(JavaQNameImpl.getInstance(s)); }

  /** <p>Clears the list of imports.</p>
   */
  public void clearImports() { imports.clear(); }

  private ArrayList myImplements = new ArrayList();
  /** <p>Returns the list of interfaces being implented by this class or
   * interface.</p>
   *
   * @see #addImplements
   */
  public JavaQName[] getImplements() {
    return (JavaQName[]) myImplements.toArray(new JavaQName[myImplements.size()]);
  }

  /** <p>Adds an interface to the list of interfaces being implemented by
   * this class or interface.</p>
   *
   * @see #getImplements
   */
  public void addImplements(JavaQName s) { myImplements.add(s); }

  /** <p>Adds an interface to the list of interfaces being implemented by
   * this class or interface.</p>
   *
   * @see #getImplements
   */
  public void addImplements(Class s) { myImplements.add(JavaQNameImpl.getInstance(s)); }

  /** <p>Clears the list of implemented interfaces.</p>
   */
  public void clearImplements() { myImplements.clear(); }

  /** <p>Returns whether the class is implementing the given interface.</p>
   */
  public boolean isImplementing(JavaQName pClass) {
    for (Iterator iter = myImplements.iterator();  iter.hasNext();  ) {
      if (iter.next().equals(pClass)) {
        return true;
      }
    }
    return false;
  }

  /** <p>Returns whether the class is implementing the given interface.</p>
   */
  public boolean isImplementing(Class pClass) {
    return isImplementing(JavaQNameImpl.getInstance(pClass));
  }

  private ArrayList myFields = new ArrayList();
  /** <p>Returns the field with the given name or null, if no such field exists.</p>
   */
  public JavaField getField(String pName) {
     if (pName == null) {
        throw new NullPointerException("A field name must not be null.");
     }
     JavaField[] fields = getFields();
     for (int i = 0;  i < fields.length;  i++) {
        if (pName.equals(fields[i].getName())) {
           return fields[i];
        }
     }
     return null;
  }
  /** <p>Returns the list of fields that this class has.</p>
   *
   * @see #newJavaField
   */
  public JavaField[] getFields() {
    return (JavaField[]) myFields.toArray(new JavaField[myFields.size()]);
  }
  /** <p>Adds a field to this classes list of fields.</p>
   *
   * @see #getFields
   */
  void addField(JavaField f) {
     String s = f.getName();
     for (int i = 0;  i < myFields.size();  i++) {
        if (s.equals(((JavaField) myFields.get(i)).getName())) {
           throw new IllegalStateException("The class " + getQName() + " already has a field " + s + ".");
        }
     }
     myFields.add(f);
  }

  /** <p>Returns the list of constructors that this class has.</p>
   *
   * @see #newJavaConstructor
   */
  public JavaConstructor[] getConstructors() {
    List result = new ArrayList();
    for (Iterator iter = myObjects.iterator();  iter.hasNext();  ) {
      ConditionalIndentationJavaSourceObject object =
        (ConditionalIndentationJavaSourceObject) iter.next();
      if (object instanceof JavaConstructor) {
        result.add(object);
      }
    }
    return (JavaConstructor[]) result.toArray(new JavaConstructor[result.size()]);
  }
  /** <p>Adds a constructor to this classes list of constructors.</p>
   *
   * @see #getConstructors
   */
  void addConstructor(JavaConstructor c) { myObjects.add(c); }
  /** <p>Returns an iterator the the classes constructors. This
   * iterator allows to remove constructors.</p>
   */
  public Iterator getConstructorIterator() {
    return new MyObjectIterator(JavaConstructor.class);
  }

  /** <p>Returns the list of methods that this class has.</p>
   *
   * @see #newJavaMethod
   */
  public JavaMethod[] getMethods() {
    List result = new ArrayList();
    for (Iterator iter = myObjects.iterator();  iter.hasNext();  ) {
       ConditionalIndentationJavaSourceObject object = (ConditionalIndentationJavaSourceObject) iter.next();
       if (object instanceof JavaMethod) {
          result.add(object);
       }
    }
    return (JavaMethod[]) result.toArray(new JavaMethod[result.size()]);
  }

  /** <p>Returns the method with the given signature or null, if there
   * is no such method.</p>
   */
  public JavaMethod getMethod(String pMethodName, JavaQName[] pParams) {
    for (Iterator iter = myObjects.iterator();  iter.hasNext();  ) {
      ConditionalIndentationJavaSourceObject object = (ConditionalIndentationJavaSourceObject) iter.next();
      if (object instanceof JavaMethod) {
        JavaMethod jm = (JavaMethod) object;
        if (jm.getName().equals(pMethodName)) {
          Parameter[] parameters = jm.getParams();
          if (parameters.length == pParams.length) {
            boolean equals = true;
            for (int i = 0;  i < parameters.length;  i++) {
              if (!parameters[i].getType().equals(pParams[i])) {
                equals = false;
                break;
              }
            }
            if (equals) {
              return jm;
            }
          }
        }
      }
    }
    return null;
  }

  private class MyObjectIterator implements Iterator {
    private final Class instanceClass;
    public MyObjectIterator(Class pInstanceClass) {
      instanceClass = pInstanceClass;
    }

    private Object result = null;
    private boolean mayRemove;
    private Iterator inner = myObjects.iterator();
    public void remove() {
      if (!mayRemove) {
        throw new IllegalStateException("remove() is only allowed immediately after next()");
      }
      inner.remove();
      mayRemove = false;
    }
    public boolean hasNext() {
      mayRemove = false;
      if (result != null) {
        return true;
      }
      while (inner.hasNext()) {
        Object o = inner.next();
        if (instanceClass.isAssignableFrom(o.getClass())) {
          result = o;
          return true;
        }
      }
      result = null;
      return false;
    }
    public Object next() {
      if (!hasNext()) {
        throw new NoSuchElementException();
      }
      Object myResult = result;
      result = null;
      mayRemove = true;
      return myResult;
    }
  }

  /** <p>Returns an iterator to the classes methods. This iterator
   * allows to remove certain methods.</p>
   */
  public Iterator getMethodIterator() {
    return new MyObjectIterator(JavaMethod.class);
  }


  /** <p>Adds a method to this classes list of methods.</p>
   *
   * @see #getMethods()
   */
  void addMethod(JavaMethod m) { myObjects.add(m); }


  public void write(IndentationTarget pTarget) throws IOException {
  	 if (!isInnerClass()) {
		String packageName = getPackageName();
		if (packageName.length() > 0) {
		  pTarget.indent(0);
		  pTarget.write("package ");
		  pTarget.write(packageName);
		  pTarget.write(";");
		  pTarget.write();
		  pTarget.indent(0);
		  pTarget.write();
		}
  	 }

    JavaQName[] myExtendedClasses = getExtends();
    JavaQName[] implementedInterfaces = getImplements();
    JavaInnerClass[] myInnerClasses = getInnerClasses();
    JavaField[] fields = getFields();
    getConstructors();
    getMethods();
      
    JavaQName[] myImports = getImports();
    Arrays.sort(myImports);
    if (myImports.length > 0) {
      for (int i = 0;  i < myImports.length;  i++) {
        pTarget.indent(0);
        pTarget.write("import ");
        pTarget.write(myImports[i].toString());
        pTarget.write(";");
        pTarget.write();
      }
      pTarget.indent(0);
      pTarget.write();
      pTarget.indent(0);
      pTarget.write();
    }
    if (comment != null) {
      comment.write(pTarget);
      pTarget.indent(0);
      pTarget.write();
    }

    pTarget.indent(0);
    if (myProtection != null  &&  !myProtection.equals(DEFAULT_PROTECTION)) {
      pTarget.write(myProtection.toString());
      pTarget.write(" ");
    }
    if (isStatic) {
      pTarget.write("static ");
    }
    if (isAbstract()) {
      pTarget.write("abstract ");
    }
    pTarget.write(getType().toString());
    pTarget.write(" ");
    String s = getClassName();
    int offset = s.lastIndexOf('.');
    if (offset > -1) {
    	s = s.substring(offset+1);
    }
    offset = s.lastIndexOf('$');
	 if (offset > -1) {
	   s = s.substring(offset+1);
	 }
    pTarget.write(s);
    pTarget.write(" ");
    for (int i = 0;  i < myExtendedClasses.length;  i++) {
      if (i > 0) {
        pTarget.write(", ");
      } else {
        pTarget.write("extends ");
      }
      pTarget.write(pTarget.asString(myExtendedClasses[i]));
      pTarget.write(" ");
    }
    if (implementedInterfaces.length > 0) {
      for (int i = 0;  i < implementedInterfaces.length;  i++) {
        if (i == 0) {
          pTarget.write("implements ");
        } else {
          pTarget.write(", ");
        }
        pTarget.write(pTarget.asString(implementedInterfaces[i]));
        pTarget.write(" ");
      }
    }
    pTarget.write("{");
    pTarget.write();

    IncreasingTarget increasingTarget = new IncreasingTarget(pTarget);
    for (int i = 0;  i < myInnerClasses.length;  i++) {
    	increasingTarget.setInterface(myInnerClasses[i].isInterface() ?
    	                              Boolean.TRUE : Boolean.FALSE);
      myInnerClasses[i].write(increasingTarget);
      increasingTarget.setInterface(null);
      pTarget.indent(0);
      pTarget.write();
    }

    if (fields != null  &&  fields.length > 0) {
      for (int i = 0;  i < fields.length;  i++) {
        fields[i].write(increasingTarget);
        pTarget.indent(0);
        pTarget.write();
      }
      pTarget.indent(0);
      pTarget.write();
    }

    for (Iterator iter = myObjects.iterator();  iter.hasNext();  ) {
       ConditionalIndentationJavaSourceObject object =
         (ConditionalIndentationJavaSourceObject) iter.next();
       object.write(increasingTarget);
      pTarget.indent(0);
      pTarget.write();
    }

    String[] myRawJavaSources = getRawJavaSources();

    for (int i = 0;  i < myRawJavaSources.length;  i++) {
      for (StringTokenizer st = new StringTokenizer(myRawJavaSources[i], "\r\n");
           st.hasMoreTokens();  ) {
        pTarget.indent(0);
        String tok = st.nextToken();
        if (tok.length() > 0) {
          pTarget.write(tok);
        }
        pTarget.write();
      }
      pTarget.indent(0);
      pTarget.write();
    }
    pTarget.indent(0);
    pTarget.write("}");
    pTarget.write();
  }

  /** <p>Returns a quoted string constant suitable for embedding
   * into Java source, but without quotes.</p>
   */
  public static String getQuotedNoQuotes(String s) {
    StringBuffer sb = new StringBuffer();
    for (int i = 0;  i < s.length();  i++) {
      char c = s.charAt(i);
      if (c == '\n') {
        sb.append("\\n");
      } else if (c == '\\') {
        sb.append("\\\\");
      } else if (c == '\t') {
        sb.append("\\t");
      } else if (c == '\r') {
        sb.append("\\r");
      } else if (c == '\f') {
        sb.append("\\f");
      } else if (c == '\"') {
        sb.append("\\\"");
      } else {
        sb.append(c);
      }
    }
    return sb.toString();
  }

  /** <p>Returns a quoted string constant suitable for embedding
   * into Java source.</p>
   */
  public static String getQuoted(String s) {
    return "\"" + getQuotedNoQuotes(s) + "\"";
  }

  private java.util.List innerClasses;
  /** <p>Adds an inner class.</p>
   */
  public void addInnerClass(JavaInnerClass pClass) {
    if (innerClasses == null) {
      innerClasses = new java.util.ArrayList();
    }
    innerClasses.add(pClass);
  }
  /** <p>Clears the list of inner classes.</p>
   */
  public void clearInnerClasses() {
    innerClasses = null;
  }
  /** <p>Returns the array of inner classes.</p>
   */
  public JavaInnerClass[] getInnerClasses() {
    if (innerClasses == null) {
      return new JavaInnerClass[0];
    }
    return (JavaInnerClass[])
      innerClasses.toArray(new JavaInnerClass[innerClasses.size()]);
  }

  private boolean isStatic;
  /** <p>Returns whether this JavaSource is static (for inner classes).</p>
   */
  protected boolean getStatic() {
    return isStatic;
  }

  /** <p>Sets whether this JavaSource is static (for inner classes).</p>
   */
  public void setStatic(boolean pStatic) {
    isStatic = pStatic;
  }

  private java.util.List rawJavaSources;
  /** <p>Adds a piece of raw Java source to the class.</p>
   */
  public void addRawJavaSource(String pSource) {
    if (pSource != null  &&  pSource.length() > 0) {
      if (rawJavaSources == null) {
        rawJavaSources = new java.util.ArrayList();
      }
      rawJavaSources.add(pSource);
    }
  }

  /** <p>Clears the list of raw Java sources.</p>
   */
  public void clearRawJavaSources() {
    rawJavaSources = null;
  }

  /** <p>Returns an array with the pieces of raw Java sources.</p>
   */
  public String[] getRawJavaSources() {
    if (rawJavaSources == null) {
      return new String[0];
    }
    return (String[]) rawJavaSources.toArray(new String[rawJavaSources.size()]);
  }

  private boolean bAbstract;
  /** <p>Returns whether class is abstract.</p>
   */
  public boolean isAbstract() {
    return bAbstract;
  }
  /** <p>Sets whether this class is abstract,</p>
   */
  public void setAbstract(boolean isAbstract) {
    this.bAbstract = isAbstract;
  }

  /** <p>Returns whether this is an interface or not.</p>
   */
  public boolean isInterface() {
    return INTERFACE.equals(getType());
  }

  /** <p>Returns whether the given JavaQName is a local class.
   * In other words, whether the package name can be omitted
   * when referencing the class.</p>
   */
  public String asString(JavaQName pQName, boolean pAddIfPossible) {
  	 return _asString(pQName, pAddIfPossible).replace('$', '.');
  }

  private String _asString(JavaQName pQName, boolean pAddIfPossible) {
  	if (isForcingFullyQualifiedName()) {
  		return pQName.toString();
  	}
  	if (pQName.isArray()) {
  		return asString(pQName.getInstanceClass(), pAddIfPossible) + "[]";
  	}
  	if (!pQName.isImportable()) {
  		return pQName.toString();
  	}
  	if ("".equals(pQName.getPackageName())) {
  		return pQName.getClassName();
  	}

  	JavaQName outerQName = pQName;
  	int offset = outerQName.getClassName().indexOf('$');
  	if (offset >= 0) {
  		String className = outerQName.getClassName().substring(0, offset);
  		outerQName = JavaQNameImpl.getInstance(outerQName.getPackageName(), className);
  	}
  	
  	offset = outerQName.getClassName().indexOf('.');
  	if (offset >= 0) {
  		String className = pQName.getClassName().substring(0, offset);
  		outerQName = JavaQNameImpl.getInstance(outerQName.getPackageName(), className);
  	}

  	if (getQName().equals(outerQName)) {
  		return pQName.getClassName();
  	} else if (getQName().getClassName().equals(outerQName.getClassName())) {
  		return pQName.toString();
  	}

  	boolean done = false;
  	boolean imported = false;

  	for (Iterator iter = imports.iterator();  !done  &&  iter.hasNext();  ) {
  		JavaQName jqName = (JavaQName) iter.next();
  		if (jqName.equals(outerQName)) {
  			done = true;
  			imported = true;
  		} else if (outerQName.getClassName().equals(jqName.getClassName())) {
  			done = true;
  			imported = false;
  		}
  	}
  	
  	if (!done) {
  		String packageName = pQName.getPackageName();
  		if (packageName.equals(getPackageName())  ||
  				packageName.equals("java.lang")) {
  			imported = true;
  			done = true;
  		}
  		if (!done) {
  			if (pAddIfPossible) {
  				addImport(outerQName);
  				done = true;
  				imported = true;
  			} else {
  				done = true;
  				imported = false;
  			}
  		}
  	}
  	
  	if (imported) {
  		return pQName.getClassName();
  	} else {
  		return pQName.toString();
  	}
  }

  private boolean forcingFullyQualifiedName = false;
  public boolean isForcingFullyQualifiedName() {
  	return forcingFullyQualifiedName;
  }
  public void setForcingFullyQualifiedName(boolean pForcingFullyQualifiedName) {
  	forcingFullyQualifiedName = pForcingFullyQualifiedName;
  }

  private boolean hasDynamicImports = true;
  public boolean hasDynamicImports() {
    return hasDynamicImports;
  }
  public void setDynamicImports(boolean pDynamicImports) {
    hasDynamicImports = pDynamicImports;
  }

  /** <p>Writes the JavaSource contents into the given Writer.</p>
   */
  public void write(Writer pTarget) throws IOException {
    if (hasDynamicImports()) {
      IndentationTarget devNullTarget = new IndentationTarget(){
        public boolean isInterface() { return JavaSource.this.isInterface(); }
        public String asString(JavaQName pQName) {
          return JavaSource.this.asString(pQName, true);
        }
        public void write(String pValue) {}
        public void write() {}
        public void indent(int i) {}
      };
      write(devNullTarget);
    }
    WriterTarget wt = new WriterTarget(){
      public boolean isInterface() { return JavaSource.this.isInterface(); }
      public String asString(JavaQName pQName) {
        return JavaSource.this.asString(pQName, false);
      }
    };
    wt.setTarget(pTarget);
    write(wt);
  }

  /** <p>Returns a string representation of this JavaSource file.</p>
   */
  public String toString() {
    StringWriter sw = new StringWriter();
    try {
       write(sw);
       return sw.toString();
    } catch (IOException e) {
      throw new IllegalStateException("Unexcpected IOException while writing into a StringWriter: " + e.getMessage());
    }
  }

  /** Creates a new instance of JavaClassInitializer */
  public JavaClassInitializer newJavaClassInitializer() {
    JavaClassInitializer result = new JavaClassInitializer();
    result.setJavaSource(this);
    myObjects.add(result);
    return result;
  }

  /** Creates a new JavaConstructor with default protection */
  public JavaConstructor newJavaConstructor() {
    return newJavaConstructor(DEFAULT_PROTECTION);
  }

  /** Creates a new JavaConstructor with the given protection */
  public JavaConstructor newJavaConstructor(JavaSource.Protection pProtection) {
    JavaConstructor result = new JavaConstructor(getClassName(), pProtection);
    result.setJavaSource(this);
    this.addConstructor(result);
    return result;
  }

  /** <p>Creates a new JavaConstructor with the given protection.
   * Equivalent to <code>newJavaConstructor(Protection.valueOf(pProtection))</code>.</p>
   */
  public JavaConstructor newJavaConstructor(String pProtection) {
	 return newJavaConstructor(Protection.valueOf(pProtection));
  }

  /** <p>Creates a new JavaConstructor with the same parameters,
   * protection and exceptions than the given. Equivalent to
   * <code>newJavaConstructor(pConstructor, false)</code>.</p>
   */
  public JavaConstructor newJavaConstructor(JavaConstructor pConstructor) {
    return newJavaConstructor(pConstructor, false);
  }

  /** <p>Creates a new JavaConstructor with the same signature
   * than the given constructors. Equivalent to
   * <code>newJavaConstructor(pConstructor, false)</code>.
   * If the <code>pSuper</code> argument is true, adds an
   * invocation of the super classes constructor with the
   * same arguments.</p>
   */
  public JavaConstructor newJavaConstructor(JavaConstructor pConstructor, boolean pSuper) {
    JavaConstructor result = newJavaConstructor(pConstructor.getProtection());
    List superParams = pSuper ? new ArrayList() : null;
    Parameter[] params = pConstructor.getParams();
    for (int i = 0;  i < params.length;  i++) {
      DirectAccessible p = result.addParam(params[i]);
      if (pSuper) {
        if (!superParams.isEmpty()) {
          superParams.add(", ");
        }
        superParams.add(p);
      }
    }
    JavaQName[] exceptions = pConstructor.getExceptions();
    for (int i = 0;  i < exceptions.length;  i++) {
      result.addThrows(exceptions[i]);
    }
    if (pSuper) {
       result.addLine("super(", superParams, ");");
    }
    return result;
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and default protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, String pType) {
    return newJavaMethod(pName, JavaQNameImpl.getInstance(pType), (Protection) null);
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, String pType, Protection pProtection) {
    return newJavaMethod(pName, JavaQNameImpl.getInstance(pType), pProtection);
  }

  /** <p>Creates a new JavaMethod with the given name, return
	* type and protection.</p>
	*/
  public JavaMethod newJavaMethod(String pName, String pType, String pProtection) {
	 return newJavaMethod(pName, JavaQNameImpl.getInstance(pType),
	                       Protection.valueOf(pProtection));
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and default protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, JavaQName pType) {
    return newJavaMethod(pName, pType, DEFAULT_PROTECTION);
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, JavaQName pType, Protection pProtection) {
    JavaMethod result = new JavaMethod(pName, pType, pProtection);
    result.setJavaSource(this);
    addMethod(result);
    return result;
  }

  /** <p>Creates a new JavaMethod with the given name, return
	* type and protection.</p>
	*/
  public JavaMethod newJavaMethod(String pName, JavaQName pType, String pProtection) {
	 JavaMethod result = new JavaMethod(pName, pType, Protection.valueOf(pProtection));
	 result.setJavaSource(this);
	 addMethod(result);
	 return result;
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and default protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, Class pType) {
    return newJavaMethod(pName, JavaQNameImpl.getInstance(pType), DEFAULT_PROTECTION);
  }

  /** <p>Creates a new JavaMethod with the given name, return
   * type and protection.</p>
   */
  public JavaMethod newJavaMethod(String pName, Class pType, Protection pProtection) {
    return newJavaMethod(pName, JavaQNameImpl.getInstance(pType), pProtection);
  }

  /** <p>Creates a new JavaMethod with the given name, return
	* type and protection.</p>
	*/
  public JavaMethod newJavaMethod(String pName, Class pType, String pProtection) {
	 return newJavaMethod(pName, JavaQNameImpl.getInstance(pType),
	                      Protection.valueOf(pProtection));
  }

  /** <p>Creates a new JavaMethod with the signature of the given method
	* <code>pMethod</code>.
	* More precise:
	* <ol>
	*   <li>The name of the created method is <code>pMethod.getName()</code>.</li>
	*   <li>The return type is set to <code>pMethod.getReturnType()</code>.</li>
	*   <li>The protection is set to <code>pMethod.
	*   <li>For any class in <code>pMethod.getParameterTypes()</code>, calls
	*     {@link JavaMethod#addParam(Class,String)} with the parameter names "p0", "p1", ...</li>
	*   <li>For any exception in <code>pMethod.getExceptionTypes()</code>, calls
	*     {@link JavaMethod#addThrows(Class)}.</li>
	* </ol>
	*/
  public JavaMethod newJavaMethod(Method pMethod) {
     JavaMethod result = newJavaMethod(pMethod.getName(),
                                       JavaQNameImpl.getInstance(pMethod.getReturnType()),
                                       JavaSource.Protection.valueOf(pMethod.getModifiers()));
     Class[] parameters = pMethod.getParameterTypes();
     if (parameters != null) {
       for (int i = 0;  i < parameters.length;  i++) {
         String parameterName = "p" + i;
         result.addParam(parameters[i], parameterName);
       }
     }
     Class[] exceptions = pMethod.getExceptionTypes();
     if (exceptions != null) {
       for (int i = 0;  i < exceptions.length;  i++) {
         result.addThrows(exceptions[i]);
       }
     }
  	  return result;
  }

  /** <p>Creates a new JavaMethod with the given methods name and signature.
   * This is useful, for example, if you have an interface and derive an
   * implementation. The following values are not cloned:
   * {@link JavaMethod#isAbstract()}, {@link JavaMethod#isStatic()},
   * {@link JavaMethod#isFinal()}, and {@link JavaMethod#isSynchronized()}.</p>
   */
  public JavaMethod newJavaMethod(JavaMethod pMethod) {
     JavaMethod jm = newJavaMethod(pMethod.getName(), pMethod.getType(), pMethod.getProtection());
     Parameter[] params = pMethod.getParams();
     for (int i = 0;  i < params.length;  i++) {
        jm.addParam(params[i].getType(), params[i].getName());
     }
     JavaQName[] exceptions = pMethod.getExceptions();
     for (int i = 0;  i < exceptions.length;  i++) {
        jm.addThrows(exceptions[i]);
     }
     return jm;
  }

  /** <p>Creates a new JavaField with the given name, type and
   * protection.</p>
   */
  public JavaField newJavaField(String pName, JavaQName pType, Protection pProtection) {
    JavaField result = new JavaField(pName, pType, pProtection);
    result.setJavaSource(this);
    addField(result);
    return result;
  }

  /** <p>Creates a new JavaField with the given name, type and
	* protection.</p>
	*/
  public JavaField newJavaField(String pName, JavaQName pType, String pProtection) {
	 JavaField result = new JavaField(pName, pType, Protection.valueOf(pProtection));
	 result.setJavaSource(this);
	 addField(result);
	 return result;
  }

  /** <p>Creates a new JavaField with the given name, type and
   * default protection.</p>
   */
  public JavaField newJavaField(String pName, JavaQName pType) {
    return newJavaField(pName, pType, DEFAULT_PROTECTION);
  }

  /** <p>Creates a new JavaField with the given name, type and
   * protection.</p>
   */
  public JavaField newJavaField(String pName, String pType, Protection pProtection) {
    return newJavaField(pName, JavaQNameImpl.getInstance(pType), pProtection);
  }

  /** <p>Creates a new JavaField with the given name, type and
	* protection.</p>
	*/
  public JavaField newJavaField(String pName, String pType, String pProtection) {
	 return newJavaField(pName, JavaQNameImpl.getInstance(pType),
	                     Protection.valueOf(pProtection));
  }

  /** <p>Creates a new JavaField with the given name, type and
   * default protection.</p>
   */
  public JavaField newJavaField(String pName, String pType) {
    return newJavaField(pName, JavaQNameImpl.getInstance(pType), (Protection) null);
  }

  /** <p>Creates a new JavaField with the given name, type and
   * protection.</p>
   */
  public JavaField newJavaField(String pName, Class pType, JavaSource.Protection pProtection) {
    return newJavaField(pName, JavaQNameImpl.getInstance(pType), pProtection);
  }

  /** <p>Creates a new JavaField with the given name, type and
	* protection. Equivalent to <code>newJavaField(pName, pType,
	* Protection.valueOf(pProtecttion)</code>.</p>
	*/
  public JavaField newJavaField(String pName, Class pType, String pProtection) {
	 return newJavaField(pName, JavaQNameImpl.getInstance(pType),
	                      Protection.valueOf(pProtection));
  }

  /** <p>Creates a new JavaField with the given name, type and
   * default protection.</p>
   */
  public JavaField newJavaField(String pName, Class pType) {
    return newJavaField(pName, JavaQNameImpl.getInstance(pType));
  }

   /** <p>Creates a new JavaInnerClass with the given name and
    * default protection.</p>
    */
   public JavaInnerClass newJavaInnerClass(String pName) {
  	   return newJavaInnerClass(pName, DEFAULT_PROTECTION);
   }

	/** <p>Creates a new JavaInnerClass with the given name and
	 * protection.</p>
	 */
	public JavaInnerClass newJavaInnerClass(String pName, Protection pProtection) {
		JavaQName innerClassName = JavaQNameImpl.getInnerInstance(getQName(), pName);
		JavaInnerClass result = new JavaInnerClass(this, innerClassName, pProtection);
		addInnerClass(result);
		return result;
	}

	/** <p>Creates a new JavaInnerClass with the given name and
	 * protection.</p>
	 */
	public JavaInnerClass newJavaInnerClass(String pName, String pProtection) {
		return newJavaInnerClass(pName, Protection.valueOf(pProtection));
	}

   /** <p>Returns, whether this is an inner class.</p>
    */
   public boolean isInnerClass() {
   	return false;
   }
  /** <p>Creates a new Java property with the given type and name.
   * Shortcut for
   * <pre>
   * String upperCaseName = Character.toUpperCase(pName.charAt(0)) +
   *     pName.substring(1);
   * if (JavaQNameImpl.VOID.equals(pType)) {
   *   newBeanProperty(pType, pName, "is" + upperCaseName, "set" + upperCaseName);
   * } else {
   *   newBeanProperty(pType, pName, "get" + upperCaseName, "set" + upperCaseName);
   * }
   * </pre>
   */
  public void newBeanProperty(JavaQName pType, String pName) {
    String upperCaseName = Character.toUpperCase(pName.charAt(0)) + pName.substring(1);
    if (JavaQNameImpl.VOID.equals(pType)) {
      newBeanProperty(pType, pName, "is" + upperCaseName, "set" + upperCaseName);
    } else {
      newBeanProperty(pType, pName, "get" + upperCaseName, "set" + upperCaseName);
    }
  }

  /** <p>Shortcut for <code>newBeanProperty(JavaQNameImpl.getInstance(pClass), pName)</code>.
   * @see #newBeanProperty(JavaQName, String)
   */
  public void newBeanProperty(Class pClass, String pName) {
    newBeanProperty(JavaQNameImpl.getInstance(pClass), pName);
  }

  /** <p>Shortcut for
   * <pre>
   * newJavaField(pFieldName, pType, JavaSource.PRIVATE);
   * JavaMethod getMethod = newJavaMethod(pGetMethodName, pType, JavaSource.PUBLIC);
   * getMethod.addLine("return this.", pFieldName, ";");
   * JavaMethod setMethod = newJavaMethod(pSetMethodName, JavaQNameImpl.VOID, JavaSource.PUBLIC);
   * setMethod.addParam(pType, pFieldName);
   * setMethod.addLine("this.", pFieldName, " = ", pFieldName, ";");
   * </pre>
   * @param pType The property type
   * @param pFieldName The name of the generated field. The generated field has private
   * access.
   * @param pGetMethodName The name of the generated get method or null, if no such
   * method is being created.
   * @param pSetMethodName The name of the generated set method or null, if no such
   * method is being created.
   */
  public void newBeanProperty(JavaQName pType, String pFieldName,
                               String pGetMethodName, String pSetMethodName) {
    newJavaField(pFieldName, pType, JavaSource.PRIVATE);
    if (pGetMethodName != null) {
      JavaMethod getMethod = newJavaMethod(pGetMethodName, pType, JavaSource.PUBLIC);
      getMethod.addLine("return this.", pFieldName, ";");
    }
    if (pSetMethodName != null) {
      JavaMethod setMethod = newJavaMethod(pSetMethodName, JavaQNameImpl.VOID, JavaSource.PUBLIC);
      setMethod.addParam(pType, pFieldName);
      setMethod.addLine("this.", pFieldName, " = ", pFieldName, ";");
    }
  }

  /** <p>Shortcut for <code>newBeanProperty(JavaQNameImpl.getInstance(pClass),
   * pFieldName, pGetMethodName, pSetMethodName)</code>.</p>
   * @see #newBeanProperty(JavaQName, String, String, String)
   */
  public void newBeanProperty(Class pClass, String pFieldName,
                               String pGetMethodName, String pSetMethodName) {
    newBeanProperty(JavaQNameImpl.getInstance(pClass), pFieldName, pGetMethodName,
    pSetMethodName);
  }
}
