/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.js.pattern;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.ws.jaxme.js.DirectAccessible;
import org.apache.ws.jaxme.js.JavaMethod;
import org.apache.ws.jaxme.js.JavaQName;
import org.apache.ws.jaxme.js.JavaQNameImpl;
import org.apache.ws.jaxme.js.JavaSource;
import org.apache.ws.jaxme.js.JavaSourceFactory;
import org.apache.ws.jaxme.js.pattern.VersionGenerator.TableInfo;
import org.apache.ws.jaxme.logging.AntProjectLoggerFactory;
import org.apache.ws.jaxme.logging.LoggerAccess;
import org.apache.ws.jaxme.logging.LoggerFactory;
import org.apache.ws.jaxme.sqls.Column;
import org.apache.ws.jaxme.sqls.Index;
import org.apache.ws.jaxme.sqls.SQLFactory;
import org.apache.ws.jaxme.sqls.Schema;
import org.apache.ws.jaxme.sqls.Table;
import org.apache.ws.jaxme.sqls.impl.ColumnImpl;
import org.apache.ws.jaxme.sqls.impl.SQLFactoryImpl;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;


/** <p>A set of Ant tasks for running the generators in the
 * pattern package.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: Ant.java,v 1.2 2004/02/16 23:39:54 jochen Exp $
 */
public class Ant {
  protected abstract static class ReallyBasicAntTask extends Task {
    private File destDir;
    private boolean settingLoggerFactory = true;
    public void setSettingLoggerFactory(boolean pSettingLoggerFactory) {
      settingLoggerFactory = pSettingLoggerFactory;
    }
    public boolean isSettingLoggerFactory() {
      return settingLoggerFactory;
    }
    public void setDestDir(File pDir) {
      destDir = pDir;
    }
    public File getDestDir() {
       return destDir;
    }
    public void finish() {
    }
    public abstract void doExecute();
    public void execute() {
      if (isSettingLoggerFactory()) {
        LoggerFactory loggerFactory = LoggerAccess.getLoggerFactory();
        if (!(loggerFactory instanceof AntProjectLoggerFactory)) {
          LoggerAccess.setLoggerFactory(new AntProjectLoggerFactory(this));
        }
      }
      finish();
      doExecute();
    }
  }

  protected abstract static class BasicAntTask extends ReallyBasicAntTask {
    private JavaQName targetClass;
    public void setTargetClass(String pTargetClass) {
	   targetClass = getJavaQName(pTargetClass);
    }
    public void finish() {
		  if (targetClass == null) {
		    throw new BuildException("The attribute 'targetClass' must be set.");
		  }
	  }
	  public abstract void generate(JavaSourceFactory pFactory, JavaQName pTargetClass)
	      throws Exception;
	  public void doExecute() {
	 	  finish();
	 	  try {
			  JavaSourceFactory factory = new JavaSourceFactory();
			  generate(factory, targetClass);
			  factory.write(getDestDir());
	 	  } catch (Exception e) {
	 		  throw new BuildException(e, getLocation());
	 	  }
	  }
  }

  protected static JavaQName getJavaQName(String pName) {
    int offset = pName.lastIndexOf('.');
    if (offset == -1) {
      return JavaQNameImpl.getInstance(pName);
    } else {
      return JavaQNameImpl.getInstance(pName.substring(0, offset),
                                        pName.substring(offset+1));
    }
  }

  public static class AntProxyGenerator extends BasicAntTask {
    private JavaQName extendedClass;
    private List implementedInterfaces = new ArrayList();

    public void setExtendedClass(String pTargetClass) {
      extendedClass = getJavaQName(pTargetClass);
    }
    public ProxyGenerator.InterfaceDescription createImplementedInterface() {
      ProxyGenerator.InterfaceDescription result = new ProxyGenerator.InterfaceDescription();
      implementedInterfaces.add(result);
      return result;
    }
    public void finish() {
    	super.finish();
      if (implementedInterfaces.size() == 0) {
        throw new BuildException("You must specify at least one interface being implemented (child element 'implementedInterface')");
      }
    }
    public void generate(JavaSourceFactory pFactory, JavaQName pTargetClass)
        throws BuildException {
        ProxyGenerator proxyGenerator = new ProxyGenerator();
        if (extendedClass != null) {
          proxyGenerator.setExtendedClass(extendedClass);
        }
      proxyGenerator.generate(pFactory, pTargetClass,
                                (ProxyGenerator.InterfaceDescription[])
                                  implementedInterfaces.toArray(new ProxyGenerator.InterfaceDescription[implementedInterfaces.size()]));
    }
  }

  public static class AntTypesafeEnumerationGenerator extends BasicAntTask {
	  private List items = new ArrayList();
	  private boolean isAddingEquals = true;

    public void setAddingEquals(boolean pAddingEquals) {
	 	  isAddingEquals = pAddingEquals;
    }

    public TypesafeEnumerationGenerator.Item createItem() {
	 	  TypesafeEnumerationGenerator.Item item = new TypesafeEnumerationGenerator.Item();
	 	  items.add(item);
	 	  return item;
	  }

    public void finish() {
	 	  super.finish();
	 	  if (items.size() == 0) {
	 		  throw new BuildException("The generated enumeration must have at least a single item.");
	 	  }
	  }

    public void generate(JavaSourceFactory pFactory, JavaQName pTargetClass)
        throws Exception {
    	TypesafeEnumerationGenerator generator = new TypesafeEnumerationGenerator();
    	generator.setAddingEquals(isAddingEquals);
		  TypesafeEnumerationGenerator.Item[] myItems = (TypesafeEnumerationGenerator.Item[])
		  this.items.toArray(new TypesafeEnumerationGenerator.Item[this.items.size()]);
		  generator.generate(pFactory, pTargetClass, myItems);
    }
  }

  public static class AntChainGenerator extends ReallyBasicAntTask {
     private List chains = new ArrayList();
     public ChainGenerator createChain() {
        ChainGenerator chain = new ChainGenerator();
        chains.add(chain);
        return chain;
     }
     public void finish() {
        if (chains.size() == 0) {
           throw new BuildException("At least one nested 'chain' element must be given.",
                                     getLocation());
        }
     }
     public void doExecute() {
        JavaSourceFactory pFactory = new JavaSourceFactory();
        for (Iterator iter = chains.iterator();  iter.hasNext();  ) {
           ChainGenerator chain = (ChainGenerator) iter.next();
           chain.generate(pFactory);
        }
        try {
           pFactory.write(getDestDir());
        } catch (IOException e) {
           throw new BuildException(e, getLocation());
        }
     }
  }

  public static class AntVersionGenerator extends BasicAntTask {
    private String driver, url, user, password, schema, verColumn;
    private List tables;
    private boolean isGeneratingLogging;

    public String getDriver() { return driver; }
    public void setDriver(String pDriver) { driver = pDriver; }
    public String getPassword() { return password; }
    public void setPassword(String pPassword) { password = pPassword; }
    public String getUrl() { return url; }
    public void setUrl(String pUrl) { url = pUrl; }
    public String getUser() { return user; }
    public void setUser(String pUser) { user = pUser; }
    public String getSchema() { return schema; }
    public void setSchema(String pSchema) { schema = pSchema; }
    public void setTables(String pTables) {
      tables = new ArrayList();
      for (StringTokenizer st = new StringTokenizer(pTables);  st.hasMoreTokens();  ) {
        String tableName = st.nextToken();
        tables.add(tableName);
      }
    }
    public List getTables() {
      return tables;
    }
    public void setVerColumn(String pColumn) {
      verColumn = pColumn;
    }
    public String getVerColumn() {
      return verColumn;
    }
    public void setGeneratingLogging(boolean pGeneratingLogging) {
      isGeneratingLogging = pGeneratingLogging;
    }
    public boolean isGeneratingLogging() {
      return isGeneratingLogging;
    }

    protected Connection getConnection() throws ClassNotFoundException, SQLException {
       String myUrl = getUrl();
       if (myUrl == null) {
         throw new NullPointerException("Missing 'url' attribute");
       }

       String myDriver = getDriver();
       if (myDriver != null) {
         try {
           Class.forName(myDriver);
         } catch (ClassNotFoundException ex) {
           try {
             Thread.currentThread().getContextClassLoader().loadClass(myDriver);
           } catch (ClassNotFoundException ex2) {
             throw ex;
           }
         }
       }       

       return DriverManager.getConnection(myUrl, getUser(), getPassword());
    }

    private class IdIncrementer implements VersionGenerator.ColumnUpdater {
      private final List columns;
      IdIncrementer(List pColumns) {
        columns = pColumns;
      }
      public void update(JavaMethod pMethod, TableInfo pTableInfo, DirectAccessible pConnection, DirectAccessible pMap, DirectAccessible pRow) {
        for (Iterator iter = columns.iterator();  iter.hasNext();  ) {
          Integer columnNum = (Integer) iter.next();
          pMethod.addLine(pRow, "[", columnNum, "] = Long.toString(Long.parseLong((String) ",
                          pRow, "[", columnNum, "])+1);");
        }
      }
    } 

    private class VerNumIncrementer implements VersionGenerator.ColumnUpdater {
      private final int columnNumber;
      VerNumIncrementer(int pColumnNumber) {
        columnNumber = pColumnNumber;
      }
      public void update(JavaMethod pMethod, TableInfo pTableInfo, DirectAccessible pConnection, DirectAccessible pMap, DirectAccessible pRow) {
        pMethod.addLine(pRow, "[" + columnNumber + "] = new Integer(((Integer) ",
                        pRow, "[" + columnNumber + "]).intValue()+1);");
      }
    }

    public void generate(JavaSourceFactory pFactory, JavaQName pTargetClass) throws Exception {
      List myTables = getTables();
      if (myTables == null) {
        throw new NullPointerException("Missing 'tables' attribute");
      }
      if (getVerColumn() == null) {
        throw new NullPointerException("Missing 'verColumn' attribute");
      }
      Column.Name columnName = new ColumnImpl.NameImpl(getVerColumn());

      SQLFactory factory = new SQLFactoryImpl();
      Schema sch = factory.getSchema(getConnection(), getSchema());

      VersionGenerator versionGenerator = new VersionGenerator();
      versionGenerator.setGeneratingLogging(isGeneratingLogging());
      boolean isFirstTable = true;

      for (Iterator iter = myTables.iterator();  iter.hasNext();  ) {
        String tableName = (String) iter.next();
        Table table = sch.getTable(tableName);
        if (table == null) {
          throw new IllegalArgumentException("Invalid table name: " + tableName);
        }

        VersionGenerator.ColumnUpdater columnUpdater;
        if (isFirstTable) {
          Column column = null;
          int columnNum = -1;
          int i = 0;
          for (Iterator colIter = table.getColumns();  colIter.hasNext();  i++) {
            Column colIterColumn = (Column) colIter.next();
            if (colIterColumn.getName().equals(columnName)) {
              column = colIterColumn;
              columnNum = i;
              break;
            }
          }
          if (column == null) {
            throw new IllegalArgumentException("No column " + columnName +
                                               " found in table " + table.getQName());
          }
          isFirstTable = false;
          columnUpdater = new VerNumIncrementer(columnNum);
        } else {
          List pkColumns = new ArrayList();
          Index primaryKey = table.getPrimaryKey();
          if (primaryKey != null) {
            for (Iterator pkIter = primaryKey.getColumns();  pkIter.hasNext();  ) {
              Column pkColumn = (Column) pkIter.next();
              int columnNum = -1;
              int i = 0;
              for (Iterator colIter = table.getColumns();  colIter.hasNext();  i++) {
                Column colIterColumn = (Column) colIter.next();
                if (colIterColumn.getName().equals(pkColumn.getName())) {
                  columnNum = i;
                  break;
                }
              }
              if (columnNum == -1) {
                throw new IllegalStateException("Primary key column " + pkColumn.getQName() +
                                                " not found in table " + table.getQName());
              }
              pkColumns.add(new Integer(columnNum));
            }
          }
          if (pkColumns.size() == 0) {
            throw new IllegalArgumentException("The table " + table.getQName() +
                                               " doesn't have a primary key.");
          }
          columnUpdater = new IdIncrementer(pkColumns);
        }
        versionGenerator.addTable(table, columnUpdater);
      }

      JavaSource js = pFactory.newJavaSource(pTargetClass);
      versionGenerator.getCloneMethod(js);
    }
  }
}
