/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.js.pattern;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.ws.jaxme.js.JavaConstructor;
import org.apache.ws.jaxme.js.JavaField;
import org.apache.ws.jaxme.js.JavaMethod;
import org.apache.ws.jaxme.js.JavaQName;
import org.apache.ws.jaxme.js.JavaSource;
import org.apache.ws.jaxme.js.JavaSourceFactory;
import org.apache.ws.jaxme.js.Parameter;


/** <p>This class is a generator for the proxy object pattern. A proxy
 * object performs the same task as an object created by the
 * {@link java.lang.reflect.Proxy} class: It delegates its method
 * calls to an internal instance.</p>
 * <p>In the case of {@link java.lang.reflect.Proxy} this works by
 * calling a so-called {@link java.lang.reflect.InvocationHandler}.
 * The InvocationHandler calls the actual object via Java reflection.</p>
 * <p>In our case, the proxy object is an instance of a generated
 * class. The main advantage of the generated approach is that you
 * can customize the proxy class quite easily by overwriting it.
 * Compared to the creation of an InvocationHandler, this saves a
 * lot of hazzle.</p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 * @version $Id: ProxyGenerator.java,v 1.3 2004/02/16 23:39:54 jochen Exp $
 */
public class ProxyGenerator {
  /** <p>This class describes a generated method. The class is used to guarantee,
   * that the generated methods are unique, even if multiple interfaces define
   * them.</p>
   */
  protected static class GeneratedMethod implements Serializable, Comparable {
   	private JavaMethod method;
   	private String name;
   	private Class[] parameters;
   	private Class declaringInterface;
		/** <p>Sets the JavaMethod generated for this method.</p>
		 */
		public void setMethod(JavaMethod pMethod) { method = pMethod; }
		/** <p>Returns the JavaMethod generated for this method.</p>
		 */
		public JavaMethod getMethod() { return method; }
		/** <p>Sets the methods name.</p>
		 */
		public void setName(String pName) { name = pName; }
		/** <p>Returns the methods name.</p>
		 */
		public String getName() { return name; }
		/** <p>Sets the methods parameters.</p>
		 */
		public void setParameters(Class[] pParameters) { parameters = pParameters; }
		/** <p>Returns the methods parameters.</p>
		 */
		public Class[] getParameters() { return parameters; }
		/** <p>Sets the interface declaring this method.</p>
		 */
		public void setDeclaringInterface(Class pInterface)  {
			declaringInterface = pInterface;
		}
		/** <p>Returns the interface declaring this method.</p>
		 */
		public Class getDeclaringInterface()  {
			return declaringInterface;
		}
		/** <p>Returns whether this GeneratedMethod equals the object <code>o</code>.
		 * This is the case, if <code>o != null</code>, <code>o instanceof GeneratedMethod</code>,
		 * and <code>compareTo(o) != 0</code>.</p>
		 */
		public boolean equals(Object o) {
			if (o == null  ||  !(o instanceof GeneratedMethod)) { return false; }
			return compareTo(o) == 0;
		}
		/** <p>Compares this GeneratedMethod to the given GeneratedMethod <code>o</code>.
		 * More precise, compares the method name, the number of parameters and the
		 * class names of the parameters, in that order.</p>
		 * @throws ClassCastException The object o is not an instance of GeneratedMethod.
		 */
		public int compareTo(Object o) {
			GeneratedMethod gm = (GeneratedMethod) o;
			if (name == null) {
				if (gm.name != null) { return -1; }
			} else if (gm.name == null) {
				return 1;
			} else {
				int result = name.compareTo(gm.name);
				if (result != 0) { return result; }
			}
			if (parameters == null) {
				if (gm.parameters != null) { return -1; }
			} else if (gm.parameters == null) {
				return 1;
			} else {
				if (parameters.length != gm.parameters.length) {
					return parameters.length - gm.parameters.length;
				}
				for (int i = 0;  i < parameters.length;  i++) {
					Class c1 = parameters[i];
					Class c2 = gm.parameters[i];
					if (c1 == null) {
						if (c2 != null) { return -1; }
					} else if (c2 == null) {
						return 1;
					} else {
						int result = c1.getName().compareTo(c2.getName());
						if (result != 0) { return result; }
					}
				}
			}
			return 0;
		}
		public int hashCode() {
			int result = (name == null) ? 0 : name.hashCode();
			if (parameters != null) {
				result += parameters.length;
				for (int i = 0;  i < parameters.length;  i++) {
					Class c = parameters[i];
					if (c != null) {
						result += c.getName().hashCode();
					}
				}
			}
			return result;
		}
  }


   /** <p>This class describes the properties of an interface, which is extended
    * by the generated class.</p>
    */
	public static class InterfaceDescription {
	 Class interfaceClass;
	 boolean isMandatory = true;

    /** <p>Sets the Java interface. Similar to
     * <code>setInterface(Class.forName(pName))</code>.</p>
     */
    public void setInterfaceName(String pName) throws ClassNotFoundException {
      ClassNotFoundException ex = null;
      try {
        Class c = Class.forName(pName);
        if (c != null) {
          setInterface(c);
          return;
        }
      } catch (ClassNotFoundException e) {
        ex = e;
      }
      try {
        Class c = Thread.currentThread().getContextClassLoader().loadClass(pName);
        if (c != null) {
          setInterface(c);
          return;
        }
      } catch (ClassNotFoundException e) {
        if (ex == null) { ex = e; }
      }
      if (ex == null) {
        ex = new ClassNotFoundException("Failed to load class: " + pName);
      }
      throw ex;
    }
		/** <p>Sets the Java interface.</p>
		 */
		public void setInterface(Class pClass) {
			if (!pClass.isInterface()) {
				throw new IllegalArgumentException("The class " + pClass.getName() +
				                                    " is not an interface.");
			}
			interfaceClass = pClass;
		}
		/** <p>Returns the Java interface.</p>
		 */
		public Class getInterface() { return interfaceClass; }
		/** <p>Sets whether this interface is mandatory. By default interfaces
		 * are mandatory and the backing objects must implement this interface.
		 * If an interface isn't mandatory, then a Proxy instance can be created
		 * even for objects which don't implement the interface. However, in that
		 * case it may happen that a ClassCastException is thrown while invoking
		 * a method declared by the interface.</p>
		 */
		public void setMandatory(boolean pMandatory) { isMandatory = pMandatory; }
		/** <p>Returns whether this interface is mandatory. By default interfaces
		 * are mandatory and the backing objects must implement this interface.
		 * If an interface isn't mandatory, then a Proxy instance can be created
		 * even for objects which don't implement the interface. However, in that
		 * case it may happen that a ClassCastException is thrown while invoking
		 * a method declared by the interface.</p>
		 */
		public boolean isMandatory() { return isMandatory; }
	}

  private JavaQName extendedClass;

  /** <p>Returns the class extended by the generated proxy class.
   * Defaults to {@link java.lang.Object}.</p>
   */
  public JavaQName getExtendedClass() {
    return extendedClass;
  }

  /** <p>Sets the class extended by the generated proxy class.
   * Defaults to {@link java.lang.Object}.</p>
   */
  public void setExtendedClass(JavaQName pExtendedClass) {
    extendedClass = pExtendedClass;
  }

	/** <p>Converts the given {@link Method} into an instance of
	 * {@link GeneratedMethod}.</p>
	 */
	protected GeneratedMethod getGeneratedMethod(Method pMethod) {
		GeneratedMethod result = new GeneratedMethod();
		result.setDeclaringInterface(pMethod.getDeclaringClass());
		result.setName(pMethod.getName());
		result.setParameters(pMethod.getParameterTypes());
		return result;
	}

  /** <p>Generated an instance of {@link JavaMethod} for the given
   * {@link Method}.</p>
   */
  protected JavaMethod getInterfaceMethod(JavaSource pJs,
   													               InterfaceDescription pInterfaceDescription,
   													               Method pMethod) {
   	JavaMethod jm = pJs.newJavaMethod(pMethod);
      Parameter[] parameters = jm.getParams();
		List callParameters = new ArrayList();
      for (int i = 0;  i < parameters.length;  i++) {
			Parameter parameter = parameters[i];
				if (callParameters.size() > 0) {
					callParameters.add(", "); 
				}
				callParameters.add(parameter.getName());
			}
		jm.addLine((Void.TYPE.equals(pMethod.getReturnType()) ? "" : " return "),
					  "((", pInterfaceDescription.getInterface(), ") backingObject).",
					  pMethod.getName(), "(", callParameters, ");");
		return jm;
	}

  /** <p>Generates the methods for a given interface.</p>
   *
   * @param pJs The Java class being generated
   * @param pGeneratedMethods A set of already generated methods; each entry in the
   *    set is an instance of {@link GeneratedMethod}. The method creates a new instance
   *    of {@link GeneratedMethod} and adds it to the set. A warning is written to
   *    {@link System#err}, if the method isn't unique.
   */
  protected void generateInterfaceMethods(JavaSource pJs, Map pGeneratedMethods,
                                            InterfaceDescription pDescription) {
    Class c = pDescription.interfaceClass;
    Method[] methods = c.getMethods();
    if (methods != null) {
      for (int i = 0;  i < methods.length;  i++) {
        Method method = methods[i];
        GeneratedMethod generatedMethod = getGeneratedMethod(method);
        GeneratedMethod existingMethod = (GeneratedMethod) pGeneratedMethods.get(generatedMethod);
        if (existingMethod != null) {
          if (generatedMethod.getDeclaringInterface().equals(existingMethod.getDeclaringInterface())) {
            // Already done, skip it
					  continue;
				  } else {
					  System.err.println("The interfaces " + existingMethod.getDeclaringInterface().getName() +
						                   "." + existingMethod.getName() + " and " +
						                   generatedMethod.getDeclaringInterface().getName() +
						                   "." + generatedMethod.getName() + " are identical, ignoring the latter.");
				  }
			  } else {
				  pGeneratedMethods.put(generatedMethod, generatedMethod);
				  generatedMethod.setMethod(getInterfaceMethod(pJs, pDescription, method));
			  }
		  }
    }
  }

  /** <p>Creates a constructor with protected access and a single argument,
   * the backing object.</p>
   */
  protected JavaConstructor getConstructor(JavaSource pJs, InterfaceDescription[] pInterfaces) {
   	JavaConstructor jcon = pJs.newJavaConstructor(JavaSource.PROTECTED);
		jcon.addParam(Object.class, "o");
    jcon.addIf("o == null");
    jcon.addThrowNew(NullPointerException.class,
                     JavaSource.getQuoted("The supplied object must not be null."));
    jcon.addEndIf();
		for (int i = 0;  i < pInterfaces.length;  i++) {
			if (pInterfaces[i].isMandatory) {
        jcon.addIf("!(o instanceof ", pInterfaces[i].getInterface(), ")");
        jcon.addThrowNew(ClassCastException.class,
                      JavaSource.getQuoted("The supplied instance of "),
                      " + o.getClass().getName() + ",
                      JavaSource.getQuoted(" is not implementing "),
                      " + ", pInterfaces[i].getInterface(), ".class.getName()");
        jcon.addEndIf();
			}
		}
		jcon.addLine("backingObject = o;");
		return jcon;
	}

  /** <p>Creates the class.</p>
   */
  protected JavaSource getJavaSource(JavaSourceFactory pFactory, JavaQName pTargetName) {
     return pFactory.newJavaSource(pTargetName, JavaSource.PUBLIC);
  }

  /** <p>Generates the <code>backingObject</code> field.</p>
   */
  protected JavaField getBackingObjectField(JavaSource pJs, InterfaceDescription[] pInterfaces) {
     return pJs.newJavaField("backingObject", Object.class, JavaSource.PRIVATE);
  }

	/** <p>Generates a class implementing the given interfaces.</p>
	 * @param pFactory The ProxyGenerator will use this factory for creating
	 *    instances of JavaSource.
	 * @param pTargetName Name of the generated class
	 * @param pInterfaces The interfaces being implemented by the generated class.
	 */
	public JavaSource generate(JavaSourceFactory pFactory, JavaQName pTargetName,
								              InterfaceDescription[] pInterfaces) {
	   for (int i = 0;  i < pInterfaces.length;  i++) {
			if (pInterfaces[i] == null  ||  pInterfaces[i].interfaceClass == null) {
				throw new NullPointerException("The interfaces being implemented must be non-null");
			}
	   }

    JavaSource js = getJavaSource(pFactory, pTargetName);
    if (getExtendedClass() != null) {
      js.addExtends(getExtendedClass());
    }
    for (int i = 0;  i < pInterfaces.length;  i++) {
      js.addImplements(pInterfaces[i].getInterface());
    }

    getBackingObjectField(js, pInterfaces);
		getConstructor(js, pInterfaces);

		Map methods = new HashMap();
		for (int i = 0;  i < pInterfaces.length;  i++) {
			generateInterfaceMethods(js, methods, pInterfaces[i]);
		}

		return js;
	}
}
