/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.sqls;

import java.util.Iterator;
import java.util.Map;

import org.apache.ws.jaxme.sqls.impl.CombinedConstraintImpl;


/**
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 */
public interface CombinedConstraint extends Constraint {
   public interface Type {
      public final static Type AND = new CombinedConstraintImpl.TypeImpl("AND");
      public final static Type OR = new CombinedConstraintImpl.TypeImpl("OR");
   }

   /** <p>Returns the type, either of
    * {@link org.apache.ws.jaxme.sqls.CombinedConstraint.Type#AND} or
    * {@link org.apache.ws.jaxme.sqls.CombinedConstraint.Type#OR}.</p>
    */
   public CombinedConstraint.Type getType();

   /** <p>Creates an AndConstraint and inserts it at the current position.</p>
    */
   public CombinedConstraint createAndConstraint();

   /** <p>Creates an OrConstraint and inserts it at the current position.</p>
    */
   public CombinedConstraint createOrConstraint();

   /** <p>Creates a condition "equals" (=) and inserts it at the current position.</p>
    */
   public BooleanConstraint createEQ();

   /** <p>Creates a condition "doesn't equal" (!=) and inserts it at the current position.</p>
    */
   public BooleanConstraint createNE();

   /** <p>Creates a condition "lower than" (<) and inserts it at the current position.</p>
    */
   public BooleanConstraint createLT();

   /** <p>Creates a condition "greater than" (<) and inserts it at the current position.</p>
    */
   public BooleanConstraint createGT();

   /** <p>Creates a condition "lower or equals" (<=) and inserts it at the current position.</p>
    */
   public BooleanConstraint createLE();

   /** <p>Creates a condition "greater or equals" (>=) and inserts it at the current position.</p>
    */
   public BooleanConstraint createGE();

   /** <p>Creates a condition "LIKE" and inserts it at the current position.</p>
    */
   public BooleanConstraint createLIKE();

   /** <p>Creates a condition "IS NULL" and inserts it at the current position.</p>
    */
   public BooleanConstraint createISNULL();

   /** <p>Creates a condition "IN" and inserts it at the current position.</p>
    */
   public BooleanConstraint createIN();

   /** <p>Creates a condition "EXISTS" with the given select
    * statement and inserts it at the current position.</p>
    */
   public void createEXISTS(SelectStatement pStatement);

   /** <p>Creates a JOIN condition matching the given foreign key. In other
    * words, if the foreign key consists of the columns <code>A</code> and
    * <code>B</code> referencing the columns <code>X</code> and <code>Y</code>,
    * then the following will be added: <code>A=X AND B=Y</code>.</p>
    * @param pKey The foreign key being matched.
    * @param pReferencingTable A reference to the table returned by the
    *   foreign keys {@link ForeignKey#getTable()} method.
    * @param pReferencedTable A reference to the table returned by the
    *   foreign keys {@link ForeignKey#getReferencedTable()} method.
    */
   public void addJoin(ForeignKey pKey, TableReference pReferencingTable,
                        TableReference pReferencedTable);

   /** <p>Creates a JOIN condition matching the given column reference.
    * In other words, if the referencing {@link ColumnSet} contains the
    * columns A and B, and the referenced column set contains {@link ColumnSet},
    * X and Y, then the following will be added: <code>A=X AND B=Y</code>.</p>
    */
   public void addJoin(TableReference pReferencingTable,
                       ColumnSet pReferencingColumnSet,
                       TableReference pReferencedTable,
                       ColumnSet pReferencedColumnSet);

   /** <p>Clones the given CombinedConstraint, mapping the column references
    * from the given constraint to the values in the given map.</p>
    * @param pMap A Map with the constraints <code>pConstraint</code> table
    *   references as keys. The values are table references of the current
    *   constraints statement.
    * @param pConstraint The constraint being cloned.
    */
   public void addConstraint(Map pMap, Constraint pConstraint);

   /** <p>Adds a check for the columns of the given column set. For example,
    * if the column set consists of the columns <code>A</code> and <code>B</code>,
    * then the following will be added: <code>A=? AND B=?</code>.</p>
    */
   public void addColumnSetQuery(ColumnSet pSet, TableReference pTableReference);

   /** <p>Returns the number of parts, that have been added with the
    * various <code>createSomething()</code> methods.</p>
    */
   public int getNumParts();

   /** <p>Returns an Iterator to the parts, that have been added with the
    * various <code>createSomething()</code> methods.</p>
    */
   public Iterator getParts();

   /** <p>Returns whether the combined constraint is inverted by adding
    * a prepending <code>NOT</code>.</p>
    */
   public boolean isNOT();

   /** <p>Sets whether the combined constraint is inverted by adding
    * a prepending <code>NOT</code>.</p>
    */
   public void setNOT(boolean pNOT);
}
