/*
 * Copyright 2003, 2004  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.

 */
package org.apache.ws.jaxme.xs.xml;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/** <p>Implementation of the <code>xs:namespaceList</code> type,
 * specified like this:
 * <pre>
 *  &lt;xs:simpleType name="namespaceList"&gt;
 *    &lt;xs:annotation&gt;
 *      &lt;xs:documentation&gt;
 *        A utility type, not for public use
 *      &lt;/xs:documentation&gt;
 *    &lt;/xs:annotation&gt;
 *    &lt;xs:union&gt;
 *      &lt;xs:simpleType&gt;
 *        &lt;xs:restriction base="xs:token"&gt;
 *          &lt;xs:enumeration value="##any"/&gt;
 *          &lt;xs:enumeration value="##other"/&gt;
 *        &lt;/xs:restriction&gt;
 *      &lt;/xs:simpleType&gt;
 *      &lt;xs:simpleType&gt;
 *        &lt;xs:list&gt;
 *          &lt;xs:simpleType&gt;
 *            &lt;xs:union memberTypes="xs:anyURI"&gt;
 *              &lt;xs:simpleType&gt;
 *                &lt;xs:restriction base="xs:token"&gt;
 *                  &lt;xs:enumeration value="##targetNamespace"/&gt;
 *                   &lt;xs:enumeration value="##local"/&gt;
 *                &lt;/xs:restriction&gt;
 *              &lt;/xs:simpleType&gt;
 *            &lt;/xs:union&gt;
 *          &lt;/xs:simpleType&gt;
 *        &lt;/xs:list&gt;
 *      &lt;/xs:simpleType&gt;
 *    &lt;/xs:union&gt;
 *  &lt;/xs:simpleType&gt;
 * </pre></p>
 *
 * @author <a href="mailto:joe@ispsoft.de">Jochen Wiedmann</a>
 */
public class XsNamespaceList {
  public static final XsNamespaceList ANY = new XsNamespaceList(){
    public boolean isAny(){ return true; }
  };
  public static final XsNamespaceList OTHER = new XsNamespaceList(){
    public boolean isOther(){ return true; }
  };

  private boolean isTargetNamespaceSet, isLocalSet;
  private List uris;

  XsNamespaceList(){
  }

  public static XsNamespaceList valueOf(String pValue) {
    if ("##any".equals(pValue)) {
      return ANY;
    } else if ("##other".equals(pValue)) {
      return OTHER;
    } else {
      XsNamespaceList result = new XsNamespaceList();
      result.uris = new ArrayList();
      for (StringTokenizer st = new StringTokenizer(pValue);  st.hasMoreTokens();  ) {
        String s = st.nextToken();
        if ("##targetNamespace".equals(s)) {
          result.isTargetNamespaceSet = true;
        } else if ("##local".equals(s)) {
          result.isLocalSet = true;
        } else {
          result.uris.add(new XsAnyURI(s));
        }
      }
      return result;
    }
  }

  /** <p>Returns whether the namespace list matches <code>##any</code>.
   * This is the case if, and only if, {@link #isOther()} returns
   * false, and {@link #getUris()} returns null.</p>
   */
  public boolean isAny() { return false; }

  /** <p>Returns whether the namespace list matches <code>##other</code>.
   * This is the case if, and only if, {@link #isAny()} returns
   * false, and {@link #getUris()} returns null.</p>
   */
  public boolean isOther() { return false; }

  /** <p>Returns the array of URI's specified in the namespace list.
   * Returns non-null if, and only if, {@link #isAny()} and
   * {@link #isOther()} return false.</p>
   * <p>If this method returns non-null, you may also use the methods
   * {@link #isTargetNamespaceSet()} and {@link #isLocalSet()}.</p>
   */
  public XsAnyURI[] getUris() {
    if (uris == null) {
      return null;
    }
    return (XsAnyURI[]) uris.toArray(new XsAnyURI[uris.size()]);
  }

  /** <p>Returns whether the URI list did contain the value
   * <code>##targetNamespace</code>.</p>
   */
  public boolean isTargetNamespaceSet() {
    return isTargetNamespaceSet;
  }

  /** <p>Returns whether the URI list did contain the value
   * <code>##targetNamespace</code>.</p>
   */
  public boolean isLocalSet() {
    return isLocalSet;
  }
}
