/////////////////////////////////////////////////////////////////////////////
// Name:        sound.cpp
// Purpose:     Example of sound playing in wxWidgets
// Author:      Vaclav Slavik
// Modified by:
// Created:     2004/01/29
// RCS-ID:      $Id: sound.cpp 64656 2010-06-20 18:18:23Z VZ $
// Copyright:   (c) 2004 Vaclav Salvik
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// for all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWidgets headers)
#ifndef WX_PRECOMP
    #include "wx/app.h"
    #include "wx/frame.h"
    #include "wx/menu.h"
    #include "wx/msgdlg.h"
    #include "wx/icon.h"
    #include "wx/textctrl.h"
    #include "wx/filedlg.h"
#endif

#include "wx/sound.h"
#include "wx/numdlg.h"
#include "wx/textdlg.h"

// ----------------------------------------------------------------------------
// resources
// ----------------------------------------------------------------------------

// the application icon (under Windows and OS/2 it is in resources)
#if defined(__WXGTK__) || defined(__WXMOTIF__) || defined(__WXMAC__) || defined(__WXMGL__) || defined(__WXX11__)
    #include "../sample.xpm"
#endif

#define WAV_FILE wxT("doggrowl.wav")

// ----------------------------------------------------------------------------
// private classes
// ----------------------------------------------------------------------------

class MyApp : public wxApp
{
public:
    virtual bool OnInit();
};


class MyFrame : public wxFrame
{
public:
    // ctor(s)
    MyFrame(const wxString& title);
    ~MyFrame() { delete m_sound; }

    // event handlers (these functions should _not_ be virtual)
    void OnSelectFile(wxCommandEvent& event);
#ifdef __WXMSW__
    void OnSelectResource(wxCommandEvent& event);
#endif // __WXMSW__
    void OnSelectMemory(wxCommandEvent& event);
    void OnQuit(wxCommandEvent& event);

    void OnPlaySync(wxCommandEvent& event);
    void OnPlayAsync(wxCommandEvent& event);
    void OnPlayAsyncOnStack(wxCommandEvent& event);
    void OnPlayLoop(wxCommandEvent& event);
    void OnStop(wxCommandEvent& event);
    void OnPlayBell(wxCommandEvent& event);

    void OnAbout(wxCommandEvent& event);

    void NotifyUsingFile(const wxString& name);


private:
    bool CreateSound(wxSound& snd) const;

    wxSound*    m_sound;
    wxString    m_soundFile;
#ifdef __WXMSW__
    wxString    m_soundRes;
#endif // __WXMSW__
    bool        m_useMemory;

    wxTextCtrl* m_tc;

    // any class wishing to process wxWidgets events must use this macro
    DECLARE_EVENT_TABLE()
};

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

// IDs for the controls and the menu commands
enum
{
    // menu items
    Sound_SelectFile = wxID_HIGHEST + 1,
#ifdef __WXMSW__
    Sound_SelectResource,
#endif // __WXMSW__
    Sound_SelectMemory,

    Sound_PlaySync,
    Sound_PlayAsync,
    Sound_PlayAsyncOnStack,
    Sound_PlayLoop,
    Sound_Stop,
    Sound_PlayBell,

    Sound_Quit = wxID_EXIT,
    Sound_About = wxID_ABOUT
};

// ----------------------------------------------------------------------------
// event tables and other macros for wxWidgets
// ----------------------------------------------------------------------------

// the event tables connect the wxWidgets events with the functions (event
// handlers) which process them. It can be also done at run-time, but for the
// simple menu events like this the static method is much simpler.
BEGIN_EVENT_TABLE(MyFrame, wxFrame)
    EVT_MENU(Sound_SelectFile,       MyFrame::OnSelectFile)
#ifdef __WXMSW__
    EVT_MENU(Sound_SelectResource,   MyFrame::OnSelectResource)
#endif // __WXMSW__
    EVT_MENU(Sound_SelectMemory,     MyFrame::OnSelectMemory)
    EVT_MENU(Sound_Quit,             MyFrame::OnQuit)
    EVT_MENU(Sound_About,            MyFrame::OnAbout)
    EVT_MENU(Sound_PlaySync,         MyFrame::OnPlaySync)
    EVT_MENU(Sound_PlayAsync,        MyFrame::OnPlayAsync)
    EVT_MENU(Sound_PlayAsyncOnStack, MyFrame::OnPlayAsyncOnStack)
    EVT_MENU(Sound_PlayLoop,         MyFrame::OnPlayLoop)
    EVT_MENU(Sound_Stop,             MyFrame::OnStop)
    EVT_MENU(Sound_PlayBell,         MyFrame::OnPlayBell)
END_EVENT_TABLE()

// Create a new application object: this macro will allow wxWidgets to create
// the application object during program execution (it's better than using a
// static object for many reasons) and also implements the accessor function
// wxGetApp() which will return the reference of the right type (i.e. MyApp and
// not wxApp)
IMPLEMENT_APP(MyApp)

// ============================================================================
// implementation
// ============================================================================

// ----------------------------------------------------------------------------
// the application class
// ----------------------------------------------------------------------------

// 'Main program' equivalent: the program execution "starts" here
bool MyApp::OnInit()
{
    if ( !wxApp::OnInit() )
        return false;

    // create the main application window
    MyFrame *frame = new MyFrame(wxT("wxWidgets Sound Sample"));

    // and show it (the frames, unlike simple controls, are not shown when
    // created initially)
    frame->Show(true);

    // success: wxApp::OnRun() will be called which will enter the main message
    // loop and the application will run. If we returned false here, the
    // application would exit immediately.
    return true;
}

// ----------------------------------------------------------------------------
// main frame
// ----------------------------------------------------------------------------

// frame constructor
MyFrame::MyFrame(const wxString& title)
       : wxFrame(NULL, wxID_ANY, title)
{
    m_sound = NULL;
    m_soundFile = WAV_FILE;
    m_useMemory = false;

    // set the frame icon
    SetIcon(wxICON(sample));

    wxMenu *menuFile = new wxMenu;
    menuFile->Append(Sound_SelectFile, wxT("Select WAV &file...\tCtrl-O"), wxT("Select a new wav file to play"));
#ifdef __WXMSW__
    menuFile->Append(Sound_SelectResource, wxT("Select WAV &resource...\tCtrl-R"), wxT("Select a new resource to play"));
    menuFile->Append(Sound_SelectMemory, wxT("Select WAV &data\tCtrl-M"), wxT("Choose to play from memory buffer"));
#endif // __WXMSW__

    menuFile->Append(Sound_Quit, wxT("E&xit\tAlt-X"), wxT("Quit this program"));

    wxMenu *playMenu = new wxMenu;
    playMenu->Append(Sound_PlaySync, wxT("Play sound &synchronously\tCtrl-S"));
    playMenu->Append(Sound_PlayAsync, wxT("Play sound &asynchronously\tCtrl-A"));
    playMenu->Append(Sound_PlayAsyncOnStack, wxT("Play sound asynchronously (&object on stack)\tCtrl-K"));
    playMenu->Append(Sound_PlayLoop, wxT("&Loop sound\tCtrl-L"));
    playMenu->AppendSeparator();
    playMenu->Append(Sound_Stop, wxT("&Stop playing\tCtrl-T"));
    playMenu->AppendSeparator();
    playMenu->Append(Sound_PlayBell, wxT("Play system bell"));

    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(Sound_About, wxT("&About...\tF1"), wxT("Show about dialog"));

    // now append the freshly created menu to the menu bar...
    wxMenuBar *menuBar = new wxMenuBar();
    menuBar->Append(menuFile, wxT("&File"));
    menuBar->Append(playMenu, wxT("&Play"));
    menuBar->Append(helpMenu, wxT("&Help"));

    // ... and attach this menu bar to the frame
    SetMenuBar(menuBar);

    m_tc = new wxTextCtrl(this, wxID_ANY, wxEmptyString,
                          wxDefaultPosition, wxDefaultSize,
                          wxTE_MULTILINE|wxTE_READONLY);
    NotifyUsingFile(m_soundFile);
}


bool MyFrame::CreateSound(wxSound& snd) const
{
#ifdef __WXMSW__
    if ( m_useMemory )
    {
        // this is the dump of cuckoo.wav
        static const unsigned char data[] =
        {
            0x52, 0x49, 0x46, 0x46, 0x00, 0x1f, 0x00, 0x00, 0x57, 0x41, 0x56,
            0x45, 0x66, 0x6d, 0x74, 0x20, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00,
            0x01, 0x00, 0x11, 0x2b, 0x00, 0x00, 0x11, 0x2b, 0x00, 0x00, 0x01,
            0x00, 0x08, 0x00, 0x64, 0x61, 0x74, 0x61, 0xdc, 0x1e, 0x00, 0x00,
            0x88, 0x84, 0x86, 0x87, 0x87, 0x83, 0x82, 0x82, 0x82, 0x81, 0x82,
            0x83, 0x80, 0x83, 0x84, 0x85, 0x80, 0x84, 0x83, 0x87, 0x86, 0x89,
            0x84, 0x85, 0x85, 0x84, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
            0x81, 0x83, 0x87, 0x85, 0x84, 0x85, 0x87, 0x85, 0x84, 0x82, 0x81,
            0x81, 0x82, 0x83, 0x84, 0x82, 0x80, 0x84, 0x84, 0x83, 0x85, 0x87,
            0x84, 0x83, 0x82, 0x85, 0x84, 0x82, 0x83, 0x80, 0x7d, 0x7f, 0x81,
            0x82, 0x82, 0x83, 0x82, 0x85, 0x87, 0x88, 0x85, 0x86, 0x87, 0x87,
            0x84, 0x84, 0x7d, 0x7f, 0x7f, 0x83, 0x80, 0x80, 0x7f, 0x82, 0x80,
            0x81, 0x83, 0x85, 0x87, 0x84, 0x87, 0x85, 0x85, 0x83, 0x85, 0x82,
            0x80, 0x7e, 0x80, 0x7f, 0x7e, 0x7f, 0x81, 0x80, 0x81, 0x84, 0x85,
            0x85, 0x86, 0x86, 0x83, 0x84, 0x87, 0x84, 0x85, 0x82, 0x81, 0x7f,
            0x80, 0x82, 0x80, 0x7d, 0x7f, 0x7e, 0x82, 0x83, 0x85, 0x83, 0x86,
            0x85, 0x88, 0x84, 0x83, 0x82, 0x84, 0x83, 0x81, 0x7f, 0x7f, 0x7e,
            0x7e, 0x80, 0x84, 0x82, 0x82, 0x83, 0x84, 0x80, 0x84, 0x87, 0x86,
            0x86, 0x85, 0x82, 0x80, 0x80, 0x7e, 0x7e, 0x7f, 0x7f, 0x80, 0x7f,
            0x81, 0x84, 0x84, 0x83, 0x86, 0x82, 0x82, 0x83, 0x85, 0x81, 0x83,
            0x7e, 0x81, 0x80, 0x83, 0x81, 0x82, 0x7d, 0x82, 0x84, 0x84, 0x84,
            0x81, 0x84, 0x83, 0x82, 0x82, 0x84, 0x80, 0x80, 0x81, 0x80, 0x7e,
            0x80, 0x83, 0x84, 0x84, 0x87, 0x85, 0x85, 0x82, 0x85, 0x82, 0x84,
            0x86, 0x84, 0x7e, 0x81, 0x80, 0x7f, 0x82, 0x83, 0x84, 0x82, 0x83,
            0x83, 0x84, 0x83, 0x85, 0x89, 0x87, 0x85, 0x84, 0x83, 0x82, 0x83,
            0x81, 0x81, 0x7d, 0x7e, 0x83, 0x83, 0x7f, 0x80, 0x85, 0x84, 0x84,
            0x86, 0x87, 0x85, 0x86, 0x86, 0x86, 0x84, 0x84, 0x84, 0x81, 0x7f,
            0x7f, 0x7f, 0x7e, 0x81, 0x81, 0x80, 0x83, 0x85, 0x85, 0x86, 0x87,
            0x88, 0x87, 0x87, 0x84, 0x81, 0x80, 0x81, 0x81, 0x82, 0x80, 0x7f,
            0x80, 0x7f, 0x81, 0x82, 0x87, 0x84, 0x88, 0x89, 0x87, 0x84, 0x85,
            0x87, 0x88, 0x82, 0x7f, 0x82, 0x7e, 0x7f, 0x80, 0x80, 0x7f, 0x83,
            0x86, 0x84, 0x85, 0x85, 0x85, 0x85, 0x86, 0x85, 0x84, 0x84, 0x84,
            0x83, 0x81, 0x81, 0x83, 0x83, 0x82, 0x82, 0x82, 0x83, 0x84, 0x85,
            0x86, 0x85, 0x84, 0x86, 0x83, 0x82, 0x80, 0x83, 0x80, 0x84, 0x81,
            0x84, 0x7f, 0x83, 0x84, 0x88, 0x83, 0x87, 0x88, 0x86, 0x85, 0x86,
            0x87, 0x87, 0x8b, 0x8a, 0x88, 0x85, 0x87, 0x88, 0x87, 0x88, 0x89,
            0x88, 0x8a, 0x8b, 0x88, 0x89, 0x88, 0x8b, 0x88, 0x88, 0x85, 0x86,
            0x83, 0x83, 0x82, 0x84, 0x83, 0x84, 0x85, 0x85, 0x83, 0x85, 0x86,
            0x87, 0x87, 0x88, 0x87, 0x86, 0x85, 0x86, 0x85, 0x7f, 0x82, 0x83,
            0x80, 0x7e, 0x81, 0x80, 0x83, 0x85, 0x87, 0x87, 0x88, 0x87, 0x89,
            0x86, 0x86, 0x88, 0x84, 0x84, 0x83, 0x80, 0x7f, 0x81, 0x83, 0x82,
            0x81, 0x83, 0x84, 0x86, 0x86, 0x86, 0x88, 0x89, 0x8a, 0x8a, 0x86,
            0x84, 0x82, 0x82, 0x81, 0x7f, 0x81, 0x81, 0x7f, 0x80, 0x84, 0x81,
            0x84, 0x84, 0x89, 0x86, 0x84, 0x84, 0x86, 0x85, 0x86, 0x86, 0x83,
            0x82, 0x84, 0x81, 0x7f, 0x80, 0x82, 0x82, 0x84, 0x84, 0x84, 0x84,
            0x86, 0x85, 0x86, 0x84, 0x86, 0x85, 0x83, 0x80, 0x80, 0x84, 0x84,
            0x83, 0x81, 0x82, 0x81, 0x82, 0x85, 0x85, 0x82, 0x84, 0x86, 0x86,
            0x84, 0x84, 0x84, 0x82, 0x83, 0x86, 0x84, 0x80, 0x80, 0x83, 0x83,
            0x81, 0x84, 0x85, 0x85, 0x87, 0x84, 0x84, 0x83, 0x83, 0x84, 0x85,
            0x83, 0x84, 0x7f, 0x83, 0x7f, 0x84, 0x89, 0x6e, 0xa4, 0x81, 0x7d,
            0x7d, 0x7f, 0x87, 0x87, 0x7c, 0x82, 0x99, 0x7a, 0x88, 0x7b, 0x79,
            0x85, 0x87, 0x7c, 0x70, 0x88, 0x7c, 0x89, 0x8f, 0x7d, 0x91, 0x85,
            0x83, 0x7c, 0x89, 0x83, 0x7d, 0x86, 0x82, 0x8f, 0x7b, 0x84, 0x68,
            0xa3, 0x65, 0xa3, 0x73, 0x7e, 0xb5, 0x6f, 0x8d, 0x96, 0x84, 0x73,
            0x9d, 0x88, 0x77, 0x6f, 0x7a, 0x90, 0x7b, 0x75, 0x6e, 0x8a, 0x84,
            0x8d, 0x87, 0x78, 0x97, 0x93, 0x8b, 0x87, 0x89, 0x8d, 0x7d, 0x87,
            0x6e, 0x84, 0x7b, 0x8a, 0x6a, 0x8e, 0x65, 0x8a, 0x7a, 0x7d, 0x99,
            0x77, 0x8c, 0x9b, 0x80, 0x9f, 0x91, 0x70, 0x85, 0x8e, 0x6e, 0x73,
            0x85, 0x7d, 0x77, 0x90, 0x8c, 0x83, 0x77, 0x80, 0x8c, 0x7e, 0x7e,
            0x90, 0x8e, 0x96, 0x7e, 0x7f, 0x6c, 0x8f, 0x76, 0x77, 0x6b, 0x8c,
            0x80, 0x8c, 0x83, 0x7d, 0x91, 0x87, 0x84, 0x92, 0x73, 0x98, 0x87,
            0x7a, 0x8e, 0x7f, 0x7e, 0x8b, 0x7c, 0x7c, 0x7f, 0x7e, 0x72, 0x89,
            0x97, 0x7f, 0x7b, 0x89, 0x89, 0x84, 0x7a, 0x94, 0x77, 0x7d, 0x7c,
            0x91, 0x71, 0x7b, 0x81, 0x81, 0x87, 0x8a, 0x7f, 0x8a, 0x9c, 0x8c,
            0x7e, 0x8b, 0x77, 0x8f, 0x76, 0x64, 0x8c, 0x78, 0x81, 0x8a, 0x71,
            0x87, 0x8b, 0x7c, 0x85, 0x8d, 0x8f, 0x94, 0x99, 0x8e, 0x82, 0x94,
            0x7b, 0x7f, 0x6e, 0x7e, 0x6b, 0x6e, 0x7c, 0x8c, 0x7a, 0x72, 0x87,
            0x8c, 0x7e, 0x97, 0x93, 0x81, 0x89, 0x96, 0x84, 0x90, 0x82, 0x6e,
            0x8e, 0x6f, 0x78, 0x7c, 0x6a, 0x8b, 0x8e, 0x70, 0x8b, 0x8b, 0x8a,
            0x98, 0x7f, 0x84, 0x89, 0x93, 0x71, 0x8b, 0x7c, 0x7f, 0x7f, 0x78,
            0x75, 0x7c, 0x88, 0x86, 0x84, 0x87, 0x8d, 0x95, 0x81, 0x81, 0x95,
            0x7c, 0x7b, 0x9b, 0x93, 0x70, 0x95, 0x71, 0x77, 0x8a, 0x72, 0x80,
            0x7c, 0x6f, 0x95, 0x8e, 0x84, 0x96, 0x75, 0x87, 0x8b, 0x8b, 0x76,
            0x8f, 0x84, 0x85, 0x8e, 0x73, 0x7f, 0x8b, 0x83, 0x76, 0x7a, 0x7f,
            0x7d, 0x89, 0x85, 0x7e, 0x91, 0x87, 0x84, 0x82, 0x80, 0x81, 0x93,
            0x68, 0x81, 0x9b, 0x75, 0x86, 0x9b, 0x7a, 0x82, 0x82, 0x80, 0x98,
            0x78, 0x8f, 0x85, 0x7c, 0x7f, 0x88, 0x75, 0x6d, 0x82, 0x74, 0x8d,
            0x91, 0x78, 0x79, 0x8f, 0x85, 0x8d, 0x92, 0x7c, 0x8b, 0x90, 0x86,
            0x74, 0x7a, 0x7f, 0x6d, 0x84, 0x62, 0x79, 0x87, 0x7b, 0x98, 0x8b,
            0x79, 0x9c, 0x97, 0x74, 0xa2, 0x89, 0x83, 0x80, 0x7e, 0x88, 0x77,
            0x7c, 0x7e, 0x7e, 0x66, 0x80, 0x8b, 0x7e, 0x91, 0x8d, 0x86, 0x92,
            0x9f, 0x88, 0x98, 0x8b, 0x6e, 0x86, 0x89, 0x8c, 0x70, 0x87, 0x71,
            0x8c, 0x8f, 0x70, 0x84, 0x8a, 0x83, 0xa5, 0x92, 0x83, 0xab, 0x86,
            0x8c, 0x9e, 0x77, 0x88, 0x91, 0x83, 0x6b, 0x76, 0x87, 0x83, 0x7c,
            0x7f, 0x8b, 0x7d, 0x79, 0x98, 0x8c, 0x84, 0x92, 0x95, 0x7b, 0x91,
            0x7d, 0x81, 0x85, 0x86, 0x72, 0x78, 0x85, 0x6d, 0x84, 0x8a, 0x79,
            0x80, 0x84, 0x88, 0x9b, 0x87, 0x7e, 0x89, 0x7e, 0x9e, 0x7f, 0x72,
            0x7b, 0x83, 0x7a, 0x7b, 0x81, 0x82, 0x88, 0x96, 0x87, 0x82, 0x71,
            0x92, 0x88, 0x78, 0x7c, 0x98, 0x7b, 0x7f, 0x86, 0x7a, 0x74, 0x81,
            0x8a, 0x85, 0x96, 0x86, 0x7c, 0x92, 0x86, 0x7f, 0x8a, 0x85, 0x76,
            0x7f, 0x6b, 0x86, 0x79, 0x7a, 0x8d, 0x7e, 0x80, 0x8c, 0x7f, 0x95,
            0x93, 0x89, 0x75, 0x99, 0x8c, 0x8b, 0x74, 0x77, 0x7c, 0x80, 0x76,
            0x82, 0x74, 0x74, 0x91, 0x86, 0x75, 0x81, 0x99, 0x88, 0x83, 0x98,
            0x87, 0x96, 0x90, 0x89, 0x80, 0x85, 0x67, 0x7b, 0x71, 0x6e, 0x77,
            0x74, 0x84, 0x95, 0x76, 0x8e, 0x86, 0x86, 0x89, 0x9d, 0x86, 0x97,
            0x86, 0x76, 0x77, 0x7f, 0x6e, 0x78, 0x7c, 0x7e, 0x7d, 0x6c, 0x82,
            0x8e, 0x86, 0x85, 0x88, 0x90, 0x80, 0x9d, 0x8a, 0x78, 0x89, 0x97,
            0x79, 0x84, 0x75, 0x77, 0x81, 0x6e, 0x6e, 0x89, 0x73, 0x98, 0x85,
            0x87, 0x8e, 0x8e, 0x81, 0x9e, 0x8e, 0x7b, 0x82, 0x81, 0x77, 0x85,
            0x83, 0x78, 0x89, 0x7e, 0x7a, 0x86, 0x81, 0x8b, 0x7c, 0x8e, 0x8a,
            0x8c, 0x7a, 0x80, 0x8c, 0x7c, 0x76, 0x96, 0x84, 0x81, 0x8e, 0x75,
            0x71, 0x95, 0x7c, 0x94, 0x7e, 0x7d, 0x93, 0x86, 0x75, 0x85, 0x82,
            0x80, 0x8d, 0x79, 0x77, 0x85, 0x83, 0x78, 0x92, 0x83, 0x7c, 0x92,
            0x80, 0x87, 0x7d, 0x83, 0x7b, 0x8d, 0x8a, 0x83, 0x7f, 0x82, 0x7b,
            0x84, 0x77, 0x8a, 0x87, 0x71, 0x83, 0x9a, 0x72, 0x8d, 0x8f, 0x7a,
            0x91, 0x82, 0x7a, 0x88, 0x83, 0x85, 0x8d, 0x74, 0x7f, 0x87, 0x78,
            0x6f, 0x81, 0x87, 0x85, 0x98, 0x7b, 0x89, 0x92, 0x8d, 0x81, 0x8f,
            0x8a, 0x87, 0x89, 0x81, 0x78, 0x76, 0x76, 0x78, 0x87, 0x83, 0x80,
            0x84, 0x7f, 0x95, 0x8a, 0x7e, 0xad, 0x8d, 0x82, 0x99, 0x85, 0x71,
            0x84, 0x7c, 0x84, 0x7e, 0x76, 0x7b, 0x76, 0x77, 0x79, 0x8b, 0x7c,
            0x94, 0x9f, 0x87, 0x86, 0x8b, 0x90, 0x8e, 0x83, 0x77, 0x7b, 0x86,
            0x76, 0x73, 0x85, 0x7f, 0x83, 0x87, 0x78, 0x85, 0x84, 0x83, 0x99,
            0x7f, 0x8b, 0x97, 0x81, 0x86, 0x90, 0x69, 0x75, 0x8e, 0x79, 0x81,
            0x81, 0x83, 0x76, 0x85, 0x88, 0x8a, 0x84, 0x7a, 0x8e, 0x86, 0x84,
            0x88, 0x84, 0x83, 0x88, 0x8c, 0x7e, 0x77, 0x80, 0x80, 0x85, 0x82,
            0x7e, 0x90, 0x90, 0x7a, 0x82, 0x78, 0x8c, 0x8a, 0x7e, 0x87, 0x88,
            0x74, 0x84, 0x7f, 0x6c, 0x90, 0x89, 0x80, 0x8b, 0x8b, 0x80, 0x85,
            0x97, 0x7e, 0x78, 0x79, 0x83, 0x82, 0x7c, 0x84, 0x82, 0x73, 0x7f,
            0x85, 0x87, 0x79, 0x83, 0x89, 0x90, 0x94, 0x95, 0x84, 0x8f, 0x7c,
            0x89, 0x7f, 0x6b, 0x78, 0x7d, 0x78, 0x7b, 0x6f, 0x73, 0x96, 0x7f,
            0x7b, 0x91, 0x85, 0x8b, 0x90, 0x8a, 0x7e, 0x95, 0x8b, 0x7f, 0x7f,
            0x7b, 0x6e, 0x75, 0x77, 0x81, 0x72, 0x86, 0x92, 0x87, 0x78, 0x88,
            0x8d, 0x92, 0x8d, 0x96, 0x82, 0x81, 0x84, 0x7d, 0x8c, 0x77, 0x6f,
            0x7d, 0x79, 0x80, 0x90, 0x7d, 0x7f, 0x91, 0x7f, 0x96, 0x89, 0x7f,
            0x8e, 0x9e, 0x80, 0x81, 0x86, 0x84, 0x7e, 0x7f, 0x69, 0x7a, 0x7e,
            0x86, 0x7c, 0x82, 0x85, 0x8d, 0x7e, 0x92, 0x8d, 0x85, 0x80, 0x92,
            0x8b, 0x72, 0x8e, 0x88, 0x7e, 0x84, 0x7a, 0x7f, 0x88, 0x78, 0x7c,
            0x8b, 0x75, 0x95, 0x93, 0x88, 0x85, 0x7d, 0x74, 0x8d, 0x89, 0x88,
            0x8b, 0x87, 0x79, 0x92, 0x90, 0x7c, 0x81, 0x8d, 0x89, 0x96, 0x85,
            0x85, 0x8d, 0x90, 0x8d, 0x76, 0x7e, 0x81, 0x90, 0x7f, 0x7f, 0x99,
            0x7d, 0x7a, 0x98, 0x88, 0x81, 0x95, 0x80, 0x91, 0x87, 0x80, 0x8f,
            0x84, 0x71, 0x7d, 0x7a, 0x7d, 0x82, 0x7d, 0x79, 0x87, 0x87, 0x8a,
            0x96, 0x88, 0x81, 0x90, 0x89, 0x87, 0x90, 0x81, 0x7e, 0x7e, 0x88,
            0x7f, 0x81, 0x70, 0x6e, 0x87, 0x7e, 0x8a, 0x89, 0x84, 0x9b, 0x9d,
            0x73, 0x90, 0x95, 0x82, 0x89, 0x86, 0x78, 0x82, 0x7b, 0x84, 0x81,
            0x73, 0x73, 0x7f, 0x79, 0x84, 0x8a, 0x88, 0x8d, 0x90, 0x86, 0x8c,
            0x92, 0x7f, 0x8b, 0x80, 0x81, 0x7e, 0x84, 0x7a, 0x70, 0x83, 0x77,
            0x82, 0x89, 0x82, 0x86, 0x87, 0x7b, 0x96, 0x8d, 0x85, 0x8d, 0x7d,
            0x7a, 0x94, 0x7c, 0x71, 0x7e, 0x87, 0x80, 0x79, 0x7a, 0x83, 0x82,
            0x7e, 0x8a, 0x86, 0x87, 0x91, 0x95, 0x7e, 0x86, 0x7a, 0x7a, 0x86,
            0x82, 0x7f, 0x7d, 0x8a, 0x75, 0x87, 0x8c, 0x7a, 0x7e, 0x8a, 0x74,
            0x8c, 0x87, 0x89, 0x8b, 0x7e, 0x7f, 0x87, 0x72, 0x7c, 0x85, 0x7c,
            0x86, 0x88, 0x84, 0x82, 0x8b, 0x7d, 0x7f, 0x81, 0x95, 0x86, 0x7b,
            0x79, 0x88, 0x7c, 0x79, 0x85, 0x79, 0x83, 0x89, 0x86, 0x77, 0x82,
            0x88, 0x89, 0x94, 0x8c, 0x7a, 0x8d, 0x84, 0x77, 0x87, 0x78, 0x78,
            0x82, 0x7d, 0x81, 0x7e, 0x75, 0x87, 0x81, 0x86, 0x94, 0x92, 0x84,
            0x90, 0x83, 0x85, 0x80, 0x85, 0x7f, 0x7d, 0x7c, 0x79, 0x7a, 0x75,
            0x8d, 0x7f, 0x71, 0x80, 0x91, 0x84, 0x83, 0x93, 0x8e, 0x90, 0x93,
            0x84, 0x79, 0x8a, 0x72, 0x83, 0x7c, 0x76, 0x79, 0x8a, 0x78, 0x81,
            0x75, 0x89, 0x8b, 0x90, 0x84, 0x92, 0x7d, 0x89, 0x90, 0x84, 0x86,
            0x86, 0x81, 0x79, 0x7d, 0x72, 0x79, 0x79, 0x7e, 0x88, 0x82, 0x85,
            0x8b, 0x84, 0x81, 0x92, 0x8b, 0x7d, 0x89, 0x9a, 0x77, 0x7a, 0x82,
            0x7e, 0x85, 0x81, 0x73, 0x85, 0x7a, 0x81, 0x83, 0x81, 0x81, 0x8c,
            0x85, 0x8d, 0x80, 0x7b, 0x93, 0x8c, 0x80, 0x80, 0x82, 0x78, 0x82,
            0x85, 0x7a, 0x83, 0x87, 0x8b, 0x8b, 0x86, 0x7f, 0x83, 0x7e, 0x88,
            0x8b, 0x87, 0x78, 0x89, 0x7d, 0x73, 0x8f, 0x84, 0x72, 0x90, 0x89,
            0x7d, 0x7e, 0x82, 0x8f, 0x94, 0x7e, 0x86, 0x8a, 0x7c, 0x80, 0x74,
            0x79, 0x83, 0x82, 0x83, 0x8a, 0x80, 0x7c, 0x94, 0x8b, 0x82, 0x89,
            0x8f, 0x87, 0x8b, 0x7d, 0x7b, 0x7e, 0x87, 0x7d, 0x7d, 0x73, 0x76,
            0x86, 0x81, 0x7e, 0x93, 0x7a, 0x89, 0x9b, 0x84, 0x86, 0x93, 0x84,
            0x81, 0x87, 0x80, 0x80, 0x70, 0x74, 0x88, 0x7b, 0x77, 0x81, 0x8a,
            0x87, 0x87, 0x7d, 0x89, 0x95, 0x8f, 0x87, 0x92, 0x80, 0x7f, 0x85,
            0x81, 0x77, 0x81, 0x81, 0x79, 0x77, 0x74, 0x72, 0x8b, 0x84, 0x8a,
            0x8d, 0x80, 0x96, 0x93, 0x7b, 0x8b, 0x8d, 0x7d, 0x92, 0x89, 0x77,
            0x79, 0x76, 0x7d, 0x81, 0x78, 0x82, 0x89, 0x77, 0x83, 0x95, 0x8a,
            0x86, 0x91, 0x8a, 0x93, 0x86, 0x78, 0x7f, 0x89, 0x76, 0x75, 0x7e,
            0x84, 0x7e, 0x85, 0x77, 0x7c, 0x8f, 0x87, 0x91, 0x88, 0x7e, 0x8c,
            0x86, 0x83, 0x8e, 0x7d, 0x7d, 0x90, 0x74, 0x75, 0x82, 0x89, 0x82,
            0x8c, 0x84, 0x82, 0x82, 0x88, 0x8d, 0x87, 0x7c, 0x88, 0x86, 0x7b,
            0x7a, 0x80, 0x7d, 0x7f, 0x85, 0x8e, 0x8d, 0x88, 0x7b, 0x86, 0x93,
            0x7f, 0x85, 0x89, 0x84, 0x84, 0x7a, 0x7b, 0x85, 0x80, 0x84, 0x83,
            0x73, 0x8a, 0x8d, 0x81, 0x8b, 0x8e, 0x7f, 0x8c, 0x96, 0x82, 0x82,
            0x82, 0x7c, 0x7c, 0x7b, 0x7e, 0x88, 0x80, 0x7d, 0x82, 0x7e, 0x81,
            0x91, 0x8f, 0x83, 0x82, 0x8f, 0x8a, 0x93, 0x8b, 0x84, 0x8b, 0x7b,
            0x77, 0x82, 0x7c, 0x7b, 0x7f, 0x7d, 0x92, 0x84, 0x7e, 0x8f, 0x8f,
            0x8f, 0x9f, 0x91, 0x8b, 0x9d, 0x94, 0x7f, 0x88, 0x8a, 0x83, 0x85,
            0x85, 0x79, 0x75, 0x80, 0x86, 0x86, 0x81, 0x8e, 0x8e, 0x8e, 0x8d,
            0x96, 0x90, 0x88, 0x90, 0x84, 0x81, 0x87, 0x7f, 0x7f, 0x79, 0x6f,
            0x86, 0x87, 0x85, 0x8c, 0x80, 0x87, 0x8b, 0x82, 0x8c, 0x96, 0x82,
            0x85, 0x90, 0x85, 0x84, 0x8a, 0x7b, 0x7c, 0x8b, 0x85, 0x78, 0x85,
            0x8f, 0x91, 0x8d, 0x8a, 0x85, 0x84, 0x8a, 0x8b, 0x85, 0x7c, 0x8a,
            0x8e, 0x7f, 0x82, 0x85, 0x7b, 0x8a, 0x88, 0x84, 0x8e, 0x7d, 0x86,
            0x8e, 0x80, 0x87, 0x93, 0x84, 0x85, 0x82, 0x7f, 0x8c, 0x8c, 0x79,
            0x87, 0x8a, 0x80, 0x84, 0x92, 0x81, 0x82, 0x87, 0x83, 0x86, 0x8a,
            0x85, 0x78, 0x7d, 0x82, 0x83, 0x79, 0x7c, 0x86, 0x87, 0x81, 0x8c,
            0x7f, 0x7c, 0x98, 0x89, 0x86, 0x87, 0x85, 0x8f, 0x90, 0x7c, 0x83,
            0x79, 0x72, 0x7c, 0x7d, 0x71, 0x81, 0x83, 0x86, 0x89, 0x88, 0x81,
            0x90, 0x8a, 0x82, 0x93, 0x93, 0x82, 0x80, 0x80, 0x79, 0x78, 0x7e,
            0x7a, 0x7a, 0x76, 0x78, 0x7f, 0x85, 0x8b, 0x97, 0x7e, 0x8f, 0x97,
            0x86, 0x89, 0x8b, 0x76, 0x85, 0x83, 0x7b, 0x77, 0x73, 0x79, 0x89,
            0x7b, 0x77, 0x85, 0x8a, 0x89, 0x8e, 0x8b, 0x82, 0x8f, 0x88, 0x89,
            0x85, 0x79, 0x7a, 0x88, 0x80, 0x74, 0x81, 0x80, 0x74, 0x78, 0x82,
            0x83, 0x8f, 0x87, 0x8f, 0x8c, 0x80, 0x88, 0x84, 0x79, 0x90, 0x7f,
            0x7b, 0x87, 0x86, 0x78, 0x7b, 0x7a, 0x7f, 0x82, 0x81, 0x83, 0x80,
            0x81, 0x89, 0x8f, 0x80, 0x80, 0x87, 0x87, 0x83, 0x7c, 0x79, 0x80,
            0x8a, 0x80, 0x84, 0x82, 0x7c, 0x7f, 0x92, 0x7e, 0x79, 0x84, 0x83,
            0x88, 0x81, 0x7a, 0x88, 0x79, 0x7a, 0x85, 0x7d, 0x7e, 0x86, 0x7d,
            0x84, 0x8f, 0x86, 0x84, 0x8a, 0x8c, 0x7e, 0x7e, 0x82, 0x89, 0x7f,
            0x79, 0x7d, 0x77, 0x77, 0x7d, 0x82, 0x7b, 0x88, 0x8f, 0x8f, 0x89,
            0x87, 0x7f, 0x8e, 0x8d, 0x7b, 0x83, 0x84, 0x77, 0x83, 0x78, 0x70,
            0x75, 0x79, 0x87, 0x87, 0x79, 0x87, 0x90, 0x89, 0x93, 0x8f, 0x88,
            0x89, 0x89, 0x7c, 0x7d, 0x80, 0x79, 0x77, 0x7b, 0x83, 0x7c, 0x76,
            0x7e, 0x88, 0x80, 0x83, 0x8d, 0x8d, 0x8a, 0x8e, 0x87, 0x83, 0x8b,
            0x85, 0x7c, 0x84, 0x73, 0x75, 0x87, 0x78, 0x7c, 0x85, 0x84, 0x85,
            0x83, 0x81, 0x8e, 0x8c, 0x92, 0x95, 0x87, 0x7d, 0x89, 0x85, 0x81,
            0x7b, 0x7e, 0x7f, 0x84, 0x82, 0x7e, 0x75, 0x81, 0x86, 0x83, 0x86,
            0x95, 0x8c, 0x80, 0x8a, 0x8f, 0x7f, 0x81, 0x8a, 0x81, 0x7f, 0x86,
            0x71, 0x81, 0x89, 0x81, 0x86, 0x89, 0x7d, 0x89, 0x83, 0x83, 0x81,
            0x84, 0x8d, 0x8b, 0x7e, 0x80, 0x85, 0x7b, 0x7e, 0x86, 0x81, 0x83,
            0x89, 0x86, 0x7d, 0x85, 0x8a, 0x8f, 0x87, 0x80, 0x79, 0x7a, 0x7f,
            0x8b, 0x81, 0x77, 0x8b, 0x88, 0x74, 0x88, 0x8c, 0x7e, 0x8a, 0x8f,
            0x85, 0x8f, 0x82, 0x7c, 0x84, 0x7f, 0x7e, 0x7d, 0x7d, 0x7d, 0x7e,
            0x77, 0x7b, 0x7a, 0x82, 0x91, 0x8e, 0x84, 0x8b, 0x8e, 0x85, 0x89,
            0x88, 0x83, 0x81, 0x81, 0x7b, 0x79, 0x76, 0x7b, 0x7e, 0x7f, 0x7a,
            0x8c, 0x83, 0x81, 0x8d, 0x7f, 0x81, 0xa0, 0x90, 0x82, 0x87, 0x7d,
            0x81, 0x80, 0x7d, 0x7d, 0x79, 0x75, 0x7b, 0x81, 0x82, 0x8b, 0x87,
            0x8a, 0x8d, 0x89, 0x87, 0x8f, 0x88, 0x8d, 0x91, 0x82, 0x77, 0x83,
            0x80, 0x77, 0x74, 0x7a, 0x79, 0x85, 0x84, 0x81, 0x84, 0x80, 0x88,
            0x97, 0x85, 0x8d, 0x94, 0x7d, 0x83, 0x85, 0x77, 0x82, 0x87, 0x80,
            0x78, 0x7e, 0x81, 0x82, 0x7c, 0x7e, 0x85, 0x8a, 0x8a, 0x8a, 0x84,
            0x81, 0x82, 0x85, 0x89, 0x8b, 0x7d, 0x7c, 0x7e, 0x7c, 0x7e, 0x8c,
            0x84, 0x7e, 0x8e, 0x89, 0x83, 0x87, 0x88, 0x92, 0x91, 0x80, 0x89,
            0x89, 0x81, 0x94, 0x8b, 0x80, 0x8c, 0x8b, 0x87, 0x8a, 0x89, 0x82,
            0x89, 0x8e, 0x8d, 0x84, 0x86, 0x8a, 0x83, 0x7b, 0x7f, 0x88, 0x82,
            0x83, 0x7f, 0x7f, 0x85, 0x8c, 0x89, 0x8c, 0x8f, 0x82, 0x89, 0x8d,
            0x7d, 0x7b, 0x83, 0x7c, 0x83, 0x88, 0x83, 0x83, 0x76, 0x83, 0x8f,
            0x84, 0x84, 0x8e, 0x87, 0x8c, 0x90, 0x89, 0x89, 0x95, 0x8e, 0x83,
            0x78, 0x72, 0x71, 0x77, 0x7b, 0x7e, 0x7c, 0x80, 0x92, 0x8f, 0x88,
            0x8a, 0x90, 0x85, 0x86, 0x84, 0x87, 0x91, 0x8b, 0x82, 0x88, 0x85,
            0x74, 0x7d, 0x72, 0x6e, 0x73, 0x7e, 0x8d, 0x91, 0x8c, 0x98, 0x90,
            0x86, 0x8e, 0x89, 0x79, 0x7d, 0x83, 0x84, 0x87, 0x87, 0x7d, 0x7f,
            0x82, 0x7d, 0x74, 0x71, 0x78, 0x7f, 0x86, 0x8f, 0x97, 0x97, 0x94,
            0x96, 0x82, 0x70, 0x73, 0x7a, 0x7b, 0x87, 0x80, 0x8a, 0x92, 0x82,
            0x7f, 0x83, 0x7a, 0x78, 0x75, 0x7f, 0x8d, 0x8c, 0x94, 0x9d, 0x89,
            0x7f, 0x84, 0x76, 0x6f, 0x78, 0x7c, 0x7c, 0x8d, 0x95, 0x91, 0x8e,
            0x86, 0x80, 0x7d, 0x7d, 0x76, 0x79, 0x7e, 0x8f, 0x95, 0x8a, 0x86,
            0x8b, 0x7b, 0x79, 0x7d, 0x72, 0x7c, 0x89, 0x91, 0x9c, 0x8e, 0x86,
            0x8a, 0x7e, 0x72, 0x6c, 0x6c, 0x7c, 0x83, 0x80, 0x90, 0x96, 0x87,
            0x84, 0x8c, 0x7b, 0x7b, 0x85, 0x7e, 0x81, 0x8d, 0x8f, 0x89, 0x89,
            0x83, 0x76, 0x6e, 0x70, 0x72, 0x7a, 0x7e, 0x8b, 0x90, 0x91, 0x9d,
            0x93, 0x7f, 0x83, 0x7f, 0x79, 0x7d, 0x79, 0x85, 0x8b, 0x89, 0x88,
            0x84, 0x70, 0x75, 0x75, 0x77, 0x77, 0x7d, 0x89, 0x9d, 0x9d, 0x8b,
            0x92, 0x8f, 0x7e, 0x77, 0x7d, 0x71, 0x79, 0x84, 0x7f, 0x80, 0x89,
            0x85, 0x80, 0x7f, 0x7a, 0x7d, 0x79, 0x7f, 0x8d, 0x8d, 0x92, 0xa0,
            0x8d, 0x8b, 0x87, 0x73, 0x6c, 0x70, 0x6f, 0x7d, 0x7e, 0x85, 0x92,
            0x91, 0x85, 0x86, 0x82, 0x7a, 0x82, 0x7e, 0x81, 0x89, 0x8b, 0x90,
            0x9a, 0x85, 0x72, 0x70, 0x78, 0x73, 0x76, 0x7d, 0x7e, 0x91, 0x8e,
            0x95, 0x86, 0x87, 0x84, 0x79, 0x6d, 0x77, 0x7b, 0x74, 0x87, 0x90,
            0x8a, 0xa0, 0x8e, 0x86, 0x89, 0x77, 0x7c, 0x78, 0x88, 0x85, 0x98,
            0x81, 0x7b, 0x85, 0x79, 0x74, 0x74, 0x77, 0x7a, 0x81, 0x86, 0x8d,
            0x99, 0x97, 0x90, 0x8f, 0x7e, 0x76, 0x7e, 0x80, 0x74, 0x7a, 0x87,
            0x90, 0x91, 0x84, 0x80, 0x76, 0x75, 0x83, 0x79, 0x7a, 0x8b, 0x8f,
            0x92, 0x96, 0x95, 0x88, 0x83, 0x7b, 0x74, 0x6e, 0x73, 0x7a, 0x80,
            0x88, 0x89, 0x8f, 0x86, 0x83, 0x81, 0x7c, 0x78, 0x81, 0x89, 0x93,
            0x9b, 0x90, 0x8e, 0x8c, 0x77, 0x79, 0x7a, 0x70, 0x6e, 0x76, 0x79,
            0x81, 0x84, 0x90, 0x9a, 0x85, 0x83, 0x83, 0x7c, 0x84, 0x8b, 0x84,
            0x88, 0x97, 0x92, 0x86, 0x7c, 0x71, 0x6d, 0x71, 0x72, 0x7a, 0x80,
            0x8b, 0x91, 0x97, 0x93, 0x91, 0x8a, 0x7d, 0x7b, 0x7d, 0x75, 0x7c,
            0x82, 0x85, 0x8f, 0x83, 0x78, 0x7d, 0x7b, 0x74, 0x7c, 0x7d, 0x87,
            0x8d, 0x92, 0x9f, 0x91, 0x83, 0x8a, 0x84, 0x73, 0x78, 0x7a, 0x79,
            0x7b, 0x82, 0x86, 0x8d, 0x91, 0x89, 0x77, 0x71, 0x76, 0x7e, 0x83,
            0x89, 0x8d, 0x8b, 0x8e, 0x90, 0x80, 0x74, 0x7e, 0x76, 0x7d, 0x87,
            0x80, 0x81, 0x8b, 0x85, 0x8b, 0x8d, 0x7d, 0x7c, 0x7a, 0x7d, 0x77,
            0x78, 0x81, 0x8e, 0x84, 0x8b, 0x8a, 0x7d, 0x78, 0x83, 0x84, 0x7c,
            0x7f, 0x88, 0x8b, 0x8d, 0x8a, 0x87, 0x81, 0x80, 0x7a, 0x75, 0x75,
            0x7e, 0x7e, 0x78, 0x84, 0x8e, 0x8d, 0x92, 0x8e, 0x7a, 0x7e, 0x89,
            0x83, 0x8b, 0x87, 0x7e, 0x86, 0x80, 0x7c, 0x7f, 0x79, 0x7a, 0x7a,
            0x7e, 0x83, 0x81, 0x81, 0x93, 0x9f, 0x96, 0x93, 0x94, 0x88, 0x82,
            0x85, 0x81, 0x7d, 0x80, 0x8a, 0x89, 0x86, 0x7e, 0x83, 0x88, 0x8a,
            0x8f, 0x8a, 0x86, 0x99, 0x93, 0x88, 0x98, 0x8f, 0x85, 0x87, 0x7e,
            0x79, 0x80, 0x7b, 0x7a, 0x7d, 0x7e, 0x86, 0x8c, 0x89, 0x85, 0x85,
            0x88, 0x8e, 0x8c, 0x8c, 0x8c, 0x83, 0x7b, 0x89, 0x89, 0x7f, 0x7d,
            0x84, 0x7d, 0x7c, 0x7b, 0x7d, 0x84, 0x85, 0x89, 0x8c, 0x83, 0x7f,
            0x8e, 0x88, 0x89, 0x8f, 0x7d, 0x82, 0x8a, 0x7c, 0x74, 0x81, 0x84,
            0x8b, 0x88, 0x85, 0x83, 0x81, 0x80, 0x80, 0x82, 0x88, 0x87, 0x88,
            0x84, 0x85, 0x83, 0x89, 0x85, 0x7d, 0x7a, 0x85, 0x82, 0x7e, 0x8b,
            0x8b, 0x86, 0x8a, 0x8f, 0x8b, 0x84, 0x77, 0x7c, 0x7e, 0x77, 0x83,
            0x86, 0x7e, 0x88, 0x8c, 0x7e, 0x87, 0x87, 0x83, 0x88, 0x8c, 0x82,
            0x80, 0x87, 0x8e, 0x8a, 0x82, 0x7c, 0x7e, 0x7b, 0x78, 0x77, 0x7b,
            0x87, 0x8b, 0x8b, 0x8c, 0x8e, 0x86, 0x8c, 0x8b, 0x81, 0x80, 0x88,
            0x79, 0x77, 0x7d, 0x7d, 0x83, 0x83, 0x81, 0x87, 0x7f, 0x7b, 0x8a,
            0x87, 0x85, 0x8e, 0x8e, 0x8b, 0x91, 0x8a, 0x7f, 0x7c, 0x7e, 0x77,
            0x77, 0x7a, 0x7a, 0x7b, 0x81, 0x89, 0x84, 0x84, 0x86, 0x8f, 0x89,
            0x86, 0x7f, 0x82, 0x8b, 0x8c, 0x82, 0x85, 0x84, 0x78, 0x81, 0x7c,
            0x72, 0x7b, 0x80, 0x7b, 0x82, 0x82, 0x8d, 0x90, 0x8a, 0x8d, 0x91,
            0x82, 0x83, 0x83, 0x7f, 0x7c, 0x82, 0x80, 0x7f, 0x7e, 0x7b, 0x7c,
            0x7d, 0x83, 0x82, 0x82, 0x83, 0x87, 0x86, 0x87, 0x8b, 0x87, 0x86,
            0x89, 0x88, 0x7d, 0x7e, 0x75, 0x7a, 0x84, 0x7e, 0x79, 0x86, 0x85,
            0x88, 0x8a, 0x81, 0x7f, 0x83, 0x84, 0x85, 0x7c, 0x7b, 0x87, 0x83,
            0x7d, 0x85, 0x88, 0x80, 0x82, 0x84, 0x7f, 0x80, 0x83, 0x86, 0x86,
            0x88, 0x85, 0x84, 0x81, 0x82, 0x7d, 0x7b, 0x78, 0x7f, 0x7b, 0x7e,
            0x8a, 0x88, 0x8a, 0x92, 0x87, 0x87, 0x87, 0x7f, 0x82, 0x81, 0x76,
            0x7a, 0x82, 0x86, 0x84, 0x7b, 0x7f, 0x82, 0x83, 0x85, 0x86, 0x7b,
            0x80, 0x90, 0x8f, 0x85, 0x8c, 0x8d, 0x83, 0x81, 0x7e, 0x76, 0x73,
            0x75, 0x78, 0x7e, 0x7f, 0x83, 0x89, 0x87, 0x8a, 0x8a, 0x82, 0x89,
            0x8b, 0x82, 0x89, 0x88, 0x82, 0x85, 0x81, 0x7d, 0x7d, 0x77, 0x77,
            0x7a, 0x79, 0x7c, 0x7e, 0x84, 0x88, 0x89, 0x83, 0x92, 0x93, 0x87,
            0x83, 0x84, 0x77, 0x7f, 0x87, 0x7c, 0x73, 0x7f, 0x81, 0x81, 0x81,
            0x80, 0x81, 0x87, 0x85, 0x8b, 0x88, 0x83, 0x89, 0x85, 0x83, 0x89,
            0x82, 0x82, 0x87, 0x81, 0x7a, 0x7c, 0x79, 0x81, 0x82, 0x84, 0x85,
            0x8a, 0x86, 0x8d, 0x8c, 0x88, 0x85, 0x82, 0x80, 0x84, 0x7e, 0x76,
            0x84, 0x86, 0x7d, 0x81, 0x8b, 0x82, 0x82, 0x87, 0x7f, 0x80, 0x89,
            0x89, 0x86, 0x7c, 0x80, 0x8c, 0x8a, 0x87, 0x86, 0x7a, 0x7e, 0x82,
            0x7d, 0x83, 0x86, 0x84, 0x87, 0x8b, 0x86, 0x83, 0x7f, 0x81, 0x84,
            0x7f, 0x7b, 0x80, 0x81, 0x7e, 0x83, 0x81, 0x82, 0x88, 0x8a, 0x88,
            0x91, 0x85, 0x81, 0x8c, 0x90, 0x7f, 0x84, 0x81, 0x7a, 0x78, 0x78,
            0x7a, 0x7e, 0x81, 0x8a, 0x88, 0x81, 0x8c, 0x8a, 0x84, 0x82, 0x83,
            0x86, 0x8f, 0x8b, 0x86, 0x86, 0x86, 0x86, 0x81, 0x7a, 0x74, 0x75,
            0x72, 0x7d, 0x87, 0x89, 0x8a, 0x92, 0x8f, 0x8b, 0x89, 0x87, 0x85,
            0x8a, 0x7e, 0x7b, 0x83, 0x7a, 0x7a, 0x86, 0x84, 0x7d, 0x7f, 0x83,
            0x89, 0x81, 0x83, 0x89, 0x88, 0x8e, 0x94, 0x89, 0x82, 0x84, 0x7c,
            0x79, 0x7b, 0x7e, 0x82, 0x85, 0x7e, 0x7e, 0x82, 0x86, 0x86, 0x85,
            0x80, 0x8a, 0x8a, 0x85, 0x82, 0x86, 0x80, 0x83, 0x87, 0x80, 0x7e,
            0x7a, 0x74, 0x7c, 0x81, 0x7a, 0x83, 0x89, 0x88, 0x85, 0x88, 0x8e,
            0x8a, 0x86, 0x8c, 0x85, 0x76, 0x80, 0x85, 0x87, 0x8d, 0x8e, 0x83,
            0x86, 0x8a, 0x8a, 0x88, 0x88, 0x8a, 0x8f, 0x8f, 0x8a, 0x88, 0x89,
            0x86, 0x85, 0x82, 0x7f, 0x86, 0x84, 0x7a, 0x7f, 0x83, 0x88, 0x8f,
            0x8a, 0x88, 0x8c, 0x81, 0x81, 0x83, 0x81, 0x7d, 0x82, 0x86, 0x87,
            0x7d, 0x7d, 0x84, 0x82, 0x81, 0x8b, 0x8b, 0x84, 0x8a, 0x8c, 0x86,
            0x88, 0x8a, 0x86, 0x86, 0x83, 0x7b, 0x77, 0x77, 0x7f, 0x83, 0x80,
            0x7b, 0x7f, 0x82, 0x87, 0x8d, 0x88, 0x8a, 0x90, 0x84, 0x84, 0x83,
            0x81, 0x85, 0x85, 0x7e, 0x82, 0x82, 0x7f, 0x7b, 0x7b, 0x81, 0x83,
            0x85, 0x8d, 0x8e, 0x86, 0x89, 0x91, 0x8c, 0x85, 0x8a, 0x85, 0x7d,
            0x7c, 0x7c, 0x74, 0x78, 0x7f, 0x81, 0x85, 0x83, 0x83, 0x84, 0x84,
            0x8a, 0x8f, 0x88, 0x8f, 0x8e, 0x88, 0x87, 0x7d, 0x75, 0x80, 0x82,
            0x7b, 0x7c, 0x78, 0x7a, 0x81, 0x84, 0x83, 0x89, 0x8c, 0x8b, 0x88,
            0x87, 0x8a, 0x89, 0x82, 0x84, 0x82, 0x78, 0x79, 0x82, 0x7b, 0x7a,
            0x84, 0x85, 0x80, 0x82, 0x82, 0x82, 0x86, 0x85, 0x8c, 0x85, 0x83,
            0x89, 0x83, 0x7e, 0x84, 0x7c, 0x7d, 0x86, 0x83, 0x82, 0x84, 0x80,
            0x83, 0x8d, 0x8d, 0x86, 0x82, 0x83, 0x82, 0x83, 0x7d, 0x7e, 0x81,
            0x88, 0x88, 0x82, 0x7d, 0x83, 0x81, 0x7e, 0x87, 0x89, 0x82, 0x84,
            0x89, 0x81, 0x83, 0x85, 0x87, 0x84, 0x7d, 0x7e, 0x81, 0x79, 0x7f,
            0x86, 0x7f, 0x85, 0x8b, 0x85, 0x86, 0x8b, 0x85, 0x88, 0x89, 0x83,
            0x7c, 0x7f, 0x7d, 0x7f, 0x82, 0x80, 0x7c, 0x7d, 0x83, 0x83, 0x80,
            0x7d, 0x84, 0x8a, 0x90, 0x8e, 0x87, 0x82, 0x89, 0x82, 0x7b, 0x7e,
            0x7b, 0x78, 0x7a, 0x7b, 0x7d, 0x7d, 0x80, 0x8d, 0x8b, 0x84, 0x8b,
            0x8b, 0x83, 0x89, 0x8a, 0x87, 0x87, 0x86, 0x7d, 0x7b, 0x78, 0x75,
            0x74, 0x7a, 0x7e, 0x7e, 0x83, 0x88, 0x87, 0x89, 0x8a, 0x8b, 0x8a,
            0x8b, 0x84, 0x80, 0x7d, 0x84, 0x82, 0x80, 0x7d, 0x75, 0x7e, 0x87,
            0x7f, 0x7e, 0x81, 0x87, 0x8a, 0x87, 0x86, 0x8b, 0x81, 0x84, 0x8b,
            0x83, 0x7c, 0x83, 0x82, 0x7d, 0x7f, 0x80, 0x80, 0x85, 0x86, 0x82,
            0x83, 0x86, 0x84, 0x87, 0x89, 0x89, 0x83, 0x81, 0x80, 0x80, 0x7d,
            0x81, 0x81, 0x80, 0x84, 0x83, 0x7e, 0x87, 0x85, 0x81, 0x88, 0x85,
            0x83, 0x8c, 0x86, 0x81, 0x7c, 0x7f, 0x88, 0x84, 0x78, 0x7d, 0x81,
            0x7d, 0x81, 0x86, 0x83, 0x82, 0x8a, 0x8c, 0x87, 0x86, 0x85, 0x84,
            0x83, 0x82, 0x7d, 0x80, 0x81, 0x81, 0x7e, 0x7e, 0x83, 0x82, 0x82,
            0x89, 0x84, 0x86, 0x8b, 0x88, 0x8a, 0x8e, 0x81, 0x7d, 0x7e, 0x7a,
            0x78, 0x7c, 0x7c, 0x7c, 0x7e, 0x83, 0x86, 0x86, 0x8b, 0x92, 0x8a,
            0x81, 0x89, 0x8f, 0x88, 0x83, 0x84, 0x82, 0x81, 0x81, 0x7e, 0x76,
            0x77, 0x7b, 0x81, 0x82, 0x84, 0x89, 0x89, 0x8f, 0x91, 0x86, 0x87,
            0x8a, 0x81, 0x81, 0x82, 0x7c, 0x82, 0x83, 0x7e, 0x80, 0x80, 0x7d,
            0x80, 0x84, 0x86, 0x83, 0x84, 0x89, 0x8c, 0x8a, 0x8b, 0x89, 0x81,
            0x7e, 0x87, 0x80, 0x77, 0x7f, 0x82, 0x7d, 0x7f, 0x87, 0x8a, 0x8c,
            0x86, 0x82, 0x85, 0x87, 0x8d, 0x8c, 0x85, 0x87, 0x86, 0x80, 0x85,
            0x7e, 0x7b, 0x80, 0x81, 0x7e, 0x82, 0x7f, 0x84, 0x8a, 0x8b, 0x87,
            0x88, 0x89, 0x89, 0x84, 0x81, 0x81, 0x7d, 0x7a, 0x80, 0x7f, 0x77,
            0x7b, 0x82, 0x7f, 0x80, 0x8b, 0x83, 0x81, 0x8c, 0x8f, 0x8b, 0x87,
            0x82, 0x85, 0x82, 0x7a, 0x7f, 0x7d, 0x7e, 0x82, 0x7c, 0x80, 0x88,
            0x88, 0x84, 0x86, 0x82, 0x87, 0x8a, 0x88, 0x83, 0x7f, 0x7c, 0x7e,
            0x7f, 0x7e, 0x7d, 0x7d, 0x7f, 0x81, 0x7d, 0x7c, 0x85, 0x88, 0x88,
            0x8c, 0x8c, 0x81, 0x8a, 0x8a, 0x83, 0x81, 0x85, 0x7f, 0x81, 0x80,
            0x84, 0x81, 0x7f, 0x89, 0x91, 0x8d, 0x8c, 0x8d, 0x8a, 0x8d, 0x8e,
            0x8a, 0x89, 0x88, 0x85, 0x84, 0x83, 0x83, 0x84, 0x81, 0x7e, 0x7e,
            0x80, 0x83, 0x87, 0x85, 0x85, 0x8a, 0x8b, 0x8a, 0x8e, 0x83, 0x7e,
            0x85, 0x82, 0x7b, 0x81, 0x7f, 0x78, 0x7f, 0x7f, 0x7d, 0x7f, 0x87,
            0x8b, 0x88, 0x81, 0x86, 0x80, 0x81, 0x89, 0x8e, 0x84, 0x82, 0x7f,
            0x7e, 0x7d, 0x7a, 0x7a, 0x81, 0x83, 0x86, 0x86, 0x84, 0x89, 0x8c,
            0x8b, 0x8a, 0x8a, 0x85, 0x83, 0x84, 0x7c, 0x7f, 0x86, 0x81, 0x81,
            0x83, 0x7c, 0x80, 0x87, 0x83, 0x82, 0x86, 0x89, 0x88, 0x89, 0x8a,
            0x84, 0x81, 0x83, 0x86, 0x7d, 0x7c, 0x84, 0x81, 0x82, 0x89, 0x87,
            0x85, 0x89, 0x85, 0x89, 0x84, 0x7e, 0x81, 0x88, 0x88, 0x84, 0x80,
            0x7b, 0x7f, 0x80, 0x7e, 0x82, 0x86, 0x83, 0x89, 0x87, 0x81, 0x86,
            0x8a, 0x87, 0x89, 0x87, 0x7f, 0x81, 0x7e, 0x7d, 0x7b, 0x82, 0x87,
            0x82, 0x7f, 0x85, 0x82, 0x82, 0x8a, 0x8c, 0x86, 0x88, 0x8b, 0x87,
            0x84, 0x83, 0x82, 0x81, 0x82, 0x7f, 0x80, 0x7d, 0x7a, 0x7f, 0x86,
            0x83, 0x83, 0x84, 0x86, 0x8f, 0x8b, 0x84, 0x89, 0x88, 0x87, 0x8a,
            0x7f, 0x7b, 0x7e, 0x7f, 0x7e, 0x81, 0x81, 0x81, 0x80, 0x85, 0x84,
            0x83, 0x8a, 0x91, 0x88, 0x84, 0x88, 0x82, 0x7c, 0x81, 0x85, 0x80,
            0x82, 0x81, 0x7c, 0x79, 0x7d, 0x7f, 0x89, 0x86, 0x86, 0x8c, 0x8a,
            0x86, 0x89, 0x84, 0x83, 0x85, 0x7f, 0x83, 0x83, 0x7a, 0x7a, 0x81,
            0x84, 0x86, 0x82, 0x7f, 0x81, 0x85, 0x85, 0x84, 0x84, 0x88, 0x8a,
            0x86, 0x86, 0x82, 0x7b, 0x7d, 0x85, 0x7d, 0x7c, 0x85, 0x88, 0x81,
            0x86, 0x87, 0x84, 0x86, 0x84, 0x82, 0x7f, 0x7d, 0x83, 0x86, 0x82,
            0x86, 0x84, 0x7f, 0x7f, 0x7b, 0x7c, 0x87, 0x86, 0x85, 0x89, 0x87,
            0x84, 0x85, 0x87, 0x86, 0x82, 0x7f, 0x7e, 0x80, 0x80, 0x7d, 0x79,
            0x80, 0x87, 0x8a, 0x82, 0x85, 0x85, 0x7f, 0x84, 0x8d, 0x84, 0x83,
            0x8b, 0x85, 0x80, 0x7e, 0x7c, 0x80, 0x7d, 0x7d, 0x80, 0x7e, 0x7f,
            0x87, 0x84, 0x85, 0x89, 0x88, 0x89, 0x89, 0x81, 0x81, 0x84, 0x80,
            0x7e, 0x81, 0x81, 0x7d, 0x7e, 0x7e, 0x80, 0x83, 0x83, 0x84, 0x85,
            0x86, 0x86, 0x88, 0x8a, 0x91, 0x89, 0x7d, 0x7f, 0x83, 0x7a, 0x7d,
            0x7d, 0x7e, 0x7e, 0x86, 0x86, 0x84, 0x7d, 0x88, 0x89, 0x85, 0x89,
            0x8c, 0x88, 0x88, 0x88, 0x81, 0x7f, 0x81, 0x82, 0x81, 0x7e, 0x7a,
            0x7b, 0x83, 0x83, 0x83, 0x85, 0x88, 0x86, 0x86, 0x87, 0x87, 0x84,
            0x84, 0x86, 0x84, 0x84, 0x84, 0x7c, 0x7d, 0x83, 0x7f, 0x7f, 0x86,
            0x88, 0x84, 0x83, 0x84, 0x8b, 0x87, 0x84, 0x85, 0x7e, 0x7d, 0x87,
            0x86, 0x7f, 0x7f, 0x81, 0x7f, 0x81, 0x83, 0x81, 0x84, 0x88, 0x88,
            0x87, 0x87, 0x82, 0x85, 0x84, 0x81, 0x7d, 0x81, 0x82, 0x85, 0x7f,
            0x80, 0x83, 0x80, 0x87, 0x8a, 0x85, 0x88, 0x87, 0x82, 0x86, 0x8a,
            0x86, 0x83, 0x80, 0x85, 0x81, 0x7d, 0x7e, 0x85, 0x7c, 0x82, 0x85,
            0x87, 0x85, 0x8b, 0x86, 0x83, 0x86, 0x8b, 0x86, 0x84, 0x7f, 0x7e,
            0x82, 0x83, 0x7f, 0x7e, 0x7d, 0x7e, 0x82, 0x85, 0x85, 0x83, 0x84,
            0x8b, 0x86, 0x84, 0x8a, 0x89, 0x85, 0x87, 0x7e, 0x7b, 0x7f, 0x7d,
            0x7d, 0x7e, 0x7f, 0x7f, 0x82, 0x82, 0x8b, 0x87, 0x81, 0x85, 0x8b,
            0x86, 0x88, 0x89, 0x85, 0x80, 0x83, 0x7e, 0x7b, 0x7d, 0x7d, 0x7b,
            0x7b, 0x7d, 0x84, 0x87, 0x86, 0x86, 0x87, 0x8b, 0x8b, 0x83, 0x82,
            0x81, 0x7e, 0x83, 0x85, 0x7c, 0x7a, 0x7f, 0x7f, 0x80, 0x81, 0x7f,
            0x82, 0x85, 0x87, 0x86, 0x88, 0x87, 0x89, 0x85, 0x82, 0x84, 0x80,
            0x7d, 0x85, 0x8a, 0x82, 0x81, 0x86, 0x85, 0x89, 0x8d, 0x8c, 0x8a,
            0x8a, 0x8c, 0x8c, 0x86, 0x89, 0x8a, 0x86, 0x87, 0x84, 0x83, 0x88,
            0x85, 0x7f, 0x84, 0x87, 0x85, 0x8b, 0x88, 0x86, 0x86, 0x8a, 0x89,
            0x85, 0x83, 0x83, 0x85, 0x84, 0x84, 0x7e, 0x7f, 0x83, 0x88, 0x83,
            0x81, 0x86, 0x85, 0x7f, 0x88, 0x89, 0x82, 0x83, 0x87, 0x85, 0x83,
            0x7e, 0x82, 0x80, 0x81, 0x83, 0x7f, 0x7b, 0x81, 0x85, 0x87, 0x8c,
            0x86, 0x86, 0x88, 0x85, 0x81, 0x85, 0x84, 0x82, 0x7f, 0x7e, 0x7a,
            0x7d, 0x7f, 0x7f, 0x81, 0x80, 0x87, 0x88, 0x85, 0x86, 0x88, 0x86,
            0x8c, 0x8d, 0x81, 0x80, 0x86, 0x7f, 0x79, 0x7b, 0x7c, 0x7e, 0x7d,
            0x83, 0x87, 0x81, 0x81, 0x89, 0x87, 0x8b, 0x8a, 0x87, 0x86, 0x87,
            0x82, 0x81, 0x7f, 0x7f, 0x7f, 0x82, 0x7e, 0x7b, 0x7d, 0x82, 0x85,
            0x85, 0x86, 0x87, 0x86, 0x86, 0x88, 0x84, 0x84, 0x87, 0x83, 0x7b,
            0x80, 0x7e, 0x7c, 0x83, 0x85, 0x7f, 0x81, 0x86, 0x82, 0x7f, 0x80,
            0x8b, 0x8a, 0x83, 0x84, 0x84, 0x81, 0x81, 0x82, 0x81, 0x80, 0x7d,
            0x7e, 0x80, 0x81, 0x7d, 0x7f, 0x81, 0x86, 0x88, 0x88, 0x86, 0x82,
            0x83, 0x85, 0x85, 0x81, 0x83, 0x85, 0x80, 0x7e, 0x80, 0x7b, 0x7e,
            0x86, 0x82, 0x80, 0x87, 0x83, 0x83, 0x85, 0x86, 0x84, 0x83, 0x85,
            0x85, 0x7a, 0x7c, 0x84, 0x82, 0x7f, 0x81, 0x84, 0x84, 0x83, 0x84,
            0x86, 0x8a, 0x8a, 0x88, 0x85, 0x80, 0x7e, 0x80, 0x81, 0x81, 0x7e,
            0x7d, 0x7c, 0x81, 0x7f, 0x85, 0x84, 0x84, 0x8a, 0x8f, 0x85, 0x87,
            0x86, 0x82, 0x84, 0x87, 0x81, 0x7e, 0x80, 0x7f, 0x7b, 0x7a, 0x82,
            0x83, 0x85, 0x8b, 0x8a, 0x84, 0x84, 0x8b, 0x88, 0x87, 0x83, 0x89,
            0x83, 0x80, 0x7b, 0x80, 0x7e, 0x7f, 0x7e, 0x82, 0x83, 0x84, 0x85,
            0x84, 0x88, 0x88, 0x89, 0x8c, 0x87, 0x82, 0x84, 0x81, 0x7f, 0x87,
            0x83, 0x7e, 0x7e, 0x81, 0x82, 0x83, 0x81, 0x84, 0x80, 0x84, 0x86,
            0x82, 0x82, 0x8a, 0x89, 0x87, 0x89, 0x88, 0x80, 0x81, 0x80, 0x78,
            0x78, 0x7b, 0x7e, 0x82, 0x80, 0x7b, 0x85, 0x88, 0x8b, 0x8e, 0x8b,
            0x85, 0x83, 0x7b, 0x78, 0x7c, 0x7c, 0x7f, 0x83, 0x82, 0x84, 0x88,
            0x85, 0x81, 0x81, 0x7f, 0x7b, 0x7c, 0x7f, 0x7e, 0x7d, 0x84, 0x89,
            0x8d, 0x91, 0x8e, 0x7f, 0x79, 0x7b, 0x75, 0x71, 0x75, 0x7b, 0x80,
            0x8b, 0x91, 0x97, 0x9a, 0x97, 0x8f, 0x85, 0x71, 0x66, 0x67, 0x69,
            0x70, 0x7a, 0x83, 0x96, 0xa1, 0xa2, 0xa0, 0x98, 0x83, 0x73, 0x68,
            0x60, 0x62, 0x6a, 0x76, 0x82, 0x99, 0xab, 0xb3, 0xaa, 0x9b, 0x7c,
            0x61, 0x53, 0x53, 0x57, 0x69, 0x7d, 0x8d, 0xa7, 0xc2, 0xca, 0xbe,
            0xa8, 0x84, 0x5a, 0x3d, 0x33, 0x3b, 0x52, 0x70, 0x91, 0xb5, 0xcf,
            0xdf, 0xd9, 0xb8, 0x83, 0x50, 0x30, 0x29, 0x30, 0x45, 0x6a, 0x91,
            0xc0, 0xe2, 0xee, 0xdf, 0xb9, 0x80, 0x48, 0x21, 0x14, 0x1c, 0x3c,
            0x69, 0x9e, 0xd4, 0xf1, 0xfe, 0xf2, 0xbf, 0x86, 0x47, 0x17, 0x10,
            0x11, 0x26, 0x5b, 0x91, 0xcf, 0xf9, 0xff, 0xf8, 0xd8, 0x97, 0x57,
            0x24, 0x11, 0x10, 0x21, 0x55, 0x83, 0xb7, 0xe9, 0xff, 0xff, 0xdc,
            0x9d, 0x5c, 0x26, 0x10, 0x10, 0x16, 0x4a, 0x80, 0xb9, 0xef, 0xff,
            0xff, 0xeb, 0xaf, 0x66, 0x2b, 0x10, 0x0f, 0x11, 0x3a, 0x6f, 0xab,
            0xe7, 0xff, 0xff, 0xf7, 0xc3, 0x7c, 0x36, 0x13, 0x10, 0x10, 0x26,
            0x64, 0x97, 0xd1, 0xff, 0xff, 0xff, 0xd7, 0x8d, 0x4a, 0x16, 0x10,
            0x10, 0x20, 0x4e, 0x82, 0xbe, 0xf6, 0xff, 0xff, 0xe3, 0xa9, 0x5d,
            0x20, 0x10, 0x0f, 0x11, 0x46, 0x7d, 0xb8, 0xf0, 0xff, 0xff, 0xf0,
            0xb9, 0x6c, 0x2a, 0x10, 0x0f, 0x10, 0x2c, 0x6a, 0xa3, 0xdf, 0xff,
            0xff, 0xfe, 0xd5, 0x93, 0x4b, 0x18, 0x10, 0x10, 0x24, 0x5a, 0x90,
            0xcc, 0xfe, 0xff, 0xff, 0xdc, 0xa0, 0x58, 0x1f, 0x10, 0x10, 0x12,
            0x4f, 0x91, 0xc8, 0xfb, 0xff, 0xff, 0xf5, 0xbd, 0x71, 0x2b, 0x11,
            0x0f, 0x10, 0x34, 0x6b, 0xa9, 0xe6, 0xff, 0xff, 0xf8, 0xcc, 0x86,
            0x3a, 0x11, 0x0f, 0x0f, 0x10, 0x43, 0x80, 0xc0, 0xef, 0xff, 0xff,
            0xe5, 0x97, 0x53, 0x1c, 0x10, 0x10, 0x1b, 0x5a, 0x91, 0xc8, 0xff,
            0xff, 0xff, 0xe5, 0xae, 0x61, 0x1e, 0x0f, 0x0f, 0x10, 0x31, 0x6d,
            0xa7, 0xec, 0xff, 0xff, 0xf8, 0xb7, 0x74, 0x2b, 0x10, 0x0f, 0x0f,
            0x17, 0x59, 0x99, 0xdd, 0xff, 0xff, 0xff, 0xd5, 0x91, 0x43, 0x14,
            0x0f, 0x0f, 0x19, 0x57, 0x8c, 0xcd, 0xfe, 0xff, 0xff, 0xdb, 0x9c,
            0x4f, 0x13, 0x0f, 0x0f, 0x10, 0x43, 0x86, 0xc8, 0xff, 0xff, 0xff,
            0xf4, 0xae, 0x5a, 0x15, 0x0f, 0x0f, 0x0f, 0x26, 0x66, 0xae, 0xf3,
            0xff, 0xff, 0xff, 0xca, 0x7b, 0x32, 0x10, 0x0f, 0x0f, 0x1e, 0x5e,
            0x97, 0xe1, 0xff, 0xff, 0xff, 0xdd, 0x90, 0x46, 0x12, 0x0f, 0x0f,
            0x14, 0x49, 0x8a, 0xce, 0xff, 0xff, 0xff, 0xdf, 0x97, 0x4b, 0x11,
            0x0f, 0x0f, 0x10, 0x39, 0x7d, 0xc1, 0xff, 0xff, 0xff, 0xfd, 0xb5,
            0x60, 0x1f, 0x0f, 0x0f, 0x0f, 0x23, 0x63, 0xa4, 0xe4, 0xff, 0xff,
            0xff, 0xd3, 0x7f, 0x35, 0x10, 0x0f, 0x0f, 0x14, 0x53, 0x92, 0xd8,
            0xff, 0xff, 0xff, 0xda, 0x8f, 0x47, 0x11, 0x0f, 0x0f, 0x10, 0x3f,
            0x82, 0xc9, 0xff, 0xff, 0xff, 0xf7, 0xb4, 0x5f, 0x19, 0x0f, 0x0f,
            0x0f, 0x2c, 0x6b, 0xb2, 0xf9, 0xff, 0xff, 0xff, 0xd1, 0x7f, 0x2c,
            0x10, 0x0f, 0x0f, 0x16, 0x5a, 0x95, 0xdd, 0xff, 0xff, 0xff, 0xde,
            0x97, 0x4c, 0x11, 0x0f, 0x0f, 0x10, 0x40, 0x83, 0xcb, 0xff, 0xff,
            0xff, 0xed, 0xa9, 0x56, 0x12, 0x0f, 0x0f, 0x0f, 0x2f, 0x6a, 0xab,
            0xf3, 0xff, 0xff, 0xff, 0xcc, 0x7b, 0x32, 0x10, 0x0f, 0x0f, 0x16,
            0x5b, 0x9a, 0xda, 0xff, 0xff, 0xff, 0xe2, 0x97, 0x49, 0x12, 0x0f,
            0x0f, 0x10, 0x3b, 0x7e, 0xbf, 0xf8, 0xff, 0xff, 0xeb, 0xad, 0x5e,
            0x1c, 0x0f, 0x0f, 0x0f, 0x2a, 0x6a, 0xa7, 0xed, 0xff, 0xff, 0xff,
            0xce, 0x7b, 0x32, 0x10, 0x0f, 0x0f, 0x12, 0x4f, 0x8e, 0xce, 0xff,
            0xff, 0xff, 0xe8, 0xaa, 0x59, 0x15, 0x0f, 0x0f, 0x10, 0x34, 0x73,
            0xb4, 0xf2, 0xff, 0xff, 0xf0, 0xb7, 0x6f, 0x2c, 0x10, 0x0f, 0x0f,
            0x1f, 0x62, 0xa0, 0xe1, 0xff, 0xff, 0xff, 0xda, 0x95, 0x40, 0x11,
            0x0f, 0x0f, 0x10, 0x42, 0x80, 0xc2, 0xfa, 0xff, 0xff, 0xef, 0xb6,
            0x64, 0x1b, 0x0f, 0x0f, 0x0f, 0x29, 0x6b, 0xa9, 0xe2, 0xff, 0xff,
            0xfc, 0xc6, 0x7c, 0x37, 0x10, 0x0f, 0x0f, 0x12, 0x4f, 0x92, 0xd3,
            0xff, 0xff, 0xff, 0xe1, 0x9e, 0x4d, 0x12, 0x0f, 0x0f, 0x10, 0x34,
            0x76, 0xb6, 0xf6, 0xff, 0xff, 0xff, 0xcc, 0x7f, 0x36, 0x10, 0x0f,
            0x0f, 0x23, 0x67, 0xa8, 0xe7, 0xff, 0xff, 0xff, 0xe4, 0x9f, 0x51,
            0x15, 0x0f, 0x0f, 0x10, 0x39, 0x7b, 0xc4, 0xfa, 0xff, 0xff, 0xe6,
            0xad, 0x62, 0x19, 0x0f, 0x0f, 0x0f, 0x1b, 0x5d, 0x9d, 0xdf, 0xff,
            0xff, 0xf8, 0xc6, 0x81, 0x30, 0x0f, 0x0f, 0x0f, 0x10, 0x45, 0x87,
            0xc9, 0xff, 0xff, 0xff, 0xea, 0xb0, 0x5c, 0x16, 0x0f, 0x0f, 0x0f,
            0x21, 0x66, 0xab, 0xe9, 0xff, 0xff, 0xf9, 0xc0, 0x78, 0x35, 0x10,
            0x0f, 0x0f, 0x1a, 0x5b, 0x98, 0xd6, 0xff, 0xff, 0xff, 0xeb, 0xab,
            0x5b, 0x20, 0x10, 0x0f, 0x13, 0x50, 0x95, 0xd2, 0xff, 0xff, 0xff,
            0xff, 0xd2, 0x81, 0x35, 0x10, 0x0f, 0x0f, 0x1c, 0x5e, 0x97, 0xd4,
            0xff, 0xff, 0xff, 0xd3, 0x8c, 0x46, 0x11, 0x0f, 0x0f, 0x10, 0x3e,
            0x7f, 0xbb, 0xf5, 0xff, 0xff, 0xe6, 0xaf, 0x5d, 0x19, 0x0f, 0x0f,
            0x0f, 0x26, 0x66, 0xa8, 0xe2, 0xff, 0xff, 0xfb, 0xcc, 0x8e, 0x3c,
            0x10, 0x0f, 0x0f, 0x11, 0x4f, 0x8d, 0xcb, 0xff, 0xff, 0xff, 0xea,
            0xac, 0x63, 0x1f, 0x10, 0x0f, 0x10, 0x32, 0x75, 0xb4, 0xef, 0xff,
            0xff, 0xff, 0xd1, 0x88, 0x41, 0x12, 0x0f, 0x10, 0x21, 0x60, 0xa0,
            0xd8, 0xff, 0xff, 0xff, 0xe1, 0xa4, 0x5a, 0x1b, 0x0f, 0x0f, 0x11,
            0x40, 0x84, 0xbb, 0xf2, 0xff, 0xff, 0xf5, 0xc0, 0x74, 0x2d, 0x10,
            0x0f, 0x0f, 0x16, 0x5a, 0x9b, 0xd7, 0xff, 0xff, 0xff, 0xd3, 0x93,
            0x4a, 0x12, 0x0f, 0x0f, 0x10, 0x3c, 0x7b, 0xb7, 0xef, 0xff, 0xff,
            0xe4, 0xae, 0x64, 0x20, 0x0f, 0x0f, 0x0f, 0x1c, 0x62, 0xa2, 0xdc,
            0xff, 0xff, 0xfc, 0xd0, 0x8a, 0x3b, 0x10, 0x0f, 0x0f, 0x10, 0x3c,
            0x82, 0xbe, 0xf4, 0xff, 0xff, 0xe2, 0xb0, 0x64, 0x20, 0x0f, 0x0f,
            0x0f, 0x2a, 0x67, 0xa6, 0xdb, 0xff, 0xff, 0xf4, 0xc4, 0x7c, 0x37,
            0x10, 0x0f, 0x0f, 0x13, 0x50, 0x8d, 0xc9, 0xfb, 0xff, 0xff, 0xdd,
            0xa4, 0x59, 0x14, 0x0f, 0x0f, 0x0f, 0x2f, 0x73, 0xaf, 0xe7, 0xff,
            0xff, 0xf6, 0xc2, 0x7f, 0x3a, 0x10, 0x0f, 0x0f, 0x13, 0x52, 0x94,
            0xcb, 0xf9, 0xff, 0xff, 0xdb, 0xa4, 0x5e, 0x24, 0x10, 0x0f, 0x10,
            0x30, 0x6e, 0xb0, 0xe8, 0xff, 0xff, 0xf2, 0xc6, 0x87, 0x3f, 0x11,
            0x0f, 0x0f, 0x17, 0x57, 0x92, 0xca, 0xfb, 0xff, 0xff, 0xdf, 0xa7,
            0x61, 0x21, 0x0f, 0x0f, 0x0f, 0x2b, 0x6e, 0xaf, 0xe0, 0xff, 0xff,
            0xf5, 0xcb, 0x90, 0x47, 0x13, 0x0f, 0x0f, 0x11, 0x41, 0x83, 0xbd,
            0xec, 0xff, 0xff, 0xe2, 0xb0, 0x6c, 0x2f, 0x10, 0x0f, 0x10, 0x2a,
            0x5f, 0x9a, 0xd2, 0xff, 0xff, 0xf4, 0xcb, 0x8e, 0x4b, 0x15, 0x0f,
            0x0f, 0x10, 0x33, 0x75, 0xaf, 0xe2, 0xff, 0xff, 0xe9, 0xc4, 0x82,
            0x3c, 0x11, 0x0f, 0x0f, 0x12, 0x43, 0x7f, 0xb8, 0xed, 0xff, 0xfa,
            0xdf, 0xaf, 0x6a, 0x25, 0x0f, 0x0f, 0x0f, 0x1b, 0x5a, 0x95, 0xd4,
            0xf8, 0xff, 0xf0, 0xc5, 0x86, 0x48, 0x16, 0x0f, 0x0f, 0x10, 0x2f,
            0x6c, 0xa7, 0xd4, 0xed, 0xed, 0xd1, 0xa7, 0x6b, 0x3b, 0x1a, 0x12,
            0x15, 0x39, 0x5e, 0x8c, 0xb9, 0xda, 0xe1, 0xd6, 0xba, 0x94, 0x66,
            0x47, 0x2b, 0x23, 0x2f, 0x50, 0x6c, 0x90, 0xaa, 0xbc, 0xc2, 0xb5,
            0x98, 0x7e, 0x66, 0x50, 0x45, 0x48, 0x55, 0x67, 0x83, 0x9c, 0xab,
            0xae, 0xa6, 0x95, 0x7c, 0x60, 0x53, 0x4c, 0x4b, 0x57, 0x68, 0x76,
            0x90, 0xa2, 0xaa, 0xab, 0xa1, 0x92, 0x7e, 0x6c, 0x5f, 0x57, 0x57,
            0x62, 0x72, 0x83, 0x92, 0x9b, 0x9e, 0x9c, 0x94, 0x84, 0x74, 0x69,
            0x60, 0x5d, 0x61, 0x65, 0x75, 0x87, 0x8e, 0x92, 0x95, 0x8e, 0x86,
            0x7d, 0x74, 0x70, 0x6c, 0x70, 0x79, 0x7f, 0x84, 0x92, 0x95, 0x94,
            0x94, 0x89, 0x7f, 0x76, 0x6d, 0x68, 0x68, 0x70, 0x7c, 0x83, 0x8b,
            0x90, 0x93, 0x91, 0x91, 0x8c, 0x83, 0x7a, 0x76, 0x70, 0x74, 0x77,
            0x7c, 0x83, 0x8f, 0x8f, 0x91, 0x8c, 0x84, 0x82, 0x82, 0x7c, 0x79,
            0x79, 0x7f, 0x82, 0x82, 0x86, 0x8b, 0x87, 0x89, 0x86, 0x7e, 0x79,
            0x78, 0x76, 0x76, 0x79, 0x7d, 0x80, 0x83, 0x88, 0x88, 0x86, 0x86,
            0x83, 0x7d, 0x79, 0x78, 0x76, 0x75, 0x79, 0x7b, 0x7d, 0x83, 0x84,
            0x80, 0x7d, 0x7b, 0x78, 0x7b, 0x79, 0x79, 0x79, 0x78, 0x7a, 0x81,
            0x81, 0x83, 0x81, 0x80, 0x84, 0x7c, 0x76, 0x78, 0x7b, 0x7c, 0x82,
            0x83, 0x81, 0x81, 0x85, 0x82, 0x7f, 0x7d, 0x7e, 0x7c, 0x79, 0x7a,
            0x79, 0x7b, 0x80, 0x87, 0x88, 0x8a, 0x8c, 0x8b, 0x88, 0x86, 0x86,
            0x82, 0x85, 0x83, 0x88, 0x87, 0x8b, 0x8c, 0x8f, 0x8c, 0x8f, 0x8c,
            0x8c, 0x8c, 0x86, 0x88, 0x8a, 0x8a, 0x8f, 0x94, 0x8f, 0x8a, 0x8d,
            0x8e, 0x87, 0x87, 0x82, 0x81, 0x7f, 0x83, 0x82, 0x83, 0x87, 0x8e,
            0x90, 0x8e, 0x89, 0x85, 0x81, 0x80, 0x82, 0x7d, 0x7e, 0x7f, 0x7e,
            0x7f, 0x82, 0x85, 0x88, 0x89, 0x88, 0x86, 0x82, 0x82, 0x81, 0x82,
            0x82, 0x85, 0x85, 0x87, 0x8b, 0x86, 0x87, 0x85, 0x7f, 0x7b, 0x7f,
            0x77, 0x7b, 0x79, 0x7d, 0x7f, 0x7e, 0x83, 0x8b, 0x8a, 0x88, 0x87,
            0x7d, 0x82, 0x81, 0x7c, 0x77, 0x7a, 0x7e, 0x7f, 0x86, 0x85, 0x81,
            0x88, 0x89, 0x86, 0x81, 0x7d, 0x7d, 0x7f, 0x7e, 0x82, 0x82, 0x83,
            0x81, 0x86, 0x86, 0x86, 0x85, 0x83, 0x82, 0x7e, 0x78, 0x7b, 0x7f,
            0x7f, 0x83, 0x88, 0x87, 0x84, 0x87, 0x87, 0x82, 0x7e, 0x80, 0x7f,
            0x7d, 0x7d, 0x80, 0x80, 0x85, 0x87, 0x86, 0x87, 0x8c, 0x89, 0x85,
            0x80, 0x7b, 0x7b, 0x7c, 0x7f, 0x7e, 0x7f, 0x81, 0x87, 0x88, 0x87,
            0x85, 0x86, 0x86, 0x86, 0x82, 0x80, 0x81, 0x80, 0x83, 0x86, 0x86,
            0x85, 0x87, 0x86, 0x82, 0x81, 0x81, 0x80, 0x7b, 0x7d, 0x80, 0x7f,
            0x81, 0x88, 0x8c, 0x88, 0x86, 0x85, 0x82, 0x82, 0x80, 0x7d, 0x7d,
            0x7d, 0x7e, 0x7d, 0x80, 0x86, 0x89, 0x89, 0x8b, 0x87, 0x86, 0x88,
            0x85, 0x80, 0x7c, 0x7c, 0x7d, 0x82, 0x83, 0x7f, 0x81, 0x84, 0x83,
            0x82, 0x84, 0x80, 0x81, 0x7f, 0x82, 0x81, 0x80, 0x83, 0x86, 0x84,
            0x87, 0x88, 0x83, 0x80, 0x81, 0x7d, 0x7b, 0x7c, 0x7e, 0x7f, 0x80,
            0x82, 0x81, 0x85, 0x88, 0x85, 0x81, 0x80, 0x81, 0x82, 0x7f, 0x7d,
            0x7f, 0x7d, 0x7e, 0x81, 0x81, 0x82, 0x85, 0x84, 0x81, 0x7d, 0x7b,
            0x7d, 0x81, 0x7e, 0x7e, 0x7e, 0x81, 0x83, 0x83, 0x83, 0x81, 0x7c,
            0x81, 0x83, 0x7c, 0x7d, 0x80, 0x80, 0x80, 0x83, 0x83, 0x82, 0x82,
            0x83, 0x80, 0x7e, 0x7d, 0x7d, 0x7d, 0x7d, 0x81, 0x7c, 0x81, 0x85,
            0x86, 0x81, 0x85, 0x81, 0x84, 0x83, 0x80, 0x80, 0x7f, 0x81, 0x7d,
            0x80, 0x7d, 0x81, 0x80, 0x83, 0x82, 0x83, 0x7d, 0x84, 0x80, 0x7e,
            0x82, 0x83, 0x83, 0x87, 0x85, 0x7c, 0x7c, 0x82, 0x7e, 0x7e, 0x7d,
            0x80, 0x80, 0x82, 0x83, 0x84, 0x83, 0x83, 0x87, 0x83, 0x84, 0x84,
            0x80, 0x80, 0x80, 0x7f, 0x78, 0x7f, 0x7c, 0x82, 0x80, 0x7f, 0x80,
            0x85, 0x84, 0x84, 0x80, 0x7e, 0x81, 0x82, 0x81, 0x81, 0x7f, 0x7f,
            0x83, 0x82, 0x7f, 0x80, 0x81, 0x7b, 0x7c, 0x7a, 0x79, 0x7f, 0x83,
            0x83, 0x81, 0x84, 0x88, 0x88, 0x83, 0x82, 0x80, 0x7f, 0x80, 0x82,
            0x7d, 0x7d, 0x7d, 0x7c, 0x7e, 0x80, 0x7f, 0x7f, 0x81, 0x83, 0x83,
            0x83, 0x80, 0x84, 0x84, 0x84, 0x81, 0x82, 0x82, 0x81, 0x7f, 0x7f,
            0x7f, 0x7f, 0x83, 0x83, 0x83, 0x81, 0x87, 0x87, 0x86, 0x87, 0x87,
            0x86, 0x84, 0x83, 0x82, 0x7c, 0x7d, 0x81, 0x7f, 0x7d, 0x81, 0x82,
            0x82, 0x84, 0x83, 0x82, 0x85, 0x85, 0x81, 0x80, 0x7f, 0x7f, 0x82,
            0x82, 0x83, 0x80, 0x7f, 0x80, 0x84, 0x87, 0x87, 0x83, 0x84, 0x86,
            0x84, 0x81, 0x82, 0x83, 0x83, 0x85, 0x83, 0x7d, 0x7b, 0x7e, 0x7e,
            0x7e, 0x7e, 0x81, 0x83, 0x82, 0x84, 0x86, 0x83, 0x85, 0x88, 0x84,
            0x81, 0x81, 0x7f, 0x7d, 0x7d, 0x7d, 0x7e, 0x82, 0x82, 0x82, 0x82,
            0x84, 0x86, 0x87, 0x87, 0x88, 0x87, 0x85, 0x85, 0x83, 0x83, 0x82,
            0x81, 0x83, 0x85, 0x7f, 0x7a, 0x7e, 0x7f, 0x80, 0x82, 0x82, 0x80,
            0x83, 0x86, 0x88, 0x88, 0x87, 0x8a, 0x88, 0x81, 0x80, 0x80, 0x7b,
            0x7e, 0x81, 0x80, 0x7d, 0x81, 0x82, 0x83, 0x83, 0x88, 0x89, 0x8b,
            0x8b, 0x8a, 0x87, 0x89, 0x87, 0x86, 0x87, 0x88, 0x84, 0x85, 0x85,
            0x83, 0x85, 0x87, 0x86, 0x88, 0x8d, 0x8a, 0x87, 0x8b, 0x8b, 0x89,
            0x89, 0x88, 0x88, 0x87, 0x84, 0x82, 0x7f, 0x80, 0x86, 0x88, 0x85,
            0x84, 0x86, 0x85, 0x86, 0x89, 0x89, 0x84, 0x87, 0x85, 0x81, 0x82,
            0x85, 0x83, 0x85, 0x85, 0x82, 0x83, 0x84, 0x85, 0x84, 0x86, 0x85,
            0x86, 0x84, 0x83, 0x80, 0x83, 0x83, 0x84, 0x83, 0x85, 0x81, 0x81,
            0x82, 0x83, 0x83, 0x83, 0x86, 0x89, 0x86, 0x85, 0x87, 0x85, 0x85,
            0x88, 0x83, 0x81, 0x81, 0x84, 0x81, 0x80, 0x7e, 0x81, 0x83, 0x88,
            0x87, 0x85, 0x82, 0x84, 0x84, 0x87, 0x87, 0x86, 0x82, 0x83, 0x7f,
            0x7d, 0x7f, 0x81, 0x7f, 0x80, 0x7e, 0x7f, 0x81, 0x81, 0x84, 0x84,
            0x85, 0x88, 0x84, 0x86, 0x87, 0x84, 0x80, 0x83, 0x81, 0x7c, 0x7b,
            0x7b, 0x7a, 0x7d, 0x7f, 0x80, 0x81, 0x83, 0x85, 0x85, 0x84, 0x88,
            0x89, 0x87, 0x85, 0x84, 0x7d, 0x80, 0x81, 0x7e, 0x7c, 0x7d, 0x7d,
            0x82, 0x82, 0x80, 0x82, 0x81, 0x86, 0x89, 0x88, 0x87, 0x88, 0x84,
            0x84, 0x82, 0x82, 0x84, 0x85, 0x80, 0x7e, 0x7f, 0x7d, 0x7e, 0x82,
            0x84, 0x82, 0x83, 0x86, 0x85, 0x84, 0x84, 0x85, 0x82, 0x85, 0x83,
            0x81, 0x7f, 0x81, 0x82, 0x82, 0x82, 0x83, 0x81, 0x84, 0x82, 0x82,
            0x82, 0x83, 0x85, 0x84, 0x81, 0x81, 0x82, 0x83, 0x85, 0x84, 0x80,
            0x80, 0x81, 0x82, 0x81, 0x81, 0x7f, 0x83, 0x86, 0x81, 0x83, 0x84,
            0x81, 0x80, 0x81, 0x82, 0x81, 0x7f, 0x80, 0x80, 0x7b, 0x7e, 0x80,
            0x83, 0x85, 0x87, 0x83, 0x82, 0x84, 0x83, 0x82, 0x81, 0x80, 0x7f,
            0x7f, 0x7d, 0x7c, 0x7d, 0x80, 0x81, 0x82, 0x82, 0x84, 0x86, 0x86,
            0x85, 0x85, 0x84, 0x85, 0x86, 0x83, 0x80, 0x7f, 0x7f, 0x81, 0x80,
            0x7b, 0x7b, 0x7f, 0x81, 0x82, 0x82, 0x86, 0x85, 0x86, 0x89, 0x88,
            0x85, 0x86, 0x85, 0x82, 0x82, 0x7f, 0x7d, 0x7b, 0x7e, 0x7e, 0x7f,
            0x7f, 0x83, 0x88, 0x87, 0x85, 0x86, 0x86, 0x88, 0x88, 0x87, 0x84,
            0x83, 0x82, 0x7f, 0x81, 0x7f, 0x80, 0x82, 0x81, 0x7e, 0x81, 0x7f,
            0x81, 0x83, 0x82, 0x82, 0x84, 0x88, 0x87, 0x83, 0x82, 0x85, 0x82,
            0x81, 0x81, 0x7e, 0x7b, 0x7f, 0x81, 0x7f, 0x80, 0x80, 0x82, 0x83,
            0x86, 0x87, 0x85, 0x83, 0x84, 0x85, 0x81, 0x80, 0x80, 0x82, 0x82,
            0x81, 0x7d, 0x82, 0x82, 0x81, 0x85, 0x82, 0x82, 0x83, 0x87, 0x83,
            0x83, 0x80, 0x84, 0x83, 0x84, 0x83, 0x85, 0x83, 0x84, 0x83, 0x80,
            0x80, 0x87, 0x84, 0x83, 0x85, 0x85, 0x83, 0x85, 0x84, 0x81, 0x81,
            0x83, 0x80, 0x82, 0x7f, 0x7f, 0x81, 0x85, 0x82, 0x88, 0x83, 0x86,
            0x87, 0x88, 0x87, 0x85, 0x7f, 0x82, 0x80, 0x80, 0x7e, 0x7f, 0x7e,
            0x80, 0x81, 0x81, 0x82, 0x84, 0x87, 0x87, 0x85, 0x87, 0x87, 0x83,
            0x86, 0x84, 0x7f, 0x82, 0x83, 0x80, 0x7e, 0x7c, 0x7f, 0x7e, 0x7d,
            0x80, 0x81, 0x83, 0x86, 0x86, 0x85, 0x84, 0x85, 0x88, 0x86, 0x85,
            0x81, 0x80, 0x80, 0x7e, 0x7e, 0x7d, 0x7d, 0x7f, 0x82, 0x83, 0x82,
            0x82, 0x85, 0x88, 0x88, 0x86, 0x86, 0x83, 0x84, 0x86, 0x83, 0x81,
            0x83, 0x81, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x86, 0x84, 0x83, 0x85,
            0x88, 0x88, 0x86, 0x86, 0x88, 0x85, 0x84, 0x83, 0x7f, 0x7f, 0x81,
            0x7f, 0x7e, 0x83, 0x7f, 0x82, 0x82, 0x83, 0x82, 0x87, 0x84, 0x89,
            0x86, 0x83, 0x81, 0x80, 0x82, 0x83, 0x81, 0x82, 0x82, 0x82, 0x82,
            0x82, 0x7f, 0x81, 0x86, 0x85, 0x83, 0x82, 0x83, 0x7e, 0x82, 0x83,
            0x83, 0x7f, 0x81, 0x81, 0x82, 0x7e, 0x83, 0x85, 0x83, 0x82, 0x83,
            0x80, 0x83, 0x85, 0x83, 0x82, 0x82, 0x82, 0x82, 0x83, 0x82, 0x83,
            0x85, 0x86, 0x88, 0x89, 0x8a, 0x8a, 0x8c, 0x8d, 0x8d, 0x8c, 0x8d,
            0x8d, 0x8a, 0x87, 0x87, 0x84, 0x85, 0x88, 0x87, 0x83, 0x83, 0x85,
            0x87, 0x86, 0x88, 0x8a, 0x8a, 0x8c, 0x8c, 0x87, 0x83, 0x86, 0x85,
            0x86, 0x85, 0x82, 0x80, 0x80, 0x7f, 0x80, 0x81, 0x85, 0x88, 0x86,
            0x86, 0x86, 0x87, 0x8a, 0x89, 0x87, 0x86, 0x85, 0x82, 0x80, 0x83,
            0x7f, 0x7f, 0x81, 0x81, 0x80, 0x81, 0x84, 0x82, 0x87, 0x87, 0x89,
            0x84, 0x8a, 0x88, 0x88, 0x81, 0x83, 0x85, 0x81, 0x82, 0x81, 0x7d,
            0x7d, 0x81, 0x83, 0x82, 0x84, 0x83, 0x85, 0x87, 0x86, 0x83, 0x84,
            0x87, 0x88, 0x86, 0x83, 0x81, 0x81, 0x81, 0x81, 0x80, 0x83, 0x85,
            0x82, 0x80, 0x81, 0x81, 0x83, 0x88, 0x89, 0x86, 0x85, 0x83, 0x84,
            0x83, 0x84, 0x84, 0x82, 0x82, 0x85, 0x82, 0x81, 0x86, 0x86, 0x86,
            0x83, 0x86, 0x84, 0x85, 0x85, 0x84, 0x81, 0x82, 0x83, 0x85, 0x83,
            0x80, 0x81, 0x82, 0x86, 0x86, 0x86, 0x85, 0x87, 0x86, 0x86, 0x84,
            0x82, 0x83, 0x85, 0x80, 0x7f, 0x81, 0x7f, 0x7f, 0x82, 0x82, 0x83,
            0x83, 0x83, 0x84, 0x82, 0x84, 0x87, 0x88, 0x86, 0x87, 0x82, 0x7e,
            0x7e, 0x82, 0x7f, 0x81, 0x81, 0x80, 0x7f, 0x81, 0x82, 0x84, 0x87,
            0x88, 0x87, 0x85, 0x85, 0x83, 0x84, 0x83, 0x7f, 0x7e, 0x81, 0x7f,
            0x80, 0x7c, 0x7e, 0x7e, 0x82, 0x85, 0x83, 0x84, 0x85, 0x87, 0x87,
            0x86, 0x88, 0x87, 0x85, 0x87, 0x83, 0x80, 0x7e, 0x80, 0x7e, 0x7e,
            0x81, 0x7f, 0x80, 0x82, 0x85, 0x84, 0x85, 0x88, 0x8a, 0x86, 0x85,
            0x85, 0x82, 0x84, 0x85, 0x81, 0x81, 0x7e, 0x7c, 0x7d, 0x7f, 0x80,
            0x84, 0x83, 0x85, 0x82, 0x82, 0x84, 0x87, 0x88, 0x87, 0x85, 0x83,
            0x84, 0x84, 0x7f, 0x7e, 0x81, 0x7f, 0x82, 0x81, 0x82, 0x7f, 0x82,
            0x84, 0x84, 0x82, 0x85, 0x85, 0x82, 0x82, 0x80, 0x82, 0x7d, 0x82,
            0x82, 0x80, 0x80, 0x82, 0x81, 0x84, 0x84, 0x83, 0x84, 0x85, 0x83,
            0x82, 0x81, 0x7f, 0x80, 0x81, 0x82, 0x80, 0x80, 0x81, 0x82, 0x82,
            0x84, 0x84, 0x84, 0x85, 0x83, 0x7f, 0x82, 0x85, 0x85, 0x83, 0x81,
            0x7f, 0x80, 0x7f, 0x82, 0x82, 0x80, 0x84, 0x85, 0x84, 0x83, 0x87,
            0x85, 0x85, 0x88, 0x87, 0x85, 0x84, 0x82, 0x80, 0x80, 0x7f, 0x7f,
            0x80, 0x81, 0x80, 0x81, 0x82, 0x84, 0x85, 0x86, 0x87, 0x88, 0x87,
            0x87, 0x86, 0x84, 0x83, 0x82, 0x81, 0x82, 0x80, 0x7c, 0x7d, 0x80,
            0x7f, 0x7e, 0x81, 0x84, 0x85, 0x85, 0x87, 0x87, 0x87, 0x89, 0x89,
            0x83, 0x82, 0x82, 0x82, 0x7e, 0x80, 0x7c, 0x7c, 0x7d, 0x7f, 0x80,
            0x84, 0x81, 0x85, 0x89, 0x87, 0x87, 0x84, 0x84, 0x84, 0x84, 0x82,
            0x82, 0x80, 0x7f, 0x80, 0x80, 0x7e, 0x81, 0x82, 0x82, 0x86, 0x83,
            0x83, 0x84, 0x86, 0x84, 0x83, 0x83, 0x86, 0x83, 0x81, 0x80, 0x7e,
            0x7f, 0x81, 0x83, 0x7f, 0x82, 0x84, 0x84, 0x83, 0x84, 0x84, 0x85,
            0x84, 0x84, 0x82, 0x82, 0x82, 0x83, 0x82, 0x81, 0x81, 0x81, 0x82,
            0x84, 0x83, 0x82, 0x83, 0x82, 0x81, 0x82, 0x81, 0x80, 0x84, 0x82,
            0x81, 0x81, 0x81, 0x80, 0x81, 0x83, 0x83, 0x81, 0x84, 0x86, 0x84,
            0x82, 0x83, 0x85, 0x84, 0x87, 0x85, 0x81, 0x83, 0x83, 0x81, 0x80,
            0x7f, 0x80, 0x85, 0x81, 0x83, 0x85, 0x84, 0x87, 0x87, 0x87, 0x87,
            0x87, 0x86, 0x85, 0x81, 0x7e, 0x81, 0x7f, 0x7f, 0x7d, 0x80, 0x7d,
            0x81, 0x84, 0x85, 0x84, 0x83, 0x88, 0x86, 0x88, 0x88, 0x88, 0x84,
            0x83, 0x82, 0x80, 0x7b, 0x80, 0x7e, 0x7e, 0x80, 0x7f, 0x80, 0x83,
            0x85, 0x83, 0x86, 0x89, 0x88, 0x88, 0x86, 0x82, 0x83, 0x82, 0x83,
            0x80, 0x80, 0x81, 0x81, 0x80, 0x83, 0x85, 0x85, 0x88, 0x8a, 0x8b,
            0x8b, 0x8c, 0x8c, 0x8a, 0x88, 0x8a, 0x87, 0x86, 0x89, 0x86, 0x82,
            0x83, 0x84, 0x83, 0x87, 0x87, 0x85, 0x83, 0x87, 0x88, 0x89, 0x87,
            0x87, 0x85, 0x86, 0x85, 0x84, 0x83, 0x84, 0x84, 0x83, 0x7f, 0x7f,
            0x80, 0x82, 0x85, 0x84, 0x83, 0x85, 0x83, 0x84, 0x82, 0x84, 0x82,
            0x84, 0x85, 0x83, 0x81, 0x82, 0x81, 0x82, 0x82, 0x84, 0x83, 0x84,
            0x84, 0x85, 0x80, 0x85, 0x87, 0x84, 0x82, 0x84, 0x82, 0x81, 0x82,
            0x81, 0x7f, 0x7f, 0x83, 0x86, 0x86, 0x85, 0x83, 0x83, 0x85, 0x87,
            0x86, 0x83, 0x82, 0x81, 0x81, 0x7f, 0x7f, 0x80, 0x82, 0x81, 0x7f,
            0x80, 0x84, 0x86, 0x86, 0x89, 0x87, 0x86, 0x86, 0x87, 0x83, 0x84,
            0x84, 0x82, 0x80, 0x80, 0x7d, 0x7b, 0x7f, 0x81, 0x82, 0x83, 0x84,
            0x85, 0x86, 0x84, 0x87, 0x85, 0x86, 0x87, 0x84, 0x80, 0x7f, 0x7e,
            0x80, 0x81, 0x7f, 0x80, 0x7f, 0x83, 0x83, 0x85, 0x82, 0x85, 0x86,
            0x8b, 0x87, 0x87, 0x83, 0x84, 0x81, 0x80, 0x82, 0x81, 0x80, 0x80,
            0x7e, 0x7c, 0x7e, 0x81, 0x83, 0x83, 0x85, 0x83, 0x85, 0x84, 0x8a,
            0x86, 0x87, 0x86, 0x87, 0x7e, 0x82, 0x80, 0x81, 0x7f, 0x83, 0x81,
            0x80, 0x7f, 0x81, 0x84, 0x83, 0x85, 0x87, 0x86, 0x86, 0x82, 0x82,
            0x81, 0x83, 0x84, 0x84, 0x80, 0x7e, 0x7f, 0x83, 0x81, 0x80, 0x82,
            0x84, 0x82, 0x86, 0x83, 0x82, 0x82, 0x84, 0x82, 0x83, 0x81, 0x83,
            0x80, 0x7f, 0x80, 0x81, 0x81, 0x83, 0x87, 0x84, 0x82, 0x84, 0x81,
            0x83, 0x84, 0x82, 0x81, 0x81, 0x7f, 0x7e, 0x7f, 0x80, 0x81, 0x83,
            0x82, 0x83, 0x83, 0x84, 0x86, 0x87, 0x85, 0x86, 0x83, 0x83, 0x81,
            0x7f, 0x7e, 0x80, 0x81, 0x82, 0x80, 0x7e, 0x7e, 0x83, 0x85, 0x85,
            0x84, 0x86, 0x87, 0x84, 0x83, 0x85, 0x83, 0x80, 0x82, 0x83, 0x7e,
            0x7d, 0x7e, 0x7e, 0x7e, 0x7f, 0x82, 0x83, 0x86, 0x83, 0x86, 0x84,
            0x86, 0x86, 0x87, 0x85, 0x83, 0x7e, 0x7f, 0x7e, 0x7b, 0x7d, 0x7f,
            0x7f, 0x7f, 0x7f, 0x7e, 0x80, 0x83, 0x86, 0x87, 0x85, 0x86, 0x86,
            0x85, 0x83, 0x82, 0x80, 0x83, 0x80, 0x7c, 0x7a, 0x7b, 0x7c, 0x7e,
            0x81, 0x80, 0x80, 0x81, 0x84, 0x84, 0x84, 0x85, 0x86, 0x84, 0x83,
            0x82, 0x7f, 0x7e, 0x80, 0x7e, 0x7e, 0x7e, 0x7e, 0x7f, 0x81, 0x80,
            0x80, 0x82, 0x84, 0x85, 0x84, 0x83, 0x80, 0x80, 0x82, 0x82, 0x80,
            0x80, 0x82, 0x81, 0x7f, 0x7f, 0x81, 0x81, 0x84, 0x88, 0x81, 0x81,
            0x84, 0x82, 0x81, 0x82, 0x81, 0x83, 0x82, 0x80, 0x81, 0x7e, 0x80,
            0x83, 0x84, 0x83, 0x83, 0x81, 0x83, 0x84, 0x82, 0x81, 0x83, 0x80,
            0x81, 0x81, 0x81, 0x7f, 0x81, 0x82, 0x80, 0x81, 0x83, 0x82, 0x82,
            0x83, 0x81, 0x84, 0x83, 0x85, 0x84, 0x81, 0x81, 0x80, 0x80, 0x80,
            0x81, 0x7d
        };

        return snd.Create(sizeof(data), data);
    }

    if ( !m_soundRes.empty() )
    {
        return snd.Create(m_soundRes, true);
    }
#endif // __WXMSW__

    return snd.Create(m_soundFile);
}


void MyFrame::NotifyUsingFile(const wxString& name)
{
    wxString msg;
    msg << wxT("Using sound file: ") << name << wxT("\n");
    m_tc->AppendText(msg);
}


// event handlers


void MyFrame::OnSelectFile(wxCommandEvent& WXUNUSED(event))
{
#if wxUSE_FILEDLG
    wxFileDialog dlg(this, wxT("Choose a sound file"),
                     wxEmptyString, wxEmptyString,
                     wxT("WAV files (*.wav)|*.wav"), wxFD_OPEN|wxFD_CHANGE_DIR);
    if ( dlg.ShowModal() == wxID_OK )
    {
        m_soundFile = dlg.GetPath();
#ifdef __WXMSW__
        m_soundRes.clear();
#endif // __WXMSW__
        m_useMemory = false;

        wxDELETE(m_sound);
        NotifyUsingFile(m_soundFile);
    }
#endif // wxUSE_FILEDLG
}

#ifdef __WXMSW__

void MyFrame::OnSelectResource(wxCommandEvent& WXUNUSED(event))
{
    m_soundRes = wxGetTextFromUser
                 (
                    wxT("Enter resource name:"),
                    wxT("wxWidgets Sound Sample"),
                    wxT("FromResource"),
                    this
                 );
    if ( m_soundRes.empty() )
        return;

    m_soundFile.clear();
    m_useMemory = false;

    wxDELETE(m_sound);

    NotifyUsingFile(wxT("Windows WAV resource"));
}

#endif // __WXMSW__

void MyFrame::OnSelectMemory(wxCommandEvent& WXUNUSED(event))
{
    m_useMemory = true;

    NotifyUsingFile(wxT("embedded sound fragment"));
}

void MyFrame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
    // true is to force the frame to close
    Close(true);
}

void MyFrame::OnPlaySync(wxCommandEvent& WXUNUSED(event))
{
    wxBusyCursor busy;
    if ( !m_sound )
    {
        m_sound = new wxSound;
        CreateSound(*m_sound);
    }

    if (m_sound->IsOk())
        m_sound->Play(wxSOUND_SYNC);
}

void MyFrame::OnPlayAsync(wxCommandEvent& WXUNUSED(event))
{
    wxBusyCursor busy;
    if ( !m_sound )
    {
        m_sound = new wxSound;
        CreateSound(*m_sound);
    }

    if (m_sound->IsOk())
        m_sound->Play(wxSOUND_ASYNC);
}

void MyFrame::OnPlayAsyncOnStack(wxCommandEvent& WXUNUSED(event))
{
    wxBusyCursor busy;
    wxSound snd;
    CreateSound(snd);
    if (snd.IsOk())
        snd.Play(wxSOUND_ASYNC);
}

void MyFrame::OnPlayLoop(wxCommandEvent& WXUNUSED(event))
{
    wxBusyCursor busy;
    if ( !m_sound )
    {
        m_sound = new wxSound;
        CreateSound(*m_sound);
    }

    if (m_sound->IsOk())
        m_sound->Play(wxSOUND_ASYNC | wxSOUND_LOOP);
}

void MyFrame::OnPlayBell(wxCommandEvent& WXUNUSED(event))
{
    wxBell();
}

void MyFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
    wxString msg;
    msg.Printf( wxT("This is the About dialog of the sound sample.\n")
                wxT("Welcome to %s"), wxVERSION_STRING);

    wxMessageBox(msg, wxT("About"), wxOK | wxICON_INFORMATION, this);
}

void MyFrame::OnStop(wxCommandEvent& WXUNUSED(event))
{
    wxSound::Stop();
}
