/*
 * ProxyPropertiesHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 98/09/18
 * Last modified by suhler on 04/08/30 09:02:32
 */

package sunlabs.brazil.handler;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.MimeHeaders;
import java.util.Enumeration;
import java.util.Properties;
import sunlabs.brazil.util.Format;

/**
 * Obtain properties format content from remote websites, and
 * add it to the current request properties.
 * Many of the handlers 
 * are designed to produce side effects, by inserting values into the
 * request properties (see {@link PropertiesHandler}).
 * If they are instead configured to produce the properties
 * in java properties format, then this handler
 * will read their output, and place the result in the request object on
 * their behalf.  This capability allows certain handlers to be run on
 * other web sites, yet behave as if they are in the handler chain.
 *
 * The following request properties are used:
 * <dl class=props>
 * <dt>type	<dd> The document type for files to process as
 *		     java properties (defaults to text/plain)
 * <dt>prepend	<dd> The prefix that should be prepended to each property
 *		     before it is inserted into the request properties
 * <dt>url	<dd> The url that should be used to fetch the remote content.
 *		     If not specified, the curent url is used instead. 
 *		     Any ${...} constructs in the url are evaluated at each
 *		     request.
 * </dl>
 * NOTE: This capability should be generalized.
 *
 * @author      Stephen Uhler
 * @version	2.2, 04/08/30
 */

public class ProxyPropertiesHandler extends GenericProxyHandler implements Handler {

    String type;	// document type for filtering
    String prepend = null;	// prepend all properties with this
    String mapUrl;

    public boolean
    init(Server server, String prefix) {
	type = server.props.getProperty(prefix + "type", "text/plain");
	server.log(Server.LOG_DIAGNOSTIC, prefix, "processing type: " + type);
	return super.init(server, prefix);
    }

    public boolean
    respond(Request request)
    throws IOException {
	request.log(Server.LOG_DIAGNOSTIC, prefix,
		"Calling ProxyProps handler respond");
	prepend = request.props.getProperty(prefix + "prepend");
	mapUrl = request.props.getProperty(prefix + "url");
	boolean result;
	if (mapUrl != null) {
	    String save = request.url;
	    request.url = Format.subst(request.props, mapUrl);
	    request.log(Server.LOG_DIAGNOSTIC, prefix + 
	    	"Mapping " + save + " -> " + mapUrl);
	    result = super.respond(request);
	    request.url = save;
	} else {
	    result = super.respond(request);
	}
	request.log(Server.LOG_DIAGNOSTIC, prefix, "PRoxy done: " + result);
	return result;
    }

    /**
     * See if the content needs to be filtered
     * Return "true" if "modifyContent" should be called
     * @param headers	mime headers for data to proxy
     */

    protected boolean shouldFilter(MimeHeaders headers) {
	String header = headers.get("Content-Type");
     	return (header != null && header.equals(type));
    }

    public byte[] modifyContent(Request request, byte[] content) {

	/*
 	 * Get snarf into byte input array, then do a url.props.load on it.
	 * If its not a properties object, just return it.  
	 * How do we know??
	 */

	ByteArrayInputStream in = new ByteArrayInputStream(content);
	Properties p;
	if (prepend != null) {
	    p = new Properties();
	} else {
	    p = request.props;
	}
	try {
	    p.load(in);
	    request.log(Server.LOG_INFORMATIONAL, prefix + 
		    "..  Got remote properties");
	} catch (java.io.IOException e) {
	    request.props.put("Error",e.toString());
	}

	/* add the prefix, if any (yuk!) */

	if (prepend != null) {
	    Properties to = request.props;
	    request.log(Server.LOG_INFORMATIONAL, prefix + 
		    "..  prepending: " + prepend);
	    Enumeration enumer = p.propertyNames();
	    while(enumer.hasMoreElements()) {
		String key = (String) enumer.nextElement();
		to.put(prepend + key, p.getProperty(key));
	    }
	    p = null;
	}
	try {
	    in.close();
	} catch (java.io.IOException e) {
	    request.log(Server.LOG_WARNING, prefix + 
		    "..  Close failed!! reading props: " + e);
	}
	return null;
    }
}
