/*
 *  Copyright (c) 2008 Mike Massonnet <mmassonnet@xfce.org>
 *
 *  GVfs mount operations inspired by the gvfs programs:
 *  Copyright (c) 2006-2007 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gio/gio.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include "menu.h"

/*
 * GObject declarations
 */

G_DEFINE_TYPE (XgmPluginMenu, xgm_plugin_menu, GTK_TYPE_MENU)

#define GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), XGM_PLUGIN_TYPE_MENU, XgmPluginMenuPrivate))

struct _XgmPluginMenuPrivate
{
  GSList               *mount_list;
  GtkWidget            *recent_menu;
  GtkWidget            *submenu; /* for open, unmount */
};

static void             xgm_plugin_menu_class_init          (XgmPluginMenuClass *klass);
static void             xgm_plugin_menu_init                (XgmPluginMenu *menu);
static void             xgm_plugin_menu_finalize            (GObject *object);

/*
 * Private methods declarations
 */

static void            _xgm_plugin_menu_free_mount_list     (XgmPluginMenu *menu);
static void            _xgm_plugin_menu_update_mounts       (XgmPluginMenu *menu);
static void            _xgm_plugin_menu_update_recent       (XgmPluginMenu *menu);

/*
 * Callbacks declarations
 */

static void             cb_add                              (GtkMenuItem *item);
static void             cb_open_mount                       (GFile *root);
static void             cb_unmount_mount                    (GFile *root);
static void             cb_unmount_done                     (GObject *source_object,
                                                             GAsyncResult *res
                                                             /*, gpointer user_data */);
static void             cb_update_menu                      (GtkWidget *menu);
static gboolean         cb_open_submenu                     (GtkWidget *mi,
                                                             GdkEventButton *event,
                                                             GFile *root);
static void             cb_close_submenu                    (GtkMenuShell *submenu,
                                                             GtkWidget *menu);



/*
 * Callbacks
 */

static void
cb_add (GtkMenuItem *item)
{
  GError *error = NULL;
  gchar *command;

  command = g_find_program_in_path ("xfce4-gvfs-mount");
  if (command == NULL)
    command = g_strdup ("xfce4-gvfs-mount");

  xfce_exec (command, FALSE, FALSE, &error);
  if (error != NULL)
    {
      xfce_err ("%s", error->message);
      g_error_free (error);
    }

  g_free (command);
}

static void
cb_open_mount (GFile *root)
{
  GError *error = NULL;
  gchar *uri;

  uri = g_file_get_uri (root);
  g_app_info_launch_default_for_uri (uri, NULL, &error);
  if (error != NULL)
    {
      xfce_err (_("Failed to open location '%s': %s\nMake sure you have setup "
                  "Fuse for your user and that gvfs-fuse-daemon is running."),
                uri, error->message);
      g_error_free (error);
    }
  g_free (uri);
}

static void
cb_mount (GtkMenuItem *mi,
          XgmPluginMenu *menu)
{
  GError *error = NULL;
  gchar *command, *uri, *full_command;

  command = g_find_program_in_path ("xfce4-gvfs-mount");
  if (command == NULL)
    command = g_strdup ("xfce4-gvfs-mount");
  uri = g_object_get_data (G_OBJECT (mi), "uri");
  full_command = g_strdup_printf ("%s -l %s", command, uri);

  xfce_exec (full_command, FALSE, FALSE, &error);
  if (error != NULL)
    {
      xfce_err ("%s", error->message);
      g_error_free (error);
    }

  g_free (command);
  g_free (full_command);
}

static void
cb_unmount_mount (GFile *root)
{
  GError *error = NULL;
  GMount *mount;

  mount = g_file_find_enclosing_mount (root, NULL, &error);
  if (error != NULL)
    {
      xfce_err (_("Error finding enclosing mount: %s"), error->message);
      g_error_free (error);
      return;
    }
  g_mount_unmount (mount, 0, NULL, (GAsyncReadyCallback)cb_unmount_done, NULL);
}

static void
cb_unmount_done (GObject *source_object,
                 GAsyncResult *res
                 /*, gpointer user_data */)
{
  GError *error = NULL;
  gboolean succeeded;
  
  succeeded = g_mount_unmount_finish (G_MOUNT (source_object), res, &error);
  g_object_unref (G_MOUNT (source_object));
  if (!succeeded)
    {
      xfce_err (_("Error unmounting mount: %s"), error->message);
      g_error_free (error);
    }
}

static void
cb_update_menu (GtkWidget *menu)
{
  _xgm_plugin_menu_update_mounts (XGM_PLUGIN_MENU (menu));
  _xgm_plugin_menu_update_recent (XGM_PLUGIN_MENU (menu));
}

static gboolean
cb_open_submenu (GtkWidget *mi,
                 GdkEventButton *event,
                 GFile *root)
{
  GtkWidget *menu;
  GtkWidget *submenu;
  GtkWidget *_mi;

  if (event->button != 3)
    return FALSE;

  menu = gtk_widget_get_parent (mi);

  if (GTK_IS_MENU (XGM_PLUGIN_MENU (menu)->priv->submenu))
    gtk_widget_destroy (XGM_PLUGIN_MENU (menu)->priv->submenu);

  submenu = XGM_PLUGIN_MENU (menu)->priv->submenu = gtk_menu_new ();
  g_signal_connect (submenu, "deactivate", G_CALLBACK (cb_close_submenu), menu);

  _mi = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (submenu), _mi);
  g_signal_connect_swapped (_mi, "activate", G_CALLBACK (cb_open_mount), root);

  _mi = gtk_image_menu_item_new_from_stock (GTK_STOCK_DISCONNECT, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (submenu), _mi);
  g_signal_connect_swapped (_mi, "activate", G_CALLBACK (cb_unmount_mount), root);

  gtk_widget_show_all (submenu);

  gtk_menu_popup (GTK_MENU (submenu), NULL, NULL, NULL, NULL, 3, gtk_get_current_event_time ());

  return TRUE;
}

static void
cb_close_submenu (GtkMenuShell *submenu,
                  GtkWidget *menu)
{
  gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
}

/*
 * Private methods
 */

static void
_xgm_plugin_menu_free_mount_list (XgmPluginMenu *menu)
{
  GSList *l;
  for (l = menu->priv->mount_list; l != NULL; l = l->next)
    gtk_widget_destroy (GTK_WIDGET (l->data));
  g_slist_free (menu->priv->mount_list);
  menu->priv->mount_list = NULL;
}

static void
_xgm_plugin_menu_update_mounts (XgmPluginMenu *menu)
{
  GVolumeMonitor *monitor;
  GList *mounts, *l;
  GMount *mount;
  GVolume *volume;
  gchar *name;
  GtkWidget *mi;
  gint n_mounts;

  _xgm_plugin_menu_free_mount_list (XGM_PLUGIN_MENU (menu));

  monitor = g_volume_monitor_get ();
  mounts = g_volume_monitor_get_mounts (monitor);

  for (n_mounts = 0, l = mounts; l != NULL; l = l->next)
    {
      mount = l->data;
      volume = g_mount_get_volume (mount);
      if (volume != NULL)
        {
          /* The plugin doesn't show what Thunar is able to show */
          g_object_unref (mount);
          g_object_unref (volume);
          continue;
        }

      name = g_mount_get_name (mount);
      mi = gtk_menu_item_new_with_label (name);

      menu->priv->mount_list = g_slist_prepend (menu->priv->mount_list, mi);
      g_signal_connect (mi, "button-press-event", G_CALLBACK (cb_open_submenu), g_mount_get_root (mount));
      g_signal_connect_swapped (mi, "activate", G_CALLBACK (cb_open_mount), g_mount_get_root (mount));
      gtk_menu_shell_insert (GTK_MENU_SHELL (menu), mi, n_mounts++);
      gtk_widget_show (mi);

      g_free (name);
      g_object_unref (mount);
    }

  if (n_mounts == 0)
    {
      mi = gtk_menu_item_new_with_label (_("No mounts"));
      menu->priv->mount_list = g_slist_prepend (menu->priv->mount_list, mi);
      gtk_menu_shell_insert (GTK_MENU_SHELL (menu), mi, 0);
      gtk_widget_set_sensitive (mi, FALSE);
      gtk_widget_show (mi);
    }

  g_list_free (mounts);
  g_object_unref (monitor);

}

static void
_xgm_plugin_menu_update_recent (XgmPluginMenu *menu)
{
  GtkWidget *mi;
  gchar *filename;
  GBookmarkFile *bookmark;
  gchar **uris;
  gchar *exec;
  gint i;
  gsize length;

  filename = g_strdup_printf ("%s/"PACKAGE"/recently-used.xbel", g_get_user_config_dir ());
  bookmark = g_bookmark_file_new ();
  g_bookmark_file_load_from_file (bookmark, filename, NULL);
  uris = g_bookmark_file_get_uris (bookmark, &length);

  gtk_container_foreach (GTK_CONTAINER (menu->priv->recent_menu), (GtkCallback)gtk_widget_destroy, NULL);

  if (length > 0)
    {
      for (i = 0; i < length; i++)
        {
          g_bookmark_file_get_app_info (bookmark, uris[i], "Xfce4 GVfs Mount", &exec, NULL, NULL, NULL);

          mi = gtk_menu_item_new_with_label (uris[i]);
          g_object_set_data_full (G_OBJECT (mi), "uri", g_strdup (uris[i]), g_free);
          g_signal_connect (mi, "activate", G_CALLBACK (cb_mount), menu);
          gtk_widget_show (mi);
          gtk_menu_shell_append (GTK_MENU_SHELL (menu->priv->recent_menu), mi);

          g_free (exec);
        }
    }
  else
    {
      mi = gtk_menu_item_new_with_label (_("Empty"));
      gtk_widget_set_sensitive (mi, FALSE);
      gtk_widget_show (mi);
      gtk_menu_shell_append (GTK_MENU_SHELL (menu->priv->recent_menu), mi);
    }

  g_strfreev (uris);
  g_bookmark_file_free (bookmark);
  g_free (filename);
}

/*
 * Public methods
 */

GtkWidget *
xgm_plugin_menu_new ()
{
  return g_object_new (XGM_PLUGIN_TYPE_MENU, NULL);
}

/*
 * GObject
 */

static void
xgm_plugin_menu_class_init (XgmPluginMenuClass *klass)
{
  GObjectClass *object_class;

  g_type_class_add_private (klass, sizeof (XgmPluginMenuPrivate));

  xgm_plugin_menu_parent_class = g_type_class_peek_parent (klass);

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = xgm_plugin_menu_finalize;
}

static void
xgm_plugin_menu_init (XgmPluginMenu *menu)
{
  GtkWidget *mi;

  menu->priv = GET_PRIVATE (menu);
  g_signal_connect (menu, "show", G_CALLBACK (cb_update_menu), NULL);

  /* Footer items */
  mi = gtk_separator_menu_item_new ();
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), mi);

  mi = gtk_menu_item_new_with_mnemonic (_("_Recent"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), mi);
  menu->priv->recent_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (mi), menu->priv->recent_menu);

  mi = gtk_image_menu_item_new_from_stock (GTK_STOCK_ADD, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), mi);
  g_signal_connect (mi, "activate",
                    G_CALLBACK (cb_add), NULL);

  /* Show all the items */
  gtk_widget_show_all (GTK_WIDGET (menu));
}

static void
xgm_plugin_menu_finalize (GObject *object)
{
  _xgm_plugin_menu_free_mount_list (XGM_PLUGIN_MENU (object));
  G_OBJECT_CLASS (xgm_plugin_menu_parent_class)->finalize (object);
}

