/*
 *  Copyright (c) 2008 Mike Massonnet <mmassonnet@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include <libxfce4panel/xfce-panel-convenience.h>

#include "menu.h"



/*
 * MyPlugin structure
 */

typedef struct _MyPlugin MyPlugin;
struct _MyPlugin
{
  XfcePanelPlugin      *panel_plugin;
  GdkPixbuf            *tooltip_pixbuf;
  GtkWidget            *button;
  GtkWidget            *image;
  GtkWidget            *menu;
};

/*
 * Panel Plugin registration
 */

static void panel_plugin_register (XfcePanelPlugin *panel_plugin);
XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL (panel_plugin_register);

/*
 * Panel Plugin functions declarations
 */

static gboolean         panel_plugin_set_size           (XfcePanelPlugin *panel_plugin,
                                                         int size,
                                                         MyPlugin *plugin);
static void             panel_plugin_free               (XfcePanelPlugin *panel_plugin,
                                                         MyPlugin *plugin);
#if GTK_CHECK_VERSION (2,12,0)
static gboolean         cb_query_tooltip                (GtkWidget *panel_plugin,
                                                         gint x,
                                                         gint y,
                                                         gboolean keyboard_mode,
                                                         GtkTooltip *tooltip,
                                                         MyPlugin *plugin);
#endif
static void             cb_button_toggled               (GtkToggleButton *button,
                                                         MyPlugin *plugin);
static void             cb_menu_deactivate              (GtkMenuShell *menu,
                                                         MyPlugin *plugin);
static void             my_plugin_position_menu         (GtkMenu *menu,
                                                         gint *x,
                                                         gint *y,
                                                         gboolean *push_in,
                                                         MyPlugin *plugin);



/*
 * Panel Plugin functions
 */

static void
panel_plugin_register (XfcePanelPlugin *panel_plugin)
{
  MyPlugin *plugin;

  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

  /* MyPlugin */
  plugin = g_slice_new0 (MyPlugin);

  /* Keep a pointer on the panel plugin */
  plugin->panel_plugin = panel_plugin;

  /* Tooltip */
#if GTK_CHECK_VERSION (2,12,0)
  gtk_widget_set_has_tooltip (GTK_WIDGET (panel_plugin), TRUE);
  g_signal_connect (panel_plugin, "query-tooltip", G_CALLBACK (cb_query_tooltip), plugin);
#endif

  /* Panel Button */
  plugin->button = xfce_create_panel_toggle_button ();
  xfce_panel_plugin_add_action_widget (panel_plugin, plugin->button);
  /* NOTE the image is set through the set_size callback */
  plugin->image = gtk_image_new ();

  gtk_container_add (GTK_CONTAINER (plugin->button), plugin->image);
  gtk_container_add (GTK_CONTAINER (panel_plugin), plugin->button);

  /* Signals */
  g_signal_connect (panel_plugin, "size-changed",
                    G_CALLBACK (panel_plugin_set_size), plugin);
  g_signal_connect (panel_plugin, "free-data",
                    G_CALLBACK (panel_plugin_free), plugin);
  g_signal_connect (plugin->button, "toggled",
                    G_CALLBACK (cb_button_toggled), plugin);

  gtk_widget_show_all (GTK_WIDGET (panel_plugin));
}

static gboolean
panel_plugin_set_size (XfcePanelPlugin *panel_plugin,
                       int size,
                       MyPlugin *plugin)
{
  GdkPixbuf *pixbuf;

  gtk_widget_set_size_request (plugin->button, size, size);

  size -= 2 + 2 * MAX (plugin->button->style->xthickness,
                       plugin->button->style->ythickness);
  pixbuf = xfce_themed_icon_load ("folder-remote", size);
  if (G_UNLIKELY (pixbuf == NULL))
    pixbuf = xfce_themed_icon_load (GTK_STOCK_DIRECTORY, size);
  gtk_image_set_from_pixbuf (GTK_IMAGE (plugin->image), pixbuf);
  g_object_unref (G_OBJECT (pixbuf));

  return TRUE;
}

static void
panel_plugin_free (XfcePanelPlugin *panel_plugin,
                   MyPlugin *plugin)
{
  gtk_widget_destroy (plugin->button);
  if (GTK_IS_MENU (plugin->menu))
    gtk_widget_destroy (plugin->menu);
  if (plugin->tooltip_pixbuf != NULL)
    g_object_unref (plugin->tooltip_pixbuf);
  g_slice_free (MyPlugin, plugin);
}

#if GTK_CHECK_VERSION (2,12,0)
static gboolean
cb_query_tooltip (GtkWidget *panel_plugin,
                  gint x,
                  gint y,
                  gboolean keyboard_mode,
                  GtkTooltip *tooltip,
                  MyPlugin *plugin)
{
  GtkIconTheme *theme;
  GdkPixbuf *pixbuf;
  gchar *text;
 
  if (plugin->tooltip_pixbuf == NULL)
    {
      theme = gtk_icon_theme_get_default ();
      pixbuf = gtk_icon_theme_load_icon (theme, "folder-remote", 32, 0, NULL);
      if (G_LIKELY (pixbuf))
        {
          plugin->tooltip_pixbuf = gdk_pixbuf_scale_simple (pixbuf, 32, 32, GDK_INTERP_BILINEAR);
          g_object_unref (pixbuf);
        }
    }
  gtk_tooltip_set_icon (tooltip, plugin->tooltip_pixbuf);

  text = g_strdup_printf ("<b>%s</b>\n%s", _("Remote file-systems"),
                          _("Mount and manage remote file-systems"));
  gtk_tooltip_set_markup (tooltip, text);
  g_free (text);

  return TRUE;
}
#endif

static void
cb_button_toggled (GtkToggleButton *button,
                   MyPlugin *plugin)
{
  if (gtk_toggle_button_get_active (button))
    {
      if (!GTK_IS_MENU (plugin->menu))
        {
          plugin->menu = xgm_plugin_menu_new ();
          gtk_menu_set_screen (GTK_MENU (plugin->menu),
                               gtk_widget_get_screen (plugin->button));
          g_signal_connect (plugin->menu, "deactivate",
                            G_CALLBACK (cb_menu_deactivate), plugin);
        }

      xfce_panel_plugin_register_menu (plugin->panel_plugin, GTK_MENU (plugin->menu));
      gtk_menu_popup (GTK_MENU (plugin->menu), NULL, NULL,
                      (GtkMenuPositionFunc)my_plugin_position_menu, plugin,
                      0, gtk_get_current_event_time ());
    }
}

static void
cb_menu_deactivate (GtkMenuShell *menu,
                    MyPlugin *plugin)
{
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (plugin->button), FALSE);
}

static void
my_plugin_position_menu (GtkMenu *menu,
                         gint *x,
                         gint *y,
                         gboolean *push_in,
                         MyPlugin *plugin)
{
  GtkWidget *button;
  GtkRequisition requisition;
  GtkOrientation orientation;

  button = plugin->button;
  orientation = xfce_panel_plugin_get_orientation (plugin->panel_plugin);
  gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
  gdk_window_get_origin (button->window, x, y);

  switch (orientation)
    {
    case GTK_ORIENTATION_HORIZONTAL:
      if (*y + button->allocation.height + requisition.height > gdk_screen_height ())
        /* Show menu above */
        *y -= requisition.height;
      else
        /* Show menu below */
        *y += button->allocation.height;

      if (*x + requisition.width > gdk_screen_width ())
        /* Adjust horizontal position */
        *x = gdk_screen_width () - requisition.width;
      break;

    case GTK_ORIENTATION_VERTICAL:
      if (*x + button->allocation.width + requisition.width > gdk_screen_width ())
        /* Show menu on the right */
        *x -= requisition.width;
      else
        /* Show menu on the left */
        *x += button->allocation.width;

      if (*y + requisition.height > gdk_screen_height ())
        /* Adjust vertical position */
        *y = gdk_screen_height () - requisition.height;
      break;

    default:
      break;
    }
}

