/*
 *  Copyright (c) 2008-2009 Mike Massonnet <mmassonnet@xfce.org>
 *
 *  DBus daemon inspired by xfce4-notifyd:
 *  Copyright (c) 2008 Brian Tarricone <bjt23@cornell.edu>
 *
 *  GVfs mount operations inspired by the gvfs programs:
 *  Copyright (c) 2006-2007 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <gio/gio.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <glade/glade.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "daemon.h"

/*
 * GObject declarations
 */

G_DEFINE_TYPE (XgmDaemon, xgm_daemon, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), XGM_TYPE_DAEMON, XgmDaemonPrivate))

struct _XgmDaemonPrivate
{
  DBusGConnection      *dbus_conn;
};

static void             xgm_daemon_class_init          (XgmDaemonClass *klass);
static void             xgm_daemon_init                (XgmDaemon *daemon);
static void             xgm_daemon_finalize            (GObject *object);

/*
 * DBus declarations
 */

#define XGM_DAEMON_DBUS_SERVICE         "org.xfce.vfs.Daemon"
#define XGM_DAEMON_DBUS_PATH            "/org/xfce/vfs/Daemon"

static gboolean         xgm_daemon_mount               (XgmDaemon *daemon,
                                                        const gchar *location,
                                                        const gchar *password,
                                                        GError **error);
static gboolean         xgm_daemon_unmount             (XgmDaemon *daemon,
                                                        const gchar *location,
                                                        GError **error);
#include "daemon-dbus.h"

/*
 * Private methods declarations
 */

static gboolean        _xgm_daemon_start               (XgmDaemon *daemon,
                                                        GError **error);
static void            _xgm_daemon_add_to_recently_used (XgmDaemon *daemon,
                                                         const gchar *location);
static void            _xgm_daemon_open_mount           (XgmDaemon *daemon,
                                                         const gchar *uri);

/*
 * Callbacks declarations
 */

static void             cb_mount_ready                 (GObject *source_object,
                                                        GAsyncResult *res,
                                                        XgmDaemon *daemon);
static void             cb_unmount_ready               (GObject *source_object,
                                                        GAsyncResult *res
                                                        /*, gpointer user_data*/);
static void             cb_ask_password                (GMountOperation *op,
                                                        const gchar *message,
                                                        const gchar *default_user,
                                                        const gchar *default_domain,
                                                        GAskPasswordFlags flags
                                                        /*, gpointer user_data*/);



/*
 * Callbacks
 */

static void
cb_mount_ready (GObject *source_object,
                GAsyncResult *res,
                XgmDaemon *daemon)
{
  GError *error = NULL;
  GFile *file;
  gchar *uri;

  file = G_FILE (source_object);
  uri = g_file_get_uri (file);

  g_file_mount_enclosing_volume_finish (file, res, &error);

  if (error != NULL)
    {
      xfce_err (error->message);
      g_error_free (error);
    }
  else
    {
      _xgm_daemon_add_to_recently_used (daemon, uri);
      _xgm_daemon_open_mount (daemon, uri);
      g_signal_emit_by_name (daemon, "mounted", uri);
    }

  g_free (uri);
  g_object_unref (file);
}

static void
cb_unmount_ready (GObject *source_object,
                  GAsyncResult *res
                  /*, gpointer user_data*/)
{
  GError *error = NULL;

  g_mount_unmount_finish (G_MOUNT (source_object), res, &error);

  if (error != NULL)
    {
      g_warning ("Error unmounting mount: %s", error->message);
      g_error_free (error);
    }

  g_object_unref (source_object);
}

static void
cb_ask_password (GMountOperation *op,
                 const gchar *message,
                 const gchar *default_user,
                 const gchar *default_domain,
                 GAskPasswordFlags flags
                 /*, gpointer user_data*/)
{
  gint res;
  GladeXML *gxml;
  GtkWidget *dialog;
  const gchar *password;

  gxml = glade_xml_new (DATADIR"/xfce4-gvfs-mount/glade/dialog.glade", "ask-password-dialog", NULL);
  dialog = glade_xml_get_widget (gxml, "ask-password-dialog");

  xfce_titled_dialog_set_subtitle (XFCE_TITLED_DIALOG (dialog), message);

  if (flags & G_ASK_PASSWORD_NEED_USERNAME)
    {
      gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (gxml, "username")), default_user);
      gtk_widget_show (glade_xml_get_widget (gxml, "username"));
      gtk_widget_show (glade_xml_get_widget (gxml, "username-label"));
    }

  if (flags & G_ASK_PASSWORD_NEED_DOMAIN)
    {
      gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (gxml, "domain")), default_domain);
      gtk_widget_show (glade_xml_get_widget (gxml, "domain"));
      gtk_widget_show (glade_xml_get_widget (gxml, "domain-label"));
    }

  if (flags & G_ASK_PASSWORD_NEED_PASSWORD)
    {
      password = g_mount_operation_get_password (op);
      if (password != NULL)
        gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (gxml, "password")), password);
      gtk_widget_show (glade_xml_get_widget (gxml, "password"));
      gtk_widget_show (glade_xml_get_widget (gxml, "password-label"));
    }

  res = gtk_dialog_run (GTK_DIALOG (dialog));
  if (res == 1)
    {
      if (flags & G_ASK_PASSWORD_NEED_USERNAME)
        g_mount_operation_set_username (op, gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (gxml, "username"))));
      if (flags & G_ASK_PASSWORD_NEED_DOMAIN)
        g_mount_operation_set_domain (op, gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (gxml, "domain"))));
      if (flags & G_ASK_PASSWORD_NEED_PASSWORD)
        g_mount_operation_set_password (op, gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (gxml, "password"))));
      g_mount_operation_reply (op, G_MOUNT_OPERATION_HANDLED);
    }
  else
    g_mount_operation_reply (op, G_MOUNT_OPERATION_ABORTED);

  gtk_widget_destroy (dialog);
  g_object_unref (gxml);
}

/*
 * Private methods
 */

static gboolean
_xgm_daemon_start (XgmDaemon *daemon,
                   GError **error)
{
  gint res;
  DBusError derror;

  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  daemon->priv->dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, error);
  if (G_UNLIKELY (daemon->priv->dbus_conn == NULL))
    return FALSE;

  dbus_error_init (&derror);
  res = dbus_bus_request_name (dbus_g_connection_get_connection (daemon->priv->dbus_conn),
                               XGM_DAEMON_DBUS_SERVICE, DBUS_NAME_FLAG_DO_NOT_QUEUE, &derror);
  if (res == DBUS_REQUEST_NAME_REPLY_EXISTS)
    {
      if (dbus_error_is_set (&derror))
        {
          dbus_set_g_error (error, &derror);
          dbus_error_free (&derror);
        }
      else
        g_set_error (error, DBUS_GERROR, DBUS_GERROR_FAILED, "Another Xfce4 GVfs daemon is running");
      return FALSE;
    }

  dbus_g_connection_register_g_object (daemon->priv->dbus_conn, XGM_DAEMON_DBUS_PATH, G_OBJECT (daemon));

  return TRUE;
}

static void
_xgm_daemon_add_to_recently_used (XgmDaemon *daemon,
                                  const gchar *location)
{
  GBookmarkFile *bookmark;
  gchar *filename;

  filename = g_strdup_printf ("%s/"PACKAGE"/recently-used.xbel", g_get_user_config_dir ());
  bookmark = g_bookmark_file_new ();
  if (g_file_test (filename, G_FILE_TEST_EXISTS))
    g_bookmark_file_load_from_file (bookmark, filename, NULL);

  g_bookmark_file_set_title (bookmark, location, location);
  g_bookmark_file_set_is_private (bookmark, location, TRUE);
  g_bookmark_file_add_group (bookmark, location, "GVfs");
  g_bookmark_file_add_application (bookmark, location, "Xfce4 GVfs Mount", "xfce4-gvfs-mount -l \%u");

  g_bookmark_file_to_file (bookmark, filename, NULL);
  g_bookmark_file_free (bookmark);
  g_free (filename);
}

static void
_xgm_daemon_open_mount (XgmDaemon *daemon,
                        const gchar *uri)
{
  g_app_info_launch_default_for_uri (uri, NULL, NULL);
}

/*
 * Public methods
 */

XgmDaemon *
xgm_daemon_new_unique (GError **error)
{
  XgmDaemon *daemon = g_object_new (XGM_TYPE_DAEMON, NULL);

  if (!_xgm_daemon_start (daemon, error))
    {
      g_object_unref (daemon);
      return NULL;
    }

  return daemon;
}

/*
 * DBus
 */

static gboolean
xgm_daemon_mount (XgmDaemon *daemon,
                  const gchar *location,
                  const gchar *password,
                  GError **error)
{
  GFile *file;
  GMountOperation *op;

  file = g_file_new_for_uri (location);

  op = g_mount_operation_new ();
  if (password != NULL && password[0] != '\0')
    {
      g_mount_operation_set_password (op, password);
    }
  g_signal_connect (op, "ask-password", G_CALLBACK (cb_ask_password), NULL);

  g_file_mount_enclosing_volume (file, 0, op, NULL, (GAsyncReadyCallback)cb_mount_ready, daemon);

  return TRUE;
}

static gboolean
xgm_daemon_unmount (XgmDaemon *daemon,
                    const gchar *location,
                    GError **error)
{
  GFile *file;
  GMount *mount;
  GError *_error = NULL;

  file = g_file_new_for_uri (location);

  mount = g_file_find_enclosing_mount (file, NULL, &_error);
  if (_error != NULL)
    {
      g_warning ("Error finding enclosing mount: %s", _error->message);
      g_error_free (_error);
      return FALSE;
    }

  g_mount_unmount (mount, 0, NULL, (GAsyncReadyCallback)cb_unmount_ready, NULL);

  g_object_unref (file);

  return TRUE;
}

/*
 * GObject
 */

static void
xgm_daemon_class_init (XgmDaemonClass *klass)
{
  GObjectClass *gobject_class;

  g_type_class_add_private (klass, sizeof (XgmDaemonPrivate));

  xgm_daemon_parent_class = g_type_class_peek_parent (klass);

  dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass), &dbus_glib_xgm_daemon_object_info);

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->finalize = xgm_daemon_finalize;

  g_signal_new ("mounted",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_LAST,
                0, NULL, NULL,
                g_cclosure_marshal_VOID__STRING,
                G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
xgm_daemon_init (XgmDaemon *daemon)
{
  daemon->priv = GET_PRIVATE (daemon);
}

static void
xgm_daemon_finalize (GObject *object)
{
  XgmDaemon *daemon = XGM_DAEMON (object);

  if (daemon->priv->dbus_conn)
    dbus_g_connection_unref (daemon->priv->dbus_conn);
}

