/*
 *  Copyright (c) 2008-2009 Mike Massonnet <mmassonnet@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <glade/glade.h>

#include "mount-dialog.h"
#include "utils.h"

/*
 * Protocols
 */

typedef enum
{
  XGM_PROTOCOL_SAMBA,
  XGM_PROTOCOL_SFTP,
  /* user created */
  XGM_PROTOCOL_GENERIC,
  N_PROTOCOLS,
} XgmProtocolType;

typedef struct _XgmProtocolEntry XgmProtocolEntry;
struct _XgmProtocolEntry
{
  gchar                *name;
  XgmProtocolType       type;
  gchar                *scheme;
  gchar                *glade_name;
};

static XgmProtocolEntry protocols[] =
{
    { "Samba", XGM_PROTOCOL_SAMBA, "smb", "samba" },
    { "SFTP", XGM_PROTOCOL_SFTP, "sftp", "sftp" },
    { "Generic", XGM_PROTOCOL_GENERIC, NULL, "generic" },
    { NULL },
};

/*
 * GObject declarations
 */

G_DEFINE_TYPE (XgmMountDialog, xgm_mount_dialog, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), XGM_TYPE_MOUNT_DIALOG, XgmMountDialogPrivate))

struct _XgmMountDialogPrivate
{
  GladeXML             *gxml;
  GladeXML             *gxml_protocol;
  const GtkWidget      *dialog;
  XgmProtocolType       protocol_type;

  gchar                *location;
};

enum
{
  PROP_0,
  LOCATION,
};

static void             xgm_mount_dialog_class_init         (XgmMountDialogClass *klass);
static void             xgm_mount_dialog_init               (XgmMountDialog *dialog);
static void             xgm_mount_dialog_finalize           (GObject *object);
static void             xgm_mount_dialog_get_property       (GObject *object,
                                                             guint prop_id,
                                                             GValue *value,
                                                             GParamSpec *pspec);
static void             xgm_mount_dialog_set_property       (GObject *object,
                                                             guint prop_id,
                                                             const GValue *value,
                                                             GParamSpec *pspec);

/*
 * Private methods declarations
 */

static void            _xgm_mount_dialog_init_settings      (XgmMountDialog *dialog);
static gchar *         _xgm_mount_dialog_get_location       (XgmMountDialog *dialog);
static void            _xgm_mount_dialog_set_location       (XgmMountDialog *dialog,
                                                             const gchar *location);
static void            _xgm_mount_dialog_set_protocol       (XgmMountDialog *dialog,
                                                             XgmProtocolType type);
static void            _xgm_mount_dialog_set_hostname       (XgmMountDialog *dialog,
                                                             const gchar *hostname);
static void            _xgm_mount_dialog_set_port           (XgmMountDialog *dialog,
                                                             gdouble port);
static void            _xgm_mount_dialog_set_path           (XgmMountDialog *dialog,
                                                             const gchar *path);
static void            _xgm_mount_dialog_set_username       (XgmMountDialog *dialog,
                                                             const gchar *username);

/*
 * Callbacks declarations
 */

static void             cb_protocol_changed                 (GtkComboBox *combobox,
                                                             XgmMountDialog *dialog);
static void             cb_anonymous_toggled                (GtkToggleButton *togglebutton,
                                                             XgmMountDialog *dialog);
static void             cb_hostname_changed                 (GtkEntry *entry,
                                                             XgmMountDialog *dialog);



/*
 * Callbacks
 */

static void
cb_protocol_changed (GtkComboBox *combobox,
                     XgmMountDialog *dialog)
{
  GtkWidget *settings, *child;
  gchar *filename;

  /* Settings frame */
  settings = glade_xml_get_widget (dialog->priv->gxml, "settings");

  /* Destroy the current child of the settings frame */
  child = gtk_bin_get_child (GTK_BIN (settings));
  if (GTK_IS_WIDGET (child))
    gtk_widget_destroy (child);

  /* Update gxml_protocol */
  if (GLADE_IS_XML (dialog->priv->gxml_protocol))
    g_object_unref (dialog->priv->gxml_protocol);

  dialog->priv->protocol_type = gtk_combo_box_get_active (combobox);
  filename = g_strdup_printf (DATADIR"/xfce4-gvfs-mount/glade/%s.glade", protocols[dialog->priv->protocol_type].glade_name);
  dialog->priv->gxml_protocol = glade_xml_new (filename, NULL, NULL);
  g_free (filename);

  if (G_UNLIKELY (!GLADE_IS_XML (dialog->priv->gxml_protocol)))
    return;

  /* Set the new child of the settings frame */
  child = glade_xml_get_widget (dialog->priv->gxml_protocol, "frame");

  if (G_LIKELY (gtk_widget_get_parent (child)))
    {
      g_object_ref (child);
      gtk_widget_unparent (child);
    }
  gtk_container_add (GTK_CONTAINER (settings), child);

  _xgm_mount_dialog_init_settings (dialog);

  /* Update mount button sensitivity */
  gtk_widget_set_sensitive (glade_xml_get_widget (dialog->priv->gxml, "button-mount"), FALSE);
}

static void
cb_anonymous_toggled (GtkToggleButton *togglebutton,
                      XgmMountDialog *dialog)
{
  GtkWidget *vbox;
  vbox = glade_xml_get_widget (dialog->priv->gxml_protocol, "table-anonymous");
  gtk_widget_set_sensitive (vbox, !gtk_toggle_button_get_active (togglebutton));
}

static void
cb_hostname_changed (GtkEntry *entry,
                     XgmMountDialog *dialog)
{
  GtkWidget *button_mount;
  const gchar *hostname, *path;
  gchar *name;
  gboolean is_sensitive = TRUE;

  hostname = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "hostname")));

  /* Update "save as" entry */
  /* TODO check if it wasn't modified by the user -- until then the field is left read-only */
  if (hostname[0] == '\0')
    {
      name = g_strdup ("");
      is_sensitive = FALSE;
    }
  else
    {
      if (dialog->priv->protocol_type == XGM_PROTOCOL_SAMBA)
        {
          path = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "path")));
          if (path[0] == '\0')
            is_sensitive = FALSE;
          name = g_strdup_printf ("\\\\%s\\%s", hostname, path);
        }
      else
        name = g_strdup_printf ("%s://%s", protocols[dialog->priv->protocol_type].scheme, hostname);
    }

  g_free (name);

  /* Upate mount button sensitivity */
  button_mount = glade_xml_get_widget (dialog->priv->gxml, "button-mount");
  gtk_widget_set_sensitive (button_mount, is_sensitive);
}

/*
 * Private methods
 */

static void
_xgm_mount_dialog_init_settings (XgmMountDialog *dialog)
{
  GtkWidget *hostname, *path, *anonymous, *username, *password;

  /* Hostname */
  hostname = glade_xml_get_widget (dialog->priv->gxml_protocol, "hostname");
  if (GTK_IS_WIDGET (hostname))
    g_signal_connect (hostname, "changed", G_CALLBACK (cb_hostname_changed), dialog);

  /* Path */
  if (dialog->priv->protocol_type == XGM_PROTOCOL_SAMBA)
    {
      path = glade_xml_get_widget (dialog->priv->gxml_protocol, "path");
      g_signal_connect (path, "changed", G_CALLBACK (cb_hostname_changed), dialog);
    }

  /* Anonymous */
  anonymous = glade_xml_get_widget (dialog->priv->gxml_protocol, "anonymous");
  if (GTK_IS_TOGGLE_BUTTON (anonymous))
    g_signal_connect (anonymous, "toggled", G_CALLBACK (cb_anonymous_toggled), dialog);

  /* Username */
  username = glade_xml_get_widget (dialog->priv->gxml_protocol, "username");
  if (GTK_IS_WIDGET (username))
    gtk_entry_set_text (GTK_ENTRY (username), g_get_user_name ());
}

static gchar *
_xgm_mount_dialog_get_location (XgmMountDialog *dialog)
{
  const gchar *scheme, *username = NULL, *hostname, *path = NULL;
  gchar *location;
  gint port = -1;
  gboolean is_anon = FALSE;
  XgmProtocolType type;

  type = dialog->priv->protocol_type;
  scheme = protocols[dialog->priv->protocol_type].scheme;
  username = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "username")));
  hostname = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "hostname")));

  /* Optional anonymous */
  if (type == XGM_PROTOCOL_SAMBA)
    {
      is_anon = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (glade_xml_get_widget (dialog->priv->gxml_protocol, "anonymous")));
    }

  /* Optional port */
  if (type == XGM_PROTOCOL_SFTP)
    {
      port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (glade_xml_get_widget (dialog->priv->gxml_protocol, "port")));
    }

  /* Optional path */
  if (type == XGM_PROTOCOL_SAMBA)
    {
      path = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "path")));
    }

  /* Build the location */
  switch (type)
    {
    case XGM_PROTOCOL_SAMBA:
      if (!is_anon)
        location = g_strdup_printf ("%s://%s@%s/%s", scheme, username, hostname, path);
      else
        location = g_strdup_printf ("%s://%s/%s", scheme, hostname, path);
      break;

    case XGM_PROTOCOL_SFTP:
      location = g_strdup_printf ("%s://%s@%s:%d", scheme, username, hostname, port);
      break;

    case XGM_PROTOCOL_GENERIC:
      location = g_strdup_printf ("%s", hostname);
      break;

    default:
      return NULL;
    }

  return location;
}

static void
_xgm_mount_dialog_set_location (XgmMountDialog *dialog,
                                const gchar *location)
{
  GtkWidget *button;
  XgmProtocolType type;
  gchar *scheme;
  gchar **split;
  gchar *username;
  gchar *hostname;
  gdouble port;
  gchar *path;
  gint i;

  g_return_if_fail (G_LIKELY (location != NULL));

  /* Parse the scheme */
  scheme = g_uri_parse_scheme (location);
  if (scheme == NULL)
    {
      g_warning ("The location has no scheme set");
      return;
    }

  type = XGM_PROTOCOL_GENERIC;
  for (i = 0; protocols[i].name != NULL; i++)
    {
      if (protocols[i].scheme != NULL && !g_ascii_strcasecmp (scheme, protocols[i].scheme))
        {
          type = i;
          break;
        }
    }
  if (type == XGM_PROTOCOL_GENERIC)
    {
      _xgm_mount_dialog_set_protocol (dialog, type);
      _xgm_mount_dialog_set_hostname (dialog, location);
      g_free (scheme);
      return;
    }

  /* Parse the hostname + path */
  split = g_strsplit (location, "/", 4);
  hostname = g_strdup (split[2]);
  path = g_strdup (split[3]);
  g_strfreev (split);

  if (hostname == NULL)
    {
      g_warning ("The location has no hostname set");
      g_free (scheme);
      return;
    }

  /* Parse hostname + username */
  username = NULL;
  split = g_strsplit (hostname, "@", 2);
  g_free (hostname);
  if (split[1] != NULL)
    {
      hostname = g_strdup (split[1]);
      username = g_strdup (split[0]);
    }
  else
    hostname = g_strdup (split[0]);
  g_strfreev (split);

  /* Parse hostname + port */
  port = 0;
  split = g_strsplit (hostname, ":", 2);
  g_free (hostname);
  if (split[1] != NULL)
    {
      hostname = g_strdup (split[0]);
      port = g_ascii_strtod (split[1], NULL);
    }
  else
    hostname = g_strdup (split[0]);
  g_strfreev (split);

  /* Update the GUI */
  _xgm_mount_dialog_set_protocol (dialog, type);
  _xgm_mount_dialog_set_hostname (dialog, hostname);
  if (port > 0)
    _xgm_mount_dialog_set_port (dialog, port);
  if (path != NULL)
    _xgm_mount_dialog_set_path (dialog, path);
  if (username != NULL)
    _xgm_mount_dialog_set_username (dialog, username);

  /* Focus the OK button */
  button = glade_xml_get_widget (dialog->priv->gxml, "button-mount");
  gtk_widget_grab_focus (button);

  g_free (scheme);
  g_free (hostname);
  g_free (path);
  g_free (username);
}

static void
_xgm_mount_dialog_set_protocol (XgmMountDialog *dialog,
                                XgmProtocolType type)
{
  g_return_if_fail (G_LIKELY (type >= 0 && type < N_PROTOCOLS));
  gtk_combo_box_set_active (GTK_COMBO_BOX (glade_xml_get_widget (dialog->priv->gxml, "protocol-combobox")), type);
}

static void
_xgm_mount_dialog_set_hostname (XgmMountDialog *dialog,
                                const gchar *hostname)
{
  g_return_if_fail (G_LIKELY (hostname != NULL));
  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "hostname")), hostname);
}

static void
_xgm_mount_dialog_set_port (XgmMountDialog *dialog,
                            gdouble port)
{
  GtkWidget *_port;

  g_return_if_fail (G_LIKELY (port > 0 && port < 65535));

  _port = glade_xml_get_widget (dialog->priv->gxml_protocol, "port");
  if (_port == NULL)
    return;

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (_port), port);
}

static  void
_xgm_mount_dialog_set_path (XgmMountDialog *dialog,
                            const gchar *path)
{
  GtkWidget *_path;

  g_return_if_fail (G_LIKELY (path != NULL));

  _path = glade_xml_get_widget (dialog->priv->gxml_protocol, "path");
  if (_path == NULL)
    return;

  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "path")), path);
}

static void
_xgm_mount_dialog_set_username (XgmMountDialog *dialog,
                                const gchar *username)
{
  GtkWidget *anonymous;

  g_return_if_fail (G_LIKELY (username != NULL));
  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "username")), username);

  anonymous = glade_xml_get_widget (dialog->priv->gxml_protocol, "anonymous");
  if (GTK_IS_TOGGLE_BUTTON (anonymous))
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (anonymous), FALSE);
}

/*
 * Public methods
 */

XgmMountDialog *
xgm_mount_dialog_new ()
{
  return g_object_new (XGM_TYPE_MOUNT_DIALOG, NULL);
}

gint
xgm_mount_dialog_run (XgmMountDialog *dialog)
{
  gint res;
  const gchar *password;

  res = gtk_dialog_run (GTK_DIALOG (dialog->priv->dialog));
  if (res == 1)
    {
      dialog->priv->location = _xgm_mount_dialog_get_location (dialog);
      password = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (dialog->priv->gxml_protocol, "password")));

      xgm_mount_with_password (dialog->priv->location, password);
    }

  return res;
}

/*
 * GObject
 */

static void
xgm_mount_dialog_class_init (XgmMountDialogClass *klass)
{
  GObjectClass *object_class;

  g_type_class_add_private (klass, sizeof (XgmMountDialogPrivate));

  xgm_mount_dialog_parent_class = g_type_class_peek_parent (klass);

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = xgm_mount_dialog_finalize;
  object_class->set_property = xgm_mount_dialog_set_property;
  object_class->get_property = xgm_mount_dialog_get_property;

  g_object_class_install_property (object_class, LOCATION,
                                   g_param_spec_string ("location", NULL, NULL,
                                                        NULL,
                                                        G_PARAM_READWRITE));
}

static void
xgm_mount_dialog_init (XgmMountDialog *dialog)
{
  gint i;
  GtkWidget *protocol_combobox;

  dialog->priv = GET_PRIVATE (dialog);

  /* Glade XML */
  dialog->priv->gxml = glade_xml_new (DATADIR"/xfce4-gvfs-mount/glade/dialog.glade", "mount-dialog", NULL);

  /* Dialog */
  dialog->priv->dialog = glade_xml_get_widget (dialog->priv->gxml, "mount-dialog");

  /* Protocols */
  protocol_combobox = glade_xml_get_widget (dialog->priv->gxml, "protocol-combobox");
  for (i = 0; protocols[i].name != NULL; i++)
    gtk_combo_box_insert_text (GTK_COMBO_BOX (protocol_combobox), protocols[i].type, protocols[i].name);
  glade_xml_signal_connect_data (dialog->priv->gxml, "cb_protocol_changed",
                                 G_CALLBACK (cb_protocol_changed), dialog);
  /* TODO Set to the last used protocol */
  _xgm_mount_dialog_set_protocol (dialog, XGM_PROTOCOL_SAMBA);
}

static void
xgm_mount_dialog_finalize (GObject *object)
{
  XgmMountDialog *dialog = XGM_MOUNT_DIALOG (object);
  g_object_unref (dialog->priv->gxml_protocol);
  g_object_unref (dialog->priv->gxml);
}

static void
xgm_mount_dialog_get_property (GObject *object,
                               guint prop_id,
                               GValue *value,
                               GParamSpec *pspec)
{
  switch (prop_id)
    {
    case LOCATION:
      g_value_set_string (value, _xgm_mount_dialog_get_location (XGM_MOUNT_DIALOG (object)));
      break;

    default:
      break;
    }
}

static void
xgm_mount_dialog_set_property (GObject *object,
                               guint prop_id,
                               const GValue *value,
                               GParamSpec *pspec)
{
  switch (prop_id)
    {
    case LOCATION:
      _xgm_mount_dialog_set_location (XGM_MOUNT_DIALOG (object), g_value_get_string (value));
      break;

    default:
      break;
    }
}

