/*
 *  xfce4-modem-lights-plugin - a mail notification applet for the xfce4 panel
 *  Copyright (c) 2005 Andreas J. Guelzow <aguelzow@taliesin.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include <libxfce4panel/xfce-panel-convenience.h>

#include <net/if.h>

#define THEME_OFFLINE_ICON "modem-disconnected"
#define THEME_ONLINE_ICON "modem-connected"
#define THEME_DIALING_ICON "modem-dialing"

typedef enum
{
	modem_disconnected = 1 << 0,
	modem_dialing = 1 << 1,
	modem_connected = 1 << 2
} modemlights_mode_t;

typedef struct
{
    XfcePanelPlugin *plugin;

	modemlights_mode_t mode;
	guint timer;
	
	GtkWidget *button;
	GtkWidget *image;
	
	GdkPixbuf *pix_online;
	GdkPixbuf *pix_offline;
	GdkPixbuf *pix_dialing;
	
	GtkTooltips *tooltip;
    
	gchar *connection_cmd;
	gchar *disconnection_cmd;
	gchar *device;
	gchar *lockfile;
	gchar *icon_disconnected;
	gchar *icon_connected;
	gchar *icon_dialing;
} XfceModemlightsPlugin;



static gboolean         interface_is_up                (const gchar *if_name);

static void             modemlights_set_mode           (XfceModemlightsPlugin *mwp,
                                                        modemlights_mode_t mode);
static void             modemlights_set_pixmaps        (XfceModemlightsPlugin *mwp,
                                                        gint which);
static gboolean         modemlights_button_release_cb  (GtkWidget *w,
                                                        GdkEventButton *evt,
                                                        gpointer user_data);
static gboolean         modemlights_timer              (gpointer data);

static XfceModemlightsPlugin* modemlights_create       (XfcePanelPlugin *plugin);

static void             modemlights_read_config        (XfceModemlightsPlugin *mwp);

static void             modemlights_write_config       (XfceModemlightsPlugin *mwp);

static gboolean         modemlights_lockfile_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_connection_cmd_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_disconnection_cmd_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_icon_disconnected_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_icon_connected_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_icon_dialing_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static gboolean         modemlights_device_focusout_cb (GtkWidget *w,
                                                        GdkEventFocus *evt,
                                                        gpointer user_data);
static void             lf_browse_cb                   (GtkWidget *b,
                                                        GtkEntry *entry);
static GtkWidget*       modemlights_create_device_entry (XfceModemlightsPlugin *mwp,
                                                        char const *label,
                                                        char const *prevalue,
                                                        GCallback cb,
                                                        GtkTable *table,
                                                        int row);
static GtkWidget*       modemlights_create_lockfile_selector (XfceModemlightsPlugin *mwp,
                                                        char const *prevalue,
                                                        GCallback cb,
                                                        GtkTable *table,
                                                        int row);
static void             ic_browse_cb                   (GtkWidget * b,
                                                        GtkEntry * entry);
static GtkWidget*       modemlights_create_icon_selector (XfceModemlightsPlugin *mwp,
                                                        char const *label,
                                                        char const *prevalue,
                                                        GCallback cb,
                                                        GtkTable *table,
                                                        int row);
static GtkWidget*       modemlights_create_entry       (XfceModemlightsPlugin *mwp,
                                                        char const *label,
                                                        char const *prevalue,
                                                        GCallback cb,
                                                        GtkTable *table,
                                                        int row);
static void             modemlights_create_info        (GtkWidget *widget,
                                                        const gchar *text);
static void             modemlights_create_options     (XfceModemlightsPlugin *mwp);

static void             modemlights_free               (XfceModemlightsPlugin *mwp);

static void             modemlights_set_size           (XfceModemlightsPlugin *mwp,
                                                        gint size);
static void             modemlights_register           (XfcePanelPlugin *plugin);



static gboolean 
interface_is_up (const gchar *if_name)
{
	struct if_nameindex *interfaces, *saved_interfaces;
	gint found = FALSE;

	if ((interfaces = if_nameindex()) == NULL)
		return FALSE;
	saved_interfaces = interfaces;

	while (interfaces->if_index != 0)
	{
		if (g_ascii_strcasecmp(interfaces->if_name, if_name) == 0)
		{
			found = TRUE;
			break;
		}
		interfaces++;
	}

	if_freenameindex(saved_interfaces);

	return found;
}

static void
modemlights_set_mode (XfceModemlightsPlugin *mwp, modemlights_mode_t mode)
{
	if (mode != mwp->mode) {
		mwp->mode = mode;
		switch (mode) {
		case modem_disconnected:
			if (mwp->pix_offline)
				xfce_scaled_image_set_from_pixbuf
					(XFCE_SCALED_IMAGE(mwp->image),
					 mwp->pix_offline);
			gtk_tooltips_set_tip(mwp->tooltip, mwp->button, 
					     _("No connection"), NULL);
			break;
		case modem_connected:
			if (mwp->pix_online)
				xfce_scaled_image_set_from_pixbuf
					(XFCE_SCALED_IMAGE(mwp->image),
					 mwp->pix_online);
			gtk_tooltips_set_tip(mwp->tooltip, mwp->button, 
					     _("Connection Established"), 
					     NULL);
			break;
		case modem_dialing:
			if (mwp->pix_dialing)
				xfce_scaled_image_set_from_pixbuf
					(XFCE_SCALED_IMAGE(mwp->image),
					 mwp->pix_dialing);
			gtk_tooltips_set_tip(mwp->tooltip, mwp->button, 
					     _("Dialing"), NULL);
			break;
		}
	}
}

static void
modemlights_set_pixmaps (XfceModemlightsPlugin *mwp, gint which)
{
    gint size = xfce_panel_plugin_get_size (mwp->plugin);

	if (modem_disconnected & which) {
		if(mwp->pix_offline) {
			g_object_unref(G_OBJECT(mwp->pix_offline));
			mwp->pix_offline = NULL;
		}
		if (mwp->icon_disconnected && *mwp->icon_disconnected)
			mwp->pix_offline = gdk_pixbuf_new_from_file_at_size
				(mwp->icon_disconnected, 
				 size, 
				 size, NULL);
		if (!mwp->pix_offline)
			mwp->pix_offline = xfce_themed_icon_load
				(THEME_OFFLINE_ICON, size);
		if (mwp->mode == modem_disconnected && mwp->pix_offline) 
			xfce_scaled_image_set_from_pixbuf
				(XFCE_SCALED_IMAGE(mwp->image),
				 mwp->pix_offline);
	}
	if (modem_connected & which) {
		if(mwp->pix_online) {
			g_object_unref(G_OBJECT(mwp->pix_online));
			mwp->pix_online = NULL;
		}
		if (mwp->icon_connected && *mwp->icon_connected)
			mwp->pix_online = gdk_pixbuf_new_from_file_at_size
				(mwp->icon_connected, 
				 size, 
				 size, NULL);
		if (!mwp->pix_online)
			mwp->pix_online = xfce_themed_icon_load
				(THEME_ONLINE_ICON, size);
		if (mwp->mode == modem_connected && mwp->pix_online) 
			xfce_scaled_image_set_from_pixbuf
				(XFCE_SCALED_IMAGE(mwp->image),
				 mwp->pix_online);
	}
	if (modem_dialing & which) {
		if(mwp->pix_dialing) {
			g_object_unref(G_OBJECT(mwp->pix_dialing));
			mwp->pix_dialing = NULL;
		}
		if (mwp->icon_dialing && *mwp->icon_dialing)
			mwp->pix_dialing = gdk_pixbuf_new_from_file_at_size
				(mwp->icon_dialing, 
				 size, 
				 size, NULL);
		if (!mwp->pix_dialing)
			mwp->pix_dialing = xfce_themed_icon_load
				(THEME_DIALING_ICON, size);
		if (mwp->mode == modem_dialing && mwp->pix_dialing) 
			xfce_scaled_image_set_from_pixbuf
				(XFCE_SCALED_IMAGE(mwp->image),
				 mwp->pix_dialing);
	}
}

static gboolean
modemlights_button_release_cb(GtkWidget *w, GdkEventButton *evt,
			      gpointer user_data)
{
	XfceModemlightsPlugin *mwp = user_data;
	
	if (evt->button != 1)
		return FALSE;

	switch(mwp->mode) {
        case modem_disconnected: 
		if(mwp->connection_cmd && (mwp->connection_cmd)) {
			xfce_exec(mwp->connection_cmd, FALSE, FALSE, NULL);
			modemlights_set_mode (mwp, modem_dialing);
		}
		break;
        case modem_dialing: 
        case modem_connected: 
		if(mwp->disconnection_cmd && (mwp->disconnection_cmd)) {
			xfce_exec(mwp->disconnection_cmd, FALSE, FALSE, NULL);
			modemlights_set_mode (mwp, modem_disconnected);
		}
		break;
	}
    
    return FALSE;
}

static gboolean    
modemlights_timer (gpointer data)
{
	XfceModemlightsPlugin *mwp = data;
	
	if (mwp->lockfile && *mwp->lockfile) {
		if (g_file_test (mwp->lockfile, G_FILE_TEST_EXISTS)) {
			if (mwp->device && *mwp->device 
			    && interface_is_up (mwp->device))
				modemlights_set_mode (mwp, modem_connected);
			else
				modemlights_set_mode (mwp, modem_dialing);
		} else
			modemlights_set_mode (mwp, modem_disconnected);
	}

	return TRUE;
}

static XfceModemlightsPlugin *
modemlights_create(XfcePanelPlugin *plugin)
{
	XfceModemlightsPlugin *mwp = g_new0(XfceModemlightsPlugin, 1);
    mwp->plugin = plugin;
    modemlights_read_config (mwp);

	mwp->mode = modem_disconnected;
	
	mwp->tooltip = gtk_tooltips_new();
	
	mwp->button = xfce_create_panel_button ();
	gtk_widget_show(mwp->button);
	g_signal_connect(mwp->button, "button-release-event",
			 G_CALLBACK(modemlights_button_release_cb), mwp);
	gtk_tooltips_set_tip(mwp->tooltip, mwp->button, 
			     _("No connection"), NULL);
	
	mwp->image = xfce_scaled_image_new();
	gtk_widget_show(mwp->image);
	gtk_container_add(GTK_CONTAINER(mwp->button), mwp->image);
	
	if(mwp->timer)
		g_source_remove (mwp->timer);
	mwp->timer = g_timeout_add (3000,
				    modemlights_timer,
				    mwp);
 
	return mwp;
}

static void
modemlights_read_config(XfceModemlightsPlugin *mwp)
{
    gchar              *file;
    XfceRc             *rc;

    file = xfce_panel_plugin_save_location (mwp->plugin, TRUE);
    DBG("RC file: `%s'", file);
    rc = xfce_rc_simple_open (file, FALSE);
    g_free (file);
    g_return_if_fail (rc);

    mwp->connection_cmd =
      g_strdup (xfce_rc_read_entry (rc, "connection_cmd", "/usr/bin/pon"));
    mwp->disconnection_cmd =
      g_strdup (xfce_rc_read_entry (rc, "disconnection_cmd", "/usr/bin/poff"));
    mwp->device =
      g_strdup (xfce_rc_read_entry (rc, "device", "ppp0"));
    mwp->lockfile =
      g_strdup (xfce_rc_read_entry (rc, "lockfile", "/var/lock/LCK..ttyS1"));
    mwp->icon_disconnected =
      g_strdup (xfce_rc_read_entry (rc, "icon_disconnected", ""));
    mwp->icon_connected =
      g_strdup (xfce_rc_read_entry (rc, "icon_connected", ""));
    mwp->icon_dialing =
      g_strdup (xfce_rc_read_entry (rc, "icon_dialing", ""));

    xfce_rc_close (rc);
}

static void
modemlights_write_config(XfceModemlightsPlugin *mwp)
{
    gchar              *file;
    XfceRc             *rc;

    file = xfce_panel_plugin_save_location (mwp->plugin, TRUE);
    rc = xfce_rc_simple_open (file, FALSE);
    g_free (file);
    g_return_if_fail (rc);

    xfce_rc_write_entry (rc, "connection_cmd",      mwp->connection_cmd);
    xfce_rc_write_entry (rc, "disconnection_cmd",   mwp->disconnection_cmd);
    xfce_rc_write_entry (rc, "device",              mwp->device);
    xfce_rc_write_entry (rc, "lockfile",            mwp->lockfile);
    xfce_rc_write_entry (rc, "icon_disconnected",   mwp->icon_disconnected);
    xfce_rc_write_entry (rc, "icon_connected",      mwp->icon_connected);
    xfce_rc_write_entry (rc, "icon_dialing",        mwp->icon_dialing);

    xfce_rc_close (rc);
}

static gboolean
modemlights_lockfile_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
				 gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *filename;
    
    if (mwp->lockfile != NULL)
	    g_free(mwp->lockfile);
    
    filename = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->lockfile = (filename != NULL) ? g_strdup(filename) : NULL;
    
    return FALSE;
}

static gboolean
modemlights_connection_cmd_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
				       gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *command;
    
    if (mwp->connection_cmd != NULL)
	    g_free(mwp->connection_cmd);
    
    command = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->connection_cmd = (command != NULL) ? g_strdup(command) : NULL;
    
    return FALSE;
}

static gboolean
modemlights_disconnection_cmd_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
					  gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *command;
    
    if (mwp->disconnection_cmd != NULL)
	    g_free(mwp->disconnection_cmd);
    
    command = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->disconnection_cmd = (command != NULL) ? g_strdup(command) : NULL;
    
    return FALSE;
}

static gboolean
modemlights_icon_disconnected_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
					  gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *filename;
    
    if (mwp->icon_disconnected != NULL)
	    g_free(mwp->icon_disconnected);
    
    filename = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->icon_disconnected = (filename != NULL) ? g_strdup(filename) : NULL;
    modemlights_set_pixmaps (mwp, modem_disconnected);    
    
    return FALSE;
}

static gboolean
modemlights_icon_connected_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
					  gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *filename;
    
    if (mwp->icon_connected != NULL)
	    g_free(mwp->icon_connected);
    
    filename = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->icon_connected = (filename != NULL) ? g_strdup(filename) : NULL;
    modemlights_set_pixmaps (mwp, modem_connected);    
    
    return FALSE;
}

static gboolean
modemlights_icon_dialing_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
					  gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *filename;
    
    if (mwp->icon_dialing != NULL)
	    g_free(mwp->icon_dialing);
    
    filename = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->icon_dialing = (filename != NULL) ? g_strdup(filename) : NULL;
    modemlights_set_pixmaps (mwp, modem_dialing);    
    return FALSE;
}

static gboolean
modemlights_device_focusout_cb(GtkWidget *w, GdkEventFocus *evt,
			       gpointer user_data)
{
    XfceModemlightsPlugin *mwp = user_data;
    gchar *name;
    
    if (mwp->device != NULL)
	    g_free(mwp->device);
    
    name = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    mwp->device = (name != NULL) ? g_strdup(name) : NULL;
    
    return FALSE;
}



static void
lf_browse_cb (GtkWidget * b, GtkEntry * entry)
{
    GtkWidget *dialog =
      gtk_file_chooser_dialog_new (_("Select lock file"),
                                   GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(entry))),
                                   GTK_FILE_CHOOSER_ACTION_OPEN,
                                   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                   GTK_STOCK_OK, GTK_RESPONSE_OK,
                                   NULL);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
        gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
        gtk_entry_set_text(entry, g_strdup(filename));
        g_free (filename);
        gtk_widget_grab_focus (GTK_WIDGET(entry));
        gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);
    }

    gtk_widget_destroy (dialog);
}

static GtkWidget *
modemlights_create_device_entry (XfceModemlightsPlugin *mwp, char const *label, 
			  char const *prevalue, GCallback cb, GtkTable *table, int row)
{
    /* Info button + label */
    GtkWidget *hbox0 = gtk_hbox_new(FALSE, 2);
    GtkWidget *btn0 = gtk_button_new();
    GtkWidget *img0 = gtk_image_new_from_stock(GTK_STOCK_INFO, GTK_ICON_SIZE_BUTTON);
    GtkWidget *lbl = gtk_label_new_with_mnemonic(label);

    gtk_misc_set_alignment (GTK_MISC(lbl), 0, 0.5);

    gtk_container_add (GTK_CONTAINER(btn0), img0);
    gtk_box_pack_start (GTK_BOX(hbox0), btn0, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX(hbox0), lbl, TRUE, TRUE, 0);
    gtk_table_attach (table, hbox0, 0, 1, row, row+1,
              GTK_FILL, GTK_FILL, 6, 6);

    g_signal_connect (btn0, "clicked",
                      G_CALLBACK (modemlights_create_info),
                      _("This is the device name (interface name) for the "
                        "established connection, such as:\n"
                        " <b>ppp0</b> \n"
                        "To determine the device name, you can execute the next "
                        "command before and after establishing the connection:\n"
                        " <b>cat /proc/net/dev</b> \n"
                        "This will show the active devices/interfaces in the first "
                        "column. The new device after establishing the connection "
                        "is the device/interface name required."));

    GtkWidget *hbox = gtk_hbox_new(FALSE, 8);
    gtk_widget_show(hbox);
    gtk_table_attach (table, hbox, 1, 2, row, row+1,
              GTK_FILL, GTK_FILL, 6, 6);

    GtkWidget *entry = gtk_entry_new();
    if(prevalue != NULL)
        gtk_entry_set_text(GTK_ENTRY(entry), prevalue);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);

    g_signal_connect(G_OBJECT(entry), "focus-out-event",
             cb, mwp);
    return hbox;
}

static GtkWidget *
modemlights_create_lockfile_selector (XfceModemlightsPlugin *mwp,
				      char const *prevalue, 
				      GCallback cb, GtkTable *table,
				      int row)
{
    /* Info button + label */
    GtkWidget *hbox0 = gtk_hbox_new(FALSE, 2);
    GtkWidget *btn0 = gtk_button_new();
    GtkWidget *img0 = gtk_image_new_from_stock(GTK_STOCK_INFO, GTK_ICON_SIZE_BUTTON);
	GtkWidget *lbl = gtk_label_new_with_mnemonic(_("Lockfile:"));

	gtk_misc_set_alignment (GTK_MISC(lbl), 0, 0.5);

    gtk_container_add (GTK_CONTAINER(btn0), img0);
	gtk_box_pack_start (GTK_BOX(hbox0), btn0, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox0), lbl, TRUE, TRUE, 0);
	gtk_table_attach (table, hbox0, 0, 1, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);

    g_signal_connect (btn0, "clicked",
                      G_CALLBACK (modemlights_create_info),
                      _("This is the name of the lock file created by the "
                        "connection command, such as:\n"
                        " <b>/var/lock/LCK..ttyS1</b> \n"
                        "You can frequently find this name by checking the "
                        "content of /var/lock before and after establishing the "
                        "connection. The absence of that file indicates to the "
                        "plugin that no connection has been established and that "
                        "no dialing is in progress."));

    /* Selector */
	GtkWidget *hbox1 = gtk_hbox_new(FALSE, 8);
	gtk_table_attach (table, hbox1, 1, 2, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);
	
	GtkWidget *entry = gtk_entry_new();
	if(prevalue != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), prevalue);
	gtk_box_pack_start(GTK_BOX(hbox1), entry, TRUE, TRUE, 0);
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);

	GtkWidget *button = gtk_button_new();
	GtkWidget *image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON);
	gtk_container_add (GTK_CONTAINER (button), image);
	gtk_box_pack_start(GTK_BOX(hbox1), button, FALSE, FALSE, 0);
	g_signal_connect (button, "clicked",
			  G_CALLBACK (lf_browse_cb), entry);

	g_signal_connect(G_OBJECT(entry), "focus-out-event",
			 cb, mwp);	

	return hbox1;
}

static void
ic_browse_cb (GtkWidget * b, GtkEntry * entry)
{
    GtkWidget *dialog =
      gtk_file_chooser_dialog_new (_("Select icon"),
                                   GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(entry))),
                                   GTK_FILE_CHOOSER_ACTION_OPEN,
                                   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                   GTK_STOCK_OK, GTK_RESPONSE_OK,
                                   NULL);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
        gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		gtk_entry_set_text(entry, g_strdup(filename));
        g_free (filename);
        gtk_widget_grab_focus (GTK_WIDGET(entry));
        gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);
    }

    gtk_widget_destroy (dialog);
}

static GtkWidget *
modemlights_create_icon_selector (XfceModemlightsPlugin *mwp, 
				  char const *label, char const *prevalue, 
				  GCallback cb, GtkTable *table,
				  int row)
{
	GtkWidget *hbox, *lbl, *entry, *button, *image;

	lbl = gtk_label_new_with_mnemonic(label);
	gtk_widget_show(lbl);
	gtk_table_attach (table, lbl, 0, 1, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);
	gtk_misc_set_alignment (GTK_MISC(lbl), 0, 0.5);

	hbox = gtk_hbox_new(FALSE, 8);
	gtk_widget_show(hbox);
	gtk_table_attach (table, hbox, 1, 2, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);
	
	entry = gtk_entry_new();
	if(prevalue != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), prevalue);
	gtk_widget_show(entry);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);

	button = gtk_button_new();
	image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show(image);
	gtk_container_add (GTK_CONTAINER (button), image);
	gtk_widget_show (button);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect (button, "clicked",
			  G_CALLBACK (ic_browse_cb), entry);

	g_signal_connect(G_OBJECT(entry), "focus-out-event",
			 cb, mwp);	
	return hbox;
}

static GtkWidget *
modemlights_create_entry (XfceModemlightsPlugin *mwp, char const *label, 
			  char const *prevalue, GCallback cb, GtkTable *table,
			  int row)
{
	GtkWidget *hbox, *lbl, *entry;

	
	lbl = gtk_label_new_with_mnemonic(label);
	gtk_widget_show(lbl);
	gtk_table_attach (table, lbl, 0, 1, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);
	gtk_misc_set_alignment (GTK_MISC(lbl), 0, 0.5);

	hbox = gtk_hbox_new(FALSE, 8);
	gtk_widget_show(hbox);
	gtk_table_attach (table, hbox, 1, 2, row, row+1, 
			  GTK_FILL, GTK_FILL, 6, 6);

	entry = gtk_entry_new();
	if(prevalue != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), prevalue);
	gtk_widget_show(entry);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);

	g_signal_connect(G_OBJECT(entry), "focus-out-event",
			 cb, mwp);	
	return hbox;
}

static void
modemlights_create_info (GtkWidget *widget,
                         const gchar *text)
{
    
    GtkWidget *dialog =
      gtk_message_dialog_new_with_markup (GTK_WINDOW(gtk_widget_get_toplevel (widget)),
                                          GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_MESSAGE_INFO,
                                          GTK_BUTTONS_CLOSE,
                                          text);
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

static void
modemlights_create_options(XfceModemlightsPlugin *mwp)
{
	GtkWidget *table;
	int row = 0;
	
	table = gtk_table_new(7,2,FALSE);
	
/* Connection Command */	
    TRACE("connection_cmd");
	modemlights_create_entry 
		(mwp, _("_Connection Command:"), 
		 mwp->connection_cmd, 
		 G_CALLBACK(modemlights_connection_cmd_focusout_cb),
		 GTK_TABLE(table), row++);

/* Disconnection Command */	
    TRACE("disconnection_cmd");
	modemlights_create_entry 
		(mwp, _("_Disconnection Command:"),
		 mwp->disconnection_cmd, 
		 G_CALLBACK(modemlights_disconnection_cmd_focusout_cb),
		 GTK_TABLE(table), row++);

/* Dev */	
    TRACE("device");
	modemlights_create_device_entry 
		(mwp, _("De_vice:"), 
		 mwp->device, 
		 G_CALLBACK(modemlights_device_focusout_cb),
		 GTK_TABLE(table), row++);

/* Lock File */	
    TRACE("lockfile");
	modemlights_create_lockfile_selector 
		(mwp, mwp->lockfile, 
		 G_CALLBACK(modemlights_lockfile_focusout_cb),
		 GTK_TABLE(table), row++);

/* Disconnected Icon */	
    TRACE("icon_disconnected");
	modemlights_create_icon_selector 
		(mwp, _("Icon (Disconnected):"), 
		 mwp->icon_disconnected, 
		 G_CALLBACK(modemlights_icon_disconnected_focusout_cb),
		 GTK_TABLE(table), row++);

/* Connecting Icon */	
    TRACE("icon_dialing");
	modemlights_create_icon_selector 
		(mwp, _("Icon (Connecting):"), 
		 mwp->icon_dialing, 
		 G_CALLBACK(modemlights_icon_dialing_focusout_cb),
		 GTK_TABLE(table), row++);

/* Connected Icon */	
    TRACE("icon_connected");
	modemlights_create_icon_selector 
		(mwp, _("Icon (Connected):"), 
		 mwp->icon_connected, 
		 G_CALLBACK(modemlights_icon_connected_focusout_cb),
		 GTK_TABLE(table), row++);

/* The dialog */
    TRACE("dialog");
    xfce_panel_plugin_block_menu (mwp->plugin);

    GtkWidget *dialog =
      xfce_titled_dialog_new_with_buttons (_("Modem Lights"),
        NULL,
        GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
        NULL);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);
	gtk_widget_show_all(GTK_DIALOG(dialog)->vbox);
    gtk_window_set_icon_name (GTK_WINDOW(dialog), "preferences-desktop");
    gtk_window_set_resizable (GTK_WINDOW(dialog), FALSE);
    gtk_dialog_run (GTK_DIALOG(dialog));
    gtk_widget_destroy (dialog);

    xfce_panel_plugin_unblock_menu (mwp->plugin);
    modemlights_write_config (mwp);
}

static void
modemlights_free(XfceModemlightsPlugin *mwp)
{
    gtk_main_quit ();
}

static void
modemlights_set_size (XfceModemlightsPlugin *mwp, gint size)
{
	modemlights_set_pixmaps 
		(mwp, modem_disconnected + modem_dialing + modem_connected);
	
	gtk_widget_set_size_request(GTK_WIDGET(mwp->plugin), size, size);
}

static void
modemlights_register (XfcePanelPlugin *plugin)
{
    xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

    XfceModemlightsPlugin *modemlights = modemlights_create (plugin);

    gtk_container_add (GTK_CONTAINER (plugin), modemlights->button);
    xfce_panel_plugin_add_action_widget (plugin, modemlights->button);
    xfce_panel_plugin_menu_show_configure (plugin);

    g_signal_connect_swapped (plugin,
                              "free-data",
                              G_CALLBACK(modemlights_free),
                              modemlights);
    g_signal_connect_swapped (plugin,
                              "save",
                              G_CALLBACK(modemlights_write_config),
                              modemlights);
    g_signal_connect_swapped (plugin,
                              "size-changed",
                              G_CALLBACK(modemlights_set_size),
                              modemlights);
    g_signal_connect_swapped (plugin,
                              "configure-plugin",
                              G_CALLBACK(modemlights_create_options),
                              modemlights);
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL (modemlights_register);

