/*  xmms_sndfile - An XMMS input plugin using libsndfile
**
**  Copyright (C) 2000, 2002 Erik de Castro Lopo
**
**  This program is free software ; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation ; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY ; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program ; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <math.h>
#include <stdio.h>
#include <pthread.h>
#include <string.h>

#include <xmms/plugin.h>
#include <xmms/configfile.h>
#include <xmms/util.h>

#include <sndfile.h>

#include "config.h"
#include "xmms_sndfile.h"

#define NAME		"libsndfile input plugin"
#define AUTHOR		"Erik de Castro Lopo"
#define CONTACT		"erikd@zip.com.au"
#define HOMEPAGE	"http://www.zip.com.au/~erikd/XMMS/"

#define BUFFER_SIZE 		8192
#define	TITLE_LEN			64

#if		CPU_IS_BIG_ENDIAN
#define	HOST_SIGNED_SHORT	FMT_S16_BE
#elif	CPU_IS_LITTLE_ENDIAN
#define	HOST_SIGNED_SHORT	FMT_S16_LE
#endif

static	GtkWidget *dialog = NULL, *button = NULL, *label = NULL ;

static	SNDFILE *sndfile = NULL ;
static	SF_INFO sfinfo ;

static	char	description [] = NAME " " VERSION ;
static	char	title_str [] = "About " NAME " " VERSION ;

static	char	about_str [] =
	"Copyright (C) 2000, 2002 " AUTHOR "\n"
	"Written and maintained by " AUTHOR " <erikd@zip.com.au>.\n"
	"Homepage: " HOMEPAGE ".\n\n"
	"This program is free software ; you can redistribute it and/or modify \n"
	"it under the terms of the GNU General Public License as published by \n"
	"the Free Software Foundation ; either version 2 of the License, or \n"
	"(at your option) any later version. \n \n"
	"This program is distributed in the hope that it will be useful, \n"
	"but WITHOUT ANY WARRANTY ; without even the implied warranty of \n"
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  \n"
	"See the GNU General Public License for more details. \n\n"
	"You should have received a copy of the GNU General Public \n"
	"License along with this program ; if not, write to \n"
	"the Free Software Foundation, Inc., \n"
	"59 Temple Place, Suite 330, \n"
	"Boston, MA  02111-1307  USA" ;

static	char 	song_title [64] ;
static	int 	song_length ;
static	int 	bit_rate = 0 ;
static	int 	decoding ;
static	int		pause_now ;
static	int 	seek_time = -1 ;

static	pthread_t decode_thread ;

static	void 	plugin_init (void) ;
static	void	about_dialog (void) ;
static	int		is_our_file (char *filename) ;
static	void 	play_start (char *filename) ;
static	void 	play_stop (void) ;
static	void 	do_pause (short pause) ;
static	void 	file_seek (int time) ;
static	int		get_time (void) ;
static	void 	get_song_info (char *filename, char **title, int *length) ;

static
InputPlugin xmms_sndfile_ip =
{	NULL,					/* Filled in by xmms	*/
	NULL,					/* Filled in by xmms	*/
	description,			/* The description that is shown in the preferences box 	*/
	plugin_init,			/* Called when the plugin is loaded	*/
	about_dialog,			/* Show the about_dialog box	*/
	NULL, 					/* Show the configure box	*/
	is_our_file,			/* Return 1 if the plugin can handle the file	*/
	NULL,					/* Scan dir	*/
	play_start,				/* Play file	*/
	play_stop,				/* Stop	*/
	do_pause,				/* Pause	*/
	file_seek,				/* Seek	*/
	NULL,					/* Set the equalizer, most plugins won't be able to do this	*/
	get_time,				/* Get the time, usually returns the output plugins output time	*/
	NULL,					/* Get volume	*/
	NULL,					/* Set volume	*/
	NULL,					/* OBSOLETE, DO NOT USE!	*/
	NULL,					/* OBSOLETE, DO NOT USE!	*/
	NULL,					/* Send data to the visualization plugins	*/
	NULL,					/* Fill in the stuff that is shown in the player window	*/
	NULL,					/* Show some text in the song title box. Filled in by xmms	*/
	get_song_info,			/* Function to grab the title string	*/
	NULL,					/* Bring up an info window for the filename passed in	*/
	NULL					/* Handle to the current output plugin. Filled in by xmms	*/
} ; /* xmms_sndfile_ip */

int
get_song_length (char *filename)
{	SNDFILE	*tmp_sndfile ;
	SF_INFO tmp_sfinfo ;

	if (! (tmp_sndfile = sf_open (filename, SFM_READ, &tmp_sfinfo)))
		return 0 ;

	sf_close (tmp_sndfile) ;
	tmp_sndfile = NULL ;

	if (tmp_sfinfo.samplerate <= 0)
		return 0 ;

	return (int) ceil (1000.0 * tmp_sfinfo.frames / tmp_sfinfo.samplerate) ;
} /* get_song_length */

static void
get_song_title (char *filename, char *title)
{	char tmp_title [TITLE_LEN] ;
	char *tmpptr ;

	if ((tmpptr = strrchr (filename, '/')))
		strncpy (tmp_title, tmpptr+1, TITLE_LEN) ;
	else
		strncpy (tmp_title, filename, TITLE_LEN) ;

	tmp_title [TITLE_LEN - 1] = 0 ;

	if ((tmpptr = strrchr (tmp_title, '.')))
		tmpptr [0] = 0 ;

	strncpy (title, tmp_title, TITLE_LEN) ;
} /* get_song_title */

static void
plugin_init (void)
{
	decoding = FALSE ;
	pause_now = FALSE ;
	dialog = NULL ;
	seek_time = -1 ;
} /* plugin_int */

static void
about_dialog (void)
{
	if (dialog)
		return ;

	dialog = gtk_dialog_new () ;
	gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		     GTK_SIGNAL_FUNC (gtk_widget_destroyed), &dialog) ;
	gtk_window_set_title (GTK_WINDOW (dialog), title_str) ;
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE) ;
	gtk_container_border_width (GTK_CONTAINER (dialog), 5) ;

	label = gtk_label_new (about_str) ;
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0) ;
	gtk_widget_show (label) ;

	button = gtk_button_new_with_label (" Close ") ;
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog)) ;
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		     FALSE, FALSE, 0) ;

	gtk_widget_show (button) ;
	gtk_widget_show (dialog) ;
	gtk_widget_grab_focus (button) ;
} /* about_dialog */

static int
is_our_file (char *filename)
{	SNDFILE	*tmp_sndfile ;
	SF_INFO tmp_sfinfo ;

	/* Have to open the file to see if libsndfile can handle it. */
	if (! (tmp_sndfile = sf_open (filename, SFM_READ, &tmp_sfinfo)))
		return FALSE ;

	/* It can so close file and return TRUE. */
	sf_close (tmp_sndfile) ;
	tmp_sndfile = NULL ;

	return TRUE ;
} /* is_our_file */

static void*
play_loop (void *arg)
{	static short buffer [BUFFER_SIZE] ;
	int samples, pcmbitwidth ;

	pcmbitwidth = 32 ;

	bit_rate = sfinfo.samplerate * pcmbitwidth * sfinfo.channels ;

	xmms_sndfile_ip.set_info (song_title, song_length, bit_rate, sfinfo.samplerate, sfinfo.channels) ;

	decoding = TRUE ;
	while (decoding)
	{
		/* sf_read_short will return 0 for all reads at EOF. */
		samples = sf_read_short (sndfile, buffer, BUFFER_SIZE) ;

		if (samples > 0 &&  decoding)
		{	xmms_sndfile_ip.output->write_audio (buffer, samples * sizeof (short)) ;
			xmms_sndfile_ip.add_vis_pcm (xmms_sndfile_ip.output->written_time (), HOST_SIGNED_SHORT, sfinfo.channels, samples * sizeof (short), buffer) ;

			/* Not sure why this is needed, but it can stutter at
			** the start of the first file if this isn't here.
			*/
			xmms_usleep (100) ;
   			}
		else
			xmms_usleep (10000) ;

#if 0
/* This stuff seems to screw up the ALSA output 
** plugin for files at higher sample rates (ie 96k).
*/
		/* Loop while waiting for the output buffer to empty. */
		while ((xmms_sndfile_ip.output->buffer_free () < (samples * sizeof (short))) && decoding)
			xmms_usleep (10000) ;
#endif

		/* Do seek if seek_time is valid. */
		if (seek_time > 0)
		{	sf_seek (sndfile, seek_time * sfinfo.samplerate, SEEK_SET) ;
			xmms_sndfile_ip.output->flush (seek_time * 1000) ;
			seek_time = -1 ;
   			} ;

  		} ; /* while (decoding) */

	pthread_exit (NULL) ;
	return NULL ;
} /* play_loop */

static void
play_start (char *filename)
{	int pcmbitwidth ;

	if (sndfile)
		return ;

	pcmbitwidth = 32 ;

	get_song_title (filename, song_title) ;

	if (! (sndfile = sf_open (filename, SFM_READ, &sfinfo)))
		return ;

	bit_rate = sfinfo.samplerate * pcmbitwidth * sfinfo.channels ;

	if (sfinfo.samplerate > 0)
		song_length = (int) ceil (1000.0 * sfinfo.frames / sfinfo.samplerate) ;
	else
		song_length = 0 ;


	if (! xmms_sndfile_ip.output->open_audio (HOST_SIGNED_SHORT, sfinfo.samplerate, sfinfo.channels))
	{	sf_close (sndfile) ;
		sndfile = NULL ;
		return ;
		} ;

	xmms_sndfile_ip.set_info (song_title, song_length, bit_rate, sfinfo.samplerate, sfinfo.channels) ;

	pthread_create (&decode_thread, NULL, play_loop, NULL) ;

	while (decoding == FALSE)
	    xmms_usleep (1000) ;

} /* play_start */

static void
play_stop (void)
{
	decoding = FALSE ;
	pause_now = FALSE ;

	pthread_join (decode_thread, NULL) ;
	xmms_sndfile_ip.output->close_audio () ;

	sf_close (sndfile) ;
	sndfile = NULL ;

	seek_time = -1 ;
} /* play_stop */

static void
do_pause (short p)
{
	if (p)
	    pause_now = TRUE ;
	else
    	pause_now = FALSE ;

	xmms_sndfile_ip.output->pause (p) ;
} /* do_pause */

static void
file_seek (int time)
{
	if (! sfinfo.seekable)
		return ;

	seek_time = time ;

	while (seek_time != -1)
		xmms_usleep (10000) ;
} /* file_seek */

static int
get_time (void)
{
	if ( ! (xmms_sndfile_ip.output->buffer_playing () && decoding))
		return -1 ;

	return xmms_sndfile_ip.output->output_time () ;
} /* get_time */

static void
get_song_info (char *filename, char **title, int *length)
{	char *tmp_title = g_malloc (TITLE_LEN * sizeof (char)) ;

	/*	Possible memory leak!!!
	**	Where does this get free()ed???
	*/

	(*length) = get_song_length (filename) ;

	get_song_title (filename, tmp_title) ;
	(*title) = tmp_title ;

} /* get_song_info */

InputPlugin *
get_iplugin_info (void)
{	return &xmms_sndfile_ip ;
} /* get_iplugin_info */
