// This file is a part of the xMule Project.
//
// Copyright (c) 2004 Theodore R. Smith (hopeseekr@xmule.ws / http://www.xmule.ws/)
// DSA-1024 Fingerprint: 10A0 6372 9092 85A2 BB7F 907B CB8B 654B E33B F1ED
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
/////////////////////////////////////////////////////////////////////////////
// Name:        file.h
// Purpose:     CFile - encapsulates low-level "file descriptor"
//              wxTempFile - safely replace the old file
// Author:      Vadim Zeitlin
// Modified by:
// Created:     29/01/98
// RCS-ID:      $Id: CFile.h,v 1.19 2004/08/28 02:37:37 HopeSeekr Exp $
// Copyright:   (c) 1998 Vadim Zeitlin <zeitlin@dptmaths.ens-cachan.fr>
// Licence:     wxWindows license
/////////////////////////////////////////////////////////////////////////////

#ifndef _WX_CFILEH__
#define _WX_CFILEH__

#if defined(__GNUG__) && !defined(__APPLE__)
#pragma interface "CFile.h"
#endif

#include <wx/filefn.h>                   // Needed for malloc

#include "mfc.h"                            // Needed for CString

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

// we redefine these constants here because S_IREAD &c are _not_ standard
// however, we do assume that the values correspond to the Unix umask bits
#define wxS_IRUSR 00400
#define wxS_IWUSR 00200
#define wxS_IXUSR 00100

#define wxS_IRGRP 00040
#define wxS_IWGRP 00020
#define wxS_IXGRP 00010

#define wxS_IROTH 00004
#define wxS_IWOTH 00002
#define wxS_IXOTH 00001

// default mode for the new files: corresponds to umask 022
#define wxS_DEFAULT   (wxS_IRUSR | wxS_IWUSR | wxS_IRGRP | wxS_IWGRP |\
                       wxS_IROTH | wxS_IWOTH)

// ----------------------------------------------------------------------------
// class CFile: raw file IO
//
// NB: for space efficiency this class has no virtual functions, including
//     dtor which is _not_ virtual, so it shouldn't be used as a base class.
// ----------------------------------------------------------------------------
class CFile
{
public:
  // more file constants
  // -------------------
    // opening mode
  enum OpenMode { read, write, read_write, write_append, write_excl };
    // standard values for file descriptor
  enum { fd_invalid = -1, fd_stdin, fd_stdout, fd_stderr };

  // static functions
  // ----------------
    // check whether a regular file by this name exists
  static bool Exists(const wxChar *name);
    // check whetther we can access the given file in given mode
    // (only read and write make sense here)
  static bool Access(const wxChar *name, OpenMode mode);

  // ctors
  // -----
    // def ctor
  CFile() { m_fd = fd_invalid; }
    // open specified file (may fail, use IsOpened())
  CFile(const wxChar *szFileName, OpenMode mode = read);
    // attach to (already opened) file
  CFile(int fd) { m_fd = fd; }

  wxString GetFilePath() {return fFilePath;}; 

  // open/close
    // create a new file (with the default value of bOverwrite, it will fail if
    // the file already exists, otherwise it will overwrite it and succeed)
  virtual bool Create(const wxChar *szFileName, bool bOverwrite = FALSE,
              int access = wxS_DEFAULT);
  virtual bool Open(const wxChar *szFileName, OpenMode mode = read,
            int access = wxS_DEFAULT);
  bool Open(CString szFileName, OpenMode mode = read,
		    int access = wxS_DEFAULT) {
    return Open(szFileName.GetData(),mode,access);
  };
  virtual bool Close();  // Close is a NOP if not opened

  // assign an existing file descriptor and get it back from CFile object
  void Attach(int fd) { Close(); m_fd = fd; }
  void Detach()       { m_fd = fd_invalid;  }
  int  fd() const { return m_fd; }

  // read/write (unbuffered)
    // returns number of bytes read or ofsInvalid on error
  virtual off_t Read(void *pBuf, off_t nCount);
  virtual off_t Readpos(void *pBuf, long pos,off_t nCount);

    // returns the number of bytes written
  virtual size_t Write(const void *pBuf, size_t nCount);
  virtual size_t Writepos(const void *pBuf,long pos, size_t nCount);
    // returns true on success
  virtual bool Write(const wxString& s, wxMBConv& conv = wxConvLocal)
  {
      const wxWX2MBbuf buf = s.mb_str(conv);
      size_t size = strlen(buf);
      return Write((const char *) buf, size) == size;
  }
    // flush data not yet written
  virtual bool Flush();

  // file pointer operations (return ofsInvalid on failure)
    // move ptr ofs bytes related to start/current off_t/end of file
  virtual off_t Seek(off_t ofs, wxSeekMode mode = wxFromStart);
    // move ptr to ofs bytes before the end
  virtual off_t SeekEnd(off_t ofs = 0) { return Seek(ofs, wxFromEnd); }
    // get current off_t
  virtual off_t Tell() const;
    // get current file length
  virtual off_t Length() const;

  // simple accessors
    // is file opened?
  bool IsOpened() const { return m_fd != fd_invalid; }
    // is end of file reached?
  bool Eof() const;
    // has an error occured?
  bool Error() const { return m_error; }

  // dtor closes the file if opened
  virtual ~CFile() { Close(); }

private:
  // copy ctor and assignment operator are private because
  // it doesn't make sense to copy files this way:
  // attempt to do it will provoke a compile-time error.
  CFile(const CFile&);
  CFile& operator=(const CFile&);

  int m_fd; // file descriptor or INVALID_FD if not opened
  bool m_error; // error memory
  wxString fFilePath;
};



#endif // _WX_FILEH__
