// This file is a part of the xMule Project.
//
// Copyright (c) 2004 Theodore R. Smith (hopeseekr@xmule.ws / http://www.xmule.ws/)
// DSA-1024 Fingerprint: 10A0 6372 9092 85A2 BB7F 907B CB8B 654B E33B F1ED
//
// Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://www.emule-project.net )
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifdef PRECOMP
    #include "xmule-headers.h"
#endif

#include "ClientCredits.h"                  // Needed for this Interface's Prototype

#include "NewFunctions.h"                   // Needed for MAP
#include "opcodes.h"                        // Needed for CREDITFILE_VERSION
#include "otherfunctions.h"                 // Needed for md4cpy
#include "Preferences.h"                    // Needed for CPreferences
#include "resource.h"                       // Needed for IDS_ERR_CREDITFILEOLD
#include "ServerList.h"                     // Needed for CServerList
#include "sockets.h"                        // Needed for CServerConnect
#include "xmule.h"                          // Needed for theApp
#include "xmuleDlg.h"                       // Needed for CxmuleDlg

#include "DynPrefs/DynPrefs.h"              // Needed for theApp.dynprefs

#include <cmath>                            // Needed for std::sqrt
#include <fcntl.h>                          // Needed for open

#if USE_CRYPTO 
    #include <cryptopp/config.h>
    #include <cryptopp/osrng.h>
    #include <cryptopp/base64.h>
    #include <cryptopp/files.h>
    #include <cryptopp/sha.h>
    #define	CryptoXMpp	CryptoPP
#else
    #include "CryptoXMpp.h"                 // Needed for Crypto++
#endif

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;

#define new DEBUG_NEW
#endif

#define		CLIENTS_MET_FILENAME	"clients.met"

MAP * mcc = NULL;
CryptoXMpp::RSASSA_PKCS1v15_SHA_Signer* m_pSignkey;
uint32 ldwForIP;

using namespace CryptoXMpp;

extern int newprefs01_opt[];

CClientCredits::CClientCredits(CreditStruct * in_credits)
{
    m_pCredits = in_credits;
    InitalizeIdent();
    m_dwUnSecureWaitTime = 0;
    m_dwSecureWaitTime = 0;
    m_dwWaitTimeIP = 0;
}

CClientCredits::CClientCredits(uchar * key)
{
    m_pCredits = new CreditStruct;
    memset(m_pCredits, 0, sizeof(CreditStruct));
    md4cpy(m_pCredits->abyKey, key);
    InitalizeIdent();
    m_dwUnSecureWaitTime =::GetTickCount();
    m_dwSecureWaitTime =::GetTickCount();
    m_dwWaitTimeIP = 0;
}

CClientCredits::~ CClientCredits()
{
    delete m_pCredits;
}

void CClientCredits::AddDownloaded(uint32 bytes, uint32 dwForIP)
{
    if ((GetCurrentIdentState(dwForIP) == IS_IDFAILED || GetCurrentIdentState(dwForIP) == IS_IDBADGUY ||
    GetCurrentIdentState(dwForIP) == IS_IDNEEDED) && theApp. clientcredits->CryptoAvailable())
    {
        return;
    }
    //encode
    uint64 current = m_pCredits->nDownloadedHi;
    current = (current << 32) + m_pCredits->nDownloadedLo + bytes;
    //recode
    m_pCredits->nDownloadedLo = (uint32) current;
    m_pCredits->nDownloadedHi = (uint32)(current >> 32);
}

void CClientCredits::ShowCHandPK(uint32 challenge, unsigned char * pk, int length)
{
    printf("    credits=%u  challenge=%u  length=%d  publickey=", (uint32) this, challenge, length);
    for (int i = 0 ; i < length ; i++)
    {
        if (i < 5 || i > length - 6)
        {
            printf("%02x", pk[i]);
        }
    }
    printf("\n");
}

void CClientCredits::AddUploaded(uint32 bytes, uint32 dwForIP)
{
    if ((GetCurrentIdentState(dwForIP) == IS_IDFAILED || GetCurrentIdentState(dwForIP) == IS_IDBADGUY || GetCurrentIdentState(dwForIP) == IS_IDNEEDED) && theApp. clientcredits->CryptoAvailable())
    {
        return;
    }
    else
    {
        uint64 current = m_pCredits->nUploadedHi;
        current = (current << 32) + m_pCredits->nUploadedLo + bytes;
        m_pCredits->nUploadedLo = (uint32) current;
        m_pCredits->nUploadedHi = (uint32)(current >> 32);
    }
}

uint64 CClientCredits::GetUploadedTotal()
{
    return((uint64) m_pCredits->nUploadedHi << 32) + m_pCredits->nUploadedLo;
}

uint64 CClientCredits::GetDownloadedTotal()
{
    return((uint64) m_pCredits->nDownloadedHi << 32) + m_pCredits->nDownloadedLo;
}

float CClientCredits::GetScoreRatio(wxUint32 dwForIP)
{
    // Check the client ident status
    if ((GetCurrentIdentState(dwForIP) == IS_IDFAILED || GetCurrentIdentState(dwForIP) == IS_IDNEEDED) &&
    theApp. clientcredits->CryptoAvailable() == true)
    {
        // bad guy - no credits for you
        return 1;
    }
    else
    {
        // Cache value
        const uint64 downloadTotal = GetDownloadedTotal();
        // Check if this client has any credit (sent >1MB)
        if (downloadTotal < 1000000)
        {
            return 1.0f;
        }
        else
        {
            // Cache value
            const uint64 uploadTotal = GetUploadedTotal();
            // Factor 1
            float result = (uploadTotal == 0) ?
            10.0f: (float)(2 * downloadTotal) / (float) uploadTotal;
            // Factor 2
            float trunk = (float) sqrt(2.0 + (double) downloadTotal/ 1048576.0);
            if (result > trunk)
            {
                result = trunk;
            }
            // Trunk final result 1..10
            if (result < 1.0f)
            {
                return 1.0f;
            }
            else if(result > 10.0f)
            {
                return 10.0f;
            }
            else
            {
                return result;
            }
        }
    }
}

// Maella end

// added by sivka [VQB: -ownCredits-]
float CClientCredits::GetMyScoreRatio(uint32 dwForIP)
{
    // check the client ident status
    if ((GetCurrentIdentState(dwForIP) == IS_IDFAILED || GetCurrentIdentState(dwForIP) == IS_IDNEEDED) && theApp. clientcredits->CryptoAvailable())
    {
        // bad guy - no credits for... me?
        return 1;
    }
    if (GetUploadedTotal() < 1000000)
    return 1;
    float result = 0;
    if (!GetDownloadedTotal())
    {
        result = 10;
    }
    else
    {
        result = (float)(((double) GetUploadedTotal() * 2.0) / (double) GetDownloadedTotal());
    }
    float result2 = 0;
    result2 = (float) GetUploadedTotal() / 1048576.0;
    result2 += 2;
    result2 = (double) sqrt((double) result2);
    if (result > result2)
    result = result2;
    if (result < 1)
    return 1;
    else if(result > 10)
    return 10;
    return result;
}

CClientCreditsList::CClientCreditsList(CPreferences * in_prefs)
{
    MapData_Init(mcc);
    m_pAppPrefs = in_prefs;
    m_nLastSaved =::GetTickCount();
    LoadList();
    InitalizeCrypting();
    ldwForIP = 0;
}

CClientCreditsList::~ CClientCreditsList()
{
    SaveList();
    MapData_RemoveAll(mcc);
    if (m_pSignkey)
    {
        delete m_pSignkey;
        m_pSignkey = NULL;
    }
}

// VQB: ownCredits // Athlazan

void CClientCreditsList::LoadList()
{
    wxString strFileName = m_pAppPrefs->GetAppDir() + wxString(CLIENTS_MET_FILENAME);
    int loadlist = open(strFileName, O_RDONLY);
    if (loadlist != - 1)
    {
        uint8 version;
        (void) read(loadlist, & version, 1);
        if (version != CREDITFILE_VERSION && version != CREDITFILE_VERSION_29)
        {
            theApp. xmuledlg->AddLogLine(false, GetResString(IDS_ERR_CREDITFILEOLD));
            (void) close(loadlist);
        }
        else
        {
            struct stat loadlistbuf;
            fstat(loadlist, & loadlistbuf);
            wxString strBakFileName = m_pAppPrefs->GetAppDir() + wxString(CLIENTS_MET_FILENAME) + ".BAK";
            BOOL bCreateBackup = TRUE;
            int hBakFile = open(strBakFileName, O_RDONLY);
            if (hBakFile != ( - 1))
            {
                struct stat hBakFilebuf;
                fstat(hBakFile, & hBakFilebuf);
                if (hBakFilebuf. st_size > loadlistbuf. st_size)
                {
                    bCreateBackup = FALSE;
                }
                (void) close(hBakFile);
            }
            if (bCreateBackup)
            {
                (void) close(loadlist);
                if (!SafeCopyFile(strFileName, strBakFileName))
                {
                    printf("info: Could not create backup file '%s'\n", strFileName. c_str());
                }
                int loadlist = open(strFileName, O_RDONLY);
                if (loadlist)
                {
                    (void) read(loadlist, & version, 1);
                }
            }
            uint32 count;
            (void) read(loadlist, & count, 4);
            // TODO: should be prime number... and 20% larger
            const uint32 dwExpired = time(NULL) - 12960000;
            // today - 150 day
            uint32 cDeleted = 0;
            for (uint32 i = 0 ; i < count ; i++)
            {
                CreditStruct * newcstruct = new CreditStruct;
                memset(newcstruct, 0, sizeof(CreditStruct));
                if (version == CREDITFILE_VERSION_29)
                {
                    (void) read(loadlist, newcstruct, sizeof(CreditStruct_29a));
                }
                else
                {
                    (void) read(loadlist, newcstruct, sizeof(CreditStruct));
                }
                if (newcstruct->nLastSeen < dwExpired)
                {
                    cDeleted++;
                    delete newcstruct;
                    continue;
                }
                CClientCredits * newcredits = new CClientCredits(newcstruct);
                MapData_Insert(mcc, newcredits, newcredits->GetKey(), 16);
            }
            (void) close(loadlist);
        }
    }
    else
    {
        theApp. xmuledlg->AddLogLine(true, GetResString(IDS_ERR_LOADCREDITFILE));
    }
}

void CClientCreditsList::SaveList()
{
    FILE * savelist;
    m_nLastSaved =::GetTickCount();
    wxString name = m_pAppPrefs->GetAppDir() + wxString(CLIENTS_MET_FILENAME);
    savelist = fopen((const char *) name, "w");
    if (savelist)
    {
        BYTE * pBuffer = new BYTE[MapData_GetCount(mcc) * sizeof(CreditStruct) ];
        wxUint32 count = 0;
        CClientCredits * cur_client;
        MAP * pos;
        pos = mcc;
        do
        {
            pos = MapData_GetNext(pos);
            if (pos)
            {
                cur_client = (CClientCredits *) MapData_GetIt(pos);
                if (cur_client)
                {
                    if (cur_client->GetUploadedTotal() || cur_client->GetDownloadedTotal())
                    {
                        memcpy(pBuffer + (count * sizeof(CreditStruct)), cur_client->GetDataStruct(),
                        sizeof(CreditStruct));
                        count++;
                    }
                }
            }
        }
        while (pos);
        wxUint8 version = CREDITFILE_VERSION;
        (void) fwrite( & version, 1, 1, savelist);
        (void) fwrite( & count, sizeof(count), 1, savelist);
        (void) fwrite(pBuffer, sizeof(CreditStruct), count, savelist);
        (void) fflush(savelist);
        (void) fclose(savelist);
        delete[] pBuffer;
    }
}

CClientCredits * CClientCreditsList::GetCredit(uchar * key)
{
    CClientCredits * result;
    MAP * pos;
    pos = MapData_Find(mcc, key, 16);
    if (pos)
    {
        result = (CClientCredits *) MapData_GetIt(pos);
    }
    else
    {
        result = new CClientCredits(key);
        MapData_Insert(mcc, result, key, 16);
    }
    result->SetLastSeen();
    return result;
}

void CClientCreditsList::Process()
{
    if (::GetTickCount() - m_nLastSaved > (13 * 60000))
    {
        SaveList();
    }
}

void CClientCredits::InitalizeIdent()
{
    if (m_pCredits->nKeySize == 0)
    {
        memset(m_abyPublicKey, 0, 80);
        // for debugging
        m_nPublicKeyLen = 0;
        IdentState = IS_NOTAVAILABLE;
    }
    else
    {
        m_nPublicKeyLen = m_pCredits->nKeySize;
        memcpy(m_abyPublicKey, m_pCredits->abySecureIdent, m_nPublicKeyLen);
        IdentState = IS_IDNEEDED;
    }
    m_dwCryptRndChallengeFor = 0;
    m_dwCryptRndChallengeFrom = 0;
    m_dwIdentIP = 0;
}

void CClientCredits::Verified(uint32 dwForIP)
{
    m_dwIdentIP = dwForIP;
    // client was verified, copy the keyto store him if not done already
    if (m_pCredits->nKeySize == 0)
    {
        m_pCredits->nKeySize = m_nPublicKeyLen;
        memcpy(m_pCredits->abySecureIdent, m_abyPublicKey, m_nPublicKeyLen);
        if (GetDownloadedTotal() > 0)
        {
            // for security reason, we have to delete all prior credits here
            m_pCredits->nDownloadedHi = 0;
            m_pCredits->nDownloadedLo = 1;
            m_pCredits->nUploadedHi = 0;
            m_pCredits->nUploadedLo = 1;
            // in order to safe this client, set 1 byte
        }
    }
    IdentState = IS_IDENTIFIED;
}

bool CClientCredits::SetSecureIdent(uchar * pachIdent, uint8 nIdentLen)
{
    // verified Public key cannot change, use only if there is not public key yet
    if (MAXPUBKEYSIZE < nIdentLen || m_pCredits->nKeySize != 0)
    {
        return false;
    }
    else
    {
        memcpy(m_abyPublicKey, pachIdent, nIdentLen);
        m_nPublicKeyLen = nIdentLen;
        IdentState = IS_IDNEEDED;
        return true;
    }
}

EIdentState CClientCredits::GetCurrentIdentState(uint32 dwForIP)
{
    if (ldwForIP != dwForIP)
    {
        ldwForIP = dwForIP;
    }
    if (IdentState != IS_IDENTIFIED)
    {
        return IdentState;
    }
    else if(dwForIP == m_dwIdentIP)
    {
        return IS_IDENTIFIED;
    }
    else
    {
        return IS_IDBADGUY;
        // mod note: clients which just reconnected after an IP change and have to ident yet will also have this state for 1-2 seconds
        // so don't try to spam such clients with "bad guy" messages (besides: spam messages are always bad)
    }
}

void CClientCreditsList::InitalizeCrypting()
{
    m_nMyPublicKeyLen = 0;
    memset(m_abyMyPublicKey, 0, 80);
    // not really needed; better for debugging tho
    m_pSignkey = NULL;
    if (m_pAppPrefs->IsSecureIdentEnabled())
    {
        bool bCreateNewKey = false;
        if (theApp.dynprefs->Get<bool>("renew-priv-key"))
        {
            printf("It's obvious I want to create some type of key...\n");
            bCreateNewKey = true;
        }
        else
        {
            FILE * hKeyFile = NULL;
            hKeyFile = fopen((const char *)(m_pAppPrefs->GetAppDir() + wxString("cryptkey.dat")), "r");
            if (hKeyFile != NULL)
            {
                (void) fclose(hKeyFile);
            }
            else
            {
                bCreateNewKey = true;
            }
        }
        if (bCreateNewKey)
        {
            //printf("Creating new Cryptkey.dat ... %08x %08x\n", time(NULL), (unsigned char) time(NULL));
            for (int i = 0 ; i < (unsigned char) time(NULL) ; i++)
            {
                (void) rand();
            }
            CreateKeyPair();
        }
        char erg[500];
        sprintf(erg, "%s/.xMule/%s", getenv("HOME"), "cryptkey.dat");
        FileSource filesource(erg, true, new Base64Decoder);
        m_pSignkey = new RSASSA_PKCS1v15_SHA_Signer(filesource);
        // calculate and store public key
        RSASSA_PKCS1v15_SHA_Verifier pubkey( * m_pSignkey);
        ArraySink asink(m_abyMyPublicKey, 80);
        pubkey. DEREncode(asink);
        m_nMyPublicKeyLen = asink. TotalPutLength();
        asink. MessageEnd();
    }
}

void CClientCreditsList::CreateKeyPair()
{
    AutoSeededRandomPool rng;
    InvertibleRSAFunction privkey;
    privkey. Initialize(rng, 384);
    Base64Encoder privkeysink(new FileSink(CString(m_pAppPrefs->GetAppDir()) + "cryptkey.dat"));
    privkey. DEREncode(privkeysink);
    privkeysink. MessageEnd();
}

uint8 CClientCreditsList::CreateSignature(CClientCredits * pTarget, uchar * pachOutput, uint8 nMaxSize, uint32 ChallengeIP, uint8 byChaIPKind)
{
    RSASSA_PKCS1v15_SHA_Signer * sigkey = NULL;
    StringSource ss_Pubkey((byte *) pTarget->GetSecureIdent(), pTarget->GetSecIDKeyLen(), true, 0);
    RSASSA_PKCS1v15_SHA_Verifier pubkey(ss_Pubkey);
    sigkey = m_pSignkey;
    // create a signature of the public key from pTarget
    SecByteBlock sbbSignature(sigkey->SignatureLength());
    AutoSeededRandomPool rng;
    byte abyBuffer[MAXPUBKEYSIZE + 9];
    uint32 keylen = pTarget->GetSecIDKeyLen();
    memcpy(abyBuffer, pTarget->GetSecureIdent(), keylen);
    // 4 additional bytes random data send from this client
    uint32 challenge = pTarget->m_dwCryptRndChallengeFrom;
    memcpy(abyBuffer + keylen, & challenge, 4);
    uint16 ChIpLen = 4;
    if (byChaIPKind != 0)
    {
        memcpy(abyBuffer + keylen + 4, & ChallengeIP, 4);
        memcpy(abyBuffer + keylen + 4 + 4, & byChaIPKind, 1);
        ChIpLen += 5;
    }
    for (unsigned int i = 0; i < sbbSignature.size(); i++)
    {
        sbbSignature. begin() [i] = 0;
    }
    sigkey->SignMessage(rng, abyBuffer, keylen + ChIpLen, sbbSignature. begin());
    if (nMaxSize > sbbSignature. size())
    {
        nMaxSize = sbbSignature. size();
    }
    for (int i = 0 ; i < nMaxSize ; i++)
    {
        pachOutput[i] = ((unsigned char *) sbbSignature. begin()) [i];
    }
    return(uint8) nMaxSize;
}

bool CClientCreditsList::VerifyIdent(CClientCredits * pTarget, uchar * pachSignature, uint8 nInputSize, uint32 dwForIP, uint8 byChaIPKind)
{
    bool bResult;
    StringSource ss_Pubkey((byte *) pTarget->GetSecureIdent(), pTarget->GetSecIDKeyLen(), true, 0);
    RSASSA_PKCS1v15_SHA_Verifier pubkey(ss_Pubkey);
    // 4 additional bytes random data send from this client +5 bytes v2
    byte abyBuffer[MAXPUBKEYSIZE + 9];
    memcpy(abyBuffer, m_abyMyPublicKey, m_nMyPublicKeyLen);
    uint32 challenge = pTarget->m_dwCryptRndChallengeFor;
    memcpy(abyBuffer + m_nMyPublicKeyLen, & challenge, 4);
    // v2 security improvments (not supported by 29b, not used as default by 29c)
    uint8 nChIpSize = 0;
    if (byChaIPKind != 0)
    {
        nChIpSize = 5;
        uint32 ChallengeIP = 0;
        switch (byChaIPKind)
        {
        case CRYPT_CIP_LOCALCLIENT:
            ChallengeIP = dwForIP;
            break;
        case CRYPT_CIP_REMOTECLIENT:
            if ((theApp. serverconnect->GetClientID() == 0) || theApp. serverconnect->IsLowID())
            {
                ChallengeIP = theApp. serverconnect->GetLocalIP();
            }
            else
            {
                ChallengeIP = theApp. serverconnect->GetClientID();
            }
            break;
            // maybe not supported in future versions:
        case CRYPT_CIP_NONECLIENT:
            ChallengeIP = 0;
            break;
        }
        memcpy(abyBuffer + m_nMyPublicKeyLen + 4, & ChallengeIP, 4);
        memcpy(abyBuffer + m_nMyPublicKeyLen + 4 + 4, & byChaIPKind, 1);
    }
    //v2 end
    bResult = pubkey. VerifyMessage(abyBuffer, m_nMyPublicKeyLen + 4 + nChIpSize, pachSignature, nInputSize);
    if (!bResult)
    {
        if (pTarget->IdentState == IS_IDNEEDED)
        pTarget->IdentState = IS_IDFAILED;
    }
    else
    {
        pTarget->Verified(dwForIP);
    }
    return bResult;
}

bool CClientCreditsList::CryptoAvailable()
{
    return(m_nMyPublicKeyLen > 0 && m_pSignkey != 0 && m_pAppPrefs->IsSecureIdentEnabled());
}

uint32 CClientCredits::GetSecureWaitStartTime(uint32 dwForIP)
{
    if (m_dwUnSecureWaitTime == 0 || m_dwSecureWaitTime == 0)
    SetSecWaitStartTime(dwForIP);
    if (m_pCredits->nKeySize != 0)
    {
        // this client is a SecureHash Client
        if (GetCurrentIdentState(dwForIP) == IS_IDENTIFIED)
        {
            // good boy
            return m_dwSecureWaitTime;
        }
        else
        {
            // not so good boy
            if (dwForIP == m_dwWaitTimeIP)
            {
                return m_dwUnSecureWaitTime;
            }
            else
            {
                // bad boy
                // this can also happen if the client has not identified himself yet, but will do later - so maybe he is not a bad boy :) .
                CString buffer2, buffer;
                m_dwUnSecureWaitTime =::GetTickCount();
                m_dwWaitTimeIP = dwForIP;
                return m_dwUnSecureWaitTime;
            }
        }
    }
    else
    {
        // not a SecureHash Client - handle it like before for now (no security checks)
        return m_dwUnSecureWaitTime;
    }
}

void CClientCredits::SetSecWaitStartTime(uint32 dwForIP)
{
    m_dwUnSecureWaitTime =::GetTickCount() - 1;
    m_dwSecureWaitTime =::GetTickCount() - 1;
    m_dwWaitTimeIP = dwForIP;
}

void CClientCredits::ClearWaitStartTime()
{
    m_dwUnSecureWaitTime = 0;
    m_dwSecureWaitTime = 0;
}

