// This file is a part of the xMule Project.
//
// Copyright (c) 2004 Theodore R. Smith (hopeseekr@xmule.ws / http://www.xmule.ws/)
// DSA-1024 Fingerprint: 10A0 6372 9092 85A2 BB7F 907B CB8B 654B E33B F1ED
//
// Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://www.emule-project.net )
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

//#include "config.h"

// Test if we have _GNU_SOURCE before the next step will mess up
// setting __USE_GNU
// (only needed for gcc-2.95 compatibility, gcc 3.2 always defines it)
//#include "wx/setup.h"

#ifdef PRECOMP
    #include "xmule-headers.h"
#endif

#include "PartFile.h"                       // Needed for this Interface's Prototype

#include "AddFileThread.h"                  // Needed for CAddFileThread
#include "BarShader.h"                      // Needed for CBarShader
#include "DownloadListCtrl.h"               // Needed for CDownloadListCtrl
#include "DownloadQueue.h"                  // Needed for CDownloadQueue
#include "ED2KLink.h"                       // Needed for CED2KLink
#include "KnownFileList.h"                  // Needed for CKnownFileList
#include "ListenSocket.h"                   // Needed for CClientReqServer
#include "otherstructs.h"                   // Needed for Gap_Struct
#include "packets.h"                        // Needed for CTag
#include "Preferences.h"                    // Needed for CPreferences
#include "SafeFile.h"                       // Needed for CSafeFile
#include "SearchList.h"                     // Needed for CSearchFile
#include "server.h"                         // Needed for CServer
#include "sockets.h"                        // Needed for CServerConnect
#include "SharedFileList.h"                 // Needed for CSharedFileList
#include "SysTray.h"                        // Needed for TBN_DLOAD
#include "TransferWnd.h"                    // Needed for CTransferWnd
#include "updownclient.h"                   // Needed for DS_ONQUEUE
#include "UploadQueue.h"                    // Needed for CUploadQueue
#include "xmule.h"                          // Needed for theApp
#include "xmuleDlg.h"                       // Needed for theApp.xmuledlg

#include "DynPrefs/DynPrefs.h"              // Needed for theApp.dynprefs

// Mario Sergio Fujikawa Ferreira <lioux@FreeBSD.org>
// to detect if this is a *BSD system
#if defined(HAVE_SYS_PARAM_H)
    #include <sys/param.h>                  // Needed for *BSD systems, apparently.
#endif

#include <arpa/inet.h>                      // Needed for inet_addr
#include <cmath>                            // Needed for floor
#include <sys/stat.h>                       // Needed for stat

#include <wx/dcmemory.h>                    // Needed for wxMemoryDC
#include <wx/filename.h>                    // Needed for wxFileName

#define min(a,b) ((a)<(b)?(a):(b))

void dump16f(FILE *f, uchar *d16)
{
    int i;
    for (i = 0 ; i < 16 ; i++)
    {
        fprintf(f, "%02X", *d16++);
    }
}

void dump16(uchar *d16)
{
    int i;
    for (i = 0 ; i < 16 ; i++)
    {
        printf("%02X", *d16++);
    }
}

extern void dump16(uchar *);

CBarShader CPartFile::s_LoadBar(3);

CBarShader CPartFile::s_ChunkBar(16);

CPartFile::CPartFile()
{
    Init();
}

CPartFile::CPartFile(CSearchFile *searchresult)
{
    Init();
    memcpy(m_abyFileHash, searchresult->GetFileHash(), 16);
    for (int i = 0 ; i != searchresult->taglist.GetCount() ; i++)
    {
        switch (searchresult->taglist[i]->tag->specialtag)
        {
        case FT_FILENAME:
            {
                m_pszFileName = nstrdup(searchresult->taglist[i]->tag->stringvalue);
                break;
            }
        case FT_FILESIZE:
            {
                m_nFileSize = searchresult->taglist[i]->tag->intvalue;
                break;
            }
        default:
            CTag *newtag = new CTag(searchresult->taglist[i]->tag);
            taglist.Add(newtag);
        }
    }
    CreatePartFile();
}

CPartFile::CPartFile(CString edonkeylink)
{
    CED2KLink *pLink = 0;
    try
    {
        pLink = CED2KLink::CreateLinkFromUrl(edonkeylink);
        CED2KFileLink *pFileLink = pLink->GetFileLink();
        if (pFileLink == 0)
        throw GetResString(IDS_ERR_NOTAFILELINK);
        InitializeFromLink(pFileLink);
    }
    catch(wxString error)
    {
        char buffer[200];
        sprintf(buffer, GetResString(IDS_ERR_INVALIDLINK), error.GetData());
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_LINKERROR), buffer);
        SetPartFileStatus(PS_ERROR);
    }
    delete pLink;
}

void
CPartFile::InitializeFromLink(CED2KFileLink *fileLink)
{
    Init();
    m_pszFileName = nstrdup(fileLink->GetName());
    m_nFileSize = fileLink->GetSize();
    memcpy(m_abyFileHash, fileLink->GetHashKey(), sizeof(m_abyFileHash));
    if (!theApp.downloadqueue->IsFileExisting(m_abyFileHash))
    CreatePartFile();
    else
    SetPartFileStatus(PS_ERROR);
}

CPartFile::CPartFile(CED2KFileLink *fileLink)
{
    InitializeFromLink(fileLink);
}

void CPartFile::Init()
{
    m_hpartfile = new CFile();
    fullname = 0;
    newdate = true;
    lastsearchtime = 0;
    lastpurgetime =::GetTickCount();
    paused = false;
    stopped = false;
    SetPartFileStatus(PS_EMPTY);
    transfered = 0;
    if (theApp.glob_prefs->GetNewAutoDown())
    {
        m_iDownPriority = PR_HIGH;
        m_bAutoDownPriority = true;
    }
    else
    {
        m_iDownPriority = PR_NORMAL;
        m_bAutoDownPriority = false;
    }
    srcarevisible = false;
    transferingsrc = 0;
    datarate = 0;
    hashsetneeded = true;
    count = 0;
    percentcompleted = 0;
    partmetfilename = 0;
    completedsize = 0;
    m_bPreviewing = false;
    //NULL;:
    lastseencomplete = 0;
    availablePartsCount = 0;
    m_ClientSrcAnswered = 0;
    m_LastNoNeededCheck = 0;
    m_iRate = 0;
    m_strComment = "";
    m_nTotalBufferData = 0;
    m_nLastBufferFlushTime = 0;
    m_bPercentUpdated = false;
    m_bRecoveringArchive = false;
    m_iGainDueToCompression = 0;
    m_iLostDueToCorruption = 0;
    m_iTotalPacketsSavedDueToICH = 0;
    m_nSavedReduceDownload = 0;
    hasRating = false;
    hasComment = false;
    m_IsA4AFAuto = false;
    /* Razor 1a - Modif by MikaelB */
    m_lastdatetimecheck = 0;
    m_category = 0;
    m_lastRefreshedDLDisplay = 0;
    m_LastSourceDropTime = 0;
}

CPartFile::~CPartFile()
{
    // Barry - Ensure all buffered data is written
    FlushBuffer();
    if (fullname)
    delete[] fullname;
    if (partmetfilename)
    delete[] partmetfilename;
    m_SrcpartFrequency.RemoveAll();
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
    delete gaplist.GetAt(pos);
}

void CPartFile::CreatePartFile()
{
    // use lowest free partfilenumber for free file (InterCeptor)
    int i = 0;
    CString filename;
    do
    {
        i++;
        if (theApp.dynprefs->Get<bool>("enable-dynamic-part"))
        {
            filename.Format("%s/%03i.pnew", theApp.glob_prefs->GetTempDir(), i);
        }
        else
        {
            filename.Format("%s/%03i.part", theApp.glob_prefs->GetTempDir(), i);
        }
    }
    while (wxFileName::FileExists(filename));
    partmetfilename = new char[15];
    if (theApp.dynprefs->Get<bool>("enable-dynamic-part"))
    {
        sprintf(partmetfilename, "%03i.pnew.met", i);
    }
    else
    {
        sprintf(partmetfilename, "%03i.part.met", i);
    }
    fullname = new char[strlen(theApp.glob_prefs->GetTempDir()) + strlen(partmetfilename) + MAX_PATH];
    sprintf(fullname, "%s/%s", theApp.glob_prefs->GetTempDir(), partmetfilename);
    char *buffer = nstrdup(partmetfilename);
    buffer[strlen(buffer) - 4] = 0;
    CTag *partnametag = new CTag(FT_PARTFILENAME, buffer);
    delete[] buffer;
    taglist.Add(partnametag);
    Gap_Struct *gap = new Gap_Struct;
    gap->start = 0;
    gap->end = m_nFileSize - 1;
    gaplist.AddTail(gap);
    char *partfull = nstrdup(fullname);
    partfull[strlen(partfull) - 4] = 0;
    if (!m_hpartfile->Create(partfull, TRUE))
    {
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_CREATEPARTFILE));
        SetPartFileStatus(PS_ERROR);
    }
    // jesh.. luotu. nyt se vaan pit avata uudestaan read-writeen..
    m_hpartfile->Close();
    if (!m_hpartfile->Open(partfull, CFile::read_write))
    {
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_CREATEPARTFILE));
        SetPartFileStatus(PS_ERROR);
    }
    delete[] partfull;
    m_SrcpartFrequency.SetSize(GetPartCount());
    for (wxUint32 i = 0 ; i != GetPartCount() ; i++)
    {
        m_SrcpartFrequency.Add((wxUint16) 0);
    }
    paused = false;
    SavePartFile(true);
}

bool CPartFile::LoadPartFile(char *in_directory, char *in_filename)
{
    // Slugfiller:
    CMap < wxUint16, wxUint16, Gap_Struct *, Gap_Struct *> gap_map;
    transfered = 0;
    partmetfilename = nstrdup(in_filename);
    directory = nstrdup(in_directory);
    char *buffer = new char[strlen(directory) + strlen(partmetfilename) + 2];
    sprintf(buffer, "%s/%s", directory, partmetfilename);
    fullname = buffer;
    CSafeFile file;
    {
        // readfile data form part.met file
        if (!file.Open(fullname, CFile::read))
        {
            theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_OPENMET), partmetfilename, m_pszFileName);
            return false;
        }
        wxUint8 version = 0;
        file.Read( &version, 1);
        if (version != PARTFILE_VERSION)
        {
            file.Close();
            theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_BADMETVERSION), partmetfilename, m_pszFileName);
            return false;
        }
        LoadDateFromFile( &file);
        LoadHashsetFromFile( &file, false);
 /* We need to forge some values here for later error
 * detection to work.
    */
        wxUint32 tagcount = 0;
        wxUint32 j = 0;
        try
        {
            if (4 != file.Read( &tagcount, 4))
            {
                tagcount = 0;
                throw CInvalidPacket("short file reading tagcount");
            }
            for (j = 0 ; j < tagcount ; j++)
            {
                try
                {
                    CTag *newtag = new CTag( &file);
                    switch (newtag->tag->specialtag)
                    {
                    case FT_FILENAME:
                        {
                            if (newtag->tag->stringvalue == NULL)
                            {
                                theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_METCORRUPT), partmetfilename, m_pszFileName);
                                delete newtag;
                                return false;
                            }
                            m_pszFileName = nstrdup(newtag->tag->stringvalue);
                            delete newtag;
                            break;
                        }
                    case FT_LASTSEENCOMPLETE:
                        {
                            lastseencomplete = newtag->tag->intvalue;
                            delete newtag;
                            break;
                        }
                    case FT_FILESIZE:
                        {
                            m_nFileSize = newtag->tag->intvalue;
                            delete newtag;
                            break;
                        }
                    case FT_TRANSFERED:
                        {
                            transfered = newtag->tag->intvalue;
                            delete newtag;
                            break;
                        }
                    case FT_CATEGORY:
                        {
                            m_category = newtag->tag->intvalue;
                            delete newtag;
                            break;
                        }
                    case FT_OLDDLPRIORITY:
                    case FT_DLPRIORITY:
                        {
                            m_iDownPriority = newtag->tag->intvalue;
                            delete newtag;
                            if (m_iDownPriority == PR_AUTO)
                            {
                                m_iDownPriority = PR_HIGH;
                                SetAutoDownPriority(true);
                            }
                            else
                            SetAutoDownPriority(false);
                            break;
                        }
                    case FT_STATUS:
                        {
                            paused = newtag->tag->intvalue;
                            stopped = paused;
                            delete newtag;
                            break;
                        }
                    case FT_OLDULPRIORITY:
                    case FT_ULPRIORITY:
                        {
                            SetUpPriority(newtag->tag->intvalue, false);
                            delete newtag;
                            if (GetUpPriority() == PR_AUTO)
                            {
                                SetUpPriority(PR_HIGH, false);
                                SetAutoUpPriority(true);
                            }
                            else
                            SetAutoUpPriority(false);
                            break;
                        }
                    default:
                        {
                            // Start Changes by Slugfiller for better exception handling
                            if ((!newtag->tag->specialtag) &&
                            (newtag->tag->tagname[0] == FT_GAPSTART ||
                            newtag->tag->tagname[0] == FT_GAPEND))
                            {
                                Gap_Struct *gap;
                                wxUint16 gapkey = atoi( &newtag->tag->tagname[1]);
                                if (!gap_map.Lookup(gapkey, gap))
                                {
                                    gap = new Gap_Struct;
                                    gap_map.SetAt(gapkey, gap);
                                    gap->start = (wxUint32) - 1;
                                    gap->end = (wxUint32) - 1;
                                }
                                if (newtag->tag->tagname[0] == FT_GAPSTART)
                                gap->start = newtag->tag->intvalue;
                                if (newtag->tag->tagname[0] == FT_GAPEND)
                                gap->end = newtag->tag->intvalue - 1;
                                delete newtag;
                                // End Changes by Slugfiller for better exception handling
                            }
                            else
                            taglist.Add(newtag);
                        }
                    }
                }
                catch(CStrangePacket)
                {
                }
            }
        }
        catch(CInvalidPacket e)
        {
 /*
 * When failing on the last tag, this might be
 * due to a failure in xMule writing the file.
    */
            if (!tagcount || (j != tagcount - 1))
            {
                printf("failure reading part file - %s\n", e.what());
                file.Close();
                return false;
            }
        }
        file.Close();
    }
    // Now to flush the map into the list (Slugfiller)
    for (POSITION pos = gap_map.GetStartPosition() ; pos != NULL ;)
    {
        Gap_Struct *gap;
        wxUint16 gapkey;
        gap_map.GetNextAssoc(pos, gapkey, gap);
        if (gap->start >= 0 &&gap->end >= 0 &&gap->start <= gap->end)
        // All tags accounted for:
        gaplist.AddTail(gap);
        else
        // Some of the tags were missing:
        delete gap;
    }
    //check if this is a backup
    if (strcasecmp(strrchr(fullname, '.'), ".backup") == 0)
    {
        char *shorten = strrchr(fullname, '.');
        *shorten = 0;
        char *oldfullname = fullname;
        fullname = new char[strlen(fullname) + 1];
        strcpy(fullname, oldfullname);
        delete[] oldfullname;
    }
    // open permanent handle
    char *searchpath = nstrdup(fullname);
    searchpath[strlen(fullname) - 4] = 0;
    if (!m_hpartfile->Open(searchpath, CFile::read_write))
    {
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_FILEOPEN), fullname, m_pszFileName);
        delete[] searchpath;
        return false;
    }
    delete[] searchpath;
    searchpath = NULL;
    m_SrcpartFrequency.SetSize(GetPartCount());
    for (wxUint32 i = 0 ; i != GetPartCount() ; i++)
    {
        m_SrcpartFrequency.Add(0);
    }
    SetPartFileStatus(PS_EMPTY);
    if (hashlist.GetCount() < GetPartCount())
    {
        hashsetneeded = true;
        return true;
    }
    else
    {
        hashsetneeded = false;
        for (int i = 0 ; i != hashlist.GetSize() ; i++)
        {
            if (IsComplete(i *PARTSIZE, ((i + 1) *PARTSIZE) - 1))
            {
                SetPartFileStatus(PS_READY);
            }
        }
    }
    if (gaplist.IsEmpty())
    {
        CompleteFile(false);
        return true;
    }
    struct stat statbuf;
    fstat(m_hpartfile->fd(), &statbuf);
    if ((time_t) date != (time_t) statbuf.st_mtime)
    {
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_REHASH), buffer, m_pszFileName);
        // rehash
        SetPartFileStatus(PS_WAITINGFORHASH);
        char *partfilename = nstrdup(partmetfilename);
        partfilename[strlen(partfilename) - 4] = 0;
        CAddFileThread::AddFile(directory, partfilename, this);
        delete[] partfilename;
    }
    UpdateCompletedInfos();
    if (completedsize > transfered)
    m_iGainDueToCompression = completedsize - transfered;
    else if(completedsize != transfered)
    m_iLostDueToCorruption = transfered - completedsize;
    return true;
    return true;
}

bool CPartFile::SavePartFile(bool Initial)
{
    switch (status)
    {
    case PS_WAITINGFORHASH:
    case PS_HASHING:
        return false;
    }
    /* Don't write anything to disk if less than 5000 bytes of free space is left. */
    wxLongLong total, free;
    if (wxGetDiskSpace(theApp.glob_prefs->GetTempDir(), &total, &free) &&
    free < 5000)
    {
        return false;
    }
    FILE *file = 0;
    char *searchpath = nstrdup(fullname);
    searchpath[strlen(fullname) - 4] = 0;
    wxString fName =::wxFindFirstFile(searchpath, wxFILE);
    delete[] searchpath;
    searchpath = NULL;
    if (fName.IsEmpty())
    {
        if (file)
        fclose(file);
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_SAVEMET), GetResString(IDS_ERR_PART_FNF) .GetData(), partmetfilename, m_pszFileName);
        return false;
    }
    struct stat sbf;
    stat(fName.GetData(), &sbf);
    date = sbf.st_mtime;
    wxUint32 lsc = lastseencomplete;
    if (!Initial)
    {
        BackupFile(fullname, ".backup");
    }
    file = fopen(fullname, "wbS");
    if (!file)
    {
        throw GetResString(IDS_ERR_OPENMETFILE);
    }
    wxUint8 version = PARTFILE_VERSION;
    fwrite( &version, 1, 1, file);
    fwrite( &date, 4, 1, file);
    fwrite( &m_abyFileHash, 16, 1, file);
    wxUint16 parts = hashlist.GetCount();
    fwrite( &parts, 2, 1, file);
    for (int x = 0 ; x != parts ; x++)
    fwrite(hashlist[x], 16, 1, file);
    wxUint32 tagcount = taglist.GetCount() + 10 + (gaplist.GetCount() *2);
    fwrite( &tagcount, 4, 1, file);
    CTag(FT_FILENAME, m_pszFileName) .WriteTagToFile(file);
    CTag(FT_FILESIZE, m_nFileSize) .WriteTagToFile(file);
    CTag(FT_TRANSFERED, transfered) .WriteTagToFile(file);
    CTag(FT_STATUS, (paused) ? 1: 0) .WriteTagToFile(file);
    CTag *prioritytag;
    wxUint8 autoprio = PR_AUTO;
    if (this->IsAutoDownPriority())
    {
        prioritytag = new CTag(FT_DLPRIORITY, autoprio);
    }
    else
    {
        prioritytag = new CTag(FT_DLPRIORITY, m_iDownPriority);
    }
    prioritytag->WriteTagToFile(file);
    delete prioritytag;
    if (this->IsAutoDownPriority())
    {
        prioritytag = new CTag(FT_OLDDLPRIORITY, autoprio);
    }
    else
    {
        prioritytag = new CTag(FT_OLDDLPRIORITY, m_iDownPriority);
    }
    prioritytag->WriteTagToFile(file);
    delete prioritytag;
    CTag *lsctag = new CTag(FT_LASTSEENCOMPLETE, lsc);
    lsctag->WriteTagToFile(file);
    delete lsctag;
    CTag *ulprioritytag;
    if (this->IsAutoUpPriority())
    {
        ulprioritytag = new CTag(FT_ULPRIORITY, autoprio);
    }
    else
    {
        ulprioritytag = new CTag(FT_ULPRIORITY, GetUpPriority());
    }
    ulprioritytag->WriteTagToFile(file);
    delete ulprioritytag;
    if (this->IsAutoUpPriority())
    {
        ulprioritytag = new CTag(FT_OLDULPRIORITY, autoprio);
    }
    else
    {
        ulprioritytag = new CTag(FT_OLDULPRIORITY, GetUpPriority());
    }
    ulprioritytag->WriteTagToFile(file);
    delete ulprioritytag;
    CTag *categorytab = new CTag(FT_CATEGORY, m_category);
    categorytab->WriteTagToFile(file);
    delete categorytab;
    for (wxUint32 j = 0 ; j != (wxUint32) taglist.GetCount() ; j++)
    taglist[j]->WriteTagToFile(file);
    char *namebuffer = new char[10];
    char *number = &namebuffer[1];
    wxUint16 i_pos = 0;
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
    {
        sprintf(number, "%d", i_pos);
        namebuffer[0] = FT_GAPSTART;
        CTag *gapstarttag = new CTag(namebuffer, gaplist.GetAt(pos)->start);
        gapstarttag->WriteTagToFile(file);
        namebuffer[0] = FT_GAPEND;
        CTag *gapendtag = new CTag(namebuffer, (gaplist.GetAt(pos)->end) + 1);
        gapendtag->WriteTagToFile(file);
        delete gapstarttag;
        delete gapendtag;
        i_pos++;
    }
    delete[] namebuffer;
    if (ferror(file))
    {
        throw CString("unexpected write error");
    }
    if (!Initial)
    {
        wxRemoveFile(wxString(fullname) + ".backup");
    }
    fclose(file);
    SafeCopyFile(fullname, wxString(fullname) + ".BAK");
    return true;
}

void CPartFile::PartFileHashFinished(CKnownFile *result)
{
    newdate = true;
    bool errorfound = false;
    for (wxUint32 i = 0 ; i != (wxUint32) hashlist.GetSize() ; i++)
    {
        if (IsComplete(i *PARTSIZE, ((i + 1) *PARTSIZE) - 1))
        {
            if (! (result->GetPartHash(i) && !memcmp(result->GetPartHash(i), this->GetPartHash(i), 16)))
            {
                theApp.xmuledlg->AddLogLine(false, GetResString(IDS_ERR_FOUNDCORRUPTION), i + 1, m_pszFileName);
                AddGap(i *PARTSIZE, ((((i + 1) *PARTSIZE) - 1) >= m_nFileSize) ? m_nFileSize - 1: ((i + 1) *PARTSIZE) - 1);
                errorfound = true;
            }
        }
    }
    delete result;
    if (!errorfound)
    {
        if (status == PS_COMPLETING)
        {
            CompleteFile(true);
            return;
        }
        else
        theApp.xmuledlg->AddLogLine(false, GetResString(IDS_HASHINGDONE), m_pszFileName);
    }
    else
    {
        SetPartFileStatus(PS_READY);
        SavePartFile();
        return;
    }
    SetPartFileStatus(PS_READY);
    SavePartFile();
    theApp.sharedfiles->SafeAddKFile(1,this);
}

void CPartFile::AddGap(wxUint32 start, wxUint32 end)
{
    POSITION pos1, pos2;
    for (pos1 = gaplist.GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        Gap_Struct *cur_gap = gaplist.GetNext(pos1);
        if (cur_gap->start >= start &&cur_gap->end <= end)
        {
            // this gap is inside the new gap - delete
            gaplist.RemoveAt(pos2);
            delete cur_gap;
            continue;
        }
        else if(cur_gap->start >= start &&cur_gap->start <= end)
        {
            // a part of this gap is in the new gap - extend limit and delete
            end = cur_gap->end;
            gaplist.RemoveAt(pos2);
            delete cur_gap;
            continue;
        }
        else if(cur_gap->end <= end &&cur_gap->end >= start)
        {
            // a part of this gap is in the new gap - extend limit and delete
            start = cur_gap->start;
            gaplist.RemoveAt(pos2);
            delete cur_gap;
            continue;
        }
        else if(start >= cur_gap->start &&end <= cur_gap->end)
        {
            // new gap is already inside this gap - return
            return;
        }
    }
    Gap_Struct *new_gap = new Gap_Struct;
    new_gap->start = start;
    new_gap->end = end;
    gaplist.AddTail(new_gap);
    UpdateDisplayedInfo();
    newdate = true;
}

bool CPartFile::IsComplete(wxUint32 start, wxUint32 end)
{
    if (end >= m_nFileSize)
    end = m_nFileSize - 1;
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
    {
        Gap_Struct *cur_gap = gaplist.GetAt(pos);
        if ((cur_gap->start >= start &&cur_gap->end <= end) || (cur_gap->start >= start
        &&cur_gap->start <= end) || (cur_gap->end <= end &&cur_gap->end >= start)
        || (start >= cur_gap->start &&end <= cur_gap->end))
        {
            return false;
        }
    }
    return true;
}

bool CPartFile::IsPureGap(wxUint32 start, wxUint32 end)
{
    if (end >= m_nFileSize)
    end = m_nFileSize - 1;
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
    {
        Gap_Struct *cur_gap = gaplist.GetAt(pos);
        if (start >= cur_gap->start &&end <= cur_gap->end)
        {
            return true;
        }
    }
    return false;
}

bool CPartFile::IsAlreadyRequested(wxUint32 start, wxUint32 end)
{
    for (POSITION pos = requestedblocks_list.GetHeadPosition() ; pos != 0 ; requestedblocks_list.GetNext(pos))
    {
        if (start >= requestedblocks_list.GetAt(pos)->StartOffset &&
        start <= requestedblocks_list.GetAt(pos)->EndOffset)
        return true;
    }
    return false;
}

bool CPartFile::GetNextEmptyBlockInPart(wxUint16 partNumber, Requested_Block_Struct *result)
{
    Gap_Struct *firstGap;
    Gap_Struct *currentGap;
    wxUint32 end;
    wxUint32 blockLimit;
    // Find start of this part
    wxUint32 partStart = (PARTSIZE *partNumber);
    wxUint32 start = partStart;
    // What is the end limit of this block, i.e. can't go outside part (or filesize)
    wxUint32 partEnd = (PARTSIZE * (partNumber + 1)) - 1;
    if (partEnd >= GetFileSize())
    partEnd = GetFileSize() - 1;
    // Loop until find a suitable gap and return true, or no more gaps and return false
    while (true)
    {
        firstGap = NULL;
        // Find the first gap from the start position
        for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
        {
            currentGap = gaplist.GetAt(pos);
            // Want gaps that overlap start<->partEnd
            if ((currentGap->start <= partEnd) && (currentGap->end >= start))
            {
                // Is this the first gap?
                if ((firstGap == NULL) || (currentGap->start < firstGap->start))
                firstGap = currentGap;
            }
        }
        // If no gaps after start, exit
        if (firstGap == NULL)
        return false;
        // Update start position if gap starts after current pos
        if (start < firstGap->start)
        start = firstGap->start;
        // If this is not within part, exit
        if (start > partEnd)
        return false;
        // Find end, keeping within the max block size and the part limit
        end = firstGap->end;
        blockLimit = partStart + (GAPBLOCKSIZE * (((start - partStart) / GAPBLOCKSIZE) + 1)) - 1;
        if (end > blockLimit)
        end = blockLimit;
        if (end > partEnd)
        end = partEnd;
        // If this gap has not already been requested, we have found a valid entry
        if (!IsAlreadyRequested(start, end))
        {
            // Was this block to be returned
            if (result != NULL)
            {
                result->StartOffset = start;
                result->EndOffset = end;
                memcpy(result->FileID, GetFileHash(), 16);
                result->transferred = 0;
            }
            return true;
        }
        else
        {
            // Reposition to end of that gap
            start = end + 1;
        }
        // If tried all gaps then break out of the loop
        if (end == partEnd)
        break;
    }
    // No suitable gap found
    return false;
}

void CPartFile::FillGap(wxUint32 start, wxUint32 end)
{
    POSITION pos1, pos2;
    for (pos1 = gaplist.GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        Gap_Struct *cur_gap = gaplist.GetNext(pos1);
        if (cur_gap->start >= start &&cur_gap->end <= end)
        {
            // our part fills this gap completly
            gaplist.RemoveAt(pos2);
            delete cur_gap;
            continue;
        }
        else if(cur_gap->start >= start &&cur_gap->start <= end)
        {
            // a part of this gap is in the part - set limit
            cur_gap->start = end + 1;
        }
        else if(cur_gap->end <= end &&cur_gap->end >= start)
        {
            // a part of this gap is in the part - set limit
            cur_gap->end = start - 1;
        }
        else if(start >= cur_gap->start &&end <= cur_gap->end)
        {
            wxUint32 buffer = cur_gap->end;
            cur_gap->end = start - 1;
            cur_gap = new Gap_Struct;
            cur_gap->start = end + 1;
            cur_gap->end = buffer;
            gaplist.InsertAfter(pos1, cur_gap);
            // [Lord KiRon]:
            break;
        }
    }
    UpdateCompletedInfos();
    UpdateDisplayedInfo();
    newdate = true;
    SavePartFile();
}

void CPartFile::UpdateCompletedInfos()
{
    wxUint32 allgaps = 0;
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ;)
    {
        POSITION prev = pos;
        Gap_Struct *cur_gap = gaplist.GetNext(pos);
        if ((cur_gap->end > m_nFileSize) || (cur_gap->start >= m_nFileSize))
        gaplist.RemoveAt(prev);
        else
        allgaps += cur_gap->end - cur_gap->start;
    }
    if (gaplist.GetCount() || requestedblocks_list.GetCount())
    {
        percentcompleted = (1.0f - (float) allgaps/m_nFileSize) *100;
        completedsize = m_nFileSize - allgaps - 1;
    }
    else
    {
        percentcompleted = 100;
        completedsize = m_nFileSize;
    }
}

void CPartFile::DrawStatusBar(wxMemoryDC *dc, wxRect rect, bool bFlat)
{
    COLORREF crProgress;
    COLORREF crHave;
    COLORREF crPending;
    COLORREF crMissing = RGB(255, 0, 0);
    if (bFlat)
    {
        crProgress = RGB(0, 150, 0);
        crHave = RGB(0, 0, 0);
        crPending = RGB(255, 255, 100);
    }
    else
    {
        crProgress = RGB(0, 224, 0);
        crHave = RGB(104, 104, 104);
        crPending = RGB(255, 208, 0);
    }
    s_ChunkBar.SetHeight(rect.height - rect.y);
    s_ChunkBar.SetWidth(rect.width - rect.x);
    s_ChunkBar.SetFileSize(m_nFileSize);
    s_ChunkBar.Fill(crHave);
    wxUint32 allgaps = 0;
    if (status == PS_COMPLETE || status == PS_COMPLETING)
    {
        s_ChunkBar.FillRange(0, m_nFileSize, crProgress);
        s_ChunkBar.Draw(dc, rect.x, rect.y, bFlat);
        percentcompleted = 100;
        completedsize = m_nFileSize;
        return;
    }
    // red gaps
    for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
    {
        Gap_Struct *cur_gap = gaplist.GetAt(pos);
        allgaps += cur_gap->end - cur_gap->start;
        bool gapdone = false;
        wxUint32 gapstart = cur_gap->start;
        wxUint32 gapend = cur_gap->end;
        for (wxUint32 i = 0 ; i != GetPartCount() ; i++)
        {
            if (gapstart >= i *PARTSIZE &&gapstart <= (i + 1) *PARTSIZE)
            {
                // is in this part?
                if (gapend <= (i + 1) *PARTSIZE)
                gapdone = true;
                else
                {
                    // and next part:
                    gapend = (i + 1) *PARTSIZE;
                }
                // paint
                COLORREF color;
                // frequency?:
                if (m_SrcpartFrequency.GetCount() >= (int) i &&m_SrcpartFrequency[i])
                color = RGB(0,
                (210 - (22 * (m_SrcpartFrequency[i] - 1)) < 0) ? 0: 210 - (22 * (m_SrcpartFrequency[i] - 1))
                , 255);
                else
                color = crMissing;
                s_ChunkBar.FillRange(gapstart, gapend + 1, color);
                // finished?:
                if (gapdone)
                break;
                else
                {
                    gapstart = gapend;
                    gapend = cur_gap->end;
                }
            }
        }
    }
    // yellow pending parts
    for (POSITION pos = requestedblocks_list.GetHeadPosition() ; pos != 0 ; requestedblocks_list.GetNext(pos))
    {
        Requested_Block_Struct *block = requestedblocks_list.GetAt(pos);
        s_ChunkBar.FillRange(block->StartOffset, block->EndOffset, crPending);
    }
    s_ChunkBar.Draw(dc, rect.x, rect.y, bFlat);
    // green progress
    float blockpixel = (float)(rect.width - rect.x) / ((float) m_nFileSize);
    RECT gaprect;
    gaprect.top = rect.y;
    gaprect.bottom = gaprect.top + 4;
    //->left;:
    gaprect.left = rect.x;
    if (!bFlat)
    {
        s_LoadBar.SetWidth((wxUint32)((float)((float)((m_nFileSize - ((allgaps == 0) ?1: allgaps)) - 1)) *blockpixel + .5f));
        s_LoadBar.Fill(crProgress);
        s_LoadBar.Draw(dc, gaprect.left, gaprect.top, false);
    }
    else
    {
        gaprect.right = rect.x + (wxUint32)((float)((float)((m_nFileSize - ((allgaps == 0) ?1: allgaps)) - 1)) *blockpixel + .5f);
        dc->SetBrush( * (wxTheBrushList->FindOrCreateBrush(wxColour(crProgress), wxSOLID)));
        dc->DrawRectangle(gaprect.left, gaprect.top, gaprect.right, gaprect.bottom);
        //draw gray progress only if flat
        gaprect.left = gaprect.right;
        gaprect.right = rect.width;
        dc->SetBrush( * (wxTheBrushList->FindOrCreateBrush(wxColour(224, 224, 224), wxSOLID)));
        dc->DrawRectangle(gaprect.left, gaprect.top, gaprect.right, gaprect.bottom);
    }
    if ((gaplist.GetCount() || requestedblocks_list.GetCount()))
    {
        percentcompleted = ((1.0f - (float) allgaps/m_nFileSize)) *100;
        completedsize = m_nFileSize - allgaps - 1;
    }
    else
    {
        percentcompleted = 100;
        completedsize = m_nFileSize;
    }
}

void CPartFile::WritePartStatus(CMemFile *file)
{
    wxUint16 parts = hashlist.GetCount();
    file->Write( &parts, 2);
    wxUint16 done = 0;
    while (done != parts)
    {
        wxUint8 towrite = 0;
        for (wxUint32 i = 0 ; i != 8 ; i++)
        {
            if (IsComplete(done *PARTSIZE, ((done + 1) *PARTSIZE) - 1))
            towrite |= (1 << i);
            done++;
            if (done == parts)
            break;
        }
        file->Write( &towrite, 1);
    }
}

int CPartFile::GetValidSourcesCount()
{
    int counter = 0;
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        srclists[sl].GetNext(pos1);
        CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
        if (cur_src->GetDownloadState() != DS_ONQUEUE &&cur_src->GetDownloadState() != DS_DOWNLOADING &&
        cur_src->GetDownloadState() != DS_NONEEDEDPARTS) counter++;
    }
    return counter;
}

wxUint16 CPartFile::GetNotCurrentSourcesCount()
{
    wxUint16 counter = 0;
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        srclists[sl].GetNext(pos1);
        CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
        if (cur_src->GetDownloadState() != DS_ONQUEUE &&cur_src->GetDownloadState() != DS_DOWNLOADING) counter++;
    }
    return counter;
}

wxUint8 CPartFile::GetStatus(bool ignorepause)
{
    if ((!paused) || status == PS_ERROR || ignorepause)
    return status;
    else
    return PS_PAUSED;
}

wxUint32 CPartFile::Process(wxUint32 reducedownload)
{
    //in percent
    wxUint16 old_trans;
    wxUint32 dwCurTick =::GetTickCount();
    // If buffer size exceeds limit, or if not written within time limit, flush data
    if ((m_nTotalBufferData > theApp.glob_prefs->GetFileBufferSize()))
    {
        // Avoid flushing while copying preview file
        if (!m_bPreviewing)
        {
            FlushBuffer();
        }
    }
    // check if we want new sources from server
    if (((!lastsearchtime) || (dwCurTick - lastsearchtime) > SERVERREASKTIME)
    &&theApp.serverconnect->IsConnected()
    &&theApp.glob_prefs->GetMaxSourcePerFileSoft() > GetSourceCount() && !stopped)
    {
        //local server
        lastsearchtime = dwCurTick;
        Packet *packet = new Packet(OP_GETSOURCES, 16);
        memcpy(packet->pBuffer, m_abyFileHash, 16);
        theApp.uploadqueue->AddUpDataOverheadServer(packet->size);
        theApp.serverconnect->SendPacket(packet, true);
    }
    old_trans = transferingsrc;
    transferingsrc = 0;
    datarate = 0;
    POSITION pos1, pos2;
    for (wxUint32 sl = 0 ; sl < SOURCESSLOTS ; sl++)
    {
        if (!srclists[sl].IsEmpty())
        {
            for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
            {
                srclists[sl].GetNext(pos1);
                CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
                switch (cur_src->GetDownloadState())
                {
                case DS_DOWNLOADING:
                    {
                        transferingsrc++;
                        wxUint32 cur_datarate = cur_src->CalculateDownloadRate();
                        datarate += cur_datarate;
                        if (reducedownload &&cur_src->GetDownloadState() == DS_DOWNLOADING)
                        {
                            //(wxUint32)(((float)reducedownload/100)*cur_datarate)/10;:
                            wxUint32 limit = reducedownload *cur_datarate/1000;
                            if (limit < 1000 &&reducedownload == 200)
                            limit += 1000;
                            else if(limit < 1)
                            limit = 1;
                            cur_src->socket->SetDownloadLimit(limit);
                        }
                        else
                        {
                            cur_src->socket->DisableDownloadLimit();
                        }
                        cur_src->SetValidSource(true);
                        break;
                    }
                case DS_BANNED:
                case DS_ERROR:
                    break;
                    // if we now have a high ip we can ask:
                case DS_LOWTOLOWIP:
                    if (((dwCurTick - lastpurgetime) > 30000) && (this->GetSourceCount() >= (theApp.glob_prefs->GetMaxSourcePerFile() *.8)))
                    {
                        theApp.downloadqueue->RemoveSource(cur_src);
                        lastpurgetime = dwCurTick;
                        break;
                    }
                    if (theApp.serverconnect->IsLowID())
                    break;
                case DS_NONEEDEDPARTS:
                    {
 /* No needed sources are dropped if over 80% of available source slots
    are full, once per 40 seconds. */
                        cur_src->SetValidSource(false);
                        if (((dwCurTick - lastpurgetime) > 40000) && (this->GetSourceCount() >= (theApp.glob_prefs->GetMaxSourcePerFile() *.8)))
                        if (!cur_src->SwapToAnotherFile())
                        {
                            theApp.downloadqueue->RemoveSource(cur_src);
                            lastpurgetime = dwCurTick;
                            //Johnny-B - nothing more to do here (good eye!):
                            break;
                        }
                        // doubled reasktime for no needed parts - save connections and traffic
                        if (! ((!cur_src->GetLastAskedTime()) || (dwCurTick - cur_src->GetLastAskedTime()) > FILEREASKTIME *2))
                        break;
                    }
                case DS_ONQUEUE:
                    {
 /* Full queue sources are dropped if over 80% of available source slots
    are filled, once per 60 seconds. */
                        cur_src->SetValidSource(true);
                        if (cur_src->IsRemoteQueueFull())
                        cur_src->SetValidSource(false);
                        if (((dwCurTick - lastpurgetime) > 60000) && (this->GetSourceCount() >= (theApp.glob_prefs->GetMaxSourcePerFile() *.8)))
                        {
                            theApp.downloadqueue->RemoveSource(cur_src);
                            lastpurgetime = dwCurTick;
                            //Johnny-B - nothing more to do here (good eye!):
                            break;
                        }
                        if (theApp.serverconnect->IsConnected() && ((!cur_src->GetLastAskedTime()) || (dwCurTick - cur_src->GetLastAskedTime()) > FILEREASKTIME - 20000))
                        cur_src->UDPReaskForDownload();
                    }
                case DS_CONNECTED:
                case DS_CONNECTING:
                    if ((dwCurTick - cur_src->GetLastAskedTime()) > FILEREASKTIME)
                    {
                        cur_src->AskForDownload();
                    }
                    break;
                case DS_NONE:
                case DS_TOOMANYCONNS:
                case DS_WAITCALLBACK:
                    if ((dwCurTick - cur_src->GetLastAskedTime()) > CONNECTION_TIMEOUT)
                    {
                        cur_src->AskForDownload();
                    }
                    break;
                }
            }
        }
    }
    // swap No needed partfiles if possible
    if (((!m_LastNoNeededCheck) || (dwCurTick - m_LastNoNeededCheck) > 10000))
    {
        m_LastNoNeededCheck = dwCurTick;
        if (this->IsA4AFAuto())
        {
            POSITION pos1, pos2;
            for (pos1 = this->A4AFSourcesList.GetHeadPosition() ; (pos2 = pos1) != NULL ;)
            {
                this->A4AFSourcesList.GetNext(pos1);
                CUpDownClient *cur_source = this->A4AFSourcesList.GetAt(pos2);
                if (! (cur_source->GetDownloadState() == DS_DOWNLOADING))
                {
                    this->A4AFSourcesList.RemoveAt(pos2);
                    cur_source->SwapToThisFile(this);
                }
            }
        }
    }
    /* Sources droping engine. Auto drop allowed type of sources at interval. */
    if ((dwCurTick - m_LastSourceDropTime) > theApp.glob_prefs->GetAutoDropTimer() *1000)
    {
        m_LastSourceDropTime = dwCurTick;
    /* If all three are enabled, use CleanUpSources() function, will save us some CPU. */
        if (theApp.glob_prefs->DropNoNeededSources() &&theApp.glob_prefs->DropFullQueueSources() &&theApp.glob_prefs->DropHighQueueRankingSources())
        {
            CleanUpSources();
        }
        else
        {
    /* Then check separately for each of them, and act accordingly. */
            if (theApp.glob_prefs->DropNoNeededSources())
            {
                RemoveNoNeededSources();
            }
            if (theApp.glob_prefs->DropFullQueueSources())
            {
                RemoveFullQueueSources();
            }
            if (theApp.glob_prefs->DropHighQueueRankingSources())
            {
                RemoveHighQueueRatingSources();
            }
        }
    }
    if (((old_trans == 0) && (transferingsrc > 0)) || ((old_trans > 0) && (transferingsrc == 0)))
    {
        SetPartFileStatus(status);
    }
    // calculate datarate, set limit etc.
    count++;
    if (count == 30)
    {
        count = 0;
        UpdateDisplayedInfo();
        if (m_bPercentUpdated == false)
        UpdateCompletedInfos();
        m_bPercentUpdated = false;
    }
    return datarate;
}

void CPartFile::AddSources(CMemFile *sources, wxUint32 serverip, wxUint16 serverport)
{
    if (stopped) return;
    wxUint8 count;
    wxUint8 debug_lowiddropped = 0;
    wxUint8 debug_possiblesources = 0;
    sources->Read( &count, 1);
    for (int i = 0 ; i != count ; i++)
    {
        wxUint32 userid;
        sources->Read( &userid, 4);
        wxUint16 port;
        sources->Read( &port, 2);
        // check first if we are this source
        // MOD Note: Do not change this part - Merkur
        if (theApp.serverconnect->GetClientID() < 16777216 &&theApp.serverconnect->IsConnected())
        {
            if ((theApp.serverconnect->GetClientID() == userid) && inet_addr(theApp.serverconnect->GetCurrentServer()->GetFullIP()) == serverip)
            continue;
        }
        else if(theApp.serverconnect->GetClientID() == userid)
        continue;
        else if(userid < 16777216 && !theApp.serverconnect->IsLocalServer(serverip, serverport))
        {
            debug_lowiddropped++;
            continue;
        }
        // MOD Note - end
        if (theApp.glob_prefs->GetMaxSourcePerFile() > this->GetSourceCount())
        {
            debug_possiblesources++;
            CUpDownClient *newsource = new CUpDownClient(port, userid, serverip, serverport, this);
            theApp.downloadqueue->CheckAndAddSource(this, newsource);
        }
    }
}

void CPartFile::NewSrcPartsInfo()
{
    // Cache part count
    wxUint16 partcount = GetPartCount();
    // Increase size if necessary
    if (m_SrcpartFrequency.GetSize() < partcount)
    {
        m_SrcpartFrequency.SetSize(partcount);
    }
    // Reset part counters
    for (int i = 0 ; i < partcount ; i++)
    {
        m_SrcpartFrequency[i] = 0;
    }
    CUpDownClient *cur_src;
    for (int sl = 0 ; sl < SOURCESSLOTS ;++sl)
    {
        if (!srclists[sl].IsEmpty())
        {
            for (POSITION pos = srclists[sl].GetHeadPosition() ; pos != 0 ;)
            {
                cur_src = srclists[sl].GetNext(pos);
                for (int i = 0 ; i != partcount ; i++)
                {
                    if (cur_src->IsPartAvailable(i))
                    m_SrcpartFrequency[i] += 1;
                }
            }
        }
    }
    UpdateDisplayedInfo();
}

bool CPartFile::GetNextRequestedBlock(CUpDownClient *sender, Requested_Block_Struct **newblocks, wxUint16 *count)
{
    wxUint16 requestedCount = *count;
    wxUint16 newblockcount = 0;
    wxUint8 *partsav = sender->GetPartStatus();
    *count = 0;
    wxUint16 randomness;
    CList < int, int > liGoodParts;
    CList < int, int > liPossibleParts;
    bool finished = false;
    while (!finished)
    {
        // Need to empty lists to avoid infinite loop when file is smaller than 180k
        // Otherwise it would keep looping to find 3 blocks, there is only one and it is requested
        liGoodParts.RemoveAll();
        liPossibleParts.RemoveAll();
        // Barry - Top priority should be to continue downloading from current blocks (if anything left to download)
        bool foundPriorityPart = false;
        wxString gettingParts;
        sender->ShowDownloadingParts(gettingParts);
        for (int i = (gettingParts.Len() - 1) ; i >= 0 ; i--)
        {
            if ((gettingParts.GetChar(i) == 'Y') && (GetNextEmptyBlockInPart(i, 0)))
            {
                liGoodParts.AddHead(i);
                foundPriorityPart = true;
            }
        }
        // Barry - Give priorty to end parts of archives and movies
        if ((!foundPriorityPart) && (IsArchive() || IsMovie()) &&theApp.glob_prefs->GetPreviewPrio())
        {
            wxUint32 partCount = GetPartCount();
            // First part
            if (sender->IsPartAvailable(0) &&GetNextEmptyBlockInPart(0, 0))
            {
                liGoodParts.AddHead(0);
                foundPriorityPart = true;
            }
            else if((partCount > 1))
            {
                // Last part
                if (sender->IsPartAvailable(partCount - 1) &&GetNextEmptyBlockInPart(partCount - 1, 0))
                {
                    liGoodParts.AddHead(partCount - 1);
                    foundPriorityPart = true;
                }
                // Barry - Better to get rarest than these, add to list, but do not exclude all others.
                // These get priority over other parts with same availability.
                else if(partCount > 2)
                {
                    // Second part
                    if (sender->IsPartAvailable(1) &&GetNextEmptyBlockInPart(1, 0))
                    liGoodParts.AddHead(1);
                    // Penultimate part
                    else if(sender->IsPartAvailable(partCount - 2) &&GetNextEmptyBlockInPart(partCount - 2, 0))
                    liGoodParts.AddHead(partCount - 2);
                }
            }
        }
        if (!foundPriorityPart)
        {
            randomness = (wxUint16) ROUND(((float) rand() /RAND_MAX) * (GetPartCount() - 1));
            for (wxUint16 i = 0 ; i != GetPartCount() ; i++)
            {
                if (sender->IsPartAvailable(randomness))
                {
                    if (partsav[randomness] && !IsComplete(randomness *PARTSIZE, ((randomness + 1) *PARTSIZE) - 1))
                    {
 /*if (IsCorruptedPart(randomness)){
 if (GetNextEmptyBlockInPart(randomness,0)){
 goodpart = randomness;
 break;
 }
 }
    else */
                        if (IsPureGap(randomness *PARTSIZE, ((randomness + 1) *PARTSIZE) - 1))
                        {
                            if (GetNextEmptyBlockInPart(randomness, 0))
                            liPossibleParts.AddHead(randomness);
                        }
                        else if(GetNextEmptyBlockInPart(randomness, 0))
                        // Barry - Add after archive/movie entries:
                        liGoodParts.AddTail(randomness);
                    }
                }
                randomness++;
                if (randomness == GetPartCount())
                randomness = 0;
            }
        }
        CList < int, int > *usedlist;
        if (!liGoodParts.IsEmpty())
        usedlist = &liGoodParts;
        else if(!liPossibleParts.IsEmpty())
        usedlist = &liPossibleParts;
        else
        {
            if (!newblockcount)
            {
                return false;
            }
            else
            break;
        }
        wxUint16 nRarest = 0xFFFF;
        wxUint16 usedpart = usedlist->GetHead();
        for (POSITION pos = usedlist->GetHeadPosition() ; pos != 0 ; usedlist->GetNext(pos))
        {
            if (m_SrcpartFrequency.GetCount() >= usedlist->GetAt(pos)
            &&m_SrcpartFrequency[usedlist->GetAt(pos) ] < nRarest)
            {
                nRarest = m_SrcpartFrequency[usedlist->GetAt(pos) ];
                usedpart = usedlist->GetAt(pos);
            }
        }
        while (true)
        {
            Requested_Block_Struct *block = new Requested_Block_Struct;
            if (GetNextEmptyBlockInPart(usedpart, block))
            {
                requestedblocks_list.AddTail(block);
                newblocks[newblockcount] = block;
                newblockcount++;
                if (newblockcount == requestedCount)
                {
                    finished = true;
                    break;
                }
            }
            else
            {
                delete block;
                break;
            }
        }
        //wend:
    }
    *count = newblockcount;
    return true;
}

void CPartFile::RemoveBlockFromList(wxUint32 start, wxUint32 end)
{
    POSITION pos1, pos2;
    for (pos1 = requestedblocks_list.GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        requestedblocks_list.GetNext(pos1);
        if (requestedblocks_list.GetAt(pos2)->StartOffset <= start &&requestedblocks_list.GetAt(pos2)->EndOffset >= end)
        requestedblocks_list.RemoveAt(pos2);
    }
}

void CPartFile::RemoveAllRequestedBlocks(void)
{
    requestedblocks_list.RemoveAll();
}

#include <pthread.h>
pthread_attr_t pattr;

void CPartFile::CompleteFile(bool bIsHashingDone)
{
    if (this->srcarevisible)
    theApp.xmuledlg->transferwnd->downloadlistctrl->HideSources(this);
    SetPartFileStatus(PS_COMPLETING);
    if (!bIsHashingDone)
    {
        datarate = 0;
        char *partfileb = nstrdup(partmetfilename);
        partfileb[strlen(partmetfilename) - 4] = 0;
        CAddFileThread::AddFile(theApp.glob_prefs->GetTempDir(), partfileb, this);
        delete[] partfileb;
        return;
    }
    else
    {
        StopFile();
        // guess I was wrong about not need to spaw a thread.. It is if the temp and incoming dirs are on different partitions/drives and the file is large...[oz]
        // use pthreads
        pthread_t tid;
        pthread_attr_init( &pattr);
        pthread_attr_setdetachstate( &pattr, PTHREAD_CREATE_DETACHED);
        pthread_create( &tid, &pattr, (void * ( *)(void *)) CompleteThreadProc, this);
    }
    theApp.xmuledlg->transferwnd->downloadlistctrl->ShowFilesCount();
    UpdateDisplayedInfo(true);
}

wxUint16 CPartFile::CompleteThreadProc(CPartFile *pFile)
{
    if (!pFile)
    return(unsigned int) - 1;
    pFile->PerformFileComplete();
    return 0;
}

// Lord KiRon - using threads for file completion
bool CPartFile::PerformFileComplete()
{
    //CSingleLock(&m_FileCompleteMutex,TRUE); // will be unlocked on exit
    wxMutexLocker sLock(m_FileCompleteMutex);
    char *partfilename = nstrdup(fullname);
    // assumes ".met" at the end:
    partfilename[strlen(fullname) - 4] = 0;
    char *newfilename = nstrdup(GetFileName());
    strcpy(newfilename, theApp.StripInvalidFilenameChars(newfilename));
    // ???:
    char *newname = new char[strlen(newfilename) + strlen(theApp.glob_prefs->GetCategory(GetCategory())->incomingpath) + MAX_PATH *2];
    CString indir;
    if (wxFileName::DirExists(theApp.glob_prefs->GetCategory(GetCategory())->incomingpath))
    {
        indir = theApp.glob_prefs->GetCategory(GetCategory())->incomingpath;
        sprintf(newname, "%s/%s", indir.GetData(), newfilename);
    }
    else
    {
        indir = theApp.glob_prefs->GetIncomingDir();
        sprintf(newname, "%s/%s", indir.GetData(), newfilename);
    }
    // add this file to the suspended uploads list
    CFileHash tmp_filehash(GetFileHash());
    theApp.uploadqueue->SuspendUpload(tmp_filehash);
    FlushBuffer();
    // close permanent handle
    m_hpartfile->Close();
    bool renamed = false;
    if (wxFileName::FileExists(newname))
    {
        renamed = true;
        int namecount = 0;
        size_t length = strlen(newfilename);
        //the file extension
        char *ext = strrchr(newfilename, '.');
        if (ext == NULL)
        ext = newfilename + length;
        //new end is the file name before extension:
        char *last = ext;
        //truncate file name:
        last[0] = 0;
        //serch for matching ()s and check if it contains a number
        if ((ext != newfilename) && (strrchr(newfilename, ')') + 1 == last))
        {
            char *first = strrchr(newfilename, '(');
            if (first != NULL)
            {
                first++;
                bool found = true;
                for (char *step = first ; step < last - 1 ; step++)
                if ( *step < '0' || *step > '9')
                {
                    found = false;
                    break;
                }
                if (found)
                {
                    namecount = atoi(first);
                    last = first - 1;
                    //truncate again:
                    last[0] = 0;
                }
            }
        }
        CString strTestName;
        do
        {
            namecount++;
            strTestName.Format("%s/%s(%d).%s", theApp.glob_prefs->GetIncomingDir(),
            newfilename, namecount, min(ext + 1, newfilename + length));
        }
        while (wxFileName::FileExists(strTestName));
        delete[] newname;
        newname = nstrdup(strTestName);
    }
    delete[] newfilename;
    if (!wxRenameFile(partfilename, newname))
    {
        if (!SafeCopyFile(partfilename, newname))
        {
            delete[] partfilename;
            delete[] newname;
            wxMutexGuiEnter();
            theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_COMPLETIONFAILED), GetFileName());
            wxMutexGuiLeave();
            paused = true;
            SetPartFileStatus(PS_ERROR);
            wxMutexGuiEnter();
            theApp.downloadqueue->StartNextFile();
            wxMutexGuiLeave();
            return FALSE;
        }
        if (!wxRemoveFile(partfilename))
        {
            printf("info: could not remove original '%s' after creating backup\n", partfilename);
        }
    }
    if (!wxRemoveFile(fullname))
    {
        wxMutexGuiEnter();
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_DELETEFAILED), fullname);
        wxMutexGuiLeave();
    }
    CString BAKName(fullname);
    BAKName.Append(".BAK");
    if (!wxRemoveFile(BAKName))
    {
        wxMutexGuiEnter();
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_DELETE), BAKName.GetData());
        wxMutexGuiLeave();
    }
    delete[] partfilename;
    delete [] fullname;
    fullname = newname;
    delete[] directory;
    directory = nstrdup(theApp.glob_prefs->GetCategory(m_category)->incomingpath);
    SetPartFileStatus(PS_COMPLETE);
    paused = false;
    wxMutexGuiEnter();
    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_DOWNLOADDONE), GetFileName());
    theApp.xmuledlg->ShowNotifier(GetResString(IDS_TBN_DOWNLOADDONE) + "\n" + GetFileName(), TBN_DLOAD);
    wxMutexGuiLeave();
    if (renamed)
    {
        wxMutexGuiEnter();
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_DOWNLOADRENAMED), (strrchr(newname, '/') ? strrchr(newname, '/') + 1: newname));
        wxMutexGuiLeave();
    }
    // TODO: What the f*** if it is already known?
    theApp.knownfiles->SafeAddKFile(1,this);
    // remove the file from the suspended uploads list
    theApp.uploadqueue->ResumeUpload(tmp_filehash);
    SetAutoUpPriority(false);
    theApp.downloadqueue->RemoveFile(this);
    //theApp.xmuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
    wxMutexGuiEnter();
    UpdateDisplayedInfo();
    theApp.xmuledlg->transferwnd->downloadlistctrl->ShowFilesCount();
    wxMutexGuiLeave();
    //SHAddToRecentDocs(SHARD_PATH, fullname); // This is a real nasty call that takes ~110 ms on my 1.4 GHz Athlon and isn't really needed afai see...[ozon]
    // Barry - Just in case
    //		transfered = m_nFileSize;
    wxMutexGuiEnter();
    theApp.downloadqueue->StartNextFile();
    wxMutexGuiLeave();
    return TRUE;
}

void CPartFile::RemoveAllSources(bool bTryToSwap)
{
    //TODO transfer sources to other downloading files if possible
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        srclists[sl].GetNext(pos1);
        if (bTryToSwap)
        {
            if (!srclists[sl].GetAt(pos2)->SwapToAnotherFile())
            theApp.downloadqueue->RemoveSource(srclists[sl].GetAt(pos2));
        }
        else
        theApp.downloadqueue->RemoveSource(srclists[sl].GetAt(pos2));
    }
    /* Razor 1a - Modif by MikaelB */
    if (!this->A4AFSourcesList.IsEmpty())
    {
        POSITION pos1, pos2;
        for (pos1 = this->A4AFSourcesList.GetHeadPosition() ; (pos2 = pos1) != NULL ;)
        {
            this->A4AFSourcesList.GetNext(pos1);
            POSITION pos3 = this->A4AFSourcesList.GetAt(pos2)->m_OtherRequests_list.Find(this);
            if (pos3)
            {
                this->A4AFSourcesList.GetAt(pos2)->m_OtherRequests_list.RemoveAt(pos3);
                theApp.xmuledlg->transferwnd->downloadlistctrl->RemoveSource(this->A4AFSourcesList.GetAt(pos2), this);
                this->A4AFSourcesList.RemoveAt(pos2);
            }
        }
    }
    /* End modif */
    UpdateFileRatingCommentAvail();
}

void CPartFile::DeleteFile()
{
    // Barry - Need to tell any connected clients to stop sending the file
    StopFile();
    theApp.sharedfiles->RemoveFile(this);
    theApp.downloadqueue->RemoveFile(this);
    theApp.xmuledlg->transferwnd->downloadlistctrl->RemoveFile(this);
    m_hpartfile->Close();
    if (!wxRemoveFile(fullname))
    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_DELETE), fullname);
    char *partfilename = nstrdup(fullname);
    partfilename[strlen(fullname) - 4] = 0;
    if (!wxRemoveFile(partfilename))
    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_DELETE), partfilename);
    CString BAKName(fullname);
    BAKName.Append(".BAK");
    if (!wxRemoveFile(BAKName))
    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_DELETE), BAKName.GetData());
    delete[] partfilename;
}

bool CPartFile::HashSinglePart(wxUint16 partnumber)
{
    if ((GetHashCount() <= partnumber) && (GetPartCount() > 1))
    {
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_HASHERRORWARNING), GetFileName());
        this->hashsetneeded = true;
        return true;
    }
    else if(!GetPartHash(partnumber) &&GetPartCount() != 1)
    {
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_INCOMPLETEHASH), GetFileName());
        this->hashsetneeded = true;
        return true;
    }
    else
    {
        uchar hashresult[16];
        m_hpartfile->Seek(PARTSIZE *partnumber);
        wxUint32 length = PARTSIZE;
        if (PARTSIZE * (partnumber + 1) > m_hpartfile->Length())
        length = (m_hpartfile->Length() - (PARTSIZE *partnumber));
        CreateHashFromFile( m_hpartfile, length, hashresult);
        if (GetPartCount() > 1)
        {
            if (memcmp(hashresult, GetPartHash(partnumber), 16))
            return false;
            else
            return true;
        }
        else
        {
            if (memcmp(hashresult, m_abyFileHash, 16))
            return false;
            else
            return true;
        }
    }
}

bool CPartFile::IsCorruptedPart(wxUint16 partnumber)
{
    return corrupted_list.Find(partnumber);
}

bool CPartFile::IsMovie()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(5);
    it_is = ((extension.CmpNoCase(".divx") == 0) || (extension.CmpNoCase(".mpeg") == 0) ||
    (extension.CmpNoCase(".vivo") == 0));
    extension = wxString(GetFileName()) .Right(4);
    it_is = (it_is || (extension.CmpNoCase(".avi") == 0) || (extension.CmpNoCase(".mpg") == 0) ||
    (extension.CmpNoCase(".m2v") == 0) || (extension.CmpNoCase(".wmv") == 0) ||
    (extension.CmpNoCase(".asf") == 0) || (extension.CmpNoCase(".mov") == 0) ||
    (extension.CmpNoCase(".bin") == 0) || (extension.CmpNoCase(".swf") == 0) ||
    (extension.CmpNoCase(".ogm") == 0));
    extension = wxString(GetFileName()) .Right(3);
    it_is = (it_is || (extension.CmpNoCase(".rm") == 0) || (extension.CmpNoCase(".qt") == 0));
    return(it_is);
}

bool CPartFile::IsArchive()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(4);
    it_is = ((extension.CmpNoCase(".zip") == 0) || (extension.CmpNoCase(".rar") == 0) ||
    (extension.CmpNoCase(".ace") == 0) || (extension.CmpNoCase(".arj") == 0) ||
    (extension.CmpNoCase(".lhz") == 0) || (extension.CmpNoCase(".tar") == 0) ||
    (extension.CmpNoCase(".bz2") == 0));
    extension = wxString(GetFileName()) .Right(3);
    it_is = (it_is || (extension.CmpNoCase(".gz") == 0) || (extension.CmpNoCase(".bz") == 0));
    return(it_is);
}

bool CPartFile::IsSound()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(4);
    it_is = ((extension.CmpNoCase(".mp3") == 0) || (extension.CmpNoCase(".mp2") == 0) ||
    (extension.CmpNoCase(".wma") == 0) || (extension.CmpNoCase(".ogg") == 0) ||
    (extension.CmpNoCase(".rma") == 0) || (extension.CmpNoCase(".wav") == 0) ||
    (extension.CmpNoCase(".mid") == 0));
    return(it_is);
}

bool CPartFile::IsCDImage()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(4);
    it_is = ((extension.CmpNoCase(".bin") == 0) || (extension.CmpNoCase(".cue") == 0) ||
    (extension.CmpNoCase(".nrg") == 0) || (extension.CmpNoCase(".ccd") == 0) ||
    (extension.CmpNoCase(".img") == 0) || (extension.CmpNoCase(".iso") == 0));
    return(it_is);
}

bool CPartFile::IsImage()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(5);
    it_is = (extension.CmpNoCase(".jpeg") == 0) || (extension.CmpNoCase(".tiff") == 0);
    extension = wxString(GetFileName()) .Right(4);
    it_is = (it_is || (extension.CmpNoCase(".jpg") == 0) || (extension.CmpNoCase(".gif") == 0) ||
    (extension.CmpNoCase(".bmp") == 0) || (extension.CmpNoCase(".rle") == 0) ||
    (extension.CmpNoCase(".psp") == 0) || (extension.CmpNoCase(".tga") == 0) ||
    (extension.CmpNoCase(".wmf") == 0) || (extension.CmpNoCase(".xpm") == 0) ||
    (extension.CmpNoCase(".png") == 0) || (extension.CmpNoCase(".pcx") == 0));
    return(it_is);
}

bool CPartFile::IsText()
{
    bool it_is;
    wxString extension = wxString(GetFileName()) .Right(5);
    it_is = (extension.CmpNoCase(".html") == 0);
    extension = wxString(GetFileName()) .Right(4);
    it_is = (it_is || (extension.CmpNoCase(".doc") == 0) || (extension.CmpNoCase(".txt") == 0) ||
    (extension.CmpNoCase(".pdf") == 0) || (extension.CmpNoCase(".ps") == 0) ||
    (extension.CmpNoCase(".htm") == 0) || (extension.CmpNoCase(".sxw") == 0) ||
    (extension.CmpNoCase(".log") == 0));
    return(it_is);
}

void CPartFile::SetDownPriority(wxUint8 np)
{
    m_iDownPriority = np;
    theApp.downloadqueue->SortByPriority();
    //theApp.xmuledlg->transferwnd->downloadlistctrl->UpdateItem(this);
    UpdateDisplayedInfo(true);
    SavePartFile();
}

void CPartFile::StopFile()
{
    // Barry - Need to tell any connected clients to stop sending the file
    // Kry - Need to set it hereto get into SetPartFileStatus(status) correctly:
    stopped = true;
    PauseFile();
    RemoveAllSources(true);
    datarate = 0;
    transferingsrc = 0;
    FlushBuffer();
    //theApp.xmuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
    UpdateDisplayedInfo(true);
}

void CPartFile::PauseFile()
{
    if (status == PS_COMPLETE || status == PS_COMPLETING) return;
    Packet *packet = new Packet(OP_CANCELTRANSFER, 0);
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++)
    {
        if (!srclists[sl].IsEmpty())
        {
            for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
            {
                srclists[sl].GetNext(pos1);
                CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
                if (cur_src->GetDownloadState() == DS_DOWNLOADING)
                {
                    theApp.uploadqueue->AddUpDataOverheadOther(packet->size);
#if defined(__DEBUG__)
                    // printf("CPartFile::PauseFile OP_CANCELTRANSFER size\n", packet->size);
#endif
                    cur_src->socket->SendPacket(packet, false, true);
                    cur_src->SetDownloadState(DS_ONQUEUE);
                }
            }
        }
    }
    delete packet;
    paused = true;
    SetPartFileStatus(status);
    datarate = 0;
    transferingsrc = 0;
    UpdateDisplayedInfo(true);
    SavePartFile();
}

void CPartFile::ResumeFile()
{
    if (status == PS_COMPLETE || status == PS_COMPLETING) return;
    paused = false;
    stopped = false;
    lastsearchtime = 0;
    SetPartFileStatus(status);
    //theApp.xmuledlg->transferwnd->downloadlistctrl->UpdateItem(this);
    UpdateDisplayedInfo(true);
}

CString CPartFile::getPartfileStatus()
{
    CString mybuffer;
    if (GetTransferingSrcCount() > 0) mybuffer = GetResString(IDS_DOWNLOADING);
    else mybuffer = GetResString(IDS_WAITING);
    switch (GetStatus())
    {
    case PS_HASHING:
    case PS_WAITINGFORHASH:
        mybuffer = GetResString(IDS_HASHING);
        break;
    case PS_COMPLETING:
        mybuffer = GetResString(IDS_COMPLETING);
        break;
    case PS_COMPLETE:
        mybuffer = GetResString(IDS_COMPLETE);
        break;
    case PS_PAUSED:
        mybuffer = GetResString(IDS_PAUSED);
        break;
    case PS_ERROR:
        mybuffer = GetResString(IDS_ERRORLIKE);
        break;
    }
    if (stopped && (GetStatus() != PS_COMPLETE))
    {
        mybuffer = GetResString(IDS_STOPPED);
    }
    return mybuffer;
}

int CPartFile::getPartfileStatusRang()
{
    int tempstatus = 0;
    if (GetTransferingSrcCount() == 0) tempstatus = 1;
    switch (GetStatus())
    {
    case PS_HASHING:
    case PS_WAITINGFORHASH:
        tempstatus = 3;
        break;
    case PS_COMPLETING:
        tempstatus = 4;
        break;
    case PS_COMPLETE:
        tempstatus = 5;
        break;
    case PS_PAUSED:
        tempstatus = 2;
        break;
    case PS_ERROR:
        tempstatus = 6;
        break;
    }
    return tempstatus;
}

wxInt32 CPartFile::getTimeRemaining()
{
    if (GetDatarate() == 0) return - 1;
    return((GetFileSize() - GetCompletedSize()) / GetDatarate());
}

void CPartFile::PreviewFile()
{
    wxString command;
    // If no player set in preferences, use mplayer.
    if (theApp.glob_prefs->GetVideoPlayer() == "")
    command.Append(wxT("mplayer"));
    else
    command.Append(theApp.glob_prefs->GetVideoPlayer());
    // Need to use quotes in case filename contains spaces.
    command.Append(wxT(" \""));
    command.Append(GetFullName());
    // Remove the .met from filename.
    for (int i = 0 ; i < 4 ; i++) command.RemoveLast();
    command.Append(wxT("\""));
    wxShell(command.c_str());
}

bool CPartFile::PreviewAvailable()
{
    return(IsMovie() &&IsComplete(0, PARTSIZE));
}

void CPartFile::UpdateAvailablePartsCount()
{
    wxUint8 availablecounter = 0;
    bool breakflag = false;
    wxUint16 iPartCount = GetPartCount();
    for (wxUint32 ixPart = 0 ; ixPart < iPartCount ; ixPart++)
    {
        breakflag = false;
        for (wxUint32 sl = 0 ; sl < SOURCESSLOTS && !breakflag ; sl++)
        {
            if (!srclists[sl].IsEmpty())
            {
                for (POSITION pos = srclists[sl].GetHeadPosition() ; pos && !breakflag ;)
                {
                    if (srclists[sl].GetNext(pos)->IsPartAvailable(ixPart))
                    {
                        availablecounter++;
                        breakflag = true;
                    }
                }
            }
        }
    }
    if (iPartCount == availablecounter &&availablePartsCount < iPartCount)
    //CTime::GetCurrentTime();:
    lastseencomplete = time(NULL);
    availablePartsCount = availablecounter;
}

Packet *CPartFile::CreateSrcInfoPacket(CUpDownClient *forClient)
{
#if defined(__DEBUG__)
    // printf("CPartFile::CreateSrcInfoPacket\n");
#endif
    int sl;
    for (sl = 0 ; sl < SOURCESSLOTS ; sl++) if(srclists[sl].IsEmpty()) return 0;
    CMemFile data;
    wxUint16 nCount = 0;
    data.Write(m_abyFileHash, 16);
    data.Write( &nCount, 2);
    bool bNeeded;
    for (sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (POSITION pos = srclists[sl].GetHeadPosition() ; pos != 0 ; srclists[sl].GetNext(pos))
    {
        bNeeded = false;
        CUpDownClient *cur_src = srclists[sl].GetAt(pos);
        if (cur_src->HasLowID())
        continue;
        // only send source which have needed parts for this client if possible
        wxUint8 *srcstatus = cur_src->GetPartStatus();
        if (srcstatus)
        {
            wxUint8 *reqstatus = forClient->GetPartStatus();
            if (reqstatus)
            {
                // only send sources which have needed parts for this client
                for (int x = 0 ; x < GetPartCount() ; x++)
                {
                    if (srcstatus[x] && !reqstatus[x])
                    {
                        bNeeded = true;
                        break;
                    }
                }
            }
            else
            {
                // if we don't know the need parts for this client, return any source
                // currently a client sends it's file status only after it has at least one complete part,
                for (int x = 0 ; x < GetPartCount() ; x++)
                {
                    if (srcstatus[x])
                    {
                        bNeeded = true;
                        break;
                    }
                }
            }
        }
        if (bNeeded)
        {
            nCount++;
            wxUint32 dwID = cur_src->GetUserID();
            wxUint16 nPort = cur_src->GetUserPort();
            wxUint32 dwServerIP = cur_src->GetServerIP();
            wxUint16 nServerPort = cur_src->GetServerPort();
            data.Write( &dwID, 4);
            data.Write( &nPort, 2);
            data.Write( &dwServerIP, 4);
            data.Write( &nServerPort, 2);
            if (forClient->GetSourceExchangeVersion() > 1)
            data.Write(cur_src->GetUserHash(), 16);
            if (nCount > 500)
            break;
        }
    }
    if (!nCount)
    return 0;
    data.Seek(16);
    data.Write( &nCount, 2);
    Packet *result = new Packet( &data, OP_EMULEPROT);
    result->opcode = OP_ANSWERSOURCES;
    if (nCount > 28)
    result->PackPacket();
    return result;
}

void CPartFile::AddClientSources(CMemFile *sources, wxUint8 sourceexchangeversion)
{
    if (stopped) return;
    wxUint16 nCount;
    sources->Read( &nCount, 2);
    for (int i = 0 ; i != nCount ; i++)
    {
        wxUint32 dwID;
        wxUint16 nPort;
        wxUint32 dwServerIP;
        wxUint16 nServerPort;
        uchar achUserHash[16];
        sources->Read( &dwID, 4);
        sources->Read( &nPort, 2);
        sources->Read( &dwServerIP, 4);
        sources->Read( &nServerPort, 2);
        if (sourceexchangeversion > 1)
        sources->Read(achUserHash, 16);
        // check first if we are this source
        if (theApp.serverconnect->GetClientID() < 16777216 &&theApp.serverconnect->IsConnected())
        {
            if ((theApp.serverconnect->GetClientID() == dwID) &&theApp.serverconnect->GetCurrentServer()->GetIP() == dwServerIP)
            continue;
        }
        else if(theApp.serverconnect->GetClientID() == dwID)
        continue;
        else if(dwID < 16777216)
        continue;
        if (theApp.glob_prefs->GetMaxSourcePerFile() > this->GetSourceCount())
        {
            CUpDownClient *newsource = new CUpDownClient(nPort, dwID, dwServerIP, nServerPort, this);
            if (sourceexchangeversion > 1)
            newsource->SetUserHash(achUserHash);
            theApp.downloadqueue->CheckAndAddSource(this, newsource);
        }
        else
        break;
    }
}

void CPartFile::UpdateAutoDownPriority()
{
    if (!IsAutoDownPriority())
    return;
    if (GetSourceCount() <= RARE_FILE)
    SetDownPriority(PR_VERYHIGH);
    else if(GetSourceCount() < 100)
    SetDownPriority(PR_HIGH);
    else
    SetDownPriority(PR_NORMAL);
}

// making this function return a higher when more sources have the extended
// protocol will force you to ask a larger variety of people for sources
int CPartFile::GetCommonFilePenalty()
{
    //TODO: implement, but never return less than MINCOMMONPENALTY!
    return MINCOMMONPENALTY;
}

 /* Barry - Replaces BlockReceived()
 Originally this only wrote to disk when a full 180k block
 had been received from a client, and only asked for data in
 180k blocks.
 This meant that on average 90k was lost for every connection
 to a client data source. That is a lot of wasted data.
 To reduce the lost data, packets are now written to a buffer
 and flushed to disk regularly regardless of size downloaded.
 This includes compressed packets.
 Data is also requested only where gaps are, not in 180k blocks.
 The requests will still not exceed 180k, but may be smaller to
 fill a gap.
    */

wxUint32 CPartFile::WriteToBuffer(wxUint32 transize, wxByte *data, wxUint32 start, wxUint32 end, Requested_Block_Struct *block)
{
    // Increment transfered bytes counter for this file
    transfered += transize;
    // This is needed a few times
    wxUint32 lenData = end - start + 1;
    if (lenData > transize)
    m_iGainDueToCompression += lenData - transize;
    // Occasionally packets are duplicated, no point writing it twice
    if (IsComplete(start, end))
    {
        return 0;
    }
    // Create copy of data as new buffer
    wxByte *buffer = new wxByte[lenData];
    memcpy(buffer, data, lenData);
    // Create a new buffered queue entry
    PartFileBufferedData *item = new PartFileBufferedData;
    item->data = buffer;
    item->start = start;
    item->end = end;
    item->block = block;
    // Add to the queue in the correct position (most likely the end)
    PartFileBufferedData *queueItem;
    bool added = false;
    POSITION pos = m_BufferedData_list.GetTailPosition();
    while (pos != NULL)
    {
        queueItem = m_BufferedData_list.GetPrev(pos);
        if (item->end > queueItem->end)
        {
            added = true;
            m_BufferedData_list.InsertAfter(pos, item);
            break;
        }
    }
    if (!added)
    m_BufferedData_list.AddHead(item);
    // Increment buffer size marker
    m_nTotalBufferData += lenData;
    // Mark this small section of the file as filled
    FillGap(item->start, item->end);
    // Update the flushed mark on the requested block
    // The loop here is unfortunate but necessary to detect deleted blocks.
    pos = requestedblocks_list.GetHeadPosition();
    while (pos != NULL)
    {
        if (requestedblocks_list.GetNext(pos) == item->block)
        item->block->transferred += lenData;
    }
    if (gaplist.IsEmpty()) FlushBuffer();
    // Return the length of data written to the buffer
    return lenData;
}

void CPartFile::FlushBuffer(void)
{
    m_nLastBufferFlushTime = GetTickCount();
    if (m_BufferedData_list.IsEmpty())
    {
        return;
    }
 /* Madcat - Check if there is at least PARTSIZE amount of free disk space
 in temp dir before flushing. If not enough space, pause the file,
    add log line and abort flushing. */
    wxLongLong total, free;
    if (wxGetDiskSpace(theApp.glob_prefs->GetTempDir(), &total, &free) &&
    free < PARTSIZE)
    {
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_OUT_OF_SPACE));
        PauseFile();
        return;
    }
    wxUint32 partCount = GetPartCount();
    bool* changedPart = new bool[partCount];
    try
    {
        // Remember which parts need to be checked at the end of the flush
        for (int partNumber = 0; (wxUint32)partNumber < partCount; ++partNumber)
        {
            changedPart[partNumber] = false;
        }
        // Ensure file is big enough to write data to (the last item will be the furthest from the start)
        PartFileBufferedData *item = m_BufferedData_list.GetTail();
        if (m_hpartfile->Length() <= item->end)
        //m_hpartfile->SetLength(item->end + 1);
        ftruncate(m_hpartfile->fd(), item->end + 1);

        // Loop through queue
        for (int i = m_BufferedData_list.GetCount(); i > 0; --i)
        {
            // Get top item
            item = m_BufferedData_list.GetHead();
            // This is needed a few times
            wxUint32 lenData = item->end - item->start + 1;
            int curpart = item->start / PARTSIZE;
            changedPart[curpart] = true;
            // Go to the correct position in file and write block of data
            m_hpartfile->Seek(item->start);
            m_hpartfile->Write(item->data, lenData);
            // Remove item from queue
            m_BufferedData_list.RemoveHead();
            // Decrease buffer size
            m_nTotalBufferData -= lenData;
            // Release memory used by this item
            delete[] item->data;
            delete item;
        }
        // Flush to disk
        m_hpartfile->Flush();
        // Check each part of the file
        wxUint32 partRange = (m_nFileSize % PARTSIZE) - 1;
        for (int partNumber = partCount - 1; partNumber >= 0; --partNumber)
        {
            if (changedPart[partNumber] == false)
            {
                // Any parts other than last must be full size
                partRange = PARTSIZE - 1;
                continue;
            }
            // Is this 9MB part complete
            if (IsComplete(PARTSIZE * partNumber, (PARTSIZE * (partNumber + 1)) - 1))
            {
                // Is part corrupt
                if (!HashSinglePart(partNumber))
                {
                    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_PARTCORRUPT), partNumber, GetFileName());
                    AddGap(PARTSIZE * partNumber, (PARTSIZE * partNumber + partRange));
                    // add part to corrupted list, if not already there
                    if (!IsCorruptedPart(partNumber))
                    {
                        corrupted_list.AddTail(partNumber);
                    }
                    // Reduce transfered amount by corrupt amount
                    m_iLostDueToCorruption += (partRange + 1);
                }
                else
                {
                    // if this part was successfully completed (although ICH is active), remove from corrupted list
                    POSITION posCorrupted = corrupted_list.Find(partNumber);
                    if (posCorrupted)
                    {
                        corrupted_list.RemoveAt(posCorrupted);
                    }

                    // Successfully completed part, make it available for sharing
                    if (status == PS_EMPTY && !hashsetneeded)
                    {
                        status = PS_READY;
                        theApp.sharedfiles->SafeAddKFile(1, this);
                    }
                }
            }
            else if (IsCorruptedPart(partNumber) && theApp.glob_prefs->IsICHEnabled())
            {
                // Try to recover with minimal loss
                if (HashSinglePart(partNumber))
                {
                    m_iTotalPacketsSavedDueToICH++;

                    FillGap(PARTSIZE * partNumber, (PARTSIZE * partNumber + partRange));
                    RemoveBlockFromList(PARTSIZE * partNumber, (PARTSIZE * partNumber + partRange));

                    POSITION posCorrupted = corrupted_list.Find(partNumber);
                    if (posCorrupted)
                    {
                        corrupted_list.RemoveAt(posCorrupted);
                    }
                    theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ICHWORKED), partNumber, GetFileName());
                    if (status == PS_EMPTY && !hashsetneeded)
                    {
                        status = PS_READY;
                        theApp.sharedfiles->SafeAddKFile(1, this);
                    }
                }
            }
            // Any parts other than last must be full size
            partRange = PARTSIZE - 1;
        }
        // Update met file
        SavePartFile();
        // Is this file finished?
        if (gaplist.IsEmpty())
        {
            CompleteFile(false);
        }
    }
    catch(...)
    {
        theApp.xmuledlg->AddLogLine(true, GetResString(IDS_ERR_WRITEERROR), GetFileName(), GetResString(IDS_UNKNOWN) .GetData());
        SetPartFileStatus(PS_ERROR);
        paused = true;
        datarate = 0;
        transferingsrc = 0;
        //theApp.xmuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
        UpdateDisplayedInfo();
    }
    delete[] changedPart;
}

// Barry - This will invert the gap list, up to caller to delete gaps when done
// 'Gaps' returned are really the filled areas, and guaranteed to be in order
void CPartFile::GetFilledList(CTypedPtrList < CPtrList, Gap_Struct *> *filled)
{
    Gap_Struct *gap;
    Gap_Struct *best;
    POSITION pos;
    wxUint32 start = 0;
    wxUint32 bestEnd = 0;
    // Loop until done
    bool finished = false;
    while (!finished)
    {
        finished = true;
        // Find first gap after current start pos
        bestEnd = m_nFileSize;
        pos = gaplist.GetHeadPosition();
        while (pos != NULL)
        {
            gap = gaplist.GetNext(pos);
            if ((gap->start > start) && (gap->end < bestEnd))
            {
                best = gap;
                bestEnd = best->end;
                finished = false;
            }
        }
        if (!finished)
        {
            // Invert this gap
            gap = new Gap_Struct;
            gap->start = start;
            gap->end = best->start - 1;
            start = best->end + 1;
            filled->AddTail(gap);
        }
        else if(best->end < m_nFileSize)
        {
            gap = new Gap_Struct;
            gap->start = best->end + 1;
            gap->end = m_nFileSize;
            filled->AddTail(gap);
        }
    }
}

void CPartFile::UpdateFileRatingCommentAvail()
{
    if (!this) return;
    bool prev = (hasComment || hasRating);
    hasComment = false;
    hasRating = false;
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        srclists[sl].GetNext(pos1);
        CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
        if (cur_src->GetFileComment() .GetLength() > 0) hasComment = true;
        if (cur_src->GetFileRate() > 0) hasRating = true;
        if (hasComment &&hasRating) break;
    }
    //theApp.xmuledlg->transferwnd.downloadlistctrl.UpdateItem(this);:
    if (prev != (hasComment || hasRating)) UpdateDisplayedInfo();
}

wxUint16 CPartFile::GetSourceCount()
{
    wxUint16 count = 0;
    for (int i = 0 ; i < SOURCESSLOTS ; i++) count += srclists[i].GetCount();
    return count;
}

bool CPartFile::HasBadRating()
{
    if (!hasRating) return false;
    POSITION pos1, pos2;
    for (int sl = 0 ; sl < SOURCESSLOTS ; sl++) if(!srclists[sl].IsEmpty())
    for (pos1 = srclists[sl].GetHeadPosition() ; (pos2 = pos1) != NULL ;)
    {
        srclists[sl].GetNext(pos1);
        CUpDownClient *cur_src = srclists[sl].GetAt(pos2);
        if (cur_src->GetFileRate() == 1) return true;
    }
    return false;
}

void CPartFile::UpdateDisplayedInfo(bool force)
{
    wxUint32 curTick =::GetTickCount();
    if (force || curTick - m_lastRefreshedDLDisplay > MINWAIT_BEFORE_DLDISPLAY_WINDOWUPDATE + (wxUint32)(rand() / (RAND_MAX/1000)))
    {
        theApp.xmuledlg->transferwnd->downloadlistctrl->UpdateItem(this);
        m_lastRefreshedDLDisplay = curTick;
    }
}

time_t CPartFile::GetLastChangeDatetime(bool forcecheck)
{
    if ((::GetTickCount() - m_lastdatetimecheck) < 60000 && !forcecheck) return m_lastdatecheckvalue;
    m_lastdatetimecheck =::GetTickCount();
    if (!::wxFileExists(m_hpartfile->GetFilePath())) m_lastdatecheckvalue = - 1;
    else
    {
        //CFileStatus filestatus;
        struct stat filestatus;
        fstat(m_hpartfile->fd(), &filestatus);
        //m_hpartfile->GetStatus(filestatus); // this; "...returns m_attribute without high-order flags" indicates a known MFC bug, wonder how many unknown there are... :)
        m_lastdatecheckvalue = filestatus.st_mtime;
    }
    return m_lastdatecheckvalue;
}

wxUint8 CPartFile::GetCategory()
{
    if (m_category > theApp.glob_prefs->GetCatCount() - 1) m_category = 0;
    return m_category;
}

CString CPartFile::GetDownloadFileInfo()
{
    if (this == NULL) return CString("");
    CString sRet;
        CString strHash = EncodeBase16(GetFileHash(), 16);;
    char lsc[50];
    char complx[50];
    char lastprogr[50];
    sprintf(complx, "%s/%s", CastItoXBytes(GetCompletedSize()) .GetData(), CastItoXBytes(GetFileSize()) .GetData());
    if (lastseencomplete == 0)
    {
        sprintf(lsc, GetResString(IDS_UNKNOWN) .MakeLower());
    }
    else
    {
        strftime(lsc, sizeof(lsc), theApp.glob_prefs->GetDateTimeFormat(), localtime((time_t *) &lastseencomplete));
    }
    if (GetFileDate() == 0)
    {
        sprintf(lastprogr, GetResString(IDS_UNKNOWN));
    }
    else
    {
        strftime(lastprogr, sizeof(lsc), theApp.glob_prefs->GetDateTimeFormat(), localtime(GetCFileDate()));
    }
    float availability = 0;
    if (GetPartCount() != 0)
    {
        availability = GetAvailablePartCount() *100 / GetPartCount();
    }
    sRet.Format(GetResString(IDS_DL_FILENAME) + ": %s (%s %s)\n\n%s\n\n"
    + GetResString(IDS_FD_HASH) + " %s\n"
    + GetResString(IDS_PARTINFOS) +
    GetResString(IDS_PARTINFOS2) + "\n%s\n%s",
    GetFileName(), CastItoXBytes(GetFileSize()).GetData(), GetResString(IDS_BYTES).GetData(),
    (GetResString(IDS_STATUS) + ": " + getPartfileStatus()) .GetData(),
    strHash.GetData(),
    GetPartMetFileName(), GetPartCount(), GetResString(IDS_AVAIL) .GetData(),
    GetAvailablePartCount(), availability,
    (int) GetPercentCompleted(), complx, GetTransferingSrcCount(),
    (GetResString(IDS_LASTSEENCOMPL) + " " + wxString(lsc)) .GetData(),
    (GetResString(IDS_FD_LASTCHANGE) + " " + wxString(lastprogr)) .GetData());
    return sRet;
}

wxString CPartFile::GetProgressString(wxUint16 size)
{
    //green:
    char crProgress = '0';
    // black:
    char crHave = '1';
    // yellow:
    char crPending = '2';
    // blue:
    char crWaiting = '3';
    // red:
    char crMissing = '4';
    wxString my_ChunkBar = "";
    for (wxUint16 i = 0 ; i <= size + 1 ; i++)
    {
        //.AppendChar(crHave);:
        my_ChunkBar += crHave;
    }
    // one more for safety
    wxUint32 m_nFileSize = GetFileSize();
    float unit = (float) size/ (float) m_nFileSize;
    wxUint32 allgaps = 0;
    if (GetStatus() == PS_COMPLETE || GetStatus() == PS_COMPLETING)
    {
        CharFillRange( &my_ChunkBar, 0, (float) m_nFileSize *unit, crProgress);
    }
    else
    {
        // red gaps
        for (POSITION pos = gaplist.GetHeadPosition() ; pos != 0 ; gaplist.GetNext(pos))
        {
            Gap_Struct *cur_gap = gaplist.GetAt(pos);
            allgaps += cur_gap->end - cur_gap->start;
            bool gapdone = false;
            wxUint32 gapstart = cur_gap->start;
            wxUint32 gapend = cur_gap->end;
            for (wxUint32 i = 0 ; i < GetPartCount() ; i++)
            {
                if (gapstart >= i *PARTSIZE &&gapstart <= (i + 1) *PARTSIZE)
                {
                    // is in this part?
                    if (gapend <= (i + 1) *PARTSIZE)
                    {
                        gapdone = true;
                    }
                    else
                    {
                        // and next part:
                        gapend = (i + 1) *PARTSIZE;
                    }
                    // paint
                    wxUint8 color;
                    if (m_SrcpartFrequency.GetCount() >= (int) i &&m_SrcpartFrequency[i])
                    {
                        // frequency?
                        color = crWaiting;
                    }
                    else
                    {
                        color = crMissing;
                    }
                    CharFillRange( &my_ChunkBar, (float) gapstart *unit, (float) gapend *unit + 1, color);
                    if (gapdone)
                    {
                        // finished?
                        break;
                    }
                    else
                    {
                        gapstart = gapend;
                        gapend = cur_gap->end;
                    }
                }
            }
        }
    }
    // yellow pending parts
    for (POSITION pos = requestedblocks_list.GetHeadPosition() ; pos != 0 ; requestedblocks_list.GetNext(pos))
    {
        Requested_Block_Struct *block = requestedblocks_list.GetAt(pos);
        CharFillRange( &my_ChunkBar, (float)(block->StartOffset + block->transferred) *unit, (float) block->EndOffset *unit, crPending);
    }
    return my_ChunkBar;
}

void CPartFile::CharFillRange(wxString *buffer, float start, float end, char color)
{
    for (wxUint32 i = (wxUint32) start ; i <= (wxUint32) end ; i++)
    {
        buffer->SetChar(i, color);
    }
}

 /* Razor 1a - Modif by MikaelB
    RemoveNoNeededSources function */

void CPartFile::RemoveNoNeededSources()
{
    POSITION position, temp_position;
    for (int slot = 0 ; slot < SOURCESSLOTS ; slot++)
    {
        if (! srclists[slot].IsEmpty())
        {
            position = srclists[slot].GetHeadPosition();
            while (position != NULL)
            {
                temp_position = position;
                srclists[slot].GetNext(position);
                CUpDownClient *client = srclists[slot].GetAt(temp_position);
                if (client->GetDownloadState() == DS_NONEEDEDPARTS)
                {
    /* If allowed, try to swap to other file. If swapping fails, remove from this one. */
                    if (theApp.glob_prefs->SwapNoNeededSources())
                    {
                        if (!client->SwapToAnotherFile())
                        {
                            theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                        }
    /* If not allowed to swap, simply remove from this one. */
                    }
                    else
                    {
                        theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                    }
                }
            }
        }
    }
}

    /* End modif */

 /* Razor 1a - Modif by MikaelB
    RemoveFullQueueSources function */

void CPartFile::RemoveFullQueueSources()
{
    POSITION position, temp_position;
    for (int slot = 0 ; slot < SOURCESSLOTS ; slot++)
    {
        if (! srclists[slot].IsEmpty())
        {
            position = srclists[slot].GetHeadPosition();
            while (position != NULL)
            {
                temp_position = position;
                srclists[slot].GetNext(position);
                CUpDownClient *client = srclists[slot].GetAt(temp_position);
                if ((client->GetDownloadState() == DS_ONQUEUE) && (client->IsRemoteQueueFull()))
                {
                    theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                }
            }
        }
    }
}

    /* End modif */

 /* Razor 1a - Modif by MikaelB
    RemoveHighQueueRatingSources function */

void CPartFile::RemoveHighQueueRatingSources()
{
    POSITION position, temp_position;
    for (int slot = 0 ; slot < SOURCESSLOTS ; slot++)
    {
        if (! srclists[slot].IsEmpty())
        {
            position = srclists[slot].GetHeadPosition();
            while (position != NULL)
            {
                temp_position = position;
                srclists[slot].GetNext(position);
                CUpDownClient *client = srclists[slot].GetAt(temp_position);
                if ((client->GetDownloadState() == DS_ONQUEUE) && (client->GetRemoteQueueRank() > theApp.glob_prefs->HighQueueRanking()))
                {
                    theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                }
            }
        }
    }
}

    /* End modif */

 /* Razor 1a - Modif by MikaelB
    CleanUpSources function */

void CPartFile::CleanUpSources()
{
    POSITION position, temp_position;
    for (int slot = 0 ; slot < SOURCESSLOTS ; slot++)
    {
        if (! srclists[slot].IsEmpty())
        {
            position = srclists[slot].GetHeadPosition();
            while (position != NULL)
            {
                temp_position = position;
                srclists[slot].GetNext(position);
                CUpDownClient *client = srclists[slot].GetAt(temp_position);
                if (client->GetDownloadState() == DS_NONEEDEDPARTS)
                {
                    if ((theApp.glob_prefs->DropNoNeededSources()) && (! client->SwapToAnotherFile()))
                    {
                        theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                    }
                }
                if ((client->GetDownloadState() == DS_ONQUEUE) && (client->IsRemoteQueueFull()))
                {
                    theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                }
                else if((client->GetDownloadState() == DS_ONQUEUE) && (client->GetRemoteQueueRank() > theApp.glob_prefs->HighQueueRanking()))
                {
                    theApp.downloadqueue->RemoveSourceFromPartFile(this, client, temp_position);
                }
            }
        }
    }
}

    /* End modif */

 /* Razor 1a - Modif by MikaelB
    AddDownloadingSource function */

void CPartFile::AddDownloadingSource(CUpDownClient *client)
{
    POSITION position = m_downloadingSourcesList.Find(client);
    if (position == NULL)
    {
        m_downloadingSourcesList.AddTail(client);
    }
}

    /* End modif */

 /* Razor 1a - Modif by MikaelB
    RemoveDownloadingSource function */

void CPartFile::RemoveDownloadingSource(CUpDownClient *client)
{
    POSITION position = m_downloadingSourcesList.Find(client);
    if (position != NULL)
    {
        m_downloadingSourcesList.RemoveAt(position);
    }
}

    /* End modif */

void CPartFile::SetPartFileStatus(wxUint8 newstatus)
{
    status = newstatus;
    if (theApp.glob_prefs->GetAllcatType() > 1)
    {
        theApp.xmuledlg->transferwnd->downloadlistctrl->Freeze();
        if (!CheckShowItemInGivenCat(this, theApp.xmuledlg->transferwnd->downloadlistctrl->curTab))
        {
            theApp.xmuledlg->transferwnd->downloadlistctrl->HideFile(this);
        }
        else
        {
            theApp.xmuledlg->transferwnd->downloadlistctrl->ShowFile(this);
        }
        theApp.xmuledlg->transferwnd->downloadlistctrl->Thaw();
        theApp.xmuledlg->transferwnd->downloadlistctrl->ShowFilesCount();
    }
    theApp.xmuledlg->transferwnd->downloadlistctrl->InitSort();
}

