# coding: utf-8
from __future__ import unicode_literals

import base64

from .common import InfoExtractor
from ..utils import qualities


class DumpertIE(InfoExtractor):
    _VALID_URL = r'https?://(?:www\.)?dumpert\.nl/mediabase/(?P<id>[0-9]+/[0-9a-zA-Z]+)'
    _TEST = {
        'url': 'http://www.dumpert.nl/mediabase/6646981/951bc60f/',
        'md5': '1b9318d7d5054e7dcb9dc7654f21d643',
        'info_dict': {
            'id': '6646981/951bc60f',
            'ext': 'mp4',
            'title': 'Ik heb nieuws voor je',
            'description': 'Niet schrikken hoor',
            'thumbnail': 're:^https?://.*\.jpg$',
        }
    }

    def _real_extract(self, url):
        video_id = self._match_id(url)
        webpage = self._download_webpage(url, video_id)

        files_base64 = self._search_regex(
            r'data-files="([^"]+)"', webpage, 'data files')

        files = self._parse_json(
            base64.b64decode(files_base64.encode('utf-8')).decode('utf-8'),
            video_id)

        quality = qualities(['flv', 'mobile', 'tablet', '720p'])

        formats = [{
            'url': video_url,
            'format_id': format_id,
            'quality': quality(format_id),
        } for format_id, video_url in files.items() if format_id != 'still']
        self._sort_formats(formats)

        title = self._html_search_meta(
            'title', webpage) or self._og_search_title(webpage)
        description = self._html_search_meta(
            'description', webpage) or self._og_search_description(webpage)
        thumbnail = files.get('still') or self._og_search_thumbnail(webpage)

        return {
            'id': video_id,
            'title': title,
            'description': description,
            'thumbnail': thumbnail,
            'formats': formats
        }
